/*                                                
 * @(#)VisualNumerics.wave  1.00 96/09/24 Steve Lang
 * 
 * Copyright (c) 1996 Visual Numerics Inc. All Rights Reserved.
 *                      Boulder, Colorado, USA
 * This software is confidential information which is proprietary to
 * and a trade secret of Visual Numerics, Inc.  Use, duplication or
 * disclosure is subject to the terms of an appropriate license agreement.
 *  
 * VISUAL NUMERICS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT. VNI SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
 * LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR
 * ITS DERIVATIVES.
 */

package VisualNumerics.wave;

/**
 * A class for creating pie charts.<p>
 *
 * This class encapsulates the PV-WAVE <tt>PIE_CHART2</tt> command
 * and all of the associated positional parameters and keywords
 * and produces a plot in a new container.<p>
 *
 * Methods for setting all of the parameters/keywords have the form 
 * <tt>setKeyword(arg1, arg2, ...)</tt> where "Keyword" is the parameter
 * or keyword name from the PV-WAVE documentation.  See the WaveGraph
 * class documentation for information on these methods.<p>
 *
 * This class also handles the communication between the applet and a PV-WAVE
 * graphic server running on the web server that served the java applet.
 *
 * @version 2.00, 4 Dec 1996
 * @author  Steve Lang
 * @see WaveGraph#setX
 * @see WaveGraph#TEK_COLOR
 * @see WaveGraph#LOADCT
 * @see WaveGraph#setBackground
 * @see WaveGraph#setCharsize
 * @see WaveGraph#setDevice
 * @see WaveGraph#setFont
 */
public class WavePie extends WaveGraph
{  

   /**
    * Creates a container of the requested size
    * @param x  the x location of the View
    * @param y  the x location of the View
    * @param xs the with of the View
    * @param ys the height of the View
    */
   public WavePie( int x, int y, int xsize, int ysize ) 
   {  super( x, y, xsize, ysize );
   }
   
   /**
    * Connects to the PV-WAVE server and creates the graphic
    * display and displays it in the container.
    */
   public void plot()
   {
      // Set the graph type
      addGraph("WavePie");
      
      // Add the data to plot
      addX();
      
      // Load color tables
      addTEK_COLOR();
      addLOADCT();
      
      // Add some special tags just for WavePie
      addXCenter();
      addYCenter();
      addRadius();
      addLabel();
      addTPosition();
      addTColor();
      addTPerct();
      addTValue();
      addTBorder();
      addTBord_color();
      addColor();
      addShade();
      addExplode();
      
      // Add all keywords supported for this plot type
      addBackground();
      addCharsize();
      addDevice();
      addFont();

      // Call the Wave graphic server
      callWave();
   }

   /**
    * Set X center (default normal coordinates)
    */
   public void setXCenter( double pos )
   {  xcenter = pos;
   }

   /**
    * Set Y center (default normal coordinates)
    */
   public void setYCenter( double pos )
   {  ycenter = pos;
   }

   /**
    * Set pie radius (default normal coordinates)
    */
   public void setRadius( double rad )
   {  radius = rad;
   }

   /**
    * Set the pie slice labels
    */
   public void setLabel(String[] labels)
   {  label = labels;
   }

   /**
    * Set the position of the labels:<P>
    *         0 : intern, 1 : extern, 2 : extern align<P>
    *         (def : 1 = extern)
    */
   public void setTPosition(int[] pos)
   {  tposition = pos;
   }

   /**
    * Set the color of the labels for each slice
    */
   public void setTColor(int[] col)
   {  tcolor = col;
   }

   /**
    * Display the % of each slice  (def: no display)
    */
   public void setTPerct( boolean flag )
   {  tperct = flag;
   }

   /**
    * Display the value of each slice  (def: no display)
    */
   public void setTValue( boolean flag )
   {  tvalue = flag;
   }

   /**
    * Display a border around the label (def: no border)
    */
   public void setTBorder( boolean flag )
   {  tborder = flag;
   }

   /**
    * Set color index for the label border
    */
   public void setTBord_color(int color)
   {  tbord_color = color;
   }

   /**
    * Set the color for each slice
    */
   public void setColor(int[] col)
   {  color = col;
   }

   /**
    * Display a shade under the pie_chart (def : 0.)
    * specifies the % of displacement for the center of the
    * shadow pie (direction is 315 degree)
    */
   public void setShade( double percent )
   {  shade = percent;
   }

   /**
    * Set % of explode  for each slice (def : no expl )
    */
   public void setExplode(double[] percent)
   {  explode = percent;
   }

   // Add methods used by this class

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addXCenter()
   {  if (xcenter != 999.0) tags = tags + "XCentre\tF\t" + xcenter + "\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addYCenter()
   {  if (ycenter != 999.0) tags = tags + "YCentre\tF\t" + ycenter + "\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addRadius()
   {  if (radius != 999.0) tags = tags + "Radius\tF\t" + radius + "\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addLabel()
   {  if (label != null) {
         if (label.length >= 1) {
            tags = tags + "Label\tS";
            for (int i=0; i < label.length; i++)
               tags = tags + "\t" + label[i];
            tags = tags + "\n";
         }
      } 
   }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addTPosition()
   {  if (tposition != null) { 
         if (tposition.length >= 1) {
            tags = tags + "TPosition\tI";
            for (int i=0; i < tposition.length; i++)
               tags = tags + "\t" + tposition[i];
            tags = tags + "\n";
         }
      } 
   }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addTColor()
   {  if (tcolor != null) { 
         if (tcolor.length >= 1) {
            tags = tags + "TColor\tI";
            for (int i=0; i < tcolor.length; i++)
               tags = tags + "\t" + tcolor[i];
            tags = tags + "\n";
         }
      } 
   }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addTPerct()
   {  if (tperct) tags = tags + "TPerct\tI\t1\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addTValue()
   {  if (tvalue) tags = tags + "TValue\tI\t1\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addTBorder()
   {  if (tborder) tags = tags + "TBorder\tI\t1\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addTBord_color()
   {  if (tbord_color != 999) tags = tags + "TBord_color\tI\t" + tbord_color
                                     + "\n"; }
   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addColor()
   {  if (color != null) { 
         if (color.length >= 1) {
            tags = tags + "Color\tI";
            for (int i=0; i < color.length; i++)
               tags = tags + "\t" + color[i];
            tags = tags + "\n";
         }
      } 
   }


   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addShade()
   {  if (shade != 999.0) tags = tags + "Shade\tF\t" + shade + "\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addExplode()
   {  if (explode != null) { 
         if (explode.length >= 1) {
            tags = tags + "Explode\tF";
            for (int i=0; i < explode.length; i++)
               tags = tags + "\t" + explode[i];
            tags = tags + "\n";
         }
      } 
   }

   /**
    * Reset all PV-WAVE plot attributes and data
    */      
   public void reset()
   {
   super.reset();
   xcenter     = 999.0;
   ycenter     = 999.0;
   radius      = 999.0;
   label       = new String[0];
   tposition   = null;
   tcolor      = null;
   tperct     = false;
   tvalue      = false;
   tborder     = false;
   tbord_color = 999;
   color      = null;
   shade       = 999.0;
   explode     = null;
   }

   protected double   xcenter  = 999.0;
   protected double   ycenter  = 999.0;
   protected double   radius   = 999.0;
   protected String[] label;
   protected int[]    tposition;
   protected int[]    tcolor;
   protected boolean  tperct   = false;
   protected boolean  tvalue   = false;
   protected boolean  tborder  = false;
   protected int      tbord_color = 999;
   protected int[]    color;
   protected double   shade    = 999.0;
   protected double[] explode;
}
