/*                                                
 * @(#)VisualNumerics.wave  1.00 96/09/24 Steve Lang
 * 
 * Copyright (c) 1996 Visual Numerics Inc. All Rights Reserved.
 *                      Boulder, Colorado, USA
 * This software is confidential information which is proprietary to
 * and a trade secret of Visual Numerics, Inc.  Use, duplication or
 * disclosure is subject to the terms of an appropriate license agreement.
 *  
 * VISUAL NUMERICS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 * OR NON-INFRINGEMENT. VNI SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
 * LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR
 * ITS DERIVATIVES.
 */

package VisualNumerics.wave;

/**
 * A class for creating Histogram plots.
 *
 * This class encapsulates the PV-WAVE <tt>HISTOGRAM</tt> and 
 * <tt>PLOT_HISTOGRAM</tt> commands 
 * and all of the associated positional parameters and keywords
 * and produces a Histogram plot in a new container.<p>
 *
 * Methods for setting all of the parameters/keywords have the form 
 * <tt>setKeyword(arg1, arg2, ...)</tt> where "Keyword" is the parameter
 * or keyword name from the PV-WAVE documentation.  See the WaveGraph
 * class documentation for information on these methods.<p>
 *
 * This class also handles the communication between the applet and a PV-WAVE
 * graphic server running on the web server that served the java applet.
 *
 * @version 2.00, 4 Dec 1996
 * @author  Steve Lang
 * @see WaveGraph#setZ
 * @see WaveGraph#TEK_COLOR
 * @see WaveGraph#LOADCT
 * @see WaveGraph#setBackground
 * @see WaveGraph#setColor
 * @see WaveGraph#setNodata
 * @see WaveGraph#setPosition
 * @see WaveGraph#setRegion
 * @see WaveGraph#setTitle
 * @see WaveGraph#setThick
 * @see WaveGraph#setXRange
 * @see WaveGraph#setXStyle
 * @see WaveGraph#setXTicklen
 * @see WaveGraph#setXTitle
 * @see WaveGraph#setXType
 * @see WaveGraph#setYRange
 * @see WaveGraph#setYStyle
 * @see WaveGraph#setYTicklen
 * @see WaveGraph#setYTitle
 * @see WaveGraph#setYType
 * @see WaveGraph#setFile
 */
public class WaveHistogram extends WaveGraph
{  
   /**
    * Creates a frame of the requested size
    * @param x  the x location of the View
    * @param y  the x location of the View
    * @param xs the with of the View
    * @param ys the height of the View
    */
   public WaveHistogram( int x, int y, int xsize, int ysize )
   {  super( x, y, xsize, ysize );
   }
   
   /**
    * Connects to the PV-WAVE server and creates the graphic
    * display and displays it in the container.
    */
   public void plot()
   {
      // Set the graph type
      addGraph("WaveHistogram");

      // Add the data to plot
      addZ();

      // Load color tables
      addTEK_COLOR();
      addLOADCT();
      
      // Add all unique keywords supported for this plot type
      addAxiscolor();
      addBinsize();
      addFillcolor();
      addFilled();
      addNoaxis();
      addStepped();
      addXmin();
      addXmax();

      // Add all keywords supported for this plot type
      // from WaveGraph
      addBackground();
      addColor();
      addNodata();
      addPosition();
      addRegion();
      addTitle();
      addThick();

      addXRange();
      addXStyle();
      addXTicklen();
      addXTitle();
      addXType();

      addYRange();
      addYStyle();
      addYTicklen();
      addYTitle();
      addYType();

      // Call the Wave graphic server
      callWave();
   }


   // Set methods unique to this class
   
   /**
    * Set the histogram axis color
    */
   public void setAxiscolor(int color)
   {  axiscolor = color;
   }

   /**
    * Set the histogram bin size
    */
   public void setBinsize(int size)
   {  binsize = size;
   }
   
   /**
    * Set the histogram fill color
    */
   public void setFillcolor(int color)
   {  fillcolor = color;
   }

   /**
    * Set fill color using standard Java color names:
    * black green red blue lightGray white cyan magenta yellow
    * darkGray orange gray pink
    */
   public void setFillcolor( String col )
   {  if (col.equals("black"))     fillcolor = 0;
      if (col.equals("green"))     fillcolor = 3;
      if (col.equals("red"))       fillcolor = 2;
      if (col.equals("blue"))      fillcolor = 4;
      if (col.equals("lightGray")) fillcolor = 15;
      if (col.equals("white"))     fillcolor = 1;
      if (col.equals("cyan"))      fillcolor = 5;
      if (col.equals("magenta"))   fillcolor = 6;
      if (col.equals("yellow"))    fillcolor = 7;
      if (col.equals("darkGray"))  fillcolor = 14;
      if (col.equals("orange"))    fillcolor = 8;
      if (col.equals("gray"))      fillcolor = 15;
      if (col.equals("pink"))      fillcolor = 13;
   }

   /**
    * Set histogram style to filled 
    */
   public void setFilled( boolean flag )
   {  filled = flag;
   }

   /**
    * Set histogram axes off 
    */
   public void setNoaxis( boolean flag )
   {  noaxis = flag;
   }

   /**
    * Set histogram to display as steps rather than bars 
    */
   public void setStepped( boolean flag )
   {  stepped = flag;
   }

   /**
    * Set min Y value to be displayed on histogram
    */
   public void setXmin( double value )
   {  xmin = value;
   }

   /**
    * Set max Y value to be displayed on histogram
    */
   public void setXmax( double value )
   {  xmax = value;
   }


   // Add methods used by this class

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addAxiscolor()
   {  if (axiscolor != 999) tags = tags + "Axiscolor\tI\t" + axiscolor + "\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addBinsize()
   {  if (binsize != 999) tags = tags + "Binsize\tI\t" + binsize + "\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addFillcolor()
   {  if (fillcolor != 999) tags = tags + "Fillcolor\tI\t" + fillcolor + "\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addFilled()
   {  if (filled) tags = tags + "Filled\tI\t1\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addNoaxis()
   {  if (noaxis) tags = tags + "Noaxis\tI\t1\n"; }
   
   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addStepped()
   {  if (stepped) tags = tags + "Stepped\tI\t1\n"; }   
   
   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addXmin()
   {  if (xmin != 999.0) tags = tags + "Xmin\tF\t" + xmin + "\n"; }

   /**
    * Protected method used by classes derived from WaveGraph
    */   
   protected void addXmax()
   {  if (xmax != 999.0) tags = tags + "Xmax\tF\t" + xmax + "\n"; }

   /**
    * Reset all PV-WAVE plot attributes and data
    */      
   public void reset()
   {
   super.reset();
   axiscolor = 999;
   binsize   = 999;
   fillcolor = 999;
   filled    = false;
   noaxis    = false;
   stepped   = false;
   xmin      = 999.0;
   xmax      = 999.0;
   }
      
   protected int      axiscolor = 999;
   protected int      binsize   = 999;
   protected int      fillcolor = 999;
   protected boolean  filled    = false;
   protected boolean  noaxis    = false;
   protected boolean  stepped   = false;
   protected double   xmin      = 999.0;
   protected double   xmax      = 999.0;
            
}
