//	Zinc Application Framework - W_WIN.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include "w_app.hpp"
#include <z_appevt.hpp>
#include <z_mouse2.hpp>
#include <z_win.hpp>

// ----- ZafWindowManager --------------------------------------------------

ZafEventMap ZAF_FARDATA ZafWindowManager::defaultEventMap[] =
{
	{ L_NONE,		0,			0,					0 }
};

ZafPaletteMap ZAF_FARDATA ZafWindowManager::defaultPaletteMap[] =
{
	{ ZAF_PM_ANY_TYPE, ZAF_PM_ANY_STATE, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_DEFAULT, ZAF_CLR_DEFAULT, ZAF_MONO_BLACK, ZAF_MONO_WHITE, ZAF_FNT_DIALOG } }
};

ZafWindowManager::~ZafWindowManager(void)
{
//??? Why isn't this in a z_ file?
	// Remove all objects from the window manager.
	while (first)
	{
		ZafWindow *window = DynamicPtrCast(First(), ZafWindow);
		Subtract(window);
		if (window->Destroyable())
			delete window;
	}

	// Clear the static variables.
	ZafWindowObject::display = ZAF_NULLP(ZafDisplay);
	ZafWindowObject::eventManager = ZAF_NULLP(ZafEventManager);
	ZafWindowObject::windowManager = ZAF_NULLP(ZafWindowManager);
}

ZafWindowObject *ZafWindowManager::Add(ZafWindowObject *window, ZafWindowObject *position)
{
	// Make sure Add is allowed.
	if (window && !window->screenID)
	{
		// Add the window to the list. (Default to first in list, 
		// or in other words the top-most window.)
		ZafList::Add(window, position ? position : First());

		// Create the window with the OS.
		window->Event(S_CREATE);
	}

	return (window);
}

ZafEventType ZafWindowManager::ApplicationEvent(const ZafEventStruct &event)
{
	// Check for zinc events.
	ZafEventType ccode = event.type;
	switch (ccode)
	{
	// ----- Document requests -----
	case A_OPEN_DOCUMENT:
	case A_PRINT_DOCUMENT:
//??? Definition needed?
		break;

	// ----- Window requests -----
	case A_OPEN_WINDOW:
	case A_CLOSE_WINDOW:
//??? Definition needed?
		break;

	// ----- Language/Locale requests -----
	case A_CHANGE_LANG_LOC:
		break;

	// ----- Layout requests -----
//??? MDI
/* START BLOCK COMMENT
**		case A_CASCADE_WINDOWS:
**			// Cascade the application windows.
**			if (First())
**			{
**				ZafEventStruct aEvent(S_SIZE);
**				aEvent.region.left = aEvent.region.top = 0;
**				for (ZafWindowObject *object = Last(); object; object = object->Previous())
**				{
**					// Move the window.
**					aEvent.region.right = aEvent.region.left + object->zafRegion.Width();
**					aEvent.region.bottom = aEvent.region.top + object->zafRegion.Height();
**					object->Event(aEvent);
**	
**					// Update the region.
**					aEvent.region.left += display->cellHeight;
**					aEvent.region.top += display->cellHeight;
**				}
**			}
**			break;
END BLOCK COMMENT */

//??? MDI
/* START BLOCK COMMENT
**		case A_TILE_WINDOWS:
**			// Tile the application windows.
**			if (first)
**			{
**				// Determine the proper size of each child.  Algorithm keeps
**				// computing until the window ratio is 1-to-1 (100% as wide as 
**				// it is tall).
**				int count = Count();
**				int width = 0, height = 0;
**				for (int rows = 1; rows <= count && width <= height; rows++)
**				{
**					width = display->columns / ((count + rows - 1) / rows);
**					height = display->lines / rows;
**				}
**	
**				// Resize the children.
**				ZafEventStruct sizeEvent(S_SIZE);
**				sizeEvent.region.left = sizeEvent.region.top = 0;
**				for (ZafWindowObject *object = Last(); object; object = object->Previous())
**				{
**					// Check for off-screen position.
**					if (sizeEvent.region.left + width > display->columns)
**					{
**						sizeEvent.region.left = 0;
**						sizeEvent.region.top += height;
**					}
**	
**					// Reposition the object.
**					sizeEvent.region.right = sizeEvent.region.left + width - 1;
**					sizeEvent.region.bottom = sizeEvent.region.top + height - 1;
**					object->Event(sizeEvent);
**	
**					// Determine the new object position.
**					sizeEvent.region.left += width;
**				}
**			}
**			break;
END BLOCK COMMENT */

	case A_MINIMIZE_WINDOWS:
		BroadcastEvent(S_MINIMIZE);
		break;

	case A_RESTORE_WINDOWS:
		BroadcastEvent(S_RESTORE);
		break;

	// ----- Errors -----
	default:
		ccode = S_ERROR;
	}

	// Return the control code.
	return (ccode);
}

ZafEventType ZafWindowManager::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	if (ccode == E_OSEVENT)
		return ZafWindowManager::OSEvent(event);

	// Process ZAF events.
	if (event.route)
		// Process routed events.
		ccode = event.route->Event(event);
	else
		// Process normal ZAF events.
		switch (ccode)
		{
		case S_CLOSE:
		case S_CLOSE_TEMPORARY:
			{
			// Get a pointer to the window that should be closed.
			ZafWindow *closeWindow;
			if (event.windowObject)
				// The window to be closed was provided with the event.
//??? An S_CLOSE_WINDOW or similar event that makes use of
//??? event.windowObject is needed.
				closeWindow = DynamicPtrCast(event.windowObject->RootObject(), ZafWindow);
			else
			{
				// Get the first window.
				closeWindow = DynamicPtrCast(First(), ZafWindow);

				// The first non-temporary window should be closed.
				if (ccode == S_CLOSE)
					while (closeWindow && closeWindow->Temporary())
						closeWindow = DynamicPtrCast(closeWindow->Next(), ZafWindow);
			}

			// Make sure it is ok to close the window.
			if (closeWindow && !closeWindow->Locked() &&
				!(ccode == S_CLOSE_TEMPORARY && !closeWindow->Temporary()) &&
				!(screenID == closeWindow->screenID && Event(ZafEventStruct(N_EXIT)) != 0) &&
				closeWindow->Event(ZafEventStruct(N_CLOSE)) == 0)
			{
				if (closeWindow->screenID == screenID)
					PostQuitMessage(0);

				// Close the window and all temporary windows in front of it.
				do
				{
					// Get the previous window.
					ZafWindowObject *prevWindow = closeWindow->Previous();

					// Close the window.
					Subtract(closeWindow);
					if (closeWindow->Destroyable())
						delete closeWindow;

					// Assign the previous window.
					closeWindow = DynamicPtrCast(prevWindow, ZafWindow);
				} while (closeWindow && closeWindow->Temporary());
			}

			// See if there are any windows left.
			if (!first)
				ccode = S_NO_OBJECT;
			}
			break;

		case S_EXIT:
			PostQuitMessage(0);
			break;

		case S_REDISPLAY:
			// Redisplay all windows.
			BroadcastEvent(event);
			break;

		case L_NEXT_WINDOW:
			{
			// Find the first non-temporary window.
			ZafWindow *firstWindow = DynamicPtrCast(First(), ZafWindow);
			while (firstWindow && firstWindow->Temporary())
				firstWindow = DynamicPtrCast(firstWindow->Next(), ZafWindow);
			}
			break;

		case N_EXIT:
			// See if it is OK to exit.
			if (exitFunction)
				ccode = (*exitFunction)(display, eventManager, this);
			else
				// A return value of zero allows application to exit.
				ccode = 0;
			break;

		// ----- Base class messages -----
		case S_ADD_OBJECT:
		case S_SUBTRACT_OBJECT:
			ccode = ZafWindow::Event(event);
			break;

		default:
			// Let first (top-most) window process the event.
			ccode = First() ? First()->Event(event) : S_NO_OBJECT;
			break;
		}

	// Return the control code.
	return (ccode);
}

ZafRegionStruct ZafWindowManager::MaxRegion(ZafWindowObject *, ZafVtJustify, ZafHzJustify)
{
	// Return the entire region of the display.
//???Regions?
//	ZafRegionStruct displayRegion = { 0, 0, display->columns - 1, display->lines - 1 };
	ZafRegionStruct displayRegion;
	displayRegion.left = displayRegion.top = 0;
	displayRegion.right = display->columns - 1;
	displayRegion.bottom = display->lines - 1;
	return (displayRegion);
}

// ----- OS Specific Functions ----------------------------------------------

ZafEventType ZafWindowManager::OSEvent(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	UINT message = event.osEvent.message;
	switch (message)
	{
	case WM_TIMER:
		if (event.osEvent.wParam == ZafMSWindowsApp::mouseTimerID)
		{
			// Test if an N_MOUSE_LEAVE event needs to be sent.
			// (If the mouse moves off of the application, this
			//  is the only way to notify the object.)
			if (mouseObject)
			{
				// See if the mouse is capured.
				HWND hwndCapture = GetCapture();
				if (hwndCapture)
				{
					if (hwndCapture == mouseObject->screenID ||
						IsChild(mouseObject->screenID, hwndCapture))
						break;
				}
				else
				{
					POINT mousePoint;
					GetCursorPos(&mousePoint);

					HWND hwndMouse = WindowFromPoint(mousePoint);
					if (hwndMouse == mouseObject->screenID ||
						IsChild(mouseObject->screenID, hwndMouse))
						break;
				}

				// Send the N_MOUSE_LEAVE event.
				mouseObject->Event(ZafEventStruct(N_MOUSE_LEAVE));
				mouseObject = ZAF_NULLP(ZafWindowObject);
			}

			// Free the mouse timer.
			KillTimer(NULL, ZafMSWindowsApp::mouseTimerID);
			ZafMSWindowsApp::mouseTimerID = 0;
		}
		else
			ccode = DispatchMessage(&event.osEvent);
		break;

	case WM_QUIT:
		// See if it is OK to exit the application.
		ccode = (Event(ZafEventStruct(N_EXIT)) == 0) ? S_EXIT : 0;
		break;

	case WM_KEYDOWN:
	case WM_KEYUP:
		if (dragObject)
		{
			if (event.osEvent.wParam == VK_ESCAPE)
			{
				dragObject = ZAF_NULLP(ZafWindowObject);
				eventManager->DeviceImage(E_MOUSE, DM_VIEW);
			}
			else
			{
				// Generate a WM_MOUSEMOVE message.
				POINT point;
				GetCursorPos(&point);
				SetCursorPos(point.x, point.y);
			}
			ccode = 0;
			break;
		}
		//Continue.
	default:
		// Message transleation is defered for Zinc objects because
		// the programmer may intercept and throw away messages. When this
		// happens, translation should not occur.
		// If the user throws away a WM_KEYDOWN message, for example,
		// a WM_CHAR message should not be generated.

		// Only translate messages not destined for Zinc objects.
		// This prevents the generation of WM_CHAR messages when the
		// user intercepts and throws away key-down messages.
		if (IsWindow(event.osEvent.hwnd) && !ZafMSWindowsApp::ObjectFromHandle(event.osEvent.hwnd))
			TranslateMessage(&event.osEvent);
		// Dispatch the Windows message.
		ccode = DispatchMessage(&event.osEvent);
		break;
	}

	return (ccode);
}

ZafWindowObject *ZafWindowManager::Subtract(ZafWindowObject *object)
{
	// Subtract the window.
	ZafWindow::Subtract(object);

	if (First())
	{
		// Reset the focus.
		if (!First()->Focus())
			First()->SetFocus(true);
	}
	else
		// Give control loop a chance to return S_NO_OBJECT.
		eventManager->Put(ZafEventStruct(S_CONTINUE));

	return (object);
}

// ----- Stubs --------------------------------------------------------------

ZafPositionStruct ZafWindowManager::ConvertToOSPosition(const ZafWindowObject *object,
	const ZafPositionStruct *zafRegion) const
{
	return ZafWindowObject::ConvertToOSPosition(object, zafRegion);
}

ZafPositionStruct ZafWindowManager::ConvertToZafPosition(const ZafWindowObject *object,
	const ZafPositionStruct *osRegion) const
{
	return ZafWindowObject::ConvertToZafPosition(object, osRegion);
}

ZafRegionStruct ZafWindowManager::ConvertToOSRegion(const ZafWindowObject *object,
	const ZafRegionStruct *region) const
{
	return ZafWindowObject::ConvertToOSRegion(object, region);
}

ZafRegionStruct ZafWindowManager::ConvertToZafRegion(const ZafWindowObject *object,
	const ZafRegionStruct *region) const
{
	return ZafWindowObject::ConvertToZafRegion(object, region);
}

void ZafWindowManager::OSMapPalette(ZafPaletteStruct &palette, ZafPaletteType, ZafPaletteState)
{
	if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
		palette.osPalette.colorForeground = GetSysColor(COLOR_WINDOWTEXT);
	if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
#if defined(ZAF_MSWINDOWS_3D)
		palette.osPalette.colorBackground = GetSysColor(COLOR_3DFACE);
#else
		palette.osPalette.colorBackground = GetSysColor(COLOR_WINDOW);
#endif
}



