//	Zinc Application Framework - W_SCROLL.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include "w_app.hpp"
#include <z_scrll2.hpp>

// ----- ZafScrollBar -----------------------------------------------------

ZafPaletteMap ZAF_FARDATA ZafScrollBar::defaultPaletteMap[] =
{
	// Palette is inherited from ZafWindowObject.
	{ ZAF_PM_NONE, 0 }
};

ZafEventType ZafScrollBar::Draw(const ZafEventStruct &, ZafEventType ccode)
{
	return (ccode);
}

ZafEventType ZafScrollBar::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	if (ccode == E_OSEVENT)
		return ZafScrollBar::OSEvent(event);

	// Process ZAF events.
	switch (ccode)
	{
	case S_REGISTER_OBJECT:
		ccode = ZafWindowObject::Event(event);
		if (SupportObject())
		{
			// Initialize the scroll bar settings.
			if (ScrollType() != ZAF_CORNER_SCROLL)
				OSSetScroll();
		}
		break;

	case S_COMPUTE_SIZE:
		ZafWindowObject::Event(event);
		if (autoSize)
		{
#if defined(ZAF_WIN32)
			if (scrollType == ZAF_VERTICAL_SLIDER)
				zafRegion.right = zafRegion.left + 22;
			else if (scrollType == ZAF_VERTICAL_SCROLL)
				zafRegion.right = zafRegion.left + GetSystemMetrics(SM_CXVSCROLL) - 1;
			else if (scrollType == ZAF_HORIZONTAL_SLIDER)
				zafRegion.bottom = zafRegion.top + 22;
			else if (scrollType == ZAF_HORIZONTAL_SCROLL)
				zafRegion.bottom = zafRegion.top + GetSystemMetrics(SM_CYHSCROLL) - 1;
#else
			if (scrollType == ZAF_VERTICAL_SCROLL || scrollType == ZAF_VERTICAL_SLIDER)
		  		zafRegion.right = zafRegion.left + GetSystemMetrics(SM_CXVSCROLL) - 1;
			else if (scrollType == ZAF_HORIZONTAL_SCROLL || scrollType == ZAF_HORIZONTAL_SLIDER)
		  		zafRegion.bottom = zafRegion.top + GetSystemMetrics(SM_CYHSCROLL) - 1;
#endif
		}
		break;

	case S_HSCROLL:
	case S_VSCROLL:
		scrollData->current = scrollData->current + event.scroll.delta;
		OSSetScroll();
		break;

	case S_HSCROLL_SET:
	case S_VSCROLL_SET:
		*scrollData = event.scroll;
		OSSetScroll();
		break;

	default:
		ccode = ZafWindowObject::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

// ----- OS Specific Functions ----------------------------------------------

ZafEventType ZafScrollBar::OSEvent(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	switch (event.osEvent.message)
	{
	case WM_HSCROLL:
	case WM_VSCROLL:
		{
#if defined (ZAF_WIN32)
		int scrollCode = (int)LOWORD(event.osEvent.wParam);
		short nPos = (short)HIWORD(event.osEvent.wParam);
#else
		int scrollCode = (int)event.osEvent.wParam;
		short nPos = (short)LOWORD(event.osEvent.lParam);
#endif

		bool invert = ScrollType() == ZAF_VERTICAL_SLIDER;
		long scrollPos = scrollData->Current();
		ZafEventStruct scrollEvent;

		switch (scrollCode)
		{
		case SB_BOTTOM:
			if (invert)
				scrollPos = scrollData->Minimum();
			else
				scrollPos = scrollData->Maximum();
			scrollEvent.type = L_LAST;
			break;

		case SB_TOP:
			if (invert)
				scrollPos = scrollData->Maximum();
			else
				scrollPos = scrollData->Minimum();
			scrollEvent.type = L_FIRST;
			break;

		case SB_LINEDOWN:
			if (invert)
				scrollPos -= scrollData->Delta();
			else
				scrollPos += scrollData->Delta();
			scrollEvent.type = L_DOWN;
			break;

		case SB_LINEUP:
			if (invert)
				scrollPos += scrollData->Delta();
			else
				scrollPos -= scrollData->Delta();
			scrollEvent.type = L_UP;
			break;

		case SB_PAGEDOWN:
			if (invert)
				scrollPos -= scrollData->Showing();
			else
				scrollPos += scrollData->Showing();
			scrollEvent.type = L_PGDN;
			break;

		case SB_PAGEUP:
			if (invert)
				scrollPos += scrollData->Showing();
			else
				scrollPos -= scrollData->Showing();
			scrollEvent.type = L_PGUP;
			break;

		case SB_THUMBPOSITION:
		case SB_THUMBTRACK:
			if (invert)
				scrollPos = scrollData->Maximum() + scrollData->Minimum() - (nPos * scrollData->Delta());
			else
				scrollPos = nPos * scrollData->Delta();
			scrollEvent.type = (scrollCode == SB_THUMBTRACK) ? L_CONTINUE_SELECT : L_END_SELECT;
			break;
		}

		if (scrollPos < scrollData->Minimum())
			scrollPos = scrollData->Minimum();
		else if (scrollPos > scrollData->Maximum())
			scrollPos = scrollData->Maximum();
		if (scrollPos != scrollData->Current())
		{
			long delta = scrollPos - scrollData->current;
			scrollData->current = scrollPos;
			OSSetScroll();
			scrollEvent.scroll = *scrollData;
			scrollEvent.scroll.delta = delta;
			if (SupportObject())
			{
				ZafEventStruct sEvent;
				if (ScrollType() == ZAF_VERTICAL_SCROLL)
					sEvent.type = N_VSCROLL;
				else
					sEvent.type = N_HSCROLL;
				sEvent.scroll.delta = delta;
				parent->Event(sEvent);
			}
			else
				(this->*memberUserFunction)(scrollEvent, L_SELECT);
		}

		ccode = 0;
		}
		break;

#if defined(ZAF_WIN32)
	case WM_CTLCOLORSTATIC:
		if (ScrollType() == ZAF_VERTICAL_SLIDER || ScrollType() == ZAF_HORIZONTAL_SLIDER)
		{
			// Get the palette.
			ZafPaletteStruct palette = parent->LogicalPalette(ZAF_PM_BACKGROUND, PaletteState());

			// Get the OS colors from the text palette.
			COLORREF colorForeground = (palette.colorForeground == ZAF_CLR_DEFAULT) ?
				palette.osPalette.colorForeground :
				display->colorTable[palette.colorForeground];
			COLORREF colorBackground = (palette.colorBackground == ZAF_CLR_DEFAULT) ?
				palette.osPalette.colorBackground :
				display->colorTable[palette.colorBackground];

			// Set the OS colors.
			SelectPalette((HDC)event.osEvent.wParam, display->logicalPalette, FALSE);
			::SetTextColor((HDC)event.osEvent.wParam, colorForeground);
			SetBkColor((HDC)event.osEvent.wParam, colorBackground);

			// Get a background brush.
			HBRUSH backgroundBrush = display->GetGdiCacheBrush(palette);
			display->ReleaseGdiCacheObject(backgroundBrush);
			ccode = (ZafEventType)backgroundBrush;
		}
		break;
#endif

	case WM_PAINT:
		ccode = DefaultCallback(event);
		break;

	case WM_KEYDOWN:
	case WM_KEYUP:
	case WM_SYSKEYDOWN:
	case WM_SYSKEYUP:
		if (event.osEvent.wParam == VK_TAB)
			ccode = S_UNKNOWN;
		else
			ccode = DefaultCallback(event);
		break;

	default:
		ccode = ZafWindowObject::OSEvent(event);
		break;
	}

	return (ccode);
}

ZafError ZafScrollBar::OSGetScroll(void)
{
	if (screenID)
	{
		if (ScrollType() == ZAF_HORIZONTAL_SLIDER || ScrollType() == ZAF_VERTICAL_SLIDER)
		{
#if defined(ZAF_WIN32)
			int scrollPos = (int)SendMessage(screenID, TBM_GETPOS, (WPARAM)0, (LPARAM)0);
#else
			int scrollPos = GetScrollPos(screenID, SB_CTL);
#endif
			if (ScrollType() == ZAF_VERTICAL_SLIDER)
				scrollData->current = (scrollData->maximum + scrollData->minimum -
					scrollPos * scrollData->delta);
			else
				scrollData->current = (scrollPos * scrollData->delta);
		}
		else
		{
			int scrollPos = GetScrollPos(screenID, SupportObject() ? (ScrollType() == ZAF_VERTICAL_SCROLL ? SB_VERT : SB_HORZ) : SB_CTL);
			scrollData->current = scrollPos * scrollData->delta;
		}
	}

	return (ZAF_ERROR_NONE);
}

void ZafScrollBar::OSMapPalette(ZafPaletteStruct &palette, ZafPaletteType type, ZafPaletteState state)
{
	// Defer to the base class.
	ZafWindowObject::OSMapPalette(palette, type, state);
}

void ZafScrollBar::OSRegisterObject(void)
{
	if (ScrollType() == ZAF_HORIZONTAL_SCROLL || ScrollType() == ZAF_VERTICAL_SCROLL)
	{
		// Set the Windows style flags.
		DWORD dwStyle = WS_CHILD;
		if (visible)
			dwStyle |= WS_VISIBLE;
		if (ScrollType() == ZAF_HORIZONTAL_SCROLL)
			dwStyle |= SBS_HORZ;
		else
			dwStyle |= SBS_VERT;

		// Create the scroll bar.
		screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, ZAF_ITEXT("SCROLLBAR"), dwStyle);

		// Initialize the scroll bar settings.
		OSSetScroll();
	}
	else if (ScrollType() == ZAF_HORIZONTAL_SLIDER || ScrollType() == ZAF_VERTICAL_SLIDER)
	{
#if defined(ZAF_WIN32)
		// Set the Windows style flags.
		DWORD dwStyle = WS_CHILD | TBS_NOTICKS;
		if (visible)
			dwStyle |= WS_VISIBLE;
		if (ScrollType() == ZAF_HORIZONTAL_SLIDER)
			dwStyle |= TBS_HORZ;
		else
			dwStyle |= TBS_VERT;

		// Create the slider.
		InitCommonControls();
		screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, TRACKBAR_CLASS, dwStyle);
#else
		// Set the Windows style flags.
		DWORD dwStyle = WS_CHILD;
		if (visible)
			dwStyle |= WS_VISIBLE;
		if (ScrollType() == ZAF_HORIZONTAL_SLIDER)
			dwStyle |= SBS_HORZ;
		else
			dwStyle |= SBS_VERT;

		// Create a scroll bar.
		screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, ZAF_ITEXT("SCROLLBAR"), dwStyle);
#endif

		// Initialize the slider settings.
		OSSetScroll();
	}
}

ZafError ZafScrollBar::OSSetScroll(void)
{
	if (screenID)
	{
		if (ScrollType() == ZAF_HORIZONTAL_SLIDER || ScrollType() == ZAF_VERTICAL_SLIDER)
		{
#if defined(ZAF_WIN32)
			SendMessage(screenID, TBM_SETRANGE, TRUE, MAKELONG(scrollData->Minimum() / scrollData->Delta(),
				scrollData->Maximum() / scrollData->Delta()));
			if (ScrollType() == ZAF_VERTICAL_SLIDER)
				SendMessage(screenID, TBM_SETPOS, TRUE, (scrollData->Maximum() + scrollData->Minimum() -
					scrollData->Current()) / scrollData->Delta());
			else
				SendMessage(screenID, TBM_SETPOS, TRUE,  scrollData->Current() / scrollData->Delta());
#else
//??? Support for long scroll values without overflow?
			SetScrollRange(screenID, SB_CTL, (int)(scrollData->Minimum() / scrollData->Delta()),
				(int)(scrollData->Maximum() / scrollData->Delta()), FALSE);
			if (ScrollType() == ZAF_VERTICAL_SLIDER)
				SetScrollPos(screenID, SB_CTL, (int)((scrollData->Maximum() + scrollData->Minimum() -
					scrollData->Current()) / scrollData->Delta()), TRUE);
			else
				SetScrollPos(screenID, SB_CTL, (int)(scrollData->Current() / scrollData->Delta()), TRUE);
#endif
		}
		else
		{
			int scrollID = SupportObject() ? ((ScrollType() == ZAF_VERTICAL_SCROLL) ? SB_VERT : SB_HORZ) : SB_CTL;
#if defined(ZAF_WIN32)
			SCROLLINFO scrollInfo;
			scrollInfo.cbSize = sizeof(scrollInfo);
			scrollInfo.fMask = SIF_ALL;
			scrollInfo.nMin = scrollData->Minimum() / scrollData->Delta();
			scrollInfo.nMax = (scrollData->Maximum() + scrollData->Showing()) / scrollData->Delta() - 1;
			scrollInfo.nPage = scrollData->Showing() / scrollData->Delta();
			scrollInfo.nPos = scrollData->Current() / scrollData->Delta();
			SetScrollInfo(screenID, scrollID, &scrollInfo, TRUE);
#else
			SetScrollRange(screenID, scrollID, (int)(scrollData->Minimum() / scrollData->Delta()),
				(int)(scrollData->Maximum() / scrollData->Delta()), FALSE);
			SetScrollPos(screenID, scrollID, (int)(scrollData->Current() / scrollData->Delta()), TRUE);
#endif
		}
	}

	return (ZAF_ERROR_NONE);
}

// ----- Stubs --------------------------------------------------------------

bool ZafScrollBar::SetFocus(bool setFocus)
{
	return ZafWindowObject::SetFocus(setFocus);
}

OSWindowID ZafScrollBar::OSScreenID(ZafScreenIDType) const
{
	return (screenID);
}

ZafEventType ZafScrollBar::ScrollEvent(const ZafEventStruct &event)
{
	return (event.type);
}

void ZafScrollBar::OSSize(void)
{
	ZafWindowObject::OSSize();
}

bool ZafScrollBar::SetVisible(bool visible)
{
	return ZafWindowObject::SetVisible(visible);
}
