#include <z_error.hpp>

//	Zinc Application Framework - W_HTIPS.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#define OEMRESOURCE			// Windows button messages and flags.
#include "w_app.hpp"
#include <z_htips.hpp>
#include <z_ctype.hpp>
#include <z_mouse1.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>

bool processDisplay = false;

// ----- ZafHelpTips -------------------------------------------------------

#if defined(ZAF_WIN32)
static ZafPaletteStruct systemPalette =
	{ ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_DEFAULT, ZAF_CLR_DEFAULT,
		ZAF_MONO_BLACK, ZAF_MONO_WHITE, ZAF_FNT_SMALL };
#endif
static ZafPaletteStruct zincPalette =
	{ ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_BLACK, ZAF_CLR_YELLOW,
		ZAF_MONO_BLACK, ZAF_MONO_WHITE, ZAF_FNT_SMALL };

void ZafHelpTips::DisplayHelpObjectTip(void)
{
	ZafWindowObject *mouseObject = zafWindowManager->mouseObject;
	if (!mouseObject)
		return;

	// Update the help object's text.
	const ZafIChar *helpObjectTip = mouseObject->HelpObjectTip();
	ZafWindowObject *helpObject = zafWindowManager->helpObject;
	if (helpObject && helpObjectTip && (helpTipsType == ZAF_HELPTIPS_HELPOBJECT ||
		helpTipsType == ZAF_HELPTIPS_BOTH))
	{
		if (replaceText)
			delete []replaceText;
		replaceText = helpObject->Text() ? strdup(helpObject->Text()) : ZAF_NULLP(ZafIChar);
		helpObject->SetText(helpObjectTip);
	}
}

void ZafHelpTips::DisplayQuickTip(void)
{
	ZafWindowObject *mouseObject = zafWindowManager->mouseObject;
	if (!mouseObject)
		return;

	// Display the quick tip.
	ZafIChar *quickTip;
	if (mouseObject->QuickTip())
 		quickTip = strdup(mouseObject->QuickTip());
	else
		return;

	if (helpTipsType == ZAF_HELPTIPS_QUICKTIP || helpTipsType == ZAF_HELPTIPS_BOTH)
	{
		ZafDisplay *display = this->display;
#if defined(ZAF_WIN32)
		if (ZafMSWindowsApp::windowsPlatform == ZAF_WIN32_WINDOWS)
		{
			quickTipPalette.osPalette.colorForeground = GetSysColor(COLOR_INFOTEXT);
			quickTipPalette.osPalette.colorBackground = GetSysColor(COLOR_INFOBK);
		}
#endif

		// Begin drawing operation.
		ZafRegionStruct drawRegion;
		drawRegion.coordinateType = ZAF_PIXEL;
		drawRegion.left = drawRegion.top = 0;
		drawRegion.right = display->columns - 1;
		drawRegion.bottom = display->lines - 1;
		display->BeginDraw(0, ID_ZAF_SCREEN, drawRegion, drawRegion);

		// Calculate position and size of quick tip using last mouse position
		// obtained with the last mouse movement.
		display->SetPalette(userPalette ? *userPalette : quickTipPalette);

		int height = 0, width = 0;
		bool multiline = false;

		if (!strchr(quickTip, '\n'))
		{
			ZafRegionStruct textSize = display->TextSize(quickTip);
			height = textSize.Height(); 
			width = textSize.Width(); 
		}
		else
		{
			multiline = true;
			ZafIChar *startLine = quickTip, *endLine;

			bool finished = false;
			do
			{
				endLine = strchr(startLine, '\n');
				if (!endLine)
					finished = true;
				else
					// Remove the new line so can parse the string.
					*endLine = '\0';

				// Get the string size.
				ZafRegionStruct textSize = display->TextSize(startLine);
				width = ZafMax(textSize.Width(), width);
				height += textSize.Height();

				// Replace the new line.
				if (!finished)
				{
					*endLine = '\n';
					startLine = ++endLine;
				}
			}while (!finished);
		}

		height += 2*display->preSpace;
		width += 2*display->cellWidth;

		drawRegion.left = lastMousePosition.column;
		drawRegion.top = lastMousePosition.line + ZAF_HELPTIPS_VT_OFFSET;
		ZafClassID objectID = mouseObject->ClassID();
		if (objectID == ID_ZAF_STRING || objectID == ID_ZAF_TEXT)
			drawRegion.top -= ZAF_HELPTIPS_VT_OFFSET / 2;
		drawRegion.right = drawRegion.left + width;
		drawRegion.bottom = drawRegion.top + height;

		// Adjust the drawRegion to not go beyond the right side of the display.
		if (drawRegion.right > (display->columns - 1))
		{
			drawRegion.right = display->columns - 1;
			drawRegion.left = drawRegion.right - width;
		}

		// Adjust the drawRegion to not go beyond the bottom of the display.
		if (drawRegion.bottom > (display->lines - 1))
		{
			drawRegion.bottom = display->lines - 1;
			drawRegion.top = drawRegion.bottom - height;
		}

		quickTipRegion = drawRegion;

		// Save the screen image under the quick tip.
		HDC hDC = GetDC(0);
		HDC bufferDC = CreateCompatibleDC(hDC);
		screenBuffer = CreateCompatibleBitmap(hDC, quickTipRegion.Width(), quickTipRegion.Height());
		SelectObject(bufferDC, screenBuffer);
		BitBlt(bufferDC, 0, 0, quickTipRegion.Width(), quickTipRegion.Height(), hDC,
			quickTipRegion.left, quickTipRegion.top, SRCCOPY);
		DeleteDC(bufferDC);
		ReleaseDC(0, hDC);

		// Draw the quick tip.
		display->Rectangle(drawRegion, 1, true);

		if (!multiline)
			display->Text(drawRegion.left, drawRegion.top, drawRegion.right, drawRegion.bottom,
				quickTip, strlen(quickTip), ZAF_HZ_CENTER, ZAF_VT_CENTER);
		else
		{
			ZafIChar *startLine = quickTip, *endLine;
			drawRegion.top += display->preSpace;
			drawRegion.left += display->cellWidth;

			bool finished = false;
			do
			{
				endLine = strchr(startLine, '\n');
				if (!endLine)
					finished = true;
				else
					// Remove the new line so can parse the string.
					*endLine = '\0';

				// Get the string size.
				ZafRegionStruct textSize = display->TextSize(startLine);
				height = textSize.Height();

				display->Text(drawRegion.left, drawRegion.top, drawRegion.right, drawRegion.bottom,
					startLine, -1, ZAF_HZ_LEFT, ZAF_VT_TOP);

				drawRegion.top += height;
		
				// Replace the new line.
				if (!finished)
				{
					*endLine = '\n';
					startLine = ++endLine;
				}
			}while (!finished);
		}
	
		// End drawing operation.
		display->EndDraw();
	}

	// Indicate that the quick tip is visible.
	quickTipVisible = true;
	allowQuickSiblingDisplay = true;

	// Turn off the timer.
	Reset(displayTimerHandle, false, 0, ZAF_NULLP(ZafEventStruct));

	// Delete the allocated string.
	delete []quickTip;
}

ZafEventType ZafHelpTips::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;
	switch (ccode)
	{
	case D_ON:
	case D_OFF:
		SetDeviceState(ccode);
		break;

	case D_STATE:
		ccode = DeviceState();
		break;

	case D_INITIALIZE:
		// Set the palette for the quickTip.
#if defined(ZAF_WIN32)
		if (ZafMSWindowsApp::windowsPlatform == ZAF_WIN32_WINDOWS)
			quickTipPalette = systemPalette;
		else
			quickTipPalette = zincPalette;
#else
		quickTipPalette = zincPalette;
#endif
		break;

	case D_DEINITIALIZE:
		// Kill the timer.
		Reset(displayTimerHandle, false, 0, ZAF_NULLP(ZafEventStruct));
		break;

	case N_MOUSE_ENTER:
		{
		// Update the help object's text.
		DisplayHelpObjectTip();

		// Elapsed time in hundredths of seconds.
		clock_t elapsedTime = (clock() - lastTime) * 100 / CLOCKS_PER_SEC;

		// Display the quick tip if necessary.
		ZafWindowObject *mouseObject = zafWindowManager->mouseObject;
		if (allowQuickSiblingDisplay && mouseObject && mouseObject->QuickTip() && elapsedTime < newHelpTipDelay)
		{
			// Set the mouse position.
			ZafEventStruct tEvent = event;
			Reset(displayTimerHandle, true, initialDelay * 10, &tEvent);

			// Display the quick tip.
			DisplayQuickTip();
		}
		else
		{
			// Indicate that we're trying to display the quick tip.
			processDisplay = true;
			quickTipVisible = false;
			allowQuickSiblingDisplay = false;

			// Set the timer and mouse position.
			ZafEventStruct tEvent = event;
			Reset(displayTimerHandle, true, initialDelay * 10, &tEvent);
		}
		}
		break;

	case N_MOUSE_LEAVE:
		{
		// Remove quick tip and update help object's text.
		RemoveHelpTip();

		// Indicate that no longer trying to display quick tip.
		processDisplay = false;
		}
		break;

	case DH_SET_HELP_OBJECT:
		SetHelpObject(event.windowObject);
		break;

	case DH_UPDATE_HELP_OBJECT:
		{
		ZafWindowObject *object = event.windowObject;
		ZafWindowObject *helpObject = zafWindowManager->helpObject;
		if (deviceState == D_ON && helpObject && (helpTipsType == ZAF_HELPTIPS_HELPOBJECT ||
			helpTipsType == ZAF_HELPTIPS_BOTH))
			helpObject->SetText(object ? object->HelpObjectTip() : replaceText);
		}	
		break;

	default:
		ccode = S_UNKNOWN;
		break;
	}

	// Return the control code.
	return (ccode);
}

void ZafHelpTips::Poll(void)
{
	if (DeviceState() == D_OFF || !processDisplay)
		return;
	
	// Don't process mouse messages if we are not the active application.
	if (!zafWindowManager->First()->Focus())
		return;

	// Get the next event (without destroying it) to see if we need to do anything.
	ZafEventStruct event;
	eventManager->Get(event, Q_NO_BLOCK | Q_BEGIN | Q_NO_DESTROY | Q_NO_POLL);
	
	// If the quick tip is currently displayed and a mouse click or keystroke 
	// occurred, remove the quick tip.
	if (event.osEvent.message == WM_LBUTTONDOWN ||
		event.osEvent.message == WM_MBUTTONDOWN ||
		event.osEvent.message == WM_RBUTTONDOWN ||
		event.osEvent.message == WM_NCLBUTTONDOWN ||
		event.osEvent.message == WM_NCMBUTTONDOWN ||
		event.osEvent.message == WM_NCRBUTTONDOWN ||
		event.InputType() == E_KEY)
	{
		if (quickTipVisible)
		{
			RemoveHelpTip();
			processDisplay = false;
		}
		else
			Reset(displayTimerHandle, false, 0, ZAF_NULLP(ZafEventStruct));
	}

	// If the quick tip is not displayed, but the mouse is moving over the
	// object and we're trying to display the quick tip, reset the timer and 
	// mouse location.
	else if (!quickTipVisible && processDisplay && event.osEvent.message == WM_MOUSEMOVE)
		Reset(displayTimerHandle, true, initialDelay * 10, &event);

	// If the quick tip is displayed and the mouse moves, update the time of the 
	// movement so can display on a sibling if the mouse moves to it quickly
	// enough.
	else if (quickTipVisible && event.osEvent.message == WM_MOUSEMOVE)
		lastTime = clock();

	// If the timer expired, display the quick tip.
	else if (!quickTipVisible && processDisplay && event.osEvent.message == WM_TIMER &&
		event.osEvent.wParam == displayTimerHandle)
		DisplayQuickTip();
}

void ZafHelpTips::RemoveHelpTip(void)
{
	// Restore the help object's text.
	ZafWindowObject *helpObject = zafWindowManager->helpObject;
	if (helpObject && (helpTipsType == ZAF_HELPTIPS_HELPOBJECT || helpTipsType == ZAF_HELPTIPS_BOTH))
	   	helpObject->SetText(replaceText);

	// Restore the screen image under the quick tip.
	if ((helpTipsType == ZAF_HELPTIPS_QUICKTIP || helpTipsType == ZAF_HELPTIPS_BOTH) &&
		quickTipVisible)
	{
		HDC hDC = GetDC(0);
		HDC bufferDC = CreateCompatibleDC(hDC);
		SelectObject(bufferDC, screenBuffer);
		BitBlt(hDC, quickTipRegion.left, quickTipRegion.top, quickTipRegion.Width(),
			quickTipRegion.Height(), bufferDC, 0, 0, SRCCOPY);
		DeleteDC(bufferDC);
		ReleaseDC(0, hDC);
		DeleteObject(screenBuffer);
	}

	// Indicate that the quick tip is not displayed.
	quickTipVisible = false;

	// Turn off the timer.
	Reset(displayTimerHandle, false, 0, ZAF_NULLP(ZafEventStruct));
}

void ZafHelpTips::Reset(OSTimerHandle &timerHandle, bool resetTimer, unsigned long interval, ZafEventStruct *event)
{
	// Kill the timer.
	if (timerHandle)
	{
		KillTimer(ZAF_NULLH(HWND), timerHandle);
		timerHandle = 0;
	}

	// Restart the timer and get the current mouse position.  We need to keep
	// track of the mouse position now, because the mouse won't be moved again
	// if the quick tip is to be displayed.
	if (resetTimer)
	{
		LPARAM lParam = event->osEvent.lParam;
#if defined(ZAF_WIN32)
		POINTS points = MAKEPOINTS(lParam);
		POINT point;
		point.x = points.x;
		point.y = points.y;
#else	
		POINT point = MAKEPOINT(lParam);
#endif
		ClientToScreen(event->osEvent.hwnd, &point);
		lastMousePosition.column = point.x;
		lastMousePosition.line = point.y;

		// Remember when this mouse movement occurred so can display a 
		// sibling's quick tip without delay if move quickly enough from object
		// to object.
		lastTime = clock();

		if (zafWindowManager->mouseObject)
			timerHandle = SetTimer(ZAF_NULLH(HWND), 0, (UINT)interval, NULL);
	}
}


