//	Zinc Application Framework - W_DSP.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include "w_app.hpp"
#include <z_dsp.hpp>
#include <z_cset.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>

#if !defined(MAKEWORD)
#	define MAKEWORD(a, b)      ((WORD)(((BYTE)(a)) | (((WORD)((BYTE)(b))) << 8)))
#endif

// Data used for initializing the pattern table. (see constructor)
static WORD zafSolidData[8] = { 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF };
static WORD zafInterleaveData[8] = { 0x5555, 0xAAAA, 0x5555, 0xAAAA, 0x5555, 0xAAAA, 0x5555, 0xAAAA };

// Data used for initializing the color table. (see constructor)
static COLORREF ZAF_FARDATA zafColors[] =
{
	RGB(0x00, 0x00, 0x00),	// 0  - ZAF_CLR_BLACK
	RGB(0x00, 0x00, 0x80),	// 1  - ZAF_CLR_BLUE
	RGB(0x00, 0x80, 0x00),	// 2  - ZAF_CLR_GREEN
	RGB(0x00, 0x80, 0x80),	// 3  - ZAF_CLR_CYAN
	RGB(0x80, 0x00, 0x00),	// 4  - ZAF_CLR_RED
	RGB(0x80, 0x00, 0x80),	// 5  - ZAF_CLR_MAGENTA
	RGB(0x80, 0x80, 0x00),	// 6  - ZAF_CLR_BROWN
	RGB(0xC0, 0xC0, 0xC0),	// 7  - ZAF_CLR_LIGHTGRAY
	RGB(0x80, 0x80, 0x80),	// 8  - ZAF_CLR_DARKGRAY
	RGB(0x00, 0x00, 0xFF),	// 9  - ZAF_CLR_LIGHTBLUE
	RGB(0x00, 0xFF, 0x00),	// 10 - ZAF_CLR_LIGHTGREEN
	RGB(0x00, 0xFF, 0xFF),	// 11 - ZAF_CLR_LIGHTCYAN
	RGB(0xFF, 0x00, 0x00),	// 12 - ZAF_CLR_LIGHTRED
	RGB(0xFF, 0x00, 0xFF),	// 13 - ZAF_CLR_LIGHTMAGENTA
	RGB(0xFF, 0xFF, 0x00),	// 14 - ZAF_CLR_YELLOW
	RGB(0xFF, 0xFF, 0xFF),	// 15 - ZAF_CLR_WHITE
	RGB(0xE0, 0xE0, 0xE0),	// 16 - COLOR_3DHILIGHT (Windows Only)
};

class ZafCache : public ZafList
{
public:
	ZafCache(int size,
		long (*CreateFunction)(long type, long value),
		void (*DestroyFunction)(long type, long value, long handle));
	virtual ~ZafCache(void);

	long Get(long type, long value);
	int Release(long handle);

private:
	int count;
	int size;
	long (*CreateFunction)(long type, long value);
	void (*DestroyFunction)(long type, long value, long handle);

	class CacheElement : public ZafElement
	{
	public:
		long handle;
		long type;
		long value;
		int useCount;

		CacheElement(long type, long value, long handle);
	};
};

ZafCache::ZafCache(int setSize,
	long (*setCreateFunction)(long type, long value),
	void (*setDestroyFunction)(long type, long value, long handle)) :
	ZafList(), size(setSize), CreateFunction(setCreateFunction),
	DestroyFunction(setDestroyFunction)
{
}

ZafCache::~ZafCache(void)
{
	for (CacheElement *element = (CacheElement *)First(); element; element = (CacheElement *)element->Next())
		DestroyFunction(element->type, element->value, element->handle);
}

long ZafCache::Get(long type, long value)
{
	// Find a matching cache element.
	CacheElement *element; 
	for (element = (CacheElement *)First(); element; element = (CacheElement *)element->Next())
		if (element->type == type && element->value == value)
		{
			// Move the element to the front of the list because it is
			// most recently used.
			if (element != first)
			{
				ZafList::Subtract(element);
				ZafList::Add(element, first);
			}

			element->useCount++;
			return element->handle;
		}

	// Create a cache element if one didn't already exist.
	long handle = CreateFunction(type, value);
	element = new CacheElement(type, value, handle);
	ZafList::Add(element, first);

	// Remove a cache element if the cache is full.
	if (++count >= size)
	{
		element = (CacheElement *)last;
		if (element->useCount <= 0)
		{
			ZafList::Subtract(element);
			DestroyFunction(element->type, element->value, element->handle);
			delete element;
			count--;
		}
	}

	return handle;
}

int ZafCache::Release(long handle)
{
	for (CacheElement *element = (CacheElement *)First(); element; element = (CacheElement *)element->Next())
		if (element->handle == handle)
		{
			element->useCount--;
			if (element->useCount == 0)
			{
				// Move unused elements to the end of the cache.
				CacheElement *positionElement = (CacheElement *)element->Next();
				while (positionElement && positionElement->useCount > 0)
					positionElement = (CacheElement *)positionElement->Next();

				if (positionElement && positionElement != element->Next())
				{
					ZafList::Subtract(element);
					ZafList::Add(element, positionElement);
				}
			}
			return (element->useCount);
		}

	return -1;
}

ZafCache::CacheElement::CacheElement(long setType, long setValue, long setHandle) :
	type(setType), value(setValue), handle(setHandle), useCount(1)
{
}

// GDI cache types.
const int ZAF_GDI_PEN		= 1;
const int ZAF_GDI_BRUSH		= 2;

long ZafGDICreate(long type, long value)
{
	int gdiType = LOBYTE(LOWORD(type));

	HGDIOBJ handle = 0;
	if (gdiType == ZAF_GDI_PEN)
	{
		int gdiWidth = HIBYTE(LOWORD(type));
		int gdiStyle = (gdiWidth > 0) ? HIWORD(type) : PS_NULL;
		COLORREF gdiColor = (COLORREF)value;

		handle = CreatePen(gdiStyle, gdiWidth, gdiColor);
	}
	else if (gdiType == ZAF_GDI_BRUSH)
	{
		HBITMAP hBitmap = (HBITMAP)value;

		handle = CreatePatternBrush(hBitmap);
	}

	return (long)handle;
}

void ZafGDIDestroy(long, long, long handle)
{
	DeleteObject((HGDIOBJ)handle);
}

static ZafCache zafGdiCache(16, ZafGDICreate, ZafGDIDestroy);

HBRUSH ZafDisplay::GetGdiCacheBrush(ZafPaletteStruct palette)
{
	return (HBRUSH)zafGdiCache.Get(ZAF_GDI_BRUSH, (long)patternTable[palette.fillPattern]);
}

HPEN ZafDisplay::GetGdiCachePen(ZafPaletteStruct palette, int width)
{
	COLORREF colorRef = (palette.colorForeground == ZAF_CLR_DEFAULT) ?
		palette.osPalette.colorForeground : colorTable[palette.colorForeground];
	OSLineStyle lineStyle = lineTable[palette.lineStyle];
	return (HPEN)zafGdiCache.Get(MAKELONG(MAKEWORD(ZAF_GDI_PEN, width), lineStyle), colorRef);
}

int ZafDisplay::ReleaseGdiCacheObject(HGDIOBJ hGdiObject)
{
	return (zafGdiCache.Release((long)hGdiObject));
}

// ----- ZafDisplay -----------------------------------------------

// Palette tables.
OSLineStyle ZAF_FARDATA ZafDisplay::lineTable[ZAF_MAXLINES];
OSColor ZAF_FARDATA ZafDisplay::colorTable[ZAF_MAXCOLORS];
OSMono ZAF_FARDATA ZafDisplay::monoTable[ZAF_MAXCOLORS];
OSFont ZAF_FARDATA ZafDisplay::fontTable[ZAF_MAXFONTS];
OSFillPattern ZAF_FARDATA ZafDisplay::patternTable[ZAF_MAXPATTERNS];
OSMode ZAF_FARDATA ZafDisplay::modeTable[ZAF_MAXMODES];

static ZafPaletteStruct ZAF_FARDATA defaultPalette =
{
	ZAF_LINE_SOLID,
	ZAF_PTN_SOLID_FILL,
	ZAF_CLR_BLACK,
	ZAF_CLR_WHITE,
	ZAF_MONO_BLACK,
	ZAF_MONO_WHITE,
	ZAF_FNT_DIALOG
};

ZafDisplay *zafDisplay = ZAF_NULLP(ZafDisplay);

ZafDisplay::ZafDisplay(int, char **) :
	hDC(0), hWnd(0), hPen(0), hBrush(0), virtualCount(0)
{
	// Initialize the global variable.
	if (!zafDisplay)
		zafDisplay = this;

	// Initialize tables.
	int index;
	for (index = 0; index < ZAF_MAXCOLORS; index++)
		colorTable[index] = 0xFFFFFFFFL;
	for (index = 0; index < ZAF_MAXFONTS; index++)
		fontTable[index] = ZAF_NULLH(OSFont);

	// Get a screen device context for initializing members.
	HDC screenDC = GetDC(0);

	// Initialize member data.
	columns = GetDeviceCaps(screenDC, HORZRES);
	lines = GetDeviceCaps(screenDC, VERTRES);
	pixelsPerInchX = GetDeviceCaps(screenDC, LOGPIXELSX);
	pixelsPerInchY = GetDeviceCaps(screenDC, LOGPIXELSY);
	miniNumeratorX = 1;
	miniDenominatorX = 10;
	miniNumeratorY = 1;
	miniDenominatorY = 10;
#if defined(ZAF_MSWINDOWS_3D)
//???
//	preSpace = 1;
//	postSpace = 1;
	preSpace = 2;
	postSpace = 2;
#else
	preSpace = 2;
	postSpace = 2;
#endif

	// Determine if the display driver is a palette device.
	paletteDevice = (GetDeviceCaps(screenDC, RASTERCAPS) & RC_PALETTE) ? true : false;

	// Allocate and initialize a LOGPALETTE structure.
	int numPaletteEntries = sizeof(zafColors) / sizeof(COLORREF);
	LOGPALETTE *logPalette = (LOGPALETTE *)new BYTE[sizeof(LOGPALETTE)
		+ (numPaletteEntries - 1) * sizeof(PALETTEENTRY)];
	logPalette->palVersion = 0x300;
	logPalette->palNumEntries = numPaletteEntries;

	// Map Zinc's reserved colors into the Windows default color palette.
	HPALETTE defaultPalette = (HPALETTE)GetStockObject(DEFAULT_PALETTE);
	for (int i = 0; i < numPaletteEntries; i++)
	{
		int index = GetNearestPaletteIndex(defaultPalette, zafColors[i]);

		// This code assumes a 256 color system palette with 10 static
		// colors at the beginning and 10 at the end.
		if (index > 9)
			index += 236;
		logPalette->palPalEntry[i].peRed = index;
		logPalette->palPalEntry[i].peGreen = 0;
		logPalette->palPalEntry[i].peBlue = 0;
		logPalette->palPalEntry[i].peFlags = PC_EXPLICIT;

		// If problems turn up with the above code, this code will work,
		// but less efficiently.
//		GetPaletteEntries(defaultPalette, index, 1, &logPalette->palPalEntry[i]);

		if (paletteDevice)
			colorTable[i] = PALETTEINDEX(i);
		else
			colorTable[i] = zafColors[i];
	}

	// Create a custom logical color palette for use in display operations.
	logicalPalette = CreatePalette(logPalette);

	// Clean up.
	delete []logPalette;

	//???Initialize monoTable here.

	// Get the text metrics of the system font.
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
	TEXTMETRICA textMetrics;
	HFONT oldFont = (HFONT)SelectObject(screenDC, GetStockObject(SYSTEM_FONT));
	GetTextMetricsA(screenDC, &textMetrics);
	SelectObject(screenDC, oldFont);
#else
	TEXTMETRIC textMetrics;
	HFONT oldFont = (HFONT)SelectObject(screenDC, GetStockObject(SYSTEM_FONT));
	GetTextMetrics(screenDC, &textMetrics);
	SelectObject(screenDC, oldFont);
#endif

	// Release screen device context.
	ReleaseDC(0, screenDC);

	// Initialize cell width and cell height.
	cellWidth = textMetrics.tmAveCharWidth + 1;
	cellHeight =
		textMetrics.tmHeight + 					// Text height
		textMetrics.tmInternalLeading + 		// Space above text
		textMetrics.tmExternalLeading + 		// Space below text
		ZAF_BORDER_WIDTH * 2 +	   				// border space
		preSpace +								// preSpace (above the border)
		postSpace;								// postSpace (below the border)

	// Initialize the line style table.
	lineTable[ZAF_LINE_SOLID] = PS_SOLID;
	lineTable[ZAF_LINE_DOTTED] = PS_DOT;

	// Initialize the pattern table.
	patternTable[ZAF_PTN_SOLID_FILL] = CreateBitmap(8, 8, 1, 1, zafSolidData);
	patternTable[ZAF_PTN_INTERLEAVE_FILL] = CreateBitmap(8, 8, 1, 1, zafInterleaveData);

	// Initialize the font table. (System, Dialog, Application, and Fixed fonts)
	fontTable[ZAF_FNT_SYSTEM] = (HFONT)GetStockObject(SYSTEM_FONT);
	fontTable[ZAF_FNT_APPLICATION] = (HFONT)GetStockObject(DEFAULT_GUI_FONT);
	fontTable[ZAF_FNT_FIXED] = (HFONT)GetStockObject(SYSTEM_FIXED_FONT);

	// Initialize the font table. (Small font)
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
	LOGFONTA	logSmallFont;
	SystemParametersInfoA(SPI_GETICONTITLELOGFONT, sizeof(LOGFONT), &logSmallFont, 0);
	fontTable[ZAF_FNT_DIALOG] = CreateFontIndirectA(&logSmallFont);
#else
	LOGFONT	logSmallFont;
	SystemParametersInfo(SPI_GETICONTITLELOGFONT, sizeof(LOGFONT), &logSmallFont, 0);
	fontTable[ZAF_FNT_DIALOG] = CreateFontIndirect(&logSmallFont);
#endif
	fontTable[ZAF_FNT_SMALL] = fontTable[ZAF_FNT_DIALOG];

	// Initialize the mix mode table.
	modeTable[ZAF_MODE_COPY].binary = R2_COPYPEN;
	modeTable[ZAF_MODE_COPY].ternary = SRCCOPY;
	modeTable[ZAF_MODE_XOR].binary = R2_XORPEN;
	modeTable[ZAF_MODE_XOR].ternary = SRCINVERT;
	mode = ZAF_MODE_COPY;

	// Initialize the Zinc palette.
	palette.lineStyle = ZAF_LINE_SOLID;
	palette.fillPattern = ZAF_PTN_SOLID_FILL;
	palette.colorForeground = ZAF_CLR_BLACK;
	palette.colorBackground = ZAF_CLR_WHITE;
	palette.monoForeground = ZAF_MONO_BLACK;
	palette.monoBackground = ZAF_MONO_WHITE;
	palette.font = ZAF_FNT_DIALOG;
	palette.osPalette.colorForeground = COLOR_WINDOWTEXT;
	palette.osPalette.colorBackground = COLOR_WINDOW;

	// Initialize the clip region.
	clipRegion.left = clipRegion.top = 0;
	clipRegion.right = columns - 1;
	clipRegion.bottom = lines - 1;
}

ZafDisplay::~ZafDisplay(void)
{
	// Clean up.
	DeleteObject(patternTable[ZAF_PTN_SOLID_FILL]);
	DeleteObject(patternTable[ZAF_PTN_INTERLEAVE_FILL]);
	DeleteObject(fontTable[ZAF_FNT_DIALOG]);
	DeleteObject(logicalPalette);
}

ZafLogicalColor ZafDisplay::AddColor(ZafLogicalColor logicalColor, ZafUInt8 red, ZafUInt8 green, ZafUInt8 blue)
{
	if (logicalColor == ZAF_CLR_NULL)
	{
		logicalColor = 0;
		while (logicalColor < ZAF_MAXCOLORS && colorTable[logicalColor] != 0xFFFFFFFFL)
			logicalColor++;

		if (logicalColor == ZAF_MAXCOLORS)
			return (ZAF_CLR_NULL);
	}

  	colorTable[logicalColor] = RGB(red, green, blue);

  	return (logicalColor);
}

ZafLogicalFont ZafDisplay::AddFont(ZafLogicalFont logicalFont, char *fontFamily,
	int pointSize, ZafFontWeight weight, ZafFontSlant slant)
{
	if (logicalFont == ZAF_FNT_NULL)
	{
		logicalFont = 0;
		while (logicalFont < ZAF_MAXFONTS && fontTable[logicalFont])
			logicalFont++;

		if (logicalFont == ZAF_MAXFONTS)
			return (ZAF_FNT_NULL);
	}
	else if (fontTable[logicalFont])
	{
		DeleteObject(fontTable[logicalFont]);
		fontTable[logicalFont] = ZAF_NULLH(OSFont);
	}

	int fnWeight;
	switch (weight)
	{
	case ZAF_FNT_WEIGHT_BOLD: fnWeight = FW_BOLD; break;
	default: fnWeight = FW_DONTCARE;
	}

	bool fbItalic = (slant == ZAF_FNT_SLANT_ITALIC) ? true : false;

	HFONT hFont = CreateFont(-pointSize, 0, 0, 0, fnWeight, fbItalic,
		0, 0, DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS,
		DEFAULT_QUALITY, DEFAULT_PITCH | FF_DONTCARE, fontFamily);

	if (hFont)
	{
		fontTable[logicalFont] = hFont;
		return (logicalFont);
	}

	return (ZAF_FNT_NULL);
}

ZafError ZafDisplay::BeginDraw(OSDisplayContext tHDC, OSDrawContext tHWnd,
	const ZafRegionStruct &, const ZafRegionStruct &)
{
	if (virtualCount++ == 0)
	{
		if (tHDC)
			SetDisplayContext(tHDC);
		else if (tHWnd == ID_ZAF_SCREEN)
		{
			hWnd = tHWnd;
			SetDisplayContext(GetDC(0));
		}
		else if (tHWnd)
		{
			hWnd = tHWnd;
			SetDisplayContext(GetDC(hWnd));
		}
		SetBkMode(hDC, TRANSPARENT);
	}

	return ZAF_ERROR_NONE;
}

ZafError ZafDisplay::EndDraw(void)
{
	if (--virtualCount == 0)
	{
		palette = defaultPalette;
		mode = ZAF_MODE_COPY;
		HDC oldDC = RestoreDisplayContext();

		if (hWnd)
		{
			if (hWnd == ID_ZAF_SCREEN)
				ReleaseDC(0, oldDC);
			else
				ReleaseDC(hWnd, oldDC);
		}
		hWnd = 0;
	}

	return ZAF_ERROR_NONE;
}

// --- Bitmap ---------------------------------------------------------------

ZafError ZafDisplay::Bitmap(int column, int line, ZafBitmapStruct &bitmap)
{
	// Create the bitmap handle(s) if necessary.
	if (!bitmap.handle && bitmap.array)
		ConvertToOSBitmap(bitmap);

	if (mode == ZAF_MODE_COPY && bitmap.mask)
	{
		// Transparancy is achieved using the "Black Source" masking method.
		// Drawing is performed flicker free by copying the screen contents
		// where the bitmap is to be drawn into a memory (buffer) bitmap,
		// perforiming the "bitblt" operations on this buffer bitmap, and
		// then copying the buffer bitmap back to the screen.

		// Create memory DC's.
		HDC bufferDC = CreateCompatibleDC(hDC);
		HDC bitmapDC = CreateCompatibleDC(hDC);

		// Create the buffer bitmap and select it into the buffer DC.
		HBITMAP bufferBitmap = CreateCompatibleBitmap(hDC, bitmap.width, bitmap.height);
		SelectObject(bufferDC, bufferBitmap);

		// Copy screen contents into the buffer bitmap.
		BitBlt(bufferDC, 0, 0, bitmap.width, bitmap.height, hDC, column, line, SRCCOPY);

		// Select the mask bitmap into the bitmap DC.
		SelectObject(bitmapDC, bitmap.mask);

		// And the mask bitmap onto the buffer bitmap.
		SetTextColor(bufferDC, RGB(0,0,0)); // Mask bitmap 'on' color.
		SetBkColor(bufferDC, RGB(0xFF,0xFF,0xFF)); // Mask bitmap 'off' color.
		BitBlt(bufferDC, 0, 0, bitmap.width, bitmap.height, bitmapDC, 0, 0, SRCAND);

		// Select the color bitmap into the bitmap DC.
		SelectObject(bitmapDC, bitmap.handle);

		// XOR the color bitmap onto  the buffer bitmap.
		BitBlt(bufferDC, 0, 0, bitmap.width, bitmap.height, bitmapDC, 0, 0, SRCINVERT);

		// Copy the buffer bitmap back to the screen.
		BitBlt(hDC, column, line, bitmap.width, bitmap.height, bufferDC, 0, 0, SRCCOPY);

		// Clean up.
		DeleteDC(bitmapDC);
		DeleteDC(bufferDC);
		DeleteObject(bufferBitmap);
	}
	else // Bitmap does not contain transparency.
	{
		// Create a memory DC.
		HDC memDC = CreateCompatibleDC(hDC);

		// Select the bitmap into the memory DC.
		SelectObject(memDC, bitmap.handle);

		// Copy the bitmap onto the screen.
		BitBlt(hDC, column, line, bitmap.width, bitmap.height, memDC, 0, 0, modeTable[mode].ternary);

		// Clean up.
		DeleteDC(memDC);
	}

	return ZAF_ERROR_NONE;
}

ZafError ZafDisplay::ConvertToOSBitmap(ZafBitmapStruct &bitmap)
{
	// Allocate local data to optimize speed.
	int width = bitmap.width, height = bitmap.height;
	ZafLogicalColor *array = bitmap.array;

	// Compute scan line size.
	int bitmapBytesPerLine = width;
	int maskBytesPerLine = (width + 7) / 8;

	// Color bitmap scan lines must be padded to end on "LONG" boundaries.
	if (bitmapBytesPerLine % sizeof(LONG))
		bitmapBytesPerLine += sizeof(LONG) - bitmapBytesPerLine % sizeof(LONG);

	// Monochrome bitmap scan lines must be padded to end on "word" boundaries.
	if (maskBytesPerLine % sizeof(WORD))
		maskBytesPerLine += sizeof(WORD) - maskBytesPerLine % sizeof(WORD);

	// Allocate bitmap initialization data.
	BYTE *bitmapData = new BYTE[bitmapBytesPerLine * height];
	BYTE *maskData = new BYTE[maskBytesPerLine * height];

	// Initialize bitmap data.
	memset(maskData, 0, maskBytesPerLine * height);
	bool transparent = false;
	int zafIndex = 0, colors = 2;
	for (int y = 0; y < height; y++)
	{
		// Compute vertical index/mask values.
		int bitmapYIndex = (height - y - 1) * bitmapBytesPerLine;
		int maskYIndex = y * maskBytesPerLine;
		BYTE maskBit = 0x80;

		for (int x = 0; x < width; x++)
		{
			if (array[zafIndex] == ZAF_CLR_BACKGROUND)
			{
				// Initialize "Black Source" and "transparent" mask values.
				transparent = true;
				bitmapData[bitmapYIndex + x] = ZAF_CLR_BLACK;
				maskData[maskYIndex + x / 8] |= maskBit;
			}
			else
			{
				// Initialize "opaque" color values.				
				bitmapData[bitmapYIndex + x] = array[zafIndex];
				if (array[zafIndex] >= colors)
					colors = array[zafIndex] + 1;
			}

			// Update index/mask values.
			zafIndex++;
			maskBit >>= 1;
			if (!maskBit)
				maskBit = 0x80;
		}
	}

	// Allocate BITMAPINFO structure.
	BITMAPINFO *bitmapInfo = (BITMAPINFO *)new char[sizeof(BITMAPINFO)
		+ (colors - 1) * (paletteDevice ? sizeof(WORD) : sizeof(RGBQUAD))];

	// Initialize HEADER information.
	bitmapInfo->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	bitmapInfo->bmiHeader.biWidth = width;
	bitmapInfo->bmiHeader.biHeight = height;
	bitmapInfo->bmiHeader.biPlanes = 1;
	bitmapInfo->bmiHeader.biBitCount = 8;
	bitmapInfo->bmiHeader.biCompression = BI_RGB;
	bitmapInfo->bmiHeader.biSizeImage = 0;
	bitmapInfo->bmiHeader.biXPelsPerMeter = 0;
	bitmapInfo->bmiHeader.biYPelsPerMeter = 0;
	bitmapInfo->bmiHeader.biClrUsed = colors;
	bitmapInfo->bmiHeader.biClrImportant = colors;

	// Initialize the color table.
	if (paletteDevice)
		for (int i = 0; i < colors; i++)
			((WORD *)bitmapInfo->bmiColors)[i] = LOWORD(colorTable[i]);
	else
		for (int i = 0; i < colors; i++)
		{
			bitmapInfo->bmiColors[i].rgbRed = GetRValue(colorTable[i]);
			bitmapInfo->bmiColors[i].rgbGreen = GetGValue(colorTable[i]);
			bitmapInfo->bmiColors[i].rgbBlue = GetBValue(colorTable[i]);
		}

	// Get a device context for the convertsion.
	HDC convertDC;
	HPALETTE savePalette = 0;
	if (hDC)
		convertDC = hDC;
	else
	{
		// Get the device context of the screen.
		convertDC = GetDC(0);
		savePalette = SelectPalette(convertDC, logicalPalette, FALSE);
	}

	// Realize logicalPalette.
	RealizePalette(convertDC);


	// Create the Bitmap.
	bitmap.handle = CreateDIBitmap(convertDC, &bitmapInfo->bmiHeader,
		CBM_INIT, bitmapData, bitmapInfo, paletteDevice ? DIB_PAL_COLORS : DIB_RGB_COLORS);

	// Release the conversion device context.
	if (!hDC)
	{
		SelectPalette(convertDC, savePalette, FALSE);
		ReleaseDC(0, convertDC);
	}

	// Create the "transpanency" mask.
	if (transparent)
		bitmap.mask = CreateBitmap(width, height, 1, 1, maskData);

	// Clean up.
	delete []bitmapInfo;
	delete []bitmapData;
	delete []maskData;

//???	// Free up the ZAF icon information.
//???	if (!bitmap.staticArray)
//???	{
//???		delete []bitmap.array;
//???		bitmap.array = ZAF_NULLP(ZafLogicalColor);
//???	}

	return ZAF_ERROR_NONE;
}

ZafError ZafDisplay::ConvertToZafBitmap(ZafBitmapStruct &bitmap)
{
	// Make sure the OS bitmap exists.
	if (!bitmap.handle)
		return ZAF_ERROR_INVALID_SOURCE;

	// Get width, height, and color format of the bitmap.
	BITMAP logBitmap;
	GetObject(bitmap.handle, sizeof(logBitmap), &logBitmap);
	bitmap.width = logBitmap.bmWidth;
	bitmap.height = logBitmap.bmHeight;

	// Initialize BITMAPINFO structure.
	BITMAPINFO *bitmapInfo = (BITMAPINFO *)new BYTE[sizeof(BITMAPINFO) +
		255 * sizeof(WORD)];
	bitmapInfo->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	bitmapInfo->bmiHeader.biWidth = bitmap.width;
	bitmapInfo->bmiHeader.biHeight = bitmap.height;
	bitmapInfo->bmiHeader.biPlanes = 1;
	bitmapInfo->bmiHeader.biBitCount = 8;
	bitmapInfo->bmiHeader.biCompression = BI_RGB;
	bitmapInfo->bmiHeader.biSizeImage = 0;
	bitmapInfo->bmiHeader.biXPelsPerMeter = 0;
	bitmapInfo->bmiHeader.biYPelsPerMeter = 0;
	bitmapInfo->bmiHeader.biClrUsed = 0;
	bitmapInfo->bmiHeader.biClrImportant = 0;

	// Get mask data.
	int maskBytesPerLine;
	BYTE *maskData = ZAF_NULLP(BYTE);
	if (bitmap.mask)
	{
		// Allocate mask bitmap pixel data.
		maskBytesPerLine = (bitmap.width + 7) / 8;
		if (maskBytesPerLine % sizeof(LONG))
			maskBytesPerLine += sizeof(LONG) - maskBytesPerLine % sizeof(LONG);
		maskData = new BYTE[maskBytesPerLine * bitmap.height];

		// Retrieve mask bitmap pixel data.
		bitmapInfo->bmiHeader.biBitCount = 1;
		GetDIBits(hDC, bitmap.mask, 0, bitmap.height, maskData, bitmapInfo,
			DIB_RGB_COLORS);
	}

	// Allocate color bitmap pixel data.
	int bitmapBytesPerLine = bitmap.width;
	if (bitmapBytesPerLine % sizeof(LONG))
		bitmapBytesPerLine += sizeof(LONG) - bitmapBytesPerLine % sizeof(LONG);
	BYTE *bitmapData = new BYTE[bitmapBytesPerLine * bitmap.height];

	// Get the color bitmap pixel data and RGB color table.
	bitmapInfo->bmiHeader.biBitCount = 8;
	GetDIBits(hDC, bitmap.handle, 0, bitmap.height, bitmapData, bitmapInfo,
		DIB_PAL_COLORS);

	// Allocate the ZAF array.
	if (bitmap.array && !bitmap.staticArray)
		delete []bitmap.array;
	bitmap.array = new ZafLogicalColor[bitmap.width * bitmap.height];
	bitmap.staticArray = false;

	// Fill the ZAF array.
	// (Optimized for speed.)
	if (bitmap.mask)
	{
		int zafIndex = 0;
		for (int y = bitmap.height - 1; y >= 0; y--)
		{
			// Initialize mask value.
			BYTE maskBit = 0x80;
			for (int x = 0; x < bitmap.width; x++)
			{
				// Test for transparency.
				if (maskData[y * maskBytesPerLine + x / 8] & maskBit)
					bitmap.array[zafIndex] = ZAF_CLR_BACKGROUND;
				else
					bitmap.array[zafIndex] = ((WORD *)bitmapInfo->bmiColors)[bitmapData[y * bitmapBytesPerLine + x]];

				// Update index/mask values.
				zafIndex++;
				maskBit >>= 1;
				if (!maskBit)
					maskBit = 0x80;
			}
		}
	}
	else
	{
		int zafIndex = 0;
		for (int y = bitmap.height - 1; y >= 0; y--)
			for (int x = 0; x < bitmap.width; x++)
				bitmap.array[zafIndex++] = ((WORD *)bitmapInfo->bmiColors)[bitmapData[y * bitmapBytesPerLine + x]];
	}

	// Clean up.
	delete []bitmapInfo;
	delete []bitmapData;
	if (maskData)
		delete []maskData;

	return ZAF_ERROR_NONE;
}

ZafError ZafDisplay::DestroyOSBitmap(ZafBitmapStruct &bitmap)
{
	// Destroy the bitmap handle and mask.
	if (!bitmap.StaticHandle())
	{
		if (bitmap.handle)
			DeleteObject(bitmap.handle);
		if (bitmap.mask)
			DeleteObject(bitmap.mask);
	}
	bitmap.handle = bitmap.mask = 0;
	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::DestroyZafBitmap(ZafBitmapStruct &bitmap)
{
	// Destroy the bitmap array.
	if (bitmap.array && !bitmap.StaticArray())
		delete []bitmap.array;
	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::InitializeOSBitmap(ZafBitmapStruct &bitmap)
{
	bitmap.handle = 0;
	bitmap.mask = 0;
	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::ResetOSBitmap(ZafBitmapStruct &bitmap, const ZafBitmapStruct &copy)
{
	bitmap.handle = copy.handle;
	bitmap.mask = copy.mask;
	return (ZAF_ERROR_NONE);
}

// --- Icon -----------------------------------------------------------------

ZafError ZafDisplay::Icon(int column, int line, ZafIconStruct &icon)
{
	// Create an icon handle if necessary.
	if (!icon.handle && icon.array)
		ConvertToOSIcon(icon);

	// Draw the icon.
	DrawIcon(hDC, column, line, icon.handle);

	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::ConvertToOSIcon(ZafIconStruct &icon)
{
	// Create and convert a bitmap struct for use in the icon conversion.
	ZafBitmapStruct bitmap;
	bitmap.width = icon.width;
	bitmap.height = icon.height;
	bitmap.array = icon.array;
	bitmap.staticArray = true;
	bitmap.handle = 0;
	bitmap.mask = 0;
	ConvertToOSBitmap(bitmap);

#if defined(ZAF_WIN32)
	// Initialize an ICONINFO structure.
	ICONINFO iconInfo;
	iconInfo.fIcon = TRUE;
	iconInfo.xHotspot = iconInfo.yHotspot = 0;
	if (!bitmap.mask)
	{
		// Compute scan line size.
		int maskBytesPerLine = (icon.width + 7) / 8;
		maskBytesPerLine += maskBytesPerLine % sizeof(WORD);

		// Create a mask bitmap with no transparancy.
		BYTE *maskData = new BYTE[maskBytesPerLine * icon.height];
		memset(maskData, 0, maskBytesPerLine * icon.height);
		iconInfo.hbmMask = CreateBitmap(icon.width, icon.height, 1, 1, maskData);
		delete []maskData;
	}
	else
		iconInfo.hbmMask = bitmap.mask;
	iconInfo.hbmColor = bitmap.handle;

	// Create Icon.
	icon.handle = CreateIconIndirect(&iconInfo);
#else
	// Get color data from the color bitmap.
	BITMAP colorBitmap;
	GetObject(bitmap.handle, sizeof(colorBitmap), &colorBitmap);
	int colorDataSize = colorBitmap.bmWidthBytes * colorBitmap.bmHeight;
	BYTE *colorData = new BYTE[colorDataSize];
	GetBitmapBits(bitmap.handle, colorDataSize, colorData);

	// Get the mask data.
	BYTE *maskData;
	if (bitmap.mask)
	{
		// Get mask data from the mask bitmap.
		BITMAP maskBitmap;
		GetObject(bitmap.mask, sizeof(maskBitmap), &maskBitmap);
		int maskDataSize = maskBitmap.bmWidthBytes * maskBitmap.bmHeight;
		maskData = new BYTE[maskDataSize];
		GetBitmapBits(bitmap.mask, maskDataSize, maskData);
	}
	else
	{
		// Initialize mask data with no transparancy.
		int maskBytesPerLine = (icon.width + 7) / 8;
		maskBytesPerLine += maskBytesPerLine % sizeof(WORD);
		int maskDataSize = maskBytesPerLine * icon.height;
		maskData = new BYTE[maskDataSize];
		memset(maskData, 0, maskDataSize);
	}

	// Create the icon handle.
	icon.handle = CreateIcon(ZafMSWindowsApp::hInstance, icon.width, icon.height,
		1, colorBitmap.bmBitsPixel, maskData, colorData);

	// Clean up.
	delete []colorData;
	delete []maskData;
#endif

	// Clean up bitmap handles.
	if (bitmap.handle)
		DeleteObject(bitmap.handle);
	if (bitmap.mask)
		DeleteObject(bitmap.mask);

	// Free up the ZAF icon information.
	if (!icon.staticArray)
	{
		delete []icon.array;
		icon.array = ZAF_NULLP(ZafLogicalColor);
	}

	return ZAF_ERROR_NONE;
}

ZafError ZafDisplay::ConvertToZafIcon(ZafIconStruct &icon)
{
	if (!icon.handle)
		return ZAF_ERROR_INVALID_SOURCE;

	// Create and intialize a bitmap struct for use in the icon conversion.
	ZafBitmapStruct bitmap;
	bitmap.array = ZAF_NULLP(ZafLogicalColor);
	bitmap.staticArray = false;
	bitmap.staticHandle = true;

#if defined(ZAF_WIN32)
	// Get ICONINFO from the icon handle.
	ICONINFO iconInfo;
	GetIconInfo(icon.handle, &iconInfo);

	// Get bitmap handles from the icon info.
	bitmap.handle = iconInfo.hbmColor;
	bitmap.mask = iconInfo.hbmMask;

	// Convert the ZAF bitap.
	ConvertToZafBitmap(bitmap);
#else
	// Create color bitmap and mask bitmap handles from the icon handle.
	icon.width = GetSystemMetrics(SM_CXICON);
	icon.height = GetSystemMetrics(SM_CYICON);

	// Create 2 memory device contexts.
	HDC memDC1 = CreateCompatibleDC(hDC);
	HDC memDC2 = CreateCompatibleDC(hDC);

	// Create a color bitmap.
	bitmap.handle = CreateCompatibleBitmap(hDC, icon.width, icon.height);

	// Select the color bitmap into memDC1;
	SelectObject(memDC1, bitmap.handle);

	// Draw the icon on to the color bitmap with black in transparent areas.
	BitBlt(memDC1, 0, 0, icon.width, icon.height, 0, 0, 0, BLACKNESS);
	DrawIcon(memDC1, 0, 0, icon.handle);

	// Create two mono (mask) bitmaps.
	bitmap.mask = CreateBitmap(icon.width, icon.height, 1, 1, 0);
	HBITMAP tempBitmap = CreateBitmap(icon.width, icon.height, 1, 1, 0);

	// Select the mask bitmaps into memDC1, and memDC2.
	SelectObject(memDC1, bitmap.mask);
	SelectObject(memDC2, tempBitmap);

	// Draw the icon on to the mask bitmaps so that when finished
	// 'bitmap.mask' has pixels set only in transparent areas.
	BitBlt(memDC1, 0, 0, icon.width, icon.height, 0, 0, 0, BLACKNESS);
	DrawIcon(memDC1, 0, 0, icon.handle);
	BitBlt(memDC2, 0, 0, icon.width, icon.height, 0, 0, 0, WHITENESS);
	DrawIcon(memDC2, 0, 0, icon.handle);
	BitBlt(memDC1, 0, 0, icon.width, icon.height, memDC2, 0, 0, SRCERASE);

	// Convert the bitmap.
	ConvertToZafBitmap(bitmap);

	// Clean up.
	DeleteDC(memDC1);
	DeleteDC(memDC2);
	DeleteObject(bitmap.handle);
	DeleteObject(bitmap.mask);
	DeleteObject(tempBitmap);
#endif

	// Copy the bitmap information.
	icon.width = bitmap.width;
	icon.height = bitmap.height;
	if (icon.array && !icon.staticArray)
		delete []icon.array;
	icon.array = bitmap.array;
	icon.staticArray = false;

	return ZAF_ERROR_NONE;
}

ZafError ZafDisplay::DestroyOSIcon(ZafIconStruct &icon)
{
	// Destroy the handle.
	if (icon.handle && !icon.StaticHandle())
		DestroyIcon(icon.handle);
	icon.handle = 0;
	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::DestroyZafIcon(ZafIconStruct &icon)
{
	// Destroy the icon array.
	if (icon.array && !icon.StaticArray())
		delete []icon.array;
	icon.array = 0;
	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::InitializeOSIcon(ZafIconStruct &icon)
{
	icon.handle = 0;
	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::ResetOSIcon(ZafIconStruct &icon, const ZafIconStruct &copy)
{
	icon.handle = copy.handle;
	return (ZAF_ERROR_NONE);
}

// --- Mouse -----------------------------------------------------------------

//??? Not implemented yet.
ZafError ZafDisplay::Mouse(int, int, ZafMouseStruct &)
{
	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::ConvertToOSMouse(ZafMouseStruct &mouse)
{
//???#if defined(ZAF_WIN32)
//???	// Create and convert a bitmap struct for use in the icon conversion.
//???	ZafBitmapStruct bitmap;
//???	bitmap.width = mouse.width;
//???	bitmap.height = mouse.height;
//???	bitmap.array = mouse.array;
//???	bitmap.staticArray = true;
//???	bitmap.handle = 0;
//???	bitmap.mask = 0;
//???	ConvertToOSBitmap(bitmap);
//???
//???	// Initialize an ICONINFO structure.
//???	ICONINFO iconInfo;
//???	iconInfo.fIcon = FALSE;
//???	iconInfo.xHotspot =	mouse.hotSpotX;
//???	iconInfo.yHotspot = mouse.hotSpotY;
//???	if (!bitmap.mask)
//???	{
//???		// Compute scan line size.
//???		int maskBytesPerLine = (mouse.width + 7) / 8;
//???		maskBytesPerLine += maskBytesPerLine % sizeof(WORD);
//???
//???		// Create a mask bitmap with no transparancy.
//???		BYTE *maskData = new BYTE[maskBytesPerLine * mouse.height];
//???		memset(maskData, 0, maskBytesPerLine * mouse.height);
//???		iconInfo.hbmMask = CreateBitmap(mouse.width, mouse.height, 1, 1, maskData);
//???		delete []maskData;
//???	}
//???	else
//???		iconInfo.hbmMask = bitmap.mask;
//???	iconInfo.hbmColor = bitmap.handle;
//???
//???	// Create Icon.
//???	mouse.cursor = CreateIconIndirect(&iconInfo);
//???#else
	int startRow = (32 - mouse.height) / 2;
	int endRow = startRow + mouse.height - 1;
	int startColumn = (32 - mouse.width) / 2;
	int endColumn = startColumn + mouse.width - 1;

	ZafUInt8 andData[128], xorData[128];
	int index = 0;
	for (int row = 0; row < 32; row++)
	{
		if (startRow <= row && row <= endRow)
		{
			int column = 0;
			for (int byte = 0; byte < 4; byte++)
			{
				andData[row * 4 + byte] = xorData[row * 4 + byte] = 0;
				ZafUInt8 bitmask = 0x80;
				for (int bit = 0; bit < 8; bit++)
				{
					if (startColumn <= column && column <= endColumn)
					{
						ZafLogicalColor color = mouse.array[index++];
						if (color == ZAF_CLR_BACKGROUND)
							andData[row * 4 + byte] |= bitmask;
						else if (color != ZAF_CLR_BLACK)
							xorData[row * 4 + byte] |= bitmask;
					}
					else
						andData[row * 4 + byte] |= bitmask;

					column++;
					bitmask = bitmask >> 1;
				}
			}
		}
		else
		{
			*((ZafUInt32 *)&andData[row * 4]) = 0xFFFFFFFFL;
			*((ZafUInt32 *)&xorData[row * 4]) = 0L;
		}
	}

	mouse.cursor = CreateCursor(ZafMSWindowsApp::hInstance,
		startColumn + mouse.hotSpotX, startRow + mouse.hotSpotY,
		32, 32, andData, xorData);
//???#endif
	return (ZAF_ERROR_NONE);
}

//??? Not implemented yet.
ZafError ZafDisplay::ConvertToZafMouse(ZafMouseStruct &)
{
	return (ZAF_ERROR_NONE);
}

//??? Not implemented yet.
ZafError ZafDisplay::DestroyOSMouse(ZafMouseStruct &mouseStruct)
{
	if (mouseStruct.cursor && !mouseStruct.staticHandle)
		DestroyCursor(mouseStruct.cursor);
	return (ZAF_ERROR_NONE);
}

//??? Not implemented yet.
ZafError ZafDisplay::DestroyZafMouse(ZafMouseStruct &)
{
	return (ZAF_ERROR_NONE);
}

//??? Not implemented yet.
ZafError ZafDisplay::InitializeOSMouse(ZafMouseStruct &)
{
	return (ZAF_ERROR_NONE);
}

//??? Not implemented yet.
ZafError ZafDisplay::ResetOSMouse(ZafMouseStruct &, const ZafMouseStruct &)
{
	return (ZAF_ERROR_NONE);
}

// --- Text -----------------------------------------------------------------

ZafError ZafDisplay::Text(int left, int top, int right, int bottom,
	const ZafIChar *text, int length,
	ZafHzJustify hzJustify,
	ZafVtJustify vtJustify,
	int hotKeyIndex, bool fill)
{
	if (!text || !*text)
		return ZAF_ERROR_NONE;

	// Set background mode.
	if (fill)
		SetBkMode(hDC, OPAQUE);

	// Set format flags for Windows API.
	const ZafIChar *drawText;
	ZafIChar *hotKeyText = ZAF_NULLP(ZafIChar);
	UINT wFormat = DT_SINGLELINE | DT_NOCLIP;
	if (hotKeyIndex > -1 && hotKeyIndex < strlen(text))
	{
		hotKeyText = new ZafIChar[strlen(text) + 2];
		strncpy(hotKeyText, text, hotKeyIndex);
		hotKeyText[hotKeyIndex] = '&';
		strcpy(&hotKeyText[hotKeyIndex + 1], &text[hotKeyIndex]);
		drawText = hotKeyText;
	}
	else
	{
		drawText = text;
		wFormat |= DT_NOPREFIX;
	}

	// Set horizontal justification flags.
	if (hzJustify == ZAF_HZ_CENTER)
		wFormat |= DT_CENTER;
	else if (hzJustify == ZAF_HZ_RIGHT)
		wFormat |= DT_RIGHT;
	else
		wFormat |= DT_LEFT;

	// Set vertical justification flags.
	if (vtJustify == ZAF_VT_TOP)
		wFormat |= DT_TOP;
	else if (vtJustify == ZAF_VT_BOTTOM)
		wFormat |= DT_BOTTOM;
	else
		wFormat |= DT_VCENTER;

	if (mode != ZAF_MODE_COPY)
	{
		// Create a memory device context (DC) with an associated bitmap.
		HDC memDC = CreateCompatibleDC(hDC);
		HBITMAP textBitmap = CreateCompatibleBitmap(hDC, right - left + 1, bottom - top + 1);
		SelectObject(memDC, textBitmap);

		// Match the memory DC attributes with the current DC.
		SetTextColor(memDC, GetTextColor(hDC));
		SetBkColor(memDC, GetBkColor(hDC));
		SetBkMode(memDC, GetBkMode(hDC));
		SelectObject(memDC, fontTable[palette.font]);

		// Initialize the memory bitmap with black.  (Black has no effect in
		// XOR operations.
		BitBlt(memDC, 0, 0, right - left + 1, bottom - top + 1, 0, 0, 0, BLACKNESS);

		// Draw the text to the memory bitmap.
		RECT rect = { 0, 0, right - left + 1, bottom - top + 1 };
		if (ZafMSWindowsApp::convertText)
		{
			char *osText = zafCodeSet->ConvertToOSString(drawText);
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
			DrawTextA(memDC, osText, length, &rect, wFormat);
#else
			DrawText(memDC, osText, length, &rect, wFormat);
#endif
			delete []osText;
		}
#if defined(ZAF_WIN32) || !defined(ZAF_UNICODE)
		else
			DrawText(memDC, drawText, length, &rect, wFormat);
#endif

		// XOR the memory bitmap onto the screen.
		BitBlt(hDC, left, top, right - left + 1, bottom - top + 1, memDC, 0, 0, modeTable[mode].ternary);

		// Clean up.
		DeleteDC(memDC);
		DeleteObject(textBitmap);
	}
	else
	{
		// Draw the text.
		RECT rect = { left, top, right + 1, bottom + 1 };
		if (ZafMSWindowsApp::convertText)
		{
			char *osText = zafCodeSet->ConvertToOSString(drawText);
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
			DrawTextA(hDC, osText, length, &rect, wFormat);
#else
			DrawText(hDC, osText, length, &rect, wFormat);
#endif
			delete []osText;
		}
#if defined(ZAF_WIN32) || !defined(ZAF_UNICODE)
		else
			DrawText(hDC, drawText, length, &rect, wFormat);
#endif
	}

	if (hotKeyText)
		delete []hotKeyText;

	// Restore the background mode.
	if (fill)
		SetBkMode(hDC, TRANSPARENT);

	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::Text(int left, int top, const ZafIChar *text, int length,
	int hotKeyIndex, bool fill)
{
	// Get the text size.
	ZafRegionStruct textSize = TextSize(text, length);
	ZafRegionStruct textRegion;

	// Set the text region.
	textRegion.left = left;
	textRegion.right = left + textSize.right - 1;
	textRegion.top = top;
	textRegion.bottom = top + textSize.bottom - 1;

	// Draw the text.
	return Text(textRegion, text, length, ZAF_HZ_LEFT, ZAF_VT_TOP, hotKeyIndex, fill);
}

ZafRegionStruct ZafDisplay::TextSize(const ZafIChar *text, int length)
{
//???Regions?
//	ZafRegionStruct textRegion = { 1, 1, 0, 0 };
	ZafRegionStruct textRegion;
	textRegion.left = textRegion.top = 1;
	textRegion.right = textRegion.bottom = 0;
	if (!text || !*text)
		return textRegion;

	// Get the text size.
	SIZE size;
	if (ZafMSWindowsApp::convertText)
	{
		char *osSizeText = zafCodeSet->ConvertToOSString(text);
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
		GetTextExtentPointA(hDC, osSizeText, length >= 0 ? length : strlen(osSizeText), &size);
#else
		GetTextExtentPoint(hDC, osSizeText, length >= 0 ? length : strlen(osSizeText), &size);
#endif
		delete []osSizeText;
	}
#if defined(ZAF_WIN32) || !defined(ZAF_UNICODE)
	else
		GetTextExtentPoint(hDC, text, length >= 0 ? length : strlen(text), &size);
#endif

	// Return a bounding rectangle for the text.
	textRegion.left = textRegion.top = 1;
	textRegion.right = size.cx;
	textRegion.bottom = size.cy;
	return (textRegion);
}


// --- Primitives -----------------------------------------------------------

ZafError ZafDisplay::Ellipse(int left, int top, int right, int bottom,
		int startAngle, int endAngle, int width, bool fill)
{
	// Pseudo-sine data.
	const int pSin[] = { 0, 871, 1736, 2588, 3420, 4226, 5000, 5735, 6427,
		7071, 7660, 8191, 8660, 9063, 9396, 9659, 9848, 9961, 10000,
		9961, 9848, 9659, 9396, 9063, 8660, 8191, 7660, 7071,
		6427, 5735, 4999, 4226, 3420, 2588, 1736, 871, 0,
		-871, -1736, -2588, -3420, -4226, -5000, -5735, -6427, -7071,
		-7660, -8191, -8660, -9063, -9396, -9659, -9848, -9961, -10000,
		-9961, -9848, -9659, -9396, -9063, -8660, -8191, -7660, -7071,
		-6427, -5735, -4999, -4226, -3420, -2588, -1736, -871, 0 };

	// Pseudo-cosine data.
	const int pCos[] = { 10000, 9961, 9848, 9659, 9396, 9063, 8660, 8191, 7660,
		7071, 6427, 5735, 4999, 4226, 3420, 2588, 1736, 871, 0,
		-871, -1736, -2588, -3420, -4226, -5000, -5735, -6427, -7071,
		-7660, -8191, -8660, -9063, -9396, -9659, -9848, -9961, -10000,
		-9961, -9848, -9659, -9396, -9063, -8660, -8191, -7660, -7071,
		-6427, -5735, -4999, -4226, -3420, -2588, -1736, -871, 0,
		871, 1736, 2588, 3420, 4226, 5000, 5735, 6427, 7071,
		7660, 8191, 8660, 9063, 9396, 9659, 9848, 9961, 10000 };

	// Normalize the starting and ending angles.
	while (startAngle < 0)
		startAngle += 360, endAngle += 360;
	if (startAngle > 360)
		startAngle %= 360;
	while (endAngle < 0)
		endAngle += 360;
	if (endAngle > 360)
		endAngle %= 360;

	// Define starting and ending radial lines for use in the windows API.
	long centerX = (left + right) * 5000L;  // (Normalized value.)
	long centerY = (top + bottom) * 5000L;  // (Normalized value.)
	int startX = (int)((centerX + 100L * pCos[startAngle / 5]) / 10000);
	int startY = (int)((centerY - 100L * pSin[startAngle / 5]) / 10000);
	int endX = (int)((centerX + 100L * pCos[endAngle / 5]) / 10000);
	int endY = (int)((centerY - 100L * pSin[endAngle / 5]) / 10000);

	// Set the pen width.
	if (width != 1)
	{
		HPEN newPen = GetGdiCachePen(palette, width);
		SelectObject(hDC, newPen);
	}

	if (fill)
	{
		if (startAngle == endAngle || startAngle == 0 && endAngle == 360)
			// Draw a complete ellipse.
			::Ellipse(hDC, left, top, right, bottom);
		else
			// Draw a pie slice.
			Pie(hDC, left, top, right, bottom, startX - 1, startY - 1,
				endX - 1, endY - 1);
	}
	else
		// Draw a non-filled ellipse or arc.
		Arc(hDC, left, top, right, bottom, startX, startY, endX, endY);

	// Restore the pen width.
	if (width != 1)
	{
		HPEN hOldPen = (HPEN)SelectObject(hDC, hPen);
		ReleaseGdiCacheObject(hOldPen);
	}

	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::Line(int column1, int line1, int column2, int line2,
		int width)
{
	// Calculate a new end point that is one pixel beyond the specified
	// endpoint because the Window's LineTo() API doesnt include the
	// end point when it draws.
	int endPointX = column2, endPointY = line2;
	if (column1 == column2)
		// The specified line is vertical, extend the "y" coordinate
		// of the end point.
		(line1 < line2) ? endPointY++ : endPointY--;
	else if (line1 == line2)
		// The specified line is horizontal, extend the "x" coordinate
		// of the end point.
		(column1 < column2) ? endPointX++ : endPointX--;
	else
	{
		// Calculate the slope of the line.  (Divide by zero errors are
		// avoided by the code above.)
		long slope = (line2 - line1) * 1000L / (column2 - column1);
		if (slope < 1000 && slope > -1000)
		{
			// This specified line is less than 45 degrees from horizontal.
			// The "x" coordinate of the enpoint can be extended, and the
			// "y" coordinate can be calculated using the slope-intercept
			// formula.
			int deltaX = column2 - column1;
			(deltaX > 0) ? deltaX++ : deltaX--;
			endPointX = column1 + deltaX;
			endPointY = (int)(line1 + deltaX * slope / 1000L);
		}
		else
		{
			// This specified line is more than 45 degrees from horizontal.
			// The "y" coordinate of the enpoint can be extended, and the
			// "x" coordinate can be calculated using the slope-intercept
			// formula.
			int deltaY = line2 - line1;
			(deltaY > 0) ? deltaY++ : deltaY--;
			endPointX = (int)(column1 + deltaY * 1000L / slope);
			endPointY = line1 + deltaY;
		}
	}

	// Set the pen width.
	if (width != 1)
	{
		HPEN newPen = GetGdiCachePen(palette, width);
		SelectObject(hDC, newPen);
	}

	// Draw the line.
	MoveToEx(hDC, column1, line1, NULL);
	LineTo(hDC, endPointX, endPointY);

	// Restore the pen width.
	if (width != 1)
	{
		HPEN hOldPen = (HPEN)SelectObject(hDC, hPen);
		ReleaseGdiCacheObject(hOldPen);
	}

	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::Pixel(int column, int line, ZafLogicalColor color, ZafLogicalColor)
{
	// Get the color.
	ZafLogicalColor drawColor = (color == ZAF_CLR_DEFAULT) ? palette.colorForeground : color;
	COLORREF colorRef = (drawColor == ZAF_CLR_DEFAULT) ?
		palette.osPalette.colorForeground : colorTable[drawColor];

	// Draw the pixel.
	SetPixel(hDC, column, line, colorRef);

	return ZAF_ERROR_NONE;
}

ZafError ZafDisplay::Polygon(int numPoints, const int *polygonPoints,
	int width, bool fill, bool close)
{
	// Allocate Windows polygon data with space to "close" polygon if necessary.
	POINT *polygon = new POINT[numPoints + 1];

	// Initialize the polygon points.
	for (int i = 0; i < numPoints; i++)
	{
		polygon[i].x = polygonPoints[i*2];
		polygon[i].y = polygonPoints[i*2+1];
	}

	// Close the polygon if specified.
	if (close && (polygon[0].x != polygon[numPoints - 1].x ||
		polygon[0].y != polygon[numPoints - 1].y))
	{
		polygon[numPoints].x = polygon[0].x;
		polygon[numPoints].y = polygon[0].y;
		numPoints++;
	}

	// Set the pen width.
	if (width != 1)
	{
		HPEN newPen = GetGdiCachePen(palette, width);
		SelectObject(hDC, newPen);
	}

	// Draw the polygon.
	if (fill)
		::Polygon(hDC, polygon, numPoints);
	else
		Polyline(hDC, polygon, numPoints);

	// Restore the pen width.
	if (width != 1)
	{
		HPEN hOldPen = (HPEN)SelectObject(hDC, hPen);
		ReleaseGdiCacheObject(hOldPen);
	}

	// Clean up.
	delete []polygon;

	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::Rectangle(int left, int top, int right, int bottom,
		int width, bool fill)
{
	// Set the pen width.
	if (width > 1)
	{
		HPEN newPen = GetGdiCachePen(palette, width);
		SelectObject(hDC, newPen);
	}

	// Set background mode.
	HBRUSH oldBrush;
	if (!fill)
		oldBrush = (HBRUSH)SelectObject(hDC, GetStockObject(NULL_BRUSH));

	// Draws outline with current pen and fills with current brush.
	if (width == 0)
	{
		RECT rect;
		rect.left = left;
		rect.top = top;
		rect.right = right + 1;
		rect.bottom = bottom + 1;
		FillRect(hDC, &rect, hBrush);
	}
	else
		::Rectangle(hDC, left, top, right + 1, bottom + 1);

	// Restore the brush.
	if (!fill)
		SelectObject(hDC, oldBrush);

	// Restore the pen.
	if (width != 1)
	{
		HPEN hOldPen = (HPEN)SelectObject(hDC, hPen);
		ReleaseGdiCacheObject(hOldPen);
	}

	return (ZAF_ERROR_NONE);
}

ZafError ZafDisplay::RectangleXORDiff(const ZafRegionStruct *oldRegion,
		const ZafRegionStruct *newRegion)
{
	RECT rect;
	if (newRegion)
	{
		// Convert to Windows region.
		rect.left = newRegion->left;
		rect.top = newRegion->top;
		rect.right = newRegion->right + 1;
		rect.bottom = newRegion->bottom + 1;

		// Draw new rectagle.
		DrawFocusRect(hDC, &rect);
	}

	if (oldRegion)
	{
		// Convert to Windows region.
		rect.left = oldRegion->left;
		rect.top = oldRegion->top;
		rect.right = oldRegion->right + 1;
		rect.bottom = oldRegion->bottom + 1;

		// Erase old rectagle.
		DrawFocusRect(hDC, &rect);
	}

	return (ZAF_ERROR_NONE);
}


// --- Drawing support ------------------------------------------------------

ZafError ZafDisplay::RegionCopy(const ZafRegionStruct &region,
	int newColumn, int newLine)
{
	// Get a screen device context if the current context is NULL.
	HDC moveDC = hDC ? hDC : GetDC(0);

	// Move the region by performing a "bitblt" operation.
	BitBlt(moveDC, newColumn, newLine, region.Width(), region.Height(),
		moveDC, region.left, region.top, SRCCOPY);

	// Release the screen device context.
	if (!hDC)
		ReleaseDC(0, moveDC);

	return ZAF_ERROR_NONE;
}

// --- Get -----------------------------------------------------------------

ZafLogicalColor ZafDisplay::Background(void) const
{
	// Return the current background color.
	return (palette.colorBackground);
}

ZafRegionStruct ZafDisplay::ClipRegion(void) const
{
	// Get the current clipping region.
	RECT clip;
	GetClipBox(hDC, &clip);

	ZafRegionStruct region;
	region.ImportPixel(clip);
	return (region);
}

OSDisplayContext ZafDisplay::DisplayContext(void) const
{
	// Return a handle to the current device contect.
	return (hDC);
}

OSDrawContext ZafDisplay::DrawContext(void) const
{
	// Return a handle to the drawing window.
	return (hWnd);
}

ZafLogicalFont ZafDisplay::Font(void) const
{
	// Return the current font.
	return (palette.font);
}

ZafLogicalColor ZafDisplay::Foreground(void) const
{
	// Return the current foreground color.
	return (palette.colorForeground);
}

ZafLogicalColor ZafDisplay::MonoBackground(void) const
{
	// Return the current monochrome background color.
	return (palette.monoBackground);
}

ZafLogicalColor ZafDisplay::MonoForeground(void) const
{
	// Return the current monochrome foreground color.
	return (palette.monoForeground);
}

ZafPaletteStruct ZafDisplay::Palette(void) const
{
	// Return the current palette.
	return (palette);
}

ZafLogicalFillPattern ZafDisplay::FillPattern(void) const
{
	// Return the current pattern.
	return (palette.fillPattern);
}

ZafLogicalMode ZafDisplay::Mode(void) const
{
	// Return the current mix mode.
	return (mode);
}

// --- Set -----------------------------------------------------------------

ZafLogicalColor ZafDisplay::SetBackground(ZafLogicalColor color)
{
	ZafLogicalColor oldColor = palette.colorBackground;
	palette.colorBackground = color;

	// Get the Windows color reference.
	COLORREF colorRef = (color == ZAF_CLR_DEFAULT) ?
		palette.osPalette.colorBackground : colorTable[color];

	// Set the background color.
	SetBkColor(hDC, colorRef);

	// Get a new background brush.
	HBRUSH newBrush = GetGdiCacheBrush(palette);

	// Select the new background brush into the DC.
	SelectObject(hDC, newBrush);

	// Release the old backgroud brush.
	if (hBrush)
		ReleaseGdiCacheObject(hBrush);

	hBrush = newBrush;

	return (oldColor);
}

ZafRegionStruct ZafDisplay::SetClipRegion(const ZafRegionStruct &region)
{
	ZafRegionStruct oldClipRegion = clipRegion;

	// Create a Windows region.
	HRGN hRegion = CreateRectRgn(region.left, region.top, region.right + 1, region.bottom + 1);

	// Copy the Windows region into the DC clipping region.  
	SelectClipRgn(hDC, hRegion);

	// Delete the Windows region.
	DeleteObject(hRegion);

	// Set the member data.
	clipRegion = region;

	return (oldClipRegion);
}

OSDisplayContext ZafDisplay::SetDisplayContext(OSDisplayContext displayContext)
{
	HDC oldDC = hDC;

	hDC = displayContext;

	// Save the current DC attributes so that changes can be restored.
	SaveDC(hDC);

	// Select the logical color paletee into the DC.
	SelectPalette(hDC, logicalPalette, FALSE);

	return (oldDC);
}

ZafLogicalFont ZafDisplay::SetFont(ZafLogicalFont font)
{
	ZafLogicalFont oldFont = palette.font;

	// Select the specified font into the DC.
	if (font >= 0)
		SelectObject(hDC, fontTable[font]);

	// Set the member variable.
	palette.font = font;

	return (oldFont);
}

ZafLogicalColor ZafDisplay::SetForeground(ZafLogicalColor color)
{
	ZafLogicalColor oldColor = palette.colorForeground;
	palette.colorForeground = color;

	// Get the Windows color reference.
	COLORREF colorRef = (color == ZAF_CLR_DEFAULT) ?
		palette.osPalette.colorForeground : colorTable[color];

	// Set the text color.
	SetTextColor(hDC, colorRef);

	// Get a new pen.
	HPEN newPen = GetGdiCachePen(palette, 1);

	// Select the new pen into the DC.
	SelectObject(hDC, newPen);

	// Release the old pen.
	if (hPen)
		ReleaseGdiCacheObject(hPen);

	hPen = newPen;

	return (oldColor);
}

ZafLogicalLineStyle ZafDisplay::SetLineStyle(ZafLogicalLineStyle newLineStyle)
{
	ZafLogicalLineStyle oldLineStyle = palette.lineStyle;

	if (newLineStyle != oldLineStyle)
	{
		palette.lineStyle = newLineStyle;

		// Get a new pen.
		HPEN newPen = GetGdiCachePen(palette, 1);

		// Select the new pen into the DC.
		SelectObject(hDC, newPen);

		// Release the old pen.
		ReleaseGdiCacheObject(hPen);

		hPen = newPen;
	}

	return (oldLineStyle);
}

ZafLogicalColor ZafDisplay::SetMonoBackground(ZafLogicalColor color)
{
	ZafLogicalColor oldColor = palette.monoBackground;

	palette.monoBackground = color;

	return (oldColor);
}

ZafLogicalColor ZafDisplay::SetMonoForeground(ZafLogicalColor color)
{
	ZafLogicalColor oldColor = palette.monoForeground;

	palette.monoForeground = color;

	return (oldColor);
}

ZafPaletteStruct ZafDisplay::SetPalette(ZafPaletteStruct tPalette)
{
	ZafPaletteStruct oldPalette = palette;

	// Copy the OS specific palette information.
	palette.osPalette = tPalette.osPalette;

	// Set each of the palette elements.
	SetLineStyle(tPalette.lineStyle);
	SetFillPattern(tPalette.fillPattern);
	SetForeground(tPalette.colorForeground);
	SetBackground(tPalette.colorBackground);
	SetFont(tPalette.font);

	return (oldPalette);
}

ZafLogicalFillPattern ZafDisplay::SetFillPattern(ZafLogicalFillPattern pattern)
{
	ZafLogicalFillPattern oldPattern = palette.fillPattern;

	if (pattern != oldPattern)
	{
		palette.fillPattern = pattern;

		// Get a new brush.
		HBRUSH newBrush = GetGdiCacheBrush(palette);

		// Select the new brush into the DC.
		SelectObject(hDC, newBrush);

		// Release the old brush.
		if (hBrush)
			ReleaseGdiCacheObject(hBrush);

		hBrush = newBrush;
	}

	return (oldPattern);
}

ZafLogicalMode ZafDisplay::SetMode(ZafLogicalMode tmode)
{
	ZafLogicalMode oldMode = mode;
	mode = tmode;

	// Set the raster operation (ROP) mode.
	SetROP2(hDC, modeTable[tmode].binary);

	return (oldMode);
}

// --- Restore --------------------------------------------------------------

OSDisplayContext ZafDisplay::RestoreDisplayContext(void)
{
	// Preserve DC handle to be returned.
	OSDisplayContext displayContext = hDC;

	if (hDC)
	{
		// Restore any changes made to the DC, and reset member variables.
		// This call does not release the DC.
		RestoreDC(hDC, -1);
		hDC = 0;

		// Clean up new pen and brush (if any).  (The original (default)
		// pen and brush were selected into the DC by the RestoreDC()
		// call above.)
		if (hPen)
		{
			ReleaseGdiCacheObject(hPen);
			hPen = 0;
		}
		if (hBrush)
		{
			ReleaseGdiCacheObject(hBrush);
			hBrush = 0;
		}
	}

	// Return the UNRELEASED device context.
	return (displayContext);
}

