//	Zinc Application Framework - W_VLIST.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include "w_app.hpp"
#include <z_combo.hpp>
#include <z_str1.hpp>

static const int MAX_TEXT_LEN = 128;

LRESULT CALLBACK ComboStringProc(HWND hwnd, UINT wMsg, WPARAM wParam, LPARAM lParam)
{
	// Route messages through the parent combo-box so that special
	// processing (drag & drop, edit mode, etc.) can be done.

	// Get a pointer to the combo box.
	HWND hCombo = GetParent(hwnd);
	ZafWindowObject *combo = ZafMSWindowsApp::ObjectFromHandle(hCombo);

	// Create a ZAF event.
	ZafEventStruct event(E_MSWINDOWS);
	event.osEvent.hwnd = hwnd;
	event.osEvent.message = wMsg;
	event.osEvent.wParam = wParam;
	event.osEvent.lParam = lParam;

	// Route the event. (Keyboard events are routed top-down.)
	ZafEventType inputType = event.InputType();
	if (combo->EditMode() && (inputType == E_KEY || inputType == E_MOUSE))
		return (combo->ZafWindowObject::Event(event));
	else if (inputType == E_KEY)
	{
		// Route the event to the root window to be processed top-down.
		// (The event has already been passed to the window manager.)
		event.route = combo;
		return (combo->RootObject()->Event(event));
	}
	else
		return (combo->Event(event));
}

// ----- ZafComboBox --------------------------------------------------------

ZafEventMap ZAF_FARDATA ZafComboBox::defaultEventMap[] =
{
	{ L_NONE,	0,	0,	0 }
};

ZafPaletteMap ZAF_FARDATA ZafComboBox::defaultPaletteMap[] =
{
	{ ZAF_PM_BACKGROUND, ZAF_PM_ANY_STATE, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_DEFAULT, ZAF_CLR_DEFAULT, ZAF_MONO_BLACK, ZAF_MONO_WHITE, ZAF_FNT_SYSTEM } },
	{ ZAF_PM_NONE, 0 }
};

ZafWindowObject *ZafComboBox::Add(ZafWindowObject *object, ZafWindowObject *position)
{
	if (object->parent)
		return (object);

	object->SetSystemObject(false);
	object->SetParentDrawBorder(true);
	object->SetParentDrawFocus(true);
	object->SetParentPalette(true);

	// Add the object to the drop-down list.
	list->ZafWindow::Add(object, position);

	if (object->SupportObject())
	{
		if (screenID)
		{
			//??? this can probably be done efficiantly at the window level
			//??? using SetWindowLong.
			Event(S_DESTROY);
			Event(S_CREATE);
		}

		return object;
	}

	// Update the current/selected attributes.
	if (object->Selected())
		NotifySelection(object, true);
	else if (list->Current() == object)
		object->SetSelected(true);

	// Insert the object into the OS control.
	if (screenID)
	{
		int index = list->Index(object);
		if (ZafMSWindowsApp::convertText && OSDraw())
		{
			char *osText = zafCodeSet->ConvertToOSString(object->Text(), ZAF_NULLP(char), FALSE);
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
			SendMessageA(screenID, CB_INSERTSTRING, (WPARAM)index, (LPARAM)osText);
#else
			SendMessage(screenID, CB_INSERTSTRING, (WPARAM)index, (LPARAM)osText);
#endif
		}
		else
			SendMessage(screenID, CB_INSERTSTRING, (WPARAM)index, OSDraw() ? (LPARAM)object->Text() : (LPARAM)object);

		if (object->Selected())
			SendMessage(screenID, CB_SETCURSEL, (WPARAM)index, (LPARAM)0);
	}

	return object;
}

ZafRegionStruct ZafComboBox::ConvertToDrawRegion(const ZafWindowObject *object,
	const ZafRegionStruct *region) const
{
	ZafRegionStruct drawRegion;
	if (region)
	{
		drawRegion = *region;
		drawRegion.left += object->zafRegion.left;
		drawRegion.right += object->zafRegion.left;
		drawRegion.top += object->zafRegion.top;
		drawRegion.bottom += object->zafRegion.top;
	}
	else
		// No conversion is necessary.
		drawRegion = object->zafRegion;
	return (drawRegion);
}

ZafEventType ZafComboBox::DrawBorder(ZafRegionStruct &region,
	ZafEventType ccode)
{
	// Defer to the base class.
	return (ZafWindowObject::DrawBorder(region, ccode));
}

ZafEventType ZafComboBox::DrawFocus(ZafRegionStruct &region, ZafEventType ccode)
{
	return ZafWindowObject::DrawFocus(region, ccode);
}

ZafEventType ZafComboBox::Draw(const ZafEventStruct &, ZafEventType ccode)
{
	// Combobox is always drawn by windows.
	// (Ownerdraw means children draw themselves insetead of being drawn by
	// the combobox.)
	return ccode;
}

ZafEventType ZafComboBox::DragDropEvent(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	switch (ccode)
	{
	case S_DROP_DEFAULT:
		if (windowManager->dragObject->MoveDraggable())
			ccode = S_DROP_MOVE;
		else if (windowManager->dragObject->CopyDraggable())
			ccode = S_DROP_COPY;
		else
		{
			ccode = S_ERROR;
			break;
		}
		//Continue.
	case S_DROP_COPY:
	case S_DROP_MOVE:
		if (acceptDrop && this != windowManager->dragObject)
		{
			if (windowManager->dragObject->IsA(ID_ZAF_COMBO_BOX) ||
				windowManager->dragObject->IsA(ID_ZAF_VT_LIST) ||
				windowManager->dragObject->IsA(ID_ZAF_HZ_LIST))
			{
				// Move the selected objects.
				ZafWindow *dragList = DynamicPtrCast(windowManager->dragObject, ZafWindow);
				ZafWindowObject *object;
				if (windowManager->dragObject->IsA(ID_ZAF_COMBO_BOX))
					object = DynamicPtrCast(dragList, ZafComboBox)->First();
				else
					object = dragList->First();
				while (object)
				{
					ZafWindowObject *nextObject = object->Next();
					if (object->Selected())
					{
						if (ccode == S_DROP_COPY)
							Add(object->Duplicate());
						else
						{
							dragList->Subtract(object);
							Add(object);
						}
					}
					object = nextObject;
				}
			}
			else
			{
				const ZafIChar *dropText = windowManager->dragObject->Text();
				if (dropText && *dropText)
				{
					ZafString *newString = new ZafString(0, 0, 0, new ZafStringData(dropText));
					newString->SetBordered(false);
					Add(newString);
					if (ccode != S_DROP_COPY)
						windowManager->dragObject->SetText(ZafLanguageData::blankString);
				}
				else
					ccode = S_ERROR;
			}
		}
		break;

	default:
		ccode = ZafWindowObject::DragDropEvent(event);
		break;
	}

	return (ccode);
}

ZafEventType ZafComboBox::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	if (ccode == E_OSEVENT)
		return ZafComboBox::OSEvent(event);

	// Switch on the event type.
	switch (ccode)
	{
	case S_INITIALIZE:
	case S_DEINITIALIZE:
		{
		// Default to the base class.
		ccode = ZafWindowObject::Event(event);
		list->Event(event);
		}
		break;

	case S_REGISTER_OBJECT:
		{
		ZafWindowObject::Event(event);
		list->SetSystemObject(false);
		list->Event(event);
		ZafList::SetCurrent(list->ZafList::Current());

		// List must be positioned at 0,0 to fix up draw coordinates 
		// since it is not a system object.
		list->zafRegion.right = list->zafRegion.Width() - 1;
		list->zafRegion.bottom = list->zafRegion.Height() - 1;
		list->zafRegion.left = list->zafRegion.top = 0;

		int index = 0;
		for (ZafWindowObject *object = list->First(); object; object = object->Next())
		{
			object->screenID = screenID;

			if (ZafMSWindowsApp::convertText && OSDraw())
			{
				char *osText = zafCodeSet->ConvertToOSString(object->Text(), ZAF_NULLP(char), FALSE);
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
				SendMessageA(screenID, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)osText);
#else
				SendMessage(screenID, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)osText);
#endif
			}
			else
				SendMessage(screenID, CB_INSERTSTRING, (WPARAM)-1, OSDraw() ? (LPARAM)object->Text() : (LPARAM)object);

			if (object->Selected())
				SendMessage(screenID, CB_SETCURSEL, (WPARAM)index, (LPARAM)0);
			index++;
		}
		}
		break;

	case S_ADD_OBJECT:
	case S_SUBTRACT_OBJECT:
		ccode = ZafWindow::Event(event);
		break;

	default:
		ccode = ZafWindowObject::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

void ZafComboBox::GetClip(ZafWindowObject *, ZafRegionStruct &, ZafRegionStruct &)
{
	// Handled automatically by Windows.
}

ZafWindowObject *ZafComboBox::NotifyFocus(ZafWindowObject *object, bool setFocus)
{
	ZafWindowObject *invalidObject = ZafWindow::NotifyFocus(object, setFocus);
	list->ZafList::SetCurrent(current);
	if (object != this && setFocus && !invalidObject)
	{
		int index = list->Index(object);
		if (SendMessage(screenID, CB_GETCURSEL, (WPARAM)0, (LPARAM)0) != index)
			SendMessage(screenID, CB_SETCURSEL, (WPARAM)index, (LPARAM)0);
	}

	return invalidObject;
}

ZafWindowObject *ZafComboBox::NotifySelection(ZafWindowObject *object, bool setSelected)
{
	if (setSelected)
	{
		SetCurrent(object);
		list->SetCurrent(object);
		if (screenID)
		{
			int index = list->Index(object);
			if (SendMessage(screenID, CB_GETCURSEL, (WPARAM)0, (LPARAM)0) != index)
				SendMessage(screenID, CB_SETCURSEL, (WPARAM)index, (LPARAM)0);
		}
	}
	return object;
}

bool ZafComboBox::SetAutoSortData(bool autoSortData)
{
	// Set the list attribute.
	return (list->SetAutoSortData(autoSortData));
}

ZafError ZafComboBox::SetText(const ZafIChar *text)
{
	if (ViewOnly() || !OSDraw())
		return ZAF_ERROR_INVALID;
	else if (screenID)
	{
		if (ZafMSWindowsApp::convertText)
		{
			char *osText = zafCodeSet->ConvertToOSString(text, ZAF_NULLP(char), FALSE);
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
			SendMessageA(screenID, WM_SETTEXT, (WPARAM)0, (LPARAM)osText);
#else
			SendMessage(screenID, WM_SETTEXT, (WPARAM)0, (LPARAM)osText);
#endif
		}
		else
			SendMessage(screenID, WM_SETTEXT, (WPARAM)0, (LPARAM)text);
	}

	return ZAF_ERROR_NONE;
}

const ZafIChar *ZafComboBox::Text(void)
{
	if (ViewOnly() || !OSDraw())
	{
		ZafWindowObject *object = list->Current();
		return (object ? object->Text() : ZAF_NULLP(ZafIChar));
	}
	else
	{
//???
		static ZafIChar text[MAX_TEXT_LEN];
		SendMessage(screenID, WM_GETTEXT, (WPARAM)MAX_TEXT_LEN, (LPARAM)text);
		return text;
	}
}

ZafWindowObject *ZafComboBox::Subtract(ZafWindowObject *object)
{
	if (screenID && !object->SupportObject())
		SendMessage(screenID, CB_DELETESTRING, (WPARAM)list->Index(object), (LPARAM)0);

	list->Subtract(object);

	if (screenID && object->SupportObject())
	{
		//??? this can probably be done efficiantly at the window level
		//??? using SetWindowLong.
		Event(S_DESTROY);
		Event(S_CREATE);
	}

	return (object);
}

// ----- Attributes ---------------------------------------------------------

bool ZafComboBox::SetFocus(bool setFocus)
{
	return ZafWindowObject::SetFocus(setFocus);
}

bool ZafComboBox::SetVisible(bool setVisible)
{
	return ZafWindowObject::SetVisible(setVisible);
}

// ----- OS Specific Functions ----------------------------------------------

void ZafComboBox::OSDestroy(void)
{
	SendMessage(screenID, CB_RESETCONTENT, 0, 0);
	for (ZafWindowObject *object = list->First(); object; object = object->Next())
		object->Event(S_DEINITIALIZE);
}

ZafEventType ZafComboBox::OSEvent(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	switch (event.osEvent.message)
	{
	case WM_COMMAND:
		{
		if (event.osEvent.hwnd == screenID)
			ccode = DefaultCallback(event);

#if defined(ZAF_WIN32)
		WORD wNotifyCode = HIWORD(event.osEvent.wParam);
#else
		WORD wNotifyCode = HIWORD(event.osEvent.lParam);
#endif
		if (wNotifyCode == CBN_SELCHANGE)
		{
			// Get index of the current item.
			int currentIndex = (int)SendMessage(screenID, CB_GETCURSEL, 0, 0);

			// Get the new current/selected object.
			ZafWindowObject *object = DynamicPtrCast(list->ZafList::Get(currentIndex), ZafWindowObject);

			// Set the focus and selection.
			if (object)
			{
				object->NotifyFocus(object, true);
				object->SetSelected(true);
				(object->*(object->memberUserFunction))(event, L_SELECT);
			}
		}
		}
		break;

#if defined(ZAF_WIN32)
	case WM_CTLCOLOREDIT:
	case WM_CTLCOLORLISTBOX:
#else
	case WM_CTLCOLOR:
#endif
		{
		// Get the palettes.
		ZafPaletteState state = PaletteState();
		ZafPaletteStruct textPalette = LogicalPalette(ZAF_PM_TEXT, state);
		ZafPaletteStruct backgroundPalette = LogicalPalette(ZAF_PM_BACKGROUND, state);

		// Get the OS colors.
		COLORREF colorForeground = (textPalette.colorForeground == ZAF_CLR_DEFAULT) ?
			textPalette.osPalette.colorForeground :
			display->colorTable[textPalette.colorForeground];
		COLORREF colorBackground = (backgroundPalette.colorBackground == ZAF_CLR_DEFAULT) ?
			backgroundPalette.osPalette.colorBackground :
			display->colorTable[backgroundPalette.colorBackground];

		// Set the OS colors.
		SelectPalette((HDC)event.osEvent.wParam, display->logicalPalette, FALSE);
		::SetTextColor((HDC)event.osEvent.wParam, colorForeground);
		SetBkColor((HDC)event.osEvent.wParam, colorBackground);

		// Get a background brush.
		HBRUSH backgroundBrush = display->GetGdiCacheBrush(backgroundPalette);
		display->ReleaseGdiCacheObject(backgroundBrush);
		ccode = (ZafEventType)backgroundBrush;
		}
		break;

	case WM_DRAWITEM:
		{
		DRAWITEMSTRUCT *drawItemStruct = (DRAWITEMSTRUCT *)event.osEvent.lParam;

		// Begin drawing operation.
		display->BeginDraw(drawItemStruct->hDC, 0, zafRegion, zafRegion);

		if (drawItemStruct->itemID == (UINT)-1)
		{
			ZafRegionStruct focusRegion;
			focusRegion.left = drawItemStruct->rcItem.left;
			focusRegion.top = drawItemStruct->rcItem.top;
			focusRegion.right = drawItemStruct->rcItem.right - 1;
			focusRegion.bottom = drawItemStruct->rcItem.bottom - 1;

			if (drawItemStruct->itemState & ODS_FOCUS)
				DrawFocus(focusRegion, S_CURRENT);
			else
				DrawFocus(focusRegion, S_NON_CURRENT);
		}
		else
		{
			ZafWindowObject *object = (ZafWindowObject *)drawItemStruct->itemData;
			object->zafRegion.left = drawItemStruct->rcItem.left;
			object->zafRegion.top = drawItemStruct->rcItem.top;
			object->zafRegion.right = drawItemStruct->rcItem.right - 1;
			object->zafRegion.bottom = drawItemStruct->rcItem.bottom - 1;

			if (drawItemStruct->itemAction & ODA_SELECT)
				drawItemStruct->itemAction |= ODA_DRAWENTIRE;

			if (drawItemStruct->itemAction & ODA_FOCUS)
			{
				if (drawItemStruct->itemState & ODS_FOCUS)
					DrawFocus(object->zafRegion, S_CURRENT);
				else
					DrawFocus(object->zafRegion, S_NON_CURRENT);
			}
			else
				object->OSEvent(event);
		}

		// End drawing operation.
		display->EndDraw();
		ccode = TRUE;
		}
		break;

	case WM_KEYDOWN:
	case WM_KEYUP:
		switch ((int)event.osEvent.wParam)
		{
		case VK_UP:
		case VK_DOWN:
		case VK_LEFT:
		case VK_RIGHT:
		case VK_PRIOR:
		case VK_NEXT:
		case VK_F4:
		case VK_ESCAPE:
			ccode = DefaultCallback(event);
			break;

		case VK_TAB:
			ccode = S_UNKNOWN;
			break;

		default:
			ccode = ZafWindowObject::OSEvent(event);
			break;
		}
		break;

	case WM_CHAR:
		if (event.osEvent.wParam == VK_TAB)
			return 0;
		break;
/* START BLOCK COMMENT
**		case WM_LBUTTONUP:
**			{
**			ccode = ZafWindowObject::OSEvent(event);
**	
**			// Make sure up clicks get to the edit field during drag & drop
**			// operations.
**	//		if (GetCapture() == screenID)
**			{
**				HWND childID = GetWindow(screenID, GW_CHILD);
**				if (childID)
**					SendMessage(childID, event.osEvent.message, event.osEvent.wParam,
**						event.osEvent.lParam);
**			}
**			}
**			break;
END BLOCK COMMENT */

	case WM_MEASUREITEM:
		if (OSDraw())
			ccode = ZafWindowObject::OSEvent(event);
		else
		{
			MEASUREITEMSTRUCT *measureItemStruct = (MEASUREITEMSTRUCT *)event.osEvent.lParam;
			if (measureItemStruct->itemID == (UINT)-1)
			{
				int maxHeight = 0;
				int maxWidth = 0;
				for (ZafWindowObject *object = list->First(); object; object = object->Next())
				{
					object->Event(ZafEventStruct(S_COMPUTE_SIZE));
					int width = object->zafRegion.Width();
					int height = object->zafRegion.Height();
					if (width > maxWidth)
						maxWidth = width;
					if (height > maxHeight)
						maxHeight = height;
				}
				measureItemStruct->itemWidth = maxWidth;
				measureItemStruct->itemHeight = maxHeight;
			}
			else
			{
				ZafWindowObject *object = (ZafWindowObject *)measureItemStruct->itemData;
				object->Event(ZafEventStruct(S_COMPUTE_SIZE));
				measureItemStruct->itemWidth = object->zafRegion.Width();
				measureItemStruct->itemHeight = object->zafRegion.Height();
			}
			ccode = TRUE;
		}
		break;

	case WM_PAINT:
	case WM_ERASEBKGND:
		// Don't call Draw().
		ccode = DefaultCallback(event);
		break;

	default:
		ccode = ZafWindowObject::OSEvent(event);
	}

	return (ccode);
}

ZafError ZafComboBox::OSGetText(void)
{
//???
	return ZAF_ERROR_NONE;
}

void ZafComboBox::OSMapPalette(ZafPaletteStruct &palette, ZafPaletteType type, ZafPaletteState state)
{
	// Get default colors from Windows.
/* START BLOCK COMMENT
**		if (type == ZAF_PM_TEXT || type == ZAF_PM_FOREGROUND)
**		{
**			if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
**			{
**				if (state & ZAF_PM_SELECTED)
**					palette.osPalette.colorForeground = GetSysColor(COLOR_HIGHLIGHTTEXT);
**				else
**					palette.osPalette.colorForeground = GetSysColor(COLOR_WINDOWTEXT);
**			}
**			if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
**			{
**				if (state & ZAF_PM_SELECTED)
**					palette.osPalette.colorBackground = GetSysColor(COLOR_HIGHLIGHT);
**				else
**					palette.osPalette.colorBackground = GetSysColor(COLOR_WINDOW);
**			}
**		}
END BLOCK COMMENT */
//	else if (type == ZAF_PM_BACKGROUND)
	if (type == ZAF_PM_BACKGROUND)
	{
		if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
		{
			if (state & ZAF_PM_SELECTED)
				palette.osPalette.colorForeground = GetSysColor(COLOR_HIGHLIGHT);
			else
				palette.osPalette.colorForeground = GetSysColor(COLOR_WINDOW);
		}
		if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
		{
			if (state & ZAF_PM_SELECTED)
				palette.osPalette.colorBackground = GetSysColor(COLOR_HIGHLIGHT);
			else
				palette.osPalette.colorBackground = GetSysColor(COLOR_WINDOW);
		}
	}
	else
		ZafWindow::OSMapPalette(palette, type, state);
}

void ZafComboBox::OSRegisterObject(void)
{
	DWORD dwStyle = WS_CHILD | WS_CLIPSIBLINGS;
	DWORD dwExStyle = 0;

	if (visible)
		dwStyle |= WS_VISIBLE;

	// Combo boxes always have borders.
#if defined(ZAF_MSWINDOWS_3D)
	if (ZafMSWindowsApp::native3D)
		dwExStyle |= WS_EX_CLIENTEDGE;
	else
		dwStyle |= WS_BORDER;
#else
	dwStyle |= WS_BORDER;
#endif

	if (Disabled() && !editMode)
		dwStyle |= WS_DISABLED;
	if (OSDraw())
	{
		if (ViewOnly())
			dwStyle |= CBS_DROPDOWNLIST;
		else
			dwStyle |= CBS_DROPDOWN | CBS_AUTOHSCROLL;
	}
	else
	{
		dwStyle |= CBS_OWNERDRAWVARIABLE;
		dwStyle |= CBS_DROPDOWNLIST;
	}

	if (list->VerticalScrollBar())
		dwStyle |= WS_VSCROLL;
	screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, ZAF_ITEXT("COMBOBOX"), dwStyle, ZAF_NULLP(ZafIChar), 0, dwExStyle);

	// Set the font.
	// Save zafRegion because setting the font automatically adjusts the 
	// combo height, and the initial sizing operation has not occured yet.
	ZafRegionStruct saveRegion = zafRegion;
	ZafPaletteStruct fontPalette = LogicalPalette(ZAF_PM_TEXT, ZAF_PM_ACTIVE | ZAF_PM_ENABLED);
	SendMessage(screenID, WM_SETFONT, (WPARAM)display->fontTable[fontPalette.font], (LPARAM)FALSE);
	zafRegion = saveRegion;

	HWND childID = GetWindow(screenID, GW_CHILD);
	if (childID)
		SetWindowLong(childID, GWL_WNDPROC, (LONG)ComboStringProc);
}

OSWindowID ZafComboBox::OSScreenID(ZafScreenIDType) const
{
	return screenID;
}

ZafError ZafComboBox::OSSetText(void)
{
//???
	return ZAF_ERROR_NONE;
}

void ZafComboBox::OSSize(void)
{
	if (zafRegion != oldRegion)
	{
		if (SystemObject())
		{
			// Convert zafRegion to OS coordinates.
			ZafRegionStruct osRegion = parent ? parent->ConvertToOSRegion(this) :
				windowManager->ConvertToOSRegion(this);

			osRegion.bottom += list->zafRegion.Height();

			// Move/Size the Windows window.
			MoveWindow(OSScreenID(ZAF_FRAMEID), osRegion.left, osRegion.top, osRegion.Width(),
				osRegion.Height(), TRUE);

			osRegion.bottom -= list->zafRegion.Height();

			oldRegion = zafRegion;
		}
		else
			ZafWindow::OSSize();
	}
}

void ZafComboBox::OSSort(void)
{
	if (screenID)
	{
		SendMessage(screenID, CB_RESETCONTENT, (WPARAM)0, (LPARAM)0);

		//??? Duplicate code.
		//??? Optimize Refresh.
		int index = 0;
		for (ZafWindowObject *object = list->First(); object; object = object->Next())
		{
			if (ZafMSWindowsApp::convertText && OSDraw())
			{
				char *osText = zafCodeSet->ConvertToOSString(object->Text(), ZAF_NULLP(char), FALSE);
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
				SendMessageA(screenID, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)osText);
#else
				SendMessage(screenID, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)osText);
#endif
			}
			else
				SendMessage(screenID, CB_INSERTSTRING, (WPARAM)-1, OSDraw() ? (LPARAM)object->Text() : (LPARAM)object);

			if (object->Selected())
				SendMessage(screenID, CB_SETCURSEL, (WPARAM)index, (LPARAM)0);
			index++;
		}
	}
}

ZafError ZafComboBox::OSUpdatePalettes(ZafPaletteType, ZafPaletteType)
{
	if (screenID)
	{
		ZafPaletteStruct fontPalette = LogicalPalette(ZAF_PM_TEXT, ZAF_PM_ACTIVE | ZAF_PM_ENABLED);
		SendMessage(screenID, WM_SETREDRAW, (WPARAM)FALSE, 0);
		SendMessage(screenID, WM_SETFONT, (WPARAM)display->fontTable[fontPalette.font], (LPARAM)FALSE);
		if (automaticUpdate && visible)
			SendMessage(screenID, WM_SETREDRAW, (WPARAM)TRUE, 0);
		Redisplay();
	}

	return (ZAF_ERROR_NONE);
}

// ----- Stubs --------------------------------------------------------------

bool ZafComboBox::SetAcceptDrop(bool acceptDrop)
{
	return (ZafWindowObject::SetAcceptDrop(acceptDrop));
}
