//	Zinc Application Framework - W_BUTTON.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#define OEMRESOURCE			// Windows button messages and flags.
#define ZAF_BUTTON_INFO		// Check Box and Radio button images.
#include "img_def.cpp"
#include "w_app.hpp"
#include <z_button.hpp>
#include <z_ctype.hpp>
#include <z_keymap.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>

//static int CHECK_BOX_SIZE = 13;

// Utility to reduce code duplication.
static ZafEventType Select(ZafButton *button, const ZafEventStruct &event, ZafEventType ccode)
{
	// Toggle selection if appropriate.
	// Don't allow de-selection of objects in single select window.
	if (button->AllowToggling() && (!button->Selected() ||
		DynamicPtrCast(button->parent, ZafWindow)->SelectionType() != ZAF_SINGLE_SELECTION))
		button->SetSelected(!button->Selected());

	// Call the user function.
	return (button->*button->memberUserFunction)(event, ccode);
}

// ----- ZafButton ---------------------------------------------------------

//??? stringData & imageData parameters should be reversed.
//??? Are ClearImage() and ClearText() necessary?

ZafEventMap ZAF_FARDATA ZafButton::defaultEventMap[] =
{
//??? Which events should be mapped?
	{ L_BEGIN_SELECT,	  	E_KEY, 		SPACE,		S_KEYDOWN },
	{ L_END_SELECT,	 	  	E_KEY, 		SPACE,		S_KEYUP },
	{ L_NONE,	0,	0,	0 }
};

ZafPaletteMap ZAF_FARDATA ZafButton::defaultPaletteMap[] =
{
	{ ZAF_PM_TEXT, ZAF_PM_ANY_STATE, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_DEFAULT, ZAF_CLR_DEFAULT, ZAF_MONO_BLACK, ZAF_MONO_WHITE, ZAF_FNT_DIALOG } },
	{ ZAF_PM_BACKGROUND, ZAF_PM_ANY_STATE, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_DEFAULT, ZAF_CLR_DEFAULT, ZAF_MONO_BLACK, ZAF_MONO_WHITE, ZAF_FNT_DIALOG } },
	{ ZAF_PM_FOCUS, ZAF_PM_ANY_STATE, { ZAF_LINE_SOLID, ZAF_PTN_SOLID_FILL, ZAF_CLR_DEFAULT, ZAF_CLR_BLACK, ZAF_MONO_WHITE, ZAF_MONO_BLACK, ZAF_FNT_DIALOG } },
	{ ZAF_PM_NONE, 0 }
};

ZafEventType ZafButton::DrawBorder(ZafRegionStruct &region, ZafEventType ccode)
{
	if (ParentDrawBorder())
	{
		ZafRegionStruct borderRegion = parent->ConvertToDrawRegion(this);
		return parent->DrawBorder(borderRegion, ccode);
	}

#if defined(ZAF_MSWINDOWS_3D)
	if (ccode == S_REDISPLAY || ccode == S_REDISPLAY_REGION)
	{
		// Draw a single pixel border.
		display->SetPalette(LogicalPalette(ZAF_PM_OUTLINE, PaletteState()));
		display->Rectangle(region, 1, false);
	}

	// Update the draw region.
	region--;
#else
	ccode = ZafWindowObject::DrawBorder(region, ccode);
#endif

	return (ccode);
}

ZafEventType ZafButton::Draw(const ZafEventStruct &, ZafEventType ccode)
{
	// Begin drawing operation.
	ZafRegionStruct drawRegion = BeginDraw();

	if (editMode && allowDefault)
	{
		display->SetPalette(parent->LogicalPalette(ZAF_PM_BACKGROUND, parent->PaletteState()));
		display->Rectangle(drawRegion, 0, true);

		display->SetLineStyle(ZAF_LINE_DOTTED);
		display->SetForeground(ZAF_CLR_BLACK);
		display->Rectangle(drawRegion);

		drawRegion -= ZAF_DEFBTN_MARGIN;
	}

	ZafPaletteState state = PaletteState();

	bool defaultButton = (DynamicPtrCast(parent, ZafWindow)->DefaultButton(false) == this) ? true : false;

	// Draw check boxes and radio buttons.
	if (ButtonType() == ZAF_CHECK_BOX || ButtonType() == ZAF_RADIO_BUTTON)
	{
		state &= ~ZAF_PM_SELECTED;

		// Erase the background.
		// (Background palette doesn't affect radio buttons and check boxes.)
//??? A better way is needed... maybe pass state parameter with draw functions.
		bool saveSelected = selected;
		selected = false;
		DrawBackground(drawRegion, ccode);
		selected = saveSelected;

		// Set the text palette.
		ZafPaletteStruct textPalette = LogicalPalette(ZAF_PM_TEXT, state);
		display->SetPalette(textPalette);

		// Compute the drawing regions.
		int imageX, imageY;
		ZafRegionStruct textRegion = drawRegion;
		ZafRegionStruct textSize;
		if (stringData)
			textSize = display->TextSize(*stringData);
		else
		{
			textSize.left = textSize.top = 1;
			textSize.right = textSize.bottom = 0;
		}
		if (HzJustify() == ZAF_HZ_RIGHT)
		{
			imageX = drawRegion.right - _check_ZafBitmap.width + 1;
			textRegion.right = imageX - 2;
			textRegion.left = textRegion.right - textSize.right - 1;
		}
		else
		{
			imageX = drawRegion.left;
			textRegion.left = imageX + _check_ZafBitmap.width + 1;
			textRegion.right = textRegion.left + textSize.Width() + 1;
		}
		imageY = drawRegion.top + (drawRegion.Height() - _check_ZafBitmap.height) / 2;

		// Draw the check or radio bitmap.
		if (ButtonType() == ZAF_RADIO_BUTTON)
			display->Bitmap(imageX, imageY, selected ? _radioSelected_ZafBitmap : _radio_ZafBitmap);
		else
			display->Bitmap(imageX, imageY, selected ? _checkSelected_ZafBitmap : _check_ZafBitmap);

		// Draw the focus.
		DrawFocus(textRegion, ccode);

		// Draw the text.
		if (stringData && (ccode == S_REDISPLAY || ccode == S_REDISPLAY_DATA || ccode == S_REDISPLAY_REGION))
		{
			if (Disabled())
			{
				ZafPaletteStruct lightPalette = textPalette;
				lightPalette.colorForeground = lightPalette.colorBackground;
				lightPalette.osPalette.colorForeground = lightPalette.osPalette.colorBackground;
				display->SetPalette(lightPalette);
				display->Text(textRegion.left + 1, textRegion.top + 1, textRegion.right + 1,
					textRegion.bottom + 1, *stringData, -1, ZAF_HZ_CENTER, ZAF_VT_CENTER, hotKeyIndex, false);
			}

			display->SetPalette(textPalette);
			display->Text(textRegion, *stringData, -1, ZAF_HZ_CENTER, ZAF_VT_CENTER, hotKeyIndex, false);
		}

		// End the drawing operation.
		EndDraw();
		return (ccode);
	}

	// Draw "normal" buttons.  (Not check boxes or radio buttons.)
	if (ccode == L_SELECT)
	{
		if (ButtonType() != ZAF_FLAT_BUTTON)
		{
			// Draw the border/default button indicator.
			if (bordered || focus || defaultButton)
				DrawBorder(drawRegion, ccode);

			// Draw the shadow.
			DrawShadow(drawRegion, depressed ? -depth : depth, S_REDISPLAY);

			// Compute the Focus Region.
			ZafRegionStruct focusRegion = drawRegion;
			focusRegion--;

			// Erase focus rect.
			if (focus)
			{
				DrawFocus(focusRegion, S_REDISPLAY);
				focusRegion++;
			}

			// BitBlt button contents.
			if (depressed)
			{
				drawRegion.right--;
				drawRegion.bottom--;
				display->RegionCopy(drawRegion, drawRegion.left + 1, drawRegion.top + 1);
			}
			else
			{
				drawRegion.left++;
				drawRegion.top++;
				display->RegionCopy(drawRegion, drawRegion.left - 1, drawRegion.top - 1);
				display->Line(drawRegion.left, drawRegion.bottom, drawRegion.right, drawRegion.bottom);
				display->Line(drawRegion.right, drawRegion.bottom - 1, drawRegion.right, drawRegion.top);
			}

			// Restore focus rect.
			if (focus)
				DrawFocus(focusRegion, S_REDISPLAY);

			// Erase old bitblt position.
			display->SetPalette(LogicalPalette(ZAF_PM_BACKGROUND, state));
			if (depressed)
			{
				display->Line(drawRegion.left, drawRegion.top, drawRegion.left, drawRegion.bottom);
				display->Line(drawRegion.left + 1, drawRegion.top, drawRegion.right, drawRegion.top);
			}
			else
			{
				display->Line(drawRegion.left, drawRegion.bottom, drawRegion.right, drawRegion.bottom);
				display->Line(drawRegion.right, drawRegion.bottom - 1, drawRegion.right, drawRegion.top);
			}

			// Only the depressed state needed to be updated.
			EndDraw();
			return (ccode);
		}
		else
			ccode = S_REDISPLAY_DATA;
	}

//??? Optimization.
//	if (ccode == S_SHOW_DEFAULT || ccode == S_HIDE_DEFAULT)
//	{
//		if (ccode == S_SHOW_DEFAULT)
//		{
//			DrawBorder(drawRegion, S_REDISPLAY);
//			if (ButtonType() != ZAF_FLAT_BUTTON)
//				DrawShadow(drawRegion, depressed ? -depth : depth, S_REDISPLAY);
//		}
//		else
//		{
//			if (ButtonType() != ZAF_FLAT_BUTTON)
//				DrawShadow(drawRegion, depressed ? -depth : depth, S_REDISPLAY);
//			display->SetPalette(LogicalPalette(ZAF_PM_BACKGROUND, state));
//			display->Rectangle(drawRegion, 1);
//		}
//
//		// Only the default state needed to be updated.
//		EndDraw();
//		return (ccode);
//	}

	// Draw a one pixel border.
	if (defaultButton || Focus() || Bordered())
		DrawBorder(drawRegion, ccode);

	// Draw the 3D shadow.
	if (ButtonType() != ZAF_FLAT_BUTTON)
		DrawShadow(drawRegion, depressed ? -depth : depth, ccode);

	// Erase the background.
	DrawBackground(drawRegion, ccode);

	// Adjust the focus region.
	if (defaultButton || Focus() || Bordered())
		drawRegion--;
	else
		drawRegion-= 2;

	// Draw the focus rectangle.
	DrawFocus(drawRegion, ccode);

	// Set the clipping region.
	display->SetClipRegion(drawRegion);

	if (ccode == S_REDISPLAY || ccode == S_REDISPLAY_DATA || ccode == S_REDISPLAY_REGION)
	{
		// Create a depression sensation by moving drawRegion down and right.
		if (depressed && ButtonType() != ZAF_FLAT_BUTTON)
		{
			drawRegion.left++;
			drawRegion.top++;
			drawRegion.right++;
			drawRegion.bottom++;
		}

		// Set the text palette.
		ZafPaletteStruct textPalette = LogicalPalette(ZAF_PM_TEXT, state);
		display->SetPalette(textPalette);

		// Compute regions.
		ZafRegionStruct textRegion;
		if (stringData)
			textRegion = display->TextSize(*stringData);
		else
		{
			textRegion.left = textRegion.top = 1;
			textRegion.right = textRegion.bottom = 0;
		}
		ZafRegionStruct imageRegion;
		imageRegion.left = imageRegion.top = 1;
		if (bitmapData)
		{
			imageRegion.right = bitmapData->width;
			imageRegion.bottom = bitmapData->height;
		}
		else
			imageRegion.right = imageRegion.bottom = 0;
		ComputeDrawRegions(drawRegion, imageRegion, textRegion, 4);

		// Draw the bitmap.
		if (bitmapData)
			display->Bitmap(imageRegion.left, imageRegion.top, *bitmapData);

		// Draw the text.
		if (stringData)
		{
			if (Disabled())
			{
				ZafPaletteStruct darkPalette = LogicalPalette(ZAF_PM_TEXT, state);
				ZafPaletteStruct lightPalette = darkPalette;
				lightPalette.colorForeground = lightPalette.colorBackground;
				lightPalette.osPalette.colorForeground = lightPalette.osPalette.colorBackground;
				display->SetPalette(lightPalette);
				display->Text(textRegion.left + 1, textRegion.top + 1, textRegion.right + 1,
					textRegion.bottom + 1, *stringData, -1, HzJustify(), ZAF_VT_CENTER, hotKeyIndex, false);
				display->SetPalette(darkPalette);
			}
			display->Text(textRegion, *stringData, -1, ZAF_HZ_CENTER, ZAF_VT_CENTER, hotKeyIndex, false);
		}
	}

	// End the drawing operation.
	EndDraw();
	return (ccode);
}

ZafEventType ZafButton::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	// Process Windows Messages.
	if (ccode == E_OSEVENT)
		return ZafButton::OSEvent(event);
	
	switch (ccode)
	{
	case S_COMPUTE_SIZE:
		zafRegion = saveRegion;
		ccode = ZafWindowObject::Event(event);
		if (ButtonType() == ZAF_CHECK_BOX || ButtonType() == ZAF_RADIO_BUTTON)
		{
			// Get the text size.
			display->SetDisplayContext(GetDC(screenID));
			display->SetFont(LogicalPalette(ZAF_PM_TEXT, 0).font);
			ZafRegionStruct textSize;
			if (stringData)
				textSize = display->TextSize(*stringData);
			else
			{
				textSize.left = textSize.top = 1;
				textSize.right = textSize.bottom = 0;
			}
			ReleaseDC(screenID, display->RestoreDisplayContext());

			// Modify the region to include only the greater of the 
			// text height and the bitmap height while preserving the
			// original center point.
			int height = ZafMax(textSize.bottom, _check_ZafBitmap.height + 1);
			int centerY = zafRegion.top + zafRegion.Height() / 2;
			zafRegion.top = centerY - height / 2;
			zafRegion.bottom = zafRegion.top + height - 1;
		}
		else if (AutoSize())
		{
//??? should use ComputeDrawRegions().
			int height;
			if (bitmapData)
			{
				// Adjust height to fit the bitmap.
				height = bitmapData->height + 4;
				if (ButtonType() == ZAF_NATIVE_BUTTON || ButtonType() == ZAF_3D_BUTTON)
					height += ZAF_SHADOW_WIDTH * 2;
				if (stringData && (*stringData)[0] && HzJustify() == ZAF_HZ_CENTER)
					height += display->cellHeight;
			}
			else
				height = display->cellHeight * 10 / 9;
			zafRegion.top = zafRegion.bottom - height + 1;
		}

		if (editMode && allowDefault)
			zafRegion += ZAF_DEFBTN_MARGIN;
		break;

	case S_CURRENT:
	case S_NON_CURRENT:
		ccode = ZafWindowObject::Event(event);

		if (buttonType == ZAF_CHECK_BOX || buttonType == ZAF_RADIO_BUTTON ||
			DynamicPtrCast(parent, ZafWindow)->DefaultButton(true) == this)
			break;
		// else continue.
	case S_REDISPLAY_DEFAULT:
		if (SystemObject())
		{
			if (OSDrawable() && OSDraw())
			{
				DWORD dwStyle =	GetWindowLong(screenID, GWL_STYLE);
				dwStyle &= 0xFFFFFFF0L;
				if (DynamicPtrCast(parent, ZafWindow)->DefaultButton(true) == this)
					dwStyle |= BS_DEFPUSHBUTTON;
				else
					dwStyle |= BS_PUSHBUTTON;
				::SendMessage(screenID, BM_SETSTYLE, (WPARAM)LOWORD(dwStyle), MAKELPARAM(TRUE, 0));
			}
			else
//??? Optimization.
//???				Draw(ZafEventStruct(S_REDISPLAY_DEFAULT), S_REDISPLAY_DEFAULT);
				Redisplay();
		}
		break;

	case S_HOT_KEY:
		if (event.key.value == hotKeyChar)
			Event(ZafEventStruct(L_SELECT));
		else
			ccode = S_UNKNOWN;
		break;

	case L_SELECT:
		Select(this, event, L_SELECT);
		break;

	default:
		ccode = ZafWindowObject::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

// ----- Attributes ---------------------------------------------------------

bool ZafButton::SetSelected(bool setSelected)
{
	if (setSelected != selected)
	{
		ZafWindowObject::SetSelected(setSelected);

		// Update the button control.
		if (screenID && SystemObject())
		{
			if (ButtonType() == ZAF_CHECK_BOX || ButtonType() == ZAF_RADIO_BUTTON)
			{
				::SendMessage(screenID, BM_SETCHECK, selected, 0);
				if (!OSDrawable() || !OSDraw())
					Draw(ZafEventStruct(L_SELECT), L_SELECT);
			}
			else if (AllowToggling() && depressed != selected)
			{
				depressed = selected;
				Draw(ZafEventStruct(L_SELECT), L_SELECT);
			}
		}
	}

	return (selected);
}

//??? should only be used internally???
bool ZafButton::SetDepressed(bool tDepressed)
{
	// Make sure the attribute has changed.
	if (depressed != tDepressed)
	{
		// Change the attribute.
		depressed = tDepressed;

		// Update the object, if appropriate.
		if (screenID && !selected)
			Draw(S_REDISPLAY, S_REDISPLAY);
	}

	// Return the current attribute.
	return depressed;
}

// ----- OS Specific Functions ----------------------------------------------

bool ZafButton::OSDrawable()
{
	// See if the button can be drawn by windows.
	if (editMode || ButtonType() == ZAF_FLAT_BUTTON ||
		((AllowToggling() || bitmapData || userPaletteData) &&
	   	ButtonType() != ZAF_CHECK_BOX && ButtonType() != ZAF_RADIO_BUTTON))
			return (false);
	return (true);
}

ZafEventType ZafButton::OSEvent(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	switch (event.osEvent.message)
	{
#if defined(ZAF_WIN32)
	case WM_CTLCOLORBTN:
#else
	case WM_CTLCOLOR:
#endif
		{
		ccode = ZafWindowObject::OSEvent(event);

		if (ButtonType() == ZAF_CHECK_BOX || ButtonType() == ZAF_RADIO_BUTTON)
		{
			ZafPaletteStruct paletteForeground = LogicalPalette(ZAF_PM_TEXT, PaletteState());
			ZafPaletteStruct paletteBackground = parent->LogicalPalette(ZAF_PM_BACKGROUND, parent->PaletteState());
			COLORREF colorForeground = (paletteForeground.colorForeground == ZAF_CLR_DEFAULT) ?
				paletteForeground.osPalette.colorForeground :
				display->colorTable[paletteForeground.colorForeground];
			COLORREF colorBackground = (paletteBackground.colorBackground == ZAF_CLR_DEFAULT) ?
				paletteBackground.osPalette.colorBackground :
				display->colorTable[paletteBackground.colorBackground];

			SelectPalette((HDC)event.osEvent.wParam, display->logicalPalette, FALSE);
			::SetTextColor((HDC)event.osEvent.wParam, colorForeground);
			SetBkColor((HDC)event.osEvent.wParam, colorBackground);

			HBRUSH backgroundBrush = display->GetGdiCacheBrush(paletteBackground);
			display->ReleaseGdiCacheObject(backgroundBrush);
			ccode = (ZafEventType)backgroundBrush;
		}
		}
		break;

	case WM_DRAWITEM:
		{
		DRAWITEMSTRUCT *drawItemStruct = (DRAWITEMSTRUCT *)event.osEvent.lParam;

		if (drawItemStruct->CtlType == ODT_BUTTON)
		{
			// Set depressed state.
			bool oldDepressed = depressed;
			if (drawItemStruct->itemState & ODS_SELECTED)
				depressed = true;
			else if (!AllowToggling() || ButtonType() == ZAF_CHECK_BOX || ButtonType() == ZAF_RADIO_BUTTON)
				depressed = false;
			else
				depressed = selected;

			if (drawItemStruct->itemAction & ODA_SELECT &&
				AllowToggling() && depressed == oldDepressed)
				ccode = TRUE;
			else
				ccode = ZafWindowObject::OSEvent(event);
		}
		else
		{
			if (drawItemStruct->itemAction & ODA_SELECT)
			{
				drawItemStruct->itemAction &= ~ODA_SELECT;
				drawItemStruct->itemAction |= ODA_DRAWENTIRE;
			}
			ccode = ZafWindowObject::OSEvent(event);
		}
		}
		break;

	case WM_KEYDOWN:
		switch (event.osEvent.wParam)
		{
		case VK_RETURN:
			if (ButtonType() != ZAF_CHECK_BOX && ButtonType() != ZAF_RADIO_BUTTON)
				Event(L_SELECT);
			else
				ccode = S_UNKNOWN;
			break;

		case VK_SPACE:
			ccode = DefaultCallback(event);
			// Test for a reapeat key message.
			if (event.osEvent.lParam & 0x40000000L)
			{
				if  (AutoRepeatSelection())
					Select(this, event, L_SELECT);
			}
			else if (SelectOnDownClick())
				Select(this, event, L_SELECT);
			break;

		default:
			ccode = ZafWindowObject::OSEvent(event);
		}
		break;

	case WM_KEYUP:
		switch (event.osEvent.wParam)
		{
		case VK_RETURN:
			ccode = 0;
			break;

		case VK_SPACE:
			if (!SelectOnDownClick())
				Select(this, event, L_SELECT);
			ccode = DefaultCallback(event);
			break;

		default:
			ccode = ZafWindowObject::OSEvent(event);
		}
		break;

	case WM_LBUTTONDOWN:
		if (Noncurrent())
		{
			SetCapture(screenID);
			::SendMessage(screenID, BM_SETSTATE, TRUE, 0);
		}
		else
			ccode = ZafWindowObject::Event(event);
		if (SelectOnDownClick())
			Select(this, event, L_SELECT);
		if (AutoRepeatSelection())
			SetTimer(screenID, ZAF_BUTTON_TIMER_ID, InitialDelay(), NULL);
		break;

	case WM_LBUTTONDBLCLK:
		ccode = ::SendMessage(screenID, WM_LBUTTONDOWN,	event.osEvent.wParam, event.osEvent.lParam);
		if (SelectOnDoubleClick())
			Select(this, event, L_DOUBLE_CLICK);
		break;

	case WM_LBUTTONUP:
		if (AutoRepeatSelection())
	  		KillTimer(screenID, ZAF_BUTTON_TIMER_ID);
		if (!SelectOnDownClick())
		{
			// See if button is in a "highlighted" (depressed) state, 
			// indicating that the mouse has not moved off of the button.
			if (::SendMessage(screenID, BM_GETSTATE, 0, 0) & 0x0004)
				Select(this, event, L_SELECT);
		}
		if (Noncurrent())
		{
			ReleaseCapture();
			::SendMessage(screenID, BM_SETSTATE, FALSE, 0);
		}
		else
			ccode = ZafWindowObject::Event(event);
		break;

	case WM_MOUSEMOVE:
		if (Noncurrent() && GetCapture() == screenID && event.osEvent.wParam & MK_LBUTTON)
		{
			ZafPositionStruct position;
			position.column = zafRegion.left + LOWORD(event.osEvent.lParam);
			position.line = zafRegion.top + HIWORD(event.osEvent.lParam);
				
			if (zafRegion.Overlap(position))
				::SendMessage(screenID, BM_SETSTATE, TRUE, 0);
			else
				::SendMessage(screenID, BM_SETSTATE, FALSE, 0);
		}
		else
		{
			ccode = ZafWindowObject::Event(event);
			if (windowManager->dragObject)
				ccode = DefaultCallback(event);
		}
		break;

	case WM_PAINT:
		// Draw() gets called by WM_DRAWITEM messages.
		ccode = DefaultCallback(event);
		break;

	case WM_ERASEBKGND:
		if (OSDraw() && OSDrawable())
			ccode = DefaultCallback(event);
		else
			ccode = ZafWindowObject::Event(event);
		break;

	case WM_TIMER:
		// See if button is in a "highlighted" (depressed) state, 
		// indicating that the mouse has not moved off of the button.
		if (AutoRepeatSelection() && (::SendMessage(screenID, BM_GETSTATE, 0, 0) & 0x0004))
		{
			Select(this, event, L_SELECT);
			SetTimer(screenID, ZAF_BUTTON_TIMER_ID, RepeatDelay(), NULL);
		}
		ccode = 0;
		break;

	default:
		ccode = ZafWindowObject::OSEvent(event);
		break;
	}

	return (ccode);
}

void ZafButton::OSMapPalette(ZafPaletteStruct &palette, ZafPaletteType type, ZafPaletteState state)
{
	// Get default colors from the operating system.
	if (type == ZAF_PM_TEXT)
	{
		if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
		{
			if (state & ZAF_PM_DISABLED)
				palette.osPalette.colorForeground = GetSysColor(COLOR_3DSHADOW);
			else
				palette.osPalette.colorForeground = GetSysColor(COLOR_BTNTEXT);
		}
		if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
		{
			if (state & ZAF_PM_DISABLED)
				palette.osPalette.colorBackground = GetSysColor(COLOR_3DHILIGHT);
			else
				palette.osPalette.colorBackground = GetSysColor(COLOR_3DFACE);
		}
	}
	else if (type == ZAF_PM_BACKGROUND)
	{
		if (ButtonType() == ZAF_CHECK_BOX || ButtonType() == ZAF_RADIO_BUTTON)
		{
			ZafPaletteStruct parentPalette = parent->LogicalPalette(type, state);
			if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
				palette.osPalette.colorForeground = (parentPalette.colorForeground == ZAF_CLR_DEFAULT) ?
					parentPalette.osPalette.colorForeground : display->colorTable[parentPalette.colorForeground];
			if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
				palette.osPalette.colorBackground = (parentPalette.colorBackground == ZAF_CLR_DEFAULT) ?
					parentPalette.osPalette.colorBackground : display->colorTable[parentPalette.colorBackground];
		}
		else
		{
			if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
				palette.osPalette.colorForeground = GetSysColor(COLOR_3DFACE);
			if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
				palette.osPalette.colorBackground = GetSysColor(COLOR_3DFACE);
		}
	}
	else if (type == ZAF_PM_FOCUS)
	{
		if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
			palette.osPalette.colorForeground = GetSysColor(COLOR_3DFACE);
	}
	else if (focus && depressed && (type == ZAF_PM_LIGHT_SHADOW || type == ZAF_PM_DARK_SHADOW))
	{
		if (palette.colorForeground == ZAF_CLR_DEFAULT && palette.osPalette.colorForeground == 0xFFFFFFFFL)
		{
			if (type == ZAF_PM_LIGHT_SHADOW)
				palette.osPalette.colorForeground = GetSysColor(COLOR_3DSHADOW);
			else if (type == ZAF_PM_DARK_SHADOW)
				palette.osPalette.colorForeground = GetSysColor(COLOR_3DSHADOW);
		}
		if (palette.colorBackground == ZAF_CLR_DEFAULT && palette.osPalette.colorBackground == 0xFFFFFFFFL)
		{
			if (type == ZAF_PM_LIGHT_SHADOW)
				palette.osPalette.colorBackground = GetSysColor(COLOR_3DFACE);
			else if (type == ZAF_PM_DARK_SHADOW)
				palette.osPalette.colorBackground = GetSysColor(COLOR_3DFACE);
		}
	}
	else
		ZafWindowObject::OSMapPalette(palette, type, state);
}

void ZafButton::OSRegisterObject(void)
{
	// Set the Windows styles.
	DWORD dwStyle = WS_CHILD | WS_CLIPSIBLINGS;

	if (!OSDraw() || !OSDrawable())
		dwStyle |= BS_OWNERDRAW;
	else
	{
		if (ButtonType() == ZAF_CHECK_BOX)
			dwStyle |= BS_CHECKBOX;
		else if (ButtonType() == ZAF_RADIO_BUTTON)
			dwStyle |= BS_RADIOBUTTON;
		else
			dwStyle |= BS_PUSHBUTTON;

		if (HzJustify() == ZAF_HZ_RIGHT)
			dwStyle |= BS_LEFTTEXT;
	}
	if (disabled && !editMode)
		dwStyle |= WS_DISABLED;
	if (visible)
		dwStyle |= WS_VISIBLE;
	if (Bordered() && OSDraw() && OSDrawable())
		dwStyle |= WS_BORDER;

	// Create the control.
	screenID = ZafMSWindowsApp::CreateSubclassedWindow(this, ZAF_ITEXT("BUTTON"), dwStyle);

	// Set the font.
	ZafPaletteStruct fontPalette = LogicalPalette(ZAF_PM_TEXT, ZAF_PM_ACTIVE | ZAF_PM_ENABLED);
	::SendMessage(screenID, WM_SETFONT, (WPARAM)display->fontTable[fontPalette.font], (LPARAM)FALSE);

	// Set the text.
	OSSetText();

	// Initialize selection state.
	if (ButtonType() == ZAF_CHECK_BOX || ButtonType() == ZAF_RADIO_BUTTON)
		::SendMessage(screenID, BM_SETCHECK, selected, 0);
}

ZafError ZafButton::OSSetImage(void)
{
	if (screenID)
	{
		Event(ZafEventStruct(S_DESTROY));
		Event(ZafEventStruct(S_CREATE));
	}

	return ZAF_ERROR_NONE;
}


ZafError ZafButton::OSSetText(void)
{
	if (screenID)
	{
		if (SystemObject() && OSDraw() && OSDrawable())
		{
			const ZafIChar *drawText = stringData ?
				stringData->Text() : ZafLanguageData::blankString;
			ZafIChar *hotKeyText = ZAF_NULLP(ZafIChar);

			// Fix ampersand characters so that they don't draw as hot keys.
			const ZafIChar *ampersand = strchr(drawText, '&');
			int hotKeyDelta = 0;
			if (ampersand)
			{
				// Count the '&' characters.
				int numAmpersands = 0;
				while (ampersand)
				{
					numAmpersands++;
					ampersand = strchr(ampersand + 1, '&');
				}

				hotKeyText = new ZafIChar[strlen(drawText) + numAmpersands + 1];

				int hotKeyTextIndex = 0;
				const ZafIChar *sourceText = drawText;
				ampersand = strchr(sourceText, '&');
				strncpy(&hotKeyText[hotKeyTextIndex], sourceText, (int)(ampersand - sourceText + 1));
				hotKeyTextIndex += (int)(ampersand - sourceText + 1);
				while (ampersand)
				{
					if (hotKeyIndex > -1 && ampersand - sourceText <= hotKeyIndex)
						hotKeyDelta++;

					hotKeyText[hotKeyTextIndex++] = '&';
					const ZafIChar *nextAmpersand = strchr(ampersand + 1, '&');
					if (nextAmpersand)
					{
						strncpy(&hotKeyText[hotKeyTextIndex], ampersand + 1,
							(int)(nextAmpersand - ampersand));
						hotKeyTextIndex += (int)(nextAmpersand - ampersand);
					}
					else
						strcpy(&hotKeyText[hotKeyTextIndex], ampersand + 1);
					ampersand = nextAmpersand;
				}

				drawText = hotKeyText;
			}

			if (hotKeyIndex > -1 && hotKeyIndex < strlen(drawText))
			{
				ZafIChar *tempText = hotKeyText;
				hotKeyText = new ZafIChar[strlen(drawText) + 2];
				strncpy(hotKeyText, drawText, hotKeyIndex + hotKeyDelta);
				hotKeyText[hotKeyIndex + hotKeyDelta] = '&';
				strcpy(&hotKeyText[hotKeyIndex + hotKeyDelta +1], &drawText[hotKeyIndex + hotKeyDelta]);
				drawText = hotKeyText;

				if (tempText)
					delete tempText;
			}

			if (ZafMSWindowsApp::convertText)
			{
				char *osText = zafCodeSet->ConvertToOSString(drawText);
#if defined(ZAF_WIN32) && defined(ZAF_UNICODE)
				::SendMessageA(screenID, WM_SETTEXT, (WPARAM)0, (LPARAM)osText);
#else
				::SendMessage(screenID, WM_SETTEXT, (WPARAM)0, (LPARAM)osText);
#endif
				delete []osText;
			}
			else
				::SendMessage(screenID, WM_SETTEXT, (WPARAM)0, (LPARAM)drawText);

			if (hotKeyText)
				delete []hotKeyText;
		}
		else
//??? Doesn't work if button is in a list.
//			Event(ZafEventStruct(S_REDISPLAY_DATA));
			Event(ZafEventStruct(S_REDISPLAY));
	}

	return ZAF_ERROR_NONE;
}

ZafError ZafButton::OSUpdatePalettes(ZafPaletteType zafTypes, ZafPaletteType osTypes)
{
	if (osDraw && screenID && systemObject)
	{
		// Determine whethor the button is to be drawn by the Windows or ZAF.
	 	LONG dwStyle = GetWindowLong(screenID, GWL_STYLE);
		bool ownerDraw = (dwStyle & BS_OWNERDRAW) ? true : false;
		bool osDrawable = OSDrawable();
		if (ownerDraw != osDrawable)
		{
			Event(S_DESTROY);
			Event(S_CREATE);
			return (ZAF_ERROR_NONE);
		}
	}

    // Defer to the base class.
   	return (ZafWindowObject::OSUpdatePalettes(zafTypes, osTypes));
}

// ----- Stubs --------------------------------------------------------------

ZafEventType ZafButton::DrawFocus(ZafRegionStruct &region, ZafEventType ccode)
{
	return (ZafWindowObject::DrawFocus(region, ccode));
}

ZafEventType ZafButton::DrawShadow(ZafRegionStruct &region, int depth, ZafEventType ccode)
{
	return (ZafWindowObject::DrawShadow(region, depth, ccode));
}

void ZafButton::OSSize(void)
{
	ZafWindowObject::OSSize();
}

bool ZafButton::SetVisible(bool setVisible)
{
	return (ZafWindowObject::SetVisible(setVisible));
}
