//	Zinc Application Framework - C_FILE.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include "c_file.hpp"
#include <z_string.hpp>
#include <z_utils.hpp>

#if defined(ZAF_MACINTOSH)
#	include <a_utils.hpp>

	OSType OldFile::fileCreator = 'Anon';
	OSType OldFile::fileType = 'TEXT';
#endif

// ----- OldFile ------------------------------------------------------------

OldFile::OldFile(const ZafIChar *tPathName, ZafFileMode zMode) :
	error(ZAF_ERROR_NONE), mode(zMode), file(0), version(0), useCount(0)
{
	// Set the pathname.
	pathName = tPathName ? strdup(tPathName) : ZAF_NULLP(ZafIChar);

	// Check the mode.
	if (DerivedAccess())
		return;

#if defined(ZAF_MACINTOSH)
	file = MacOpen(pathName, fileCreator, fileType, zMode);
#else
	// Determine the mode.
	ZafIChar openMode[6];
	if (OpenCreate())
		strcpy(openMode, ZAF_ITEXT("a+"));
	else if (Create())
		strcpy(openMode, ZAF_ITEXT("w+"));
	else if (ReadWrite())
		strcpy(openMode, ZAF_ITEXT("r+"));
	else
		strcpy(openMode, ZAF_ITEXT("r"));

	if (BinaryMode())
		strcat(openMode, ZAF_ITEXT("b"));
	else if (TextMode())
		strcat(openMode, ZAF_ITEXT("t"));

	// Open the file.
	file = fopen(pathName, openMode);
#endif
}

OldFile::~OldFile(void)
{
	// Delete the file.
	if (!DerivedAccess() && file)
#if defined(ZAF_MACINTOSH)
		FSClose(file);
#else
		fclose(file);
#endif

	// Delete the pathname.
	if (pathName)
		delete []pathName;
}

ZafOffset OldFile::GetLength(void) const
{
	// Return the file length.
#if defined(ZAF_MACINTOSH)
	long position = 0L;
	GetEOF(file, &position);
	return (position);
#else
	return (file ? ftell(file) : -1);
#endif
}

ZafError OldFile::Rename(const ZafIChar *newPathName)
{
	// Rename the file.
#if defined(ZAF_MACINTOSH)
	Str255 fName, tName;
#	if defined(ZAF_UNICODE) || defined(ZAF_ISO8859_1)
	zafDefaultCharMap->ConvertToOSString(pathName, (char *)fName, false);
	zafDefaultCharMap->ConvertToOSString(newPathName, (char *)tName, false);
#	else
	strcpy((char *)fName, pathName);
	strcpy((char *)tName, newPathName);
#	endif
	c2pstr((char *)fName);
	c2pstr((char *)tName);
	OSErr result = HRename(0, 0L, fName, tName);
#else
	int result = ::rename(pathName, newPathName);
#endif

	delete []pathName;
	pathName = strdup(newPathName);
	return (result ? ZAF_ERROR_INVALID : ZAF_ERROR_NONE);
}

ZafError OldFile::Seek(ZafOffset offset, ZafSeek location)
{
	// Seek into the file.
	if (file)
	{
#if defined(ZAF_MACINTOSH)
		short fromLoc = fsFromStart;
		switch (location)
		{
		case ZAF_SEEK_START:
			fromLoc = fsFromStart;
			break;
		case ZAF_SEEK_CURRENT:
			fromLoc = fsFromMark;
			break;
		case ZAF_SEEK_END:
			fromLoc = fsFromLEOF;
			break;
		}
		return (SetFPos(file, fromLoc, offset) == noErr ? ZAF_ERROR_NONE : ZAF_ERROR_INVALID);
#else
		return (fseek(file, offset, (int)location) ? ZAF_ERROR_INVALID : ZAF_ERROR_NONE);
#endif
	}
	return (ZAF_ERROR_INVALID);
}

ZafError OldFile::SetError(ZafError newError)
{
	// Reset the error.
	error = newError;
	return (error);
}

ZafVersion OldFile::SetVersion(ZafVersion newVersion)
{
	// Reset the version number.
	version = newVersion;
	return (version);
}

ZafOffset OldFile::Tell(void) const
{
	// Tell the current offset.
#if defined(ZAF_MACINTOSH)
	long position;
	GetFPos(file, &position);
	return (position);
#else
	return (file ? ftell(file) : -1);
#endif
}

// ----- Read Functions -----------------------------------------------------

int OldFile::Read(ZafInt8 &value)
{
	return (Read(&value, sizeof(value), 1));
}

int OldFile::Read(ZafUInt8 &value)
{
	return (Read((char *)&value, sizeof(value), 1));
}

#if (ZAF_WORD_SIZE != 64)
int OldFile::Read(int &value)
{
	ZafInt32 temp;
	int ccode = Read(temp);
	value = (int)temp;
	return (ccode);
}
#endif

#if defined(ZAF_BIGENDIAN)
int OldFile::Read(ZafInt16 &value)
{
	int i = (Read((char *)&value, sizeof(value), 1));
	SwapInt16(&value);
	return (i);
}

int OldFile::Read(ZafUInt16 &value)
{
	int i = (Read((char *)&value, sizeof(value), 1));
	SwapUInt16(&value);
	return (i);
}

int OldFile::Read(ZafInt32 &value)
{
	int i = (Read((char *)&value, sizeof(value), 1));
	SwapInt32(&value);
	return (i);
}

int OldFile::Read(ZafUInt32 &value)
{
	int i = (Read((char *)&value, sizeof(value), 1));
	SwapUInt32(&value);
	return (i);
}

#else
int OldFile::Read(ZafInt16 &value)
{
	return (Read((char *)&value, sizeof(value), 1));
}

int OldFile::Read(ZafUInt16 &value)
{
	return (Read((char *)&value, sizeof(value), 1));
}

int OldFile::Read(ZafInt32 &value)
{
	return (Read((char *)&value, sizeof(value), 1));
}

int OldFile::Read(ZafUInt32 &value)
{
	return (Read((char *)&value, sizeof(value), 1));
}
#endif

int OldFile::Read(ZafIChar *text, int maxLength)
{
	// Note: The argument maxLength is the maximum number of ZafIChars 
	// to Read, whereas the return int is the number of bytes Read.

	*text = '\0';
	ZafUInt16 size;
	int cnt1 = Read(size);
	if (cnt1 != sizeof(size))
		return (cnt1);
	// Unicode support can load both Unicode and ISO 8859-1 strings
	// without data loss.  The upper 128 characters of OEM strings
	// are bogus in Unicode mode.  ISO and OEM support can load the
	// first 256 or 128 Unicode characters respectivly.  Both loose
	// information outside this range.
	int doUni = (size & ZAF_UNICODE_FLAG);
	size &= ~ZAF_UNICODE_FLAG;
	if (maxLength != -1 && size > maxLength)
		return (-1);
	int cnt2 = 0;
	if (size)
#if defined(ZAF_UNICODE)
		if (doUni)		// Byte Swap each short
		{
			for (int i = 0; i < size; i++)
			{
				ZafUInt16 c;
				cnt2 += Read(c);
				text[i] = c;
			}
		}
		else
		{
			for (int i = 0; i < size; i++)
			{
				ZafUInt8 tmp;
				cnt2 += Read(tmp);
				text[i] = tmp;
			}
		}
#else
		if (doUni)
			for (int i = 0; i < size; i++)
			{
				ZafUInt16 tmp;
				cnt2 += Read(tmp);
				text[i] = (char)tmp;
			}
		else
			cnt2 += Read(text, sizeof(ZafIChar), size);
#endif
	text[size] = '\0';
	return (cnt2 != (size * sizeof(text[0])) ? cnt2 : cnt1 + cnt2);
}

int OldFile::Read(ZafIChar **text)
{
	ZafUInt16 size;
	int cnt1 = Read(size);
	int cnt2 = 0;
	int doUni = (size & ZAF_UNICODE_FLAG);
	size &= ~ZAF_UNICODE_FLAG;
	if (cnt1 == sizeof(size) && size)
	{
		// See Unicode related comments in previous function.
		*text = new ZafIChar[size + 1];
#if defined(ZAF_UNICODE)
		if (doUni)		// Byte Swap each short
		{
			for (int i = 0; i < size; i++)
			{
				ZafUInt16 c;
				cnt2 += Read(c);
				(*text)[i] = c;
			}
		}
		else
		{
			for (int i = 0; i < size; i++)
			{
				ZafUInt8 tmp;
				cnt2 += Read(tmp);
				(*text)[i] = tmp;
			}
		}
#else
		if (doUni)
			for (int i = 0; i < size; i++)
			{
				ZafUInt16 tmp;
				cnt2 += Read(tmp);
				(*text)[i] = (char)tmp;
			}
		else
			cnt2 += Read(*text, sizeof(ZafIChar), size);
#endif
		(*text)[size] = '\0';
	}
	else
		*text = ZAF_NULLP(ZafIChar);
	return (cnt2 != (size * sizeof((*text)[0])) ? cnt2 : cnt1 + cnt2);
}

int OldFile::Read(void *buffer, int size, int num)
{
#if defined(ZAF_MACINTOSH)
	long count = size * num;
	OSErr theError = FSRead(file, &count, (Ptr)buffer);
	if (theError == noErr || theError == eofErr)
		return ((int)count);
	else
		return (0);
#else
	return (size * fread((char *)buffer, size, num, file));
#endif
}

// ----- Write Functions ----------------------------------------------------

int OldFile::Write(ZafInt8 value)
{
	return (Write(&value, sizeof(value), 1));
}

int OldFile::Write(ZafUInt8 value)
{
	return (Write((char *)&value, sizeof(value), 1));
}

#if (ZAF_WORD_SIZE != 64)
int OldFile::Write(int value)
{
	ZafInt32 temp = (ZafInt16)value;
	return (Write(temp));
}
#endif

#if defined(ZAF_BIGENDIAN)
int OldFile::Write(ZafInt16 value)
{
	SwapInt16(&value);
	return (Write((char *)&value, sizeof(value), 1));
}

int OldFile::Write(ZafUInt16 value)
{
	SwapUInt16(&value);
	return (Write((char *)&value, sizeof(value), 1));
}

int OldFile::Write(ZafInt32 value)
{
	SwapInt32(&value);
	return (Write((char *)&value, sizeof(value), 1));
}

int OldFile::Write(ZafUInt32 value)
{
	SwapUInt32(&value);
	return (Write((char *)&value, sizeof(value), 1));
}

int OldFile::Write(const ZafIChar *text)
{
#if defined(ZAF_UNICODE)
	ZafInt16 length = (text ? strlen(text) : 0) | ZAF_UNICODE_FLAG;
	Write(length);
	int j = 0;
	for (int i=0; i < length; i++)
		j += Write((ZafUInt16)text[i]);
	return (j);
#else
	ZafInt16 length = (text ? strlen(text) : 0);
	Write(length);
	int j = 0;
	for (int i=0; i < length; i++)
		j += Write(text[i]);
	return (j);
#endif
}
#else
int OldFile::Write(ZafInt16 value)
{
	return (Write((char *)&value, sizeof(value), 1));
}

int OldFile::Write(ZafUInt16 value)
{
	return (Write((char *)&value, sizeof(value), 1));
}

int OldFile::Write(ZafInt32 value)
{
	return (Write((char *)&value, sizeof(value), 1));
}

int OldFile::Write(ZafUInt32 value)
{
	return (Write((char *)&value, sizeof(value), 1));
}

int OldFile::Write(const ZafIChar *text)
{
#if defined(ZAF_UNICODE)
	ZafInt16 length = (ZafInt16)(text ? strlen(text) : 0) | ZAF_UNICODE_FLAG;
#else
	ZafInt16 length = (ZafInt16)(text ? strlen(text) : 0);
#endif
	Write(length);
	return (length ? Write((void *)text, sizeof(ZafIChar), length) : 0);
}
#endif

int OldFile::Write(void *buffer, int size, int num)
{
#if defined(ZAF_MACINTOSH)
	long count = size * num;
	if (FSWrite(file, &count, (Ptr)buffer) == noErr)
		return ((int)count);
	else
		return (0);
#else
	return (size * fwrite((char *)buffer, size, num, file));
#endif
}

