//	Zinc Application Framework - MAINTEST.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include "maintest.hpp"

// This table contains the list of all events that can be received by an 
// object.  EventLog.count gets updated by GenericTestWin::LogEvent().
// This list is displayed when the tester selects the "Event Log" button.
ListEntry ZAF_FARDATA GenericTestWin::EventLog[] =
{
	{ ZAF_ITEXT("S_ERROR"),              0, S_ERROR},
	{ ZAF_ITEXT("S_UNKNOWN"),            0, S_UNKNOWN},
	{ ZAF_ITEXT("S_NO_OBJECT"),          0, S_NO_OBJECT},
	{ ZAF_ITEXT("S_MINIMIZE"),           0, S_MINIMIZE},
	{ ZAF_ITEXT("S_MAXIMIZE"),           0, S_MAXIMIZE},
	{ ZAF_ITEXT("S_CONTINUE"),           0, S_CONTINUE},
	{ ZAF_ITEXT("S_INITIALIZE"),         0, S_INITIALIZE},
	{ ZAF_ITEXT("S_CREATE"),             0, S_CREATE},
	{ ZAF_ITEXT("S_CLOSE"),              0, S_CLOSE},
	{ ZAF_ITEXT("S_NON_CURRENT"),        0, S_NON_CURRENT},
	{ ZAF_ITEXT("S_REDISPLAY_DATA"),     0, S_REDISPLAY_DATA},
	{ ZAF_ITEXT("S_REDISPLAY"),          0, S_REDISPLAY},
	{ ZAF_ITEXT("S_CLOSE_TEMPORARY"),    0, S_CLOSE_TEMPORARY},
	{ ZAF_ITEXT("S_DEINITIALIZE"),       0, S_DEINITIALIZE},
	{ ZAF_ITEXT("S_RESTORE"),            0, S_RESTORE},
	{ ZAF_ITEXT("S_REGISTER_OBJECT"),    0, S_REGISTER_OBJECT},
	{ ZAF_ITEXT("S_DESTROY"),            0, S_DESTROY},
	{ ZAF_ITEXT("S_REDISPLAY_REGION"),   0, S_REDISPLAY_REGION},
	{ ZAF_ITEXT("S_HOT_KEY"),            0, S_HOT_KEY},
	{ ZAF_ITEXT("S_SIZE"),               0, S_SIZE},
	{ ZAF_ITEXT("S_COMPUTE_SIZE"),       0, S_COMPUTE_SIZE},
	{ ZAF_ITEXT("S_CURRENT"),            0, S_CURRENT},
	{ ZAF_ITEXT("S_VSCROLL"),            0, S_VSCROLL},
	{ ZAF_ITEXT("S_HSCROLL"),            0, S_HSCROLL},
	{ ZAF_ITEXT("S_VSCROLL_SET"),        0, S_VSCROLL_SET},
	{ ZAF_ITEXT("S_HSCROLL_SET"),        0, S_HSCROLL_SET},
	{ ZAF_ITEXT("S_VSCROLL_CHECK"),      0, S_VSCROLL_CHECK},
	{ ZAF_ITEXT("S_HSCROLL_CHECK"),      0, S_HSCROLL_CHECK},
	{ ZAF_ITEXT("S_VSCROLL_COMPUTE"),    0, S_VSCROLL_COMPUTE},
	{ ZAF_ITEXT("S_HSCROLL_COMPUTE"),    0, S_HSCROLL_COMPUTE},
	{ ZAF_ITEXT("S_ADD_OBJECT"),         0, S_ADD_OBJECT},
	{ ZAF_ITEXT("S_SUBTRACT_OBJECT"),    0, S_SUBTRACT_OBJECT},
	{ ZAF_ITEXT("S_DRAG_MOVE"),          0, S_DRAG_MOVE},
	{ ZAF_ITEXT("S_DRAG_COPY"),          0, S_DRAG_COPY},
	{ ZAF_ITEXT("S_DRAG_LINK"),          0, S_DRAG_LINK},
	{ ZAF_ITEXT("S_DROP_LINK"),          0, S_DROP_LINK},
	{ ZAF_ITEXT("S_DRAG_DEFAULT"),       0, S_DRAG_DEFAULT},
	{ ZAF_ITEXT("S_DROP_DEFAULT"),       0, S_DROP_DEFAULT},
	{ ZAF_ITEXT("S_COPY_DATA"),          0, S_COPY_DATA},
	{ ZAF_ITEXT("S_SET_DATA"),           0, S_SET_DATA},
	{ ZAF_ITEXT("S_MDI_CASCADE_WINDOWS"),0, S_MDI_CASCADE_WINDOWS},
	{ ZAF_ITEXT("S_MDI_CLOSE"),          0, S_MDI_CLOSE},
	{ ZAF_ITEXT("S_MDI_MAXIMIZE"),       0, S_MDI_MAXIMIZE},
	{ ZAF_ITEXT("S_MDI_MINIMIZE"),       0, S_MDI_MINIMIZE},
	{ ZAF_ITEXT("S_MDI_RESTORE"),        0, S_MDI_RESTORE},
	{ ZAF_ITEXT("S_MDI_TILE_WINDOWS"),   0, S_MDI_TILE_WINDOWS},
	{ ZAF_ITEXT("N_CLOSE"),              0, N_CLOSE},
	{ ZAF_ITEXT("N_EXIT"),               0, N_EXIT},
	{ ZAF_ITEXT("N_CURRENT"),            0, N_CURRENT},
	{ ZAF_ITEXT("N_NON_CURRENT"),        0, N_NON_CURRENT},
	{ ZAF_ITEXT("N_SIZE"),               0, N_SIZE},
	{ ZAF_ITEXT("N_MOVE"),               0, N_MOVE},
	{ ZAF_ITEXT("N_VSCROLL"),            0, N_VSCROLL},
	{ ZAF_ITEXT("N_HSCROLL"),            0, N_HSCROLL},
	{ ZAF_ITEXT("N_CHANGE_PAGE"),        0, N_CHANGE_PAGE},
	{ ZAF_ITEXT("L_ALT_KEY"),            0, L_ALT_KEY},
	{ ZAF_ITEXT("L_EXIT"),               0, L_EXIT},
	{ ZAF_ITEXT("L_VIEW"),               0, L_VIEW},
	{ ZAF_ITEXT("L_SELECT"),             0, L_SELECT},
	{ ZAF_ITEXT("L_BEGIN_SELECT"),       0, L_BEGIN_SELECT},
	{ ZAF_ITEXT("L_CONTINUE_SELECT"),    0, L_CONTINUE_SELECT},
	{ ZAF_ITEXT("L_END_SELECT"),         0, L_END_SELECT},
	{ ZAF_ITEXT("L_BEGIN_ESCAPE"),       0, L_BEGIN_ESCAPE},
	{ ZAF_ITEXT("L_CONTINUE_ESCAPE"),    0, L_CONTINUE_ESCAPE},
	{ ZAF_ITEXT("L_END_ESCAPE"),         0, L_END_ESCAPE},
	{ ZAF_ITEXT("L_HELP"),               0, L_HELP},
	{ ZAF_ITEXT("L_CANCEL"),             0, L_CANCEL},
	{ ZAF_ITEXT("L_DOUBLE_CLICK"),       0, L_DOUBLE_CLICK},
	{ ZAF_ITEXT("L_MOVE_MODE"),          0, L_MOVE_MODE},
	{ ZAF_ITEXT("L_SIZE_MODE"),          0, L_SIZE_MODE},
	{ ZAF_ITEXT("L_RESTORE"),            0, L_RESTORE},
	{ ZAF_ITEXT("L_MINIMIZE"),           0, L_MINIMIZE},
	{ ZAF_ITEXT("L_MAXIMIZE"),           0, L_MAXIMIZE},
	{ ZAF_ITEXT("L_SYSTEM_MENU"),        0, L_SYSTEM_MENU},
	{ ZAF_ITEXT("L_NEXT_WINDOW"),        0, L_NEXT_WINDOW},
	{ ZAF_ITEXT("L_UP"),                 0, L_UP},
	{ ZAF_ITEXT("L_DOWN"),               0, L_DOWN},
	{ ZAF_ITEXT("L_LEFT"),               0, L_LEFT},
	{ ZAF_ITEXT("L_RIGHT"),              0, L_RIGHT},
	{ ZAF_ITEXT("L_PREVIOUS"),           0, L_PREVIOUS},
	{ ZAF_ITEXT("L_NEXT"),               0, L_NEXT},
	{ ZAF_ITEXT("L_FIRST"),              0, L_FIRST},
	{ ZAF_ITEXT("L_LAST"),               0, L_LAST},
	{ ZAF_ITEXT("L_PGUP"),               0, L_PGUP},
	{ ZAF_ITEXT("L_PGDN"),               0, L_PGDN},
	{ ZAF_ITEXT("L_EXTEND_PREVIOUS"),    0, L_EXTEND_PREVIOUS},
	{ ZAF_ITEXT("L_EXTEND_NEXT"),        0, L_EXTEND_NEXT},
	{ ZAF_ITEXT("L_EXTEND_FIRST"),       0, L_EXTEND_FIRST},
	{ ZAF_ITEXT("L_EXTEND_LAST"),        0, L_EXTEND_LAST},
	{ ZAF_ITEXT("L_CUT"),                0, L_CUT},
	{ ZAF_ITEXT("L_PASTE"),              0, L_PASTE},
	{ ZAF_ITEXT("L_DELETE"),             0, L_DELETE},
	{ ZAF_ITEXT("L_DELETE_WORD"),        0, L_DELETE_WORD},
	{ ZAF_ITEXT("L_DELETE_EOL"),         0, L_DELETE_EOL},
	{ ZAF_ITEXT("L_INSERT_TOGGLE"),      0, L_INSERT_TOGGLE},
	{ ZAF_ITEXT("L_WORD_LEFT"),          0, L_WORD_LEFT},
	{ ZAF_ITEXT("L_WORD_RIGHT"),         0, L_WORD_RIGHT},
	{ ZAF_ITEXT("L_BOL"),                0, L_BOL},
	{ ZAF_ITEXT("L_EOL"),                0, L_EOL},
	{ ZAF_ITEXT("L_BACKSPACE"),          0, L_BACKSPACE},
	{ ZAF_ITEXT("L_COPY"),               0, L_COPY},
	{ ZAF_ITEXT("L_MARK_LEFT"),          0, L_MARK_LEFT},
	{ ZAF_ITEXT("L_MARK_RIGHT"),         0, L_MARK_RIGHT},
	{ ZAF_ITEXT("L_MARK_UP"),            0, L_MARK_UP},
	{ ZAF_ITEXT("L_MARK_DOWN"),          0, L_MARK_DOWN},
	{ ZAF_ITEXT("L_MARK_BOL"),           0, L_MARK_BOL},
	{ ZAF_ITEXT("L_MARK_EOL"),           0, L_MARK_EOL},
	{ ZAF_ITEXT("L_MARK_PGUP"),          0, L_MARK_PGUP},
	{ ZAF_ITEXT("L_MARK_PGDN"),          0, L_MARK_PGDN},
	{ ZAF_ITEXT("L_MARK_WORD_RIGHT"),    0, L_MARK_WORD_RIGHT},
	{ ZAF_ITEXT("L_MARK_WORD_LEFT"),     0, L_MARK_WORD_LEFT},
	{ ZAF_ITEXT("S_DECREMENT"),          0, S_DECREMENT},
	{ ZAF_ITEXT("S_INCREMENT"),          0, S_INCREMENT},
	{ ZAF_ITEXT("L_MDI_NEXT_WINDOW"),    0, L_MDI_NEXT_WINDOW},
	{ ZAF_ITEXT("L_MDI_MOVE_MODE"),      0, L_MDI_MOVE_MODE},
	{ ZAF_ITEXT("L_MDI_SIZE_MODE"),      0, L_MDI_SIZE_MODE},
	{ ZAF_ITEXT("L_NONE"),               0, L_NONE},
	{ ZAF_ITEXT("A_OPEN_DOCUMENT"),      0, A_OPEN_DOCUMENT},
	{ ZAF_ITEXT("A_PRINT_DOCUMENT"),     0, A_PRINT_DOCUMENT},
	{ ZAF_ITEXT("A_OPEN_WINDOW"),        0, A_OPEN_WINDOW},
	{ ZAF_ITEXT("A_CLOSE_WINDOW"),       0, A_CLOSE_WINDOW},
	{ ZAF_ITEXT("A_CHANGE_LANG_LOC"),    0, A_CHANGE_LANG_LOC},
	{ ZAF_ITEXT("A_MINIMIZE_WINDOWS"),   0, A_MINIMIZE_WINDOWS},
	{ ZAF_ITEXT("A_RESTORE_WINDOWS"),    0, A_RESTORE_WINDOWS},
	{ ZAF_NULLP(ZafIChar),               0, 0 }
};

ZafHelpTips *helpTip = 0;

GenericTestWin::GenericTestWin(int left, int top, int width, int height) :
	ZafWindow(left, top, width, height),
	attrList(0), testObject(0), testObjectParent(0), testObjectSibling(0)
{
	SetSelectionType(ZAF_MULTIPLE_SELECTION);
	SetOSDraw(false);
	SetStringID(ZAF_ITEXT("TEST_WINDOW"));
	Add(new ZafBorder);
	Add(new ZafMaximizeButton);
	Add(new ZafMinimizeButton);
	Add(new ZafSystemButton);
	Add(new ZafTitle(new ZafStringData("")));

	ZafStatusBar *helpBar = new ZafStatusBar(0, 0, 0, 1);
	helpBar->SetStringID(ZAF_ITEXT("STATUS_BAR"));
	ZafString *helpBarString = new ZafString(0, 0, 32, new ZafStringData("", 64));
	helpBarString->SetBordered(true);
	helpBar->Add(helpBarString);
	Add(helpBar);
	helpTip->SetHelpObject(helpBarString);

	Add(new ZafPrompt(20, 0, 25, new ZafStringData("Settings")));

	// Create attribute list.
	attrList = new ZafVtList(1, 1, 25, 7);
	attrList->SetStringID(ZAF_ITEXT("ATTR_LIST"));
	attrList->Add(new ZafScrollBar(0, 0, 0, 0, ZAF_NULLP(ZafScrollData)));
	attrList->SetSelectionType(ZAF_MULTIPLE_SELECTION);
	attrList->SetAutoSortData(true);
	attrList->SetOSDraw(false);

	// Add attribute check boxes to list.
	int count = 0;
	ZafIChar buf[64];
	ZafButton *attr = ZAF_NULLP(ZafButton);
	for (count = 0; AttributeTable[count].text; count++)
	{
		sprintf(buf, ZAF_ITEXT("%s %d"), AttributeTable[count].text, AttributeTable[count].count);
		attr = new ZafButton(0, 0, 15, 1, new ZafStringData(buf));
		attr->SetHzJustify(ZAF_HZ_LEFT);
		attr->SetButtonType(ZAF_CHECK_BOX);
		attr->SetSendMessageWhenSelected(true);
		attr->SetValue(AttributeTable[count].value);
		attr->SetNumberID((ZafNumberID)AttributeTable[count].value);
		attrList->Add(attr);
	}
	Add(attrList);

	// Create tool tip combo box.
	ZafComboBox *toolTipCombo = new ZafComboBox(27, 3, 20, 3);
	toolTipCombo->SetViewOnly(true);
	toolTipCombo->SetStringID(ZAF_ITEXT("TOOL_TIP_COMBO"));
	toolTipCombo->Add(new ZafScrollBar(0, 0, 0, 0, ZAF_NULLP(ZafScrollData)));
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("(no help tip) *"));
	attr->SetStringID(ZAF_ITEXT("None"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_TOOL_TIP);
	toolTipCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Single line help tip*"));
	attr->SetStringID(ZAF_ITEXT("Single"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_TOOL_TIP);
	toolTipCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Multi-line help tip *"));
	attr->SetStringID(ZAF_ITEXT("Multi"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_TOOL_TIP);
	toolTipCombo->Add(attr);

	Add(toolTipCombo);

	// Create help context combo box.
	ZafComboBox *helpContextCombo = new ZafComboBox(27, 4, 20, 3);
	helpContextCombo->SetViewOnly(true);
	helpContextCombo->SetStringID(ZAF_ITEXT("HELP_CONTEXT_COMBO"));
	helpContextCombo->Add(new ZafScrollBar(0, 0, 0, 0, ZAF_NULLP(ZafScrollData)));
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("(none) *"));
	attr->SetStringID(ZAF_ITEXT("HELP_CONTEXT_NONE"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_HELP_CONTEXT);
	helpContextCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Help 1 *"));
	attr->SetStringID(ZAF_ITEXT("HELP_CONTEXT_1"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_HELP_CONTEXT);
	helpContextCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Help 2 *"));
	attr->SetStringID(ZAF_ITEXT("HELP_CONTEXT_2"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_HELP_CONTEXT);
	helpContextCombo->Add(attr);
	Add(helpContextCombo);

	// Create text color combo box.
	Add(new ZafPrompt(27, 5, 9, new ZafStringData("Text:**")));
	ZafComboBox *textColorCombo = new ZafComboBox(37, 5, 10, 3);
	textColorCombo->SetViewOnly(true);
	textColorCombo->SetStringID(ZAF_ITEXT("TEXT_COLOR_COMBO"));
	textColorCombo->Add(new ZafScrollBar(0, 0, 0, 0, ZAF_NULLP(ZafScrollData)));
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Default"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_TEXT_DEFAULT);
	attr->SetSelected(true);
	textColorCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Black"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_TEXT_BLACK);
	textColorCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Red"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_TEXT_RED);
	textColorCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Yellow"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_TEXT_YELLOW);
	textColorCombo->Add(attr);

	Add(textColorCombo);

	// Create background color combo box.
	Add(new ZafPrompt(27, 6, 9, new ZafStringData("Backgr:**")));
	ZafComboBox *backgroundColorCombo = new ZafComboBox(37, 6, 10, 3);
	backgroundColorCombo->SetViewOnly(true);
	backgroundColorCombo->SetStringID(ZAF_ITEXT("BACK_COLOR_COMBO"));
	backgroundColorCombo->Add(new ZafScrollBar(0, 0, 0, 0, ZAF_NULLP(ZafScrollData)));
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Default"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_BACK_DEFAULT);
	attr->SetSelected(true);
	backgroundColorCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Black"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_BACK_BLACK);
	backgroundColorCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Red"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_BACK_RED);
	backgroundColorCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Yellow"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_BACK_YELLOW);
	backgroundColorCombo->Add(attr);

	Add(backgroundColorCombo);

	// Create font combo box.
	Add(new ZafPrompt(27, 7, 9, new ZafStringData("Font:**")));
	ZafComboBox *fontCombo = new ZafComboBox(37, 7, 10, 3);
	fontCombo->SetViewOnly(true);
	fontCombo->SetStringID(ZAF_ITEXT("FONT_COMBO"));
	fontCombo->Add(new ZafScrollBar(0, 0, 0, 0, ZAF_NULLP(ZafScrollData)));
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Small"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_FONT_SMALL);
	fontCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Dialog"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_FONT_DIALOG);
	fontCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("System"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_FONT_SYSTEM);
	fontCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Application"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_FONT_APPLICATION);
	fontCombo->Add(attr);
	attr = new ZafButton(0, 0, 15, 1, new ZafStringData("Fixed"));
	attr->SetButtonType(ZAF_FLAT_BUTTON);
	attr->SetHzJustify(ZAF_HZ_LEFT);
	attr->SetSendMessageWhenSelected(true);
	attr->SetValue(TST_SET_FONT_FIXED);
	fontCombo->Add(attr);

	Add(fontCombo);

	// Create store button.
	ZafButton *store = new ZafButton(1, 10, 10, 1, new ZafStringData("Store *"));
	store->SetAutoSize(false);
	store->SetSendMessageWhenSelected(true);
	store->SetValue(TST_STORE);
	Add(store);

	// Create event log button.
	ZafButton *eventLog = new ZafButton(12, 10, 10, 1, new ZafStringData("Event Log"));
	eventLog->SetStringID(ZAF_ITEXT("EVENT_LOG"));
	eventLog->SetAutoSize(false);
	eventLog->SetSendMessageWhenSelected(true);
	eventLog->SetValue(TST_SHOW_EVENT_LOG);
	Add(eventLog);

	// Create duplicate button.
	ZafButton *duplicate = new ZafButton(23, 10, 11, 1, new ZafStringData("Duplicate *"));
	duplicate->SetAutoSize(false);
	duplicate->SetSendMessageWhenSelected(true);
	duplicate->SetValue(TST_DUPLICATE);
	Add(duplicate);

	// Create redraw button.
	ZafButton *redraw = new ZafButton(35, 10, 11, 1, new ZafStringData("Redraw **"));
	redraw->SetAutoSize(false);
	redraw->SetSendMessageWhenSelected(true);
	redraw->SetValue(TST_REDRAW);
	Add(redraw);

	// Create addSubtract button.
	ZafButton *addSubtract = new ZafButton(50, 10, 11, 1, new ZafStringData("Subtract"));
	addSubtract->SetStringID(ZAF_ITEXT("ADD_SUBTRACT"));
	addSubtract->SetAutoSize(false);
	addSubtract->SetSendMessageWhenSelected(true);
	addSubtract->SetValue(TST_SUBTRACT_TESTOBJECT);
	Add(addSubtract);

	Add(new ZafPrompt(60, 0, 25, new ZafStringData("Object")));
}

GenericTestWin::GenericTestWin(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist),
	attrList(0), testObject(0), testObjectParent(0), testObjectSibling(0)
{
	// Initialize pointers to important objects.
	attrList = DynamicPtrCast(GetObject(ZAF_ITEXT("ATTR_LIST")), ZafVtList);
	testObject = GetObject(ZAF_ITEXT("TEST_OBJECT"));

	// Persisted windows don't have a derived test object, so don't track events.
	ZafWindowObject *eventLog = GetObject(ZAF_ITEXT("EVENT_LOG"));
	eventLog->SetDisabled(true);
}

GenericTestWin::GenericTestWin(GenericTestWin &copy): ZafWindow(copy),
	attrList(0), testObject(0), testObjectParent(0), testObjectSibling(0)
{
	// Initialize pointers to important objects.
	attrList = DynamicPtrCast(GetObject(ZAF_ITEXT("ATTR_LIST")), ZafVtList);
	testObject = GetObject(ZAF_ITEXT("TEST_OBJECT"));

	// Copied windows don't have a derived test object, so don't track events.
	ZafWindowObject *eventLog = GetObject(ZAF_ITEXT("EVENT_LOG"));
	eventLog->SetDisabled(true);
}

GenericTestWin::~GenericTestWin(void)
{
	// Delete testObject if it is not attached to the window.
/* START BLOCK COMMENT
**		if (!testObject->screenID)
**			delete testObject;
END BLOCK COMMENT */
}

ZafEventType GenericTestWin::Draw(const ZafEventStruct &event, ZafEventType ccode)
{
	ZafWindow::Draw(event, ccode);

	// Compute the actual draw region.
	ZafRegionStruct drawRegion = BeginDraw();

	ZafPaletteStruct lightShadow = LogicalPalette(ZAF_PM_LIGHT_SHADOW, 0);
	ZafPaletteStruct darkShadow = LogicalPalette(ZAF_PM_DARK_SHADOW, 0);
	ZafPaletteStruct whiteShadow = lightShadow;

	// Copy the background colors to the foreground since display->Line() uses 
	// the foreground color.  The "shadow" palettes have different colors in 
	// the foreground and background entries so they can generate two shades
	// of shadow.
	whiteShadow.colorForeground = whiteShadow.colorBackground;

	// Draw a divider line between the settings and the test object(s).
	ZafWindowObject *object = GetObject(ZAF_ITEXT("BACK_COLOR_COMBO"));
	int column = drawRegion.left + object->zafRegion.right + 2 * display->cellWidth;
	int top = drawRegion.top;
	int bottom = drawRegion.bottom;

	display->SetPalette(darkShadow);
	display->Line(column, top, column, bottom);
	display->SetPalette(whiteShadow);
	display->Line(column + 1, top, column + 1, bottom);

	EndDraw();

	return (ccode);
}

ZafEventType GenericTestWin::Event(const ZafEventStruct &event)
{
	ZafEventType ccode = event.type;

	// Update count for attribute item.
	if (ccode >= TST_TOGGLE_FIRST && ccode <= TST_TOGGLE_LAST)
	{
		for (int count = 0; AttributeTable[count].text; count++)
			if (AttributeTable[count].value == ccode)
			{
				AttributeTable[count].count++;
				ZafWindowObject *attr = attrList->GetObject((ZafNumberID)ccode);
				ZafIChar buf[64];
				sprintf(buf, ZAF_ITEXT("%s %d"), AttributeTable[count].text, AttributeTable[count].count);
				attr->SetText(buf);
				break;
			}
	}

	switch (ccode)
	{
	case S_INITIALIZE:
		ZafWindow::Event(event);
		testObjectParent = DynamicPtrCast(testObject->parent, ZafWindow);
		break;

	// TST_TOGGLE_*** sets testObject's attribute to the selected status of
	// the attribute check box.  It then updates the attribute check box
	// using the attribute accessor function to test if the attribute was
	// successfully set.
	case TST_TOGGLE_BORDERED:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_BORDERED);
			testObject->SetBordered(attr->Selected());
			attr->SetSelected(testObject->Bordered());
		}
		break;

	case TST_TOGGLE_VISIBLE:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_VISIBLE);
			testObject->SetVisible(attr->Selected());
			attr->SetSelected(testObject->Visible());
		}
		break;

	case TST_TOGGLE_ACCEPT_DROP:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_ACCEPT_DROP);
			testObject->SetAcceptDrop(attr->Selected());
			attr->SetSelected(testObject->AcceptDrop());
		}
		break;

	case TST_TOGGLE_AUTOMATIC_UPDATE:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_AUTOMATIC_UPDATE);
			testObject->SetAutomaticUpdate(attr->Selected());
			attr->SetSelected(testObject->AutomaticUpdate(false));
		}
		break;

	case TST_TOGGLE_CHANGED:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_CHANGED);
			testObject->SetChanged(attr->Selected());
			attr->SetSelected(testObject->Changed());
		}
		break;

	case TST_TOGGLE_COPY_DRAGGABLE:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_COPY_DRAGGABLE);
			testObject->SetCopyDraggable(attr->Selected());
			attr->SetSelected(testObject->CopyDraggable());
		}
		break;

	case TST_TOGGLE_LINK_DRAGGABLE:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_LINK_DRAGGABLE);
			testObject->SetLinkDraggable(attr->Selected());
			attr->SetSelected(testObject->LinkDraggable());
		}
		break;

	case TST_TOGGLE_MOVE_DRAGGABLE:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_MOVE_DRAGGABLE);
			testObject->SetMoveDraggable(attr->Selected());
			attr->SetSelected(testObject->MoveDraggable());
		}
		break;

	case TST_TOGGLE_FOCUS:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_FOCUS);
			testObject->SetFocus(attr->Selected());
			attr->SetSelected(testObject->Focus());
		}
		break;

	case TST_TOGGLE_NONCURRENT:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_NONCURRENT);
			testObject->SetNoncurrent(attr->Selected());
			attr->SetSelected(testObject->Noncurrent());
		}
		break;

	case TST_TOGGLE_DISABLED:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_DISABLED);
			testObject->SetDisabled(attr->Selected());
			attr->SetSelected(testObject->Disabled());
		}
		break;

	case TST_TOGGLE_OSDRAW:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_OSDRAW);
			testObject->SetOSDraw(attr->Selected());
			attr->SetSelected(testObject->OSDraw());
		}
		break;

	case TST_TOGGLE_PARENTDRAW_BORDER:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_PARENTDRAW_BORDER);
			testObject->SetParentDrawBorder(attr->Selected());
			attr->SetSelected(testObject->ParentDrawBorder());
		}
		break;

	case TST_TOGGLE_PARENTPALETTE:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_PARENTPALETTE);
			testObject->SetParentPalette(attr->Selected());
			attr->SetSelected(testObject->ParentPalette());
		}
		break;

	case TST_TOGGLE_PARENTDRAW_FOCUS:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_PARENTDRAW_FOCUS);
			testObject->SetParentDrawFocus(attr->Selected());
			attr->SetSelected(testObject->ParentDrawFocus());
		}
		break;

	case TST_TOGGLE_SELECTED:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_SELECTED);
			testObject->SetSelected(attr->Selected());
			attr->SetSelected(testObject->Selected());
		}
		break;

	case TST_TOGGLE_EDIT_MODE:
		{
			ZafWindowObject *attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_EDIT_MODE);
			testObject->SetEditMode(attr->Selected());
			attr->SetSelected(testObject->EditMode());
		}
		break;

	case TST_SET_TOOL_TIP:
		{
			ZafComboBox *helpTipCombo = DynamicPtrCast(GetObject(ZAF_ITEXT("TOOL_TIP_COMBO")), ZafComboBox);
			ZafStringID stringID = helpTipCombo->Current()->StringID();
			if (!strcmp(stringID, ZAF_ITEXT("None")))
			{
				testObject->SetHelpObjectTip(ZAF_NULLP(ZafIChar));
				testObject->SetQuickTip(ZAF_NULLP(ZafIChar));
			}
			else if (!strcmp(stringID, ZAF_ITEXT("Single")))
			{
				ZafIChar *text = ZAF_ITEXT("Single line help object tip");
				testObject->SetHelpObjectTip(text);
				text = ZAF_ITEXT("Single line quick tip");
				testObject->SetQuickTip(text);
			}
			else
			{
				ZafIChar *text = ZAF_ITEXT("Multi-line help object text");
				testObject->SetHelpObjectTip(text);
				text = ZAF_ITEXT("Multi-line \nquick tip.");
				testObject->SetQuickTip(text);
			}

			const ZafIChar *quickTip = testObject->QuickTip();

			if (!quickTip)
				helpTipCombo->First()->SetSelected(true);
			else if (!strncmp(quickTip, ZAF_ITEXT("Single"), 6))
				helpTipCombo->Get(1)->SetSelected(true);
			else
				helpTipCombo->Get(2)->SetSelected(true);
		}
		break;

	case TST_SET_HELP_CONTEXT:
		{
			ZafComboBox *helpCombo = DynamicPtrCast(GetObject(ZAF_ITEXT("HELP_CONTEXT_COMBO")), ZafComboBox);
			ZafWindowObject *attr = helpCombo->Current();
			ZafStringID newHelpContext = ZAF_NULLP(ZafIChar);
			if (strcmp(attr->StringID(), ZAF_ITEXT("HELP_CONTEXT_NONE")))
				newHelpContext = helpCombo->Current()->StringID();
			testObject->SetHelpContext(newHelpContext);

			const ZafIChar *helpContext = testObject->HelpContext();
			attr = helpCombo->First();
			while (attr && helpContext && strcmp(helpContext, attr->StringID()))
				attr = attr->Next();

			if (attr)
				attr->SetSelected(true);
		}
		break;

	case TST_SET_TEXT_DEFAULT:
		testObject->SetTextColor(ZAF_CLR_NULL);
		break;

	case TST_SET_TEXT_RED:
		testObject->SetTextColor(ZAF_CLR_RED);
		break;

	case TST_SET_TEXT_YELLOW:
		testObject->SetTextColor(ZAF_CLR_YELLOW);
		break;

	case TST_SET_TEXT_BLACK:
		testObject->SetTextColor(ZAF_CLR_BLACK);
		break;

	case TST_SET_BACK_DEFAULT:
		testObject->SetBackgroundColor(ZAF_CLR_NULL);
		break;

	case TST_SET_BACK_RED:
		testObject->SetBackgroundColor(ZAF_CLR_RED);
		break;

	case TST_SET_BACK_YELLOW:
		testObject->SetBackgroundColor(ZAF_CLR_YELLOW);
		break;

	case TST_SET_BACK_BLACK:
		testObject->SetBackgroundColor(ZAF_CLR_BLACK);
		break;

	case TST_SET_FONT_SMALL:
	case TST_SET_FONT_DIALOG:
	case TST_SET_FONT_APPLICATION:
	case TST_SET_FONT_SYSTEM:
	case TST_SET_FONT_FIXED:
		{
			if (ccode == TST_SET_FONT_SMALL)
				testObject->SetFont(ZAF_FNT_SMALL);
			else if (ccode == TST_SET_FONT_DIALOG)
				testObject->SetFont(ZAF_FNT_DIALOG);
			else if (ccode == TST_SET_FONT_SYSTEM)
				testObject->SetFont(ZAF_FNT_SYSTEM);
			else if (ccode == TST_SET_FONT_APPLICATION)
				testObject->SetFont(ZAF_FNT_APPLICATION);
			else if (ccode == TST_SET_FONT_FIXED)
				testObject->SetFont(ZAF_FNT_FIXED);

			ZafComboBox *fontCombo = DynamicPtrCast(GetObject(ZAF_ITEXT("FONT_COMBO")), ZafComboBox);
			ZafLogicalFont font = testObject->Font();

			if (font == ZAF_FNT_SMALL)
				fontCombo->Get(0)->SetSelected(true);
			if (font == ZAF_FNT_DIALOG)
				fontCombo->Get(1)->SetSelected(true);
			if (font == ZAF_FNT_SYSTEM)
				fontCombo->Get(2)->SetSelected(true);
			if (font == ZAF_FNT_APPLICATION)
				fontCombo->Get(3)->SetSelected(true);
			if (font == ZAF_FNT_FIXED)
				fontCombo->Get(4)->SetSelected(true);
		}
		break;

	// Store the window, then load a copy of it from the dat file.
	case TST_STORE:
		{
			// If the testObject is not on the window, add it so it will be 
			// stored.
			if (!testObject->screenID)
				Event(TST_ADD_TESTOBJECT);
			ZafStorage *storage = new ZafStorage(ZAF_ITEXT("test.dat"), ZAF_FILE_CREATE | ZAF_FILE_READWRITE);
			ZafObjectPersistence *persist = new ZafObjectPersistence(storage);
			Write(*persist);
			storage->Save();
			ObjectTestWin *newWin = new ObjectTestWin(StringID(), *persist);
			windowManager->Add(newWin);
			newWin->SynchronizeAttributes();
			delete persist;
			delete storage;
		}
		break;

	// Display the event log.
	case TST_SHOW_EVENT_LOG:
		{
			ZafVtList *eventList = new ZafVtList(1, 1, 25, 7);
			eventList->SetRegionType(ZAF_AVAILABLE_REGION);
			eventList->Add(new ZafScrollBar(0, 0, 0, 0, ZAF_NULLP(ZafScrollData)));

			// Add attribute items to list.
			ZafIChar buf[64];
			ZafButton *eventItem = ZAF_NULLP(ZafButton);
			for (int count = 0; EventLog[count].text; count++)
			{
				sprintf(buf, ZAF_ITEXT("%s %d"), EventLog[count].text, EventLog[count].count);
				eventItem = new ZafButton(0, 0, 15, 1, buf);
				eventItem->SetHzJustify(ZAF_HZ_LEFT);
				eventItem->SetButtonType(ZAF_FLAT_BUTTON);
				eventList->Add(eventItem);
			}

			ZafWindow *eventWindow = new ZafWindow(40, 10, 30, 6);
			eventWindow->AddGenericObjects(new ZafStringData("Event Log"));
//			eventWindow->SetModal(true);
			eventWindow->Add(eventList);
			windowManager->Add(eventWindow);
		}
		break;

	// Redraw the window.  Sometimes changing an attribute doesn't update the
	// display properly.  Redrawing may indicate that the attribute _did_ get
	// set just not shown.
	case TST_REDRAW:
		Event(S_REDISPLAY);
		break;

	case TST_DUPLICATE:
		{
			// If the testObject is not on the window, add it so it will be 
			// duplicated.
			if (!testObject->screenID)
				Event(TST_ADD_TESTOBJECT);
			ObjectTestWin *newWin = DynamicPtrCast(Duplicate(), ObjectTestWin);
			windowManager->Add(newWin);
			newWin->SynchronizeAttributes();
		}
		break;

	case TST_SUBTRACT_TESTOBJECT:
		{
			ZafButton *addSubtract = DynamicPtrCast(GetObject(ZAF_ITEXT("ADD_SUBTRACT")), ZafButton);
			addSubtract->SetValue(TST_ADD_TESTOBJECT);
			addSubtract->SetText(ZAF_ITEXT("Add"));
			testObjectSibling = testObject->Next();
			testObjectParent->Subtract(testObject);
		}
		break;

	case TST_ADD_TESTOBJECT:
		{
			ZafButton *addSubtract = DynamicPtrCast(GetObject(ZAF_ITEXT("ADD_SUBTRACT")), ZafButton);
			addSubtract->SetValue(TST_SUBTRACT_TESTOBJECT);
			addSubtract->SetText(ZAF_ITEXT("Subtract"));
			testObjectParent->Add(testObject, testObjectSibling);
		}
		break;

	default:
		ccode = ZafWindow::Event(event);
		break;
	}

	return (ccode);
}

void GenericTestWin::LogEvent(ZafEventType ccode)
{
	// Update the count for the event ccode.
	for (int count = 0; EventLog[count].text; count++)
	{
		if (EventLog[count].value == ccode)
		{
			EventLog[count].count++;
			break;
		}
	}
}

void GenericTestWin::SynchronizeAttributes()
{
	ZafWindowObject *attr = 0;

	// Indicate if the testObject has a particular attribute setting.
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_BORDERED);
	if (attr)
		attr->SetSelected(testObject->Bordered());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_VISIBLE);
	if (attr)
		attr->SetSelected(testObject->Visible());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_ACCEPT_DROP);
	if (attr)
		attr->SetSelected(testObject->AcceptDrop());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_AUTOMATIC_UPDATE);
	if (attr)
		attr->SetSelected(testObject->AutomaticUpdate(false));
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_CHANGED);
	if (attr)
		attr->SetSelected(testObject->Changed());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_COPY_DRAGGABLE);
	if (attr)
		attr->SetSelected(testObject->CopyDraggable());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_LINK_DRAGGABLE);
	if (attr)
		attr->SetSelected(testObject->LinkDraggable());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_MOVE_DRAGGABLE);
	if (attr)
		attr->SetSelected(testObject->MoveDraggable());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_FOCUS);
	if (attr)
		attr->SetSelected(testObject->Focus());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_NONCURRENT);
	if (attr)
		attr->SetSelected(testObject->Noncurrent());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_DISABLED);
	if (attr)
		attr->SetSelected(testObject->Disabled());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_OSDRAW);
	if (attr)
		attr->SetSelected(testObject->OSDraw());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_PARENTDRAW_BORDER);
	if (attr)
		attr->SetSelected(testObject->ParentDrawBorder());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_PARENTPALETTE);
	if (attr)
		attr->SetSelected(testObject->ParentPalette());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_PARENTDRAW_FOCUS);
	if (attr)
		attr->SetSelected(testObject->ParentDrawFocus());
	attr = attrList->GetObject((ZafNumberID)TST_TOGGLE_SELECTED);
	if (attr)
		attr->SetSelected(testObject->Selected());

	ZafComboBox *helpCombo = DynamicPtrCast(GetObject(ZAF_ITEXT("HELP_CONTEXT_COMBO")), ZafComboBox);
	const ZafIChar *helpContext = testObject->HelpContext();
	attr = helpCombo->First();
	while (attr && helpContext && strcmp(helpContext, attr->StringID()))
		attr = attr->Next();
	if (attr)
		attr->SetSelected(true);

	ZafComboBox *helpTipCombo = DynamicPtrCast(GetObject(ZAF_ITEXT("TOOL_TIP_COMBO")), ZafComboBox);
	const ZafIChar *quickTip = testObject->QuickTip();
	if (!quickTip)
		helpTipCombo->First()->SetSelected(true);
	else if (!strncmp(quickTip, ZAF_ITEXT("Single"), 6))
		helpTipCombo->Get(1)->SetSelected(true);
	else
		helpTipCombo->Get(2)->SetSelected(true);

	ZafComboBox *fontCombo = DynamicPtrCast(GetObject(ZAF_ITEXT("FONT_COMBO")), ZafComboBox);
	ZafLogicalFont font = testObject->Font();

	if (font == ZAF_FNT_SMALL)
		fontCombo->Get(0)->SetSelected(true);
	if (font == ZAF_FNT_DIALOG)
		fontCombo->Get(1)->SetSelected(true);
	if (font == ZAF_FNT_SYSTEM)
		fontCombo->Get(2)->SetSelected(true);
}

ObjectTestWin::ObjectTestWin(const ZafIChar *name, ZafObjectPersistence &persist) :
	GenericTestWin(name, persist)
{
	// Nothing to do.
}

ObjectTestWin::ObjectTestWin(ObjectTestWin &copy): GenericTestWin(copy)
{
	// Nothing to do.
}

int ZafApplication::Main(void)
{
	// Ensure main() is linked properly.
	LinkMain();

	helpTip = new ZafHelpTips(D_ON, ZAF_HELPTIPS_BOTH);
	zafEventManager->Add(helpTip);

	ZafStorage *storage = new ZafStorage(ZAF_ITEXT("help.dat"), ZAF_FILE_READWRITE);
	if (!storage->Error())
		zafHelpSystem = new ZafHelpSystem(storage, ZAF_NULLP(ZafIChar));

	// Add the test window.
	ObjectTestWin *testWindow = new ObjectTestWin;
	zafWindowManager->Add(testWindow);
	testWindow->SynchronizeAttributes();

	// Set windowManager's screenID so if the main window is closed, the
	// entire app is shut down.
	zafWindowManager->screenID = testWindow->screenID;

	// Get the user input.
	Control();

	if (zafHelpSystem)
		delete zafHelpSystem;
	if (storage)
		delete storage;

	// Return success.
	return (0);
}


