//	Zinc Application Framework - Z_VLIST.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_vlist.hpp>
#define ZAF_VT_LIST_INFO
#include "gbl_def.cpp"

// ----- ZafVtList ----------------------------------------------------------

ZafVtList::ZafVtList(int left, int top, int width, int height) :
	ZafWindow(left, top, width, height),
	autoSortData(false)
{
	// Initialize the list information.
	SetBordered(true);

#	if defined(ZAF_MOTIF)
	itemWidth = itemHeight = 0;
	firstVisible = lastVisible = cancelObject = ZAF_NULLP(ZafWindowObject);
#	endif
}

ZafVtList::ZafVtList(const ZafVtList &copy) :
	ZafWindow(copy),
	autoSortData(copy.autoSortData)
{
	// Set all children as non-system objects.  ZafWindow copy
	// constructor will not call ZafVtList::Add() since it is in the base
	// class phase of construction when the children are added.
	for (ZafWindowObject *object = First(); object; object = object->Next())
	{
		object->SetSystemObject(false);
		object->SetParentPalette(true);
	}

#	if defined(ZAF_MOTIF)
	itemWidth = itemHeight = 0;
	firstVisible = lastVisible = cancelObject = ZAF_NULLP(ZafWindowObject);
#	endif
}

ZafVtList::~ZafVtList(void)
{
}

void ZafVtList::Destroy(void)
{
	// Destroy the list elements (high level).
	OSDestroy();

	// Destroy the list elements (low level).
	ZafList::Destroy();
}

ZafWindowObject *ZafVtList::Duplicate(void)
{
	return (new ZafVtList(*this));
}

ZafLogicalEvent ZafVtList::LogicalEvent(const ZafEventStruct &event)
{
	ZafLogicalEvent ccode = MapEvent(defaultEventMap, event);
	return ((ccode == L_NONE) ? ZafWindow::LogicalEvent(event) : ccode);
}

const ZafPaletteStruct *ZafVtList::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafWindow::MapClassPalette(type, state));
}

bool ZafVtList::SetAutoSortData(bool setAutoSortData)
{
	// Make sure the attribute has changed.
	if (autoSortData != setAutoSortData)
	{
		autoSortData = setAutoSortData;

		// Reset the sort function.
		if (autoSortData)
		{
			SetCompareFunction((ZafCompareFunction)CompareAscending);
			Sort();
		}
		else if (compareFunction == (ZafCompareFunction)CompareAscending)
			compareFunction = ZAF_NULLF(ZafCompareFunction);
	}

	// Return the current attribute.
	return (autoSortData);
}

ZafLogicalColor ZafVtList::SetBackgroundColor(ZafLogicalColor color, ZafLogicalColor mono)
{
	// Make sure there is a userPalette.
	if (!userPaletteData)
		SetUserPaletteData(new ZafPaletteData());

	// Add the background color entry.
	ZafPaletteStruct backgroundPalette = userPaletteData->GetPalette(ZAF_PM_BACKGROUND, ZAF_PM_NOT_SELECTED);
	backgroundPalette.colorBackground = color;
	backgroundPalette.monoBackground = mono;
	userPaletteData->AddPalette(ZAF_PM_BACKGROUND, ZAF_PM_NOT_SELECTED, backgroundPalette);

	// Add the selected text color entry.
	backgroundPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED);
	backgroundPalette.colorForeground = color;
	backgroundPalette.monoForeground = mono;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED, backgroundPalette);

	// Return the current color.
	return (color);
}

bool ZafVtList::SetCopyDraggable(bool setCopyDraggable)
{
	// Defer to the root class.
	return (ZafWindowObject::SetCopyDraggable(setCopyDraggable));
}

bool ZafVtList::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

ZafLogicalFont ZafVtList::SetFont(ZafLogicalFont font)
{
	// Make sure there is a userPalette.
	if (!userPaletteData)
		SetUserPaletteData(new ZafPaletteData());

	// Add the new entries.
	ZafPaletteStruct textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED);
	textPalette.font = font;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED, textPalette);
	textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED);
	textPalette.font = font;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED, textPalette);
	textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_ANY_STATE);
	textPalette.font = font;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_ANY_STATE, textPalette);

	// Return the current font.
	return (font);
}

bool ZafVtList::SetLinkDraggable(bool setLinkDraggable)
{
	// Defer to the root class.
	return (ZafWindowObject::SetLinkDraggable(setLinkDraggable));
}

bool ZafVtList::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

bool ZafVtList::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafVtList::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafVtList::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

bool ZafVtList::SetMoveDraggable(bool setMoveDraggable)
{
	// Defer to the root class.
	return (ZafWindowObject::SetMoveDraggable(setMoveDraggable));
}

bool ZafVtList::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

ZafLogicalColor ZafVtList::SetTextColor(ZafLogicalColor color, ZafLogicalColor mono)
{
	// Make sure there is a userPalette.
	if (!userPaletteData)
		SetUserPaletteData(new ZafPaletteData());

	// Add the text color entry.
	ZafPaletteStruct textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED);
	textPalette.colorForeground = color;
	textPalette.monoForeground = mono;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED, textPalette);

	// Add the selected text color entry.
	textPalette = userPaletteData->GetPalette(ZAF_PM_BACKGROUND, ZAF_PM_SELECTED);
	textPalette.colorBackground = color;
	textPalette.monoBackground = mono;
	userPaletteData->AddPalette(ZAF_PM_BACKGROUND, ZAF_PM_SELECTED, textPalette);

	// Return the current color.
	return (color);
}

void ZafVtList::Sort(void)
{
	// Sort on the Zinc level.
	ZafList::Sort();

	// Sort from the OS level.
	OSSort();
}

// ----- Persistent functions -----------------------------------------------

ZafVtList::ZafVtList(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY))
{
	// Read the data.
	ZafVtList::ReadData(persist);

	// Set all children as non-system objects.  ZafWindow persistent
	// constructor will not call ZafVtList::Add() since it is in the base
	// class phase of construction when the children are added.
	for (ZafWindowObject *object = First(); object; object = object->Next())
	{
		object->SetSystemObject(false);
		object->SetParentPalette(true);
	}

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());

#	if defined(ZAF_MOTIF)
	itemWidth = itemHeight = 0;
	firstVisible = lastVisible = cancelObject = ZAF_NULLP(ZafWindowObject);
#	endif
}

ZafElement *ZafVtList::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafVtList(name, persist));
}

void ZafVtList::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafUInt16 flag1;
	ZafFile *file = persist.File();
	*file >> flag1;
	autoSortData = (flag1 & 0x0001) ? true : false;
}

void ZafVtList::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafWindow::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafVtList::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafVtList::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	ZafUInt16 flag1 = 0;
	ZafFile *file = persist.File();
	flag1 |= autoSortData ? 0x0001 : 0;
	*file << flag1;
}
