//	Zinc Application Framework - Z_UTIME.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <time.h>
#include <z_ctype.hpp>
#include <z_stdlib.hpp>
#include <z_stdarg.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>
#include <z_utime.hpp>
#if defined(ZAF_MOTIF)
#	if defined(SCO_UNIX)
	extern "C" int gettimeofday(struct timeval *tp, struct timezone *tzp);
#	endif
#	include <sys/time.h>
#elif defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS)
#	include <dos.h>
#endif
#include <z_app.hpp>
#define ZAF_UTIME_DATA_INFO
#include "lang_def.cpp"
#include "gbl_def.cpp"

// ----- ZafUTimeData -------------------------------------------------------

// We don't really support leap seconds.
const long uSecsPerDay = (24L * 60L * 60L * 1000L);
static int ZAF_FARDATA _daysInMonth[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
static ZafIChar *_dFormat = ZAF_ITEXT("%m/%d/%y");
static ZafIChar *_rFormat = ZAF_ITEXT("%H:%M");
static ZafIChar *_tFormat = ZAF_ITEXT("%H:%M:%S");
static struct
{
	ZafIChar *name;
	int offset;
} _timeZones[] =
{
	{ ZAF_NULLP(ZafIChar), 0 }
};

ZafUTimeData::ZafUTimeData(void) :
	ZafFormatData(), jDay(0), uSec(0), recurse(false)
{
	// Check the printf binding.
	Bind();

	// Check the language and local information.
	LanguageAllocate();

	// Clear the utime from the system and get basisYear.
	Clear();
}

ZafUTimeData::ZafUTimeData(ZafUInt32 seconds) :
	ZafFormatData(), jDay(0), uSec(0), recurse(false)
{
	// Check the printf binding.
	Bind();

	// Check the language and local information.
	LanguageAllocate();

	// Clear the utime from the system and get basisYear.
	Clear();

	// Get the utime based on separate values.
	SetUTime(seconds);
}

ZafUTimeData::ZafUTimeData(int year, int month, int day,
	int hour, int minute, int second, int milliSecond) :
	ZafFormatData(), jDay(0), uSec(0), recurse(false)
{
	// Check the printf binding.
	Bind();

	// Check the language and local information.
	LanguageAllocate();

	// Clear the utime from the system and get basisYear.
	Clear();

	// Get the utime based on separate values.
	SetUTime(year, month, day, hour, minute, second, milliSecond);
}

ZafUTimeData::ZafUTimeData(const ZafIChar *string, const ZafIChar *format) :
	ZafFormatData(), jDay(0), uSec(0), recurse(false)
{
	// Check the printf binding.
	Bind();

	// Check the language and local information.
	LanguageAllocate();

	// Clear the utime from the system and get basisYear.
	Clear();

	// Get the utime based on a string value.
	SetUTime(string, format);
}

ZafUTimeData::ZafUTimeData(const ZafUTimeData &utime) :
	ZafFormatData(utime),
	basisYear(utime.basisYear), jDay(utime.jDay), uSec(utime.uSec),
	recurse(false)
{
	// Check the printf binding.
	Bind();

	// Check the language and local information.
	LanguageAllocate();
}

ZafUTimeData::~ZafUTimeData(void)
{
	// Check the language and local information.
	LanguageFree();
}

void ZafUTimeData::Bind(void)
{
	// Bind in the sprintf characters.
	static bool bound = false;
	if (!bound)
	{
		// Check for a valid locale.
		if (!zafLocale)
			ZafI18nData::I18nAllocate();

		// Set the lookup characters.
		bound = true;
		ZafStandardArg::SetSprintf('U', ZafUTimeData::Format);
		ZafStandardArg::SetSscanf('U', ZafUTimeData::Parse);
	}
}

void ZafUTimeData::Clear(void)
{
	// Set the time values according to the system time.
#if defined(ZAF_MACINTOSH)
	DateTimeRec t;
	GetTime(&t);
	SetUTime(t.year, t.month, t.day, t.hour, t.minute, t.second, 0);
#elif defined(__DVX__) || defined(_MSC_VER) || defined(__SC__)
	time_t t;
	time(&t);
	struct tm *localTime = localtime(&t);
	SetUTime(1900+localTime->tm_year, localTime->tm_mon+1, localTime->tm_mday,
		localTime->tm_hour, localTime->tm_min, localTime->tm_sec, 0);
#elif defined(__BCPLUSPLUS__) || defined(__TCPLUSPLUS__)
	struct time tinfo;
	gettime(&tinfo);
	struct date dinfo;
	getdate(&dinfo);
	SetUTime(dinfo.da_year, dinfo.da_mon, dinfo.da_day,
		tinfo.ti_hour, tinfo.ti_min, tinfo.ti_sec, tinfo.ti_hund * 10);
#elif defined(__WATCOMC__) && !defined(__QNX__)
	struct dostime_t tInfo;
	_dos_gettime(&tInfo);
	struct dosdate_t dInfo;
	_dos_getdate(&dInfo);
	SetUTime(dInfo.year, dInfo.month, dInfo.day,
		tInfo.hour, tInfo.minute, tInfo.second, tInfo.hsecond * 10);
#else
	struct timeval tv;
	struct timezone tz;
	gettimeofday(&tv, &tz);
	time_t t;
	time(&t);
	struct tm *localTime = localtime(&t);	// Used for consistent DST info.
	SetUTime(1900+localTime->tm_year, localTime->tm_mon+1, localTime->tm_mday,
		localTime->tm_hour, localTime->tm_min, localTime->tm_sec,
		(int)(tv.tv_usec / 1000));
#endif

	// Set the default basisYear to be the first year of the current century.
	int tYear = Year();
	basisYear = tYear - (tYear % 100);
}

void ZafUTimeData::ConvertJDay(int *pYear, int *pMonth, int *pDay, int *pDayOfWeek) const
{
	// Start with empty dates.
	long year = 0, month = 0, day = 0;

	// JDay == 0 and uSec < 0 is a "blank".
	if (jDay != 0 || uSec >= 0)
	{
		// Compute the separate date values algorithm 5 of Practical
		// Astronomy with your Calculator.  This includes a fix for the
		// Gregorian calendar change in 1582.  Protestant countries should
		// change that value to when they adopted the Gregorian calendar.
		long C = jDay;
		if (C > (long)zafLocale->beginGregorian)
		{
			long A = (100 * C - 186721625L) / 3652425L;
			C += 1 + A - A / 4;
		}
		C += 1524;
		long D = (100 * C - 12210L) / 36525L;
		long E = (1461L * D) / 4;
		long G = ((C - E) * 10000) / 306001L;
		day = C - E - (G * 306001L) / 10000;
		month = G - ((G > 13) ? 13 : 1);
		year = D - ((month > 2) ? 4716 : 4715);
		if (year <= 0)		// There is no year 0 (1 BC = -1).
			year--;
	}

	// Set the integer values.
	if (pYear)
		*pYear = (int)year;
	if (pMonth)
		*pMonth = (int)month;
 	if (pDay)
		*pDay = (int)day;
	if (pDayOfWeek)
		*pDayOfWeek = (int)((jDay + 1) % 7);
}

void ZafUTimeData::ConvertUSec(int *hour, int *minute, int *second, int *milliSecond) const
{
	// uSec < 0 means "blank"
	long tmp = (uSec < 0 ? 0 : uSec);

	// Set the integer variables.
	if (hour)
		*hour = (int)(tmp / 3600000L);
	tmp %= 3600000L;
	if (minute)
		*minute = (int)(tmp / 60000L);
	tmp %= 60000L;
	if (second)
		*second = (int)(tmp / 1000L);
	tmp %= 1000L;
	if (milliSecond)
		*milliSecond = (int)tmp;
}

int ZafUTimeData::Day(void) const
{
	// Compute the value.
	int day;
	ConvertJDay(0, 0, &day, 0);
	return (day);
}

int ZafUTimeData::DayOfWeek(void) const
{
	// Compute the value.
	int dayOfWeek;
	ConvertJDay(0, 0, 0, &dayOfWeek);
	return (dayOfWeek);
}

int ZafUTimeData::DaysInMonth(int month) const
{
	// Compute the specified month.
	if (month == 0)
		ConvertJDay(0, &month, 0, 0);
	if (month == 2 && LeapYear())
		return (29);
	return (month ? _daysInMonth[month - 1] : 0);
}

long ZafUTimeData::Difference(const ZafUTimeData &utime) const
{
	long _jDay = jDay - utime.jDay;
	long _uSec = uSec - utime.uSec;
	return (_jDay * uSecsPerDay + _uSec) / 10;
}

ZafData *ZafUTimeData::Duplicate(void)
{
	return (new ZafUTimeData(*this));
}

int ZafUTimeData::FormattedText(ZafIChar *buffer, int maxLength, const ZafIChar *format) const
{
	// Format the text.
	return (Sprintf(buffer, maxLength, format ? format : zafLocale->dateTimeStringOutputFormat));
}

int ZafUTimeData::Hour(void) const
{
	// Compute the value.
	int hour;
	ConvertUSec(&hour, 0, 0, 0);
	return (hour);
}

void ZafUTimeData::LanguageAllocate(void)
{
	// Check for initialization.
	if (dayStrings)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!dayStrings && zafDataManager)
	{
		dayStrings = DynamicPtrCast(zafDataManager->AllocateData(dayName, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);
		monthStrings = DynamicPtrCast(zafDataManager->AllocateData(monthName, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);
		timeStrings = DynamicPtrCast(zafDataManager->AllocateData(timeName, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);
	}

	// Default to code initialization.
	if (!dayStrings)
		dayStrings = new ZafLanguageData(defaultDayStrings);
	if (!monthStrings)
		monthStrings = new ZafLanguageData(defaultMonthStrings);
	if (!timeStrings)
		timeStrings = new ZafLanguageData(defaultTimeStrings);
}

void ZafUTimeData::LanguageFree(bool globalRequest)
{
	// Destroy the static images.
	if (globalRequest && dayStrings)
	{
		delete dayStrings;
		dayStrings = ZAF_NULLP(ZafLanguageData);
		delete monthStrings;
		monthStrings = ZAF_NULLP(ZafLanguageData);
		delete timeStrings;
		timeStrings = ZAF_NULLP(ZafLanguageData);
	}
}

bool ZafUTimeData::LeapYear(void) const
{
	// Compute the leap year.
	int year;
	ConvertJDay(&year, 0, 0, 0);
	return ((year % 400 == 0) || ((year % 4 == 0) && (year % 100 != 0)) ? true : false);
}

void ZafUTimeData::MakeCanonical(void)
{
	// Compute the canonical date.
	if (uSec >= uSecsPerDay)
	{
		jDay += uSec / uSecsPerDay;
		uSec %= uSecsPerDay;
	}
	else
	{
		while (uSec < 0)
		{
			jDay--;
			uSec += uSecsPerDay;
		}
	}
}

int ZafUTimeData::MilliSecond(void) const
{
	// Compute the value.
	int milliSecond;
	ConvertUSec(0, 0, 0, &milliSecond);
	return (milliSecond);
}

int ZafUTimeData::Minute(void) const
{
	// Compute the value.
	int minute;
	ConvertUSec(0, &minute, 0, 0);
	return (minute);
}

int ZafUTimeData::Month(void) const
{
	// Compute the value.
	int month;
	ConvertJDay(0, &month, 0, 0);
	return (month);
}

int ZafUTimeData::Second(void) const
{
	// Compute the value.
	int second;
	ConvertUSec(0, 0, &second, 0);
	return (second);
}

ZafError ZafUTimeData::SetBasisYear(int tBasisYear)
{
	// Reset the basis year.
	// basisYear may be any year, and it is not required to be divisible by 100.
	// For example, if basisYear is 1950, a 2-digit year specified by the user
	// will fall in the range 1950..2049.
	PushLevel();
	basisYear = tBasisYear;
	PopLevel();

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafError ZafUTimeData::SetUTime(ZafUInt32 seconds)
{
	// Change the time values.
	PushLevel();
	SetUTime(1970, 0, 0, 0, 0, 0, 0);
	uSec += seconds;
	jDay += uSec / uSecsPerDay;
	uSec %= uSecsPerDay;
	PopLevel();

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafError ZafUTimeData::SetUTime(const ZafIChar *buffer, const ZafIChar *format)
{
	// Reset the time based on the buffer.
	PushLevel();
	if (!Sscanf(buffer, format ? format : zafLocale->dateTimeStringInputFormat))
	{
		PopLevel();
		return (ZAF_ERROR_OUT_OF_RANGE);
	}
	PopLevel();

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafError ZafUTimeData::SetUTime(int year, int month, int day, int hour,
	int minute, int second, int millisecond)
{
	PushLevel();

	if (year + month == 0)
		jDay = day;
	else
	{
		// Compute the Julian date from three integer values using
		// algorithm 4 of Practical Astronomy with your Calculator.
		// This include a fix for the Gregorian calendar change in
		// 1582.  Protestant countries should change that value to
		// when they adopted the Gregorian calendar.
		if (month <= 2)
		{
			month += 12;
			year--;
		}
		jDay = 365L * year + (153L * (month + 1)) / 5 + day + 1720995L;
		jDay += ((year >= 0) ? year : (year-3)) / 4;
		// Gregorian date fix
		if ((jDay > (long)(zafLocale->beginGregorian + zafLocale->skipGregorian)) ||
		    (jDay > 2299160L + 11))
//		if (jDay > zafLocale->beginGregorian + zafLocale->skipGregorian)
			jDay += 2 - (year / 100) + ((year / 100) / 4);
	}
	uSec = millisecond + 1000L * (second + 60L * (minute + 60L * hour));

	MakeCanonical();

	// If all zeros are sent in, mark this as "blank".
	if (year + month + day + uSec == 0)
		uSec = -1;

	PopLevel();

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafError ZafUTimeData::SetUTime(const ZafUTimeData &utime)
{
	// Change the time values.
	PushLevel();
	jDay = utime.jDay;
	uSec = utime.uSec;
	PopLevel();

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafError ZafUTimeData::SetZoneOffset(int tZoneOffset)
{
	// Reset the zone offset.
	PushLevel();
	zoneOffset = tZoneOffset;
	PopLevel();

	// Return success.
	return (ZAF_ERROR_NONE);
}

long ZafUTimeData::Value(void) const
{
	return (Difference(ZafUTimeData(1900, 0, 0, 0, 0, 0, 0)));
}

int ZafUTimeData::Year(void) const
{
	// Compute the value.
	int year;
	ConvertJDay(&year, 0, 0, 0);
	return (year);
}

// ----- Operators ----------------------------------------------------------

ZafUTimeData ZafUTimeData::operator+(const ZafUTimeData &utime)
{
	// Add the specified time value.
	ZafUTimeData tmp = *this;
	tmp += utime;
	return tmp;
}

ZafUTimeData ZafUTimeData::operator-(const ZafUTimeData &utime)
{
	// Subtract the specified time value.
	ZafUTimeData tmp = *this;
	tmp -= utime;
	return tmp;
}

ZafUTimeData ZafUTimeData::operator++(void)
{
	// Reset the value.
	PushLevel();
	*this += 1L;
	PopLevel();

	// Return a reference pointer.
	return *this;
}

ZafUTimeData ZafUTimeData::operator++(int)
{
	// Reset the value.
	PushLevel();
	ZafUTimeData tmp = *this;
	*this += 1L;
	PopLevel();

	// Return a temporary reference pointer.
	return tmp;
}

ZafUTimeData ZafUTimeData::operator--(void)
{
	// Reset the value.
	PushLevel();
	*this -= 1L;
	PopLevel();

	// Return a reference pointer.
	return *this;
}

ZafUTimeData ZafUTimeData::operator--(int)
{
	// Reset the value.
	PushLevel();
	ZafUTimeData tmp = *this;
	*this -= 1L;
	PopLevel();

	// Return a temporary reference pointer.
	return tmp;
}

ZafUTimeData &ZafUTimeData::operator=(const ZafUTimeData &utime)
{
	// Reset the time values.
	SetUTime(utime);
	return (*this);
}

ZafUTimeData &ZafUTimeData::operator=(long seconds)
{
	// Reset the time values.
	SetUTime(seconds);
	return (*this);
}

ZafUTimeData &ZafUTimeData::operator+=(const ZafUTimeData &utime)
{
	// Reset the time values.
	PushLevel();
	jDay += utime.jDay;
	uSec += utime.uSec;
	MakeCanonical();
	PopLevel();

	// Return a reference pointer.
	return (*this);
}

ZafUTimeData &ZafUTimeData::operator-=(const ZafUTimeData &utime)
{
	// Reset the time values.
	PushLevel();
	jDay -= utime.jDay;
	uSec -= utime.uSec;
	MakeCanonical();
	PopLevel();

	// Return a reference pointer.
	return (*this);
}

ZafUTimeData ZafUTimeData::operator+(long seconds)
{
	return (*this + ZafUTimeData(seconds));
}

ZafUTimeData ZafUTimeData::operator-(long seconds)
{
	return (*this - ZafUTimeData(seconds));
}

// ----- sprintf/sscanf functions -------------------------------------------

void ZafUTimeData::Format(va_list *argList, ZafIChar **ptrBuffer, ZafIChar, int, int, int)
{
	ZafUTimeData *utime = va_arg(*argList, ZafUTimeData *);
	int len = utime->strftime(*ptrBuffer, 256, ZAF_NULLP(ZafIChar));
	*ptrBuffer += len;
}

int ZafUTimeData::Parse(va_list *argList, ZafIChar **ptrBuffer,
	ZafIChar , int , int , const ZafIChar **)
{
	ZafUTimeData *utime = va_arg(*argList, ZafUTimeData *);
	ZafIChar *newPtr = utime->strptime(*ptrBuffer, ZAF_NULLP(ZafIChar));
	if (newPtr != *ptrBuffer)
	{
		*ptrBuffer = newPtr;
		return 1;
	}
	return (0);
}

int ZafUTimeData::Sprintf(ZafIChar *buffer, int maxLength, const ZafIChar *format) const
{
	// Get the output format, defined as the first format in the buffer.
	ZafIChar oFormat[ZAF_FORMAT_LENGTH];
	ParseFormat(oFormat, format);
	return (strftime(buffer, maxLength, oFormat));
}

int ZafUTimeData::Sscanf(const ZafIChar *buffer, const ZafIChar *format)
{
	bool match = false;
	ZafIChar iFormat[ZAF_FORMAT_LENGTH];
	format = ParseFormat(iFormat, format);
	while (!match && *iFormat)
	{
		if (!strptime(buffer, iFormat))
			format = ParseFormat(iFormat, format);
		else
			match = true;
	}
	return (match ? 1 : 0);
}

int ZafUTimeData::strftime(ZafIChar *string, int maxsize, const ZafIChar *format) const
{
	// Check for valid utime members.
	if (jDay == 0 && uSec < 0)
	{
		*string = 0;
		return (0);
	}
	// Check for a valid locale.
	int currentMaxsize = maxsize;

	// Get the current utime values.
	int hour, minute, second, milliSecond;
	ConvertUSec(&hour, &minute, &second, &milliSecond);
	int pYear, pMonth, pDay, pDayOfWeek;
	ConvertJDay(&pYear, &pMonth, &pDay, &pDayOfWeek);

	int pDayOfYear = 0;
	for (int j = 0; j < pMonth-1; j++)
		pDayOfYear += _daysInMonth[j];
	pDayOfYear += pDay;
	if (LeapYear() && pMonth > 2)
		pDayOfYear++;

	ZafUTimeData firstDay(pYear, 1, 1, 0, 0, 0, 0);
	int firstDayOfYear = firstDay.DayOfWeek();

	ZafIChar *ECval = ZAF_NULLP(ZafIChar), *EYval = ZAF_NULLP(ZafIChar);
	int Eyval = 0, i = 0;
	for ( ; i < zafLocale->eraTableLength; i++)
	{
		if ((long)zafLocale->eraTable[i].startDate <= jDay &&
			jDay <= (long)zafLocale->eraTable[i].endDate)
		{
			ECval = zafLocale->eraTable[i].eraName;
			EYval = zafLocale->eraTable[i].eraFormat;
			ZafUTimeData tmp; tmp.jDay = zafLocale->eraTable[i].startDate;
			int baseYear;
			tmp.ConvertJDay(&baseYear, ZAF_NULLP(int), ZAF_NULLP(int), ZAF_NULLP(int));
			if (zafLocale->eraTable[i].direction)
				Eyval = zafLocale->eraTable[i].offset + pYear - baseYear;
			else
				Eyval = zafLocale->eraTable[i].offset + baseYear - pYear;
		}
	}

	// Check the format information.
	for (i = 0; format[i]; i++)
	{
		int doCopy = true;
		ZafIChar tmpStr[ZAF_FORMAT_LENGTH];
		const ZafIChar *copyStr = tmpStr;
		if (format[i] == '%')
		{
			i++;
			if (format[i] == 0)
				break;
			switch (format[i])
			{
			case 'a':
				copyStr = dayStrings->GetMessage((ZafNumberID)(ZAF_ABDAY_1+pDayOfWeek));
				break;
			case 'A':
				copyStr = dayStrings->GetMessage((ZafNumberID)(ZAF_DAY_1+pDayOfWeek));
				break;
			case 'b':
				copyStr = monthStrings->GetMessage((ZafNumberID)(ZAF_ABMON_1+pMonth-1));
				break;
			case 'B':
				copyStr = monthStrings->GetMessage((ZafNumberID)(ZAF_MON_1+pMonth-1));
				break;
			case 'c':
				maxsize -= strftime(string, maxsize, zafLocale->dateTimeStringOutputFormat);
				doCopy = false;
				break;
			case 'C':
				itoa(pYear / 100, tmpStr, 10, 2);
				break;
			case 'd':
				itoa(pDay, tmpStr, 10, 2);
				break;
			case 'D':
				maxsize -= strftime(string, maxsize, _dFormat);
				doCopy = false;
				break;
			case 'e':
				itoa(pDay, tmpStr, 10, 0);
				break;
			case 'E':	// Era conversions
				i++;
				switch (format[i])
				{
				case 'C':
					if (ECval)
						copyStr = ECval;
					else
						itoa(pYear / 100, tmpStr, 10, 2);
					break;
				case 'y':
					if (ECval)
						itoa(Eyval, tmpStr, 10, 0);
					else
						itoa(pYear % 100, tmpStr, 10, 2);
					break;
				case 'Y':
					maxsize -= strftime(string, maxsize, EYval);
					doCopy = false;
					break;
				}
				break;
			case 'g':
				copyStr = monthStrings->GetMessage((ZafNumberID)(ZAF_SHMON_1+pMonth-1));
				break;
			case 'G':
				copyStr = dayStrings->GetMessage((ZafNumberID)(ZAF_SHDAY_1+pDayOfWeek));
				break;
			case 'h':
				itoa(hour, tmpStr, 10, 0);
				break;
			case 'H':
				itoa(hour, tmpStr, 10, 2);
				break;
			case 'i':
			{
				int tmp = hour % 12;
				itoa((tmp == 0 ? 12 : tmp), tmpStr, 10, 0);
				break;
			}
			case 'I':
			{
				int tmp = hour % 12;
				itoa((tmp == 0 ? 12 : tmp), tmpStr, 10, 2);
				break;
			}
			case 'j':
				itoa(pDayOfYear, tmpStr, 10, 3);
				break;
			case 'J':
				itoa(milliSecond / 100, tmpStr, 10, 1);
				break;
			case 'k':
				itoa(milliSecond / 10, tmpStr, 10, 2);
				break;
			case 'K':
				itoa(milliSecond, tmpStr, 10, 3);
				break;
			case 'l':
			case 'L':
				i++;
				switch (format[i])
				{
				case 'm':
					itoa(12*pYear + pMonth, tmpStr, 10, 0);
					break;
				case 'd':
					itoa(jDay, tmpStr, 10, 0);
					break;
				case 'H':
					itoa(24 * jDay + uSec / (1000*60*60), tmpStr, 10, 0);
					break;
				case 'M':
					itoa(24 * 60 * jDay + uSec / (1000*60), tmpStr, 10, 0);
					break;
				case 'S':
					itoa(24 * 60 * 60 * jDay + uSec / (1000), tmpStr, 10, 0);
					break;
				case 0:
					i--;
					break;
				}
				break;
			case 'm':
				itoa(pMonth, tmpStr, 10, 2);
				break;
			case 'M':
				itoa(minute, tmpStr, 10, 2);
				break;
			case 'n':
				tmpStr[0] = '\n';
				tmpStr[1] = 0;
				break;
			// case 'O':	alternate digits
			case 'p':
				copyStr = timeStrings->GetMessage((hour < 12) ? ZAF_AM : ZAF_PM);
				break;
			case 'r':
				maxsize -= strftime(string, maxsize, zafLocale->time12StringOutputFormat);
				doCopy = false;
				break;
			case 'R':
				maxsize -= strftime(string, maxsize, _rFormat);
				doCopy = false;
				break;
			case 's':
				itoa(second, tmpStr, 10, 0);
				break;
			case 'S':
				itoa(second, tmpStr, 10, 2);
				break;
			case 't':
				tmpStr[0] = '\t';
				tmpStr[1] = 0;
				break;
			case 'T':
				maxsize -= strftime(string, maxsize, _tFormat);
				doCopy = false;
				break;
			case 'u':
				itoa((pDayOfWeek == 0 ? 7 : pDayOfWeek), tmpStr, 10, 0);
				break;
			case 'U':
			{
				int bias = (firstDayOfYear <= 0 ? 6 : firstDayOfYear - 1);
				itoa((pDayOfYear+bias)/7, tmpStr, 10, 2);
				break;
			}
			case 'v':
				itoa(pMonth, tmpStr, 10, 0);
				break;
			case 'V':
				itoa(minute, tmpStr, 10, 0);
				break;
			case 'w':
				itoa(pDayOfWeek, tmpStr, 10, 0);
				break;
			case 'W':
			{
				int bias = (firstDayOfYear <= 1 ? 5 + firstDayOfYear : firstDayOfYear - 2);
				itoa((pDayOfYear+bias)/7, tmpStr, 10, 2);
				break;
			}
			case 'x':
				maxsize -= strftime(string, maxsize, zafLocale->dateStringOutputFormat);
				doCopy = false;
				break;
			case 'X':
				maxsize -= strftime(string, maxsize, zafLocale->timeStringOutputFormat);
				doCopy = false;
				break;
			case 'y':
				itoa(pYear % 100, tmpStr, 10, 2);
				break;
			case 'Y':
				itoa(pYear, tmpStr, 10, 0);
				break;
			case 'Z':
				copyStr = timeStrings->GetMessage(ZAF_TZ);
				break;
			case '%':
				tmpStr[0] = '%';
				tmpStr[1] = 0;
				break;
			default:
				tmpStr[0] = 0;
				break;
			}
		}
		else
		{
			tmpStr[0] = format[i];
			tmpStr[1] = 0;
		}
		if (doCopy)
		{
			maxsize -= strlen(copyStr);
			if (maxsize <= 0)
				return (0);
			strcpy(string, copyStr);
		}
		string += strlen(string);
	}
	*string = format[i];
	return (currentMaxsize-maxsize);
}

ZafIChar *ZafUTimeData::strptime(const ZafIChar *string, const ZafIChar *format)
{
	if (!string)
	{
		jDay = 0;
		uSec = -1;
		return ((ZafIChar *)string);
	}

	if (*string == 0)
	{
		jDay = 0;
		uSec = -1;
		return ((ZafIChar *)string);
	}
	if (!format)
		format = zafLocale->dateTimeStringInputFormat;
	if (recurse)
		jDay = uSec = 0;
	int saveRecurse = recurse;

	// Get the default utime values.
	int hour, minute, second, milliSecond;
	ConvertUSec(&hour, &minute, &second, &milliSecond);
	int pYear, pMonth, pDay, pDayOfWeek;
	ConvertJDay(&pYear, &pMonth, &pDay, &pDayOfWeek);

	// Check the format information.
	int pDayOfYear;
	int pWeek;
	int timeOffset = 0;
	for (int i=0; format[i]; i++)
	{
//??? White space should be ignored?
//		if (IsSpace(format[i]))
//		{
//			while (IsSpace(*nextChar))
//			       nextChar++;
//		}
//		else if (format[i] == '%')
		while (IsSpace(*string))
	       string++;
		ZafIChar *nextChar = (ZafIChar *)string;
		if (format[i] == '%')
		{
			i++;
			if (format[i] == 0)
				return (ZAF_NULLP(ZafIChar));
			switch (format[i])
			{
			case 'a':
			case 'A':
			{
				int count = 0;
				for (int j = 0; j < 7; j++)
				{
					const ZafIChar *tmp = dayStrings->GetMessage((ZafNumberID)(j+ZAF_ABDAY_1));
					if (strnicmp(string, tmp, strlen(tmp)) == 0)
					{
						count++;
						pDayOfWeek = j;
						nextChar += strlen(tmp);
						continue;
					}
					tmp = dayStrings->GetMessage((ZafNumberID)(j+ZAF_DAY_1));
					if (strnicmp(string, tmp, strlen(tmp)) == 0)
					{
						count++;
						pDayOfWeek = j;
						nextChar += strlen(tmp);
						continue;
					}
				}
				if (count != 1)
					return (ZAF_NULLP(ZafIChar));
				break;
			}
			case 'h':
			case 'b':
			case 'B':
			{
				int count = 0;
				for (int j = 0; j < 12; j++)
				{
					const ZafIChar *tmp = monthStrings->GetMessage((ZafNumberID)(j+ZAF_MON_1));
					if (strnicmp(string, tmp, strlen(tmp)) == 0)
					{
						count++;
						pMonth = j + 1;
						nextChar += strlen(tmp);
						continue;
					}
					tmp = monthStrings->GetMessage((ZafNumberID)(j+ZAF_ABMON_1));
					if (strnicmp(string, tmp, strlen(tmp)) == 0)
					{
						count++;
						pMonth = j + 1;
						nextChar += strlen(tmp);
						continue;
					}
					tmp = monthStrings->GetMessage((ZafNumberID)(j+ZAF_SHMON_1));
					if (strnicmp(string, tmp, strlen(tmp)) == 0)
					{
						count++;
						pMonth = j + 1;
						nextChar += strlen(tmp);
						continue;
					}
				}
				if (count != 1)
					return (ZAF_NULLP(ZafIChar));
				break;
			}
			case 'c':
				recurse = true;
				nextChar = strptime(string, zafLocale->dateTimeStringInputFormat);
				break;
			case 'C': // This doesn't match the XOpen guide
				pYear = (int)strtol(string, &nextChar, 10);
				pYear *= 100;
				break;
			case 'd':
				pDay = (int)strtol(string, &nextChar, 10);
				if (pDay < 1 || pDay > 31)
					return (ZAF_NULLP(ZafIChar));
				break;
			case 'D':
				{
				recurse = true;

				ZafIChar iFormat[ZAF_FORMAT_LENGTH];
				const ZafIChar *tFormat = ParseFormat(iFormat, zafLocale->dateStringInputFormat);
				while (*iFormat)
				{
					nextChar = strptime(string, iFormat);
					if (!nextChar)
						tFormat = ParseFormat(iFormat, tFormat);
					else
						break;
				}
				if (nextChar)
					ConvertJDay(&pYear, &pMonth, &pDay, &pDayOfWeek);
				else
					return ZAF_NULLP(ZafIChar);
				}
				break;

			// case 'e':	see 'd'
			// case 'E':	// Era conversions
			// case 'h':	see 'b'
			case 'H':
				hour = (int)strtol(string, &nextChar, 10);
				if (hour < 0 || hour > 23)
					return (ZAF_NULLP(ZafIChar));
				break;
			case 'I':
				hour = (int)strtol(string, &nextChar, 10);
				if (hour < 1 || hour > 12)
					return (ZAF_NULLP(ZafIChar));
				break;
			case 'j':
				pDayOfYear = (int)strtol(string, &nextChar, 10);
				if (pDayOfYear < 1 || pDayOfYear > 366)
					return (ZAF_NULLP(ZafIChar));
				break;
			case 'm':
				pMonth = (int)strtol(string, &nextChar, 10);
				if (pMonth < 1 || pMonth > 12)
					return (ZAF_NULLP(ZafIChar));
				break;
			case 'M':
				minute = (int)strtol(string, &nextChar, 10);
				if (minute < 0 || minute > 59)
					return (ZAF_NULLP(ZafIChar));
				break;
			case 't':
			case 'n':
				while (IsSpace(*nextChar))
				       nextChar++;
				break;
			// case 'O':	????
			case 'p':
				{
				const ZafIChar *tmp = timeStrings->GetMessage(ZAF_AM);
				if (strnicmp(string, tmp, strlen(tmp)) == 0)
				{
					nextChar += strlen(tmp);
					if (hour == 12)
						timeOffset = -12;
					break;
				}
				tmp = timeStrings->GetMessage(ZAF_PM);
				if (strnicmp(string, tmp, strlen(tmp)) == 0)
				{
					if (hour != 12)
						timeOffset = 12;
					nextChar += strlen(tmp);
					break;
				}
				return (ZAF_NULLP(ZafIChar));
				}
			case 'r': // This doesn't match the XOpen guide
				recurse = true;
				nextChar = strptime(string, zafLocale->timeStringInputFormat);
				break;
			case 'R':
				recurse = true;
				nextChar = strptime(string, _rFormat);
				break;
			case 'S':
				second = (int)strtol(string, &nextChar, 10);
				if (second < 0 || second > 61)
					return (ZAF_NULLP(ZafIChar));
				break;
			// case 't':	see 'n'
			case 'T':
				{
				recurse = true;

				ZafIChar iFormat[ZAF_FORMAT_LENGTH];
				const ZafIChar *tFormat = ParseFormat(iFormat, zafLocale->timeStringInputFormat);
				while (*iFormat)
				{
					nextChar = strptime(string, iFormat);
					if (!nextChar)
						tFormat = ParseFormat(iFormat, tFormat);
					else
						break;
				}
				if (nextChar)
					ConvertUSec(&hour, &minute, &second, &milliSecond);
				else
					return ZAF_NULLP(ZafIChar);
				}
				break;

			case 'U':
				pWeek = (int)strtol(string, &nextChar, 10);
				if (pWeek < 0 || pWeek > 55)
					return (ZAF_NULLP(ZafIChar));
				break;
			case 'w':
				pDayOfWeek = (int)strtol(string, &nextChar, 10);
				if (pDayOfWeek < 0 || pDayOfWeek > 6)
					return (ZAF_NULLP(ZafIChar));
				break;
			case 'W':
				pDayOfWeek = (int)strtol(string, &nextChar, 10);
				if (pDayOfWeek < 1 || pDayOfWeek > 7)
					return (ZAF_NULLP(ZafIChar));
				if (pDayOfWeek == 7)
					pDayOfWeek = 0;
				break;
			case 'x':
				recurse = true;
				nextChar = strptime(string, zafLocale->dateStringInputFormat);
				break;
			case 'X':
				recurse = true;
				nextChar = strptime(string, zafLocale->timeStringInputFormat);
				break;
			case 'y':
				{
				pYear = (int)strtol(string, &nextChar, 10);
				if (nextChar == string || pYear < 0 || pYear > 99)
					return (ZAF_NULLP(ZafIChar));

				// Support for basisYear not required to be divisible by 100.
				int basisYearShort = basisYear % 100;
				int basisYearCentury = basisYear - basisYearShort;
				if (pYear < basisYearShort)
					basisYearCentury += 100;

				pYear += basisYearCentury;
				break;
				}
			case 'Y':
				pYear = (int)strtol(string, &nextChar, 10);
				if (nextChar == string)
					return (ZAF_NULLP(ZafIChar));
				break;

			case 'Z':
			{
				for (int j = 0; _timeZones[j].name; j++)
					if (strnicmp(string, _timeZones[j].name, strlen(_timeZones[j].name)) == 0)
					{
						zoneOffset = _timeZones[j].offset;
						nextChar += strlen(_timeZones[j].name);
						break;
					}
				break;
			}
			case '%':
				if (*nextChar != '%')
					return (ZAF_NULLP(ZafIChar));
				nextChar++;
				break;
			default:
				break;
			}
		}
		else
		{
			if (format[i] != *nextChar)
				return (ZAF_NULLP(ZafIChar));
			nextChar++;
		}
		recurse = saveRecurse;

		// Check to see if match string has run out of match data.
		if (nextChar == ZAF_NULLP(ZafIChar))
			return (ZAF_NULLP(ZafIChar));
		string = nextChar;
	}

	// Final check for errors.
	if (SetUTime(pYear, pMonth, pDay, hour + timeOffset, minute, second, 0) == ZAF_ERROR_NONE)
		return ((ZafIChar *)string);
	return (ZAF_NULLP(ZafIChar));
}

// ----- Persistent functions -----------------------------------------------

ZafUTimeData::ZafUTimeData(const ZafIChar *name, ZafDataPersistence &persist) :
	ZafFormatData(name, persist.PushLevel(className, classID, ZAF_PERSIST_FILE))
{
	Bind();

	// Read the data.
	ZafUTimeData::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafUTimeData::Read(const ZafIChar *name, ZafDataPersistence &persist)
{
	return (new ZafUTimeData(name, persist));
}

void ZafUTimeData::ReadData(ZafDataPersistence &persist)
{
	// Read the data.
	ZafInt32 tJDay, tUSec;
	ZafInt16 tBasisYear, tZoneOffset;
	ZafFile *file = persist.File();
	*file >> tJDay >> tUSec >> tBasisYear >> tZoneOffset;
	jDay = tJDay;
	uSec = tUSec;
	basisYear = tBasisYear;
	zoneOffset = tZoneOffset;
}

void ZafUTimeData::Write(ZafDataPersistence &persist)
{
	// Write the object.
	ZafFormatData::Write(persist.PushLevel(className, classID, ZAF_PERSIST_FILE));
	ZafUTimeData::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafUTimeData::WriteData(ZafDataPersistence &persist)
{
	// Write the data.
	ZafInt32 tJDay = jDay, tUSec = uSec;
	ZafInt16 tBasisYear = basisYear, tZoneOffset = zoneOffset;
	ZafFile *file = persist.File();
	*file << tJDay << tUSec << tBasisYear << tZoneOffset;
}

