//	Zinc Application Framework - Z_TIME1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_time1.hpp>
#include <z_error.hpp>
#include <z_string.hpp>
#include <z_app.hpp>
#define ZAF_TIME_INFO
#include "lang_def.cpp"
#include "gbl_def.cpp"

static ZafIChar timeSeparator[3] = { '.','.',0 };
static ZafIChar rangeSeparator[3] = { ',',' ',0 };
static ZafStringID _timeDataName = ZAF_ITEXT("timeData");

// ----- ZafTime -----------------------------------------------------------

ZafTime::ZafTime(int left, int top, int width, ZafTimeData *zTimeData) :
	ZafString(left, top, width, new ZafStringData),
	timeData(ZAF_NULLP(ZafTimeData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the time data.
	SetTimeData(zTimeData ? zTimeData : new ZafTimeData);
}

ZafTime::ZafTime(int left, int top, int width, int hour, int minute,
	int second, int milliSecond) :
	ZafString(left, top, width, new ZafStringData),
	timeData(ZAF_NULLP(ZafTimeData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the time data.
	SetTimeData(new ZafTimeData(hour, minute, second, milliSecond));
}

ZafTime::ZafTime(const ZafTime &copy) :
	ZafString(copy), timeData(ZAF_NULLP(ZafTimeData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Copy the time data.
	if (copy.TimeData()->Destroyable())
		SetTimeData(new ZafTimeData(*copy.TimeData()));
	else
		SetTimeData(copy.TimeData());
}

ZafTime::~ZafTime(void)
{
	// Remove the data notification.
	timeData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Check the language and local information.
	LanguageFree();

	// Restore the time information.
	if (timeData->Destroyable() && timeData->NotifyCount() == 0)
		delete timeData;
}

ZafError ZafTime::Decrement(ZafData *data)
{
	// Check for a valid object.
	ZafTimeData *time = DynamicPtrCast(data, ZafTimeData);
	if (!time)
		return (ZAF_ERROR_INVALID_CLASS);

	// Decrement the value.
	ZafTimeData _time(*timeData);
	OSGetTime();
	*timeData -= *time; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafWindowObject *ZafTime::Duplicate(void)
{
	return (new ZafTime(*this));
}

ZafEventType ZafTime::Event(const ZafEventStruct &event)
{
	// Process OS specific messages.
	ZafEventType ccode = event.type;
	if (event.InputType() == E_KEY)
		ccode = LogicalEvent(event);

	// Check for zinc events.
	switch (ccode)
	{
	// ----- Create messages -----
	case S_INITIALIZE:
		// Check for an unanswered field.
		if (!Unanswered())
			OSSetTime();
		ccode = ZafString::Event(event);
		break;

	// ----- Action messages -----
	case L_SELECT:
	case N_NON_CURRENT:
		{
		// Keep a temporary time in case of error.
		ZafTimeData oldValue(*timeData);
		OSGetTime();

		// Call the associated user function.
		ccode = ZafString::Event(event);

		// Check for errors.
		if (ccode != 0 && ccode != ZAF_ERROR_LEAVE_INVALID)
			timeData->SetTime(oldValue);

		// Set the formatted bignum text into the field.
		OSSetTime();
		}
		break;

	case S_COPY_DATA:
		{
		ZafWindowObject *object = event.windowObject;
		ZafTime *time = DynamicPtrCast(object, ZafTime);
		if (time)
			timeData->SetTime(*time->TimeData());
		}
		break;

	case S_SET_DATA:
		if (event.windowObject)
		{
			ZafWindowObject *object = event.windowObject;
			ZafTime *time = DynamicPtrCast(object, ZafTime);
			if (time)
				SetTimeData(time->TimeData());
		}
		else
			SetTimeData(new ZafTimeData(*timeData));
		break;

	// ----- Default or unknown messages -----
	default:
		ccode = ZafString::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafError ZafTime::Increment(ZafData *data)
{
	// Check for a valid object.
	ZafTimeData *time = DynamicPtrCast(data, ZafTimeData);
	if (!time)
		return (ZAF_ERROR_INVALID_CLASS);

	// Increment the value.
	ZafTimeData _time(*timeData);
	OSGetTime();
	*timeData += *time; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

void ZafTime::LanguageAllocate(void)
{
	// Check for initialization.
	if (errorStrings)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!errorStrings && zafDataManager)
		errorStrings = DynamicPtrCast(zafDataManager->AllocateData(className, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

	// Default to code initialization.
	if (!errorStrings)
		errorStrings = new ZafLanguageData(defaultErrorStrings);
}

void ZafTime::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && errorStrings && errorStrings->Destroyable() &&
		errorStrings->NotifyCount() == 0)
	{
		delete errorStrings;
		errorStrings = ZAF_NULLP(ZafLanguageData);
	}
}

ZafError ZafTime::OSGetTime(void)
{
	// Disable notification.
	ZafUpdateType update = timeData->Update(this);
	timeData->SetUpdate(this, ZAF_UPDATE_NONE);

	// Set the time based on the string data.
	OSGetText();
	error = timeData->SetTime(StringData()->Text(), InputFormatText());

	// Restore notification.
	timeData->SetUpdate(this, update);
	return (error);
}

ZafError ZafTime::OSSetTime(void)
{
	// Set the string based on the time data.
	ZafIChar text[ZAF_STRING_DATA_LENGTH];
	timeData->FormattedText(text, ZAF_STRING_DATA_LENGTH, OutputFormatText());
	stringData->SetText(text);

	// Return the current status.
	return (ZAF_ERROR_NONE);
}

ZafError ZafTime::SetTimeData(ZafTimeData *tTimeData)
{
	// Remove the data notification.
	if (timeData)
		timeData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Reset the time data.
	if (timeData && timeData != tTimeData &&
		timeData->Destroyable() && timeData->NotifyCount() == 0)
		delete timeData;
	timeData = tTimeData ? tTimeData : new ZafTimeData;
	if (!timeData->StringID())
		timeData->SetStringID(_timeDataName);

	// Add the data notification.
	timeData->AddNotification(this, (ZafUpdateFunction)Update);
	return (OSSetTime());
}

bool ZafTime::SetLowerCase(bool )
{
	// lowerCase is set to false by this class.
	lowerCase = false;
	return (lowerCase);
}

bool ZafTime::SetPassword(bool )
{
	// password is set to false by this class.
	password = false;
	return (password);
}

ZafError ZafTime::SetStringData(ZafStringData *)
{
	// String data cannot be set for this class.
	return (ZAF_ERROR_INVALID);
}

bool ZafTime::SetUpperCase(bool )
{
	// upperCase is set to false by this class.
	upperCase = false;
	return (upperCase);
}

ZafError ZafTime::Validate(bool processError)
{
	// Check for an absolute time error. Don't set the error member to ZAF_ERROR_NONE.
	ZafTimeData currentTime;
	ZafIChar *stringTime = stringData->DynamicText();
	ZafError tError = currentTime.SetTime(stringTime, InputFormatText());

	// Check for a range error.
	if (RangeText() && tError == ZAF_ERROR_NONE)
		tError = ZAF_ERROR_OUT_OF_RANGE;
	int numRanges = 0;
	ZafTimeData low, high;
	ZafIChar minTime[ZAF_STRING_DATA_LENGTH];
	ZafIChar maxTime[ZAF_STRING_DATA_LENGTH];
	ZafIChar rBuffer[1024]; // Localized range string for error messages.
	rBuffer[0] = '\0';
	for (const ZafIChar *tRange = RangeText(); tRange && tError == ZAF_ERROR_OUT_OF_RANGE; numRanges++)
	{
		tRange = ParseRange(tRange, minTime, maxTime);
		low.SetTime(minTime, OutputFormatText());
		high.SetTime(maxTime, OutputFormatText());
		if ((!minTime[0] || currentTime >= low) && (!maxTime[0] || currentTime <= high))
			tError = ZAF_ERROR_NONE;
		else if (processError)
		{
			if (rBuffer[0])
				strcat(rBuffer, rangeSeparator);
			if (minTime[0])
				low.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
			strcat(rBuffer, timeSeparator);
			if (maxTime[0])
				high.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Process the error code.
	SetInvalid(false);
	if (tError == ZAF_ERROR_NONE)
	{
		// Set up the new time.
		currentTime.FormattedText(stringTime, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		SetText(stringTime);
		delete []stringTime;
		return (ZAF_ERROR_NONE);
	}
	else
	{
		// Keep the error code.
		error = tError;
		if (!zafErrorSystem)
		{
			// Restore the original time.
			timeData->FormattedText(stringTime, ZAF_STRING_DATA_LENGTH, OutputFormatText());
			SetText(stringTime);
			delete []stringTime;
			return (error);
		}
		else if (!processError)
		{
			delete []stringTime;
			return (error);
		}
	}

	// Check for open-ended range errors.
	if (error == ZAF_ERROR_OUT_OF_RANGE && numRanges == 1)
	{
		if (minTime[0] && !maxTime[0])
		{
			error = ZAF_ERROR_LESS_THAN_RANGE;
			low.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
		else if (!minTime[0] && maxTime[0])
		{
			error = ZAF_ERROR_GREATER_THAN_RANGE;
			high.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Generate the error message and wait for a response.
	ZafIChar *title = ZafLanguageData::blankString;
	ZafIChar *errorString = errorStrings->GetMessage((ZafNumberID)error, true);
	if (errorString &&
		zafErrorSystem->ReportError(windowManager, title, ZAF_DIALOG_OK | ZAF_DIALOG_CANCEL, errorString, stringTime, rBuffer) == S_DLG_OK)
		error = ZAF_ERROR_LEAVE_INVALID; // Keep the new value.
	delete []stringTime;
	return (error);
}

bool ZafTime::SetVariableName(bool )
{
	// variableName is set to false by this class.
	variableName = false;
	return (variableName);
}

// ----- Persistent functions -----------------------------------------------

ZafTime::ZafTime(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafString(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	timeData(ZAF_NULLP(ZafTimeData))
{
	// Check the language and local information.
	LanguageAllocate();

	ZafTime::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafTime::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafTime(name, persist));
}

void ZafTime::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafIChar timeName[ZAF_MAXNAMELEN];
	ZafFile *file = persist.File();
	*file >> timeName;
	SetTimeData(timeName[0] ? new ZafTimeData(timeName, persist) : new ZafTimeData);
}

void ZafTime::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafString::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafTime::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafTime::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	const ZafIChar *timeName = timeData ? timeData->StringID() : ZAF_NULLP(ZafIChar);
	ZafFile *file = persist.File();
	*file << timeName;
	if (timeData)
		timeData->Write(persist);
}

