//	Zinc Application Framework - Z_TABLE2.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_table.hpp>
#include <z_str1.hpp>

// ----- ZafTableRecord -----------------------------------------------------

ZafTableRecord::ZafTableRecord(int width, int height) :
	ZafWindow(0, 0, width, height),
	offset(-1)
{
}

ZafTableRecord::ZafTableRecord(const ZafTableRecord &copy) :
	ZafWindow(copy),
	offset(-1)
{
}

ZafWindowObject *ZafTableRecord::Add(ZafWindowObject *object, ZafWindowObject *position)
{
	// Prevent object registration unless the object is already on the screen.
	if (!screenID)
		object->SetSystemObject(false);

	// Add the object to the table record.
	return (ZafWindow::Add(object, position));
}

void ZafTableRecord::ConvertRegion(ZafRegionStruct &region, ZafCoordinateType newType)
{
	// Go directly to region convert so preSpace and postSpace are not used.
	region.ConvertCoordinates(newType);
}

ZafWindowObject *ZafTableRecord::Duplicate(void)
{
	return (new ZafTableRecord(*this));
}

ZafEventType ZafTableRecord::Event(const ZafEventStruct &event)
{
	// Check for zinc events.
	ZafWindowObject *object;
	ZafEventType ccode = LogicalEvent(event);
	switch (ccode)
	{
	case S_CURRENT:
		ZafWindow::Event(event);

		// Update data pointers of virtualRecord in case user has a link to it.
		SetVirtualData(ccode);
		break;

	// ----- Create messages -----
	case S_REGISTER_OBJECT:
		RegisterObject();
		BroadcastEvent(event);
		break;

	case S_COMPUTE_SIZE:
		{
		int height = zafRegion.Height();
		int width = parent->MaxRegion(this).Width();
		zafRegion.top = Previous() ? Previous()->zafRegion.bottom + 1 : 0;
		zafRegion.bottom = zafRegion.top + height - 1;
		zafRegion.left = 0;
		zafRegion.right = width - 1;
		}
		break;

	// ----- Selection messages -----
	case L_BEGIN_SELECT:
		{
		// Get mouse position relative to this object.
		ZafPositionStruct mousePosition = event.position;

#		if defined(ZAF_MSDOS)
		if (!clientRegion.Overlap(mousePosition))
			return (ZafWindow::Event(event));

		//	Adjust mouse position to be relative to client region.
		mousePosition.column -= clientRegion.left;
		mousePosition.line -= clientRegion.top;
#		endif

		// Try to find an overlapping item.
		for (object = First(); object; object = object->Next())
			if (object->zafRegion.Overlap(mousePosition))
			{
				if (!object->SystemObject())
				{
					ZafTable *table = DynamicPtrCast(parent, ZafTable);
					ZafTableRecord *record = table->Current();
					if (record && record->Current())
						record->Current()->SetSystemObject(false);
					object->SetSystemObject(true);
					object->SetFocus(true);
				}
				else
				{
					ZafPositionStruct mousePosition = ConvertToOSPosition(this, &event.position);
					ZafEventStruct mEvent(event);
					mEvent.position = mousePosition;
					ZafWindow::Event(event);
				}
				break;
			}
		}
		break;

	default:
		ccode = ZafWindow::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafWindowObject *ZafTableRecord::NotifyFocus(ZafWindowObject *focusObject, bool setFocus)
{
	// If getting focus, let base class handle.  If losing focus, focusObject
	// will be the parent if focus is moving to another sibling.  Let base
	// class handle if focusObject is NOT the parent.
	if (setFocus || focusObject != parent)
		return (ZafWindow::NotifyFocus(focusObject, setFocus));

	// A different record is becoming current, so reset data pointers.
	SetVirtualData(N_NON_CURRENT);
	return (ZafWindow::NotifyFocus(focusObject, setFocus));
}

bool ZafTableRecord::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

bool ZafTableRecord::SetBordered(bool )
{
	// bordered is false for this class.
	bordered = false;
	return (bordered);
}

bool ZafTableRecord::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

bool ZafTableRecord::SetDisabled(bool )
{
	// disabled is false for this class.
	disabled = false;
	return (disabled);
}

bool ZafTableRecord::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

bool ZafTableRecord::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafTableRecord::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafTableRecord::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

bool ZafTableRecord::SetNoncurrent(bool )
{
	// noncurrent is false for this class.
	noncurrent = false;
	return (noncurrent);
}

ZafOffset ZafTableRecord::SetOffset(ZafOffset tOffset)
{
	// Make sure the offset has changed.
	if (offset != tOffset)
		offset = tOffset;

	// Return the current offset.
	return (offset);
}

bool ZafTableRecord::SetParentDrawFocus(bool )
{
	// parentDrawFocus is true for this class.
	parentDrawFocus = true;
	return (parentDrawFocus);
}

ZafRegionType ZafTableRecord::SetRegionType(ZafRegionType )
{
	// regionType is ZAF_INSIDE_REGION for this class.
	regionType = ZAF_INSIDE_REGION;
	return (regionType);
}

ZafSelectionType ZafTableRecord::SetSelectionType(ZafSelectionType )
{
	// selectionType is ZAF_SINGLE_SELECTION for this class.
	selectionType = ZAF_SINGLE_SELECTION;
	return (selectionType);
}

bool ZafTableRecord::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

bool ZafTableRecord::SetSupportObject(bool )
{
	// supportObject is false for this class.
	supportObject = false;
	return (supportObject);
}

bool ZafTableRecord::SetTemporary(bool )
{
	// temporary is false for this class.
	temporary = false;
	return (temporary);
}

void ZafTableRecord::SetVirtualData(ZafEventType ccode)
{
	// N_NON_CURRENT called from NotifyFocus.  Reset stringData pointers 
	// to own copies since no longer should be updating virtualRecord's.
	// S_CURRENT called from Event(S_CURRENT).  Becoming current.  Set 
	// virtualRecord's data to our data and then reset our stringData 
	// pointers to virtualRecord's in case user has a link to them.
	ZafEventStruct copyEvent(S_COPY_DATA);
	ZafEventStruct setEvent(S_SET_DATA);
	ZafTableRecord *virtualRecord = DynamicPtrCast(parent, ZafTable)->VirtualRecord();
	for (ZafWindowObject *recObj = First(), *virtObj = virtualRecord->First();
		recObj && virtObj; recObj = recObj->Next(), virtObj = virtObj->Next())
	{
		// Turn off updates.
		bool saveUpdate = recObj->AutomaticUpdate(false);
		recObj->SetAutomaticUpdate(false);

		if (ccode == N_NON_CURRENT)
			// Cause the field to allocate a new data object.
			setEvent.windowObject = ZAF_NULLP(ZafWindowObject);
		else if (ccode == S_CURRENT)
		{
			// Copy data to virtualRecord data fields.
			copyEvent.windowObject = recObj;
			virtObj->Event(copyEvent);

			// Cause the field to use virtualRecord's data object.
			setEvent.windowObject = virtObj;
		}

		// Cause current record fields to point to new data object.
		recObj->Event(setEvent);

		// Restore updates.
		recObj->SetAutomaticUpdate(saveUpdate);
	}
}

ZafWindowObject *ZafTableRecord::Subtract(ZafWindowObject *object)
{
	// Remove the object from the table record.
	return (ZafWindow::Subtract(object));
}

// ----- Persistent functions -----------------------------------------------

ZafTableRecord::ZafTableRecord(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	offset(-1)
{
	// Only ZafTable's virtualRecord is persisted.  Other records are created
	// at run-time as needed.

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafTableRecord::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafTableRecord(name, persist));
}

void ZafTableRecord::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafWindow::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));

	ZafTableRecord::WriteData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

