//	Zinc Application Framework - Z_TABLE1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_table.hpp>

static ZafNumberID _colNumberID = ZAF_NUMID_COLUMN_HEADER;
static ZafStringID _colStringID = ZAF_ITEXT("ZAF_COLUMN_HEADER");
static ZafNumberID _rowNumberID = ZAF_NUMID_ROW_HEADER;
static ZafStringID _rowStringID = ZAF_ITEXT("ZAF_ROW_HEADER");
static ZafNumberID _corNumberID = ZAF_NUMID_CORNER_HEADER;
static ZafStringID _corStringID = ZAF_ITEXT("ZAF_CORNER_HEADER");
static ZafStringID _virtualFieldName = ZAF_ITEXT("virtualField");

ZafTableHeader *ZafTable::ColumnTableHeader(void)
{
	for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
		if (object->NumberID() == ZAF_NUMID_COLUMN_HEADER)
			return (DynamicPtrCast(object, ZafTableHeader));
	return (ZAF_NULLP(ZafTableHeader));
}

ZafTableHeader *ZafTable::RowTableHeader(void)
{
	for (ZafWindowObject *object = SupportFirst(); object; object = object->Next())
		if (object->NumberID() == ZAF_NUMID_ROW_HEADER)
			return (DynamicPtrCast(object, ZafTableHeader));
	return (ZAF_NULLP(ZafTableHeader));
}

// ----- ZafTableHeader -----------------------------------------------------

ZafTableHeader::ZafTableHeader(int width, int height, ZafTableHeaderType tHeaderType) :
	ZafWindow(0, 0, width, height),
	headerType((ZafTableHeaderType)-1),
	virtualField(ZAF_NULLP(ZafWindowObject))
{
	SetHeaderType(tHeaderType);
	SetBordered(true);
	SetSupportObject(true);
	SetNoncurrent(true);
	SetOSDraw(false);
	SetRegionType(ZAF_AVAILABLE_REGION);
}

ZafTableHeader::ZafTableHeader(const ZafTableHeader &copy) :
	ZafWindow(copy),
	headerType((ZafTableHeaderType)-1),
	virtualField(ZAF_NULLP(ZafWindowObject))
{
	// Need to call the headerType attribute function so it can set
	// numberID and stringID.
	SetHeaderType(copy.headerType);

	if (First())
	{
		ZafWindowObject *object;

		// ZafWindow copy constructor will have duplicated row header fields,
		// which are created by the table using virtualField.  So we need
		// to remove them.
		if (headerType == ZAF_ROW_HEADER)
		{
			while (First())
			{
				object = First();
				ZafWindow::Subtract(object);
				delete object;
			}
		}
		else
		{
			// The header objects were added by the ZafWindow 
			// copy constructor using the ZafWindow::Add() function.
			// They need to be added using ZafTableHeader::Add().
			ZafList tempList;
			while (First())
			{
				object = First();
				Subtract(object);
				tempList.Add(object);
			}
			while (tempList.First())
			{
				object = DynamicPtrCast(tempList.First(), ZafWindowObject);
				tempList.Subtract(object);
				Add(object);
			}
		}
	}

	if (copy.virtualField)
		SetVirtualField(copy.virtualField->Duplicate());
}

ZafTableHeader::~ZafTableHeader(void)
{
	// Delete the virtual field.
	if (virtualField)
		delete virtualField;
}

ZafWindowObject *ZafTableHeader::Add(ZafWindowObject *object, ZafWindowObject *position)
{
	// Add the object to the table header.
	object->SetSystemObject(false); 		// Prevent object registration.
	ZafWindow::Add(object, position);

	// Return a pointer to the object.
	return (object);
}

void ZafTableHeader::ConvertRegion(ZafRegionStruct &region, ZafCoordinateType newType)
{
	// Go directly to region convert so preSpace and postSpace are not used.
	region.ConvertCoordinates(newType);
}

ZafWindowObject *ZafTableHeader::Duplicate(void)
{
	return (new ZafTableHeader(*this));
}

ZafEventType ZafTableHeader::Event(const ZafEventStruct &event)
{
	// Check for zinc events.
	ZafEventType ccode = event.type;
	switch (ccode)
	{
	case S_REGISTER_OBJECT:
		// Register the table header, but not the children.
		RegisterObject();
		BroadcastEvent(event);
		break;

	case S_COMPUTE_SIZE:
		{
		int height = zafRegion.bottom - zafRegion.top;
		int width = zafRegion.right - zafRegion.left;
		ccode = ZafWindowObject::Event(event);
		if (headerType == ZAF_CORNER_HEADER || headerType == ZAF_ROW_HEADER)
			zafRegion.right = zafRegion.left + width;
		if (headerType == ZAF_CORNER_HEADER || headerType == ZAF_COLUMN_HEADER)
			zafRegion.bottom = zafRegion.top + height;
		}
		break;

	default:
		ccode = ZafWindow::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

const ZafPaletteStruct *ZafTableHeader::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafWindow::MapClassPalette(type, state));
}

bool ZafTableHeader::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

bool ZafTableHeader::SetBordered(bool )
{
	// bordered is true for this class.
	bordered = true;
	return (bordered);
}

bool ZafTableHeader::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

bool ZafTableHeader::SetDisabled(bool )
{
	// disabled is true for this class.
	disabled = true;
	return (disabled);
}

bool ZafTableHeader::SetFocus(bool )
{
	// focus is false for this class.
	focus = false;
	return (focus);
}

ZafTableHeaderType ZafTableHeader::SetHeaderType(ZafTableHeaderType setHeaderType)
{
	// Make sure the attribute has changed.
	if (headerType != setHeaderType && !screenID)
	{
		headerType = setHeaderType;
		if (HeaderType() == ZAF_CORNER_HEADER)
		{
			SetNumberID(_corNumberID);
			SetStringID(_corStringID);
		}
		else if (HeaderType() == ZAF_COLUMN_HEADER)
		{
			SetNumberID(_colNumberID);
			SetStringID(_colStringID);
		}
		else if (HeaderType() == ZAF_ROW_HEADER)
		{
			SetNumberID(_rowNumberID);
			SetStringID(_rowStringID);
		}
	}

	// Return the current attribute.
	return (headerType);
}

const ZafIChar *ZafTableHeader::SetHelpContext(const ZafIChar *)
{
	// help context is empty for this class.
	return (ZAF_NULLP(ZafIChar));
}

bool ZafTableHeader::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

bool ZafTableHeader::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafTableHeader::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafTableHeader::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

bool ZafTableHeader::SetNoncurrent(bool )
{
	// noncurrent is true for this class.
	noncurrent = true;
	return (noncurrent);
}

ZafRegionType ZafTableHeader::SetRegionType(ZafRegionType )
{
	// regionType is ZAF_AVAILABLE_REGION for this class.
	regionType = ZAF_AVAILABLE_REGION;
	return (regionType);
}

ZafSelectionType ZafTableHeader::SetSelectionType(ZafSelectionType )
{
	// selectionType is ZAF_SINGLE_SELECTION for this class.
	selectionType = ZAF_SINGLE_SELECTION;
	return (selectionType);
}

bool ZafTableHeader::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

bool ZafTableHeader::SetSupportObject(bool )
{
	// supportObject is true for this class.
	supportObject = true;
	return (supportObject);
}

bool ZafTableHeader::SetTemporary(bool )
{
	// temporary is false for this class.
	temporary = false;
	return (temporary);
}

ZafUserFunction ZafTableHeader::SetUserFunction(ZafUserFunction )
{
	// userFunction is null for this class.
	userFunction = ZAF_NULLF(ZafUserFunction);
	return (userFunction);
}

ZafWindowObject *ZafTableHeader::SetVirtualField(ZafWindowObject *tVirtualField)
{
	// Make sure the attribute has changed.
	if (!screenID)
	{
		if (virtualField)
			delete virtualField;
		virtualField = tVirtualField;
	}

	// Return the current virtual field.
	return (virtualField);
}

ZafWindowObject *ZafTableHeader::Subtract(ZafWindowObject *object)
{
	// Defer to the base class.
	return (ZafWindow::Subtract(object));
}

// ----- Persistent functions -----------------------------------------------

ZafTableHeader::ZafTableHeader(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	headerType((ZafTableHeaderType)-1),
	virtualField(ZAF_NULLP(ZafWindowObject))
{
	ZafTableHeader::ReadData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());

	if (First())
	{
		ZafWindowObject *object = First();
		// ZafWindow persistent constructor loaded the virtualField (if there 
		// was one) and added it as a child, so need to remove it and set it 
		// as the virtualField if this is a row header.
		if (headerType == ZAF_ROW_HEADER)
		{
			Subtract(object);
			SetVirtualField(object);
		}
		// Otherwise, the header objects were added by the ZafWindow 
		// persistent constructor using the ZafWindow::Add() function.
		// They need to be added using ZafTableHeader::Add().
		else
		{
			ZafList tempList;
			while (First())
			{
				object = First();
				Subtract(object);
				tempList.Add(object);
			}
			while (tempList.First())
			{
				object = DynamicPtrCast(tempList.First(), ZafWindowObject);
				tempList.Subtract(object);
				Add(object);
			}
		}
	}
}

ZafElement *ZafTableHeader::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafTableHeader(name, persist));
}

void ZafTableHeader::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafUInt16 flag1;
	ZafFile *file = persist.File();
	*file >> flag1;
	// flag1
	headerType = (ZafTableHeaderType)((int)flag1 & 0x000F);
}

void ZafTableHeader::Write(ZafObjectPersistence &persist)
{
	// Bypass ZafWindow calls to keep the children from being saved.
	ZafWindowObject::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafWindow::WriteData(persist);

	// Write children if not a row header.  Row header uses virtualField.
	if (headerType != ZAF_ROW_HEADER)
		ZafWindow::WriteChildren(persist);
	ZafTableHeader::WriteData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafTableHeader::WriteData(ZafObjectPersistence &persist)
{
	if (virtualField && !virtualField->StringID())
		virtualField->SetStringID(_virtualFieldName);

	// Write the header data.
	ZafUInt16 flag1 = 0;
	// flag1
	flag1 |= (ZafUInt16)(headerType & 0x000F);
	ZafFile *file = persist.File();
	*file << flag1;

	// Write the virtualField.
	if (virtualField)
	{
		virtualField->Write(persist.PushLevel(virtualField->ClassName(), virtualField->ClassID(), ZAF_PERSIST_DIRECTORY));
		persist.PopLevel();
	}
}


