//	Zinc Application Framework - Z_TABLE.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_table.hpp>
#include <z_scrll2.hpp>
#define ZAF_TABLE_INFO
#include "gbl_def.cpp"

static ZafStringID _virtualRecordName = ZAF_ITEXT("virtualRecord");

// ----- ZafTable -----------------------------------------------------------

ZafTable::ZafTable(int left, int top, int width, int height) :
	ZafWindow(left, top, width, height),
	virtualRecord(ZAF_NULLP(ZafTableRecord)),
	currentOffset(-1), maxOffset(-1), grid(false)
{
	SetBordered(true);
	SetOSDraw(false);
}

ZafTable::ZafTable(const ZafTable &copy) :
	ZafWindow(copy),
	virtualRecord(ZAF_NULLP(ZafTableRecord)),
	currentOffset(-1), maxOffset(copy.maxOffset), grid(copy.grid)
{
	// ZafWindow copy constructor will have duplicated the visible records in
	// the table, so we need to get rid of those and start over.
	ZafWindowObject *object;
	while (First())
	{
		object = First();
		ZafWindow::Subtract(object);
		delete object;
	}
	SetVirtualRecord(DynamicPtrCast(copy.virtualRecord->Duplicate(), ZafTableRecord));
}

ZafTable::~ZafTable(void)
{
	// Delete the virtual record.
	if (virtualRecord)
		delete virtualRecord;
}

ZafWindowObject *ZafTable::Add(ZafWindowObject *object, ZafWindowObject *position)
{
	// Check for a support object.
	if (object->SupportObject())
		return (ZafWindow::Add(object, position));

	// Check for a valid table record.
	ZafTableRecord *record = DynamicPtrCast(object, ZafTableRecord);
	if (!record)
		return (ZAF_NULLP(ZafWindowObject));

	// Create a row field for the new record.
	ZafTableHeader *header = RowTableHeader();
	ZafWindowObject *rowField = ZAF_NULLP(ZafWindowObject);
	if (header && header->VirtualField())
		rowField = header->VirtualField()->Duplicate();

	// Read the record data if the table is already on the screen.
	if (screenID)
	{
		// Seek to the proper record position.
		ZafOffset newOffset = 0;
		ZafTableRecord *positionRecord = DynamicPtrCast(position, ZafTableRecord);
		if (positionRecord && positionRecord->Offset() != -1)
			newOffset = positionRecord->Offset();
		else if (Last() && Last()->Offset() != -1)
			newOffset = Last()->Offset() + 1;
		SeekOptimized(newOffset);

		// Read the record.
		if (record->Offset() != currentOffset)
		{
			ReadRecord(record, rowField);
			record->SetOffset(currentOffset);
		}
	}

	// Add the table record and row field.
	record->SetSystemObject(false); 		// Prevent object registration.
	ZafWindow::Add(record, position);
	if (header)
	{
		// Set the row field's region.
		if (screenID)
		{
			rowField->ConvertCoordinates(ZAF_PIXEL);
			ZafRegionStruct region = rowField->Region();
			int height = region.bottom - region.top;
			region.top = record->zafRegion.top;
			region.bottom = record->zafRegion.top + height;
			rowField->SetRegion(region);
		}

		// Add the row field.
		ZafWindowObject *headerPosition = header->Get(Index(record));
		header->Add(rowField, headerPosition);
	}

	if (screenID)
	{
		// Size all subsequent table records.
		for (record = record->Next(); record; record = record->Next())
		{
			record->Event(S_COMPUTE_SIZE);
			record->OSSize();

			// Update the row header.
			if (rowField)
			{
				rowField = rowField->Next();
				ZafRegionStruct region = rowField->Region();
				int height = region.bottom - region.top;
				region.top = record->zafRegion.top;
				region.bottom = record->zafRegion.top + height;
				rowField->SetRegion(region);
				rowField->OSSize();
			}
		}
	}

	// Return the object pointer.
	return (object);
}

ZafWindowObject *ZafTable::Duplicate(void)
{
	return (new ZafTable(*this));
}

ZafEventType ZafTable::Event(const ZafEventStruct &event)
{
	// Check for OS logical events.
	ZafEventType ccode = LogicalEvent(event);

	// Check for zinc events.
	ZafTableRecord *object;
	switch (ccode)
	{
	// ----- Position & Size messages -----
	case N_SIZE:
		ZafWindow::Event(event);

		// Apply geometry to the children.
		{
		ZafTableRecord *record = First();
		ZafOffset offset = record ? record->Offset() : 0;
		int line = record ? record->zafRegion.bottom : 0;
		int height = 0;
		int bottom = clientRegion.bottom - clientRegion.top;
		while (line + height <= bottom)
		{
			// Get the next table record.
			if (record)
				;
			else if (virtualRecord && offset <= maxOffset)
			{
				record = DynamicPtrCast(virtualRecord->Duplicate(), ZafTableRecord);
				Add(record);
			}
			else
				break;

			// Update the bottom line and get the next record.
			line = record->zafRegion.bottom;
			height = record->zafRegion.bottom - record->zafRegion.top;
			record = record->Next();
			offset++;
		}

		// Delete any remaining records that don't fit on the screen.
		if (record && record != First())
		{
			// Start with the last record and work backward.  Working forward
			// would cause later records to be shifted up and redrawn right
			// before they get deleted.

			ZafTableRecord *deleteRecord = Last();
			ZafTableHeader *header = RowTableHeader();
			ZafWindowObject *rowField = header ? header->Get(Index(deleteRecord)) : 0;

			// Delete until the record prior to 'record'. 'record' gets deleted.
			ZafWindowObject *keepRecord = record->Previous();
			while (deleteRecord && deleteRecord != keepRecord)
			{
				ZafTableRecord *previousRecord = deleteRecord->Previous();

				// Write changed records.
				if (deleteRecord->Changed())
				{
					SeekOptimized(deleteRecord->Offset());
					WriteRecord(deleteRecord, rowField);
					deleteRecord->SetChanged(false);
				}

				// Save current record info in virtualRecord.
				if (deleteRecord == Current())
				{
					virtualRecord->SetOffset(Current()->Offset());
					ZafWindowObject *deleteCurrent = deleteRecord->Current();
					virtualRecord->SetCurrent(deleteCurrent ? virtualRecord->Get(deleteCurrent->ListIndex()) : deleteCurrent);
					current = 0;
				}

				Subtract(deleteRecord);
				delete deleteRecord;
				if (rowField)
				{
					ZafWindowObject *previousRowField = rowField->Previous();
					header->Subtract(rowField);
					delete rowField;
					rowField = previousRowField;
				}
				deleteRecord = previousRecord;
			}
		}

		// Adjust the scroll information.
		ScrollEvent(S_VSCROLL_COMPUTE);
		}
		break;

	// ----- Selection messages -----
#if !defined(ZAF_MSDOS)
	case L_BEGIN_SELECT:
		// Try to find an overlapping item.
		for (object = First(); object; object = object->Next())
			if (object->zafRegion.Overlap(event.position))
			{
				// Check for the current object.
				ZafTableRecord *record = Current();
				ZafWindowObject *curObject = record ? record->Current() : ZAF_NULLP(ZafWindowObject);
				if (curObject && curObject->SystemObject())
					curObject->SetSystemObject(false);

				// Make sure the record is active.
				object->SetFocus(true);
				object->Event(event);
				break;
			}
		break;
#endif

	// ----- Movement messages -----
	case L_FIRST:
	case L_LAST:
	case L_UP:
	case L_DOWN:
	case L_LEFT:
	case L_RIGHT:
	case L_PREVIOUS:
	case L_NEXT:
	case L_PGDN:
	case L_PGUP:
		ccode = MoveEvent(ccode);
		break;

	case S_REDISPLAY_DATA:
		for (object = First(); object; object = object->Next())
			object->Event(event);
		if (RowTableHeader())
			RowTableHeader()->Event(event);
		break;

	// ----- Miscellaneous messages -----
	case S_HOT_KEY:
		ccode = S_UNKNOWN;
		break;

	default:
		// Shift current record back into view if the user typed a key.
		if (!current && event.InputType() == E_KEY && First())
			ScrollEvent(S_VSCROLL_CHECK);
		ccode = ZafWindow::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafLogicalEvent ZafTable::LogicalEvent(const ZafEventStruct &event)
{
	ZafLogicalEvent ccode = MapEvent(defaultEventMap, event);
	return ((ccode == L_NONE) ? ZafWindow::LogicalEvent(event) : ccode);
}

const ZafPaletteStruct *ZafTable::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafWindow::MapClassPalette(type, state));
}

ZafEventType ZafTable::MoveEvent(const ZafEventStruct &event)
{
	// If no objects, return.
	if (!First())
		return (S_ERROR);

	// Make sure current record in view.
	if (!current)
		ScrollEvent(S_VSCROLL_CHECK);

	// Determine the new selection.
	ZafEventType ccode = event.type;
	ZafTableRecord *oldRecord = Current();
	ZafWindowObject *oldFocus = oldRecord->Current();
	ZafOffset oldOffset = oldRecord->Offset();
	ZafTableRecord *newRecord = oldRecord;
	ZafWindowObject *newFocus = oldFocus;
	ZafOffset newOffset = oldOffset;
	switch (ccode)
	{
	case L_UP:
		if (newOffset > 0)
			newOffset--;
		else
			return (S_ERROR);
		break;

	case L_DOWN:
		if (newOffset < maxOffset)
			newOffset++;
		else
			return (S_ERROR);
		break;

	case L_LEFT:
		if (newFocus && newFocus->Previous())
			newFocus = newFocus->Previous();
		else
			return (S_ERROR);
		break;

	case L_RIGHT:
		if (newFocus && newFocus->Next())
			newFocus = newFocus->Next();
		else
			return (S_ERROR);
		break;

	case L_PGUP:
		if (Current() != First())
			newOffset = First()->Offset();
		else if (newOffset > 0)
		{
			ZafOffset count = Count() - 1;
			newOffset = (newOffset > count) ? newOffset - count : 0;
		}
		else
			return (S_ERROR);
		break;

	case L_PGDN:
		if (Current() != Last())
			newOffset = Last()->Offset();
		else if (newOffset < maxOffset)
		{
			ZafOffset count = Count() - 1;
			newOffset = (newOffset + count < maxOffset) ? newOffset + count : maxOffset;
		}
		else
			return (S_ERROR);
		break;

	case L_FIRST:
		if (newOffset > 0)
			newOffset = 0;
		else
			return (S_ERROR);
		break;

	case L_LAST:
		if (newOffset < maxOffset)
			newOffset = maxOffset;
		else
			return (S_ERROR);
		break;

	case L_PREVIOUS:
		if (newFocus && newFocus->Previous())
			newFocus = newFocus->Previous();
		else if (newOffset > 0)
		{
			newOffset--;
			newFocus = newRecord->Last();
		}
		else
			return (S_ERROR);
		break;

	case L_NEXT:
		if (newFocus && newFocus->Next())
			newFocus = newFocus->Next();
		else if (newOffset < maxOffset)
		{
			newOffset++;
			newFocus = newRecord->First();
		}
		else
			return (S_ERROR);
		break;

	default:
		return (S_ERROR);
	}

	// Shift the records, if appropriate.  Assign a new current record as well.
	if (newOffset < First()->Offset())
		ShiftRecords(newOffset - First()->Offset(), newOffset, newFocus);
	else if (newOffset > Last()->Offset())
		ShiftRecords(newOffset - Last()->Offset(), newOffset, newFocus);
	else if (newOffset != oldOffset || newFocus != oldFocus)
	{
		newRecord = Get((int)(newOffset - First()->Offset()));

		// Change the focus object.
		if (virtualRecord->First())
		{
			newFocus = newRecord->Get(newFocus->ListIndex());
			oldFocus->SetSystemObject(false);
			newFocus->SetSystemObject(true);
			newFocus->SetFocus(true);
		}
		else
			newRecord->SetFocus(true);
	}

	// Check the scroll information.
	ScrollEvent(S_VSCROLL_COMPUTE);

	// Grab equivalent events to give the appearance of speed!
	ZafEventStruct tEvent;
	ZafEventType tCode = event.type;
	do
	{
		if (eventManager->Get(tEvent, Q_NO_BLOCK) != 0)
			break;
		else if (tEvent.InputType() != E_KEY || LogicalEvent(tEvent) != event.type)
			tCode = zafWindowManager->Event(tEvent);
	} while (tCode != S_EXIT && tCode != S_NO_OBJECT);

	// Return the control code.
	return (ccode);
}

ZafEventType ZafTable::ScrollEvent(const ZafEventStruct &event)
{
	// Process the scrolling request.  Scrolling at the table level is
	// based on the number of table records.
	ZafEventType ccode = event.type;
	switch (ccode)
	{
	case S_VSCROLL_CHECK:
		// Shift current record into view.
		if (!current && First())
		{
			ZafOffset cOffset = virtualRecord->Offset();
			if (cOffset < First()->Offset())
				ShiftRecords(cOffset - First()->Offset());
			else if (cOffset > Last()->Offset())
				ShiftRecords(cOffset - Last()->Offset());
			Event(S_VSCROLL_COMPUTE);
		}
		break;

	case S_VSCROLL:
	case N_VSCROLL:
		// Shift the records according to the delta.
		ShiftRecords(event.scroll.delta);

		// Adjust the scroll values.
		if (ccode == S_VSCROLL)
			Event(S_VSCROLL_COMPUTE);
		break;

	case S_VSCROLL_COMPUTE:
		{
		// Check for a scroll-bar.
		ZafScrollBar *scrollBar = VerticalScrollBar();
		if (!scrollBar)
			return (S_ERROR);

		// Reset the scroll values.
		ZafScrollStruct vScroll;
		vScroll.delta = 1;
		vScroll.showing = Count();
		vScroll.minimum = 0;
		vScroll.maximum = maxOffset - vScroll.showing + 1;
		vScroll.current = First() ? First()->Offset() : 0;

		// Adjust the vertical scroll-bar values.
		scrollBar->ScrollData()->SetScroll(vScroll);
		}
		break;

	default:
		// No other messages apply to the table.
		return (S_ERROR);
	}

	// Return the control code.
	return (ccode);
}

bool ZafTable::ShiftRecords(long delta, ZafOffset newOffset, ZafWindowObject *newFocus)
{
	// Check for a valid delta.
	if (!delta || !First())
		return (false);

	ZafTableRecord *oldRecord = Current();
	ZafWindowObject *oldFocus = oldRecord ? oldRecord->Current() : 0;
	ZafOffset oldOffset = oldRecord ? oldRecord->Offset() : 0;

	int i;
	int count = Count();
	ZafTableHeader *header = RowTableHeader();
	ZafTableRecord *record;
	ZafWindowObject *rowField;

	// Keep track of the current record's offset in virtualRecord in case 
	// the current record is getting scrolled off the viewable table.
	if (current)
		virtualRecord->SetOffset(Current()->Offset());

	// Virtualize the current field to give a smooth scroll.
	if (oldFocus)
	{
		oldFocus->SetSystemObject(false);

		// Keep track of the current record's current field in case
		// the current record is getting scrolled off the viewable table.
		if (current)
			virtualRecord->SetCurrent(virtualRecord->Get(oldFocus->ListIndex()));
	}

	// Check the record count.
	if (delta > 0 && delta < count)
	{
		// If current record is getting scrolled off the viewable table, set
		// current to 0.  current will be reassigned, if appropriate, at the
		// end of the function.
		if (current && oldOffset - delta < First()->Offset())
		{
			Current()->SetFocus(false);
			current = 0;
		}
		// current is getting scrolled back into view.
		else if (!current && virtualRecord->Offset() - delta <= Last()->Offset())
			newOffset = virtualRecord->Offset();
		// current was, and is remaining, in view.
		else if (newOffset < 0)
			newOffset = oldOffset;

		// Write changed records.
		rowField = header ? header->First() : 0;
		for (i = 0, record = First(); i < delta && record; i++, record = record->Next())
		{
			if (record->Changed())
			{
				SeekOptimized(record->Offset());
				WriteRecord(record, rowField);
				record->SetChanged(false);
			}
			if (rowField)
				rowField = rowField->Next();
		}

		// Shift the objects up in the table.
		for ( ; delta > 0; delta--)
		{
			// Get the record and row field.
			record = First();
			rowField = header ? header->First() : 0;

			// Temporarily remove the object from the list.
			ZafList::Subtract(record);
			SeekOptimized(Last() ? Last()->Offset() + 1 : record->Offset() + 1);
			ZafList::Add(record);
			if (rowField)
			{
				header->ZafList::Subtract(rowField);
				header->ZafList::Add(rowField);
			}

			// Read the new record contents.
			bool updateRecord = record->AutomaticUpdate();
			bool updateRowField = rowField ? rowField->AutomaticUpdate() : false;
			record->SetAutomaticUpdate(false);
			if (rowField)
				rowField->SetAutomaticUpdate(false);
			ReadRecord(record, rowField);
			if (rowField)
				rowField->SetAutomaticUpdate(updateRowField);
			record->SetAutomaticUpdate(updateRecord);
			record->SetOffset(currentOffset);
		}
	}
	else if (delta < 0 && delta > -count)
	{
		// If current record is getting scrolled off the viewable table, set
		// current to 0.  current will be reassigned, if appropriate, at the
		// end of the function.
		if (current && oldOffset - delta > Last()->Offset())
		{
			Current()->SetFocus(false);
			current = 0;
		}
		// current is getting scrolled back into view.
		else if (!current && virtualRecord->Offset() - delta >= First()->Offset())
			newOffset = virtualRecord->Offset();
		// current was, and is remaining, in view.
		else if (newOffset < 0)
			newOffset = oldOffset;

		// Write changed records.
		rowField = header ? header->Last() : 0;
		for (i = 0, record = Last(); i > delta && record; i--, record = record->Previous())
		{
			if (record->Changed())
			{
				SeekOptimized(record->Offset());
				WriteRecord(record, rowField);
				record->SetChanged(false);
			}
			if (rowField)
				rowField = rowField->Previous();
		}

		// Shift the objects down in the table.
		for ( ; delta < 0; delta++)
		{
			// Get the record and row field.
			record = Last();
			rowField = header ? header->Last() : 0;

			// Temporarily remove the object from the list.
			ZafList::Subtract(record);
			SeekOptimized(First() ? First()->Offset() - 1 : record->Offset() - 1);
			ZafList::Add(record, First());
			if (rowField)
			{
				header->ZafList::Subtract(rowField);
				header->ZafList::Add(rowField, header->First());
			}

			// Read the new record contents.
			bool updateRecord = record->AutomaticUpdate();
			bool updateRowField = rowField ? rowField->AutomaticUpdate() : false;
			record->SetAutomaticUpdate(false);
			if (rowField)
				rowField->SetAutomaticUpdate(false);
			ReadRecord(record, rowField);
			if (rowField)
				rowField->SetAutomaticUpdate(updateRowField);
			record->SetAutomaticUpdate(updateRecord);
			record->SetOffset(currentOffset);
		}
	}
	else
	{
		// Current record is getting scrolled off the viewable table, so set
		// current to 0.  current will be reassigned, if appropriate, at the
		// end of the function.
		if (current)
		{
			Current()->SetFocus(false);
			current = 0;
		}
		// current is getting scrolled back into view.
		else if (!current &&
			((virtualRecord->Offset() - delta <= Last()->Offset()) ||
			(virtualRecord->Offset() - delta >= First()->Offset())))
			newOffset = virtualRecord->Offset();

		// Write changed records.
		rowField = header ? header->First() : 0;
		for (record = First(); record; record = record->Next())
		{
			if (record->Changed())
			{
				SeekOptimized(record->Offset());
				WriteRecord(record, rowField);
				record->SetChanged(false);
			}
			if (rowField)
				rowField = rowField->Next();
		}

		// Read all new records.
		ZafOffset offset = First()->Offset() + delta;
		rowField = header ? header->First() : 0;
		for (record = First(); record; record = record->Next())
		{
			// Read the new record.
			SeekOptimized(offset++);
			bool updateRecord = record->AutomaticUpdate();
			bool updateRowField = rowField ? rowField->AutomaticUpdate() : false;
			record->SetAutomaticUpdate(false);
			if (rowField)
				rowField->SetAutomaticUpdate(false);
			ReadRecord(record, rowField);
			record->SetAutomaticUpdate(updateRecord);
			if (rowField)
				rowField->SetAutomaticUpdate(updateRowField);
			record->SetOffset(currentOffset);
			if (rowField)
				rowField = rowField->Next();
		}
	}

	// Reposition the records.
	rowField = header ? header->First() : 0;
	for (record = First(); record; record = record->Next())
	{
		// Update the record.
		int height = record->zafRegion.bottom - record->zafRegion.top;
		record->zafRegion.top = record->Previous() ? record->Previous()->zafRegion.bottom + 1 : 0;
		record->zafRegion.bottom = record->zafRegion.top + height;

		// Update the row header.
		if (rowField)
		{
			ZafRegionStruct region = rowField->Region();
			int height = region.bottom - region.top;
			region.top = record->zafRegion.top;
			region.bottom = record->zafRegion.top + height;
			rowField->SetRegion(region);
			rowField = rowField->Next();
		}
	}

	// Refresh the window.
	RedisplayData();

	// Find the new current record if newOffset is >= 0.  Otherwise, not 
	// changing current.
	if (newOffset >= First()->Offset() && newOffset <= Last()->Offset())
	{
		ZafTableRecord *newRecord = Get((int)(newOffset - First()->Offset()));
		if (current)
		{
			if (!newFocus)
				newFocus = oldFocus;
			newFocus = (newRecord && newFocus) ? newRecord->Get(newFocus->ListIndex()) : ZAF_NULLP(ZafWindowObject);
			if (newFocus)
			{
				newFocus->SetSystemObject(true);
				newFocus->SetFocus(true);
			}
			else
				newRecord->SetFocus(true);
		}
		else
		{
			if (oldFocus)
			{
				newFocus = newRecord->Get(virtualRecord->Current()->ListIndex());
				newFocus->SetSystemObject(true);
				newFocus->SetFocus(true);
			}
			else
				newRecord->SetFocus(true);
		}
	}

	// Success.
	return (true);
}

bool ZafTable::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

ZafOffset ZafTable::SetCurrentOffset(ZafOffset tCurrentOffset)
{
	currentOffset = tCurrentOffset;
	return (currentOffset);
}

bool ZafTable::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

bool ZafTable::SetGrid(bool tGrid)
{
	// Check the attribute.
	if (grid != tGrid && !screenID)
		grid = tGrid;
	return (grid);
}

bool ZafTable::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

ZafOffset ZafTable::SetMaxOffset(ZafOffset tMaxOffset)
{
	maxOffset = tMaxOffset;
	return (maxOffset);
}

bool ZafTable::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafTable::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafTable::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

bool ZafTable::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

bool ZafTable::SetTemporary(bool )
{
	// temporary is false for this class.
	temporary = false;
	return (temporary);
}

ZafTableRecord *ZafTable::SetVirtualRecord(ZafTableRecord *newVirtualRecord)
{
	// Make sure this is a valid request.
	if (!screenID)
	{
		if (virtualRecord)
			delete virtualRecord;
		virtualRecord = newVirtualRecord;
	}

	// Return a pointer to the virtual record.
	return (virtualRecord);
}

ZafWindowObject *ZafTable::Subtract(ZafWindowObject *object)
{
	// Remove the record.
	ZafWindowObject *nextObject = ZafWindow::Subtract(object);

	// Shift the table.
	if (screenID)
	{
		ZafRegionStruct newRegion = object->zafRegion;
		for (ZafWindowObject *record = nextObject; record; record = record->Next())
		{
			ZafRegionStruct saveRegion = record->zafRegion;
			record->zafRegion = newRegion;
			newRegion = saveRegion;
			record->OSSize();
		}
	}

	// Return the next object.
	return (nextObject);
}

// ----- Data Members -------------------------------------------------------

ZafError ZafTable::DeleteRecord(void)
{
	return (ZAF_ERROR_INVALID);
}

ZafError ZafTable::InsertRecord(ZafTableRecord *, ZafWindowObject *)
{
	return (ZAF_ERROR_INVALID);
}

ZafError ZafTable::ReadRecord(ZafTableRecord *, ZafWindowObject *)
{
	return (ZAF_ERROR_NONE);
}

ZafError ZafTable::SeekNextRecord(void)
{
	if (currentOffset > maxOffset)
		return (ZAF_ERROR_INVALID);
	currentOffset++;
	return (ZAF_ERROR_NONE);
}

ZafError ZafTable::SeekOptimized(ZafOffset newOffset)
{
	// Optimize the seek.
	if (currentOffset == newOffset)
		; // no seek is necessary;
	else if (currentOffset + 1 == newOffset)
		SeekNextRecord();
	else if (currentOffset - 1 == newOffset)
		SeekPreviousRecord();
	else
		SeekRandomRecord(newOffset, ZAF_SEEK_START); // no hope!
	return (ZAF_ERROR_NONE);
}

ZafError ZafTable::SeekPreviousRecord(void)
{
	if (currentOffset == 0)
		return (ZAF_ERROR_INVALID);
	currentOffset--;
	return (ZAF_ERROR_NONE);
}

ZafError ZafTable::SeekRandomRecord(ZafOffset offset, ZafSeek location)
{
	// Reset the location.
	if (location == ZAF_SEEK_CURRENT)
		offset += currentOffset;
	else if (location == ZAF_SEEK_END)
		offset += maxOffset;

	// Check for out-of-bounds.
	if (offset < 0 || offset > maxOffset)
		return (ZAF_ERROR_INVALID);

	// Reset the current value.
	currentOffset = offset;
	return (ZAF_ERROR_NONE);
}

ZafOffset ZafTable::Tell(void)
{
	// Return the current position.
	return (currentOffset);
}

ZafError ZafTable::WriteRecord(ZafTableRecord *, ZafWindowObject *)
{
	return (ZAF_ERROR_INVALID);
}

// ----- Persistent functions -----------------------------------------------

ZafTable::ZafTable(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	virtualRecord(ZAF_NULLP(ZafTableRecord)),
	currentOffset(-1), maxOffset(-1), grid(false)
{
	// ZafWindow persistent constructor loaded the virtualRecord and added it
	// as a child, so need to remove it.
	while (First())
	{
		ZafWindowObject *object = First();
		ZafWindow::Subtract(object);
		delete object;
	}

	ZafTable::ReadData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafTable::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafTable(name, persist));
}

void ZafTable::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafIChar virtualRecordName[ZAF_MAXNAMELEN];
	ZafFile *file = persist.File();
	ZafInt32 value32;
	ZafUInt16 flag1;
	*file >> value32 >> flag1 >> virtualRecordName;
	maxOffset = (ZafOffset)value32;
	grid = (flag1 & 0x0001) ? true : false;

	// Read the virtualRecord.
	if (*virtualRecordName)
		SetVirtualRecord(new ZafTableRecord(virtualRecordName, persist));
}

void ZafTable::Write(ZafObjectPersistence &persist)
{
	// Bypass ZafWindow calls to keep the children from being saved.
	ZafWindowObject::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafWindow::WriteData(persist);
	ZafWindow::WriteSupportChildren(persist);
	ZafTable::WriteData(persist);

	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafTable::WriteData(ZafObjectPersistence &persist)
{
	if (virtualRecord && !virtualRecord->StringID())
	{
		virtualRecord->SetStringID(_virtualRecordName);
		virtualRecord->Event(S_INITIALIZE);
	}
	const ZafIChar *virtualRecordName = virtualRecord ? virtualRecord->StringID() : ZAF_NULLP(ZafIChar);

	ZafFile *file = persist.File();
	ZafInt32 value32 = (ZafInt32)maxOffset;
	ZafUInt16 flag1 = grid ? 0x0001 : 0;
	*file << value32 << flag1 << virtualRecordName;

	// Write the virtualRecord.
	if (virtualRecord)
		virtualRecord->Write(persist);
}

