//	Zinc Application Framework - Z_STR1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_str1.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>
#define ZAF_STRING_INFO
#include "gbl_def.cpp"

static ZafStringID _inputFormatDataName = ZAF_ITEXT("inputFormatData");
static ZafStringID _outputFormatDataName = ZAF_ITEXT("outputFormatData");
static ZafStringID _rangeDataName = ZAF_ITEXT("rangeData");
static ZafStringID _stringDataName = ZAF_ITEXT("stringData");

// ----- ZafString ----------------------------------------------------------

ZafString::ZafString(int left, int top, int width, const ZafIChar *text,
	int maxLength) :
	ZafWindowObject(left, top, width, 1),
	autoClear(true), invalid(false), unanswered(false), lowerCase(false),
	password(false), upperCase(false), variableName(false), viewOnly(false),
	hzJustify(ZAF_HZ_LEFT), inputFormatData(ZAF_NULLP(ZafStringData)),
	outputFormatData(ZAF_NULLP(ZafStringData)),
	rangeData(ZAF_NULLP(ZafStringData)), stringData(ZAF_NULLP(ZafStringData))
{
	// Initialize the string information.
	SetBordered(true);
	memberUserFunction = (MemberUserFunction)&ZafString::DefaultValidateFunction;

	// Initialize the string data.
	SetStringData(new ZafStringData(text, maxLength));

#if defined(ZAF_MSDOS) || defined(ZAF_CURSES)
	cursor = firstDisplayedChar = length = 0;
	beginMark = endMark = -1;
#endif
}

ZafString::ZafString(int left, int top, int width, ZafStringData *zStringData) :
	ZafWindowObject(left, top, width, 1),
	autoClear(true), invalid(false), unanswered(false), lowerCase(false),
	password(false), upperCase(false), variableName(false), viewOnly(false),
	hzJustify(ZAF_HZ_LEFT), inputFormatData(ZAF_NULLP(ZafStringData)),
	outputFormatData(ZAF_NULLP(ZafStringData)),
	rangeData(ZAF_NULLP(ZafStringData)), stringData(ZAF_NULLP(ZafStringData))
{
	// Initialize the string information.
	SetBordered(true);
	memberUserFunction = (MemberUserFunction)&ZafString::DefaultValidateFunction;

	// Initialize the string data.
	if (zStringData)
		SetStringData(zStringData);
	else
		SetStringData(new ZafStringData);

#if defined(ZAF_MSDOS) || defined(ZAF_CURSES)
	cursor = firstDisplayedChar = 0;
	beginMark = endMark = -1;
#endif
}

ZafString::ZafString(const ZafString &copy) :
	ZafWindowObject(copy),
	autoClear(copy.autoClear), invalid(copy.invalid),
	unanswered(copy.unanswered), lowerCase(copy.lowerCase),
	password(copy.password), upperCase(copy.upperCase),
	variableName(copy.variableName), viewOnly(copy.viewOnly),
	hzJustify(copy.hzJustify),
	inputFormatData(ZAF_NULLP(ZafStringData)),
	outputFormatData(ZAF_NULLP(ZafStringData)),
	rangeData(ZAF_NULLP(ZafStringData)),
	stringData(ZAF_NULLP(ZafStringData))
{
	// Initialize the string information.
	if (copy.stringData->Destroyable())
		SetStringData(new ZafStringData(*copy.stringData));
	else
		SetStringData(copy.stringData);

	if (copy.inputFormatData)
	{
		if (copy.inputFormatData->Destroyable())
			SetInputFormatData(new ZafStringData(*copy.inputFormatData));
		else
			SetInputFormatData(copy.inputFormatData);
	}

	if (copy.outputFormatData)
	{
		if (copy.outputFormatData->Destroyable())
			SetStringData(new ZafStringData(*copy.outputFormatData));
		else
			SetInputFormatData(copy.outputFormatData);
	}

	if (copy.rangeData)
	{
		if (copy.rangeData->Destroyable())
			SetStringData(new ZafStringData(*copy.rangeData));
		else
			SetRangeData(copy.rangeData);
	}

#if defined(ZAF_MSDOS) || defined(ZAF_CURSES)
	cursor = firstDisplayedChar = 0;
	beginMark = endMark = -1;
#endif
}

ZafString::~ZafString(void)
{
	// Remove the data notification.
	stringData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Restore the string information.
	if (stringData->Destroyable() && stringData->NotifyCount() == 0)
		delete stringData;
	if (inputFormatData && inputFormatData->Destroyable() && inputFormatData->NotifyCount() == 0)
		delete inputFormatData;
	if (outputFormatData && outputFormatData->Destroyable() && outputFormatData->NotifyCount() == 0)
		delete outputFormatData;
	if (rangeData && rangeData->Destroyable() && rangeData->NotifyCount() == 0)
		delete rangeData;
}

ZafError ZafString::Decrement(ZafData *)
{
	// This class has no decrement capabilities.
	return (ZAF_ERROR_INVALID_CLASS);
}

ZafEventType ZafString::DefaultValidateFunction(const ZafEventStruct &event, ZafEventType ccode)
{
	// Start with the number validation.
	ZafEventType errorCode = (ccode == L_SELECT || ccode == N_NON_CURRENT) ? Validate(true) : ZAF_ERROR_NONE;

	// Call the base user function.
	return (errorCode ? errorCode : DefaultUserFunction(event, ccode));
}

ZafWindowObject *ZafString::Duplicate(void)
{
	return (new ZafString(*this));
}

ZafError ZafString::Increment(ZafData *)
{
	// This class has no increment capabilities.
	return (ZAF_ERROR_INVALID_CLASS);
}

ZafLogicalEvent ZafString::LogicalEvent(const ZafEventStruct &event)
{
	ZafLogicalEvent ccode = MapEvent(defaultEventMap, event);
	return ((ccode == L_NONE) ? ZafWindowObject::LogicalEvent(event) : ccode);
}

const ZafPaletteStruct *ZafString::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafWindowObject::MapClassPalette(type, state));
}

const ZafIChar *ZafString::ParseRange(const ZafIChar *buffer, ZafIChar *minValue, ZafIChar *maxValue)
{
	// Parse the minimum value.
	int position = 0, offset = 0;
	while (buffer[offset] != '\0' && buffer[offset] != '/' &&
		buffer[offset] != '\n' &&
		(buffer[offset] != '.' || buffer[offset+1] != '.'))
		minValue[position++] = buffer[offset++];
	minValue[position] = '\0';

	// See if it is a standalone value.
	if (buffer[offset] == '/' || buffer[offset] == '\0' || buffer[offset] == '\n')
	{
		strcpy(maxValue, minValue);
		return (buffer[offset] ? &buffer[++offset] : ZAF_NULLP(ZafIChar));
	}

	// Increment the offset.
	while (buffer[offset] == '.')
		offset++;

	// Parse the maximum value.
	position = 0;
	while (buffer[offset] != '\0' && buffer[offset] != '/' && buffer[offset] != '\n')
		maxValue[position++] = buffer[offset++];
	maxValue[position] = '\0';

	// Return the offset position.
	return (buffer[offset] ? &buffer[++offset] : ZAF_NULLP(ZafIChar));
}

bool ZafString::SetAutoClear(bool tAutoClear)
{
	// Make sure the attribute has changed.
	if (autoClear != tAutoClear)
		autoClear = tAutoClear;

	// Return the current attribute.
	return autoClear;
}

ZafHzJustify ZafString::SetHzJustify(ZafHzJustify setHzJustify)
{
	// Make sure the request is valid.
	if (!screenID && hzJustify != setHzJustify)
		hzJustify = setHzJustify;

	// Return the current attribute.
	return (hzJustify);
}

ZafError ZafString::SetInputFormat(const ZafIChar *setInputFormat)
{
	// Make sure the request is valid.
	if (screenID)
		return (ZAF_ERROR_INVALID);

	// Reset the format data.
	if (inputFormatData)
		return (inputFormatData->SetText(setInputFormat));
	return (SetInputFormatData(new ZafStringData(setInputFormat)));
}

ZafError ZafString::SetOutputFormat(const ZafIChar *setOutputFormat)
{
	// Make sure the request is valid.
	if (screenID)
		return (ZAF_ERROR_INVALID);

	// Reset the format data.
	if (outputFormatData)
		return (outputFormatData->SetText(setOutputFormat));
	return (SetOutputFormatData(new ZafStringData(setOutputFormat)));
}

ZafError ZafString::SetInputFormatData(ZafStringData *tInputFormatData)
{
	// Make sure the request is valid.
	if (screenID)
		return (ZAF_ERROR_INVALID);

	// Reset the string data.
	if (inputFormatData && inputFormatData != tInputFormatData &&
		inputFormatData->Destroyable() && inputFormatData->NotifyCount() == 0)
		delete inputFormatData;
	inputFormatData = tInputFormatData;

	// Reset the string identification.
	if (inputFormatData && !inputFormatData->StringID())
		inputFormatData->SetStringID(_inputFormatDataName);
	return (ZAF_ERROR_NONE);
}

ZafError ZafString::SetOutputFormatData(ZafStringData *tOutputFormatData)
{
	// Make sure the request is valid.
	if (screenID)
		return (ZAF_ERROR_INVALID);

	// Reset the string data.
	if (outputFormatData && outputFormatData != tOutputFormatData &&
		outputFormatData->Destroyable() && outputFormatData->NotifyCount() == 0)
		delete outputFormatData;
	outputFormatData = tOutputFormatData;

	// Reset the string identification.
	if (outputFormatData && !outputFormatData->StringID())
		outputFormatData->SetStringID(_outputFormatDataName);
	return (ZAF_ERROR_NONE);
}

bool ZafString::SetInvalid(bool setInvalid)
{
	// Make sure the request is valid.
	if (invalid != setInvalid)
		invalid = setInvalid;
	return (invalid);
}

bool ZafString::SetLowerCase(bool setLowerCase)
{
	// Make sure the request is valid.
	if (lowerCase != setLowerCase && !screenID)
	{
		lowerCase = setLowerCase;
		if (lowerCase && upperCase)
			upperCase = false;
	}

	// Return the current attribute.
	return (lowerCase);
}

bool ZafString::SetPassword(bool setPassword)
{
	// Make sure the request is valid.
	if (password != setPassword && !screenID)
		password = setPassword;

	// Return the current attribute.
	return (password);
}

ZafError ZafString::SetRange(const ZafIChar *setRange)
{
	// Make sure the request is valid.
	if (screenID)
		return (ZAF_ERROR_INVALID);

	// Reset the range data.
	if (rangeData)
		return (rangeData->SetText(setRange));
	return (SetRangeData(new ZafStringData(setRange)));
}

ZafError ZafString::SetRangeData(ZafStringData *tRangeData)
{
	// Make sure the request is valid.
	if (screenID)
		return (ZAF_ERROR_INVALID);

	// Reset the range data.
	if (rangeData && rangeData != tRangeData &&
		rangeData->Destroyable() && rangeData->NotifyCount() == 0)
		delete rangeData;
	rangeData = tRangeData;

	// Add the data notification.
	if (rangeData && !rangeData->StringID())
		rangeData->SetStringID(_rangeDataName);
	return (ZAF_ERROR_NONE);
}

ZafError ZafString::SetStringData(ZafStringData *tStringData)
{
	// Remove the data notification.
	if (stringData)
		stringData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Reset the string data.
	if (stringData && stringData != tStringData &&
		stringData->Destroyable() && stringData->NotifyCount() == 0)
		delete stringData;
	stringData = tStringData ? tStringData : new ZafStringData;
	if (!stringData->StringID())
		stringData->SetStringID(_stringDataName);

	// Add the data notification.
	stringData->AddNotification(this, (ZafUpdateFunction)Update);
	return (OSSetText());
}

ZafError ZafString::SetText(const ZafIChar *text)
{
	// Drag-drop and other run-time text operations count as user input.
	if (screenID)
		unanswered = false;

	// Set the OS text.
	stringData->SetText(text);
	return (ZAF_ERROR_NONE);
}

bool ZafString::SetUnanswered(bool setUnanswered)
{
	// Make sure the request is valid.
	if (unanswered != setUnanswered)
	{
		// Re-compute the text before resetting the flag.
		if (setUnanswered)
			stringData->Clear();

		unanswered = setUnanswered;
	}

	// Return the current attribute.
	return (unanswered);
}

bool ZafString::SetUpperCase(bool setUpperCase)
{
	// Make sure the request is valid.
	if (upperCase != setUpperCase && !screenID)
	{
		upperCase = setUpperCase;
		if (upperCase && lowerCase)
			lowerCase = false;
	}

	// Return the current attribute.
	return (upperCase);
}

bool ZafString::SetVariableName(bool setVariableName)
{
	// Check for equivalent flag.
	if (variableName != setVariableName && !screenID)
		variableName = setVariableName;
	return (variableName);
}

bool ZafString::SetViewOnly(bool setViewOnly)
{
	// Check for equivalent flag.
	if (viewOnly != setViewOnly && !screenID)
		viewOnly = setViewOnly;

	// Return the current attribute.
	return (viewOnly);
}

const ZafIChar *ZafString::Text(void)
{
	// Get the os text.
	OSGetText();
	return (stringData->Text());
}

ZafError ZafString::Validate(bool )
{
	// The base class does not do any validation.
	return (ZAF_ERROR_NONE);
}

// ----- Persistent functions -----------------------------------------------

ZafString::ZafString(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindowObject(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	inputFormatData(ZAF_NULLP(ZafStringData)),
	outputFormatData(ZAF_NULLP(ZafStringData)),
	rangeData(ZAF_NULLP(ZafStringData)),
	stringData(ZAF_NULLP(ZafStringData))
{
	// Initialize the string information.
	memberUserFunction = (MemberUserFunction)&ZafString::DefaultValidateFunction;

	// Read the data.
	ZafString::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());

#if defined(ZAF_MSDOS) || defined(ZAF_CURSES)
	cursor = firstDisplayedChar = 0;
	beginMark = endMark = -1;
#endif
}

ZafElement *ZafString::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafString(name, persist));
}

void ZafString::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafUInt16 flag1, flag2;
	ZafIChar stringName[ZAF_MAXNAMELEN], inputFormatName[ZAF_MAXNAMELEN];
	ZafIChar outputFormatName[ZAF_MAXNAMELEN], rangeName[ZAF_MAXNAMELEN];
	ZafFile *file = persist.File();
	*file >> flag1 >> flag2 >> stringName >> inputFormatName >> outputFormatName >> rangeName;
	autoClear = (flag1 & 0x0001) ? true : false;
	invalid = (flag1 & 0x0002) ? true : false;
	unanswered = (flag1 & 0x0004) ? true : false;
	lowerCase = (flag1 & 0x0008) ? true : false;
	password = (flag1 & 0x0010) ? true : false;
	upperCase = (flag1 & 0x0020) ? true : false;
	variableName = (flag1 & 0x0040) ? true : false;
	viewOnly = (flag1 & 0x0080) ? true : false;
	hzJustify = (ZafHzJustify)((int)flag2 & 0x000F);

	// Read the string data.
	if (*stringName)
		SetStringData(new ZafStringData(stringName, persist));
	if (*inputFormatName)
		inputFormatData = new ZafStringData(inputFormatName, persist);
	if (*outputFormatName)
		outputFormatData = new ZafStringData(outputFormatName, persist);
	if (*rangeName)
		rangeData = new ZafStringData(rangeName, persist);
}

void ZafString::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafWindowObject::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafString::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafString::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	ZafUInt16 flag1 = 0, flag2 = 0;
	flag1 |= autoClear ? 0x0001 : 0;
	flag1 |= invalid ? 0x0002 : 0;
	flag1 |= unanswered ? 0x0004 : 0;
	flag1 |= lowerCase ? 0x0008 : 0;
	flag1 |= password ? 0x0010 : 0;
	flag1 |= upperCase ? 0x0020 : 0;
	flag1 |= variableName ? 0x0040 : 0;
	flag1 |= viewOnly ? 0x0080 : 0;
	flag2 |= (ZafUInt16)(hzJustify & 0x000F);
	const ZafIChar *stringName = stringData ? stringData->StringID() : ZAF_NULLP(ZafIChar);
	const ZafIChar *inputFormatName = inputFormatData ? inputFormatData->StringID() : ZAF_NULLP(ZafIChar);
	const ZafIChar *outputFormatName = outputFormatData ? outputFormatData->StringID() : ZAF_NULLP(ZafIChar);
	const ZafIChar *rangeName = rangeData ? rangeData->StringID() : ZAF_NULLP(ZafIChar);
	ZafFile *file = persist.File();
	*file << flag1 << flag2 << stringName << inputFormatName << outputFormatName << rangeName;

	// Write the string data.
	if (stringData)
		stringData->Write(persist);
	if (inputFormatData)
		inputFormatData->Write(persist);
	if (outputFormatData)
		outputFormatData->Write(persist);
	if (rangeData)
		rangeData->Write(persist);
}

