//	Zinc Application Framework - Z_SCRLL.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_scrll1.hpp>
#define ZAF_SCROLL_DATA_INFO
#include "gbl_def.cpp"

// ----- ZafScrollData ------------------------------------------------------

ZafScrollData::ZafScrollData(long tMinimum, long tMaximum, long tCurrent,
	long tDelta, long tShowing) :
	ZafData()
{
	minimum = tMinimum;
	maximum = tMaximum;
	current = tCurrent;
	if (current < minimum)
		current = minimum;
	else if (current > maximum)
		current = maximum;
	delta = tDelta;
	showing = tShowing;
}

ZafScrollData::ZafScrollData(ZafScrollStruct &scroll) :
	ZafData()
{
	minimum = scroll.minimum;
	maximum = scroll.maximum;
	current = scroll.current;
	if (current < minimum)
		current = minimum;
	else if (current > maximum)
		current = maximum;
	showing = scroll.showing;
	delta = scroll.delta;
}

ZafScrollData::ZafScrollData(const ZafScrollData &copy) :
	ZafData(copy)
{
	minimum = copy.minimum;
	maximum = copy.maximum;
	current = copy.current;
	if (current < minimum)
		current = minimum;
	else if (current > maximum)
		current = maximum;
	showing = copy.showing;
	delta = copy.delta;
}

ZafScrollData::~ZafScrollData(void)
{
}

void ZafScrollData::Clear(void)
{
	// Reset the values.
	if (minimum != 0 || maximum != 0 || current != 0 ||
		showing != 1 || delta != 1)
	{
		PushLevel();
		minimum = maximum = current = 0;
		showing = delta = 1;
		PopLevel();
	}
}

ZafData *ZafScrollData::Duplicate(void)
{
	return (new ZafScrollData(*this));
}

ZafError ZafScrollData::SetScroll(long tMinimum, long tMaximum,
	long tCurrent, long tDelta, long tShowing)
{
	// Check the current argument.
	if (tCurrent < tMinimum)
		tCurrent = tMinimum;
	else if (tCurrent > tMaximum)
		tCurrent = tMaximum;

	// Reset the values.
	if (minimum != tMinimum || maximum != tMaximum || current != tCurrent ||
		delta != tDelta || showing != tShowing)
	{
		PushLevel();
		minimum = tMinimum;
		maximum = tMaximum;
		current = tCurrent;
		delta = tDelta;
		showing = tShowing;
		PopLevel();
	}

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafError ZafScrollData::SetScroll(const ZafScrollStruct &scroll)
{
	// Check the current argument.
	long tCurrent = scroll.current;
	if (tCurrent < scroll.minimum)
		tCurrent = scroll.minimum;
	else if (tCurrent > scroll.maximum)
		tCurrent = scroll.maximum;

	// Reset the data.
	if (minimum != scroll.minimum || maximum != scroll.maximum ||
		current != tCurrent || delta != scroll.delta ||
		showing != scroll.showing)
	{
		PushLevel();
		minimum = scroll.minimum;
		maximum = scroll.maximum;
		current = tCurrent;
		showing = scroll.showing;
		delta = scroll.delta;
		PopLevel();
	}

	// Return success.
	return (ZAF_ERROR_NONE);
}

// ----- Persistent functions -----------------------------------------------

ZafScrollData::ZafScrollData(const ZafIChar *name, ZafDataPersistence &persist) :
	ZafData(name, persist.PushLevel(className, classID, ZAF_PERSIST_FILE))
{
	// Read the data.
	ZafScrollData::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafScrollData::Read(const ZafIChar *name, ZafDataPersistence &persist)
{
	return (new ZafScrollData(name, persist));
}

void ZafScrollData::ReadData(ZafDataPersistence &persist)
{
	// Read the data.
	ZafInt32 tCurrent, tMin, tMax, tShow, tDelta;
	ZafFile *file = persist.File();
	*file >> tCurrent >> tMin >> tMax >> tShow >> tDelta;

	current = tCurrent;
	minimum = tMin;
	maximum = tMax;
	showing = tShow;
	delta = tDelta;
}

void ZafScrollData::Write(ZafDataPersistence &persist)
{
	// Write the object.
	ZafData::Write(persist.PushLevel(className, classID, ZAF_PERSIST_FILE));
	ZafScrollData::WriteData(persist);	
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafScrollData::WriteData(ZafDataPersistence &persist)
{
	// Write the data.
	ZafInt32 tCurrent, tMin, tMax, tShow, tDelta;

	tCurrent = current;
	tMin = minimum;
	tMax = maximum;
	tShow = showing;
	tDelta = delta;

	ZafFile *file = persist.File();
	*file << tCurrent << tMin << tMax << tShow << tDelta;
}

