//	Zinc Application Framework - Z_REAL1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_stdio.hpp>
#include <z_string.hpp>
#include <z_real1.hpp>
#include <z_utils.hpp>
#include <z_error.hpp>
#include <z_app.hpp>
#define ZAF_REAL_INFO
#include "lang_def.cpp"
#include "gbl_def.cpp"

static ZafIChar numSeparator[3] = { '.','.',0 };
static ZafIChar rangeSeparator[3] = { ',',' ',0 };
static ZafStringID _realDataName = ZAF_ITEXT("realData");

// ----- ZafReal -----------------------------------------------------------

ZafReal::ZafReal(int left, int top, int width, ZafRealData *zRealData) :
	ZafString(left, top, width, new ZafStringData),
	realData(ZAF_NULLP(ZafRealData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the real data.
	SetRealData(zRealData ? zRealData : new ZafRealData);
}

ZafReal::ZafReal(int left, int top, int width, double value) :
	ZafString(left, top, width, new ZafStringData),
	realData(ZAF_NULLP(ZafRealData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the real data.
	SetRealData(new ZafRealData(value));
}

ZafReal::ZafReal(const ZafReal &copy) :
	ZafString(copy), realData(ZAF_NULLP(ZafRealData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Copy the real data.
	if (copy.RealData()->Destroyable())
		SetRealData(new ZafRealData(*copy.RealData()));
	else
		SetRealData(copy.RealData());
}

ZafReal::~ZafReal(void)
{
	// Remove the data notification.
	realData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Check the language and local information.
	LanguageFree();

	// Restore the real information.
	if (realData->Destroyable() && realData->NotifyCount() == 0)
		delete realData;
}

ZafError ZafReal::Decrement(ZafData *data)
{
	// Check for a valid object.
	ZafRealData *real = DynamicPtrCast(data, ZafRealData);
	if (!real)
		return (ZAF_ERROR_INVALID_CLASS);

	// Decrement the value.
	ZafRealData _real(*realData);
	OSGetReal();
	*realData -= *real; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafWindowObject *ZafReal::Duplicate(void)
{
	return (new ZafReal(*this));
}

ZafEventType ZafReal::Event(const ZafEventStruct &event)
{
	// Process OS specific messages.
	ZafEventType ccode = event.type;
	if (event.InputType() == E_KEY)
		ccode = LogicalEvent(event);

	// Check for zinc events.
	switch (ccode)
	{
	// ----- Create messages -----
	case S_INITIALIZE:
		// Check for an unanswered string.
		if (!Unanswered())
			OSSetReal();
		ccode = ZafString::Event(event);
		break;

	// ----- Action messages -----
	case L_SELECT:
	case N_NON_CURRENT:
		{
		// Keep a temporary number in case of error.
		ZafRealData oldValue(*realData);
		OSGetReal();

		// Call the associated user function.
		ccode = ZafString::Event(event);

		// Check for errors.
		if (ccode != 0 && ccode != ZAF_ERROR_LEAVE_INVALID)
			realData->SetReal(oldValue);

		// Set the formatted real text into the field.
		OSSetReal();
		}
		break;

	case S_COPY_DATA:
		{
		ZafWindowObject *object = event.windowObject;
		ZafReal *realNum = DynamicPtrCast(object, ZafReal);
		if (realNum)
			realData->SetReal(*realNum->RealData());
		}
		break;

	case S_SET_DATA:
		if (event.windowObject)
		{
			ZafWindowObject *object = event.windowObject;
			ZafReal *realNum = DynamicPtrCast(object, ZafReal);
			if (realNum)
				SetRealData(realNum->RealData());
		}
		else
			SetRealData(new ZafRealData(*realData));
		break;

	// ----- Default or unknown messages -----
	default:
		ccode = ZafString::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafError ZafReal::Increment(ZafData *data)
{
	// Check for a valid object.
	ZafRealData *real = DynamicPtrCast(data, ZafRealData);
	if (!real)
		return (ZAF_ERROR_INVALID_CLASS);

	// Increment the value.
	ZafRealData _real(*realData);
	OSGetReal();
	*realData += *real; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

void ZafReal::LanguageAllocate(void)
{
	// Check for initialization.
	if (errorStrings)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!errorStrings && zafDataManager)
		errorStrings = DynamicPtrCast(zafDataManager->AllocateData(className, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

	// Default to code initialization.
	if (!errorStrings)
		errorStrings = new ZafLanguageData(defaultErrorStrings);
}

void ZafReal::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && errorStrings && errorStrings->Destroyable() &&
		errorStrings->NotifyCount() == 0)
	{
		delete errorStrings;
		errorStrings = ZAF_NULLP(ZafLanguageData);
	}
}

ZafError ZafReal::OSGetReal(void)
{
	// Disable notification.
	ZafUpdateType update = realData->Update(this);
	realData->SetUpdate(this, ZAF_UPDATE_NONE);

	// Set the real based on the string data.
	OSGetText();
	error = realData->SetReal(StringData()->Text(), InputFormatText());

	// Restore notification.
	realData->SetUpdate(this, update);
	return (error);
}

ZafError ZafReal::OSSetReal(void)
{
	// Set the string based on the real data.
	ZafIChar text[ZAF_STRING_DATA_LENGTH];
	realData->FormattedText(text, ZAF_STRING_DATA_LENGTH, OutputFormatText());
	stringData->SetText(text);

	// Return the current status.
	return (ZAF_ERROR_NONE);
}

ZafError ZafReal::SetRealData(ZafRealData *tRealData)
{
	// Remove the data notification.
	if (realData)
		realData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Reset the real data.
	if (realData && realData != tRealData &&
		realData->Destroyable() && realData->NotifyCount() == 0)
		delete realData;
	realData = tRealData ? tRealData : new ZafRealData;
	if (!realData->StringID())
		realData->SetStringID(_realDataName);

	// Add the data notification.
	realData->AddNotification(this, (ZafUpdateFunction)Update);
	return (OSSetReal());
}

bool ZafReal::SetLowerCase(bool )
{
	// lowerCase is set to false by this class.
	lowerCase = false;
	return (lowerCase);
}

bool ZafReal::SetPassword(bool )
{
	// password is set to false by this class.
	password = false;
	return (password);
}

ZafError ZafReal::SetStringData(ZafStringData *)
{
	// String data cannot be set for this class.
	return (ZAF_ERROR_INVALID);
}

bool ZafReal::SetUpperCase(bool )
{
	// upperCase is set to false by this class.
	upperCase = false;
	return (upperCase);
}

ZafError ZafReal::Validate(bool processError)
{
	// Check for an absolute number error. Don't set the error member to ZAF_ERROR_NONE.
	ZafIChar *stringNumber = stringData->DynamicText();
	ZafRealData currentNumber;
	ZafError tError = currentNumber.SetReal(stringNumber, InputFormatText());

	// Check for a range error.
	if (RangeText() && tError == ZAF_ERROR_NONE)
		tError = ZAF_ERROR_OUT_OF_RANGE;
	int numRanges = 0;
	ZafRealData low, high;
	ZafIChar minNumber[ZAF_STRING_DATA_LENGTH];
	ZafIChar maxNumber[ZAF_STRING_DATA_LENGTH];
	ZafIChar rBuffer[1024];				// Localized range string for error messages.
	rBuffer[0] = '\0';
	for (const ZafIChar *tRange = RangeText(); tRange && tError == ZAF_ERROR_OUT_OF_RANGE; numRanges++)
	{
		tRange = ParseRange(tRange, minNumber, maxNumber);
		low.SetReal(minNumber, OutputFormatText());
		high.SetReal(maxNumber, OutputFormatText());
		if ((!minNumber[0] || currentNumber >= low) && (!maxNumber[0] || currentNumber <= high))
			tError = ZAF_ERROR_NONE;
		else if (processError)
		{
			if (rBuffer[0])
				strcat(rBuffer, rangeSeparator);
			if (minNumber[0])
				low.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
			strcat(rBuffer, numSeparator);
			if (maxNumber[0])
				high.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Process the error code.
	SetInvalid(false);
	if (tError == ZAF_ERROR_NONE)
	{
		// Set up the new number.
		currentNumber.FormattedText(stringNumber, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		SetText(stringNumber);
		delete []stringNumber;
		return (ZAF_ERROR_NONE);
	}
	else
	{
		// Keep the error code.
		error = tError;
		if (!zafErrorSystem)
		{
			// Restore the original number.
			currentNumber.FormattedText(stringNumber, ZAF_STRING_DATA_LENGTH, OutputFormatText());
			SetText(stringNumber);
			delete []stringNumber;
			return (error);
		}
		else if (!processError)
		{
			delete []stringNumber;
			return (error);
		}
	}

	// Check for open-ended range errors.
	if (error == ZAF_ERROR_OUT_OF_RANGE && numRanges == 1)
	{
		if (minNumber[0] && !maxNumber[0])
		{
			error = ZAF_ERROR_LESS_THAN_RANGE;
			low.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
		else if (!minNumber[0] && maxNumber[0])
		{
			error = ZAF_ERROR_GREATER_THAN_RANGE;
			high.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Generate the error message and wait for a response.
	ZafIChar *title = ZafLanguageData::blankString;
	ZafIChar *errorString = errorStrings->GetMessage((ZafNumberID)error, true);
	if (errorString &&
		zafErrorSystem->ReportError(windowManager, title, ZAF_DIALOG_OK | ZAF_DIALOG_CANCEL, errorString, stringNumber, rBuffer) == S_DLG_OK)
		error = ZAF_ERROR_LEAVE_INVALID; // Keep the new value.
	delete []stringNumber;
	return (error);
}

bool ZafReal::SetVariableName(bool )
{
	// variableName is false for this class.
	variableName = false;
	return (variableName);
}

// ----- Persistent functions -----------------------------------------------

ZafReal::ZafReal(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafString(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	realData(ZAF_NULLP(ZafRealData))
{
	// Check the language and local information.
	LanguageAllocate();

	ZafReal::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafReal::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafReal(name, persist));
}

void ZafReal::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafIChar realName[ZAF_MAXNAMELEN];
	ZafFile *file = persist.File();
	*file >> realName;
	SetRealData(realName[0] ? new ZafRealData(realName, persist) : new ZafRealData);
}

void ZafReal::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafString::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafReal::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafReal::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	const ZafIChar *realName = realData ? realData->StringID() : ZAF_NULLP(ZafIChar);
	ZafFile *file = persist.File();
	*file << realName;
	if (realData)
		realData->Write(persist);
}

