//	Zinc Application Framework - Z_PRST.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_data.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>
#if defined(ZAF_PERSIST_ALL)
#	include <z_datall.hpp>
#endif
#define ZAF_DATA_PERSISTENCE_INFO
#include "gbl_def.cpp"

static ZafIChar _objectDataName[] = ZAF_ITEXT("objectData");

// ----- ZafDataPersistence -------------------------------------------------

ZafDataPersistence::ZafDataPersistence(ZafFileSystem *tFileSystem, ZafFile *tFile) :
	current(0), error(ZAF_ERROR_NONE), fileSystem(tFileSystem)
{
	dataConstructor = defaultDataConstructor;
	for (int i = 0; i < ZAF_MAXFILES; i++)
	{
		entry[i].level = 0;
		entry[i].type = ZAF_PERSIST_ROOT_DIRECTORY;
		entry[i].className = 0;
		entry[i].classID = 0;
		entry[i].file = ZAF_NULLP(ZafFile);
		entry[i].restorePath[0] = '\0';
	}
	entry[current].file = tFile;
}

ZafDataPersistence::ZafDataPersistence(DataConstructor *tDataConstructor) :
	current(0), error(ZAF_ERROR_NONE), fileSystem(ZAF_NULLP(ZafFileSystem))
{
	dataConstructor = tDataConstructor;
	for (int i = 0; i < ZAF_MAXFILES; i++)
	{
		entry[i].level = 0;
		entry[i].type = ZAF_PERSIST_ROOT_DIRECTORY;
		entry[i].className = 0;
		entry[i].classID = 0;
		entry[i].file = ZAF_NULLP(ZafFile);
		entry[i].restorePath[0] = '\0';
	}
}

ZafDataPersistence::ZafDataPersistence(const ZafDataPersistence &copy) :
	current(0), error(ZAF_ERROR_NONE), fileSystem(ZAF_NULLP(ZafFileSystem))
{
	dataConstructor = copy.dataConstructor;
}

ZafDataPersistence::~ZafDataPersistence(void)
{
	// Clear the tables.
	ClearDataConstructors();
}

bool ZafDataPersistence::AddDataConstructor(ZafClassID classID, ZafClassName className, ZafDataConstructor constructor)
{
	// Check for an existing function.
	int offset = 0;
	if (dataConstructor && dataConstructor != defaultDataConstructor)
	{
		// Try to find a matching class.
		for ( ; dataConstructor[offset].classID != ID_END; offset++)
			if (classID == dataConstructor[offset].classID)
			{
				// Check for invalid function.
				if (constructor && constructor != dataConstructor[offset].constructor)
					return (false);

				// Increment the use count.
				dataConstructor[offset].useCount++;
				return (true);
			}

		// Reallocate the array.
		if (((offset + 1) % ZAF_DYNAMIC_ARRAY_INCREMENT) == 0)
		{
			DataConstructor *tempArray = dataConstructor;
			dataConstructor = new DataConstructor[offset + ZAF_DYNAMIC_ARRAY_INCREMENT + 1];
			memcpy(dataConstructor, tempArray, offset * sizeof(DataConstructor));
			delete []tempArray;
		}
	}
	else
		dataConstructor = new DataConstructor[ZAF_DYNAMIC_ARRAY_INCREMENT];

	// Set the read function.
	dataConstructor[offset].useCount = 0;
	dataConstructor[offset].classID = classID;
	dataConstructor[offset].className = className;
	dataConstructor[offset].constructor = constructor;

	// Set the end-of-array.
	offset++;
	dataConstructor[offset].useCount = 0;
	dataConstructor[offset].classID = ID_END;
	dataConstructor[offset].className = '\0';
	dataConstructor[offset].constructor = ZAF_NULLF(ZafDataConstructor);

	// Return the read function.
	return (true);
}

void ZafDataPersistence::AllocateFile(const ZafIChar *name)
{
	// Reset the file system.
	if (!fileSystem)
	{
		entry[current+1].file = entry[current].file;
		entry[current].restorePath[0] = '\0';
		return;
	}

	if (!name || !name[0])
		name = entry[current].className;
	if (entry[current].type == ZAF_PERSIST_ROOT_DIRECTORY ||
		entry[current].type == ZAF_PERSIST_ROOT_FILE)
	{
		// Move to the root directory.
		fileSystem->GetCWD(entry[current].restorePath, ZAF_MAXPATHLEN);
		fileSystem->ChDir(ZafFileSystem::rootDirectoryName);
		if (fileSystem->ChDir(entry[current].className))
		{
			fileSystem->MkDir(entry[current].className);
			fileSystem->ChDir(entry[current].className);
		}

		// Allocate a named directory.
		if (entry[current].type == ZAF_PERSIST_ROOT_DIRECTORY &&
			fileSystem->ChDir(name))
		{
			fileSystem->MkDir(name, ZAF_NULLP(ZafStringIDChar), entry[current].classID);
			fileSystem->ChDir(name);
		}

		// Move to the proper sub-directory.
		// printf("push root %s\n", name);
		entry[current].file = fileSystem->Open(_objectDataName, ZAF_FILE_OPENCREATE | ZAF_FILE_READWRITE);
	}
	else if (entry[current].type == ZAF_PERSIST_DIRECTORY)
	{
		// Copy the restore path.
		strcpy(entry[current].restorePath, ZafFileSystem::parentDirectoryName);

		// Allocate a named directory.
		if (fileSystem->ChDir(name))
		{
			fileSystem->MkDir(name, ZAF_NULLP(ZafStringIDChar), entry[current].classID);
			fileSystem->ChDir(name);
		}

		// Move to the proper sub-directory.
		// printf("push directory %s\n", name);
		entry[current].file = fileSystem->Open(_objectDataName, ZAF_FILE_OPENCREATE | ZAF_FILE_READWRITE);
	}
	else
	{
		// Set an empty restore path.
		entry[current].restorePath[0] = '\0';

		// Move to the proper sub-directory.
		// printf("push file %s\n", name);
		entry[current].file = fileSystem->Open(name, ZAF_FILE_OPENCREATE | ZAF_FILE_READWRITE,
			ZAF_NULLP(ZafStringIDChar), entry[current].classID);
	}

	// Keep the error code.
	if (entry[current].file->Error() != ZAF_ERROR_NONE)
		SetError(entry[current].file->Error());
}

void ZafDataPersistence::ClearDataConstructors(void)
{
	// Check for a valid persist table.
	if (dataConstructor && dataConstructor != defaultDataConstructor)
	{
		delete []dataConstructor;
		dataConstructor = ZAF_NULLP(DataConstructor);
	}
}

ZafDataConstructor ZafDataPersistence::GetDataConstructor(ZafClassID classID, ZafClassName className)
{
	// Check for a valid table.
	if (!dataConstructor)
		return (0);

	// Try to find a matching entry.
	for (int i = 0; dataConstructor[i].classID != ID_END; i++)
		if ((classID && classID == dataConstructor[i].classID) ||
			(className == dataConstructor[i].className) ||
			(className && dataConstructor[i].className && !streq(className, dataConstructor[i].className)))
			return (dataConstructor[i].constructor);

	// Read function not found.
	return (0);
}

ZafDataPersistence &ZafDataPersistence::PopLevel(void)
{
	// Remove the file.
	if (--entry[current].level > 0)
		return (*this);

	// Decrement the level count.
	if (fileSystem)
	{
		// Delete the file and keep the error code.
		if (entry[current].file)
		{
			if (entry[current].file->Error() != ZAF_ERROR_NONE)
				SetError(entry[current].file->Error());
			fileSystem->Close(entry[current].file);
			entry[current].file = ZAF_NULLP(ZafFile);
		}

		// Restore the directory.
		// printf("pop %s\n", entry[current].restorePath);
		if (entry[current].restorePath[0])
			fileSystem->ChDir(entry[current].restorePath);
	}
	if (current)
		current--;
	return (*this);
}

ZafDataPersistence &ZafDataPersistence::PushLevel(ZafClassName className, ZafClassID classID, ZafPersistEntryType type)
{
	// Check for a new entry.
	if (entry[current].file)
	{
		current++;
		entry[current].level = 1;
		entry[current].file = ZAF_NULLP(ZafFile);
		entry[current].restorePath[0] = '\0';
	}
	else
		entry[current].level++;

	// Check for entry update.
	if (entry[current].level == 1)
	{
		entry[current].type = type;
		entry[current].className = className;
		entry[current].classID = classID;
	}

	return (*this);
}

bool ZafDataPersistence::SubtractDataConstructor(ZafClassID classID, ZafClassName className)
{
	// Check for a valid table.
	if (!dataConstructor)
		return (0);

	// Try to find a matching entry.
	for (int i = 0; dataConstructor[i].classID != ID_END; i++)
		if ((classID && classID == dataConstructor[i].classID) ||
			(className && !streq(className, dataConstructor[i].className)))
		{
			// Check for an empty array.
			if (i == 0 && dataConstructor[i+1].classID == 0)
			{
				delete []dataConstructor;
				dataConstructor = ZAF_NULLP(DataConstructor);
			}
			else
			{
				// Remove the data from the notification array.
				for (int j = i; dataConstructor[j].classID; j++)
					dataConstructor[j] = dataConstructor[j+1];
			}

			// Entry found.
			return (true);
		}

	// Entry not found.
	return (false);
}


