//	Zinc Application Framework - Z_INT1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_stdio.hpp>
#include <z_string.hpp>
#include <z_int1.hpp>
#include <z_utils.hpp>
#include <z_error.hpp>
#include <z_app.hpp>
#define ZAF_INTEGER_INFO
#include "lang_def.cpp"
#include "gbl_def.cpp"

static ZafIChar numSeparator[3] = { '.','.',0 };
static ZafIChar rangeSeparator[3] = { ',',' ',0 };
static ZafStringID _integerDataName = ZAF_ITEXT("integerData");

// ----- ZafInteger ---------------------------------------------------------

ZafInteger::ZafInteger(int left, int top, int width, ZafIntegerData *zIntegerData) :
	ZafString(left, top, width, new ZafStringData),
	integerData(ZAF_NULLP(ZafIntegerData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the integer data.
	SetIntegerData(zIntegerData ? zIntegerData : new ZafIntegerData);
}

ZafInteger::ZafInteger(int left, int top, int width, long value) :
	ZafString(left, top, width, new ZafStringData),
	integerData(ZAF_NULLP(ZafIntegerData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the integer data.
	SetIntegerData(new ZafIntegerData(value));
}

ZafInteger::ZafInteger(const ZafInteger &copy) :
	ZafString(copy), integerData(ZAF_NULLP(ZafIntegerData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Copy the integer data.
	if (copy.IntegerData()->Destroyable())
		SetIntegerData(new ZafIntegerData(*copy.IntegerData()));
	else
		SetIntegerData(copy.IntegerData());
}

ZafInteger::~ZafInteger(void)
{
	// Remove the data notification.
	integerData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Check the language and local information.
	LanguageFree();

	// Restore the integer information.
	if (integerData->Destroyable() && integerData->NotifyCount() == 0)
		delete integerData;
}

ZafError ZafInteger::Decrement(ZafData *data)
{
	// Check for a valid object.
	ZafIntegerData *integer = DynamicPtrCast(data, ZafIntegerData);
	if (!integer)
		return (ZAF_ERROR_INVALID_CLASS);

	// Decrement the value.
	ZafIntegerData _integer(*integerData);
	OSGetInteger();
	*integerData -= *integer; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafWindowObject *ZafInteger::Duplicate(void)
{
	return (new ZafInteger(*this));
}

ZafEventType ZafInteger::Event(const ZafEventStruct &event)
{
	// Process OS specific messages.
	ZafEventType ccode = event.type;
	if (event.InputType() == E_KEY)
		ccode = LogicalEvent(event);

	// Check for zinc events.
	switch (ccode)
	{
	// ----- Create messages -----
	case S_INITIALIZE:
		// Check for an unanswered field.
		if (!Unanswered())
			OSSetInteger();
		ccode = ZafString::Event(event);
		break;

	// ----- Action messages -----
	case L_SELECT:
	case N_NON_CURRENT:
		{
		// Keep a temporary number in case of error.
		ZafIntegerData oldValue(*integerData);
		OSGetInteger();

		// Call the associated user function.
		ccode = ZafString::Event(event);

		// Check for errors.
		if (ccode != 0 && ccode != ZAF_ERROR_LEAVE_INVALID)
			integerData->SetInteger(oldValue);

		// Set the formatted integer text into the field.
		OSSetInteger();
		}
		break;

	case S_COPY_DATA:
		{
		ZafWindowObject *object = event.windowObject;
		ZafInteger *integer = DynamicPtrCast(object, ZafInteger);
		if (integer)
			integerData->SetInteger(*integer->IntegerData());
		}
		break;

	case S_SET_DATA:
		if (event.windowObject)
		{
			ZafWindowObject *object = event.windowObject;
			ZafInteger *integer = DynamicPtrCast(object, ZafInteger);
			if (integer)
				SetIntegerData(integer->IntegerData());
		}
		else
			SetIntegerData(new ZafIntegerData(*integerData));
		break;

	// ----- Default or unknown messages -----
	default:
		ccode = ZafString::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafError ZafInteger::Increment(ZafData *data)
{
	// Check for a valid object.
	ZafIntegerData *integer = DynamicPtrCast(data, ZafIntegerData);
	if (!integer)
		return (ZAF_ERROR_INVALID_CLASS);

	// Increment the value.
	ZafIntegerData _integer(*integerData);
	OSGetInteger();
	*integerData += *integer; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

void ZafInteger::LanguageAllocate(void)
{
	// Check for initialization.
	if (errorStrings)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!errorStrings && zafDataManager)
		errorStrings = DynamicPtrCast(zafDataManager->AllocateData(className, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

	// Default to code initialization.
	if (!errorStrings)
		errorStrings = new ZafLanguageData(defaultErrorStrings);
}

void ZafInteger::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && errorStrings && errorStrings->Destroyable() &&
		errorStrings->NotifyCount() == 0)
	{
		delete errorStrings;
		errorStrings = ZAF_NULLP(ZafLanguageData);
	}
}

ZafError ZafInteger::OSGetInteger(void)
{
	// Disable notification.
	ZafUpdateType update = integerData->Update(this);
	integerData->SetUpdate(this, ZAF_UPDATE_NONE);

	// Set the integer based on the string data.
	OSGetText();
	error = integerData->SetInteger(StringData()->Text(), InputFormatText());

	// Restore notification.
	integerData->SetUpdate(this, update);
	return (error);
}

ZafError ZafInteger::OSSetInteger(void)
{
	// Set the string based on the integer data.
	ZafIChar text[ZAF_STRING_DATA_LENGTH];
	integerData->FormattedText(text, ZAF_STRING_DATA_LENGTH, OutputFormatText());
	stringData->SetText(text);

	// Return the current status.
	return (ZAF_ERROR_NONE);
}

ZafError ZafInteger::SetIntegerData(ZafIntegerData *tIntegerData)
{
	// Remove the data notification.
	if (integerData)
		integerData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Reset the integer data.
	if (integerData && integerData != tIntegerData &&
		integerData->Destroyable() && integerData->NotifyCount() == 0)
		delete integerData;
	integerData = tIntegerData ? tIntegerData : new ZafIntegerData;
	if (!integerData->StringID())
		integerData->SetStringID(_integerDataName);

	// Add the data notification.
	integerData->AddNotification(this, (ZafUpdateFunction)Update);
	return (OSSetInteger());
}

bool ZafInteger::SetLowerCase(bool )
{
	// lowerCase is set to false by this class.
	lowerCase = false;
	return (lowerCase);
}

bool ZafInteger::SetPassword(bool )
{
	// password is set to false by this class.
	password = false;
	return (password);
}

ZafError ZafInteger::SetStringData(ZafStringData *)
{
	// String data cannot be set for this class.
	return (ZAF_ERROR_INVALID);
}

bool ZafInteger::SetUpperCase(bool )
{
	// upperCase is set to false by this class.
	upperCase = false;
	return (upperCase);
}

ZafError ZafInteger::Validate(bool processError)
{
	// Check for an absolute number error. Don't set the error member to ZAF_ERROR_NONE.
	ZafIChar *stringNumber = stringData->DynamicText();
	ZafIntegerData currentNumber;
	ZafError tError = currentNumber.SetInteger(stringNumber, InputFormatText());

	// Check for a range error.
	if (RangeText() && tError == ZAF_ERROR_NONE)
		tError = ZAF_ERROR_OUT_OF_RANGE;
	int numRanges = 0;
	ZafIntegerData low, high;
	ZafIChar minNumber[ZAF_STRING_DATA_LENGTH];
	ZafIChar maxNumber[ZAF_STRING_DATA_LENGTH];
	ZafIChar rBuffer[1024]; // Localized range string for error messages.
	rBuffer[0] = '\0';
	for (const ZafIChar *tRange = RangeText(); tRange && tError == ZAF_ERROR_OUT_OF_RANGE; numRanges++)
	{
		tRange = ParseRange(tRange, minNumber, maxNumber);
		low.SetInteger(minNumber, OutputFormatText());
		high.SetInteger(maxNumber, OutputFormatText());
		if ((!minNumber[0] || currentNumber >= low) && (!maxNumber[0] || currentNumber <= high))
			tError = ZAF_ERROR_NONE;
		else if (processError)
		{
			if (rBuffer[0])
				strcat(rBuffer, rangeSeparator);
			if (minNumber[0])
				low.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
			strcat(rBuffer, numSeparator);
			if (maxNumber[0])
				high.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Process the error code.
	if (tError == ZAF_ERROR_NONE)
	{
		delete []stringNumber;
		return (ZAF_ERROR_NONE);
	}
	else
	{
		// Keep the error code.
		error = tError;
		if (!zafErrorSystem || !processError)
		{
			delete []stringNumber;
			return (error);
		}
		else if (error == ZAF_ERROR_OUT_OF_RANGE && numRanges == 1)
		{
			// Check for open-ended range errors.
			if (minNumber[0] && !maxNumber[0])
			{
				error = ZAF_ERROR_LESS_THAN_RANGE;
				low.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
			}
			else if (!minNumber[0] && maxNumber[0])
			{
				error = ZAF_ERROR_GREATER_THAN_RANGE;
				high.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
			}
		}
	}

	// Generate the error message and wait for a response.
	ZafIChar *title = ZafLanguageData::blankString;
	ZafIChar *errorString = errorStrings->GetMessage((ZafNumberID)error, true);
	if (errorString &&
		zafErrorSystem->ReportError(windowManager, title, ZAF_DIALOG_OK | ZAF_DIALOG_CANCEL, errorString, stringNumber, rBuffer) == S_DLG_OK)
		error = ZAF_ERROR_LEAVE_INVALID; // Keep the new value.
	delete []stringNumber;
	return (error);
}

bool ZafInteger::SetVariableName(bool )
{
	// variableName is false for this class.
	variableName = false;
	return (variableName);
}

// ----- Persistent functions -----------------------------------------------

ZafInteger::ZafInteger(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafString(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	integerData(ZAF_NULLP(ZafIntegerData))
{
	// Check the language and local information.
	LanguageAllocate();

	ZafInteger::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafInteger::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafInteger(name, persist));
}

void ZafInteger::ReadData(ZafObjectPersistence &persist)
{
	ZafFile *file = persist.File();

	// Read the integer data.
	ZafIChar integerName[ZAF_MAXNAMELEN];
	*file >> integerName;
	SetIntegerData(integerName[0] ? new ZafIntegerData(integerName, persist) : new ZafIntegerData);
}

void ZafInteger::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafString::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafInteger::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafInteger::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	const ZafIChar *integerName = integerData ? integerData->StringID() : ZAF_NULLP(ZafIChar);
	ZafFile *file = persist.File();
	*file << integerName;
	if (integerData)
		integerData->Write(persist);
}

