//	Zinc Application Framework - Z_IMAGE.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_image.hpp>
#include <z_utils.hpp>
#define ZAF_IMAGE_INFO
#include "gbl_def.cpp"

// ----- ZafImage -----------------------------------------------------------

ZafImage::ZafImage(int left, int top, int tWidth, int tHeight,
	const ZafIChar *_pathName, int _pathID) :
	ZafWindowObject(left, top, tWidth, tHeight),
	pathName(ZAF_NULLP(ZafIChar)), pathID(_pathID), 
	autoSize(false), scaled(false), tiled(false), wallpaper(false)
{
	// Set the default attributes.
	SetOSDraw(false);
	SetNoncurrent(true);

	// Set the image information.
	if (_pathName)
		pathName = strdup(_pathName);
	width = height = 0;
#if defined (ZAF_MSWINDOWS)
	imageBuffer.hBitmap = 0;
	imageBuffer.hPalette = 0;
#else
	imageBuffer = 0;
#endif

#if defined(ZAF_MACINTOSH)
	staticPICT = false;
#endif

	// Level 1 - Load the image from the application resources.
	if (!LoadImageFromApplication())

	// Level 2 - Load the image from the native file.
		if (!LoadImageFromFile())
			error = ZAF_ERROR_CONSTRUCTOR;
}

ZafImage::ZafImage(const ZafImage &copy) :
	ZafWindowObject(copy),
	pathName(ZAF_NULLP(ZafIChar)), pathID(copy.pathID), 
	autoSize(copy.autoSize), scaled(copy.scaled), tiled(copy.tiled),
	wallpaper(copy.wallpaper)
{
	if (copy.pathName)
		pathName = strdup(copy.pathName);
#if defined (ZAF_MSWINDOWS)
	imageBuffer.hBitmap = 0;
	imageBuffer.hPalette = 0;
#else
	imageBuffer = 0;
#endif

#if defined(ZAF_MACINTOSH)
	staticPICT = false;
#endif

	// Level 1 - Load the image from the application resources.
	if (!LoadImageFromApplication())

	// Level 2 - Load the image from the native file.
		if (!LoadImageFromFile())
			error = ZAF_ERROR_CONSTRUCTOR;
}

ZafImage::~ZafImage(void)
{
	// Destroy the image and pathName.
	DestroyImageHandle();
	if (pathName)
		delete []pathName;
}

ZafWindowObject *ZafImage::Duplicate(void)
{
	return (new ZafImage(*this));
}

bool ZafImage::SetAcceptDrop(bool )
{
	// acceptDrop is false for this class.
	acceptDrop = false;
	return (acceptDrop);
}

bool ZafImage::SetAutoSize(bool setAutoSize)
{
	// Make sure the attribute has changed.
	if (autoSize != setAutoSize && !screenID)
		autoSize = setAutoSize;

	// Return the current attribute.
	return (autoSize);
}

bool ZafImage::SetCopyDraggable(bool )
{
	// copyDraggable is false for this class.
	copyDraggable = false;
	return (copyDraggable);
}

bool ZafImage::SetFocus(bool )
{
	// focus is false for this class.
	focus = false;
	return (focus);
}

ZafLogicalFont ZafImage::SetFont(ZafLogicalFont )
{
	// This class does not permit font changes.
	return (ZAF_FNT_NULL);
}

const ZafIChar *ZafImage::SetHelpContext(const ZafIChar *)
{
	// help context is empty for this class.
	return (ZAF_NULLP(ZafIChar));
}

const ZafIChar *ZafImage::SetHelpObjectTip(const ZafIChar *)
{
	// helpObjectTip cannot be set for this class.
	return (ZAF_NULLP(ZafIChar));
}

bool ZafImage::SetLinkDraggable(bool )
{
	// linkDraggable is false for this class.
	linkDraggable = false;
	return (linkDraggable);
}

bool ZafImage::SetMoveDraggable(bool )
{
	// moveDraggable is false for this class.
	moveDraggable = false;
	return (moveDraggable);
}

bool ZafImage::SetNoncurrent(bool )
{
	// noncurrent is true for this class.
	noncurrent = true;
	return (noncurrent);
}

bool ZafImage::SetScaled(bool setScaled)
{
	// Make sure the attribute has changed.
	if (scaled != setScaled && !screenID)
		scaled = setScaled;

	// Return the current attribute.
	return (scaled);
}

ZafLogicalColor ZafImage::SetTextColor(ZafLogicalColor , ZafLogicalColor )
{
	// This class does not permit text color changes.
	return (ZAF_CLR_NULL);
}

bool ZafImage::SetTiled(bool setTiled)
{
	// Make sure the attribute has changed.
	if (tiled != setTiled && !screenID)
		tiled = setTiled;

	// Return the current attribute.
	return (tiled);
}

bool ZafImage::SetWallpaper(bool setWallpaper)
{
	// Make sure the attribute has changed.
	if (wallpaper != setWallpaper && !parent)
	{
		wallpaper = setWallpaper;
		supportObject = wallpaper ? true : false;
	}

	// Return the current attribute.
	return (wallpaper);
}

// ----- Persistent functions -----------------------------------------------

ZafImage::ZafImage(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindowObject(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	pathName(ZAF_NULLP(ZafIChar)), pathID(-1), width(0), height(0),
	autoSize(false), scaled(false), tiled(false), wallpaper(false)
{
	// Set the image information.
	SetOSDraw(false);
#if defined (ZAF_MSWINDOWS)
	imageBuffer.hBitmap = 0;
	imageBuffer.hPalette = 0;
#else
	imageBuffer = 0;
#endif

#if defined(ZAF_MACINTOSH)
	staticPICT = false;
#endif

	// Read the data.
	ZafImage::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafImage::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafImage(name, persist));
}

void ZafImage::ReadData(ZafObjectPersistence &persist)
{
	ZafFile *file = persist.File();

	// Load the image information.
	ZafUInt16 flag1;
	ZafIChar _pathName[ZAF_MAXPATHLEN];
	*file >> _pathName >> pathID >> flag1;
	autoSize = (flag1 & 0x0001) ? true : false;
	scaled = (flag1 & 0x0002) ? true : false;
	tiled = (flag1 & 0x0004) ? true : false;
	wallpaper = (flag1 & 0x0008) ? true : false;

	if (_pathName[0])
		pathName = strdup(_pathName);

	// Level 1 - Load the image from the application resources.
	if (!LoadImageFromApplication())

	// Level 2 - Load the image from the native file.
		if (!LoadImageFromFile())
			error = ZAF_ERROR_CONSTRUCTOR;
}

void ZafImage::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafWindowObject::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafImage::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafImage::WriteData(ZafObjectPersistence &persist)
{
	// Write the image information.
	ZafUInt16 flag1 = 0;
	flag1 |= autoSize ? 0x0001 : 0;
	flag1 |= scaled ? 0x0002 : 0;
	flag1 |= tiled ? 0x0004 : 0;
	flag1 |= wallpaper ? 0x0008 : 0;
	ZafFile *file = persist.File();
	const ZafIChar *_pathName = pathName ? pathName : ZAF_NULLP(ZafIChar);
	*file << _pathName << pathID << flag1;
}

