//	Zinc Application Framework - Z_HLIST.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_hlist.hpp>
#define ZAF_HZ_LIST_INFO
#include "gbl_def.cpp"

// ----- ZafHzList ----------------------------------------------------------

ZafHzList::ZafHzList(int left, int top, int width, int height,
	int zCellWidth, int zCellHeight) :
	ZafWindow(left, top, width, height),
	autoSortData(false), cellWidth(zCellWidth), cellHeight(zCellHeight)
{
	// Initialize the list information.
	SetBordered(true);

#	if defined(ZAF_MOTIF)
	itemWidth = itemHeight = 0;
	firstVisible = lastVisible = cancelObject = ZAF_NULLP(ZafWindowObject);
#	endif
}

ZafHzList::ZafHzList(const ZafHzList &copy) :
	ZafWindow(copy),
	autoSortData(copy.autoSortData), cellWidth(copy.cellWidth),
	cellHeight(copy.cellHeight)
{
	// Set all children as non-system objects.  ZafWindow copy
	// constructor will not call ZafHzList::Add() since it is in the base
	// class phase of construction when the children are added.
	for (ZafWindowObject *object = First(); object; object = object->Next())
	{
		object->SetSystemObject(false);
		object->SetParentPalette(true);
	}

#	if defined(ZAF_MOTIF)
	itemWidth = itemHeight = 0;
	firstVisible = lastVisible = cancelObject = ZAF_NULLP(ZafWindowObject);
#	endif
}

ZafHzList::~ZafHzList(void)
{
}

void ZafHzList::Destroy(void)
{
	// Destroy the list elements (high level).
	OSDestroy();

	// Destroy the list elements (low level).
	ZafList::Destroy();
}

ZafWindowObject *ZafHzList::Duplicate(void)
{
	return (new ZafHzList(*this));
}

ZafLogicalEvent ZafHzList::LogicalEvent(const ZafEventStruct &event)
{
	ZafLogicalEvent ccode = MapEvent(defaultEventMap, event);
	return ((ccode == L_NONE) ? ZafWindow::LogicalEvent(event) : ccode);
}

const ZafPaletteStruct *ZafHzList::MapClassPalette(ZafPaletteType type, ZafPaletteState state)
{
	const ZafPaletteStruct *palette = MapPalette(defaultPaletteMap, type, state);
	return (palette ? palette : ZafWindow::MapClassPalette(type, state));
}

bool ZafHzList::SetAutoSortData(bool tAutoSortData)
{
	// Make sure the attribute has changed.
	if (autoSortData != tAutoSortData)
	{
		autoSortData = tAutoSortData;

		// Reset the sort function.
		if (autoSortData)
		{
			SetCompareFunction((ZafCompareFunction)CompareAscending);
			Sort();
		}
		else if (compareFunction == (ZafCompareFunction)CompareAscending)
			compareFunction = ZAF_NULLF(ZafCompareFunction);
	}

	// Return the current attribute.
	return (autoSortData);
}

ZafLogicalColor ZafHzList::SetBackgroundColor(ZafLogicalColor color, ZafLogicalColor mono)
{
	// Make sure there is a userPalette.
	if (!userPaletteData)
		SetUserPaletteData(new ZafPaletteData());

	// Add the background color entry.
	ZafPaletteStruct backgroundPalette = userPaletteData->GetPalette(ZAF_PM_BACKGROUND, ZAF_PM_NOT_SELECTED);
	backgroundPalette.colorBackground = color;
	backgroundPalette.monoBackground = mono;
	userPaletteData->AddPalette(ZAF_PM_BACKGROUND, ZAF_PM_NOT_SELECTED, backgroundPalette);

	// Add the selected text color entry.
	backgroundPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED);
	backgroundPalette.colorForeground = color;
	backgroundPalette.monoForeground = mono;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED, backgroundPalette);

	// Return the current color.
	return (color);
}

int ZafHzList::SetCellHeight(int setCellHeight)
{
	// Make sure the request is valid.
	if (!screenID)
		cellHeight = setCellHeight;

	// Return the current attribute.
	return (cellHeight);
}

int ZafHzList::SetCellWidth(int setCellWidth)
{
	// Make sure the request is valid.
	if (!screenID)
		cellWidth = setCellWidth;

	// Return the current attribute.
	return (cellWidth);
}

bool ZafHzList::SetCopyDraggable(bool setCopyDraggable)
{
	// Defer to the root class.
	return (ZafWindowObject::SetCopyDraggable(setCopyDraggable));
}

bool ZafHzList::SetDestroyable(bool )
{
	// destroyable is false for this class.
	destroyable = false;
	return (destroyable);
}

ZafLogicalFont ZafHzList::SetFont(ZafLogicalFont font)
{
	// Make sure there is a userPalette.
	if (!userPaletteData)
		SetUserPaletteData(new ZafPaletteData());

	// Add the new entries.
	ZafPaletteStruct textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED);
	textPalette.font = font;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED, textPalette);
	textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED);
	textPalette.font = font;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_SELECTED | ZAF_PM_ENABLED, textPalette);
	textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_ANY_STATE);
	textPalette.font = font;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_ANY_STATE, textPalette);

	// Return the current font.
	return (font);
}

bool ZafHzList::SetLinkDraggable(bool setLinkDraggable)
{
	// Defer to the root class.
	return (ZafWindowObject::SetLinkDraggable(setLinkDraggable));
}

bool ZafHzList::SetLocked(bool )
{
	// locked is false for this class.
	locked = false;
	return (locked);
}

bool ZafHzList::SetMaximized(bool )
{
	// maximized is false for this class.
	maximized = false;
	return (maximized);
}

bool ZafHzList::SetMinimized(bool )
{
	// minimized is false for this class.
	minimized = false;
	return (minimized);
}

bool ZafHzList::SetMoveable(bool )
{
	// moveable is false for this class.
	moveable = false;
	return (moveable);
}

bool ZafHzList::SetMoveDraggable(bool setMoveDraggable)
{
	// Defer to the root class.
	return (ZafWindowObject::SetMoveDraggable(setMoveDraggable));
}

bool ZafHzList::SetSizeable(bool )
{
	// sizeable is false for this class.
	sizeable = false;
	return (sizeable);
}

void ZafHzList::Sort(void)
{
	// Sort on the Zinc level.
	ZafList::Sort();

	// Sort from the OS level.
	OSSort();
}

ZafLogicalColor ZafHzList::SetTextColor(ZafLogicalColor color, ZafLogicalColor mono)
{
	// Make sure there is a userPalette.
	if (!userPaletteData)
		SetUserPaletteData(new ZafPaletteData());

	// Add the text color entry.
	ZafPaletteStruct textPalette = userPaletteData->GetPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED);
	textPalette.colorForeground = color;
	textPalette.monoForeground = mono;
	userPaletteData->AddPalette(ZAF_PM_TEXT, ZAF_PM_NOT_SELECTED | ZAF_PM_ENABLED, textPalette);

	// Add the selected text color entry.
	textPalette = userPaletteData->GetPalette(ZAF_PM_BACKGROUND, ZAF_PM_SELECTED);
	textPalette.colorBackground = color;
	textPalette.monoBackground = mono;
	userPaletteData->AddPalette(ZAF_PM_BACKGROUND, ZAF_PM_SELECTED, textPalette);

	// Return the current color.
	return (color);
}

// ----- Persistent functions -----------------------------------------------

ZafHzList::ZafHzList(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafWindow(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY))
{
	// Read the data.
	ZafHzList::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());

	// Set all children as non-system objects.  ZafWindow persistent
	// constructor will not call ZafHzList::Add() since it is in the base
	// class phase of construction when the children are added.
	for (ZafWindowObject *object = First(); object; object = object->Next())
	{
		object->SetSystemObject(false);
		object->SetParentPalette(true);
	}

#	if defined(ZAF_MOTIF)
	itemWidth = itemHeight = 0;
	firstVisible = lastVisible = cancelObject = ZAF_NULLP(ZafWindowObject);
#	endif
}

ZafElement *ZafHzList::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafHzList(name, persist));
}

void ZafHzList::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafUInt16 flag1;
	ZafFile *file = persist.File();
	*file >> flag1 >> cellWidth >> cellHeight;
	autoSortData = (flag1 & 0x0001) ? true : false;
}

void ZafHzList::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafWindow::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafHzList::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafHzList::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	ZafUInt16 flag1 = 0;
	ZafFile *file = persist.File();
	flag1 |= autoSortData ? 0x0001 : 0;
	*file << flag1 << cellWidth << cellHeight;
}

