//	Zinc Application Framework - Z_FILE.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_file.hpp>
#include <z_string.hpp>
#define ZAF_FILE_INFO
#include "gbl_def.cpp"

#if defined(ZAF_BIGENDIAN)
static void ReverseEndian(void *value, int bytes)
{
	// Converts big endian to little endian or little endian to big endian.
 	ZafUInt8 *buffer = (ZafUInt8 *)value;
	for (int i = 0; i < bytes / 2; i++)
	{
		int temp = buffer[i];
		buffer[i] = buffer[bytes - i - 1];
		buffer[bytes - i - 1] = temp;
	}
}
#endif

// ----- ZafFile ------------------------------------------------------------

ZafFile::ZafFile(ZafFileMode setMode) :
	error(ZAF_ERROR_NONE), mode(setMode)
{
}

ZafFile::~ZafFile(void)
{
}

// ----- Read functions -----------------------------------------------------

int ZafFile::Read(ZafInt8 &value)
{
	return (ReadData(&value, sizeof(value)));
}

int ZafFile::Read(ZafUInt8 &value)
{
	return (ReadData(&value, sizeof(value)));
}

int ZafFile::Read(ZafInt16 &value)
{
	int bytesRead = ReadData(&value, sizeof(value));
#if defined(ZAF_BIGENDIAN)
	ReverseEndian(&value, sizeof(value));
#endif
	return bytesRead;
}

int ZafFile::Read(ZafUInt16 &value)
{
	int bytesRead = ReadData(&value, sizeof(value));
#if defined(ZAF_BIGENDIAN)
	ReverseEndian(&value, sizeof(value));
#endif
	return bytesRead;
}

int ZafFile::Read(ZafInt32 &value)
{
	int bytesRead = ReadData(&value, sizeof(value));
#if defined(ZAF_BIGENDIAN)
	ReverseEndian(&value, sizeof(value));
#endif
	return bytesRead;
}

int ZafFile::Read(ZafUInt32 &value)
{
	int bytesRead = ReadData(&value, sizeof(value));
#if defined(ZAF_BIGENDIAN)
	ReverseEndian(&value, sizeof(value));
#endif
	return bytesRead;
}

int ZafFile::Read(ZafIChar *string, int length)
{
//??? Could be optimized?
	*string = '\0';
	ZafUInt16 readSize;
	int cnt1 = Read(readSize);
	if (cnt1 != sizeof(readSize))
	{
		// Couldn't read the length of the string.
		// error is set appropriately in ReadData().
		return (cnt1);
	}
	// Unicode support can load both Unicode and ISO 8859-1 strings
	// without data loss.  The upper 128 characters of OEM strings
	// are bogus in Unicode mode.  ISO and OEM support can load the
	// first 256 and 128 Unicode characters respectively.  Both lose
	// information outside this range.
	int doUni = (readSize & ZAF_UNICODE_FLAG);
	readSize &= ~ZAF_UNICODE_FLAG;
	if (length != -1 && (readSize + 1) > length)
	{
		// The buffer is not large enough to hold the string and a null terminator.
		error = ZAF_ERROR_FILE_READ;
		Seek(-cnt1, ZAF_SEEK_CURRENT);	// Put the file pointer back to where it was.
		return (-1);
	}
	int cnt2 = 0;
	if (readSize)
#if defined(ZAF_UNICODE)
		if (doUni)		// Byte Swap each short
		{
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt16 c;
				cnt2 += Read(c);
				string[i] = c;
			}
		}
		else
		{
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt8 tmp;
				cnt2 += Read(tmp);
				string[i] = tmp;
			}
		}
#else
		if (doUni)
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt16 tmp;
				cnt2 += Read(tmp);
				string[i] = (char)tmp;
			}
		else
			cnt2 += ReadData(string, readSize);
#endif
	string[readSize] = '\0';

	if (cnt2 != (int)(readSize * sizeof(string[0])))
	{
		// Couldn't read the whole string.
		// Set error, in case of eof (which isn't an error in ReadData()).
		error = ZAF_ERROR_FILE_READ;
		return (cnt2);
	}
	return (cnt1 + cnt2);
}

int ZafFile::Read(ZafIChar **string)
{
//??? Could be optimized?
	ZafUInt16 readSize;
	int cnt1 = Read(readSize);
	int cnt2 = 0;
	int doUni = (readSize & ZAF_UNICODE_FLAG);
	readSize &= ~ZAF_UNICODE_FLAG;
	if (cnt1 == sizeof(readSize) && readSize)
	{
		// See Unicode related comments in previous function.
		*string = new ZafIChar[readSize + 1];
#if defined(ZAF_UNICODE)
		if (doUni)		// Byte Swap each short
		{
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt16 c;
				cnt2 += Read(c);
				(*string)[i] = c;
			}
		}
		else
		{
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt8 tmp;
				cnt2 += Read(tmp);
				(*string)[i] = tmp;
			}
		}
#else
		if (doUni)
			for (int i = 0; i < readSize; i++)
			{
				ZafUInt16 tmp;
				cnt2 += Read(tmp);
				(*string)[i] = (char)tmp;
			}
		else
			cnt2 += ReadData(*string, readSize);
#endif
		(*string)[readSize] = '\0';
	}
	else
	{
		// If we got here because we couldn't read the length of the string,
		// error was set appropriately in ReadData().
		*string = ZAF_NULLP(ZafIChar);
	}

	if (cnt2 != (int)(readSize * (doUni ? 2 : 1)))
	{
		// Couldn't read the whole string.
		// Set error, in case of eof (which isn't an error in ReadData()).
		error = ZAF_ERROR_FILE_READ;
		return (cnt2);
	}
	return (cnt1 + cnt2);
}

int ZafFile::Read(void *buffer, int size, int num)
{
	return ReadData(buffer, size * num);
}

// The definition of ZAF_WORD_SIZE == 64 is the following:
// 8 bit chars, 16 bit shorts, 32 bit ints, and 64 bit longs.
// In this case, ZafInt32 is typedefed to int, rather than long, so
// Read(int &value) is the same as Read(ZafInt32 &value).
#if ZAF_WORD_SIZE != 64
int ZafFile::Read(int &value)
{
	ZafInt32 tempValue;
	int bytesRead = Read(tempValue);
	value = (int)tempValue;
	return (bytesRead);
}
#endif

// ----- Write functions ----------------------------------------------------

int ZafFile::Write(ZafInt8 value)
{
	return WriteData(&value, sizeof(value));
}

int ZafFile::Write(ZafUInt8 value)
{
	return WriteData(&value, sizeof(value));
}

int ZafFile::Write(ZafInt16 value)
{
#if defined(ZAF_BIGENDIAN)
	ReverseEndian(&value, sizeof(value));
#endif
	return WriteData(&value, sizeof(value));
}

int ZafFile::Write(ZafUInt16 value)
{
#if defined(ZAF_BIGENDIAN)
	ReverseEndian(&value, sizeof(value));
#endif
	return WriteData(&value, sizeof(value));
}

int ZafFile::Write(ZafInt32 value)
{
#if defined(ZAF_BIGENDIAN)
	ReverseEndian(&value, sizeof(value));
#endif
	return WriteData(&value, sizeof(value));
}

int ZafFile::Write(ZafUInt32 value)
{
#if defined(ZAF_BIGENDIAN)
	ReverseEndian(&value, sizeof(value));
#endif
	return WriteData(&value, sizeof(value));
}

int ZafFile::Write(const ZafIChar *string)
{
//??? Could be optimized?
	// Don't write the null terminator.
	ZafUInt16 tSize = (ZafUInt16)(string == ZAF_NULLP(ZafIChar) ? 0 : strlen(string));
	// See Unicode comments in previous functions.
#if defined(ZAF_UNICODE)
	int i;
	int isISO = 1;
	// Let's try to store the string as an ISO string (save lots of space).
	for (i=0; i < tSize; i++)
		if (string[i] > 0x00ff)
			isISO = 0;
	int cnt1 = Write((ZafUInt16)(tSize | (isISO ? 0 : ZAF_UNICODE_FLAG)));
	if (cnt1 != sizeof(tSize))
	{
		// Couldn't write the length of the string. error is set in WriteData().
		return (cnt1);
	}
	int cnt2 = 0;
	// Must (possibly) byte swap the file
	if (isISO)
		for (i = 0; i < tSize; i++)
		{	
			ZafUInt8 c = string[i];
			cnt2 += Write(c);
		}
	else
		for (i = 0; i < tSize; i++)
		{	
			ZafUInt16 c = string[i];
			cnt2 += Write(c);
		}
#else
	int cnt1 = Write(tSize);
	if (cnt1 != sizeof(tSize))
	{
		// Couldn't write the length of the string. error is set in WriteData().
		return (cnt1);
	}
	int cnt2 = 0;
	cnt2 += WriteData((void *)string, tSize);
#endif

	if (cnt2 != (int)(tSize * sizeof(string[0])))
	{
		// Couldn't write the whole string. error is set in WriteData().
		return (cnt2);
	}
	return (cnt1 + cnt2);
}

int ZafFile::Write(const void *buffer, int size, int num)
{
	return WriteData(buffer, size * num);
}

// The definition of ZAF_WORD_SIZE == 64 is the following:
// 8 bit chars, 16 bit shorts, 32 bit ints, and 64 bit longs.
// In this case, ZafInt32 is typedefed to int, rather than long, so
// Write(int &value) is the same as Write(ZafInt32 &value).
#if ZAF_WORD_SIZE != 64
int ZafFile::Write(int value)
{
	ZafInt32 tempValue = (ZafInt32)value;
	return Write(tempValue);
}
#endif

