//	Zinc Application Framework - Z_DSKFIL.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#if defined(__BCPLUSPLUS__) || defined(__TCPLUSPLUS__)
#	include <dir.h>
#	include <direct.h>
#elif defined(__hpux) || defined(__sgi) || defined(sun) || defined(_SINIX) || defined(_IBM_RS6000) || defined(SCO_UNIX) || defined(__QNX__) || defined(__GNUC__)
#	include <dirent.h>
#	include <sys/stat.h>
#elif (defined(__SC__) && !defined(macintosh)) || defined(_MSC_VER) || (defined(__WATCOMC__) && !defined(__QNX__)) || defined(__HIGHC__) || defined(__IBMCPP__) || defined(__GLOCKENSPIEL__)
#	include <direct.h>
#endif
#if defined(__SC__) && !defined(macintosh)
#	define _dos_getdrive dos_getdrive
#	define _dos_setdrive dos_setdrive
#endif
#if defined(_MSC_VER) && defined(DOS386)
#	include <pldos32.h>
#endif
#include <z_env.hpp>
#if defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(ZAF_OS2) || defined(__DVX__)
#	if !defined(__IBMCPP__) && !defined(__GLOCKENSPIEL__)
#		include <dos.h>
#	endif
#endif

#include <z_ctype.hpp>
#include <z_dskfil.hpp>
#include <z_cset.hpp>
#include <z_string.hpp>
#include <z_unistd.hpp>
#include <z_utils.hpp>

#if defined(ZAF_MACINTOSH)
#	include "a_utils.hpp"

	OSType ZafDiskFile::fileCreator = 'Anon';
	OSType ZafDiskFile::fileType = 'TEXT';
#endif

// ----- ZafDiskFileSystem --------------------------------------------------

void ZafDiskFileSystem::AppendFullPath(ZafIChar *fullPath,
	const ZafIChar *pathName, const ZafIChar *fileName,
	const ZafIChar *extension)
{
	if (pathName != fullPath)
		strcpy(fullPath, pathName);
	int i = strlen(fullPath);
#if defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(ZAF_OS2) || defined(__DVX__)
	if (i && fullPath[i-1] != zafCodeSet->dirSepStr[0] && fullPath[i-1] != ':')
#else
	if (i && fullPath[i-1] != zafCodeSet->dirSepStr[0])
#endif
		strcat(fullPath, zafCodeSet->dirSepStr);
	if (fileName)
		strcat(fullPath, fileName);
	if (extension)
		ChangeExtension(fullPath, extension);
}

void ZafDiskFileSystem::ChangeExtension(ZafIChar *pathName,
	const ZafIChar *newExtension)
{
	ZafIChar *oldExtension = strrchr(pathName, '.');
	if (oldExtension)
		*oldExtension = '\0';
	if (newExtension)
		strcat(pathName, newExtension);
}

void ZafDiskFileSystem::MakeFullPath(ZafIChar *tmppath)
{
	// Check for a null string;
	if (!tmppath)
		return;

#if defined(ZAF_POSIX)
	if (tmppath[0] != zafCodeSet->dirSepStr[0])
	{
		ZafIChar save[ZAF_MAXPATHLEN];
		strcpy(save, tmppath);
		getcwd(tmppath, ZAF_MAXPATHLEN);
		AppendFullPath(tmppath, tmppath, save);
	}
#elif defined(ZAF_MACINTOSH)
	// A full path may be a qualified partial path, still usable by the Toolbox.
	if (!strchr(tmppath, zafCodeSet->dirSepStr[0]))
		strcat(tmppath, zafCodeSet->dirSepStr);
#else
	if (tmppath[0] == zafCodeSet->dirSepStr[0])
	{
		memmove((char *)&tmppath[2], (const char *)&tmppath[0], (strlen(tmppath)+1) * sizeof(tmppath[0]));
		tmppath[1] = ':';
#if defined(ZAF_WIN32)
		unsigned drive;
		char currentDrive[MAX_PATH];
		GetCurrentDirectoryA(MAX_PATH, currentDrive);
		drive = ToUpper(currentDrive[0]) - 'A' + 1;
		tmppath[0] = drive - 1 + 'A';
#elif defined(ZAF_OS2)
		ULONG pDrive, lDrive;
		DosQueryCurrentDisk(&pDrive, &lDrive);
		tmppath[0] = pDrive - 1 + 'A';
#elif defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(__DVX__)
		unsigned drive;
		_dos_getdrive(&drive);
		tmppath[0] = (ZafIChar)(drive - 1 + 'A');
#else
		????;   This is an error;
#endif
	}
	else if (tmppath[0] && tmppath[1] == ':' && tmppath[2] != zafCodeSet->dirSepStr[0])
	{
		ZafIChar save[ZAF_MAXPATHLEN];
		strcpy(save, &tmppath[2]);
#if defined(ZAF_WIN32)
#	if defined(ZAF_UNICODE)
		char currentDrive[MAX_PATH], newDrive[MAX_PATH];
		GetCurrentDirectoryA(MAX_PATH, currentDrive);
		newDrive[0] = tmppath[0];
		newDrive[1] = ':';
		newDrive[2] = 0;
		SetCurrentDirectoryA(newDrive);
		GetCurrentDirectoryA(ZAF_MAXPATHLEN, newDrive);
		MultiByteToWideChar(CP_OEMCP, MB_PRECOMPOSED, newDrive, -1, &tmppath[2], ZAF_MAXPATHLEN-2);
		SetCurrentDirectoryA(currentDrive);
#	else
		char currentDrive[MAX_PATH];
		GetCurrentDirectory(MAX_PATH, currentDrive);
		SetCurrentDirectory(tmppath);
		GetCurrentDirectory(ZAF_MAXPATHLEN, tmppath);
		SetCurrentDirectory(currentDrive);
#	endif
#elif defined(ZAF_OS2)
		ULONG drive, xxx;
		DosQueryCurrentDisk(&drive, &xxx);
		DosSetDefaultDisk(tmppath[0] - 'a' + 1);
		getcwd(&tmppath[2], CCHMAXPATH);
		DosSetDefaultDisk(drive);
#elif defined(ZAF_MSDOS) || defined(ZAF_MSWINDOWS) || defined(__DVX__)
		unsigned drive, xxx;
		_dos_getdrive(&drive);
		_dos_setdrive(tmppath[0] - 'A' + 1, &xxx);
		getcwd(tmppath, ZAF_MAXPATHLEN);
		_dos_setdrive(drive, &xxx);
#else
		????; This is an error;
#endif
		AppendFullPath(tmppath, tmppath, save);
	}
	else if (!(tmppath[0] && tmppath[1] == ':' && tmppath[2] == zafCodeSet->dirSepStr[0]))
	{
		ZafIChar save[ZAF_MAXPATHLEN];
		strcpy(save, tmppath);
		getcwd(tmppath, ZAF_MAXPATHLEN);
		AppendFullPath(tmppath, tmppath, save);
	}
#endif
}

void ZafDiskFileSystem::TempName(ZafIChar *tempname)
{
#if defined(ZAF_POSIX) || (defined(ZAF_OS2) && defined(__IBMCPP__))
	tmpnam(tempname);
#else
	ZafIChar tmppath[ZAF_MAXPATHLEN];
	tmpnam(tmppath);
	MakeFullPath(tempname);
	AppendFullPath(tempname, tempname, tmppath);
#endif
}

ZafDiskFileSystem::ZafDiskFileSystem(void) :
	ZafFileSystem()
{
}

ZafDiskFileSystem::~ZafDiskFileSystem(void)
{
}

int ZafDiskFileSystem::ChDir(const ZafIChar *newPath)
{
	// Move to the new directory.
#if defined(ZAF_MACINTOSH)
	// Not implemented.
#	pragma unused(newPath)
	return (-1);
#else
	return (chdir(newPath));
#endif
}

int ZafDiskFileSystem::FindFirst(ZafFileInfoStruct &fileInfo,
	const ZafIChar *searchName, ZafStringID stringID, ZafNumberID numberID)
{
	// Initialize the info information.
	fileInfo.name = ZAF_NULLP(ZafIChar);
	fileInfo.stringID = ZAF_NULLP(ZafIChar);
	fileInfo.numberID = 0;
	fileInfo.length = -1;
	fileInfo.attributes = ZAF_FATTRIB_NONE;
	fileInfo.internalHandle = ZAF_NULLP(void);

	// Find the file.
	return (FindNext(fileInfo, searchName, stringID, numberID));
}

int ZafDiskFileSystem::FindNext(ZafFileInfoStruct &fileInfo,
	const ZafIChar *searchName, ZafStringID , ZafNumberID )
{
	// Initialize the search variables.
	int tError = 0;
	char osName[ZAF_MAXPATHLEN];
	zafCodeSet->ConvertToOSString(searchName, osName, false);

	// Prepare for the next lookup.
	if (fileInfo.name)
	{
		delete (ZafIChar *)fileInfo.name;
		fileInfo.name = ZAF_NULLP(ZafIChar);
	}
	fileInfo.length = -1;
	fileInfo.attributes = ZAF_FATTRIB_NONE;

	// Check for find-first implementations.
#if defined(ZAF_WIN32)
	HANDLE *searchHandle = (HANDLE *)fileInfo.internalHandle;
	WIN32_FIND_DATA findData ;
	if (!searchHandle)
	{
		fileInfo.internalHandle = searchHandle = new HANDLE;
		*searchHandle = FindFirstFile(osName, &findData);
		if (searchHandle == INVALID_HANDLE_VALUE)
			tError = -1;
	}
	else if (FindNextFile(searchHandle, &findData) != 0)
		tError = -1;

	// Fill the fileInfo structure.
	if (!tError)
	{
		fileInfo.name = zafCodeSet->ConvertToZafString(findData.cFileName);
		fileInfo.length = findData.nFileSizeLow;
		if (findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
			fileInfo.attributes |= ZAF_FATTRIB_DIRECTORY;
		if (findData.dwFileAttributes & FILE_ATTRIBUTE_READONLY)
			fileInfo.attributes |= ZAF_FATTRIB_READONLY;
	}
#elif defined(__BCPLUSPLUS__) || defined(__TCPLUSPLUS__)
	ffblk *info = (ffblk *)fileInfo.internalHandle;
	if (!info)
	{
		fileInfo.internalHandle = info = new ffblk;
		if (findfirst(osName, info, FA_RDONLY | FA_DIREC) != 0)
			tError = -1;
	}
	else if (findnext(info) != 0)
		tError = -1;

	// Fill the fileInfo structure.
	if (!tError)
	{
		fileInfo.name = zafCodeSet->ConvertToZafString(info->ff_name);
		fileInfo.length = info->ff_fsize;
		if (info->ff_attrib & FA_DIREC)
			fileInfo.attributes |= ZAF_FATTRIB_DIRECTORY;
		if (info->ff_attrib & FA_RDONLY)
			fileInfo.attributes |= ZAF_FATTRIB_READONLY;
	}
#elif defined(_MSC_VER)
	_find_t *info = (_find_t *)fileInfo.internalHandle;
	if (!info)
	{
		fileInfo.internalHandle = info = new _find_t;
		if (_dos_findfirst(osName, _A_RDONLY | _A_SUBDIR, info) != 0)
			tError = -1;
	}
	else if (_dos_findnext(info) != 0)
		tError = -1;

	// Fill the fileInfo structure.
	if (!tError)
	{
		fileInfo.name = zafCodeSet->ConvertToZafString(info->ff_name);
		fileInfo.length = info->ff_fsize;
		if (info->ff_attrib & FA_DIREC)
			fileInfo.attributes |= ZAF_FATTRIB_DIRECTORY;
		if (info->ff_attrib & FA_RDONLY)
			fileInfo.attributes |= ZAF_FATTRIB_READONLY;
	}
#elif defined(__WATCOMC__) && !defined(__QNX__)
	find_t *info = (find_t *)fileInfo.internalHandle;
	if (!info)
	{
		fileInfo.internalHandle = info = new find_t;
		if (_dos_findfirst(osName, _A_SUBDIR, info) != 0)
			tError = -1;
	}
	else if (_dos_findnext(info) != 0)
		tError = -1;

	// Fill the fileInfo structure.
	if (!tError)
	{
		fileInfo.name = zafCodeSet->ConvertToZafString(info->name);
		fileInfo.length = info->size;
		if (info->attrib & _A_SUBDIR)
			fileInfo.attributes |= ZAF_FATTRIB_DIRECTORY;
		if (info->attrib & _A_RDONLY)
			fileInfo.attributes |= ZAF_FATTRIB_READONLY;
	}
#elif defined(ZAF_MOTIF)
	DIR *directory = (DIR *)fileInfo.internalHandle;
	if (!directory)
		fileInfo.internalHandle = directory = opendir(".");

	dirent *info = readdir(directory);
	if (info)
	{
		struct stat fileStats;
		stat(info->d_name, &fileStats);
		fileInfo.name = zafCodeSet->ConvertToZafString(info->d_name);
		fileInfo.length = fileStats.st_size;
		if (fileStats.st_mode & S_IFDIR)
			fileInfo.attributes |= ZAF_FATTRIB_DIRECTORY;
		if ((fileStats.st_mode & (S_IWUSR | S_IWGRP | S_IWOTH)) == 0)
			fileInfo.attributes |= ZAF_FATTRIB_READONLY;
	}
	else if (directory)
	{
		tError = -1;
		closedir(directory);
		fileInfo.internalHandle = ZAF_NULLP(void);
	}
#endif

	// Return the error status.
	return (tError);
}

int ZafDiskFileSystem::FindClose(ZafFileInfoStruct &fileInfo)
{
	int tError = -1;
#if defined(ZAF_WIN32)
	HANDLE *searchHandle = (HANDLE *)fileInfo.internalHandle;
 	if (searchHandle)
	{
		::FindClose(searchHandle);
		delete searchHandle;
		tError = 0;
	}
#elif defined(__BCPLUSPLUS__) || defined(__TCPLUSPLUS__)
 	if (fileInfo.internalHandle)
	{
		delete fileInfo.internalHandle;
		tError = 0;
	}
#elif defined(_MSC_VER)
 	if (fileInfo.internalHandle)
	{
		delete fileInfo.internalHandle;
		tError = 0;
	}
#elif defined(__WATCOMC__) && !defined(__QNX__)
 	if (fileInfo.internalHandle)
	{
		delete fileInfo.internalHandle;
		tError = 0;
	}
#elif defined(ZAF_MOTIF)
	DIR *directory = (DIR *)fileInfo.internalHandle;
 	if (directory)
	{
		closedir(directory);
		tError = 0;
	}
#elif defined(ZAF_MACINTOSH)
 	if (fileInfo.internalHandle)
	{
		delete fileInfo.internalHandle;
		tError = 0;
	}
#endif
 	fileInfo.internalHandle = ZAF_NULLP(void);
	return (tError);
}

int ZafDiskFileSystem::GetCWD(ZafIChar *pathName, int pathLength)
{
	// Get the current working directory.
#if defined(ZAF_MACINTOSH)
	// Not implemented.
#	pragma unused(pathName)
#	pragma unused(pathLength)
	return (-1);
#else
	return (getcwd(pathName, pathLength) ? 0 : -1);
#endif
}

int ZafDiskFileSystem::MkDir(const ZafIChar *pathName, ZafStringID ,
	ZafNumberID)
{
	char osName[ZAF_MAXPATHLEN];
	zafCodeSet->ConvertToOSString(pathName, osName, false);

#if defined(ZAF_POSIX)
	return (mkdir(osName, 0666));
#elif defined(ZAF_MACINTOSH)
	long createdDirID;
	return (DirCreate(0, 0L, c2pstr(osName), &createdDirID));
#else
	return (mkdir(osName));
#endif
}

void ZafDiskFileSystem::Close(ZafFile *file)
{
	ZafDiskFile *diskFile = (ZafDiskFile *)file;
	delete diskFile;
}

void ZafDiskFileSystem::DeleteDriveNames(ZafIChar *driveName[])
{
	int index = 0;
	while (driveName[index])
		delete []driveName[index++];
	delete []driveName;
}

#if defined(ZAF_X11) || defined(ZAF_MACINTOSH)
int ZafDiskFileSystem::GetDriveNames(ZafIChar **getDriveName[])
{
	// Motif and Macintosh do not have drives.
	*getDriveName = 0;
	return (-1);
}
#else
int ZafDiskFileSystem::GetDriveNames(ZafIChar **getDriveName[])
{
#if defined(_MSC_VER)
	int currentDrive = _getdrive() + 1;
	int drives = 0;
	bool available[26];
	for (int drive = 0; drive < 26; drive++)
	{
		if (_chdrive(drive + 1) == 0)
		{
			drives++;
			available[drive] = true;
		}
		else
			available[drive] = false;
	}
	_chdrive(currentDrive);

	if (getDriveName)
	{
		ZafIChar **driveName = new ZafIChar *[drives + 1];
		int availableIndex = 0;
		for (int index = 0; index < drives; index++)
		{
			while (!available[availableIndex++])
				;
			driveName[index] = new ZafIChar[3];
			sprintf(driveName[index], "%c:", 'A' + availableIndex);
		}
		driveName[drives] = ZAF_NULLP(ZafIChar);

		*getDriveName = driveName;
	}
	return (drives);
#elif defined(__BCPLUSPLUS__) || defined(__TCPLUSPLUS__)
	int currentDrive = getdisk() + 1;
	int drives = 0;
	bool available[26];
	for (int drive = 0; drive < 26; drive++)
	{
		if (_chdrive(drive + 1) == 0)
		{
			drives++;
			available[drive] = true;
		}
		else
			available[drive] = false;
	}
	_chdrive(currentDrive);

	if (getDriveName)
	{
		ZafIChar **driveName = new ZafIChar *[drives + 1];
		int availableIndex = 0;
		for (int index = 0; index < drives; index++)
		{
			while (!available[availableIndex++])
				;
			driveName[index] = new ZafIChar[3];
			sprintf(driveName[index], "%c:", 'A' + availableIndex);
		}
		driveName[drives] = ZAF_NULLP(ZafIChar);

		*getDriveName = driveName;
	}
	return (drives);
#elif defined(__WATCOMC__) && !defined(__QNX__)
	unsigned currentDrive, total;
	_dos_getdrive(&currentDrive);
	int drives = 0;
	bool available[26];
	for (int drive = 0; drive < 26; drive++)
	{
		unsigned setDrive, getDrive;
		setDrive = drive + 1;
		_dos_setdrive(setDrive, &total);
		_dos_getdrive(&getDrive);

		if (getDrive == setDrive)
		{
			drives++;
			available[drive] = true;
		}
		else
			available[drive] = false;
	}
	_dos_setdrive(currentDrive, &total);

	if (getDriveName)
	{
		ZafIChar **driveName = new ZafIChar *[drives + 1];
		int availableIndex = 0;
		for (int index = 0; index < drives; index++)
		{
			while (!available[availableIndex++])
				;
			driveName[index] = new ZafIChar[3];
			sprintf(driveName[index], ZAF_ITEXT("%c:"), 'A' + availableIndex);
		}
		driveName[drives] = ZAF_NULLP(ZafIChar);

		*getDriveName = driveName;
	}
	return (drives);
#else
	//??? Not implemented yet.
	return (-1);
#endif
}
#endif

ZafFile *ZafDiskFileSystem::Open(const ZafIChar *fileName,
	const ZafFileMode mode, ZafStringID, ZafNumberID)
{
	ZafDiskFile *file = new ZafDiskFile(fileName, mode);
	return file;
}

int ZafDiskFileSystem::Remove(const ZafIChar *name)
{
	char osName[ZAF_MAXPATHLEN];
	zafCodeSet->ConvertToOSString(name, osName, false);

#if defined(ZAF_MACINTOSH)
	return (HDelete(0, 0L, c2pstr(osName)) == noErr ? 0 : -1);
#else
	return (remove(osName));
#endif
}

int ZafDiskFileSystem::Rename(const ZafIChar *oldName, const ZafIChar *newName)
{
	char osOldName[ZAF_MAXPATHLEN], osNewName[ZAF_MAXPATHLEN];
	zafCodeSet->ConvertToOSString(oldName, osOldName, false);
	zafCodeSet->ConvertToOSString(newName, osNewName, false);

#if defined(ZAF_MACINTOSH)
	return (HRename(0, 0L, c2pstr(osOldName), c2pstr(osNewName)) == noErr ? 0 : -1);
#else
	return (::rename(osOldName, osNewName));
#endif
}

int ZafDiskFileSystem::RmDir(const ZafIChar *pathName, bool deleteContents)
{
	char osPath[ZAF_MAXPATHLEN];
	zafCodeSet->ConvertToOSString(pathName, osPath, false);

	if (deleteContents)
	{
		ZafIChar currentDirectory[ZAF_MAXPATHLEN];
		GetCWD(currentDirectory, ZAF_MAXPATHLEN);
		ChDir(pathName);

		ZafFileInfoStruct fileInfo;
		int finished = FindFirst(fileInfo, ZAF_ITEXT("*"));
		while (!finished)
		{
			if (fileInfo.Directory())
				RmDir(fileInfo.name, true);
			else
				Remove(fileInfo.name);
			finished = FindNext(fileInfo, ZAF_ITEXT("*"));
		}

		ChDir(currentDirectory);
	}

	int result = 0;
#if defined(__BCPLUSPLUS__) || defined(__TCPLUSPLUS__) || defined(__WATCOMC__) || defined(ZAF_X11)
	if (rmdir(osPath) != 0)
		result = -1;
#elif defined(_MSC_VER)
	if (_rmdir(osPath) != 0)
		result = -1;
#elif defined(ZAF_MACINTOSH)
	if (HDelete(0, 0L, c2pstr(osPath)) != noErr)
		result = -1;
#else
	//??? Not Implemented yet.
#endif
	return (result);
}

#if defined(ZAF_X11) || defined(ZAF_MACINTOSH)
int ZafDiskFileSystem::SetDrive(const ZafIChar *)
{
	// Motif and Macintosh do not have the concept of a current drive.
	return (-1);
}
#else
int ZafDiskFileSystem::SetDrive(const ZafIChar *driveName)
{
#if defined(__BCPLUSPLUS__) || defined(__TCPLUSPLUS__) || defined(_MSC_VER)
	return (_chdrive(toupper(*driveName) - 'A' + 1));
#elif defined(__WATCOMC__) && !defined(__QNX__)
	unsigned setDrive = toupper(*driveName) - 'A' + 1;
	unsigned getDrive = 0;
	unsigned drives;
	_dos_setdrive(setDrive, &drives);
	_dos_getdrive(&getDrive);
	return ((getDrive == setDrive) ? 0 : -1);
#else
	//??? Not implemented yet.
	return (-1);
#endif
}
#endif

// ----- ZafDiskFile --------------------------------------------------------

ZafDiskFile::ZafDiskFile(const ZafIChar *tPathName, ZafFileMode newMode) :
	ZafFile(newMode),
	useCount(0), file(0)
{
	// Set the pathname.
	pathName = tPathName ? strdup(tPathName) : ZAF_NULLP(ZafIChar);

	// Check the mode.
	if (DerivedAccess())
		return;

#if defined(ZAF_MACINTOSH)
	file = MacOpen(pathName, fileCreator, fileType, mode);
#else
	// Determine the mode.
	ZafIChar openMode[6];
	if (OpenCreate())
		strcpy(openMode, ZAF_ITEXT("a+"));
	else if (Create())
		strcpy(openMode, ZAF_ITEXT("w+"));
	else if (ReadWrite())
		strcpy(openMode, ZAF_ITEXT("r+"));
	else
		strcpy(openMode, ZAF_ITEXT("r"));

	if (BinaryMode())
		strcat(openMode, ZAF_ITEXT("b"));
	else if (TextMode())
		strcat(openMode, ZAF_ITEXT("t"));

	// Open the file.
	file = fopen(pathName, openMode);
#endif

	if (!file)
		error = ZAF_ERROR_FILE_OPEN;
}

ZafDiskFile::~ZafDiskFile(void)
{
	// Delete the file.
	if (!DerivedAccess() && file)
#if defined(ZAF_MACINTOSH)
		FSClose(file);
#else
		fclose(file);
#endif

	// Delete the pathname.
	if (pathName)
		delete []pathName;
}

ZafOffset ZafDiskFile::Length(void) const
{
	// Return the file length.
	ZafOffset position = -1;
	if (file)
#if defined(ZAF_MACINTOSH)
		GetEOF(file, &position);
#else
		position = ftell(file);
#endif
	return (position);
}

int ZafDiskFile::Seek(ZafOffset offset, ZafSeek location)
{
	// Seek into the file.
	int result = -1;
	if (file)
	{
#if defined(ZAF_MACINTOSH)
		short fromLoc = fsFromStart;
		switch (location)
		{
		case ZAF_SEEK_START:
			fromLoc = fsFromStart;
			break;
		case ZAF_SEEK_CURRENT:
			fromLoc = fsFromMark;
			break;
		case ZAF_SEEK_END:
			fromLoc = fsFromLEOF;
			break;
		}
		result = SetFPos(file, fromLoc, offset);
#else
		result = fseek(file, offset, (int)location);
#endif
	}

	if (result)
		error = ZAF_ERROR_FILE_POSITION;
	return (result);
}

ZafOffset ZafDiskFile::Tell(void) const
{
	// Tell the current offset.
	ZafOffset position = -1;
	if (file)
#if defined(ZAF_MACINTOSH)
		GetFPos(file, &position);
#else
		position = ftell(file);
#endif
	return (position);
}

// ----- I/O ----------------------------------------------------------------

int ZafDiskFile::ReadData(void *buffer, int size)
{
#if defined(ZAF_MACINTOSH)
	long count = size;
	OSErr theError = FSRead(file, &count, (Ptr)buffer);
	if (theError == noErr || theError == eofErr)
		return ((int)count);
	else
	{
		error = ZAF_ERROR_FILE_READ;
		return (0);
	}
#else
	int result = fread((char *)buffer, 1, size, file);
	if (ferror(file))
		error = ZAF_ERROR_FILE_READ;
	return (result);
#endif
}

int ZafDiskFile::WriteData(const void *buffer, int size)
{
	// Check for a valid size.
	if (size == 0)
		return (size);

#if defined(ZAF_MACINTOSH)
	long count = size;
	if (FSWrite(file, &count, (Ptr)buffer) == noErr)
	{
		if (count != (long)size)
			error = ZAF_ERROR_FILE_WRITE;
		return ((int)count);
	}
	else
	{
		error = ZAF_ERROR_FILE_WRITE;
		return (0);
	}
#else
	int result = fwrite((char *)buffer, 1, size, file);
	if (result != size)
		error = ZAF_ERROR_FILE_WRITE;
	return (result);
#endif
}

