//	Zinc Application Framework - Z_DATE1.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_date1.hpp>
#include <z_error.hpp>
#include <z_string.hpp>
#include <z_app.hpp>
#define ZAF_DATE_INFO
#include "lang_def.cpp"
#include "gbl_def.cpp"

static ZafIChar dateSeparator[3] = { '.','.',0 };
static ZafIChar rangeSeparator[3] = { ',',' ',0 };
static ZafStringID _dateDataName = ZAF_ITEXT("dateData");

// ----- ZafDate ------------------------------------------------------------

ZafDate::ZafDate(int left, int top, int width, ZafDateData *zDateData) :
	ZafString(left, top, width, new ZafStringData),
	dateData(ZAF_NULLP(ZafDateData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the date data.
	SetDateData(zDateData ? zDateData : new ZafDateData);
}

ZafDate::ZafDate(int left, int top, int width, int year, int month, int day) :
	ZafString(left, top, width, new ZafStringData),
	dateData(ZAF_NULLP(ZafDateData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Initialize the date data.
	SetDateData(new ZafDateData(year, month, day));
}

ZafDate::ZafDate(const ZafDate &copy) :
	ZafString(copy), dateData(ZAF_NULLP(ZafDateData))
{
	// Check the language and local information.
	LanguageAllocate();

	// Copy the date data.
	if (copy.DateData()->Destroyable())
		SetDateData(new ZafDateData(*copy.DateData()));
	else
		SetDateData(copy.DateData());
}

ZafDate::~ZafDate(void)
{
	// Remove the data notification.
	dateData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Check the language and local information.
	LanguageFree();

	// Restore the date information.
	if (dateData->Destroyable() && dateData->NotifyCount() == 0)
		delete dateData;
}

ZafError ZafDate::Decrement(ZafData *data)
{
	// Check for a valid object.
	ZafDateData *date = DynamicPtrCast(data, ZafDateData);
	if (!date)
		return (ZAF_ERROR_INVALID_CLASS);

	// Decrement the value.
	ZafDateData _date(*dateData);
	OSGetDate();
	*dateData -= *date; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

ZafWindowObject *ZafDate::Duplicate(void)
{
	return (new ZafDate(*this));
}

ZafEventType ZafDate::Event(const ZafEventStruct &event)
{
	// Process OS specific messages.
	ZafEventType ccode = event.type;
	if (event.InputType() == E_KEY)
		ccode = LogicalEvent(event);

	// Check for zinc events.
	switch (ccode)
	{
	// ----- Create messages -----
	case S_INITIALIZE:
		// Check for an unanswered field.
		if (!Unanswered())
			OSSetDate();
		ccode = ZafString::Event(event);
		break;

	// ----- Action messages -----
	case L_SELECT:
	case N_NON_CURRENT:
		{
		// Keep a temporary date in case of error.
		ZafDateData oldValue(*dateData);
		OSGetDate();

		// Call the associated user function.
		ccode = ZafString::Event(event);

		// Check for errors.
		if (ccode != 0 && ccode != ZAF_ERROR_LEAVE_INVALID)
			dateData->SetDate(oldValue);
		else
			OSSetDate();
		}
		break;

	case S_COPY_DATA:
		{
		ZafWindowObject *object = event.windowObject;
		ZafDate *date = DynamicPtrCast(object, ZafDate);
		if (date)
			dateData->SetDate(*date->DateData());
		}
		break;

	case S_SET_DATA:
		if (event.windowObject)
		{
			ZafWindowObject *object = event.windowObject;
			ZafDate *date = DynamicPtrCast(object, ZafDate);
			if (date)
				SetDateData(date->DateData());
		}
		else
			SetDateData(new ZafDateData(*dateData));
		break;

	// ----- Default or unknown messages -----
	default:
		ccode = ZafString::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

ZafError ZafDate::Increment(ZafData *data)
{
	// Check for a valid object.
	ZafDateData *date = DynamicPtrCast(data, ZafDateData);
	if (!date)
		return (ZAF_ERROR_INVALID_CLASS);

	// Increment the value.
	ZafDateData _date(*dateData);
	OSGetDate();
	*dateData += *date; // update is automatic.

	// Return success.
	return (ZAF_ERROR_NONE);
}

void ZafDate::LanguageAllocate(void)
{
	// Check for initialization.
	if (errorStrings)
		return;
	ZafApplication::AddStaticModule(LanguageFree);

	// Try the zafDataManager.
	if (!errorStrings && zafDataManager)
		errorStrings = DynamicPtrCast(zafDataManager->AllocateData(className, ZafLanguageData::className, ZafLanguageData::classID), ZafLanguageData);

	// Default to code initialization.
	if (!errorStrings)
		errorStrings = new ZafLanguageData(defaultErrorStrings);
}

void ZafDate::LanguageFree(bool globalRequest)
{
	// Destroy the static language.
	if (globalRequest && errorStrings && errorStrings->Destroyable() &&
		errorStrings->NotifyCount() == 0)
	{
		delete errorStrings;
		errorStrings = ZAF_NULLP(ZafLanguageData);
	}
}

ZafError ZafDate::OSGetDate(void)
{
	// Disable notification.
	ZafUpdateType update = dateData->Update(this);
	dateData->SetUpdate(this, ZAF_UPDATE_NONE);

	// Get the string data from the OS.
	OSGetText();

	// Set the date based on the string data.
	error = dateData->SetDate(stringData->Text(), InputFormatText());

	// Restore notification.
	dateData->SetUpdate(this, update);
	return (error);
}

ZafError ZafDate::OSSetDate(void)
{
	// Set the string based on the date data.
	ZafIChar text[ZAF_STRING_DATA_LENGTH];
	dateData->FormattedText(text, ZAF_STRING_DATA_LENGTH, OutputFormatText());
	stringData->SetText(text);

	// Return the current status.
	return (ZAF_ERROR_NONE);
}

ZafError ZafDate::SetDateData(ZafDateData *tDateData)
{
	// Remove the data notification.
	if (dateData)
		dateData->SubtractNotification(this, (ZafUpdateFunction)Update);

	// Reset the date data.
	if (dateData && dateData != tDateData &&
		dateData->Destroyable() && dateData->NotifyCount() == 0)
		delete dateData;
	dateData = tDateData ? tDateData : new ZafDateData;
	if (!dateData->StringID())
		dateData->SetStringID(_dateDataName);

	// Add the data notification.
	dateData->AddNotification(this, (ZafUpdateFunction)Update);
	return (OSSetDate());
}

bool ZafDate::SetLowerCase(bool )
{
	// lowerCase is set to false by this class.
	lowerCase = false;
	return (lowerCase);
}

bool ZafDate::SetPassword(bool )
{
	// password is set to false by this class.
	password = false;
	return (password);
}

ZafError ZafDate::SetStringData(ZafStringData *)
{
	// String data cannot be set for this class.
	return (ZAF_ERROR_INVALID);
}

bool ZafDate::SetUpperCase(bool )
{
	// upperCase is set to false by this class.
	upperCase = false;
	return (upperCase);
}

ZafError ZafDate::Validate(bool processError)
{
	// Check for an absolute date error. Don't set the error member to ZAF_ERROR_NONE.
	ZafDateData currentDate;
	ZafIChar *stringDate = stringData->DynamicText();
	ZafError tError = currentDate.SetDate(stringDate, InputFormatText());

	// Check for a range error.
	if (RangeText() && tError == ZAF_ERROR_NONE)
		tError = ZAF_ERROR_OUT_OF_RANGE;
	int numRanges = 0;
	ZafDateData low, high;
	ZafIChar minDate[ZAF_STRING_DATA_LENGTH];
	ZafIChar maxDate[ZAF_STRING_DATA_LENGTH];
	ZafIChar rBuffer[1024]; // Localized range string for error messages.
	rBuffer[0] = '\0';
	for (const ZafIChar *tRange = RangeText(); tRange && tError == ZAF_ERROR_OUT_OF_RANGE; numRanges++)
	{
		tRange = ParseRange(tRange, minDate, maxDate);
		low.SetDate(minDate, OutputFormatText());
		high.SetDate(maxDate, OutputFormatText());
		if ((!minDate[0] || currentDate >= low) && (!maxDate[0] || currentDate <= high))
			tError = ZAF_ERROR_NONE;
		else if (processError)
		{
			if (rBuffer[0])
				strcat(rBuffer, rangeSeparator);
			if (minDate[0])
				low.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
			strcat(rBuffer, dateSeparator);
			if (maxDate[0])
				high.FormattedText(&rBuffer[strlen(rBuffer)], ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Process the error code.
	SetInvalid(false);
	if (tError == ZAF_ERROR_NONE)
	{
		// Set up the new date.
		currentDate.FormattedText(stringDate, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		SetText(stringDate);
		delete []stringDate;
		return (ZAF_ERROR_NONE);
	}
	else
	{
		// Keep the error code.
		error = tError;
		if (!zafErrorSystem)
		{
			// Restore the original date.
			dateData->FormattedText(stringDate, ZAF_STRING_DATA_LENGTH, OutputFormatText());
			SetText(stringDate);
			delete []stringDate;
			return (error);
		}
		else if (!processError)
		{
			delete []stringDate;
			return (error);
		}
	}

	// Check for open-ended range errors.
	if (error == ZAF_ERROR_OUT_OF_RANGE && numRanges == 1)
	{
		if (minDate[0] && !maxDate[0])
		{
			error = ZAF_ERROR_LESS_THAN_RANGE;
			low.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
		else if (!minDate[0] && maxDate[0])
		{
			error = ZAF_ERROR_GREATER_THAN_RANGE;
			high.FormattedText(rBuffer, ZAF_STRING_DATA_LENGTH, OutputFormatText());
		}
	}

	// Generate the error message and wait for a response.
	ZafIChar *title = ZafLanguageData::blankString;
	ZafIChar *errorString = errorStrings->GetMessage((ZafNumberID)error, true);
	if (errorString &&
		zafErrorSystem->ReportError(windowManager, title, ZAF_DIALOG_OK | ZAF_DIALOG_CANCEL, errorString, stringDate, rBuffer) == S_DLG_OK)
		error = ZAF_ERROR_LEAVE_INVALID; // Keep the new value.
	delete []stringDate;
	return (error);
}

bool ZafDate::SetVariableName(bool )
{
	// variableName is false for this class.
	variableName = false;
	return (variableName);
}

// ----- Persistent functions -----------------------------------------------

ZafDate::ZafDate(const ZafIChar *name, ZafObjectPersistence &persist) :
	ZafString(name, persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY)),
	dateData(ZAF_NULLP(ZafDateData))
{
	// Check the language and local information.
	LanguageAllocate();

	ZafDate::ReadData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

ZafElement *ZafDate::Read(const ZafIChar *name, ZafObjectPersistence &persist)
{
	return (new ZafDate(name, persist));
}

void ZafDate::ReadData(ZafObjectPersistence &persist)
{
	// Read the data.
	ZafIChar dateName[ZAF_MAXNAMELEN];
	ZafFile *file = persist.File();
	*file >> dateName;
	SetDateData(dateName[0] ? new ZafDateData(dateName, persist) : new ZafDateData);
}

void ZafDate::Write(ZafObjectPersistence &persist)
{
	// Write the object.
	ZafString::Write(persist.PushLevel(className, classID, ZAF_PERSIST_DIRECTORY));
	ZafDate::WriteData(persist);
	persist.PopLevel();
	if (persist.Error() != ZAF_ERROR_NONE)
		SetError(persist.Error());
}

void ZafDate::WriteData(ZafObjectPersistence &persist)
{
	// Write the data.
	const ZafIChar *dateName = dateData ? dateData->StringID() : ZAF_NULLP(ZafIChar);
	ZafFile *file = persist.File();
	*file << dateName;
	if (dateData)
		dateData->Write(persist);
}

