//	Zinc Application Framework - Z_APP.CPP
//	COPYRIGHT (C) 1990-1997.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA

#include <z_devall.hpp>
#include <z_win.hpp>
#include <z_app.hpp>
#include <z_string.hpp>
#include <z_utils.hpp>
#include <z_i18n.hpp>
#define ZAF_APPLICATION_INFO
#include "gbl_def.cpp"

// ----- ZafApplication -----------------------------------------------------

ZafApplication::ZafApplication(int _argc, char **_argv) :
	argc(_argc), argv(ZAF_NULLP(ZafIChar *))
{
	// Allocate the display.
	zafDisplay = new ZafDisplay(_argc, _argv);

	zafDataManager = new ZafDataManager;

	// Call ResetI18n() to reflect changes based on the display.
	ZafI18nData::I18nAllocate();

	// Allocate the arguments.
	if (argc)
	{
		argv = new ZafIChar *[argc];
		for (int i = 0; i < argc; i++)
			argv[i] = zafCodeSet->ConvertToZafString(_argv[i]);
	}

	// Allocate the event manager.
	zafEventManager = new ZafEventManager;
	zafEventManager->Add(new ZafKeyboard);
	zafEventManager->Add(new ZafMouse);
	zafEventManager->Add(new ZafCursor);

	// Allocate the window manager.
	zafWindowManager = new ZafWindowManager;
}

ZafApplication::~ZafApplication(void)
{
	// Free the arguments.
	if (argc)
	{
		for (int i = 0; i < argc; i++)
			delete []argv[i];
		delete []argv;
	}

	// Free the associated high-level managers.
	if (zafWindowManager)
		delete zafWindowManager;
	zafWindowManager = ZAF_NULLP(ZafWindowManager);
	if (zafEventManager)
		delete zafEventManager;
	zafEventManager = ZAF_NULLP(ZafEventManager);

	// Free static memory modules (these modules need the zafDisplay pointer).
	if (staticModule)
	{
		for (int i = 0; staticModule[i]; i++)
			(*staticModule[i])(true);
		delete []staticModule;
		staticModule = ZAF_NULLP(ZafStaticModule);
	}

	// Free the associated low-level managers.
	if (zafDisplay)
		delete zafDisplay;
	zafDisplay = ZAF_NULLP(ZafDisplay);
	delete zafDataManager;
	zafDataManager = ZAF_NULLP(ZafDataManager);
}

bool ZafApplication::AddStaticModule(ZafStaticModule module)
{
	// Make sure it is a valid request.
	if (!module)
		return (false);

	// Check the array.
	int offset = 0;
	if (staticModule)
	{
		// Find the last item.
		while (staticModule[offset])
			if (staticModule[offset] == module)
				return (true);
			else
				offset++;
		// Reallocate the array.
		if (((offset + 1) % ZAF_DYNAMIC_ARRAY_INCREMENT) == 0)
		{
			ZafStaticModule *tempArray = staticModule;
			staticModule = new ZafStaticModule[offset + ZAF_DYNAMIC_ARRAY_INCREMENT + 1];
			memcpy(staticModule, tempArray, offset * sizeof(ZafStaticModule));
			delete []tempArray;
		}
	}
	else
		staticModule = new ZafStaticModule[ZAF_DYNAMIC_ARRAY_INCREMENT];

	// Reset the array.
	staticModule[offset+1] = ZAF_NULLF(ZafStaticModule);
	staticModule[offset] = module;
	return (true);
}

ZafEventType ZafApplication::Control(void)
{
	// Make sure there is a window attached to the window manager.
	if (!zafWindowManager->First())
		return (S_NO_OBJECT);

	// Wait for user response.
	ZafEventStruct event;
	ZafEventType ccode;
	do
	{
		zafEventManager->Get(event, Q_NORMAL);
		ccode = zafWindowManager->Event(event);
	} while (ccode != S_EXIT && ccode != S_NO_OBJECT);

	// Return the final code.
	return (ccode);
}

bool ZafApplication::SubtractStaticModule(ZafStaticModule module)
{
	// Make sure it is a valid request.
	if (!staticModule || !module)
		return (false);

	// Find the matching module.
	int offset = -1, count = 0;
	while (staticModule[count])
	{
		if (staticModule[count] == module)
			offset = count;
		count++;
	}

	// Reset the array.
	if (offset != -1)
	{
		memcpy(&staticModule[offset], &staticModule[offset+1], (count - offset) * sizeof(ZafStaticModule));
		return (true);
	}
	return (false);
}

