// MyEdit.cpp : implementation file
//
// Source module for Boot Options version 1.0
// Copyright 1996 Ziff Davis Publishing
// First published in PC Magazine September 24, 1996
// Written by Gregory A. Wolking.
//
//   This creates the CFilterEdit control class. It is identical to the
// CEdit control class except that the control can filter out one or more
// characters specified by the user, whether they are typed at the keyboard
// or pasted into the field from the clipboard.
//   By default, the control does not perform any filtering. To
// activate filtering, use the SetFilterText() member function. To cancel filtering,
// call SetFilterText() using either an empty string or NULL as the first parameter.
//   Since this control handles its own EN_UPDATE message, that message
// will not be reflected back to the parent window. If your main window handles
// the control's EN_CHANGE message, note that the control's window text has already
// been filtered before EN_CHANGE is generated.
//
// CFilterEdit adds the following public member functions:
//		void SetFilterText(LPCTSTR text, BOOL apply = TRUE, BOOL method = TRUE)
//		void SetFilterText(int filter, BOOL apply = TRUE, BOOL method = TRUE)
//		BOOL GetFilterText(CString& s)
// See function definitions for parameters and return values.
//
//   Note that the backspace _key_ cannot be excluded but the backspace
// _character_ can. If backspace character is excluded, user cannot paste a 
// backspace character from the clipboard but can still use the backspace key to
// edit the field.

#include "stdafx.h"
#include "MyEdit.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CFilterEdit

CFilterEdit::CFilterEdit()
{
	m_pchFilter = "";	// Initialize filter string to empty.
	ignore = FALSE;		// Clear recursion flag.
}

CFilterEdit::~CFilterEdit()
{
}

BEGIN_MESSAGE_MAP(CFilterEdit, CEdit)
	//{{AFX_MSG_MAP(CFilterEdit)
	ON_WM_CHAR()
	ON_CONTROL_REFLECT(EN_UPDATE, OnUpdate)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CFilterEdit message handlers

// Filter characters typed at the keyboard.
void CFilterEdit::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (nChar != 8)		// If character was not a backspace
	{
		if (exclude ?							// Based on exclude flag,
			(m_pchFilter.Find(nChar) >= 0) :	// If char is in filter
			(m_pchFilter.Find(nChar) < 0))		// or not in filter,
				return;							// Ignore it.
	}
	// Since we've already filtered the character,
	// there's no need for OnUpdate to do further processing.
	ignore = TRUE;
	CEdit::OnChar(nChar, nRepCnt, nFlags);	// Call base class handler.
}

// Filter text pasted from the clipboard.
void CFilterEdit::OnUpdate() 
{
	if (ignore)				// Check for recursion.
	{
		ignore = FALSE;		// Reset recursion flag
		return;				// and quit.
	}
	if (m_pchFilter.GetLength())	// If filter string is not empty,
	{
		int start, end, pos, adjust = 0;
		CString a, b;

		GetWindowText(a);			// Get current contents.
		if (exclude)				// Apply filter based on exclude flag.
		{
			// Exclude characters in filter string.
			while ((pos = a.FindOneOf(m_pchFilter)) >= 0)
			{
				++adjust;						// Increment caret adjustment.
				b = a.Left(pos);				// Get all chars prior to filter char.
				if (pos < a.GetLength() -1)		// Any chars after filter char?
					b += a.Mid(pos + 1);		// tack them on if so.
				a = b;							// Update and continue searching.
			}

		}
		else
		{
			// Exclude characters not in filter string.
			int l = a.GetLength();				// Get length of text.
			for (pos = 0; pos < l; pos++)		// For each character,
			{
				if (m_pchFilter.Find(a.GetAt(pos)) >= 0)// Is it in the filter string?
					b += a.GetAt(pos);					// If so, add it to the result
				else
					++ adjust;			// Increment adjustment to caret position.
			}
		}
		if (adjust)							// If text actually changed,
		{
			ignore = TRUE;					// set flag to prevent recursion.
			GetSel(start, end);				// Get caret position.
			start -= adjust;				// Adjust it for characters removed.
			SetWindowText(exclude ? a : b);	// Install modified text.
			SetSel(start, start);			// Restore caret position.
		}
	}
}

///////////////////////
// Function sets the filter string and method.
// Parameters:
//    LPCTSTR text (required):	case-sensitive string of filter characters.
//								Duplicate characters will be stripped.
//    BOOL apply (optional):	TRUE = apply filter to control text immediately.
//								FALSE = apply filter only if text changes.
//								If omitted, default is TRUE.
//    BOOL method (optional):	TRUE = exclude characters in filter string.
//								FALSE = exclude characters _not_ in filter string.
//								If omitted, default is TRUE.
//////////////////////
void CFilterEdit::SetFilterText(LPCTSTR text, BOOL apply, BOOL method)
{
	int pos, i, len;

	exclude = method;			// Transfer exclusion flag to member variable.
	m_pchFilter.Empty();		// Zap filter string.
	if (text == NULL)			// Was pointer NULL?
		return;					// Done if so.
	len = strlen(text);
	pos = 0;
	while (text[pos])			// Strip any duplicated characters
	{
		for (i = pos + 1; i < len; i++)
		{
			if (text[i] == text[pos])
				break;
		}
		if (i == len)
			m_pchFilter += text[pos];
		++pos;
	}
	if (apply && !m_pchFilter.IsEmpty())
		OnUpdate();
}

////////////////////////////////
// This version sets a predefined filter.
// Works the same as the above version except that
// the first parameter is an int (instead of an LPCTSTR) that can
// be one or more of the following constants ORed together:
//  FILTER_UC_ALPHA = uppercase letters (A-Z)
//  FILTER_LC_ALPHA = lowercase letters (a-z)
//  FILTER_DIGITS = digits (0 - 9)
//  FILTER_SEPARATORS = space, tab, comma, semicolon, equal sign, slash, backslash.
//  FILTER_PARENS = left and right parentheses.
//  FILTER_BRACKETS = left and right square brackets
//  FILTER_BRACES = left and right curly braces
//  FILTER_MATH = *, +, -, ^
//	FILTER_QUOTE = quotation mark (")
// (These identifiers are enumerated in MyEdit.h)
////////////////////////////////
void CFilterEdit::SetFilterText(int filter, BOOL apply, BOOL method)
{
	CString s;

	if (filter & FILTER_UC_ALPHA)
		s = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	if (filter & FILTER_LC_ALPHA)
		s += "abcdefghijklmnopqrstuvwxyz";
	if (filter & FILTER_DIGITS)
		s += "0123456789";
	if (filter & FILTER_SEPARATORS)
		s += " \t,;=/\\";
	if (filter & FILTER_PARENS)
		s += "()";
	if (filter & FILTER_BRACKETS)
		s += "[]";
	if (filter & FILTER_BRACES)
		s += "{}";
	if (filter & FILTER_MATH)
		s += "*+-^";
	if (filter & FILTER_QUOTE)
		s += '\"';
	SetFilterText(s, apply, method);
}

//////////////////
// Retrieve filter state.
// Parameter: s = reference to a CString object which will receive a copy
//				  of the filter string.
// Return value: Filter method (TRUE or FALSE)
//////////////////
BOOL CFilterEdit::GetFilterText(CString& s)
{
	s =  m_pchFilter;
	return exclude;
}
