// $Header: /MeShare/Src/FC.S 41    1/08/96 23:06 Dan $
macro_file FC;

#INCLUDE metools.sh

#IFDEF windows
	#INCLUDE windows.sh
	#INCLUDE dialog.sh
	#INCLUDE colors.sh
	#INCLUDE tbwin.sh
	#INCLUDE mew.sh
	#INCLUDE mewhelp.sh
	#INCLUDE mewlib.sh
	#include KEYS.SH
	#include VCS.SH
	#include WINDOW.SH

	#ifdef _Debug_
		#include DBUG.SH
		#include SPRINTF.SH
	#endif

	#define FILECOMP_HELPLINK "COMPARE"
	#define FILECOMPSET_HELPLINK "FILE COMPARE SETTINGS"
#ENDIF


/***************************************************************************
NAME: FC

DESCRIPTION:
 FC() performs a file compare on two files.  To use it, load up a file,
 and the run FC.  You will be prompted for a direction to split the
 window, and then you will be asked to select a file to compare the first
 one to.  The two files will be compared and then you are taken into
 "Synchronized compare mode".  The current line in both files will be
 synchronized, even across differences.  The differences in the files will
 be highlighted.  You can even edit the files, and FC will do its best to
 keep everything syncronized.  This macro makes use of the event_macro
 window variable to intercept all keystrokes going to the compared
 windows.


NOTES:
 Most file compares are oriented towards the concept of an "original" file
 and a "changed" file.  They then generate there output based on insertions
 and deletions in the "original file".  This macro does not make any
 assumptions about "original" or "changed" files.  Instead, it simply locates
 the differences, and records them for both files.


INTERFACE
	 run FC for the actual file compare
	 run FC_CONFIG for the file compare configuration.
	 run FC_REPORT to generate a report on the compare

	 All other functions are internal to fc, and should NOT be executed by the
	 user.


FUNCTIONS:
	 fc_match_pos     keeps the two windows synchronized during sync mode
	 fc_next_dif      takes the cursor to the next/previous difference
	 fc_report				generates readable reports on the differences
	 fc_reset					gets rid of the compare color coding
	 fc_config				configuration dialog
	 fc_config_x			used by fc_config
	 fc_eproc         process all keystrokes to the compare windows
	 fc_update				updates the compare data after lines are inserted or
										deleted
	 fc               the main macro function

AUTHOR:   Todd Johnson

HISTORY:  03-18-93 01:52pm  Original release
					04-08-93 01:39am  Re-wrote fc_resynch to provide for more accurate,
														faster resynching.  Added fc_config to allow
														for easy configuration
					04-19-93 03:58pm  Revamped FC to take advantage of the event_macro
														window variable.  FC is now "modeless".  The
														file compare  is run, and then control is
														returned to the user.  You can now edit the
														files just like any other, and fc will try to
														keep the files synced.


						(C)Copyright 1993 by American Cybnernetics, Inc.
***************************************************************************/

prototype mesys
{
	void cursor_to_line( int l );
}

global
{
	str fc_config_str;
	int fc_wcmd_next "~fc_wcmd_next";
	int fc_wcmd_prev "~fc_wcmd_prev";
	int fc_wcmd_help "~fc_wcmd_help";
	int fc_WCmd_CDif "~fc_WCmd_CDif";
	int g_FcSelWin	 "!FcSelWin";
}

#define FC_EVENT_MODE EVENT_MODE_ENHANCED | EVENT_WCMD | EVENT_DELETE_WIN | EVENT_MOUSE_VSCROLL | EVENT_MOUSE_HSCROLL


// Return 1 if file compare is not enabled for the current window
int Fc_Enabled( ) {
	if ( Caps( Copy( Event_Macro, 1, 8 ) ) == "FC_EPROC") {
		return ( 0 );
	}
	return ( 1 );

}  // Fc_Enabled


void fc_match_pos( int w1, w2, tw, &l1, &l2 )
/*********************************************************************
	This routine is used to keep the cursors in synch after the actual
	file compare has been done.
 *********************************************************************/
{
	int jx, tundo = undo_stat,m;
	int TRefresh = Refresh;
	int tpersistent_blocks = persistent_blocks;
	persistent_blocks = true;

	str ss1[4], ss2[4], cc1[4], cc2[4];

	#IFNDEF windows
	set_virtual_display;
	#ENDIF

	undo_stat = false;

	Refresh = False;
	if(window_id == w2 )
	{
		m = 1;
		jx = w2;
		w2 = w1;
		w1 = jx;
		ss1 = "/S2=";
		ss2 = "/S1=";
		cc1 = "/C2=";
		cc2 = "/C1=";
	}
	else
	{
		m = 0;
		ss2 = "/S2=";
		ss1 = "/S1=";
		cc2 = "/C2=";
		cc1 = "/C1=";
		switch_win_id(w1);
	}

	int tl = c_line, nl = 0, s1,s2, c1,c2, old_c_row = c_row, new_c_row = -1;
	int TCol = C_Col;
	int TLeftOffset = Left_Offset;
// 	refresh = false;

	switch_win_id( tw );
	tof;
	while( !at_eof )
	{
		if(parse_int(cc1, get_line) > tl )
		{
			if( c_line == 1 )
			{
				nl = tl;
			}
			else
			{
				up;
				s1 = parse_int(ss1, get_line );
				s2 = parse_int(ss2, get_line );
				nl = s2 + (tl - s1);
			}
			new_c_row = old_c_row;
			goto done;
		}
		down;
		if( parse_int(ss1, get_line) > tl )
		{
			up;
			nl = parse_int(cc2, get_line);
			new_c_row = old_c_row;
			goto done;
		}
		down;
	}
	up;
	if(!at_eof) {
		s1 = parse_int(ss1, get_line );
		s2 = parse_int(ss2, get_line );
		nl = s2 + (tl - s1);
	}
	else
		nl = tl;
	new_c_row = old_c_row;
done:
	if ( nl < 1 )
	{
		nl=1;
	}
	if( m )
	{
		l2 = c_line;
	}
	else
		l1 = c_line;
	switch_win_id( w2 );
	if( new_c_row != -1 )
	{
		if((new_c_row != c_row) || (nl != c_line))
		{
			if( new_c_row > win_cheight)
			{
				new_c_row = win_cheight;
			}
			int ttc = c_row;

			while(c_row < new_c_row )
				down;
			while(c_row > new_c_row )
				up;

			int ttl = c_line;
			goto_line(nl);



			if((ttc == new_c_row) || (ttl != c_line ))
				update_window;
		}
	//	update_window;
	}
	else
		cursor_to_line( nl );
	// refresh = true;
	// update_window;
	if( m )
	{
		l1 = c_line;
	}
	else
		l2 = c_line;
// 	refresh = true;
	if(!xpos("-Cmpr", window_name, 1 ))
	{
		window_name += "-Cmpr";
		update_window;
	}
	if(left_offset != TLeftOffset)
	{
		Left_Offset = TLeftOffset;
		update_window;
	}
	Goto_Col( TCol );
// 	refresh = false;
	switch_win_id( w1 );
	if(!xpos("-Cmpr", window_name, 1 ))
	{
		window_name += "-Cmpr";
// 		refresh = true;
		update_window;
	}
	if(left_offset != TLeftOffset)
	{
		Left_Offset = TLeftOffset;
	}
	Goto_Col( TCol );
// 	refresh = false;
	persistent_blocks = tpersistent_blocks;
	Refresh = TRefresh;
	undo_stat = tundo;
	#IFNDEF windows
	update_virtual_display;
	reset_virtual_display;
	#ENDIF
}


/****************************************************************
 Takes the cursor to the next or previous difference
		d == 0, goto next difference
		d == 1, goto previous difference
		w1      the first compare window id
		w2      the second compare window id
		tw      the temporary window id
 ****************************************************************/
int fc_next_dif(int d, w1, w2, tw )
{
	int jx, tundo = undo_stat,m,  cl = c_line, jl,
			result = true;
	int TRefresh = Refresh;

	str cc1[4], cc2[4];
	undo_stat = false;

	#IFNDEF windows
	set_virtual_display;
	#ENDIF

	Refresh = False;
	if(window_id == w2 )
	{
		jx = w2;
		w2 = w1;
		w1 = jx;
		cc1 = "/C2=";
		cc2 = "/C1=";
	}
	else
	{
		cc2 = "/C2=";
		cc1 = "/C1=";
		switch_win_id(w1);
	}
	int s1,s2;
// 	refresh = false;
	reg_exp_stat= true;
	switch_win_id( tw );
	tof;
	while( !at_eof && (jl = parse_int( cc1, get_line ) <= cl ))
	{
		down;
		down;
	}
	if(d)
	{
		do
		{
			up; eol;
			jl = search_bwd("/C",0);
		}
		while( (jl !=0) && (c_line > 1) && (parse_int( cc1, get_line ) >= cl ));
		if( !jl )
		{
			goto done;
		}
	};
	if( !at_eof )
	{
		s1 = parse_int(cc1, get_line);
		s2 = parse_int(cc2, get_line);
		switch_win_id( w2 );
		jx = (win_y2 - win_y1) / 2;
		if( jx < 1 )
			jx = 1;
		while(c_row > jx)
			up;
		while(c_row < jx)
			down;
		goto_line( s2 );
		//redraw;
		update_window;
		switch_win_id( w1 );
// 		refresh = true;
		jx = (win_y2 - win_y1) / 2;
		if( jx < 1 )
			jx = 1;
		while(c_row > jx)
			up;
		while(c_row < jx)
			down;
		goto_line( s1 );
		update_window;
	}
	else
	{
		result = false;
		switch_win_id(w1);
	}

done:
	undo_stat = tundo;
	#IFNDEF windows
	update_virtual_display;
	reset_virtual_display;
	#ENDIF
	Refresh = TRefresh;
	return(result);
}


void Fc_Reset( int W ) {
/******************** Multi-Edit VOID Macro Function ************************

 DESCRIPTION:  resets all of the line color attributes in a file

 PARAMETERS:   w  the window id to reset
*****************************04-08-93 01:46am*******************************/

	int SavRefresh = Refresh;
	int ActiveWin	 = Cur_Window;

	str GStr = "!FcEventSav_";

	Refresh = False;
	if ( Switch_Win_Id( W ) ) {
		Event_Macro = "";
		Mark_Pos;
		Tof;
		while( !At_Eof ) {
			Line_Attr = 0;
			Down;
		}
		Goto_Mark;
		Rm( "Set_Syntax_Parms" );
		// Restore previous Event_Macro and Event_Mode
		GStr += Str( Window_Id );
		Event_Macro = Global_Str( GStr );
		Event_Mode = Global_Int( GStr );
		Set_Global_Str( GStr, "" );
		Set_Global_Int( GStr, 0 );
	}
	Switch_Window( ActiveWin );
	Refresh = SavRefresh;

}  // Fc_Reset

void Fc_Cleanup( int Wid1, int Wid2, int TWid ) {

//  if ( Parse_Int( "/TBH=", Global_Str( "!FC-" + Hex_Str( TWid ) ) ) ) {
//    DestroyWindow(parse_int("/TBH=", global_str("!FC-" + hex_str(twid) )));
//  }
	Set_Global_Str( "!FC-" + Hex_Str( TWid ), "" );

  if ( Switch_Win_Id( TWid ) ) {
		Delete_Window;
	}
	Fc_Reset( Wid1  );
	Fc_Reset( Wid2 );
	Make_Message( "" );

}  // Fc_Cleanup

void fc_report( ) {
/******************************************************************
 This function generats a report on the file compare.
 It has two modes:
		 report_mode 0    Make a short report, showing only line #s
		 report_mode 1    Make a detailed report, showing the actual text
 *****************************************************************/

	int report_win, ln = 0, cc1, cc2, ss1, ss2, jx;

	int w1 = parse_int("/W1=", event_macro),
			w2 = parse_int("/W2=", event_macro),
			tw = parse_int("/TW=", event_macro),
			wn1,wn2,twn,
			report_mode = 0,
			twin = window_id;

	refresh = false;

	if( !switch_win_id( tw ) )
	{
		rm("MEError^MessageBox /T=ERROR/B=1/M=No valid file compare information available.");
		return();
	}
	twn = cur_window;

	RM("Xmenu /L=Select report type/B=1/T=1/M=Summary report(FC)Detailed report(FC)");
	if( return_int < 1 )
	{
		switch_win_id(twin);
		return();
	}

	report_mode = return_int - 1;

	rm("CreateWindow");
	rm("SetWindowNames");
	file_name = "FCREPORT.TXT";
	report_win = cur_window;
	switch_win_id( w1 );
	wn1 = cur_window;
	put_line_to_win( "File compare: " + date + " " + time, ++ln, report_win, false );
	put_line_to_win( "File 1: " + file_name, ++ln, report_win, false );
	switch_win_id( w2 );
	wn2 = cur_window;
	put_line_to_win( "File 2: " + file_name, ++ln, report_win, false );
	++ln;

	switch_win_id( tw );
	tof;
	while( !at_eof )
	{
		cc1 = parse_int( "/C1=", get_line );
		cc2 = parse_int( "/C2=", get_line );
		down;
		ss1 = parse_int( "/S1=", get_line ) - 1;
		ss2 = parse_int( "/S2=", get_line ) - 1;
		down;
		put_line_to_win("Difference: File 1, lines " + str(cc1) + " to " + str(ss1), ++ln, report_win, false );
		put_line_to_win("            File 2, lines " + str(cc2) + " to " + str(ss2), ++ln, report_win, false );
		if( report_mode )
		{
			++ln;
			put_line_to_win("  File 1 detail:", ++ln, report_win, false );
			for( jx = cc1; jx <= ss1; jx++ )
			{
				put_line_to_win("   " + str( jx ) + ": " + get_line_from_win( jx, wn1 ), ++ln, report_win, false );
			}
			++ln;
			put_line_to_win("  File 2 detail:", ++ln, report_win, false );
			for( jx = cc2; jx <= ss2; jx++ )
			{
				put_line_to_win("   " + str( jx ) + ": " + get_line_from_win( jx, wn2 ), ++ln, report_win, false );
			}
			++ln; ++ln;
		}
	}
	switch_window( report_win );
	return();
}

void FC_Set_Keynames( ) {
/******************** Multi-Edit VOID Macro Function ************************

 NAME:         FC_Set_Keynames()

 DESCRIPTION:  Sets the keyname message str.

*****************************07-29-93 10:36pm*******************************/

	return ( );
	str key_str[80],tstr;
	// Find current key assignments
	int tmp_int = parse_int("/NKEY=", fc_config_str);
	rm('setup^makekey /K1='+str(tmp_int%256)+'/K2='+str(tmp_int>>8));
	key_str = 'NextDif'  + return_str;
	tmp_int = parse_int("/PKEY=", fc_config_str);
	rm('setup^makekey /K1='+str(tmp_int%256)+'/K2='+str(tmp_int>>8));
	key_str += ' PrevDif' + return_str;
	tmp_int = parse_int("/RKEY=", fc_config_str);
	rm('setup^makekey /K1='+str(tmp_int%256)+'/K2='+str(tmp_int>>8));
	key_str += ' Report'+return_str;
	tstr = fc_config_str;
	var_remove_str('/KEYSTR=',tstr);
	fc_config_str = tstr + '/KEYSTR='+key_str;

}

#ifdef Windows

int fc_config( int set_defaults = parse_int("/D=", mparm_str) )	trans
/****************** Multi-Edit INTEGER Macro Function ***********************

 NAME:         fc_config()

 DESCRIPTION:  configuration dialog for fc

 PARAMETERS:

 RETURNS:      TRUE if accepted, FALSE if ESC was hit.

*****************************04-09-93 01:18am*******************************/
{
	str tstr = "";

	int menu = menu_create,
			tr = refresh,
			change_color,
			result = 1,
			insert_color;

	refresh = false;
	if( fc_config_str == "" )
	{
		fc_config_str = "/SH=1";
	}

	if ( set_defaults )
	{
		return(1);
	}
	tstr = fc_config_str;

	int dlg;
	DlgCreate(dlg);

	DlgAddCtrl( dlg, DLG_BitmapStatic, "BT_FL_114",
							1, 1,0,0, 1000, 0, "" );


	DlgAddCtrl( dlg, DLG_GroupBox, "",
							7,1,42,6, 1002, 0, "" );

	DlgAddCtrl( dlg, DLG_CheckBox, "&Syntax highlighting during file compare",
							DLG_PosOffset + 1, DLG_PosOffset + 1,0,0, 2001, 0, "" );
	DlgSetInt( dlg, 2001, Parse_Int("/SH=", tstr ));
	DlgAddCtrl( dlg, DLG_CheckBox, "Ignore &case",
							DLG_PosOffset, DLG_PosOffset + 1,0,0, 2002, 0, "" );
	DlgSetInt( dlg, 2002, Parse_Int("/IGC=", tstr ));
	DlgAddCtrl( dlg, DLG_CheckBox, "Ignore &indent",
							DLG_PosOffset, DLG_PosOffset + 1,0,0, 2003, 0, "" );
	DlgSetInt( dlg, 2003, Parse_Int("/IGI=", tstr ));
	DlgAddCtrl( dlg, DLG_CheckBox, "Ignore &blank lines",
							DLG_PosOffset, DLG_PosOffset + 1,0,0, 2004, 0, "" );
	DlgSetInt( dlg, 2004, Parse_Int("/IGB=", tstr ));

	DlgAddCtrl( Dlg, dlg_CheckBox, "&No window list",
			dlg_PosOffset, dlg_PosOffset + 1,
			0, 0,
			2005, 0, "" );
	DlgSetInt( Dlg, 2005, Parse_Int( "/NWL=", TStr ) );

	DlgAddCtrl( Dlg, dlg_Static, "&Window split",
			dlg_PosOffset, dlg_PosOffset + 2,
			0, 0,
			100, 0, "" );
	DlgAddCtrl( Dlg, dlg_Choice, "Prompt()Right()Left()Up()Down()Full Screen()",
			dlg_PosOffset + 14, dlg_PosOffset,
			15, 0,
			2006, 0, "" );
	DlgSetInt( Dlg, 2006, Parse_Int( "/WS=", TStr ) );

	DlgAddCtrl( dlg, DLG_PushButton, "OK",
							1, DLG_PosOffset + 2,DLG_StanBtnWidth,0, 101, DLGF_DefButton, "/R=1" );
	DlgAddCtrl( dlg, DLG_PushButton, "Cancel",
							DLG_PosOffset + Dlg_StanBtnWidth + 3, DLG_PosOffset,DLG_StanBtnWidth,0, 102, 0, "/R=0" );

	DlgAddCtrl( dlg, DLG_PushButton, "&Help",
							40, DLG_PosOffset,DLG_StanBtnWidth,0, 103, 0, "/R=2" );

	result = DlgExecute( dlg, 2001, "File Compare Setup",
						FILECOMPSET_HELPLINK, "", 0 );

	if(result)
	{
		fc_config_str =
			"/SH=" + str(DlgGetInt( dlg, 2001)) +
			"/IGC=" + str(DlgGetInt( dlg, 2002)) +
			"/IGI=" + str(DlgGetInt( dlg, 2003)) +
			"/IGB=" + str(DlgGetInt( dlg, 2004)) +
			"/NWL=" + Str( DlgGetInt( Dlg, 2005 ) ) +
			"/WS=" + Str( DlgGetInt( Dlg, 2006 ) );
		Set_Global_Int('SETUP_CHANGED',Global_Int('SETUP_CHANGED') | $01);
	}
	refresh = tr;

	dlgkill(dlg);
	return(result);
}

#ELSE
int fc_config( int set_defaults = parse_int("/D=", mparm_str) ) trans
{

	str tstr = "";

	int menu = menu_create,
			tr = refresh,
			change_color,
			insert_color;

	refresh = false;
	if( fc_config_str == "" )
	{
		fc_config_str = "/SH=0/CC=79/IC=48/RW=37120/NKEY=37120/PKEY=36096/RKEY=5118";
	}

	if ( set_defaults )
	{
		return(1);
	}
	tstr = fc_config_str;

	var_parse_int("/CC=", tstr, change_color );
	var_parse_int("/IC=", tstr, insert_color );

	menu_set_item( menu, 1, "Syntax highlighting during interactive compare", "", "/L=1/C=1/QK=1", 13,
															parse_int("/SH=", tstr), 0 );
	menu_set_item( menu, 2, "Next difference key:    ", "", "/QK=1/L=3/W=20/C=1", 9,parse_int("/NKEY=", tstr), 0 );
	menu_set_item( menu, 3, "Previous difference key:", "", "/QK=1/W=20/L=4/C=1", 9,parse_int("/PKEY=", tstr), 0 );
	menu_set_item( menu, 4, "Report key:             ", "", "/QK=1/W=20/L=5/C=1", 9,parse_int("/RKEY=", tstr), 0 );

	int color_item = 8;
	menu_set_item( menu, 5, "Colors:", "", "/L=7/C=1/QK=1", 10,0, 0 );
	menu_set_item( menu, 6, " Unchanged line  ", "", "/L=8/C=2/ATTR=" + str(w_t_color), 10,0, 0 );
	menu_set_item( menu, 7, "                 ", "", "/L=9/C=2/ATTR=" + str(w_t_color), 10,0, 0 );
	menu_set_item( menu, 8, " Lines different ", "", "/L=10/C=2/ATTR=" + str(change_color), 10,0, 0 );
	menu_set_item( menu, 9, "                 ", "", "/L=11/C=2/ATTR=" + str(w_t_color), 10,0, 0 );
	menu_set_item( menu,10, " Lines inserted  ", "", "/L=12/C=2/ATTR=" + str(insert_color), 10,0, 0 );
	menu_set_item( menu,11, "                 ", "", "/L=13/C=2/ATTR=" + str(w_t_color), 10,0, 0 );

	menu_set_item( menu, 12, "  1 ", "fc_config_x /#=" + str(color_item) + "/M=" + str(menu), "/QK=4/L=10/C=20/M=1/R=4", 11, 0, 0 );
	menu_set_item( menu, 13, "  2 ", "fc_config_x /#=" + str(color_item + 2) + "/M=" + str(menu), "/QK=4/L=12/C=20/M=1/R=5", 11, 0, 0 );

	return_int = menu;
	rm("USERIN^DATA_IN /T=File Compare Configuration/#=13/HN=1/H=COMPARE" );

	if(return_int)
	{
		fc_config_str = "/SH=" + str(menu_item_int( menu, 1, 2 )) +
										"/CC=" + parse_str("/ATTR=", menu_item_str( menu, color_item,3)) +
										"/IC=" + parse_str("/ATTR=", menu_item_str( menu, color_item + 2,3)) +
										"/RW=" + str(menu_item_int( menu, 2, 2 )) +
										"/NKEY=" + str(menu_item_int( menu, 2, 2 )) +
										"/PKEY=" + str(menu_item_int( menu, 3, 2 )) +
										"/RKEY=" + str(menu_item_int( menu, 4, 2 ))
										;
		fc_set_keynames;
	}

	menu_delete(menu);
	refresh = tr;
	return(return_int);
}

void fc_config_x()
	trans2
{
	int menu = parse_int( "/M=", mparm_str);
	int mn = parse_int("/#=", mparm_str);
	int attr;

	str tstr;

	tstr = menu_item_str( menu,  mn, 3 );
	attr = parse_int("/ATTR=", tstr );
	var_remove_str( "/ATTR=", tstr );

	return_int = attr;
	rm( "SETUP^COLORCHART");

	if(return_int >= 0)
		attr = return_int;

	tstr += "/ATTR=" + str(attr );
	menu_set_str( menu, mn, 3, tstr );
	return_int = 0;
}
#ENDIF

void Fc_CDifSrch( int Cmd = Parse_Int( "/C=", MParm_Str ) ) {
/******************************************************************************
															 Multi-Edit Macro
															 27-Mar-95  14:29

	Function: Search for changed lines in the Composit difference file.

	Entry   : int Cmd
									0 - Search fwd for Added lines.
									1 - Search bwd for Added lines.
									2 - Search fwd for Deleted lines.
									3 - Search bwd for Deleted lines.
									4 - Search fwd for any changed lines.
									5 - Search bwd for any changed lines.

							 Copyright (C) 1995 by American Cybernectics, Inc.
********************************************************************( ldh )***/

	int TRefresh = Refresh;
	int Color;
	int Found = False;
	int Cur_LA = Line_Attr;

	Refresh = False;
	if ( !XPos( "-CDif", Window_Name, 1 ) ) {
		Window_Name += "-CDif";
	}
	Working;
	Mark_Pos;
	switch ( Cmd ) {
		case 0:															// Forward to next added line
			while ( ( Line_Attr == Cur_LA ) && !At_Eof ) {
				Down;
			}
			while ( ( Line_Attr != fc_Inserted_Attr ) && !At_Eof ) {
				Down;
			}
			if ( Line_Attr == fc_Inserted_Attr) {
				Found = True;
			}
			break;

		case 1:                             // Backward to previous added line
			while ( ( Line_Attr == Cur_LA ) && ( C_Line != 1 ) ) {
				Up;
			}
			while ( ( Line_Attr != fc_Inserted_Attr ) && ( C_Line != 1 ) ) {
				Up;
			}
			while ( ( Line_Attr == fc_Inserted_Attr ) && ( C_Line != 1 ) ) {
				Up;
			}
			if ( Line_Attr != fc_Inserted_Attr ) {
				Down;
			}
			if ( Line_Attr == fc_Inserted_Attr ) {
				Found = True;
			}
			break;

		case 2:                             // Forward to next deleted line
			while ( ( Line_Attr == Cur_LA ) && !At_Eof ) {
				Down;
			}
			while ( ( Line_Attr != fc_Different_Attr ) && !At_Eof ) {
				Down;
			}
			if ( Line_Attr == fc_Different_Attr ) {
				Found = True;
			}
			break;

		case 3:                             // Backward to previous deleted line
			while ( ( Line_Attr == Cur_LA ) && ( C_Line != 1 ) ) {
				Up;
			}
			while ( ( Line_Attr != fc_Different_Attr ) && ( C_Line != 1 ) ) {
				Up;
			}
			while ( ( Line_Attr == fc_Different_Attr ) && ( C_Line != 1 ) ) {
				Up;
			}
			if ( Line_Attr != fc_Different_Attr ) {
				Down;
			}
			if ( Line_Attr == fc_Different_Attr ) {
				Found = True;
			}
			break;

		case 4:															// Forward to next difference
			while ( ( Line_Attr == Cur_LA ) && !At_Eof ) {
				Down;
			}
			while ( ( Line_Attr == 0 ) && !At_Eof ) {
				Down;
			}
			if ( ( Line_Attr == fc_Inserted_Attr ) ||
					( Line_Attr == fc_Different_Attr ) ) {
				Found = True;
			}
			break;

		case 5:															// Backward to previous difference
			while ( ( Line_Attr == Cur_LA ) && ( C_Line != 1 ) ) {
				Up;
			}
			while ( ( Line_Attr == 0 ) && ( C_Line != 1 ) ) {
				Up;
			}
			Cur_LA = Line_Attr;
			while ( ( Line_Attr == Cur_La) && ( C_Line != 1 ) ) {
				Up;
			}
			if ( Line_Attr != Cur_LA ) {
				Down;
			}
			if ( ( Line_Attr == fc_Inserted_Attr ) ||
					( Line_Attr == fc_Different_Attr ) ) {
				Found = True;
			}
			break;
	}
	if ( !Found ) {
		Goto_Mark;
	}
	else {
		Pop_Mark;
		if ( C_Line != 1 ) {

			int Tmp;
			int Value;

			Tmp = C_Line;
			Value = ( ( Win_Y2 - Win_Y1 ) / 2 );
			while ( C_Row > Value ) {
				Up;
			}
			while ( C_Row < Value ) {
				Down;
			}
			Goto_Line( Tmp );
		}
	}
	Redraw;
	Refresh = TRefresh;

} // Fc_CDifSrch

int Fc_CDiff( int DifWin, int Win1, int Win2, int UseWin2 ) {
/******************************************************************************
															 Multi-Edit Macro
															 27-Mar-95  14:28

	Function:
	Entry   :
	Exit    :
	Globals :
	Macros  :

							 Copyright (C) 1995 by American Cybernectics, Inc.
********************************************************************( ldh )***/

	int OrgWin = Window_Id;
	int TRefresh = Refresh;
	int TInsert_Mode = Insert_Mode;
	int TUndo_Stat = Undo_Stat;
	int TSave_Resets_Changed = Save_Resets_Changed;
	int C1;
	int C2;
	int S1;
	int S2;
	int DifWinNo;
	int Win1No;
	int LstWin = 0;
	int Cnt = 0;
	int DifLines;
	int LnOffset = 0;
	int Jx;

	str Line;
	str FName;

	Refresh = False;
	Insert_Mode = True;
	Undo_Stat = False;
	vcsSaveWin = Window_Id;               // Save window_id for VcsShowWin macro
	Switch_Win_Id( DifWin );
	DifWinNo = Cur_Window;
	Eof;
	if ( C_Line == 1 ) {
		Switch_Win_Id( OrgWin );
		goto Exit;
	}
	DifLines = C_Line + 1;

	Make_Message( "Creating composite difference file..." );
	Switch_Win_Id( Win2 );
	FName = TempFileName( Truncate_Path( File_Name ) );
	if ( UseWin2 ) {
		File_Name = FName;
		LstWin = Window_Id;
		Read_Only = False;
		Fc_Reset( Win2 );
		Fc_Reset( Win1 );
	}
	else {
		Mark_Pos;
		Tof;
		Block_Begin;
		Eof;
		Block_End;
		Win1No = Cur_Window;

		if ( Switch_File( FName ) ) {
			LstWin = Window_Id;
			Switch_Win_Id( Win2 );
			Block_Off;
			Goto_Mark;
			goto Exit1;
		}
		Switch_Window( Window_Count );			// Create list window
		Create_Window;
		Window_Copy( Win1No );
		Block_Off;
		File_Name = FName;
		Save_Resets_Changed = True;
		Save_File;
		Save_Resets_Changed = TSave_Resets_Changed;
		Del_File( File_Name );
		LstWin = Window_Id;
		Read_Only = False;

		Switch_Win_Id( Win2 );
		Block_Off;
		Goto_Mark;
	}
	Switch_Win_Id( Win1 );
	Win1No = Cur_Window;

	Switch_Win_Id( LstWin );
	while ( ++Cnt < DifLines ) {
		Line = Get_Line_From_Win( Cnt, DifWinNo );
		C1 = Parse_Int( "/C1=", Line );
		C2 = Parse_Int( "/C2=", Line );
		Line = Get_Line_From_Win( ++Cnt, DifWinNo );
		S1 = Parse_Int( "/S1=", Line );
		S2 = Parse_Int( "/S2=", Line );
		Jx = S1 - C1;
		if ( Jx ) {
			Goto_Line( C2 + LnOffset - 1 );
			LnOffset += Jx;
			while ( C1 < S1 ) {
				Eol;
				Cr;
				Line = Get_Line_From_Win( C1++, Win1No );
				Put_Line( Line );
				Line_Attr = fc_Inserted_Attr;
				Line_Changed = False;
			}
		}
		Jx = S2 - C2;
		if ( Jx ) {
			Goto_Line( C2 + LnOffset );
			while ( C2++ < S2 ) {
				Line_Attr = fc_Different_Attr;
				Down;
			}
		}
	}
	File_Changed = False;
	Read_Only = True;
	Rm( "ExtSetup" );
	Tof;
	while ( Line_Attr == 0 ) {
		Down;
	}
	Rm( "Select_Window" );
	Rm( "SetWindowNames" );
	Window_Name += "-CDif";
	Cur_File_Time = 0;
	if ( C_Line != 1 ) {

		int Value = ( Win_Y2 - Win_Y1 ) / 2;

		Jx = C_Line;
		while ( C_Row > Value ) {
			Up;
		}
		while ( C_Row < Value ) {
			Down;
		}
		Goto_Line( Jx );
	}
	Key_To_Window( k_AltShftA, "Fc_CDifSrch /C=0" );
	Key_To_Window( k_AltShftS, "Fc_CDifSrch /C=1" );
	Key_To_Window( k_AltShftD, "Fc_CDifSrch /C=2" );
	Key_To_Window( k_AltShftF, "Fc_CDifSrch /C=3" );
	Key_To_Window( k_AltShftX, "Fc_CDifSrch /C=4" );
	Key_To_Window( k_AltShftE, "Fc_CDifSrch /C=5" );

	Key_To_Window( k_AltPgDown, "Fc_CDifSrch /C=4" );
	Key_To_Window( k_AltPgUp, "Fc_CDifSrch /C=5" );

	if ( UseWin2 ) {
		if ( Switch_Win_Id( DifWin ) ) {
			Delete_Window;
		}
		Switch_Win_Id( Win1 );
		Goto_Mark;
	}

Exit1:
	Switch_Win_Id( LstWin );
	Rm( "Select_Window /BM=1/VIS=1" );

Exit:
	Undo_Stat = TUndo_Stat;
	Insert_Mode = TInsert_Mode;
	Refresh = TRefresh;

	return ( LstWin );

}  // Fc_CDiff

void fc_update( int ol, nl, wid1, wid2, twid ) {
/******************** Multi-Edit VOID Macro Function ************************

 NAME:         fc_update

 DESCRIPTION:  Handles updating the fc data after lines have been inserted
							 or deleted.  Should only be called by fc_eproc
							 The update is done by taking the offset of ol and nl.

 PARAMETERS:   ol				old line number
							 nl				new line number
							 wid1, wid2  compare window ids
							 twid     compare data window id


*****************************04-19-93 04:27pm*******************************/

	int tr = refresh, twin = cur_window,
			tl, offset;
	str tc[4],ts[4], tstr[80];

	refresh = false;

	offset = nl - ol;		// get offset

	if(window_id == wid1)  // Determine the data that is being updated
	{
		tc = "/C1=";
		ts = "/S1=";
	}
	else
	{
		tc = "/C2=";
		ts = "/S2=";
	}

	if(switch_win_id(twid))
	{
		tof;                      // process whole file
		while(!at_eof)
		{
			tstr = get_line;
			tl = parse_int(tc,tstr);
			if(tl >= ol)
			{
				tl = tl + offset;
				var_remove_str( tc, tstr );
				tstr += tc + str(tl);
				put_line(tstr);
			}
			else
			{
				tl = parse_int(ts,tstr);
				if(tl >= ol)
				{
					tl = tl + offset;
					var_remove_str( ts, tstr );
					tstr += ts + str(tl);
					put_line(tstr);
				}
			}
			down;
		}

		switch_window(twin);
	}


	refresh = tr;
}
global
{
	int fc_ev_count;
}

int Fc_EProc( int En ) {
/****************** Multi-Edit INTEGER Macro Function ***********************

 NAME:         fc_eproc()

 DESCRIPTION:  Even processor for file compare windows

*****************************04-18-93 02:04pm*******************************/

	int Wid1 	 = Parse_Int( "/W1=", MParm_Str );
	int Wid2 	 = Parse_Int( "/W2=", MParm_Str );
	int TWid 	 = Parse_Int( "/TW=", MParm_Str );
	int Zoomed = Parse_Int( "/WZ=", MParm_Str );
	int Bid  	 = Buffer_Id;
	int CL1  	 = C_Line + 1;
	int Ww   	 = Window_Id;
	int cl,cp,crr,co,
			result = 1,
			wcmd,
			wlp = wcmd_last_lparam,
			l1,l2;

	int tr = refresh;
	int SelWin = 0;
	switch ( En & 0xFFFF ) {

#ifdef Windows
		case EVENT_WCMD :
			// Set a mark in the third marker set
			Set_Mark_Record( 1, 3, Cl1, C_Col, C_Row + 1, Left_Offset );
			Result = False;
			WCmd = En >> 16;

			if ( WCmd == Parse_Int( "/NWC=", MParm_Str ) ) {
				Fc_Next_Dif( 0, Wid1, Wid2, TWid );
			}
			else if( WCmd == Parse_Int( "/PWC=", MParm_Str ) ) {
				Fc_Next_Dif( 1, wid1, wid2, twid );
			}
			else if( WCmd == Parse_Int( "/HWC=", MParm_Str ) ) {
				MewHelp( FILECOMP_HELPLINK );
			}
			else if ( WCmd == Parse_Int( "/CWC=", MParm_Str ) ) {
				Fc_CDiff( TWid, Wid1, Wid2, False );
			}
			else {
      	if ( WCmd == Global_Int( "~CLOSE_WCMD_ID" ) ) {
					call Remove_Comp2;
					g_FcSelWin = SelWin;
        }
				WCmd_Execute( Cur_Window, WCmd, WLp );
      	if ( WCmd == Global_Int( "~CLOSE_WCMD_ID" ) ) {
					Switch_Win_Id( SelWin );
					Rm( "SetWindowNames" );
					g_FcSelWin = 0;
					break;
				}
			}
			call Check2;
			break;
#endif

		case EVENT_KEY_ENHANCED :
			Result = 0;
			call Process_0;
			break;

		case EVENT_MOUSE_HSCROLL :
		case EVENT_MOUSE_VSCROLL :
		case EVENT_CURSOR_LINE :
		case EVENT_CURSOR_COL :
			// synch the positions
			Fc_Match_Pos( Wid1, Wid2, TWid, L1, L2 );
			call Display_Keys;
			break;


		case EVENT_DELETE_WIN :
			call Remove_Comp2;
			Switch_Win_Id( SelWin );
			break;

	}
	Refresh = Tr;
	return ( Result );


Process_0:
		// Check for next/previous difference keys
		L1 = ( Key2 << 8 ) + Key1;
		if ( L1 == Parse_Int( "/NKEY=", Fc_Config_Str ) ) {
			Fc_Next_Dif( 0, Wid1, Wid2, TWid );
		}
		else if ( L1 == Parse_Int( "/PKEY=", Fc_Config_Str ) ) {
			Fc_Next_Dif( 1, Wid1, Wid2, TWid );
		}
		else if ( L1 == Parse_Int( "/RKEY=", Fc_Config_Str ) ) {
			Fc_Report;
		}
		else {
			// Set a mark in the third marker set
			Set_Mark_Record( 1, 3, CL1, C_Col, C_Row + 1, Left_Offset );

			// Execute the user's operation
			Pass_Event_Through;

Check2:
			// If we are still in the same window then process
			if ( Ww == Window_Id ) {
				// Get the mark we set
				Get_Mark_Record( 1, 3, Cl, Cp, Crr, Co );
				//if((c_line != (cl1 - 1)) || (cl != cl1))
			 //	{
					// If the lines do not match, then something was inserted or deleted
					if ( Cl != Cl1 ) {
						// Update the tmp window
						Fc_Update( Cl1, Cl, Wid1, Wid2, TWid );
					}
					// Now synch the positions
					Fc_Match_Pos( Wid1, Wid2, TWid, L1, L2 );
					call Display_Keys;
			 //	}
			}
			else {
				// Check for the existance of the windows.  If either of them is gone,
				// then remove the tmp window and cancel the compare
				call Remove_Comp;
			}
		}
		ret;


Display_Keys:

#ifndef Windows
	Make_Message( Parse_Str( "/KEYSTR=", Fc_Config_Str ) );
#endif

	ret;


Remove_Comp2:
	Ww = Window_Id;
	Refresh = False;
	if ( Ww == Wid1 ) {
		if ( Switch_Win_Id( Wid2 ) ) {
			Rm( "UnSplitWin /ID=" + Str( Wid1 ) );
			if ( Zoomed ) {
				SendMessage( Client_Handle, WM_MDIMAXIMIZE, Window_Handle, 0 );
			}
		}
	}
	else if ( Ww == Wid2 ) {
		if ( Switch_Win_Id( Wid1 ) ) {
			Rm( "UnSplitWin /ID=" + Str( Wid2 ) );
			if ( Zoomed ) {
				SendMessage( Client_Handle, WM_MDIMAXIMIZE, Window_Handle, 0 );
			}
		}
	}
	SelWin = Window_Id;
	Switch_Win_Id( Ww );
	call Cleanup;
	Switch_Win_Id( Ww );
	ret;


Remove_Comp:
	Ww = Window_Id;
	Refresh = False;
	if ( !Switch_Win_Id( Wid1 ) ) {
		call Cleanup;
	}
	else if ( !Switch_win_id( Wid2 ) ) {
		call Cleanup;
	}
	Switch_Win_Id( Ww );
	ret;


Cleanup:
	Fc_Cleanup( Wid1, Wid2, TWid );
	ret;

}  // Fc_EProc

int Fc(
			str FName = Parse_Str( "/F=", MParm_Str ),
			int Split = Parse_Int( "/S=", MParm_Str )
		)
/******************** Multi-Edit VOID Macro Function ************************

 NAME:         fc()

 DESCRIPTION:   This is the main file compare macro fc_mode lets you override
								whether or not this is a new compare Normally, fc automatically
								detects whether or not to start a new compare, based on the
								window you are in.

 PARAMETERS:

 RETURNS:      	Return_Int

									0 		- Compare ok
									1 		- Files the same
									2 		- Error

*****************************04-09-93 01:29am*******************************/
{
	int w1 = cur_window, w2, tw, wn, highlighting;
	int wid1 = window_id, wid2, c1 = 1, c2 = 1, lc1, lc2, lx1, lx2, cc1, cc2;
	int tt, dc = 0, jx, fc_done = false;
	int fc_tmp_win;
	int Fc_Mode = Parse_Int( "/M=", MParm_Str );
	int CDWid;
	int Result = 0;
	int WindowZoomed;
	int NoWinList;

	str SplitStr = "RightLeft Up   Down";
	str TStr;
	str OrgFile = File_Name;

	Refresh = False;

	if ( Fc_Config_Str == ""  ) {
		Fc_Config( True );
	}
	if ( Svl( FName ) != 0 ) {
		Fc_Mode = True;
	}

	Fc_Tmp_Win = Parse_Int( "/TW=", Event_Macro );

	// Return to original window
// 	switch_win_id( wid1 );

	// Get the ids from the last compare
	Wid1 = Parse_Int( "/W1=", Event_Macro );
	Wid2 = Parse_Int( "/W2=", Event_Macro );

	// Check to see if we are in one of the windows from the last compare
	// If we are, then check for the existance of the second window
	// If both are true, then start from where we left off with the last compare
	// instead of restarting.
	if ( Window_Id == Wid1 ) {
		if ( Switch_Win_Id( Wid2 ) ) {
			Fc_Mode = True;
		}
	}
	else if ( Window_Id == Wid2 ) {
		if ( Switch_Win_Id( Wid1 ) ) {
			Fc_Mode = True;
		}
	}
	if ( Fc_Mode ) {
		Fc_Mode = False;
		if ( Switch_Win_Id( Wid1 ) ) {
			W1 = Cur_Window;
			if ( Switch_Win_Id( Wid2 ) ) {
				W2 = Cur_Window;
				Fc_Mode = True;
			}
		}
	}
	// if fc_mode == 0, start a new compare
	if ( !Fc_Mode ) {
		Switch_Window( W1 );
		Wid1 = Window_Id;
		if ( Split == 0 ) {
			Split = Parse_Int( "/WS=", Fc_Config_Str );
		}
		TStr = "";
		Result = 0;
		if ( Svl( FName ) != 0 ) {
      Return_Str = FName;
      Rm( "LdFiles /DE=1/NC=1/NW=1/CW=1/NHA=1/NB=1" );
			if ( Error_Level ) {
				Error_Level = 0;
				Delete_Window;
				Switch_Win_Id( Wid1 );
				Result = 2;
			}
			TStr = "/ID=" + Str( Window_Id );
		}
		if ( Result == 0 ) {
			if ( Split == 1 ) {
				Split = SplitWinDlg( "FC Split Window", 0x1F );
				if ( Split == 0 ) {
					goto Exit;
				}
				else {
					Split -= 9;
				}
			}
			if ( Split < 6 ) {
        Refresh = True;
				Switch_Win_Id( Wid1 );
				Refresh = False;
				if ( Split > 1 ) {
					Split -= 2;
					TStr += "/DIR=" + Remove_Space( Copy( SplitStr, ( Split * 5 ) + 1, 5 ) );
				}
				if ( BriefMode ) {
					Rm( "BRIEF^BrMakeSpecialWin" );
					Wid1 = Window_Id;
					TStr += "/NL=1/NR=1/NH=1";
				}
				WindowZoomed = IsZoomed( Window_Handle );
				NoWinList = Parse_Int( "/NWL=", Fc_Config_Str );
				Rm( "SplitWin /WL=" + Str( !NoWinList ) + TStr );
				if ( !XPos( "/ID=", TStr, 1 ) ) {
					if ( !Return_Int ) {
						Make_Message( "Compare aborted" );
						Result = 2;
					}
					else if ( Caps( File_Name ) == Caps( OrgFile ) ) {
						Result = 1;
						if ( NoWinList ) {
							TStr = "/LC=0/CW=0";
							goto FilePrompt;
						}
					}
				}
				Rm( "Select_Window" );
				Redraw;
			}
			else if ( Window_Id == Wid1 ) {
				TStr = "/CW=1/NB=1";
				Result = 2;

FilePrompt:
        Return_Str = OrgFile;
				Rm( "MEUTIL1^File_Prompt /T=Filename to Compare" );
				if ( Return_Int > 0 ) {
        	Rm( "LdFiles /DE=1/NC=1/NW=1/NHA=1" + TStr );
        	if ( Error_Level ) {
						Error_Level = 0;
						if ( Result = 2 ) {
							Delete_Window;
							Switch_Win_Id( Wid1 );
						}
					}
					else {
						Result = 0;
					}
				}
			}
		}
		Refresh = False;
		Wid2 = Window_Id;
		W2 = Cur_Window;
		if ( Result ) {
			if ( Result == 1 ) {
				call RmWindow;
				Make_Message( "Compare aborted" );
				Result = 2;
			}
			goto Exit;
		}
	}
	else {
		Working;
		Fc_Reset( Wid1  );
		Fc_Reset( Wid2 );
		Rm( "SetWindowNames" );
	}
	Highlighting = Parse_Int( "/SH=", Fc_Config_Str );

	if ( !Highlighting ) {
		Keywords_Str = "";
	}
	Mark_Pos;
	Eof;
	Lc2 = C_Line;
	Tof;

	Switch_Window( W1 );
	if ( !Highlighting ) {
		Keywords_Str = "";
	}
	Mark_Pos;
	Eof;
	Lc1 = C_Line;
	Tof;

	call Compare;

	if ( Result == 1 ) {
		Make_Message( "Files the same" );
		call RmWindow;
		Goto_Mark;
		goto Exit;
	}
	if ( Split == 6 ) {
		Fc_CDiff( Fc_Tmp_Win, Wid1, Wid2, True );
		goto Exit;
	}
	Push_Labels;

	Switch_Window( W1 );
	if ( Event_Macro != "" ) {
		if ( Fc_Enabled( ) == 1 ) {
			Set_Global_Str( "!FcEventSav_" + Str( Window_Id ), Event_Macro );
			Set_Global_Int( "!FcEventSav_" + Str( Window_Id ), Event_Mode );
		}
		else {
			WindowZoomed = Parse_Int( "/WZ=", Event_Macro );
		}
		Event_Macro = "";
		Event_Mode = 0;
	}
	Wn = Cur_Window;
	if( Fc_Mode ) {
		Goto_Mark;
	}
	else {
		Tof;
	}
	Redraw;
	Cc1 = C_Line;
	Switch_Window( W2 );
	if ( Event_Macro != "" ) {
		if ( Fc_Enabled( ) == 1 ) {
			Set_Global_Str( "!FcEventSav_" + Str( Window_Id ), Event_Macro );
			Set_Global_Int( "!FcEventSav_" + Str( Window_Id ), Event_Mode );
		}
		else {
			WindowZoomed = Parse_Int( "/WZ=", Event_Macro );
		}
		Event_Macro = "";
		Event_Mode = 0;
	}
	if( Fc_Mode ) {
		Goto_Mark;
	}
	else {
		Tof;
	}
	Cc2 = C_Line;
	Redraw;

	// If this is a new compare, then go to the first difference
	if ( !Fc_Mode ) {
		Fc_Next_Dif( 0, Wid1, Wid2, Fc_Tmp_Win );
	}
	call Set_Events;
	Rm( "SetWindowNames" );
	Redraw;

#ifndef Windows
	Fc_Set_Keynames( );
	Make_Message( Parse_Str( "/KEYSTR=", Fc_Config_Str ) );
#endif

Exit:
	Return_Int = Result;
	return ( Result );


Set_Events:

	int Twx = Cur_Window;
	int TB1NextWCmd = Fc_WCmd_Next;
	int TB1PrevWCmd = Fc_WCmd_Prev;
	int TB1HelpWCmd = Fc_WCmd_Help;
	int TB1CDifWCmd	= Fc_WCmd_CDif;

	Switch_Window( W1 );

#ifdef Windows

	/*
	int tb1handle =
			ToolboxCreate(frame_handle,"Diff",TBS_FLOATING,25,
									25,client_handle,
									0);

			ToolboxAddTextItem(tb1handle,"Next", tb1nextwcmd,TS_OneState,"Next difference",
									"");

			ToolboxAddTextItem(tb1handle,"Prev", tb1prevwcmd,TS_OneState,"Previous difference",
									"");

			ToolBoxSetPos( tb1handle, 25,25);

			ToolboxShow( tb1handle );
	 */
	Set_Global_Str( "!FC-" + Hex_Str( Fc_Tmp_Win ),
			"/NWC=" + Str( TB1NextWCmd ) + "/PWC=" + Str( TB1PrevWCmd ) +
			"/HWC=" + Str( TB1HelpWCmd ) + "/CWC=" + Str( TB1CDifWCmd )
			// + "/TBH=" + str( tb1handle )
			);
#endif

	Event_Macro  = "Fc_EProc /W1=" + Str( Wid1 ) + "/W2=" + Str( Wid2 ) +
			"/TW=" + Str( Fc_Tmp_Win ) + "/NWC=" + Str( TB1NextWCmd ) +
			"/PWC=" + Str( TB1PrevWCmd ) + "/HWC=" + Str( TB1HelpWCmd ) +
			"/CWC=" + Str( TB1CDifWCmd ) + "/WZ=" + Str( WindowZoomed );
	Event_Mode = FC_EVENT_MODE;

	Switch_Window( W2 );
	Event_Macro  = "Fc_EProc /W1=" + Str( Wid1 ) + "/W2=" + Str( Wid2 ) +
			"/TW=" + Str( Fc_Tmp_Win ) + "/NWC=" + Str( TB1NextWCmd ) +
			"/PWC=" + Str( TB1PrevWCmd ) + "/HWC=" + Str( TB1HelpWCmd ) +
			"/CWC=" + Str( TB1CDifWCmd ) + "/WZ=" + Str( WindowZoomed );
	Event_Mode = FC_EVENT_MODE;
	// If filename passed then the file could be deleted, fix so no error shows
	if ( Svl( FName ) ) {
		Cur_File_Time = 0;
	}
	Switch_Window( Twx );
	ret;


Compare:  // Perform the next part of the compare
	Refresh = False;
	Make_Message( "Comparing files...");
	Working;
	Wn = Window_Id;
	// look for our file compare temp window if it doesn't exist, then create it
	if ( !Switch_Win_Id( Fc_Tmp_Win ) || ( !Fc_Tmp_Win ) ) {
		Create_Window;
		Window_Attr = 0x81;
		Fc_Tmp_Win = Window_Id;
		File_Name = "!Fc_Dif";
	}
	if ( Switch_Win_Id( Fc_Tmp_Win ) ) {
		// now erase it
		Erase_Window;
		Window_Attr = 0x81;
		File_Name = "!Fc_Dif";
	}
	Tw = Cur_Window;

	// Mark positions in windows
	Switch_Win_Id( Wid1 );
	W1 = Cur_Window;
	Switch_Win_Id( Wid2 );
	W2 = Cur_Window;

	Lx1 = Lc1;
	Lx2 = Lc2;
	// Do the compare
	int Flags = 0;

	if ( Parse_Int( "/IGB=", Fc_Config_Str ) ) {
		Flags |= _FC_IGNORE_BLANK;
	}
	if ( Parse_Int( "/IGC=", Fc_Config_Str ) ) {
		Flags |= _FC_IGNORE_CASE;
	}
	if ( Parse_Int( "/IGI=", Fc_Config_Str ) ) {
		Flags |= _FC_IGNORE_INDENT;
	}

	FCompare( W1, W2, Tw, C1, C2, Lx1, Lx2, Dc, Lc1, Lc2, Flags );

	Switch_Window( Tw );
	Eof;
	if ( C_Line == 1 ) {
		Delete_Window;											// Delete temp window
		Result = 1;                         // Files the same
	}
	Switch_Win_Id( Wn );
	Make_Message( "Compare complete." );
	ret;


RmWindow:
	Switch_Win_Id( Wid1 );
	Rm( "UnSplitWin /ID=" + Str( Wid2 ) );
	if ( WindowZoomed ) {
		SendMessage( Client_Handle, WM_MDIMAXIMIZE, Window_Handle, 0 );
	}
	if ( Switch_Win_Id( Wid2 ) ) {
		Rm( "USERIN^CheckFile" );
		Delete_Window;
    Switch_Win_Id( Wid1 );
    if ( BriefMode ) {
      Rm( "BRIEF^BrDelSpecialWin" );
    }
	}
	Switch_Win_Id( Wid1 );
	ret;

}  // Fc
