/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyWave.c,v $
   $Revision: 1.10.14.1 $ $Date: 1995/06/09 22:00:26 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's wave functions.

   Written by: Colin Hui

   (c) Copyright 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinySceneToken.h"
#include "tinyWave.h"
#include "tinyUtils.h"

static int DK_waveNodeWriteAsciiAnimation (  FILE *, DK_Wave *);
static int DK_waveNodeReadAsciiAnimation (  FILE *, DK_Wave *);

/*******************************************************************************

   $$L DK_waveAllocate

   This function allocates a wave structure.

   Returned Value: DK_Wave structure

*******************************************************************************/

DK_Wave *DK_waveAllocate( void )
{
   DK_Wave *wave;
   int i;

   if( wave = (DK_Wave *) calloc( 1, sizeof( DK_Wave ) ) )
   {
      wave->wave_name  = NULL;
      wave->wave_prefix  = NULL;
      _DK_VectorInit(wave->scale, 1.0, 1.0, 1.0);
      _DK_VectorInit(wave->rotate, 0.0, 0.0, 0.0);
      _DK_VectorInit(wave->translation, 0.0, 0.0, 0.0);
      wave->decay = 0.0;
      wave->spread = 0.0;
      wave->pos_constraint_name = NULL;
      wave->pos_constraint_type = 0;
      wave->path = NULL;
      wave->path_name = NULL;
      wave->constr_list = NULL;
      wave->w_amodel = NULL;
      wave->next  = NULL;
      for (i = 0; i < DK_FCV_WAVE_NUM; i++)
      {
	  wave->fcv[i] = NULL;
	  wave->active_fcv[i].active = FALSE;
	  wave->active_fcv[i].fcv = NULL;
      }
   }

   return( wave );
}

/*******************************************************************************

   $$L DK_wave_amodelAllocate

   This function allocates a wave attach model structure.

   Returned Value: DK_Wave_amodel structure

*******************************************************************************/

DK_Wave_amodel *DK_wave_amodelAllocate( void )
{
   DK_Wave_amodel *awave;

   if( awave = (DK_Wave_amodel *) calloc( 1, sizeof( DK_Wave_amodel ) ) )
   {
      awave->next  = NULL;
      awave->model_name  = NULL;
      awave->model_branch  = FALSE;
   }

   return( awave );
}

/*******************************************************************************

   $$L DK_waveDispose

   This function disposes a wave structure.

   Returned Value: None

*******************************************************************************/

void DK_waveDispose
    (
	DK_Wave **wv
    )
{
    if( wv )
    {
        DK_Wave *wave = *wv;

        while ( wave )
        {
            DK_Wave *save = wave;

            _DK_FREE( wave->wave_name, -1 );
            _DK_FREE( wave->wave_prefix, -1 );
            _DK_FREE( wave->pos_constraint_name, -1 );
            if( wave->path )
                DK_splineDispose( &wave->path );
            _DK_FREE( wave->path_name, -1 );
	    DK_constraintDispose( &wave->constr_list );
            DK_fcurveArrayDispose( wave->fcv, DK_FCV_WAVE_NUM );
            DK_actFcurveArrayDispose( wave->active_fcv, DK_FCV_WAVE_NUM );
            if( wave->w_amodel )
                DK_wave_amodelDispose( &wave->w_amodel );

            wave = wave->next;
            free( save );
        }

        *wv = NULL;
    }
}

/*******************************************************************************

   $$L DK_wave_amodelDispose

   This function disposes a wave_amodel structure.

   Returned Value: None

*******************************************************************************/

void DK_wave_amodelDispose
    (
	DK_Wave_amodel **wv
    )
{
    if( wv )
    {
        DK_Wave_amodel *amodel = *wv;

        while( amodel )
        {
            DK_Wave_amodel *save = amodel;

            _DK_FREE( amodel->model_name, -1 );
            amodel = amodel->next;
            free( save );
        }

        *wv = NULL;
    }
}

/*******************************************************************************

   $$L DK_waveNodeWriteAsciiAnimation

   This function write the waves.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_waveNodeWriteAsciiAnimation
    (
	 FILE *file,  	/* file io pointer */
	DK_Wave *wave	/* wave pointer */
    )
{
    DK_Wave_amodel *att;
    DK_Vector dummy;

    if (wave == NULL)
	return(FALSE);

    DK_incIndent(1);

    _DK_OUT_INDENT();
    _DK_OUT_STRING(wave->wave_name);

    if (wave->fcv[DK_FCV_WAVE_SCLX] != NULL || wave->fcv[DK_FCV_WAVE_SCLY] != NULL ||
        wave->fcv[DK_FCV_WAVE_SCLZ] != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_WAVE_NODE_SCALE_TOKEN);
        DK_value3WriteAsciiAnimation(file, 1.0, 1.0, 1.0, wave->fcv[DK_FCV_WAVE_SCLX],
		wave->fcv[DK_FCV_WAVE_SCLY], wave->fcv[DK_FCV_WAVE_SCLZ]);
    }

    if (wave->fcv[DK_FCV_WAVE_ROTX] != NULL || wave->fcv[DK_FCV_WAVE_ROTY] != NULL ||
	wave->fcv[DK_FCV_WAVE_ROTZ] != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_WAVE_NODE_ROTATE_TOKEN);
        DK_value3WriteAsciiAnimation(file, 0.0, 0.0, 0.0, wave->fcv[DK_FCV_WAVE_ROTX],
		wave->fcv[DK_FCV_WAVE_ROTY], wave->fcv[DK_FCV_WAVE_ROTZ]);
    }

    dummy.x = 0.0;
    dummy.y = 0.0;
    dummy.z = 0.0;
    if (wave->fcv[DK_FCV_WAVE_TRNX] != NULL || wave->fcv[DK_FCV_WAVE_TRNY] != NULL ||
	wave->fcv[DK_FCV_WAVE_TRNZ] != NULL || wave->fcv[DK_FCV_WAVE_POS] != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_WAVE_NODE_TRANSLATE_TOKEN);
        DK_positionWriteAsciiAnimation(file, &dummy,
	    wave->fcv[DK_FCV_WAVE_POS], wave->path,
	    wave->pos_constraint_name, wave->pos_constraint_type,
	    wave->fcv[DK_FCV_WAVE_TRNX], wave->fcv[DK_FCV_WAVE_TRNY], 
	    wave->fcv[DK_FCV_WAVE_TRNZ],
	    wave->path_name);
    }

    if (wave->fcv[DK_FCV_WAVE_DCY] != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_WAVE_NODE_DECAY_TOKEN);
        DK_valueWriteAsciiAnimation(file, 0.0, wave->fcv[DK_FCV_WAVE_DCY]);
    }

    if (wave->fcv[DK_FCV_WAVE_SPD] != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_WAVE_NODE_SPREAD_TOKEN);
        DK_valueWriteAsciiAnimation(file, 0.0, wave->fcv[DK_FCV_WAVE_SPD]);
    }

    att = wave->w_amodel;
    while (att != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_WAVE_NODE_MODEL_TOKEN);
        _DK_OUT_STRING(att->model_name);
        if (att->model_branch == TRUE)
	    _DK_OUT_TOKEN(DK_A_YES_TOKEN);
        else
	    _DK_OUT_TOKEN(DK_A_NO_TOKEN);
	att = att->next;
    }

    DK_incIndent(-1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_waveNodeReadAsciiAnimation

   This function read the waves.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_waveNodeReadAsciiAnimation
    (
	 FILE *file,  	/* file io pointer */
	DK_Wave *wave
    )
{
    char *name = NULL;
    float dummy;
    int branch;
    DK_Vector vdummy;
    DK_Wave_amodel *att, *att2;

    name = DK_stringReadAsciiAnimation(file);
    if (wave != NULL)
	wave->wave_name = name;
    else
	_DK_FREE(name, -1);

    DK_in_animToken(file);
    while (TRUE)
    {
	if (strcmp(DK_A_WAVE_NODE_SCALE_TOKEN, DK_animToken) == 0) 
	{
	    if (wave != NULL)
                DK_value3ReadAsciiAnimation(file, &dummy, &dummy, &dummy, 
			&wave->fcv[DK_FCV_WAVE_SCLX], &wave->fcv[DK_FCV_WAVE_SCLY],
			&wave->fcv[DK_FCV_WAVE_SCLZ]);
	    else
                DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
	}
	else if (strcmp(DK_A_WAVE_NODE_ROTATE_TOKEN, DK_animToken) == 0) 
	{
	    if (wave != NULL)
                DK_value3ReadAsciiAnimation(file, &dummy, &dummy, &dummy, 
			&wave->fcv[DK_FCV_WAVE_ROTX], &wave->fcv[DK_FCV_WAVE_ROTY],
			&wave->fcv[DK_FCV_WAVE_ROTZ]);
	    else
                DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
	}
	else if (strcmp(DK_A_WAVE_NODE_TRANSLATE_TOKEN, DK_animToken) == 0) 
	{
	    if (wave != NULL)
	        DK_positionReadAsciiAnimation(file, &vdummy, &wave->fcv[DK_FCV_WAVE_POS], 
			&wave->path, &wave->pos_constraint_name, 
			&wave->pos_constraint_type, 
			&wave->fcv[DK_FCV_WAVE_TRNX], &wave->fcv[DK_FCV_WAVE_TRNY],
			&wave->fcv[DK_FCV_WAVE_TRNZ],
			&wave->path_name);
	    else
	        DK_positionReadAsciiAnimation(file, NULL, NULL,
			NULL, NULL, NULL, NULL, NULL, NULL, NULL);
        }
	else if (strcmp(DK_A_WAVE_NODE_DECAY_TOKEN, DK_animToken) == 0) 
	{
	    if (wave != NULL)
	        DK_valueReadAsciiAnimation(file, &dummy, &wave->fcv[DK_FCV_WAVE_DCY], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
	else if (strcmp(DK_A_WAVE_NODE_SPREAD_TOKEN, DK_animToken) == 0) 
	{
	    if (wave != NULL)
	        DK_valueReadAsciiAnimation(file, &dummy, &wave->fcv[DK_FCV_WAVE_SPD], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
	else if (strcmp(DK_A_WAVE_NODE_MODEL_TOKEN, DK_animToken) == 0) 
	{
	    name = DK_stringReadAsciiAnimation(file);
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
		branch = TRUE;
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
		branch = FALSE;
	    else 
		branch = FALSE;
	    if ((att = DK_wave_amodelAllocate()) != NULL)
	    {
		att->model_name = name;
		att->model_branch = branch;
		att2 = wave->w_amodel;
		if (att2 == NULL)
		    wave->w_amodel = att;
		else
		{
		    while (att2 != NULL && att2->next != NULL)
		        att2 = att2->next;
		    att2->next = att;
		}
	    }
	    else
		_DK_FREE(name, -1);
            DK_in_animToken(file);
	}
	else
	    return(TRUE);
    }
}

/*******************************************************************************

   $$L DK_waveWriteAsciiAnimation

   This function write waves.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_waveWriteAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	char *filename,		/* wave filename */
	DK_Wave *wave
    )
{
    int rtn = FALSE;

    DK_incIndent(1);

    _DK_OUT_INDENT();
    _DK_OUT_STRING(filename);
    
    while (wave != NULL) 
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN(DK_A_WAVE_NODE_TOKEN);
	rtn |= DK_waveNodeWriteAsciiAnimation(file, wave);
        wave = wave->next;
    }
    DK_incIndent(-1);

    return(rtn);
}

/*******************************************************************************

   $$L DK_waveReadAsciiAnimation

   This function read waves.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_waveReadAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	char **wave_filename,
	DK_Wave **wave
    )
{
    int rtn = TRUE;
    char *name;
    DK_Wave *last_w, *w;

    if (wave != NULL && *wave != NULL)
    {
	last_w = *wave;
	while (last_w->next != NULL)
	    last_w = last_w->next;
    }

    name = DK_stringReadAsciiAnimation(file);
    if (name == NULL || name[0] == '\0')
    {
        DK_in_animToken(file);
	return(FALSE);
    }
    if (wave_filename != NULL)
    {
	if (*wave_filename != NULL)
	    _DK_FREE(*wave_filename, -1);
        *wave_filename = name;
    }
    else
	_DK_FREE(name, -1);

    /* read each node */
    DK_in_animToken(file);
    while (TRUE)
    {
	if (strcmp(DK_A_WAVE_NODE_TOKEN, DK_animToken) == 0)
	{
	    /* allocate wave structure and link it to the end */
	    w = DK_waveAllocate();
	    if (wave != NULL)
	    {
		if (*wave == NULL)
		    *wave = w;
		else
		    last_w->next = w;
	    }
	    last_w = w;
	    rtn = DK_waveNodeReadAsciiAnimation(file, w);
	}
	else
	    return(rtn);
    }
}
