/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyTexture.c,v $
   $Revision: 1.27.14.3 $ $Date: 1995/07/21 23:08:34 $
   Checkin by: $Author: alain $

   This file contains a subset of the SOFTIMAGE's textures (2D & 3D) functions,
   its content is sufficient to read, write and manage textures data structure
   (2D & 3D) into SOFTIMAGE's file format...

   Written by: Laurent Lauzon

   (c) Copyright 1990, 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/
#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinyTexture.h"
#include "tinySceneToken.h"
#include "tinyScene.h"
#include "tinyUtils.h"
#include "tinyMental.h"

DK_String   DK_TextureAnimName[] =
   {
      "STATIC",
      "SEQUENCE",
      "LIST"
   };

DK_String   DK_TextureMethodName[] =
   {
      "--------",
      "XY",
      "XZ",
      "YZ",
      "UV",
      "CYLINDRICAL",
      "SPHERICAL",
      "AFTER_REFLECT",
      "WITHOUT_REFLECT"
   };

DK_String   DK_Texture3DTypeName[] =
   {
      "MARBLE",
      "WOOD",
      "CLOUD"
   };

/******************************************************************************/
/*** Static function prototypes ***********************************************/
/******************************************************************************/

static DK_TextureAnim    DK_textureGetAnimationByName( DK_String );
static DK_TextureMethod  DK_textureGetMethodByName( DK_String );
static DK_Texture       *DK_textureReadBinaryFile( DK_String );
static DK_Texture       *DK_textureReadAsciiFile( DK_String );
static DK_Boolean        DK_textureWriteBinaryFile( DK_String, DK_Texture * );
static DK_Boolean        DK_textureWriteAsciiFile( DK_String, DK_Texture * );

static DK_Texture3DType  DK_texture3DGetTypeByName( DK_String );
static DK_Texture3D     *DK_texture3DReadBinaryFile( DK_String );
static DK_Texture3D     *DK_texture3DReadAsciiFile( DK_String );
static DK_Boolean        DK_texture3DWriteBinaryFile( DK_String, DK_Texture3D * );
static DK_Boolean        DK_texture3DWriteAsciiFile( DK_String, DK_Texture3D * );

/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

/******************************************************************************/
/*** 2D Textures maps *********************************************************/
/******************************************************************************/

DK_Texture *DK_textureAllocate( void )
{
   DK_Texture *txt;
   long i;

   if( txt = (DK_Texture *) calloc( 1, sizeof( DK_Texture ) ) )
   {
      txt->pic_name      = NULL;
      txt->txt_name      = NULL;
      txt->txt_prefix    = NULL;
      txt->animation     = DK_TXT_STATIC;
      txt->method        = DK_TXT_XY;
      txt->uCount        = 1;
      txt->vCount        = 1;
      txt->uScal         = 1.0;
      txt->vScal         = 1.0;
      txt->uOffset       = 0.0;
      txt->vOffset       = 0.0;
      txt->pixelInterp   = FALSE;
      txt->ambientFct    = 0.0;
      txt->diffuseFct    = 0.0;
      txt->specularFct   = 0.0;
      txt->reflectFct    = 0.0;
      txt->transpFct     = 0.0;
      txt->roughnessFct  = 0.0;
      txt->reflectMappingFct    = 1.0;
      txt->rotation      = 0.0;
      txt->mask          = DK_TXT_NO_MASK;
      txt->effect        = DK_TXT_INTENSITY;
      txt->blend         = 1.0;
      txt->cropping      = FALSE;
      txt->uMinCrop      = 0;
      txt->vMinCrop      = 0;
      txt->uMaxCrop      = 0;
      txt->vMaxCrop      = 0;
      txt->uvSwap        = FALSE;
      txt->uAlternate    = FALSE;
      txt->vAlternate    = FALSE;
    
      _DK_VectorInit(txt->geo, 1.0, 1.0, 1.0);
      _DK_VectorInit(txt->ctr, 0.0, 0.0, 0.0);
      _DK_VectorInit(txt->rot, 0.0, 0.0, 0.0);
      _DK_VectorInit(txt->scal, 1.0, 1.0, 1.0);
      _DK_VectorInit(txt->trans, 0.0, 0.0, 0.0);
      _DK_MatrixIdInit( txt->txtsup );

      txt->blkwht	 = FALSE;
      txt->shader	 = NULL;

      for ( i = 0; i < DK_FCV_TXT_NUM; i++)
      {
	 txt->fcv[i] = (DK_Fcurve *)NULL;
	 txt->active_fcv[i].active = FALSE;
	 txt->active_fcv[i].fcv = (DK_Fcurve *)NULL;
      }
      
      txt->constr_list	 = NULL;
      txt->displacement = FALSE;

      txt->next          = NULL;

      txt->schem_posx = 0.0;
      txt->schem_posy = 0.0;
      txt->schem_userppos = 0;
   }

   return( txt );
}

void DK_textureDispose( DK_Texture **txt )
{
   if( txt )
   {
      DK_Texture *texture = *txt;

      while( texture )
      {
         DK_Texture *save = texture;

         _DK_FREE( texture->pic_name, -1 );
         _DK_FREE( texture->txt_name, -1 );
         _DK_FREE( texture->txt_prefix, -1 );
         DK_fcurveArrayDispose( texture->fcv, DK_FCV_TXT_NUM );
         DK_actFcurveArrayDispose( texture->active_fcv, DK_FCV_TXT_NUM );
         DK_udfDispose( &texture->shader );
	 DK_constraintDispose( &texture->constr_list );

         texture = texture->next;
         free( save );
      }

      *txt = NULL;
   }
}

DK_Boolean DK_textureWriteAscii(  FILE *file, DK_Texture *txt, DK_String space, short no )
{
   fprintf( file, "\n%stexture    [%1hd]\n", space, no );
   fprintf( file, "%s{\n", space );
   fprintf( file, "%s  name       \"%s\"\n", space, txt->pic_name );
   fprintf( file, "%s  glbname    \"%s\"\n", space, txt->txt_name );
   fprintf( file, "%s  anim       %s\n", space,
                                         DK_TextureAnimName[ txt->animation ] );
   fprintf( file, "%s  method     %s\n", space,
                                         DK_TextureMethodName[ txt->method ]);
   fprintf( file, "%s  repeat     %hd  %hd\n", space, txt->uCount, txt->vCount);
   fprintf( file, "%s  scaling    %*.*f  %*.*f\n", space,  DK_FM,  DK_FD, txt->uScal,
                                                           DK_FM,  DK_FD, txt->vScal );
   fprintf( file, "%s  offset     %*.*f  %*.*f\n", space,  DK_FM,  DK_FD, txt->uOffset,
                                                          DK_FM,  DK_FD, txt->vOffset );
   if( txt->pixelInterp )
      fprintf( file, "%s  pixelInterp\n", space );
   if( txt->uvSwap )
      fprintf( file, "%s  uvswap\n", space );
   if( txt->uAlternate )
      fprintf( file, "%s  uAlternate\n", space );
   if( txt->vAlternate )
      fprintf( file, "%s  vAlternate\n", space );
   switch( txt->effect )
   {
      case DK_TXT_ALPHA:
         fprintf( file, "%s  effect     ALPHA\n", space );
         break;
      case DK_TXT_INTENSITY:
         fprintf( file, "%s  effect     INTENSITY\n", space );
         break;
   }
   switch( txt->mask )
   {
      case DK_TXT_ALPHA:
         fprintf( file, "%s  mask       ALPHA\n", space );
         break;
      case DK_TXT_INTENSITY:
         fprintf( file, "%s  mask       INTENSITY\n", space );
         break;
   }
   fprintf( file, "%s  blending   %*.*f\n", space,  DK_FM,  DK_FD, txt->blend );
   fprintf( file, "%s  ambient    %*.*f\n", space,  DK_FM,  DK_FD, txt->ambientFct );
   fprintf( file, "%s  diffuse    %*.*f\n", space,  DK_FM,  DK_FD, txt->diffuseFct );
   fprintf( file, "%s  specular   %*.*f\n", space,  DK_FM,  DK_FD, txt->specularFct );
   fprintf( file, "%s  reflect    %*.*f\n", space,  DK_FM,  DK_FD, txt->reflectFct );
   fprintf( file, "%s  transp     %*.*f\n", space,  DK_FM,  DK_FD, txt->transpFct );
   fprintf( file, "%s  roughness  %*.*f\n", space,  DK_FM,  DK_FD, txt->roughnessFct );
   fprintf( file, "%s  reflMap    %*.*f\n", space,  DK_FM,  DK_FD, txt->reflectMappingFct );
   fprintf( file, "%s  rotation   %*.*f\n", space,  DK_FM,  DK_FD, txt->rotation );
   if( txt->cropping )
      fprintf( file, "%s  cropping   %hd  %hd  %hd  %hd\n", space,
               txt->uMinCrop, txt->vMinCrop, txt->uMaxCrop, txt->vMaxCrop );

   fprintf( file, "%s  txtsup_rot   %*.*f %*.*f %*.*f\n", space,  
					DK_FM,  DK_FD, txt->rot.x,
					DK_FM,  DK_FD, txt->rot.y,
					DK_FM,  DK_FD, txt->rot.z );
   fprintf( file, "%s  txtsup_trans %*.*f %*.*f %*.*f\n", space,  
					DK_FM,  DK_FD, txt->trans.x,
					DK_FM,  DK_FD, txt->trans.y,
					DK_FM,  DK_FD, txt->trans.x );
   fprintf( file, "%s  txtsup_scal  %*.*f %*.*f %*.*f\n", space,  
					DK_FM,  DK_FD, txt->scal.x,
					DK_FM,  DK_FD, txt->scal.y,
					DK_FM,  DK_FD, txt->scal.z );

   if( txt->shader )
      DK_udfWriteAscii( file, txt->shader, space, "txt2d_shader" );
   
   if( txt->displacement )
       fprintf( file, "%s  displacement\n", space);
	       
   fprintf( file, "%s}\n", space );

   return( TRUE );
}

DK_Texture *DK_textureReadAscii(  FILE *file )
{
   char		 str[ 80 ], str2[ 80 ];
   DK_Texture	*txt;

   if( (txt = DK_textureAllocate()) == NULL )
      return( NULL );

   /* get rid of number */
   fscanf( file, "%s", str );

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( txt );

   do
   {
      fscanf( file, "%s", str );
      if( strcmp( str, "name" ) == 0 )
         txt->pic_name = DK_stringReadAscii( file );
      else if( strcmp( str, "glbname" ) == 0 )
         txt->txt_name = DK_stringReadAscii( file );
      else if( strcmp( str, "anim" ) == 0 )
      {
         fscanf( file, "%s", str2 );
         txt->animation = DK_textureGetAnimationByName( str2 );
      }
      else if( strcmp( str, "method" ) == 0 )
      {
         fscanf( file, "%s", str2 );
         txt->method = DK_textureGetMethodByName( str2 );
      }
      else if( strcmp( str, "repeat" ) == 0 )
         fscanf( file, "%hd %hd", &txt->uCount, &txt->vCount );
      else if( strcmp( str, "scaling" ) == 0 )
         fscanf( file, "%f %f", &txt->uScal, &txt->vScal );
      else if( strcmp( str, "offset" ) == 0 )
         fscanf( file, "%f %f", &txt->uOffset, &txt->vOffset );
      else if( strcmp( str, "pixelInterp" ) == 0 )
         txt->pixelInterp = TRUE;
      else if( strcmp( str, "uvswap" ) == 0 )
         txt->pixelInterp = TRUE;
      else if( strcmp( str, "uAlternate" ) == 0 )
         txt->uAlternate = TRUE;
      else if( strcmp( str, "vAlternate" ) == 0 )
         txt->vAlternate = TRUE;
      else if( strcmp( str, "ambient" ) == 0 )
         fscanf( file, "%f", &txt->ambientFct );
      else if( strcmp( str, "diffuse" ) == 0 )
         fscanf( file, "%f", &txt->diffuseFct );
      else if( strcmp( str, "specular" ) == 0 )
         fscanf( file, "%f", &txt->specularFct );
      else if( strcmp( str, "reflect" ) == 0 )
         fscanf( file, "%f", &txt->reflectFct );
      else if( strcmp( str, "transp" ) == 0 )
         fscanf( file, "%f", &txt->transpFct );
      else if( strcmp( str, "roughness" ) == 0 )
         fscanf( file, "%f", &txt->roughnessFct );
      else if( strcmp( str, "reflMap" ) == 0 )
         fscanf( file, "%f", &txt->reflectMappingFct );
      else if( strcmp( str, "rotation" ) == 0 )
         fscanf( file, "%f", &txt->rotation );
      else if( strcmp( str, "alpha" ) == 0 )
      {
         /* OLD way (DO NOT use it) */
         txt->mask = DK_TXT_ALPHA;
         txt->blend = 1.0;
      }
      else if( strcmp( str, "mask" ) == 0 )
      {
         fscanf( file, "%s", str2 );
         if( strcmp( "ALPHA", str2 ) == 0 )
            txt->mask = DK_TXT_ALPHA;
         else if( strcmp( "INTENSITY", str2 ) == 0 )
            txt->mask = DK_TXT_INTENSITY;
      }
      else if( strcmp( str, "effect" ) == 0 )
      {
         fscanf( file, "%s", str2 );
         if( strcmp( "ALPHA", str2 ) == 0 )
            txt->effect = DK_TXT_ALPHA;
         else if( strcmp( "INTENSITY", str2 ) == 0 )
            txt->effect = DK_TXT_INTENSITY;
      }
      else if( strcmp( str, "blending" ) == 0 )
         fscanf( file, "%f", &txt->blend );
      else if( strcmp( str, "cropping" ) == 0 )
      {
         txt->cropping = TRUE;
         fscanf( file, "%hd %hd %hd %hd\n", &txt->uMinCrop, &txt->vMinCrop,
                                            &txt->uMaxCrop, &txt->vMaxCrop );
      }

      else if ( strcmp( str, "txtsup_rot" ) == 0)
	 fscanf( file, "%f %f %f\n", &txt->rot.x, &txt->rot.y, &txt->rot.z );
      else if ( strcmp( str, "txtsup_trans" ) == 0)
	 fscanf( file, "%f %f %f\n", &txt->trans.x, &txt->trans.y, &txt->trans.z );
      else if ( strcmp( str, "txtsup_scal" ) == 0)
	 fscanf( file, "%f %f %f\n", &txt->scal.x, &txt->scal.y, &txt->scal.z );

      else if( strcmp( str, "txt2d_shader" ) == 0 )
	 txt->shader = DK_udfReadAscii( file );
      else if( strcmp( str, "displacement" ) == 0 )
         txt->displacement = TRUE;

   } while( strcmp( str, "}" ) != 0 );

   return( txt );
}

DK_Boolean DK_textureWriteBinary( FILE *file, DK_Texture *txt, DK_FileProt *prot )
{
   short	 nb, i;
   float	 val;

   for( i = 0; prot->texture[ i ] != 0; i++ )
   {
      switch( prot->texture[ i ] )
      {
         case 1:  /*  name of picture (or list) */
            DK_stringWriteBinary( file, txt->pic_name );
            break;

         case 2:  /*  animation type */
            nb = txt->animation;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 3:  /*  U repetition */
            DK_byteswap_fwrite( &txt->uCount, 2, 1, file );
            break;

         case 4:  /*  V repetition */
            DK_byteswap_fwrite( &txt->vCount, 2, 1, file );
            break;

         case 5:  /*  U Scaling */
            DK_byteswap_fwrite( &txt->uScal, 4, 1, file );
            break;

         case 6:  /*  V Scaling */
            DK_byteswap_fwrite( &txt->vScal, 4, 1, file );
            break;

         case 7:  /*  U Offset */
            DK_byteswap_fwrite( &txt->uOffset, 4, 1, file );
            break;

         case 8:  /*  V Offset */
            DK_byteswap_fwrite( &txt->vOffset, 4, 1, file );
            break;

         case 9:  /*  interpolation */
            DK_byteswap_fwrite( &txt->pixelInterp, 2, 1, file );
            break;

         case 10: /*   Application method */
            nb = txt->method;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 11: /* ambient factor */
            DK_byteswap_fwrite( &txt->ambientFct, 4, 1, file );
            break;

         case 12: /* diffuse factor */
            DK_byteswap_fwrite( &txt->diffuseFct, 4, 1, file );
            break;

         case 13: /* specular factor */
            DK_byteswap_fwrite( &txt->specularFct, 4, 1, file );
            break;

         case 14: /* roughness factor */
            DK_byteswap_fwrite( &txt->roughnessFct, 4, 1, file );
            break;

         case 15: /* transparency factor */
            DK_byteswap_fwrite( &txt->transpFct, 4, 1, file );
            break;

         case 16: /* reflection map factor */
            DK_byteswap_fwrite( &txt->reflectMappingFct, 4, 1, file );
            break;

         case 17: /* refractive index */
            DK_byteswap_fwrite( &val, 4, 1, file );
            break;

         case 18:    /* environment rotation */
            DK_byteswap_fwrite( &txt->rotation, 4, 1, file );
            break;

         case 19:    /* OLD WAY alpha use */
            nb = 1;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 20:    /* blending factor */
            DK_byteswap_fwrite( &txt->blend, 4, 1, file );
            break;

         case 21:    /* cropping */
            nb = txt->cropping;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 22:    /* cropping umin */
            DK_byteswap_fwrite( &txt->uMinCrop, 2, 2, file );
            break;

         case 23:    /* cropping vmin */
            DK_byteswap_fwrite( &txt->vMinCrop, 2, 2, file );
            break;

         case 24:    /* cropping umax */
            DK_byteswap_fwrite( &txt->uMaxCrop, 2, 2, file );
            break;

         case 25:    /* cropping vmax */
            DK_byteswap_fwrite( &txt->vMaxCrop, 2, 2, file );
            break;

         case 26:    /* swap UV */
            nb = txt->uvSwap;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 27:    /* alt horz */
            nb = txt->uAlternate;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 28:    /* alt vert */
            nb = txt->vAlternate;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 29:    /* reflectivity */
            DK_byteswap_fwrite( &txt->reflectFct, 4, 1, file );
            break;

         case 30: /* mask */
            DK_byteswap_fwrite( &txt->mask, 2, 1, file );
            break;

         case 31: /* effect */
            DK_byteswap_fwrite( &txt->effect, 2, 1, file );
            break;

         case 32:    /* txtsup Rotation */
            DK_byteswap_fwrite( &txt->rot, sizeof(DK_Vector), 1, file );
            break;

         case 33:    /* txtsup Scaling */
            DK_byteswap_fwrite( &txt->scal, sizeof(DK_Vector), 1, file );
            break;

         case 34:    /* txtsup Translation */
            DK_byteswap_fwrite( &txt->trans, sizeof(DK_Vector), 1, file );
            break;
      
         case 35:    /* mapping Black and/or White */
            DK_byteswap_fwrite( &txt->blkwht, sizeof(short), 1, file );
            break;
        
	 case 36:   /* Texture 2D shader */
            DK_udfWriteBinary( file, prot, txt->shader );
            break;

	 case 37:   /* Displacement */
	    DK_byteswap_fwrite( &txt->displacement, sizeof(short), 1, file );
            break;

         case 50:  /*  name */
            DK_stringWriteBinary( file, txt->txt_name );
            break;
   
         default:
	    if ( DK_option.verbose == TRUE)
                fprintf( DK_option.msgFd, "Error: %s %d): illegal token: %hd\n",
                                       __FILE__, __LINE__, prot->texture[ i ] );
            break;
      }
   }

   return( TRUE );
}

DK_Texture *DK_textureReadBinary(  FILE *file, DK_FileProt *prot )
{
   short	 nb, tok, ind = 0;
   DK_Texture	*txt;
   float	 val;
   char		*ptr;

   if( (txt = DK_textureAllocate()) == NULL )
      return( NULL );

   if( (prot != NULL) && (prot->texture != NULL) )
      tok = prot->texture[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );
   while( tok != 0 /* End of texture definition */ )
   {
      switch( tok )
      {
         case 1:  /*  name of picture (or list) */
            txt->pic_name = DK_stringReadBinary( file );
            break;

         case 2:  /*  animation type */
            DK_byteswap_fread( &nb, 2, 1, file );
            txt->animation = (DK_TextureAnim) nb;
            break;

         case 3:  /*  U repetition */
            DK_byteswap_fread( &txt->uCount, 2, 1, file );
            break;

         case 4:  /*  V repetition */
            DK_byteswap_fread( &txt->vCount, 2, 1, file );
            break;

         case 5:  /*  U Scaling */
            DK_byteswap_fread( &txt->uScal, 4, 1, file );
            break;

         case 6:  /*  V Scaling */
            DK_byteswap_fread( &txt->vScal, 4, 1, file );
            break;

         case 7:  /*  U Offset */
            DK_byteswap_fread( &txt->uOffset, 4, 1, file );
            break;

         case 8:  /*  V Offset */
            DK_byteswap_fread( &txt->vOffset, 4, 1, file );
            break;

         case 9:  /*  interpolation */
            DK_byteswap_fread( &txt->pixelInterp, 2, 1, file );
            break;

         case 10: /*   Application method */
            DK_byteswap_fread( &nb, 2, 1, file );
            txt->method = (DK_TextureMethod) nb;
            break;

         case 11: /* ambient factor */
            DK_byteswap_fread( &txt->ambientFct, 4, 1, file );
            break;

         case 12: /* diffuse factor */
            DK_byteswap_fread( &txt->diffuseFct, 4, 1, file );
            break;

         case 13: /* specular factor */
            DK_byteswap_fread( &txt->specularFct, 4, 1, file );
            break;

         case 14: /* roughness factor */
            DK_byteswap_fread( &txt->roughnessFct, 4, 1, file );
            break;

         case 15: /* transparency factor */
            DK_byteswap_fread( &txt->transpFct, 4, 1, file );
            break;

         case 16: /* reflection map factor */
            DK_byteswap_fread( &txt->reflectMappingFct, 4, 1, file );
            break;

         case 17: /* refractive index */
            DK_byteswap_fread( &val, 4, 1, file );
            break;

         case 18:   /* environment rotation */
            DK_byteswap_fread( &txt->rotation, 4, 1, file );
            break;

         case 19:   /* alpha use */
            DK_byteswap_fread( &nb, 2, 1, file );
            if( nb == 1 )
            {
               txt->mask = DK_TXT_ALPHA;
               txt->blend = 1.0;
            }
            break;

         case 20:   /* blending factor */
            DK_byteswap_fread( &txt->blend, 4, 1, file );
            break;

         case 21:   /* cropping */
            DK_byteswap_fread( &nb, 2, 1, file );
            txt->cropping = nb;
            break;

         case 22:   /* cropping umin */
            DK_byteswap_fread( &txt->uMinCrop, 2, 2, file );
            break;

         case 23:   /* cropping vmin */
            DK_byteswap_fread( &txt->vMinCrop, 2, 2, file );
            break;

         case 24:   /* cropping umax */
            DK_byteswap_fread( &txt->uMaxCrop, 2, 2, file );
            break;

         case 25:   /* cropping vmax */
            DK_byteswap_fread( &txt->vMaxCrop, 2, 2, file );
            break;

         case 26:   /* swap UV */
            DK_byteswap_fread( &nb, 2, 1, file );
            txt->uvSwap = nb;
            break;

         case 27:   /* alt horz */
            DK_byteswap_fread( &nb, 2, 1, file );
            txt->uAlternate = nb;
            break;

         case 28:   /* alt vert */
            DK_byteswap_fread( &nb, 2, 1, file );
            txt->vAlternate = nb;
            break;

         case 29:   /* reflectivity */
            DK_byteswap_fread( &txt->reflectFct, 4, 1, file );
            break;

         case 30: /* mask */
            DK_byteswap_fread( &txt->mask, 2, 1, file );
            break;

         case 31: /* effect */
            DK_byteswap_fread( &txt->effect, 2, 1, file );
            break;

         case 32:    /* txtsup Rotation */
            DK_Vector_fread( &txt->rot, sizeof(DK_Vector), 1, file );
            break;

         case 33:    /* txtsup Scaling */
            DK_Vector_fread( &txt->scal, sizeof(DK_Vector), 1, file );
            break;

         case 34:    /* txtsup Translation */
            DK_Vector_fread( &txt->trans, sizeof(DK_Vector), 1, file );
            break;
      
         case 35:    /* mapping Black and/or White */
            DK_byteswap_fread( &txt->blkwht, sizeof(short), 1, file );
            break;
      
         case 36:   /* Texture 2D shader */
            txt->shader = DK_udfReadBinary( file, prot );
            break;

         case 37:   /* Displacement */
            DK_byteswap_fread( &txt->displacement, sizeof(short), 1, file );
            break;
      
         case 50:    /* global name */
            txt->txt_name = DK_stringReadBinary( file );
            break;

         default:
            DK_fileSkipField( file, tok );
      }
      if( (prot != NULL) && (prot->texture != NULL) )
         tok = prot->texture[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }
   return( txt );
}

static DK_TextureAnim DK_textureGetAnimationByName( DK_String name )
{
   DK_TextureAnim	 type = DK_TXT_STATIC;
   int		 i;


   for( i = 0; i < 3; i++ )
      if( strcmp( name, DK_TextureAnimName[ i ] ) == 0 )
         type = (DK_TextureAnim) i;

   return( type );
}

static DK_TextureMethod DK_textureGetMethodByName( DK_String name )
{
   DK_TextureMethod	 type = DK_TXT_XY;
   int		 i;


   for( i = 0; i < 9; i++ )
      if( strcmp( name, DK_TextureMethodName[ i ] ) == 0 )
         type = (DK_TextureMethod) i;

   return( type );
}


DK_Texture *DK_textureReadFile( DK_String filename )
{
   char		 buf[256];
   long          id;
    FILE		*file;

   sprintf( buf, "%s.txt", filename );
   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( NULL );

   DK_byteswap_fread( &id, 4, 1, file );
   fclose( file );

   if( id == 'TXMP' )
      return( DK_textureReadAsciiFile( filename ) );
   else if( id == DK_BINARY_MAGIC_NUMBER || id == DK_MAGIC_T2D_BIN )
      return( DK_textureReadBinaryFile( filename ) );
   else
      return( NULL );
}

DK_Boolean DK_textureWriteFile( DK_String filename, DK_Texture *txt , DK_FileFormat fmt )
{
   if( fmt ==  DK_FIL_ASCII )
      return( DK_textureWriteAsciiFile( filename, txt ) );
   else if( fmt ==  DK_FIL_BINARY )
      return( DK_textureWriteBinaryFile( filename, txt ) );
   else
      return( FALSE );
}

static DK_Boolean DK_textureWriteAsciiFile( DK_String filename, DK_Texture *txt )
{
    FILE		*file;
   char		buf[256];
   DK_Boolean      success;

   sprintf( buf, "%s.txt", filename );
   if( ( file = fopen( buf, "wb" ) ) == NULL )
      return( FALSE );

   success = DK_headerWriteAscii( file, "TXMP" );
   success &= DK_textureWriteAscii( file, txt, "", 0 );
   /* Verify if file was saved correctly */
   fflush( file );
   success = (ferror( file ) == 0 );

   fclose( file );
   return( success );
}

DK_Boolean DK_fdtextureWriteBinaryFile( FILE *file, DK_Texture *txt )
{
   DK_FileProt	*prot;
   DK_Boolean	 success;

   DK_newheaderWriteBinary( file, DK_MAGIC_T2D_BIN, NULL );
   DK_prototypeInitTexture( &prot, TRUE );
   DK_prototypeWrite( file, prot );
   DK_String_fwrite( "TXMP", 4, 1, file );
   success = DK_textureWriteBinary( file, txt, prot );
   DK_prototypeDispose( &prot );
   return( success );
}

static DK_Boolean DK_textureWriteBinaryFile( DK_String filename, DK_Texture *txt )
{
    FILE		*file;
   char		 buf[256];
   DK_FileProt	*prot;
   DK_Boolean	 success;

   sprintf( buf, "%s.txt", filename );
   if( ( file = fopen( buf, "wb" ) ) == NULL )
      return( FALSE );

   DK_fdtextureWriteBinaryFile( file, txt );

   /* Verify if file was saved correctly */
   fflush( file );
   success = (ferror( file ) == 0 );
   fclose( file );
   return( success );
}

static DK_Texture *DK_textureReadAsciiFile( DK_String filename )
{
   char		 str[ 80 ];
    FILE		*file;
   DK_Texture	*txt;
   long		 id;
   float	 version;

   sprintf( str, "%s.txt", filename );
   if( ( file = fopen( str, "rb" ) ) == NULL )
      return( NULL );

   if( (id = DK_headerReadAscii( file, &version )) != 'TXMP' )
      return( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str , "texture" ) == 0 )
      txt = DK_textureReadAscii( file );

   fclose( file );

   return( txt );
}

DK_Texture *DK_fdtextureReadBinaryFile( FILE *file )
{
   float	 version;
   long		 id;
   DK_FileProt	*prot;
   DK_Texture	*txt;

   id = DK_headerReadBinary( file, &version, NULL );
   if( id == 'PROT' )
   {
      prot = DK_prototypeRead( file );
      DK_byteswap_fread( &id, 4, 1, file );
   }
   if( id == 'TXMP' )
      txt = DK_textureReadBinary( file, prot );

   DK_prototypeDispose( &prot );

   return( txt );
}

static DK_Texture *DK_textureReadBinaryFile( DK_String filename )
{
   char		 buf[256];
   FILE		*file;
   DK_Texture	*txt;

   sprintf( buf, "%s.txt", filename );
   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( NULL );

   txt = DK_fdtextureReadBinaryFile( file );

   fclose( file );
   return( txt );
}



/******************************************************************************/
/*** 3D Procedurals Textures **************************************************/
/******************************************************************************/

DK_Texture3D *DK_texture3DAllocate( void )
{
   DK_Texture3D *txt;
   long i;

   if( txt = (DK_Texture3D *) calloc( 1, sizeof( DK_Texture3D ) ) )
   {
      txt->name           = NULL;
      txt->prefix         = NULL;
      txt->type           = DK_TXT_3D_MARBLE;
      txt->spacing        = 1.0;
      txt->taper          = 0.0;
      txt->strength       = 0.0;
      txt->nbIterations   = 1;
      txt->power          = 1.0;

      txt->mask           = DK_TXT_NO_MASK;
      txt->effect         = DK_TXT_ALPHA;
      txt->blend          = 1.0;

      txt->colour0.r = txt->colour0.g = txt->colour0.b = txt->colour0.a = 0.0;
      txt->colour1.r = txt->colour1.g = txt->colour1.b = txt->colour1.a = 0.0;
      txt->colour2.r = txt->colour2.g = txt->colour2.b = txt->colour2.a = 0.0;
      txt->colour3.r = txt->colour3.g = txt->colour3.b = txt->colour3.a = 0.0;
      txt->colour4.r = txt->colour4.g = txt->colour4.b = txt->colour4.a = 0.0;

      txt->factor1 = 0.25;
      txt->factor2 = 0.50;
      txt->factor3 = 0.75;

      txt->scaling.x     = txt->scaling.y     = txt->scaling.z     = 1.0;
      txt->rotation.x    = txt->rotation.y    = txt->rotation.z    = 0.0;
      txt->translation.x = txt->translation.y = txt->translation.z = 0.0;

      txt->ambientFct    = 0.0;
      txt->diffuseFct    = 0.0;
      txt->specularFct   = 0.0;
      txt->reflectFct    = 0.0;
      txt->transpFct     = 0.0;
      txt->roughnessFct  = 0.0;

      for (i = 0; i < DK_FCV_TXT3D_NUM; i++)
      {
	 txt->fcv[i] = (DK_Fcurve *)NULL;
	 txt->active_fcv[i].active = FALSE;
	 txt->active_fcv[i].fcv = (DK_Fcurve *)NULL;
      }

      txt->shader        = NULL;
      txt->displacement  = FALSE;

      txt->constr_list	 = NULL;

      txt->next = NULL;

      txt->schem_posx = 0.0;
      txt->schem_posy = 0.0;
      txt->schem_userppos = 0;
   }

   return( txt );
}

void DK_texture3DDispose( DK_Texture3D **txt )
{
   if( txt )
   {
      DK_Texture3D *texture = *txt;

      while( texture )
      {
         DK_Texture3D *save = texture;

         _DK_FREE( texture->name, -1 );
         _DK_FREE( texture->prefix, -1 );
         DK_fcurveArrayDispose( texture->fcv, DK_FCV_TXT3D_NUM );
         DK_actFcurveArrayDispose( texture->active_fcv, DK_FCV_TXT3D_NUM );
         DK_udfDispose( &texture->shader );
	 DK_constraintDispose( &texture->constr_list );

         texture = texture->next;
         free( save );
      }

      *txt = NULL;
   }
}

DK_Boolean DK_texture3DWriteBinary(  FILE *file, DK_Texture3D *txt, DK_FileProt *prot )
{
   short	 tok, nb, i;
   float	 tmp;

   for( i = 0; prot->texture3D[ i ] != 0; i++ )
   {
      tok = prot->texture3D[ i ];

      switch( tok )
      {
         case 1:  /* name of texture */
            DK_stringWriteBinary( file, txt->name );
            break;

         case 2: /* 3D texture type */
            nb = (short) txt->type;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 3: /* Number of iteration */
            nb = txt->nbIterations;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 4: /* strength */
            DK_byteswap_fwrite( &txt->strength, 4, 1, file );
            break;

         case 5: /* spacing */
            DK_byteswap_fwrite( &txt->spacing, 4, 1, file );
            break;

         case 6: /* stretch */
            tmp = 1.0;
            DK_byteswap_fwrite( &tmp, 4, 1, file );
            break;

         case 22: /* color 1 (RGBA) */
            DK_RGBA_fwrite( &txt->colour0, sizeof( DK_Colour32 ), 1, file );
            break;

         case 23: /* color 2 (RGBA) */
            DK_RGBA_fwrite( &txt->colour1, sizeof( DK_Colour32 ), 1, file );
            break;

         case 24: /* color 3 (RGBA) */
            DK_RGBA_fwrite( &txt->colour2, sizeof( DK_Colour32 ), 1, file );
            break;

         case 25: /* color 4 (RGBA) */
            DK_RGBA_fwrite( &txt->colour3, sizeof( DK_Colour32 ), 1, file );
            break;

         case 26: /* color 5 (RGBA) */
            DK_RGBA_fwrite( &txt->colour4, sizeof( DK_Colour32 ), 1, file );
            break;

         case 7: /* color 1 (RGB) */
            DK_RGB_fwrite( (DK_Colour24 *) (&txt->colour0), sizeof( DK_Colour24 ), 1, file );
            break;

         case 8: /* color 2 (RGB) */
            DK_RGB_fwrite(  (DK_Colour24 *) (&txt->colour2), sizeof( DK_Colour24 ), 1, file );
            break;

         case 9: /* color 3 (RGB) */
            DK_RGB_fwrite( (DK_Colour24 *) (&txt->colour4), sizeof( DK_Colour24 ), 1, file );
            break;

         case 10: /* factor 2 */
            DK_byteswap_fwrite( &txt->factor1, 4, 1, file );
            break;

         case 27: /* factor 3 */
            DK_byteswap_fwrite( &txt->factor2, 4, 1, file );
            break;

         case 28: /* factor 4 */
            DK_byteswap_fwrite( &txt->factor3, 4, 1, file );
            break;

         case 11: /* Scaling */
            DK_byteswap_fwrite( &txt->scaling, sizeof( DK_Vector ), 1, file );
            break;

         case 12: /* rotation */
            DK_byteswap_fwrite( &txt->rotation, sizeof( DK_Vector ), 1, file );
            break;

         case 13: /* translation */
            DK_byteswap_fwrite( &txt->translation, sizeof( DK_Vector ), 1, file );
            break;

         case 14: /* ambient factor */
            DK_byteswap_fwrite( &txt->ambientFct, 4, 1, file );
            break;

         case 15: /* diffuse factor */
            DK_byteswap_fwrite( &txt->diffuseFct, 4, 1, file );
            break;

         case 16: /* specular factor */
            DK_byteswap_fwrite( &txt->specularFct, 4, 1, file );
            break;

         case 17: /* roughness factor */
            DK_byteswap_fwrite( &txt->roughnessFct, 4, 1, file );
            break;

         case 18: /* transparency factor */
            DK_byteswap_fwrite( &txt->transpFct, 4, 1, file );
            break;

         case 19: /* reflection factor */
            DK_byteswap_fwrite( &txt->reflectFct, 4, 1, file );
            break;

         case 20: /* refractive index */
            DK_byteswap_fwrite( &tmp, 4, 1, file );
            break;

         case 21: /* power */
            DK_byteswap_fwrite( &txt->power, 4, 1, file );
            break;

         case 29: /* taper */
            DK_byteswap_fwrite( &txt->taper, 4, 1, file );
            break;

         case 30: /* use alpha */
            nb = 1;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 31: /* blending */
            DK_byteswap_fwrite( &txt->blend, 4, 1, file );
            break;

         case 32: /* mask */
            DK_byteswap_fwrite( &txt->mask, 2, 1, file );
            break;

         case 33: /* effect */
            DK_byteswap_fwrite( &txt->effect, 2, 1, file );
            break;

	 case 34:   /* Texture 3D shader */
            DK_udfWriteBinary( file, prot, txt->shader );
            break;

	 case 35:   /* Displacement */
	    DK_byteswap_fwrite( &txt->displacement, sizeof(short), 1, file );
            break;
   
      }
   }

   return( TRUE );
}

DK_Texture3D *DK_texture3DReadBinary(  FILE *file, DK_FileProt *prot )
{
   short	 tok, nb, ind = 0;
   DK_Texture3D	*txt;
   float	 tmp;
   DK_Boolean	 old = FALSE;

   if( (txt = DK_texture3DAllocate()) == NULL )
      return( NULL );

   if( (prot != NULL) && (prot->texture3D != NULL) )
      tok = prot->texture3D[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );

   while( tok != 0 /* End of texture 3D definition */ )
   {
      switch( tok )
      {
         case 1:  /* name of texture */
            txt->name = DK_stringReadBinary( file );
            break;

         case 2: /* 3D texture type */
            DK_byteswap_fread( &nb, 2, 1, file );
            txt->type = (DK_Texture3DType) nb;
            break;

         case 3: /* Number of iteration */
            DK_byteswap_fread( &nb, 2, 1, file );
            txt->nbIterations = nb;
            break;

         case 4: /* strength */
            DK_byteswap_fread( &txt->strength, 4, 1, file );
            break;

         case 5: /* spacing */
            DK_byteswap_fread( &txt->spacing, 4, 1, file );
            break;

         case 6: /* stretch */
            DK_byteswap_fread( &tmp, 4, 1, file );
            break;

         case 22: /* color 1 (RGBA) */
            DK_RGBA_fread( &txt->colour0, sizeof( DK_Colour32 ), 1, file );
            break;

         case 23: /* color 2 (RGBA) */
            DK_RGBA_fread( &txt->colour1, sizeof( DK_Colour32 ), 1, file );
            break;

         case 24: /* color 3 (RGBA) */
            DK_RGBA_fread( &txt->colour2, sizeof( DK_Colour32 ), 1, file );
            break;

         case 25: /* color 4 (RGBA) */
            DK_RGBA_fread( &txt->colour3, sizeof( DK_Colour32 ), 1, file );
            break;

         case 26: /* color 5 (RGBA) */
            DK_RGBA_fread( &txt->colour4, sizeof( DK_Colour32 ), 1, file );
            break;

         case 7: /* color 1 (RGB) */
            DK_RGB_fread( (DK_Colour24 *) (&txt->colour0), sizeof( DK_Colour24 ), 1, file );
            txt->colour0.a = 0.0;
            old = TRUE;
            break;

         case 8: /* color 2 (RGB) */
            DK_RGB_fread( (DK_Colour24 *) (&txt->colour2), sizeof( DK_Colour24 ), 1, file );
            txt->colour2.a = 0.0;
            break;

         case 9: /* color 3 (RGB) */
            DK_RGB_fread( (DK_Colour24 *) (&txt->colour4), sizeof( DK_Colour24 ), 1, file );
            txt->colour4.a = 0.0;
            break;

         case 10: /* factor 2 */
            DK_byteswap_fread( &txt->factor1, 4, 1, file );
            break;

         case 27: /* factor 3 */
            DK_byteswap_fread( &txt->factor2, 4, 1, file );
            break;

         case 28: /* factor 4 */
            DK_byteswap_fread( &txt->factor3, 4, 1, file );
            break;

         case 11: /* Scaling */
            DK_Vector_fread( &txt->scaling, sizeof( DK_Vector ), 1, file );
            break;

         case 12: /* rotation */
            DK_Vector_fread( &txt->rotation, sizeof( DK_Vector ), 1, file );
            break;

         case 13: /* translation */
            DK_Vector_fread( &txt->translation, sizeof( DK_Vector ), 1, file );
            break;

         case 14: /* ambient factor */
            DK_byteswap_fread( &txt->ambientFct, 4, 1, file );
            break;

         case 15: /* diffuse factor */
            DK_byteswap_fread( &txt->diffuseFct, 4, 1, file );
            break;

         case 16: /* specular factor */
            DK_byteswap_fread( &txt->specularFct, 4, 1, file );
            break;

         case 17: /* roughness factor */
            DK_byteswap_fread( &txt->roughnessFct, 4, 1, file );
            break;

         case 18: /* transparency factor */
            DK_byteswap_fread( &txt->transpFct, 4, 1, file );
            break;

         case 19: /* reflection factor */
            DK_byteswap_fread( &txt->reflectFct, 4, 1, file );
            break;

         case 20: /* refractive index */
            DK_byteswap_fread( &tmp, 4, 1, file );
            break;

         case 21: /* power */
            DK_byteswap_fread( &txt->power, 4, 1, file );
            break;

         case 29: /* taper */
            DK_byteswap_fread( &txt->taper, 4, 1, file );
            break;

         case 30: /* use alpha */
            DK_byteswap_fread( &nb, 2, 1, file );
            if( nb == 1 )
            {
               txt->mask = DK_TXT_ALPHA;
               txt->blend = 1.0;
            }
            break;

         case 31: /* blending */
            DK_byteswap_fread( &txt->blend, 4, 1, file );
            break;

         case 32: /* mask */
            DK_byteswap_fread( &txt->mask, 2, 1, file );
            break;

         case 33: /* effect */
            DK_byteswap_fread( &txt->effect, 2, 1, file );
            break;

         case 34:   /* Texture 3D shader */
            txt->shader = DK_udfReadBinary( file, prot );
            break;

         case 35:   /* Displacement */
            DK_byteswap_fread( &txt->displacement, sizeof(short), 1, file );
            break;
      
         default:
            DK_fileSkipField( file, tok );
      }
      if( (prot != NULL) && (prot->texture3D != NULL) )
         tok = prot->texture3D[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   /* Initialisations pour garder la compatibilite avec les vieux DK_modeles. */

   if ( old )
   {
      txt->colour1.r = ( txt->colour0.r + txt->colour2.r ) / 2.0;
      txt->colour1.g = ( txt->colour0.g + txt->colour2.g ) / 2.0;
      txt->colour1.b = ( txt->colour0.b + txt->colour2.b ) / 2.0;
      txt->colour1.a = 0.0;

      txt->colour3.r = ( txt->colour2.r + txt->colour4.r ) / 2.0;
      txt->colour3.g = ( txt->colour2.g + txt->colour4.g ) / 2.0;
      txt->colour3.b = ( txt->colour2.b + txt->colour4.b ) / 2.0;
      txt->colour3.a = 0.0;

      txt->factor2 = txt->factor1;
      txt->factor1 = txt->factor2 / 2.0;
      txt->factor3 = txt->factor2 + ( 1.0 - txt->factor2 ) / 2.0;
   }

   return( txt );
}

DK_Boolean DK_texture3DWriteAscii(  FILE *file, DK_Texture3D *txt, DK_String space, short no)
{
   fprintf( file, "\n%stexture3D   [%1hd]\n", space, no );
   fprintf( file, "%s{\n", space );
   fprintf( file, "%s  name       \"%s\"\n", space, txt->name );
   fprintf( file, "%s  type       %s\n", space,
                                         DK_Texture3DTypeName[ (int) txt->type ] );
   fprintf( file, "%s  nbIterations    %hd\n", space, txt->nbIterations );
   fprintf( file, "%s  strength   %*.*f\n", space,  DK_FM,  DK_FD, txt->strength );
   fprintf( file, "%s  spacing    %*.*f\n", space,  DK_FM,  DK_FD, txt->spacing );
   fprintf( file, "%s  power      %*.*f\n", space,  DK_FM,  DK_FD, txt->power );
   fprintf( file, "%s  taper      %*.*f\n", space,  DK_FM,  DK_FD, txt->taper );
   switch( txt->effect )
   {
      case DK_TXT_ALPHA:
         fprintf( file, "%s  effect     ALPHA\n", space );
         break;
      case DK_TXT_INTENSITY:
         fprintf( file, "%s  effect     INTENSITY\n", space );
         break;
   }
   switch( txt->mask )
   {
      case DK_TXT_ALPHA:
         fprintf( file, "%s  mask       ALPHA\n", space );
         break;
      case DK_TXT_INTENSITY:
         fprintf( file, "%s  mask       INTENSITY\n", space );
         break;
   }
   fprintf( file, "%s  blending   %*.*f\n", space,  DK_FM,  DK_FD, txt->blend );
   fprintf( file, "%s  colour0   %*.*f %*.*f %*.*f %*.*f\n", space,  DK_FM,  DK_FD,
                          txt->colour0.r,  DK_FM,  DK_FD, txt->colour0.g,  DK_FM,  DK_FD,
                          txt->colour0.b,  DK_FM,  DK_FD, txt->colour0.a );
   fprintf( file, "%s  colour1   %*.*f %*.*f %*.*f %*.*f\n", space,  DK_FM,  DK_FD,
                          txt->colour1.r,  DK_FM,  DK_FD, txt->colour1.g,  DK_FM,  DK_FD,
                          txt->colour1.b,  DK_FM,  DK_FD, txt->colour1.a );
   fprintf( file, "%s  factor1   %*.*f\n", space,  DK_FM,  DK_FD, txt->factor1 );
   fprintf( file, "%s  colour2   %*.*f %*.*f %*.*f %*.*f\n", space,  DK_FM,  DK_FD,
                          txt->colour2.r,  DK_FM,  DK_FD, txt->colour2.g,  DK_FM,  DK_FD,
                          txt->colour2.b,  DK_FM,  DK_FD, txt->colour2.a );
   fprintf( file, "%s  factor2   %*.*f\n", space,  DK_FM,  DK_FD, txt->factor2 );
   fprintf( file, "%s  colour3   %*.*f %*.*f %*.*f %*.*f\n", space,  DK_FM,  DK_FD,
                          txt->colour3.r,  DK_FM,  DK_FD, txt->colour3.g,  DK_FM,  DK_FD,
                          txt->colour3.b,  DK_FM,  DK_FD, txt->colour3.a );
   fprintf( file, "%s  factor3   %*.*f\n", space,  DK_FM,  DK_FD, txt->factor3 );
   fprintf( file, "%s  colour4   %*.*f %*.*f %*.*f %*.*f\n", space,  DK_FM,  DK_FD,
                          txt->colour4.r,  DK_FM,  DK_FD, txt->colour4.g,  DK_FM,  DK_FD,
                          txt->colour4.b,  DK_FM,  DK_FD, txt->colour4.a );
   fprintf( file, "%s  scaling    %*.*f  %*.*f  %*.*f\n", space,  DK_FM,  DK_FD,
            txt->scaling.x,  DK_FM,  DK_FD, txt->scaling.y,  DK_FM,  DK_FD, txt->scaling.z );
   fprintf( file, "%s  rotation   %*.*f  %*.*f  %*.*f\n", space,  DK_FM,  DK_FD,
            txt->rotation.x,  DK_FM,  DK_FD, txt->rotation.y,  DK_FM,  DK_FD, txt->rotation.z );
   fprintf( file, "%s  translation  %*.*f  %*.*f  %*.*f\n", space,  DK_FM,  DK_FD,
            txt->translation.x,  DK_FM,  DK_FD, txt->translation.y,  DK_FM,  DK_FD,
            txt->translation.z );

   fprintf( file, "%s  ambient    %*.*f\n", space,  DK_FM,  DK_FD, txt->ambientFct );
   fprintf( file, "%s  diffuse    %*.*f\n", space,  DK_FM,  DK_FD, txt->diffuseFct );
   fprintf( file, "%s  specular   %*.*f\n", space,  DK_FM,  DK_FD, txt->specularFct );
   fprintf( file, "%s  reflect    %*.*f\n", space,  DK_FM,  DK_FD, txt->reflectFct );
   fprintf( file, "%s  transp     %*.*f\n", space,  DK_FM,  DK_FD, txt->transpFct );
   fprintf( file, "%s  roughness  %*.*f\n", space,  DK_FM,  DK_FD, txt->roughnessFct );

   if( txt->shader )
      DK_udfWriteAscii( file, txt->shader, space, "txt3d_shader" );
   
   if( txt->displacement )
       fprintf( file, "%s  displacement\n", space);
	       

   fprintf( file, "%s}\n", space );

   return( TRUE );
}

DK_Texture3D *DK_texture3DReadAscii(  FILE *file )
{
   char		 str[ 80 ], str2[ 80 ];
   DK_Texture3D   *txt;

   if( (txt = DK_texture3DAllocate()) == NULL )
      return( NULL );

   /* get rid of number */
   fscanf( file, "%s", str );

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( txt );

   do
   {
      fscanf( file, "%s", str );
      if( strcmp( str, "name" ) == 0 )
         txt->name = DK_stringReadAscii( file );
      else if( strcmp( str, "type" ) == 0 )
      {
         fscanf( file, "%s", str2 );
         txt->type = DK_texture3DGetTypeByName( str2 );
      }
      else if( strcmp( str, "nbIterations" ) == 0 )
         fscanf( file, "%hd", &txt->nbIterations );
      else if( strcmp( str, "strength" ) == 0 )
         fscanf( file, "%f", &txt->strength );
      else if( strcmp( str, "spacing" ) == 0 )
         fscanf( file, "%f", &txt->spacing );
      else if( strcmp( str, "power" ) == 0 )
         fscanf( file, "%f", &txt->power );
      else if( strcmp( str, "taper" ) == 0 )
         fscanf( file, "%f", &txt->taper );
      else if( strcmp( str, "alpha" ) == 0 )
      {
         /* OLD way (DO NOT use it) */
         txt->mask = DK_TXT_ALPHA;
         txt->blend = 1.0;
      }
      else if( strcmp( str, "mask" ) == 0 )
      {
         fscanf( file, "%s", str2 );
         if( strcmp( "ALPHA", str2 ) == 0 )
            txt->mask = DK_TXT_ALPHA;
         else if( strcmp( "INTENSITY", str2 ) == 0 )
            txt->mask = DK_TXT_INTENSITY;
      }
      else if( strcmp( str, "effect" ) == 0 )
      {
         fscanf( file, "%s", str2 );
         if( strcmp( "ALPHA", str2 ) == 0 )
            txt->effect = DK_TXT_ALPHA;
         else if( strcmp( "INTENSITY", str2 ) == 0 )
            txt->effect = DK_TXT_INTENSITY;
      }
      else if( strcmp( str, "blending" ) == 0 )
         fscanf( file, "%f", &txt->blend );
      else if( strcmp( str, "colour0" ) == 0 )
         fscanf( file, "%f %f %f %f", &txt->colour0.r, &txt->colour0.g,
                                      &txt->colour0.b, &txt->colour0.a );
      else if( strcmp( str, "colour1" ) == 0 )
         fscanf( file, "%f %f %f %f", &txt->colour1.r, &txt->colour1.g,
                                      &txt->colour1.b, &txt->colour1.a );
      else if( strcmp( str, "factor1" ) == 0 )
         fscanf( file, "%f", &txt->factor1 );
      else if( strcmp( str, "colour2" ) == 0 )
         fscanf( file, "%f %f %f %f", &txt->colour2.r, &txt->colour2.g,
                                      &txt->colour2.b, &txt->colour2.a );
      else if( strcmp( str, "factor2" ) == 0 )
         fscanf( file, "%f", &txt->factor2 );
      else if( strcmp( str, "colour3" ) == 0 )
         fscanf( file, "%f %f %f %f", &txt->colour3.r, &txt->colour3.g,
                                      &txt->colour3.b, &txt->colour3.a );
      else if( strcmp( str, "factor3" ) == 0 )
         fscanf( file, "%f", &txt->factor3 );
      else if( strcmp( str, "colour4" ) == 0 )
         fscanf( file, "%f %f %f %f", &txt->colour4.r, &txt->colour4.g,
                                      &txt->colour4.b, &txt->colour4.a );
      else if( strcmp( str, "scaling" ) == 0 )
         fscanf( file, "%f %f %f", &txt->scaling.x, &txt->scaling.y,
                                                    &txt->scaling.z );
      else if( strcmp( str, "rotation" ) == 0 )
         fscanf( file, "%f %f %f", &txt->rotation.x, &txt->rotation.y,
                                                    &txt->rotation.z );
      else if( strcmp( str, "translation" ) == 0 )
         fscanf( file, "%f %f %f", &txt->translation.x, &txt->translation.y,
                                                        &txt->translation.z );
      else if( strcmp( str, "ambient" ) == 0 )
         fscanf( file, "%f", &txt->ambientFct );
      else if( strcmp( str, "diffuse" ) == 0 )
         fscanf( file, "%f", &txt->diffuseFct );
      else if( strcmp( str, "specular" ) == 0 )
         fscanf( file, "%f", &txt->specularFct );
      else if( strcmp( str, "reflect" ) == 0 )
         fscanf( file, "%f", &txt->reflectFct );
      else if( strcmp( str, "transp" ) == 0 )
         fscanf( file, "%f", &txt->transpFct );
      else if( strcmp( str, "roughness" ) == 0 )
         fscanf( file, "%f", &txt->roughnessFct );
      else if( strcmp( str, "txt3d_shader" ) == 0 )
	 txt->shader = DK_udfReadAscii( file );
      else if( strcmp( str, "displacement" ) == 0 )
         txt->displacement = TRUE;

   } while( strcmp( str, "}" ) != 0 );

   return( txt );

}

static DK_Texture3DType DK_texture3DGetTypeByName( DK_String name )
{
   DK_Texture3DType	 type = DK_TXT_3D_MARBLE;
   int		 i;


   for( i = 0; i < 3; i++ )
      if( strcmp( name, DK_Texture3DTypeName[ i ] ) == 0 )
         type = (DK_Texture3DType) i;

   return( type );
}


DK_Texture3D *DK_texture3DReadFile( DK_String filename )
{
   char		 buf[256];
   long          id;
    FILE		*file;

   sprintf( buf, "%s.t3D", filename );
   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( NULL );

   DK_byteswap_fread( &id, 4, 1, file );
   fclose( file );

   if( id == 'TX3D' )
      return( DK_texture3DReadAsciiFile( filename ) );
   else if( id == DK_BINARY_MAGIC_NUMBER || id == DK_MAGIC_T3D_BIN )
      return( DK_texture3DReadBinaryFile( filename ) );
   else
      return( NULL );
}

DK_Boolean DK_texture3DWriteFile( DK_String filename, DK_Texture3D *txt , DK_FileFormat fmt )
{
   if( fmt ==  DK_FIL_ASCII )
      return( DK_texture3DWriteAsciiFile( filename, txt ) );
   else if( fmt ==  DK_FIL_BINARY )
      return( DK_texture3DWriteBinaryFile( filename, txt ) );
   else
      return( FALSE );
}

static DK_Boolean DK_texture3DWriteAsciiFile( DK_String filename, DK_Texture3D *txt )
{
    FILE		*file;
   char		buf[256];
   DK_Boolean      success;

   sprintf( buf, "%s.t3D", filename );
   if( ( file = fopen( buf, "wb" ) ) == NULL )
      return( FALSE );

   success = DK_headerWriteAscii( file, "TX3D" );
   success &= DK_texture3DWriteAscii( file, txt, "", 0 );
   /* Verify if file was saved correctly */
   fflush( file );
   success = (ferror( file ) == 0 );

   fclose( file );
   return( success );
}

DK_Boolean DK_fdtexture3DWriteBinaryFile( FILE *file, DK_Texture3D *txt )
{
   DK_FileProt	*prot = NULL;
   DK_Boolean	 success;

   DK_newheaderWriteBinary( file, DK_MAGIC_T3D_BIN, NULL );
   DK_prototypeInitTexture3D( &prot, TRUE );
   DK_prototypeWrite( file, prot );
   DK_String_fwrite( "TX3D", 4, 1, file );
   success = DK_texture3DWriteBinary( file, txt, prot );
   DK_prototypeDispose( &prot );
   return( success );
}

static DK_Boolean DK_texture3DWriteBinaryFile( DK_String filename, DK_Texture3D *txt )
{
   FILE		*file;
   char		 buf[256];
   DK_Boolean	 success;

   sprintf( buf, "%s.t3D", filename );
   if( ( file = fopen( buf, "wb" ) ) == NULL )
      return( FALSE );

   DK_fdtexture3DWriteBinaryFile( file, txt );

   /* Verify if file was saved correctly */
   fflush( file );
   success = (ferror( file ) == 0 );
   fclose( file );
   return( success );
}

static DK_Texture3D *DK_texture3DReadAsciiFile( DK_String filename )
{
   char		 str[ 80 ];
    FILE		*file;
   DK_Texture3D	*txt;
   long		 id;
   float	 version;

   sprintf( str, "%s.t3D", filename );
   if( ( file = fopen( str, "rb" ) ) == NULL )
      return( NULL );

   if( (id = DK_headerReadAscii( file, &version )) != 'TX3D' )
      return( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str , "texture3D" ) == 0 )
      txt = DK_texture3DReadAscii( file );

   fclose( file );

   return( txt );
}

DK_Texture3D *DK_fdtexture3DReadBinaryFile( FILE *file )
{
   float	 version;
   long		 id;
   DK_FileProt	*prot;
   DK_Texture3D	*txt;

   id = DK_headerReadBinary( file, &version, NULL );
   if( id == 'PROT' )
   {
      prot = DK_prototypeRead( file );
      DK_byteswap_fread( &id, 4, 1, file );
   }
   if( id == 'TX3D' )
      txt = DK_texture3DReadBinary( file, prot );

   DK_prototypeDispose( &prot );
   return( txt );
}

static DK_Texture3D *DK_texture3DReadBinaryFile( DK_String filename )
{
   char		 buf[256];
   FILE		*file;
   DK_Texture3D	*txt;

   sprintf( buf, "%s.t3D", filename );
   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( NULL );

   txt = DK_fdtexture3DReadBinaryFile( file );

   fclose( file );
   return( txt );
}

/*******************************************************************************

   $$L DK_texture3DWriteAsciiAnimation

   This function writes a 3D texture.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_texture3DWriteAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Texture3D *txt		/* 3d texture pointer */
    )
{
    if (txt == NULL)
	return(FALSE);

    DK_incIndent(1);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_NAME_TOKEN);
    _DK_OUT_STRING(txt->name);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_STDK_REN_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->strength, txt->fcv[DK_FCV_TXT3D_STR]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_SPAC_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->spacing, txt->fcv[DK_FCV_TXT3D_SPA]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL0_TOKEN);
    DK_value3WriteAsciiAnimation(file, txt->colour0.r, txt->colour0.g, txt->colour0.b,
	txt->fcv[DK_FCV_TXT3D_COL0R], txt->fcv[DK_FCV_TXT3D_COL0G], 
	txt->fcv[DK_FCV_TXT3D_COL0B]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL0A_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->colour0.a, txt->fcv[DK_FCV_TXT3D_COL0A]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL1_TOKEN);
    DK_value3WriteAsciiAnimation(file, txt->colour1.r, txt->colour1.g, txt->colour1.b,
	txt->fcv[DK_FCV_TXT3D_COL1R], txt->fcv[DK_FCV_TXT3D_COL1G], 
	txt->fcv[DK_FCV_TXT3D_COL1B]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL1A_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->colour1.a, txt->fcv[DK_FCV_TXT3D_COL1A]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL2_TOKEN);
    DK_value3WriteAsciiAnimation(file, txt->colour2.r, txt->colour2.g, txt->colour2.b,
	txt->fcv[DK_FCV_TXT3D_COL2R], txt->fcv[DK_FCV_TXT3D_COL2G],
	txt->fcv[DK_FCV_TXT3D_COL2B]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL2A_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->colour2.a, txt->fcv[DK_FCV_TXT3D_COL2A]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL3_TOKEN);
    DK_value3WriteAsciiAnimation(file, txt->colour3.r, txt->colour3.g, txt->colour3.b,
	txt->fcv[DK_FCV_TXT3D_COL3R], txt->fcv[DK_FCV_TXT3D_COL3G],
	txt->fcv[DK_FCV_TXT3D_COL3B]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL3A_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->colour3.a, txt->fcv[DK_FCV_TXT3D_COL3A]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL4_TOKEN);
    DK_value3WriteAsciiAnimation(file, txt->colour4.r, txt->colour4.g, txt->colour4.b,
	txt->fcv[DK_FCV_TXT3D_COL4R], txt->fcv[DK_FCV_TXT3D_COL4G],
	txt->fcv[DK_FCV_TXT3D_COL4B]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_COL4A_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->colour4.a, txt->fcv[DK_FCV_TXT3D_COL4A]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_FAC1_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->factor1, txt->fcv[DK_FCV_TXT3D_FAC1]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_FAC2_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->factor2, txt->fcv[DK_FCV_TXT3D_FAC2]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_FAC3_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->factor3, txt->fcv[DK_FCV_TXT3D_FAC3]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_SCL_TOKEN);
    DK_value3WriteAsciiAnimation(file, txt->scaling.x, txt->scaling.y, txt->scaling.z,
	txt->fcv[DK_FCV_TXT3D_SCLX], txt->fcv[DK_FCV_TXT3D_SCLY],
	txt->fcv[DK_FCV_TXT3D_SCLZ]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_ROT_TOKEN);
    DK_value3WriteAsciiAnimation(file, txt->rotation.x, txt->rotation.y, txt->rotation.z,
	txt->fcv[DK_FCV_TXT3D_ROTX], txt->fcv[DK_FCV_TXT3D_ROTY],
	txt->fcv[DK_FCV_TXT3D_ROTZ]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_TRAN_TOKEN);
    DK_value3WriteAsciiAnimation(file, txt->translation.x, txt->translation.y,
    	txt->translation.z, txt->fcv[DK_FCV_TXT3D_TRNX], txt->fcv[DK_FCV_TXT3D_TRNY],
	txt->fcv[DK_FCV_TXT3D_TRNZ]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_AMB_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->ambientFct, txt->fcv[DK_FCV_TXT3D_AMB]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_DIFF_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->diffuseFct, txt->fcv[DK_FCV_TXT3D_DIF]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_SPEC_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->specularFct, txt->fcv[DK_FCV_TXT3D_SPC]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_ROUGH_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->roughnessFct, txt->fcv[DK_FCV_TXT3D_ROU]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_BLEND_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->blend, txt->fcv[DK_FCV_TXT3D_BLD]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_TRANSP_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->transpFct, txt->fcv[DK_FCV_TXT3D_TRS]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_REFL_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->reflectFct, txt->fcv[DK_FCV_TXT3D_REF]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_POWER_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->power, txt->fcv[DK_FCV_TXT3D_POW]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT3D_TAPER_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->taper, txt->fcv[DK_FCV_TXT3D_TAP]);

    DK_incIndent(-1);
    return(TRUE);
}

/*******************************************************************************

   $$L DK_texture3DReadAsciiAnimation

   This function reads a 3D texture.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_texture3DReadAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Texture3D *txt		/* DK_texture 3d pointer */
    )
{

    char *cptr;
    int first = TRUE;

    DK_in_animToken(file);
    while (TRUE)
    {
        if (strcmp(DK_A_TXT3D_NAME_TOKEN, DK_animToken) == 0)
	{
	    cptr = DK_stringReadAsciiAnimation(file);
	    if (txt != NULL)
		txt->name = cptr;
	    else if (cptr != NULL)
	    {
		_DK_FREE(cptr, -1);
	    }
	    DK_in_animToken(file);
	}
        else if (strcmp(DK_A_TXT3D_STDK_REN_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL) 
	        DK_valueReadAsciiAnimation(file, &txt->strength, 
			&txt->fcv[DK_FCV_TXT3D_STR], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_SPAC_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL) 
	        DK_valueReadAsciiAnimation(file, &txt->spacing, 
			&txt->fcv[DK_FCV_TXT3D_SPA], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_COL0_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_value3ReadAsciiAnimation(file, &txt->colour0.r, &txt->colour0.g,
			&txt->colour0.b, &txt->fcv[DK_FCV_TXT3D_COL0R],
			&txt->fcv[DK_FCV_TXT3D_COL0G], &txt->fcv[DK_FCV_TXT3D_COL0B]);
	    else
	        DK_value3ReadAsciiAnimation(file,NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp(DK_A_TXT3D_COL0A_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->colour0.a, 
			&txt->fcv[DK_FCV_TXT3D_COL0A], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_COL1_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_value3ReadAsciiAnimation(file, &txt->colour1.r, &txt->colour1.g,
			&txt->colour1.b, &txt->fcv[DK_FCV_TXT3D_COL1R],
			&txt->fcv[DK_FCV_TXT3D_COL1G], &txt->fcv[DK_FCV_TXT3D_COL1B]);
	    else
	        DK_value3ReadAsciiAnimation(file,NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp(DK_A_TXT3D_COL1A_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->colour1.a,
			&txt->fcv[DK_FCV_TXT3D_COL1A], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_COL2_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_value3ReadAsciiAnimation(file, &txt->colour2.r, &txt->colour2.g,
			&txt->colour2.b, &txt->fcv[DK_FCV_TXT3D_COL2R],
			&txt->fcv[DK_FCV_TXT3D_COL2G], &txt->fcv[DK_FCV_TXT3D_COL2B]);
	    else
	        DK_value3ReadAsciiAnimation(file,NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp(DK_A_TXT3D_COL2A_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->colour2.a,
			&txt->fcv[DK_FCV_TXT3D_COL2A], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_COL3_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_value3ReadAsciiAnimation(file, &txt->colour3.r, &txt->colour3.g,
			&txt->colour3.b, &txt->fcv[DK_FCV_TXT3D_COL3R],
			&txt->fcv[DK_FCV_TXT3D_COL3G], &txt->fcv[DK_FCV_TXT3D_COL3B]);
	    else
	        DK_value3ReadAsciiAnimation(file,NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp(DK_A_TXT3D_COL3A_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->colour3.a,
			&txt->fcv[DK_FCV_TXT3D_COL3A], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_COL4_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_value3ReadAsciiAnimation(file, &txt->colour4.r, &txt->colour4.g, 
			&txt->colour4.b, &txt->fcv[DK_FCV_TXT3D_COL4R],
			&txt->fcv[DK_FCV_TXT3D_COL4G], &txt->fcv[DK_FCV_TXT3D_COL4B]);
	    else
	        DK_value3ReadAsciiAnimation(file,NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp(DK_A_TXT3D_COL4A_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->colour4.a,
			&txt->fcv[DK_FCV_TXT3D_COL4A], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_FAC1_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->factor1, 
			&txt->fcv[DK_FCV_TXT3D_FAC1], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_FAC2_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->factor2, 
			&txt->fcv[DK_FCV_TXT3D_FAC2], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_FAC3_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->factor3, 
			&txt->fcv[DK_FCV_TXT3D_FAC3], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_SCL_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_value3ReadAsciiAnimation(file,&txt->scaling.x, &txt->scaling.y, 
		    &txt->scaling.z, &txt->fcv[DK_FCV_TXT3D_SCLX], 
		    &txt->fcv[DK_FCV_TXT3D_SCLY], &txt->fcv[DK_FCV_TXT3D_SCLZ]);
	    else
	        DK_value3ReadAsciiAnimation(file,NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp(DK_A_TXT3D_ROT_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_value3ReadAsciiAnimation(file,&txt->rotation.x, &txt->rotation.y,
		    &txt->rotation.z, &txt->fcv[DK_FCV_TXT3D_ROTX],
		    &txt->fcv[DK_FCV_TXT3D_ROTY], &txt->fcv[DK_FCV_TXT3D_ROTZ]);
	    else
	        DK_value3ReadAsciiAnimation(file,NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp(DK_A_TXT3D_TRAN_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_value3ReadAsciiAnimation(file,&txt->translation.x, &txt->translation.y,
		    &txt->translation.z, &txt->fcv[DK_FCV_TXT3D_TRNX], 
		    &txt->fcv[DK_FCV_TXT3D_TRNY], &txt->fcv[DK_FCV_TXT3D_TRNZ]);
	    else
	        DK_value3ReadAsciiAnimation(file,NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp(DK_A_TXT3D_AMB_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->ambientFct,
			&txt->fcv[DK_FCV_TXT3D_AMB], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_DIFF_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->diffuseFct, 
			&txt->fcv[DK_FCV_TXT3D_DIF], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_SPEC_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->specularFct, 
			&txt->fcv[DK_FCV_TXT3D_SPC], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_ROUGH_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->roughnessFct, 
			&txt->fcv[DK_FCV_TXT3D_ROU], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_BLEND_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->blend, 
			&txt->fcv[DK_FCV_TXT3D_BLD], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_TRANSP_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->transpFct, 
			&txt->fcv[DK_FCV_TXT3D_TRS], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_REFL_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->reflectFct,
			&txt->fcv[DK_FCV_TXT3D_REF], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_POWER_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL) 
	        DK_valueReadAsciiAnimation(file, &txt->power, 
			&txt->fcv[DK_FCV_TXT3D_POW], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_TAPER_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->taper, 
			&txt->fcv[DK_FCV_TXT3D_TAP], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_RIND_TOKEN, DK_animToken) == 0)
	{
	    /* ignore the refraction index, an old version parameter */
	    DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT3D_STRETCH_TOKEN, DK_animToken) == 0)
	{
	    /* ignore the stretch, an old version parameter */
	    DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
	else
	{
	    if (first)
		DK_in_animToken(file);
	    return(TRUE);
	}
	first = FALSE;
    }
}

/*******************************************************************************

   $$L DK_textureWriteAsciiAnimation

   This function writes a texture.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_textureWriteAsciiAnimation
    (
	 FILE *file,	/* file io pointer */
	DK_Texture *txt	/* texture pointer */
    )
{
    if (txt == NULL)
	return(FALSE);

    DK_incIndent(1);
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_NAME_TOKEN);
    _DK_OUT_STRING(txt->pic_name);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_USCL_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->uScal, txt->fcv[DK_FCV_TXT_U_SCL]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_VSCL_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->vScal, txt->fcv[DK_FCV_TXT_V_SCL]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_UOFF_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->uOffset, txt->fcv[DK_FCV_TXT_U_OFF]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_VOFF_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->vOffset, txt->fcv[DK_FCV_TXT_V_OFF]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_AMB_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->ambientFct, txt->fcv[DK_FCV_TXT_AMB]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_DIFF_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->diffuseFct, txt->fcv[DK_FCV_TXT_DIFF]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_SPEC_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->specularFct, txt->fcv[DK_FCV_TXT_SPEC]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_ROUGH_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->roughnessFct, txt->fcv[DK_FCV_TXT_ROU]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_TRANSP_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->transpFct, txt->fcv[DK_FCV_TXT_TRSP]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_REFLM_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->reflectMappingFct, txt->fcv[DK_FCV_TXT_REFLM]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_ENV_ROT_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->rotation, txt->fcv[DK_FCV_TXT_ROT]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_BLEND_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->blend, txt->fcv[DK_FCV_TXT_BLD]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_TXT_REFL_TOKEN);
    DK_valueWriteAsciiAnimation(file, txt->reflectFct, txt->fcv[DK_FCV_TXT_REFL]);

    DK_incIndent(-1);
    return(TRUE);
}

/*******************************************************************************

   $$L DK_textureReadAsciiAnimation

   This function reads a texture.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_textureReadAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Texture *txt		/* texture list pointer */
    )
{
    char *cptr;
    int first = TRUE;

    DK_in_animToken(file);
    while (TRUE)
    {
        if (strcmp(DK_A_TXT_NAME_TOKEN, DK_animToken) == 0)
	{
	    cptr = DK_stringReadAsciiAnimation(file);
	    if (txt != NULL)
		txt->pic_name = cptr;
	    else if (cptr != NULL)
	    {
		_DK_FREE(cptr, -1);
	    }
	    DK_in_animToken(file);
	}
        else if (strcmp(DK_A_TXT_USCL_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->uScal, &txt->fcv[DK_FCV_TXT_U_SCL],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_VSCL_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->vScal, &txt->fcv[DK_FCV_TXT_V_SCL],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_UOFF_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->uOffset, &txt->fcv[DK_FCV_TXT_U_OFF],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_VOFF_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->vOffset, &txt->fcv[DK_FCV_TXT_V_OFF],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_AMB_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->ambientFct, &txt->fcv[DK_FCV_TXT_AMB],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_DIFF_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->diffuseFct, &txt->fcv[DK_FCV_TXT_DIFF],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_SPEC_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->specularFct, &txt->fcv[DK_FCV_TXT_SPEC],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_ROUGH_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->roughnessFct, &txt->fcv[DK_FCV_TXT_ROU],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_TRANSP_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->transpFct, &txt->fcv[DK_FCV_TXT_TRSP],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_REFLM_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->reflectMappingFct,
			&txt->fcv[DK_FCV_TXT_REFLM], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_REFR_TOKEN, DK_animToken) == 0)
	{
	    /* ignore the refrac, an old version parameter */
	    DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_ENV_ROT_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->rotation, &txt->fcv[DK_FCV_TXT_ROT],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_BLEND_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->blend, &txt->fcv[DK_FCV_TXT_BLD], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp(DK_A_TXT_REFL_TOKEN, DK_animToken) == 0)
	{
	    if (txt != NULL)
	        DK_valueReadAsciiAnimation(file, &txt->reflectFct, &txt->fcv[DK_FCV_TXT_REFL],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
	else
	{
	    if (first)
		DK_in_animToken(file);
	    return(TRUE);
	}
	first = FALSE;
    }
}

