/***************************************************************************************************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyTag.c,v $
   $Revision: 1.4.14.1 $ $Date: 1995/06/09 22:00:21 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's tags functions,
   its content is sufficient to read, write and manage picture data structure
   into SOFTIMAGE's file format...

   Written by: Laurent Lauzon

   (c) Copyright 1992, 1993 SOFTIMAGE Inc.

***************************************************************************************************************************************************************/

#include "tinyByteswap.h"
#include "tinyTag.h"

/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

DK_TagImage *DK_tagAllocate( void )
{
   DK_TagImage *tag;

   if( tag = (DK_TagImage *) calloc( 1, sizeof( DK_TagImage ) ) )
   {
      tag->width  = 0;
      tag->height = 0;
      tag->ratio  = 1.0;
      tag->fields = DK_NO_FIELD;
      tag->depth  = 0;
      tag->tags   = NULL;
   }
   return( tag );
}

DK_Boolean DK_tagAllocateChannels( 
   DK_TagImage *tag 
)
{
   int	i, j, k, scanBytes;

   if( tag->depth == 0 || tag->height == 0 || tag->width == 0 )
      return( FALSE );

   scanBytes = _DK_SCAN_LENGTH( tag->width );

   tag->tags =
         (unsigned char ***) malloc( sizeof( unsigned char **) * tag->depth );
   for( i = 0; i < tag->depth; i++ )
   {
      tag->tags[i] =
          (unsigned char **) malloc( sizeof( unsigned char *) * tag->height );
      for( j = 0; j < tag->height; j++ )
      {
         tag->tags[i][j] = (unsigned char *) malloc( scanBytes );
         for( k = 0; k < scanBytes; k++ )
            tag->tags[i][j][k] = 0;
      }
   }

   return( TRUE );
}

DK_Boolean DK_tagDispose( 
   DK_TagImage **tag 
)
{
   int	i, j;

   if( *tag )
   {
      if( (*tag)->tags )
      {
         for( i = 0; i < (*tag)->depth; i++ )
         {
            for( j = 0; j < (*tag)->height; j++ )
            {
               free( (*tag)->tags[i][j] );
            }

            free( (*tag)->tags[i] );
         }

         free( (*tag)->tags );
      }
      free( *tag );
   }

   return( TRUE );
}

void DK_tagWriteScanline( 
   FILE        *file, 
   DK_TagImage *tag, 
   int          scanline 
)
{
   int	          i;	  /* index of bit-plane */
   int	          ind;
   int	          j;
   int	          k;	  /* index of pixel on scanline */
   int	          count;  /* number of pixel still to write */
			  /*    on scanline (= width - k) */
   int            nbBits; /* number of pixel writed in one shot */
   unsigned short val;	  /* buffer of coded sequence */
   DK_Boolean	  first;
   DK_Boolean	  same;
   DK_Boolean	  pix;

   for( i = 0; i < tag->depth; i++ )
   {
      for( count=tag->width, ind=0; count>0; count -= nbBits, ind += nbBits )
      {
        if( count <= 14 )
        {
          /******************************************************/
          /* less or 14 bits to write, use a raw sequecne of 14 */
          /******************************************************/
           val = 0x0000;
           nbBits = count;
           for( j = 0; j < nbBits; j++ )
              if( _DK_GET_PIXEL( tag, i, scanline, ind+j ) )
                 val |= (1<<(13-j));
           DK_byteswap_fwrite( &val, 2, 1, file );
        }
        else
        {
           /*********************************************/
           /* verify if the 14 next pixels are the same */
           /*********************************************/
           first = (_DK_GET_PIXEL( tag, i, scanline, ind ) != 0);
           same = TRUE;
           for( j = 1; j < 14; j++ )
           {
              pix = (_DK_GET_PIXEL( tag, i, scanline, ind+j ) != 0);
              if( pix != first )
              {
                 same = FALSE;
                 break;
              }
           }

           /****************************************/
           /* if same then count how many more are */
           /****************************************/
           if( same )
           {
              for( j = 14; j < count; j++ )
              {
                 pix = (_DK_GET_PIXEL( tag, i, scanline, ind+j ) != 0);
                 if( pix != first )
                    break;
              }
              val = (first) ? 0xc000 : 0x8000;
              nbBits = j;
              val |= nbBits;
              DK_byteswap_fwrite( &val, 2, 1, file );
           }
           else
           {
             /*********************************/
             /* use a raw sequecne of 14 bits */
             /*********************************/
              val = 0x0000;
              nbBits = 14;
              for( j = 0; j < nbBits; j++ )
                 if( _DK_GET_PIXEL( tag, i, scanline, ind+j ) )
                    val |= (1<<(13-j));
               DK_byteswap_fwrite( &val, 2, 1, file );
            }
         }
      }
   }
}

void DK_tagReadScanline( 
   FILE        *file, 
   DK_TagImage *tag, 
   int          scanline 
)
{
   int			count, nbBits, i, j, ind;
   unsigned short	val;

   
   for( i = 0; i < tag->depth; i++ )
   {
      for( count=tag->width, ind=0; count>0; count -= nbBits, ind += nbBits )
      {
         DK_byteswap_fread( &val, 2, 1, file );
         switch( val & 0xc000 )
         {
            case 0x0000:	/* raw 14 bits */
               nbBits = 14;
               if( count < nbBits )
                  nbBits = count;
               for( j = 0; j < nbBits; j++ )
               {
                  _DK_SET_PIXEL( tag, i, scanline, (ind+j), (val & 0x2000) );
                  val = val << 1;
               }
               break;

            case 0x4000:	/* long raw */
               nbBits  = (val & 0x3fff);
               for( j = 0; j < nbBits; j++ )
               {
                  if( (j & 15) == 0 )
                     DK_byteswap_fread( &val, 2, 1, file );
                  _DK_SET_PIXEL( tag, i, scanline, (ind+j), (val & 0x8000) );
                  val = val << 1;
               }
               break;

            case 0x8000:	/* long serie of 0s */
               nbBits = (val & 0x3fff);
               for( j = 0; j < nbBits; j++ )
                  _DK_SET_PIXEL( tag, i, scanline, (ind+j), 0 );
               break;

            case 0xc000:	/* long serie of 1s */
               nbBits = (val & 0x3fff);
               for( j = 0; j < nbBits; j++ )
                  _DK_SET_PIXEL( tag, i, scanline, (ind+j), 1 );
               break;
         }
      }
   }
}

DK_TagImage *DK_tagReadFile( 
   DK_String filename 
)
{
   FILE        *file;
   DK_TagImage *tag;
   int          i;

   if( ( tag = DK_tagAllocate() ) == NULL )
      return( NULL );

   if( file = DK_tagReadFileHeader( filename, tag ) )
   {
      for( i = 0; i < tag->height; i++ )
         DK_tagReadScanline( file, tag, i );

      return( tag );
   }
   return( NULL );
}

FILE *DK_tagReadFileHeader( 
   DK_String    filename, 
   DK_TagImage *tag 
)
{
   char  buf[ 256 ];
   FILE *file;
   long  id;

   if( ( strlen( filename ) > 4 ) &&
       ( strcmp( ".tag", &filename[ strlen( filename ) - 4 ] ) == 0 ) )
      strcpy( buf, filename );
   else
      sprintf( buf, "%s.tag", filename );

   if( file = fopen( buf, "rb" ) )
   {
      id = DK_headerReadBinary( file, NULL, NULL );
      if( id != 'STAG' )
      {
         fclose( file );
         return( NULL );
      }

      DK_byteswap_fread( &tag->width,  2, 1, file );
      DK_byteswap_fread( &tag->height, 2, 1, file );
      DK_byteswap_fread( &tag->ratio,  4, 1, file );
      DK_byteswap_fread( &tag->fields, 2, 1, file );
      DK_byteswap_fread( &tag->depth,  2, 1, file );
   }

   return( file );
}

DK_Boolean DK_tagWriteFile( 
   DK_String    filename, 
   DK_TagImage *tag 
)
{
   short  i;
   FILE  *file;

   if( ( file = DK_tagWriteFileHeader( filename, tag ) ) == NULL )
      return( FALSE );

   for( i = 0; i < tag->height; i++ )
      DK_tagWriteScanline( file, tag, i );

   fclose(file);
   return( TRUE );
}

FILE *DK_tagWriteFileHeader( 
   DK_String    filename, 
   DK_TagImage *tag 
)
{
   char  buf[ 256 ];
   FILE *file;

   if( ( strlen( filename ) > 4 ) &&
       ( strcmp( ".tag", &filename[ strlen( filename ) - 4 ] ) == 0 ) )
      strcpy( buf, filename );
   else
      sprintf( buf, "%s.tag", filename );

   if( file = fopen( buf, "wb" ) )
   {
      DK_headerWriteBinary( file, NULL );
      DK_String_fwrite( "STAG", 4, 1, file );

      DK_byteswap_fwrite( &tag->width, 2, 1, file );
      DK_byteswap_fwrite( &tag->height, 2, 1, file );
      DK_byteswap_fwrite( &tag->ratio, 4, 1, file );
      DK_byteswap_fwrite( &tag->fields, 2, 1, file );
      DK_byteswap_fwrite( &tag->depth, 2, 1, file );
   }

   return( file );
}

