/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinySpline.c,v $
   $Revision: 1.13.14.1 $ $Date: 1995/06/09 22:00:20 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's spline functions,
   its content is sufficient to read, write and manage spline data structure
   into SOFTIMAGE's file format...

   Written by: Laurent Lauzon

   (c) Copyright 1990, 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinySceneToken.h"
#include "tinyUtils.h"
#include "tinySpline.h"

/* Printable name for spline type */
DK_String DK_SplineTypeName[] =
   {
      "LINEAR",
      "BEZIER",
      "BSPLINE",
      "CARDINAL",
   };

DK_String DK_BezSegmentName[] =
   {
      "--------",
      "CURVE",
      "LINEAR"
   };

DK_String DK_BezConstraintName[] =
   {
      "--------",
      "C0",
      "C1",
      "C2",
      "G1",
      "G2"
   };

/******************************************************************************/
/*** Static function prototypes ***********************************************/
/******************************************************************************/

static DK_Boolean  DK_splineWriteBinaryFile( DK_String, DK_Spline * );
static DK_Boolean  DK_splineWriteAsciiFile( DK_String, DK_Spline * );
static DK_Spline  *DK_splineReadAsciiFile( DK_String );
static DK_Spline  *DK_splineReadBinaryFile( DK_String );
static void        DK_splineReadControlPointsAscii(  FILE *, DK_Spline * );
static short       DK_bezierGetSegmentByName( char * );
static short       DK_bezierGetConstraintByName( char * );
static DK_Spline  *DK_bezierReadBinary(  FILE *, DK_FileProt * );

/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

DK_Spline *DK_splineAllocate( void )
{
   DK_Spline *spl;

   if( spl = (DK_Spline *) calloc( 1, sizeof( DK_Spline ) ) )
   {
      spl->name       = NULL;
      spl->phantom    = FALSE;
      spl->close      = FALSE;
      spl->type       = DK_SPL_LINEAR;
      spl->nbKeys     = 0;
      spl->keys       = NULL;
      spl->tension    = 0.0;
      spl->step       = 5;
      spl->constraint =  DK_BEZ_C1;
      spl->segment    =  DK_BEZ_CURVE;
      spl->next       = NULL;

      /* display */
      spl->wire_colour = DK_DEFAULT_WIRE_COLOUR;
      spl->line_type = DK_VISIBLE;
      spl->selected = FALSE;

      spl->keyframed = FALSE;
   }

   return( spl );
}

void DK_splineDispose( DK_Spline **spl )
{
   if( spl )
   {
      DK_Spline *spline = *spl;

      while( spline )
      {
         DK_Spline *save = spline;

         _DK_FREE( spline->name, -1 );
         _DK_FREE( spline->keys, -1 );

         spline = spline->next;
         free( save );
      }

      *spl = NULL;
   }
}

DK_Boolean DK_splineWriteBinary(  FILE *file, DK_Spline *spl, DK_FileProt *prot )
{
   DK_SplineKey  *key;
   short    ind, ind1, i, nb;

   for( ind = 0; prot->spline[ ind ] != 0; ind++ )
   {
      switch( prot->spline[ ind ] )
      {
         case 0:	/* end of spline definition */
            break;

         case 1:	/* spline type */
            nb = (short) spl->type;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 2:	/* points list (OLD WAY) */
            if( spl->type != DK_SPL_BEZIER )
            {
               DK_byteswap_fwrite( &spl->nbKeys, 2, 1, file );
               for( i = spl->nbKeys, key = spl->keys; i; i--, key++ )
                  DK_Vector_fwrite( &key->position, sizeof( DK_Point ), 1, file );
            }
            else
            {
               nb = spl->nbKeys * 3 - 2;
               if( spl->close )
                  nb += 2;
               if( nb < 0 )
                  nb = 0;
               DK_byteswap_fwrite( &nb, 2, 1, file );
               for( i = 0, key = spl->keys; i < spl->nbKeys; i++, key++ )
               {
                  if( i != 0 )
                     DK_Vector_fwrite( &key->previous, sizeof( DK_Vector ), 1, file );
                  DK_Vector_fwrite( &key->position, sizeof( DK_Vector ), 1, file );
                  if( i != spl->nbKeys - 1 )
                     DK_Vector_fwrite( &key->next, sizeof( DK_Vector ), 1, file );
               }
               if( spl->close )
               {
                  key = &spl->keys[ spl->nbKeys - 1];
                  DK_Vector_fwrite( &key->next, sizeof( DK_Vector ), 1, file );
                  key = spl->keys;
                  DK_Vector_fwrite( &key->previous, sizeof( DK_Vector ), 1, file );
               }
            }
            break;

         case 3: 	/* close */
            DK_byteswap_fwrite( &spl->close, 2, 1, file );
            break;

         case 4:	/* tension */
            DK_byteswap_fwrite( &spl->tension, 4, 1, file );
            break;

         case 5:	/* drawing step */
            DK_byteswap_fwrite( &spl->step, 2, 1, file );
            break;

         case 6:	/* Phantom key */
            DK_byteswap_fwrite( &spl->phantom, 2, 1, file );
            break;

         case 7:	/* keys list */
            DK_byteswap_fwrite( &spl->nbKeys, 2, 1, file );
            for( i = spl->nbKeys, key = spl->keys; i > 0; i--, key++ )
            {
               for( ind1 = 0; prot->splKey[ ind1 ] != 0; ind1++ )
               {
                  switch( prot->splKey[ ind1 ] )
                  {
                     case 1:	/* previous */
                        DK_Vector_fwrite( &key->previous, sizeof( DK_Point ), 1, file );
                        break;
                     case 2:	/* position */
                        DK_Vector_fwrite( &key->position, sizeof( DK_Point ), 1, file );
                        break;
                     case 3:	/* next */
                        DK_Vector_fwrite( &key->next, sizeof( DK_Point ), 1, file );
                        break;
                     case 4:	/* constraint */
                        DK_byteswap_fwrite( &key->constraint, 2, 1, file );
                        break;
                     case 5:	/* segment */
                        DK_byteswap_fwrite( &key->segment, 2, 1, file );
                        break;
                     case 6:	/* flag */
                        DK_byteswap_fwrite( &key->flag, 2, 1, file );
                        break;
                  }
               }
            }
            break;

         case 8:	/* name */
            DK_stringWriteBinary( file, spl->name );
            break;
      }
   }

   return( TRUE );
}

DK_SplineKey *DK_splineKeyAllocate( short nb )
{
   DK_SplineKey	*key, *ptr;
   int		 i;

   if( key = (DK_SplineKey *) calloc( nb, sizeof( DK_SplineKey ) ) )
   {
      for( i = 0, ptr = key; i < nb; i++, ptr++ )
      {
         ptr->previous.x = key->previous.y = key->previous.z = 0.0;
         ptr->position.x = key->position.y = key->position.z = 0.0;
         ptr->next.x     = key->next.y     = key->next.z     = 0.0;
         ptr->constraint =  DK_BEZ_C1;
         ptr->segment    =  DK_BEZ_CURVE;
         ptr->flag       = 0;
      }
   }
   return( key );
}

DK_Spline *DK_splineReadBinary(  FILE *file, DK_FileProt *prot )
{
   DK_Spline	*spl;
   short	 i, ind = 0, ind1 = 0, tok, nb;
   DK_SplineKey	*key;
   DK_Boolean	 old = FALSE;

   if( (spl = DK_splineAllocate()) == NULL )
      return( NULL );

   if( (prot != NULL) && (prot->spline != NULL) )
      tok = prot->spline[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );
   while( tok != 0 /* End of spline definition */ )
   {
      switch( tok )
      {
         case 1: /* spline type */
            DK_byteswap_fread( &nb, 2, 1, file );
            spl->type = (DK_SplineType) nb;
            break;

         case 2: /* Control Points list */
            old = TRUE;
            DK_byteswap_fread( &spl->nbKeys, 2, 1, file );
            key = spl->keys = DK_splineKeyAllocate( spl->nbKeys );
            for( i = spl->nbKeys; i > 0; i--, key++ )
               DK_Vector_fread( &key->position, sizeof( DK_Vector ), 1, file );
            break;

         case 7: /* Keys list (new way */
            DK_byteswap_fread( &spl->nbKeys, 2, 1, file );
            key = spl->keys = DK_splineKeyAllocate( spl->nbKeys );
            for( i = 0; i < spl->nbKeys; i++, key++ )
            {
               ind1 = 0;
               if( (prot != NULL) && (prot->splKey != NULL) )
                  tok = prot->splKey[ ind1 ++ ];
               else
                  DK_byteswap_fread( &tok, 2, 1, file );
               while( tok != 0 /* End of spline key definition */ )
               {
                  switch( tok )
                  {
                     case 1: /* previous */
                        DK_Vector_fread( &key->previous, sizeof( DK_Vector ), 1, file );
                        break;
                     case 2: /* position */
                        DK_Vector_fread( &key->position, sizeof( DK_Vector ), 1, file );
                        break;
                     case 3: /* next */
                        DK_Vector_fread( &key->next, sizeof( DK_Vector ), 1, file );
                        break;
                     case 4: /* constraint */
                        DK_byteswap_fread( &key->constraint, 2, 1, file );
                        break;
                     case 5: /* segment */
                        DK_byteswap_fread( &key->segment, 2, 1, file );
                        break;
                     case 6: /* flag */
                        DK_byteswap_fread( &key->flag, 2, 1, file );
                        break;

                     default:
                        DK_fileSkipField( file, tok );

                  }
                  if( (prot != NULL) && (prot->splKey != NULL) )
                     tok = prot->splKey[ ind1 ++ ];
                  else
                     DK_byteswap_fread( &tok, 2, 1, file );
               }
            }
            break;

         case 3: /* close */
            DK_byteswap_fread( &spl->close, 2, 1, file );
            break;

         case 4: /* tension */
            DK_byteswap_fread( &spl->tension, 4, 1, file );
            break;

         case 5: /* drawing step */
            DK_byteswap_fread( &spl->step, 2, 1, file );
            break;

         case 6: /* phantom */
            DK_byteswap_fread( &spl->phantom, 2, 1, file );
            break;

         case 8: /* name */
            spl->name = DK_stringReadBinary( file );
            break;

         default:
            DK_fileSkipField( file, tok );

      }
      if( (prot != NULL) && (prot->spline != NULL) )
         tok = prot->spline[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   if( old && (spl->type == DK_SPL_BEZIER) )
      DK_splineShrinkBezier( spl );

   return( spl );
}

DK_SplineType DK_splineGetTypeByName( char *name )
{
   DK_SplineType	 type = DK_SPL_LINEAR;
   int		 i;


   for( i = 0; i < 4; i++ )
      if( strcmp( name, DK_SplineTypeName[ i ] ) == 0 )
         type = (DK_SplineType) i;

   return( type );
}

DK_Spline *DK_splineReadAscii(  FILE *file, short *nbHoles, DK_Spline ***holes )
{
   DK_Spline	*spl,
		*hol;
   char		 str[ 80 ],
		 type[ 80 ];


   if( (spl = DK_splineAllocate()) == NULL )
      return( NULL );

   hol = NULL;

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( spl );

   do
   {
      fscanf( file, "%s", str );

      if( strcmp( "name", str ) == 0 )
         spl->name = DK_stringReadAscii( file );
      else if( strcmp( "type", str ) == 0 )
      {
         fscanf( file, "%s", type );
         spl->type = DK_splineGetTypeByName( type );
      }
      else if( strcmp( "phantom", str ) == 0 )
         spl->phantom = TRUE;
      else if( strcmp( "close", str ) == 0 )
         spl->close = TRUE;
      else if( strcmp( "nbKeys", str ) == 0 )
         fscanf( file, "%hd", &spl->nbKeys );
      else if( strcmp( "tension", str ) == 0 )
         fscanf( file, "%f", &spl->tension );
      else if( strcmp( "step", str ) == 0 )
         fscanf( file, "%hd", &spl->step );
      else if( strcmp( "constraint", str ) == 0 )
      {
         fscanf( file, "%s", type );
         spl->constraint = DK_bezierGetConstraintByName( type );
      }
      else if( strcmp( "segment", str ) == 0 )
      {
         fscanf( file, "%s", type );
         spl->segment = DK_bezierGetSegmentByName( type );
      }
      else if( strcmp( "controlPoints", str ) == 0 )
         DK_splineReadControlPointsAscii( file, spl );
      else if( strcmp( "hole", str ) == 0 )
      {
         /* get rid of number */
         fscanf( file, "%s", type );

         hol =  DK_splineReadAscii( file, NULL, NULL );

         if( nbHoles == NULL )
            DK_splineDispose( &hol );
         else if( (*nbHoles) == 0 )
         {
            (*nbHoles) = 1;
            (*holes) = (DK_Spline **) calloc( 1, sizeof( DK_Spline * ) );
            (*holes)[ 0 ] = hol;
         }
         else
         {
            (*nbHoles)++;
            (*holes) = (DK_Spline **) realloc( (*holes), sizeof( DK_Spline * ) * (*nbHoles) );
            (*holes)[ (*nbHoles) - 1 ] = hol;
         }
      }
   } while( strcmp( "}", str ) != 0 );

   return( spl );
}

static void DK_splineReadControlPointsAscii(  FILE *file, DK_Spline *spl )
{
   char		 str[ 80 ],
		 str2[ 80 ],
		 type[ 80 ];
   DK_SplineKey	*key = NULL;


   if( (spl->keys = DK_splineKeyAllocate( spl->nbKeys )) == NULL )
      return;

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return;

   do
   {
      fscanf( file, "%s", str );

      if( str[0] == '[' )
      {
         if( key == NULL )
            key = spl->keys;
         else
            key++;
      }
      else if( strcmp( "position", str ) == 0 )
         fscanf( file, "%f %f %f", &key->position.x, &key->position.y,
                                   &key->position.z );
      else if( strcmp( "previous", str ) == 0 )
         fscanf( file, "%f %f %f", &key->previous.x, &key->previous.y,
                                   &key->previous.z );
      else if( strcmp( "next", str ) == 0 )
         fscanf( file, "%f %f %f", &key->next.x, &key->next.y, &key->next.z );
      else if( strcmp( "constraint", str ) == 0 )
      {
         fscanf( file, "%s", type );
         key->constraint = DK_bezierGetConstraintByName( type );
      }
      else if( strcmp( "segment", str ) == 0 )
      {
         fscanf( file, "%s", type );
         key->segment = DK_bezierGetSegmentByName( type );
      }
      else if( strcmp( "flag", str ) == 0 )
      {
         fscanf( file, "%s", str2 );
         if( strcmp( str2, "(" ) != 0 )
            return;
         do
         {
            fscanf( file, "%s", str2 );

            if( strcmp( "SELECTED", str2 ) == 0 )
               key->flag |=  DK_KEY_SELECTED;
         } while( strcmp( str2, ")" ) != 0 );
      }
   } while( strcmp( "}", str ) != 0 );
}

static short DK_bezierGetSegmentByName( char *name )
{
   short	 type =  DK_BEZ_CURVE;
   int		 i;

   for( i = 0; i < 3; i++ )
      if( strcmp( name, DK_BezSegmentName[ i ] ) == 0 )
         type = i;

   return( type );
}

static short DK_bezierGetConstraintByName( char *name )
{
   short	 type =  DK_BEZ_C0;
   int		 i;

   for( i = 0; i < 6; i++ )
      if( strcmp( name, DK_BezConstraintName[ i ] ) == 0 )
         type = i;

   return( type );
}

DK_Boolean DK_splineWriteAscii(  FILE *file, DK_Spline *spl, DK_String space )
{
   DK_SplineKey	*key;
   int		 i;
   
   fprintf( file, "\n%sspline\n", space );
   fprintf( file, "%s{\n", space );

   fprintf( file, "%s  name       \"%s\"\n", space, spl->name );
   fprintf( file, "%s  type       %s\n", space, DK_SplineTypeName[ spl->type ]);
   fprintf( file, "%s  nbKeys     %hd\n", space, spl->nbKeys );
   fprintf( file, "%s  tension    %*.*f\n", space,  DK_FM,  DK_FD, spl->tension );
   fprintf( file, "%s  step       %hd\n", space, spl->step );
   if( spl->close )
      fprintf( file, "%s  close\n", space );
   if( spl->phantom )
      fprintf( file, "%s  phantom\n", space );

   if( spl->type ==  DK_SPL_BEZIER )
   {
      fprintf( file, "%s  constraint %s\n", space,
                     DK_BezConstraintName[ spl->constraint ] );
      fprintf( file, "%s  segment    %s\n", space,
                     DK_BezSegmentName[ spl->segment ] );
   }

   fprintf( file, "\n%s  controlPoints\n", space );
   fprintf( file, "%s  {\n", space );
   for( i = 0, key = spl->keys; i < spl->nbKeys; i++, key++ )
   {
      fprintf( file, "%s    [%1d]  position   %*.*f  %*.*f  %*.*f\n", space,
                        i,  DK_FM,  DK_FD, key->position.x,  DK_FM,  DK_FD, key->position.y,
                         DK_FM,  DK_FD, key->position.z );
      if( spl->type ==  DK_SPL_BEZIER )
      {
         fprintf( file, "%s         previous   %*.*f  %*.*f  %*.*f\n", space,
                         DK_FM,  DK_FD, key->previous.x,  DK_FM,  DK_FD, key->previous.y,
                         DK_FM,  DK_FD, key->previous.z );
         fprintf( file, "%s         next       %*.*f  %*.*f  %*.*f\n", space,
                         DK_FM,  DK_FD, key->next.x,  DK_FM,  DK_FD, key->next.y,
                         DK_FM,  DK_FD, key->next.z );
         fprintf( file, "%s         constraint %s\n", space,
                        DK_BezConstraintName[ key->constraint ] );
         fprintf( file, "%s         segment    %s\n", space,
                        DK_BezSegmentName[ key->segment ] );
      }
      if( key->flag &  DK_KEY_SELECTED )
         fprintf( file, "%s         flag    ( SELECTED )\n", space );
   }
   fprintf( file, "%s  }\n", space );

   fprintf( file, "%s}\n", space );
   return( TRUE );
}


DK_Spline *DK_splineReadFile( DK_String filename )
{
   char		 buf[256];
   long          id;
    FILE		*file;

   sprintf( buf, "%s.spl", filename );
   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( NULL );

   DK_byteswap_fread( &id, 4, 1, file );
   fclose( file );

   if( id == 'SPLN' )
      return( DK_splineReadAsciiFile( filename ) );
   else if( id == DK_BINARY_MAGIC_NUMBER )
      return( DK_splineReadBinaryFile( filename ) );
   else
      return( NULL );
}

DK_Boolean DK_splineWriteFile( DK_String filename, DK_Spline *spl , DK_FileFormat fmt )
{
   if( fmt ==  DK_FIL_ASCII )
      return( DK_splineWriteAsciiFile( filename, spl ) );
   else if( fmt ==  DK_FIL_BINARY )
      return( DK_splineWriteBinaryFile( filename, spl ) );
   else
      return( FALSE );
}

/******************************************************************************/
/*** Static functions *********************************************************/
/******************************************************************************/

static DK_Boolean DK_splineWriteAsciiFile( DK_String filename, DK_Spline *spl )
{
    FILE		*file;
   char		buf[256];
   DK_Boolean      success;

   sprintf( buf, "%s.spl", filename );
   if( ( file = fopen( buf, "wb" ) ) == NULL )
      return( FALSE );

   success = DK_headerWriteAscii( file, "SPLN" );
   success &= DK_splineWriteAscii( file, spl, "" );
   /* Verify if file was saved correctly */
   fflush( file );
   success = (ferror( file ) == 0 );

   fclose( file );
   return( success );
}

static DK_Boolean DK_splineWriteBinaryFile( DK_String filename, DK_Spline *spl )
{
    FILE		*file;
   char		 buf[256];
   DK_FileProt	*prot;
   DK_Boolean	 success;

   sprintf( buf, "%s.spl", filename );
   if( ( file = fopen( buf, "wb" ) ) == NULL )
      return( FALSE );

   DK_headerWriteBinary( file, NULL );
   DK_prototypeInitSpline( &prot, TRUE );
   DK_prototypeWrite( file, prot );
   DK_String_fwrite( "SPLN", 4, 1, file );
   success = DK_splineWriteBinary( file, spl, prot );
   /* Verify if file was saved correctly */
   fflush( file );
   success = (ferror( file ) == 0 );
   fclose( file );
   DK_prototypeDispose( &prot );
   return( success );
}

static DK_Spline *DK_splineReadAsciiFile( DK_String filename )
{
   char		 str[ 80 ];
    FILE		*file;
   DK_Spline	*spl;
   long		 id;
   float	 version;

   sprintf( str, "%s.spl", filename );
   if( ( file = fopen( str, "rb" ) ) == NULL )
      return( NULL );

   if( (id = DK_headerReadAscii( file, &version )) != 'SPLN' )
      return( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str , "spline" ) == 0 )
      spl = DK_splineReadAscii( file, NULL, NULL );

   fclose( file );

   return( spl );
}

static DK_Spline *DK_splineReadBinaryFile( DK_String filename )
{
   char		 buf[256];
    FILE		*file;
   float	 version;
   long		 id;
   DK_FileProt	*prot;
   DK_Spline	*spl;

   sprintf( buf, "%s.spl", filename );
   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( NULL );

   id = DK_headerReadBinary( file, &version, NULL );
   if( id == 'PROT' )
   {
      prot = DK_prototypeRead( file );
      DK_byteswap_fread( &id, 4, 1, file );
   }
   if( id == 'SPLN' )
      spl = DK_splineReadBinary( file, prot );
   else if( id == 'BEZR' )
      spl = DK_bezierReadBinary( file, prot );

   fclose( file );

   DK_prototypeDispose( &prot );

   return( spl );
   
}

static DK_Spline *DK_bezierReadBinary(  FILE *file, DK_FileProt *prot )
{
   DK_Spline     *bez;
   DK_SplineKey  *key;
   short       i, ind, ind1, tok;

   if( (bez = DK_splineAllocate()) == NULL )
      return( NULL );

   bez->type = DK_SPL_BEZIER;

   ind = 0;
   if( (prot != NULL) && (prot->bezier != NULL) )
      tok = prot->bezier[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );
   while( tok != 0 /* End of bezier definition */ )
   {
      switch( tok )
      {
         case 1: /* keys */
            DK_byteswap_fread( &bez->nbKeys, 2, 1, file );
            key = bez->keys = DK_splineKeyAllocate( bez->nbKeys );
            for( i = 0; i < bez->nbKeys; i++, key++ )
            {
               ind1 = 0;
               if( (prot != NULL) && (prot->bezKey != NULL) )
                  tok = prot->bezKey[ ind1 ++ ];
               else
                  DK_byteswap_fread( &tok, 2, 1, file );
               while( tok != 0 /* End of bezier key definition */ )
               {
                  switch( tok )
                  {
                     case 1: /* previous */
                        DK_Vector_fread( &key->previous, sizeof( DK_Point ), 1, file );
                        break;
                     case 2: /* position */
                        DK_Vector_fread( &key->position, sizeof( DK_Point ), 1, file );
                        break;
                     case 3: /* next */
                        DK_Vector_fread( &key->next, sizeof( DK_Point ), 1, file );
                        break;
                     case 4: /* constraint */
                        DK_byteswap_fread( &key->constraint, 2, 1, file );
                        break;
                     case 5: /* segment */
                        DK_byteswap_fread( &key->segment, 2, 1, file );
                        break;
                     case 6: /* flag */
                        DK_byteswap_fread( &key->flag, 2, 1, file );
                        break;

                     default:
                        DK_fileSkipField( file, tok );

                  }
                  if( (prot != NULL) && (prot->bezKey != NULL) )
                     tok = prot->bezKey[ ind1 ++ ];
                  else
                     DK_byteswap_fread( &tok, 2, 1, file );
               }
            }
            break;
         case 2: /* close */
            DK_byteswap_fread( &bez->close, 2, 1, file );
            break;
         case 3: /* default constraint */
            DK_byteswap_fread( &bez->constraint, 2, 1, file );
            break;
         case 4: /* default segment */
            DK_byteswap_fread( &bez->segment, 2, 1, file );
            break;
         case 5: /* step */
            DK_byteswap_fread( &bez->step, 2, 1, file );
            break;

         default:
            DK_fileSkipField( file, tok );

      }
      if( (prot != NULL) && (prot->bezier != NULL) )
         tok = prot->bezier[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   return( bez );
}

void DK_splineShrinkBezier( DK_Spline *spl )
{
   short	 nb, i;
   DK_SplineKey	*newKeys, *nkey, *key;

   if( spl->close )
      nb = spl->nbKeys / 3;
   else
      nb = (spl->nbKeys - 1) / 3 + 1;

   nkey = newKeys = DK_splineKeyAllocate( nb );
   key = spl->keys;

   for( i = 0; i < nb; i++, nkey++ )
   {
      if( i != 0 )
      {
         nkey->previous = key->position;
         key++;
      }

      nkey->position = key->position;
      key++;

      if( i != (nb - 1) || spl->close )
      {
         nkey->next = key->position;
         key++;
      }
   }

   if( spl->close )
      newKeys->previous = key->position;

   free( spl->keys );
   spl->keys = newKeys;
   spl->nbKeys = nb;

}

/*******************************************************************************

   $$L DK_splinekeyWriteAsciiAnimation

   This function writes a spline key.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_splinekeyWriteAsciiAnimation
    (
	 FILE *file,	/* file io pointer */
	DK_SplineKey *splinekey  /* spline key pointer */
    )
{
    if (splinekey == NULL)
	return(FALSE);

    DK_incIndent(1);
    _DK_OUT_INDENT();

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_KEY_PREV_TOKEN);
    DK_vectorWriteAsciiAnimation(file, &splinekey->previous);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_KEY_POS_TOKEN);
    DK_vectorWriteAsciiAnimation(file, &splinekey->position);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_KEY_NEXT_TOKEN);
    DK_vectorWriteAsciiAnimation(file, &splinekey->next);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_KEY_CONSTRAINT_TOKEN);
    _DK_OUT_SVALUE(splinekey->constraint);

    _DK_OUT_4SPACES();
    _DK_OUT_TOKEN( DK_A_SPLINE_KEY_SEGMENT_TOKEN);
    _DK_OUT_SVALUE(splinekey->segment);

    _DK_OUT_4SPACES();
    _DK_OUT_TOKEN( DK_A_SPLINE_KEY_FLAG_TOKEN);
    _DK_OUT_SVALUE( splinekey->flag);

    DK_incIndent(-1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_splinekeyReadAsciiAnimation

   This function reads a spline key.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_splinekeyReadAsciiAnimation
    (
	 FILE *file,	/* file io pointer */
	DK_SplineKey *splinekey  /* fkey pointer */
    )
{
    while (TRUE)
    {
	if (strcmp( DK_A_SPLINE_KEY_PREV_TOKEN, DK_animToken) == 0)
	{
	    if (splinekey != NULL)
                DK_vectorReadAsciiAnimation(file, &(splinekey->previous));
	    else
                DK_vectorReadAsciiAnimation(file, NULL);
	}
	else if (strcmp( DK_A_SPLINE_KEY_POS_TOKEN, DK_animToken) == 0)
	{
	    if (splinekey != NULL)
                DK_vectorReadAsciiAnimation(file, &(splinekey->position));
	    else
                DK_vectorReadAsciiAnimation(file, NULL);
	}
	else if (strcmp( DK_A_SPLINE_KEY_NEXT_TOKEN, DK_animToken) == 0)
	{
	    if (splinekey != NULL)
                DK_vectorReadAsciiAnimation(file, &(splinekey->next));
	    else
                DK_vectorReadAsciiAnimation(file, NULL);
	}
	else if (strcmp( DK_A_SPLINE_KEY_CONSTRAINT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (splinekey != NULL)
	        splinekey->constraint = DK_animSValue;
	}
	else if (strcmp( DK_A_SPLINE_KEY_SEGMENT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (splinekey != NULL)
	        splinekey->segment = DK_animSValue;
	}
	else if (strcmp( DK_A_SPLINE_KEY_FLAG_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (splinekey != NULL)
	        splinekey->flag = DK_animSValue;

	    /* terminate the splinekey read */
	    DK_in_animToken(file);
	    return(TRUE);
	}
	else 
	{
	    return(TRUE);
	}
        if (DK_in_animToken(file) == FALSE)
	    return(FALSE);
    }

}

/*******************************************************************************

   $$L DK_splineWriteAsciiAnimation

   This function writes a spline.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_splineWriteAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Spline *spl		/* spline pointer */
    )
{
    int i;
    DK_SplineKey *v;

    if (spl == NULL) 
	return(FALSE);

    DK_incIndent(1);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_NAME_TOKEN);
    _DK_OUT_STRING(spl->name);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_TYPE_TOKEN);
    switch (spl->type)
    {
	case  DK_SPL_LINEAR:
	    _DK_OUT_TOKEN(DK_A_SPLINE_LINEAR_TYPE_TOKEN);
	    break;
	case  DK_SPL_BEZIER:
	    _DK_OUT_TOKEN(DK_A_SPLINE_BEZIER_TYPE_TOKEN);
	    break;
	case  DK_SPL_BSPLINE:
	    _DK_OUT_TOKEN(DK_A_SPLINE_BSPLINE_TYPE_TOKEN);
	    break;
	case  DK_SPL_CARDINAL:
	    _DK_OUT_TOKEN(DK_A_SPLINE_CARDINAL_TYPE_TOKEN);
	    break;
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_CLOSE_TOKEN);
    _DK_OUT_SVALUE( spl->close);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_TENSION_TOKEN);
    _DK_OUT_FVALUE(spl->tension);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_STEP_TOKEN);
    _DK_OUT_SVALUE( spl->step);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_PHANTOM_TOKEN);
    _DK_OUT_SVALUE( spl->phantom);

    if (spl->keyframed == TRUE)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_SPLINE_KEYFRAME_FLAG_TOKEN);
        _DK_OUT_SVALUE( spl->keyframed);
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SPLINE_KEYLIST_TOKEN);
    _DK_OUT_SVALUE( spl->nbKeys);
    for (i = 0, v = spl->keys; i < spl->nbKeys; i++, v++)
	DK_splinekeyWriteAsciiAnimation(file, v);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_SPLINE_WIRE_COLOUR_TOKEN);
    _DK_OUT_SVALUE( spl->wire_colour);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_SPLINE_LINE_TYPE_TOKEN);
    _DK_OUT_SVALUE( spl->line_type);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_SPLINE_SELECTED_TOKEN);
    switch (spl->selected)
    {
	case  DK_UNSELECTED:
	    _DK_OUT_TOKEN(DK_A_UNSELECTED_TOKEN);
	    break;
	case  DK_NODE_SELECTED:
	    _DK_OUT_TOKEN(DK_A_NODE_SELECTED_TOKEN);
	    break;
	case  DK_BRANCH_SELECTED:
	    _DK_OUT_TOKEN(DK_A_BRANCH_SELECTED_TOKEN);
	    break;
	case  DK_INHERIT_SELECTED:
	    _DK_OUT_TOKEN(DK_A_INHERIT_SELECTED_TOKEN);
	    break;
    }

    DK_incIndent(-1);
    return(TRUE);
}

/*******************************************************************************

   $$L DK_splineReadAsciiAnimation

   This function allocates and reads a spline.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_splineReadAsciiAnimation
    (
	 FILE *file,	/* file io pointer */
	DK_Spline *spl    /* spline pointer */
    )
{
    int read, i, first = TRUE;
    DK_HVector tv;
    DK_SplineKey *v;
    DK_SVALUE nb;
    int oldKeysList = FALSE;
    char *name = NULL;

    while (TRUE)
    {
 	read = TRUE;

	if (strcmp(DK_A_SPLINE_NAME_TOKEN, DK_animToken) == 0)
	{
	    name = DK_stringReadAsciiAnimation(file);
	    if (spl != NULL)
		spl->name = name;
	    else if (name != NULL)
		_DK_FREE(name, -1);
	}
	else if (strcmp(DK_A_SPLINE_TYPE_TOKEN, DK_animToken) == 0)
	{
	    if (DK_in_animToken(file) == FALSE)
		return(FALSE);
	    if (spl != NULL) 
	    {
		if (strcmp(DK_A_SPLINE_LINEAR_TYPE_TOKEN, DK_animToken) == 0)
	            spl->type =  DK_SPL_LINEAR;
		else if (strcmp(DK_A_SPLINE_BEZIER_TYPE_TOKEN, DK_animToken) == 0)
	            spl->type =  DK_SPL_BEZIER;
		else if (strcmp(DK_A_SPLINE_BSPLINE_TYPE_TOKEN, DK_animToken) == 0)
	            spl->type =  DK_SPL_BSPLINE;
		else if (strcmp(DK_A_SPLINE_CARDINAL_TYPE_TOKEN, DK_animToken) == 0)
	            spl->type =  DK_SPL_CARDINAL;
		else if ( DK_option.verbose == TRUE)
		    fprintf( DK_option.msgFd, "DK_SplineReadAsciiAnimation: \
error, unknown spline type DK_animToken '%s'.\n",DK_animToken);
	        if (spl->type ==  DK_SPL_CARDINAL) 
		    spl->tension = 0.5;
	    }
	}
        else if (strcmp( DK_A_SPLINE_NB_TOKEN, DK_animToken) == 0)
	{
	    oldKeysList = TRUE;
	    DK_in_animSValue(file, &nb);
	    if (spl != NULL) 
	    {
	        spl->nbKeys = nb;
                spl->keys = DK_splineKeyAllocate(spl->nbKeys);
	        if (spl->keys == NULL) 
		{
		    /* skip */
                    for( i = 0; i < nb; i++)
		        DK_hvectorReadAsciiAnimation(file, NULL);
	        }
	        else 
		{
                    v = spl->keys;
                    for( i = 0; i < spl->nbKeys; i++, v++ ) 
		    {
		        DK_hvectorReadAsciiAnimation(file, &tv);
			v->position.x = tv.x;
			v->position.y = tv.y;
			v->position.z = tv.z;
                        v->flag = FALSE;
                    }
	        }
	    }
	    else 
	    {
		/* skip */
		for( i = 0; i < nb; i++)
		    DK_hvectorReadAsciiAnimation(file, NULL);
	    }

	}
        else if (strcmp( DK_A_SPLINE_KEYLIST_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &nb);
            if (DK_in_animToken(file) == FALSE)
	        return(FALSE);
	    if (spl != NULL) 
	    {
	        spl->nbKeys = nb;
                spl->keys = DK_splineKeyAllocate(spl->nbKeys);
	        if (spl->keys == NULL) 
		{
                    for( i = 0; i < nb; i++)
		        DK_splinekeyReadAsciiAnimation(file, NULL);
	        }
	        else 
		{
                    v = spl->keys;
                    for( i = 0; i < spl->nbKeys; i++, v++ ) 
		        DK_splinekeyReadAsciiAnimation(file, v);
	        }
	    }
	    else 
	    {
		/* skip */
		for( i = 0; i < nb; i++)
		    DK_splinekeyReadAsciiAnimation(file, NULL);
	    }

	    read = FALSE;
	}
        else if (strcmp( DK_A_SPLINE_CLOSE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (spl != NULL)
	        spl->close = DK_animSValue;
	}
        else if (strcmp( DK_A_SPLINE_TENSION_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	    if (spl != NULL)
	        spl->tension = DK_animFValue;
	}
        else if (strcmp( DK_A_SPLINE_STEP_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (spl != NULL)
	        spl->step = DK_animSValue;
	}
        else if (strcmp( DK_A_SPLINE_PHANTOM_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (spl != NULL)
	        spl->phantom = DK_animSValue;
	}
        else if (strcmp( DK_A_SPLINE_KEYFRAME_FLAG_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (spl != NULL)
	        spl->keyframed = DK_animSValue;
	}
        else if (strcmp( DK_A_SPLINE_WIRE_COLOUR_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (spl != NULL)
	        spl->wire_colour = DK_animSValue;
	}
        else if (strcmp(DK_A_SPLINE_WIRE_COLOUR_INDEX_TOKEN, DK_animToken) == 0)
        {
            DK_in_animSValue_na(file);
            DK_in_animSValue_na(file);
	    /* forget the two indexes for the moment */
            if (spl != NULL)
    	        spl->wire_colour = DK_DEFAULT_WIRE_COLOUR;
    	    DK_in_animToken(file);
        }
	else if (strcmp( DK_A_SPLINE_LINE_TYPE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (spl != NULL)
	        spl->line_type = DK_animSValue;
	}
	else if (strcmp( DK_A_SPLINE_SELECTED_TOKEN, DK_animToken) == 0)
	{
	    if (DK_in_animToken(file) == FALSE)
		return(FALSE);
	    if (spl != NULL)
	    {
	        if (strcmp(DK_A_UNSELECTED_TOKEN, DK_animToken) == 0)
		    spl->selected =  DK_UNSELECTED;
	        else if (strcmp(DK_A_NODE_SELECTED_TOKEN, DK_animToken) == 0)
		    spl->selected =  DK_NODE_SELECTED;
	        else if (strcmp(DK_A_BRANCH_SELECTED_TOKEN, DK_animToken) == 0)
		    spl->selected =  DK_BRANCH_SELECTED;
	        else if (strcmp(DK_A_INHERIT_SELECTED_TOKEN, DK_animToken) == 0)
		    spl->selected =  DK_INHERIT_SELECTED;
		else if ( DK_option.verbose == TRUE)
		    fprintf( DK_option.msgFd, "DK_SplineReadAsciiAnimation: error, \
unknown select DK_animToken '%s'.\n", DK_animToken);
	    }

	    if (spl != NULL&&
	        oldKeysList == TRUE && spl->type ==  DK_SPL_BEZIER )
		   DK_splineShrinkBezier( spl );

	    return(DK_in_animToken(file));
	}
	else 
	{
	    if (first)
		DK_in_animToken(file);

	    if (spl != NULL && oldKeysList && spl->type ==  DK_SPL_BEZIER )
	        DK_splineShrinkBezier( spl );

	    return(TRUE);
	}
	if (read == TRUE && DK_in_animToken(file) == FALSE)
	    return(FALSE);
	first = FALSE;
    }
}

