
/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyScene.c,v $
   $Revision: 1.26.14.1 $ $Date: 1995/06/09 22:00:18 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's scene functions,
   its content is sufficient to read, and write scene data structure
   into SOFTIMAGE's file format...

   Written by: Colin Hui

   (c) Copyright 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#define MAIN 1

#include "tinySoftType.h"
#include "tinySceneToken.h"
#include "tinyScene.h"
#include "tinyUtils.h"

extern DK_Boolean DK_CLASSIC_SCALING;

/* static functions */
static void DK_renderInitialize ( DK_RenderingSetup *);
static int  DK_pictureWriteAsciiAnimation ( FILE *, DK_RenderingSetup *);
static int  DK_pictureReadAsciiAnimation ( FILE *, DK_RenderingSetup *);
static int  DK_headerWriteAsciiAnimation ( FILE *, char *, float);
static int  DK_headerReadAsciiAnimation ( FILE *, float *);
static int  DK_pathWriteAsciiAnimation ( FILE *, DK_Scene *);
static int  DK_pathReadAsciiAnimation ( FILE *, DK_Scene *);
static int  DK_renderWriteAsciiAnimation ( FILE *, DK_RenderingSetup *);
static void DK_layoutReadAsciiAnimation ( FILE *);

/*******************************************************************************

   $$L DK_renderInitialize

   This function initializes the render structure.

   Returned Value: None

*******************************************************************************/

static void DK_renderInitialize
    (
	DK_RenderingSetup *ren
    )
{
    long i;

    ren->type = DK_REN_SOFTIMAGE;
    ren->start = 1;
    ren->end = 1;
    ren->step = 1;
    ren->frameBuffer = TRUE;
    ren->pictureFileName = strdup("default");
    ren->pictureFile = TRUE;
    ren->xResolution = 500;
    ren->yResolution = 424;
    ren->pixelRatio = 1;
    ren->picFormat = DK_FMT_CUSTOM_ASPECT;
    ren->aspectRatio = 1.18;
    ren->aspectWidth = 0.0;
    ren->aspectHeight = 0.0;

    ren->rayMaxDepth = 1;
    ren->rayTrianglesPerLeaf = 10;
    ren->rayMaxTreeDepth = 30;

    ren->antialiasType = DK_REN_ANTIALIAS_BARTLET;
    ren->samplingFilter = 1;
    ren->maxLevel = 1;
    ren->adaptiveThreshold = .05;
    ren->backCulling = TRUE;
    ren->colourClip = DK_REN_CLIP_COMPONENT;

    ren->fade.active = FALSE;
    ren->fade.start = 20;
    ren->fade.end = 100;
    ren->fade.colour.r = 0;
    ren->fade.colour.g = 0;
    ren->fade.colour.b = 0;
    ren->fade.transparency = 0;
    for (i = 0; i < DK_FCV_FAD_NUM; i++)
    {
       ren->fade.fcv[i] = (DK_Fcurve *)NULL;
       ren->fade.active_fcv[i].active = FALSE;
       ren->fade.active_fcv[i].fcv = (DK_Fcurve *)NULL;
    }

    ren->ambiance.r = .3;
    ren->ambiance.g = .3;
    ren->ambiance.b = .3;

    ren->fog.active = FALSE;
    ren->fog.start = 0.0;
    ren->fog.end = 0.0;
    ren->fog.colour.r = ren->fog.colour.g = ren->fog.colour.b = 0.0;
    ren->fog.density = 0.0;
    ren->fog.base = 0.0;
    ren->fog.thickness = 0.0;
    for (i = 0; i < DK_FCV_FOG_NUM; i++)
    {
       ren->fog.fcv[i] = (DK_Fcurve *)NULL;
       ren->fog.active_fcv[i].active = FALSE;
       ren->fog.active_fcv[i].fcv = (DK_Fcurve *)NULL;
    }

    ren->preFrameFile  = NULL;
    ren->postFrameFile = NULL;

    ren->motion_blur = 0;
    ren->shutter_speed = 1.0;
    ren->min_movement = 5.0;

    ren->onfields = 0;
    ren->firstfield = 1;

    ren->statFile = NULL;
    ren->statTiming = FALSE;
    ren->statMemory = FALSE;
    ren->statTree = FALSE;
    ren->statLine = FALSE;

    ren->do_dither = FALSE;
    ren->add_noise = FALSE;
    ren->noise_amount = 0.0;

    ren->nbOutputShader = 0;
    ren->nbLensShader = 0;
    ren->outputShader = NULL;
    ren->lensShader = NULL;
    ren->atmosphere = NULL;
}

/*******************************************************************************

   $$L DK_sceneAllocate

   This function allocates a scene structure.

   Returned Value: DK_Scene structure

*******************************************************************************/

DK_Scene *DK_sceneAllocate( void )
{
   DK_Scene *scn;

   if( scn = (DK_Scene *) calloc( 1, sizeof( DK_Scene ) ) )
   {
      scn->name          	= NULL;
      scn->object_path         	= NULL;
      scn->texture_path        	= NULL;
      scn->picture_path         = NULL;
      scn->material_lib        	= NULL;
      scn->texture_lib        	= NULL;
      scn->palette_lib        	= NULL;

      scn->light          	= NULL;
      scn->model          	= NULL;
      scn->wave          	= NULL;
      scn->wave_filename        = NULL;
      scn->spline          	= NULL;

      /* play control */
      scn->start = 1;
      scn->end = 100;
      scn->step = 1;
      scn->current = 1;
      scn->pause = 0;
      scn->rate = 30;

      /* render */
      DK_renderInitialize(&scn->render);

      /* init camera */
      DK_cameraInitialize( &scn->camera );

      scn->setup_filename     = NULL;
      scn->udfFileInfo = NULL;
   }

   return( scn );
}

/*******************************************************************************

   $$L DK_sceneDispose

   This function disposes a scene structure.

   Returned Value: None

*******************************************************************************/

void DK_sceneDispose
    (
	DK_Scene **scn
    )
{
    if (scn != NULL && *scn != NULL)
    {
        DK_Scene *scene = *scn;

        _DK_FREE( scene->name, -1 );
        _DK_FREE( scene->object_path, -1 );
        _DK_FREE( scene->texture_path, -1 );
        _DK_FREE( scene->picture_path, -1 );
        _DK_FREE( scene->material_lib, -1 );
        _DK_FREE( scene->texture_lib, -1 );
        _DK_FREE( scene->palette_lib, -1 );

        _DK_FREE( scene->camera.pos_constraint_name, -1 );
        _DK_FREE( scene->camera.int_constraint_name, -1 );
        if( scene->camera.pos_path )
            DK_splineDispose( &scene->camera.pos_path );
        if( scene->camera.int_path )
            DK_splineDispose( &scene->camera.int_path );
        _DK_FREE( scene->camera.pos_path_name, -1 );
        _DK_FREE( scene->camera.int_path_name, -1 );

	DK_constraintDispose( &scene->camera.constr_list );
        DK_fcurveArrayDispose( scene->camera.fcv, DK_FCV_CAM_NUM );
        DK_actFcurveArrayDispose( scene->camera.active_fcv, DK_FCV_CAM_NUM );

        if( scene->light )
            DK_lightDispose( &scene->light );
        if( scene->model )
            DK_modelDispose( &scene->model );
        if( scene->spline )
            DK_splineDispose( &scene->spline );

        _DK_FREE( scene->wave_filename, -1 );
        if( scene->wave )
            DK_waveDispose( &scene->wave );

        _DK_FREE( scene->render.pictureFileName, -1 );
        DK_fcurveArrayDispose( scene->render.fade.fcv, DK_FCV_FAD_NUM );
        DK_actFcurveArrayDispose( scene->render.fade.active_fcv, DK_FCV_FAD_NUM );
        _DK_FREE( scene->render.preFrameFile, -1 );
        _DK_FREE( scene->render.postFrameFile, -1 );
        _DK_FREE( scene->render.statFile, -1 );


        if (scene->render.nbOutputShader) {
           int i;

  	   for (i = 0; i < scene->render.nbOutputShader; i++) {
              _DK_FREE( scene->render.outputShader[i].name, -1);
              _DK_FREE( scene->render.outputShader[i].description, -1);
           }
           _DK_FREE( scene->render.outputShader, -1);
        }

        if (scene->render.nbLensShader) {
           int i;

  	   for (i = 0; i < scene->render.nbLensShader; i++) {
              _DK_FREE( scene->render.lensShader[i].name, -1);
              _DK_FREE( scene->render.lensShader[i].description, -1);
           }
           _DK_FREE( scene->render.lensShader, -1);
        }

        if (scene->render.atmosphere) {
           _DK_FREE( scene->render.atmosphere->name, -1);
           _DK_FREE( scene->render.atmosphere->description, -1);
           _DK_FREE( scene->render.atmosphere, -1);
        }

	_DK_FREE( scene->setup_filename, -1 );

        /*
        *** Free up custom shaders' parameter filenames
        */
        if (scene->udfFileInfo) {
	   DK_UdfFileInfo *cur, *next;
	   
	   for (cur = scene->udfFileInfo; cur; cur = next) {
              next = cur->next;
              _DK_FREE (cur->filename, -1);
              _DK_FREE (cur->instance, -1);
	      _DK_FREE (cur, -1);
           }
        }

        _DK_FREE(*scn, -1);
    }
}

static int DK_pictureWriteAsciiAnimation
    (
	FILE *file,		/* file io pointer */
	DK_RenderingSetup *ren	/* pointer to picture variable */
    )
{
    DK_incIndent(1);
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_PIC_TYPE_TOKEN);
    switch (ren->picFormat)
    {
	case DK_FMT_35MM_137:
	    _DK_OUT_TOKEN(DK_A_PIC_35_137_TYPE_TOKEN);
	    break;
	case DK_FMT_35MM_166:
	    _DK_OUT_TOKEN(DK_A_PIC_35_166_TYPE_TOKEN);
	    break;
	case DK_FMT_35MM_185:
	    _DK_OUT_TOKEN(DK_A_PIC_35_185_TYPE_TOKEN);
	    break;
	case DK_FMT_70MM_PANAVISION:
	    _DK_OUT_TOKEN(DK_A_PIC_70_PAN_TYPE_TOKEN);
	    break;
	case DK_FMT_70MM_IMAX:
	    _DK_OUT_TOKEN(DK_A_PIC_70_IMAX_TYPE_TOKEN);
	    break;
	case DK_FMT_70MM_OMNIMAX:
	    _DK_OUT_TOKEN(DK_A_PIC_70_OMNI_TYPE_TOKEN);
	    break;
	case DK_FMT_NTSC:
	    _DK_OUT_TOKEN(DK_A_PIC_NTSC_TYPE_TOKEN);
	    break;
	case DK_FMT_PAL:
	    _DK_OUT_TOKEN(DK_A_PIC_PAL_TYPE_TOKEN);
	    break;
	case DK_FMT_HDTV:
	    _DK_OUT_TOKEN(DK_A_PIC_HDTV_TYPE_TOKEN);
	    break;
	case DK_FMT_SLIDE_35MM:
	    _DK_OUT_TOKEN(DK_A_PIC_SLIDE_35_TYPE_TOKEN);
	    break;
	case DK_FMT_6CM_X_6CM:
	    _DK_OUT_TOKEN(DK_A_PIC_6X6_TYPE_TOKEN);
	    break;
	case DK_FMT_6CM_X_9CM:
	    _DK_OUT_TOKEN(DK_A_PIC_6X9_TYPE_TOKEN);
	    break;
	case DK_FMT_4in_X_5in:
	    _DK_OUT_TOKEN(DK_A_PIC_4X5_TYPE_TOKEN);
	    break;
	case DK_FMT_CUSTOM_ASPECT:
	    _DK_OUT_TOKEN(DK_A_PIC_CUSTOM_ASPECT_TYPE_TOKEN);
	    break;
	case DK_FMT_CUSTOM_SIZE:
	    _DK_OUT_TOKEN(DK_A_PIC_CUSTOM_SIZE_TYPE_TOKEN);
	    break;
    }
    if( ren->picFormat == DK_FMT_CUSTOM_ASPECT ) 
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_PIC_ASPECT_TOKEN);
        _DK_OUT_FVALUE(ren->aspectRatio);
    }
    else if (ren->picFormat == DK_FMT_CUSTOM_SIZE)
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_PIC_WIDTH_TOKEN);
        _DK_OUT_FVALUE(ren->aspectWidth);
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_PIC_HEIGHT_TOKEN);
        _DK_OUT_FVALUE(ren->aspectHeight);
    }
    DK_incIndent(-1);
    return(TRUE);
}

static int DK_pictureReadAsciiAnimation
    (
	FILE *file,		/* file io pointer */
	DK_RenderingSetup *ren	/* pointer to picture variable */
    )
{
    int first = TRUE;

    while (TRUE)
    {
	if (strcmp(DK_A_PIC_TYPE_TOKEN, DK_animToken) == 0)
	{
	    if (DK_in_animToken(file) == FALSE)
		return(FALSE);
	    if (strcmp(DK_A_PIC_35_137_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_35MM_137;
	    else if (strcmp(DK_A_PIC_35_166_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_35MM_166;
	    else if (strcmp(DK_A_PIC_35_185_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_35MM_185;
	    else if (strcmp(DK_A_PIC_70_PAN_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_70MM_PANAVISION;
	    else if (strcmp(DK_A_PIC_70_IMAX_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_70MM_IMAX;
	    else if (strcmp(DK_A_PIC_70_OMNI_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_70MM_OMNIMAX;
	    else if (strcmp(DK_A_PIC_NTSC_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_NTSC;
	    else if (strcmp(DK_A_PIC_PAL_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_PAL;
	    else if (strcmp(DK_A_PIC_HDTV_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_HDTV;
	    else if (strcmp(DK_A_PIC_SLIDE_35_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_SLIDE_35MM;
	    else if (strcmp(DK_A_PIC_6X6_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_6CM_X_6CM;
	    else if (strcmp(DK_A_PIC_6X9_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_6CM_X_9CM;
	    else if (strcmp(DK_A_PIC_4X5_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_4in_X_5in;
	    else if (strcmp(DK_A_PIC_CUSTOM_ASPECT_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_CUSTOM_ASPECT;
	    else if (strcmp(DK_A_PIC_CUSTOM_SIZE_TYPE_TOKEN, DK_animToken) == 0)
	        ren->picFormat = DK_FMT_CUSTOM_SIZE;
	    else if ( DK_option.verbose == TRUE)
		fprintf( DK_option.msgFd, "DK_PictureReadAsciiAnimation: \
error, unknown picture format '%s'.\n",DK_animToken);
	}
	else if (strcmp(DK_A_PIC_ASPECT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue(file, &ren->aspectRatio);
	}
	else if (strcmp(DK_A_PIC_WIDTH_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue(file, &ren->aspectWidth);
	}
	else if (strcmp(DK_A_PIC_HEIGHT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue(file, &ren->aspectHeight);
	}
	else 
	{
	    if (first)
		DK_in_animToken(file);
	    return(TRUE);
	}
	if (DK_in_animToken(file) == FALSE)
	    return(FALSE);
	first = FALSE;
	ren->pictureFile = TRUE;
    }
}


/*******************************************************************************

   $$L DK_headerWriteAsciiAnimation

   This function writes the file header.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_headerWriteAsciiAnimation
    (
	FILE *file, 	/* file io pointer */
	char *comment,  /* user comment */
	float version   /* version number */
    )
{
    char h[DK_HEADER_LENGTH];

    _DK_OUT_TOKEN(DK_A_SCENE_ID_TOKEN);
    _DK_OUT_INDENT();
    sprintf(h,"%s v%4.2f",comment, version);
    _DK_OUT_TOKEN(h);
    _DK_OUT_INDENT();
    _DK_OUT_INDENT();

    return(TRUE);
}

/*******************************************************************************

   $$L DK_headerReadAsciiAnimation

   This function reads the file header.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_headerReadAsciiAnimation
    (
	FILE *file,	  /* file io pointer */
	float *version    /* version number */
    )
{
    int ci, i;

    /* skip white space */
    while ((ci = fgetc(file)) != EOF && (ci == ' ' || ci == '\t' || ci == '\n'))
	_DK_NOP();
    if (ci == EOF)
	return(FALSE);
    /* read up to and including the new line */
    DK_animTokenLine[0] = ci;
    i = 1;
    while ((ci = fgetc(file)) != EOF && ci != '\n')
	DK_animTokenLine[i++] = ci;
    DK_animTokenLine[i] = '\0';

    /* get version number, start at the end and try to find a 'v' */
    while (i >= 0)
    {
	if (DK_animTokenLine[i] == 'v')
	    break;
	else
	    i--;
    }
    if (i >= 0)
	*version = (float) atof(DK_animTokenLine+i+1);
    return(TRUE);
}

/*******************************************************************************

   $$L DK_pathWriteAsciiAnimation

   This function writes the paths.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_pathWriteAsciiAnimation
    (
	FILE *file,	/* file io pointer */
	DK_Scene *scn	/* scene pointer */
    )
{
    if (scn == NULL)
	return(FALSE);

    DK_incIndent(1);
    if (scn->object_path != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_OBJ_PATH_TOKEN);
        _DK_OUT_STRING(scn->object_path);
    }
    
    if (scn->texture_path != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_TEX_PATH_TOKEN);
        _DK_OUT_STRING(scn->texture_path);
    }

    if (scn->picture_path != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_PIC_PATH_TOKEN);
        _DK_OUT_STRING(scn->picture_path);
    }

    if (scn->material_lib != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_MAT_LIB_TOKEN);
        _DK_OUT_STRING(scn->material_lib);
    }

    if (scn->texture_lib != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_TEX_LIB_TOKEN);
        _DK_OUT_STRING(scn->texture_lib);
    }
    
    if (scn->palette_lib != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_PAL_LIB_TOKEN);
        _DK_OUT_STRING(scn->palette_lib);
    }
    DK_incIndent(-1);
    
    return(TRUE);
}

/*******************************************************************************

   $$L DK_pathReadAsciiAnimation

   This function reads the paths.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_pathReadAsciiAnimation
    (
	FILE *file,	/* file io pointer */
	DK_Scene *scn	/* scene pointer */
    )
{
    DK_in_animToken(file);
    while (TRUE)
    {
	if (strcmp(DK_A_OBJ_PATH_TOKEN, DK_animToken) == 0)
	{
	    if (scn->object_path != NULL)
		_DK_FREE(scn->object_path, -1);
	    scn->object_path = DK_stringReadAsciiAnimation(file);
	}
	else if (strcmp(DK_A_TEX_PATH_TOKEN, DK_animToken) == 0)
	{
	    if (scn->texture_path != NULL)
		_DK_FREE(scn->texture_path, -1);
	    scn->texture_path = DK_stringReadAsciiAnimation(file);
	}
	else if (strcmp(DK_A_PIC_PATH_TOKEN, DK_animToken) == 0)
	{
	    if (scn->picture_path != NULL)
		_DK_FREE(scn->picture_path, -1);
	    scn->picture_path = DK_stringReadAsciiAnimation(file);
	}
	else if (strcmp(DK_A_MAT_LIB_TOKEN, DK_animToken) == 0)
	{
	    if (scn->material_lib != NULL)
		_DK_FREE(scn->material_lib, -1);
	    scn->material_lib = DK_stringReadAsciiAnimation(file);
	}
	else if (strcmp(DK_A_TEX_LIB_TOKEN, DK_animToken) == 0)
	{
	    if (scn->texture_lib != NULL)
		_DK_FREE(scn->texture_lib, -1);
	    scn->texture_lib = DK_stringReadAsciiAnimation(file);
	}
	else if (strcmp(DK_A_PAL_LIB_TOKEN, DK_animToken) == 0)
	{
	    if (scn->palette_lib != NULL)
		_DK_FREE(scn->palette_lib, -1);
	    scn->palette_lib = DK_stringReadAsciiAnimation(file);
	}
	else
	    return(TRUE);
        DK_in_animToken(file);
    }
}


/*******************************************************************************

   $$L DK_renderWriteAsciiAnimation

   This function writes render parameters.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_renderWriteAsciiAnimation
    (
	FILE *file,		/* file io pointer */
	DK_RenderingSetup *renSup	/* renSupder pointer */
    )
{
    if (renSup == NULL)
	return(FALSE);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_TYPE_TOKEN);
    switch (renSup->type)
    {
        case DK_REN_WIREFRAME:
	    _DK_OUT_TOKEN(DK_A_REN_WIRE_TYPE_TOKEN);
	    break;
        case DK_REN_HIDDENLINE:
	    _DK_OUT_TOKEN(DK_A_REN_HIDDEN_TYPE_TOKEN);
	    break;
        case DK_REN_SOFTIMAGE:
	    _DK_OUT_TOKEN(DK_A_REN_RENDER_TYPE_TOKEN);
	    break;
        case DK_REN_HARDWARE:
	    _DK_OUT_TOKEN(DK_A_REN_HARDWARE_TYPE_TOKEN);
	    break;
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_FRAME_TOKEN);
    _DK_OUT_SVALUE( renSup->start);
    _DK_OUT_SVALUE( renSup->end);
    _DK_OUT_SVALUE( renSup->step);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_fBUF_TOKEN);
    if (renSup->frameBuffer)
	_DK_OUT_TOKEN(DK_A_ACTIVE_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_INACTIVE_TOKEN);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_OUTFILE_TOKEN);
    _DK_OUT_STRING(renSup->pictureFileName);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_RES_TOKEN);
    _DK_OUT_SVALUE( renSup->xResolution);
    _DK_OUT_SVALUE( renSup->yResolution);

    if (renSup->pictureFile == TRUE)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_REN_PICFORMAT_TOKEN);
	DK_pictureWriteAsciiAnimation(file, renSup);
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_RATIO_TOKEN);
    _DK_OUT_FVALUE( renSup->pixelRatio);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_FILTER_TOKEN);
    _DK_OUT_SVALUE( renSup->samplingFilter);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_RDEPTH_TOKEN);
    _DK_OUT_SVALUE( renSup->rayMaxDepth);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_TLEAF_TOKEN);
    _DK_OUT_SVALUE( renSup->rayTrianglesPerLeaf);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_TDEPTH_TOKEN);
    _DK_OUT_SVALUE( renSup->rayMaxTreeDepth);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_PRESCRIPT_TOKEN);
    _DK_OUT_STRING(renSup->preFrameFile);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_POSTSCRIPT_TOKEN);
    _DK_OUT_STRING(renSup->postFrameFile);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_TIME_STAT_TOKEN);
    if (renSup->statTiming)
	_DK_OUT_TOKEN(DK_A_YES_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_NO_TOKEN);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_MEMORY_STAT_TOKEN);
    if (renSup->statMemory)
	_DK_OUT_TOKEN(DK_A_YES_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_NO_TOKEN);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_TREE_STAT_TOKEN);
    if (renSup->statTree)
	_DK_OUT_TOKEN(DK_A_YES_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_NO_TOKEN);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_LINE_STAT_TOKEN);
    if (renSup->statLine)
	_DK_OUT_TOKEN(DK_A_YES_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_NO_TOKEN);

    if (renSup->antialiasType == DK_REN_ANTIALIAS_BARTLET || 
	renSup->antialiasType == DK_REN_ANTIALIAS_ADAPTIVE)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_REN_ANTIALIAS_TOKEN);
        switch (renSup->antialiasType)
        {
	    case DK_REN_ANTIALIAS_BARTLET:
	        _DK_OUT_TOKEN(DK_A_REN_FIXED_TOKEN);
	        break;
            case DK_REN_ANTIALIAS_ADAPTIVE:
	        _DK_OUT_TOKEN(DK_A_REN_ADAPTIVE_TOKEN);
	        break;
        }
        _DK_OUT_SVALUE(renSup->samplingFilter);
        _DK_OUT_SVALUE(renSup->maxLevel);
        _DK_OUT_FVALUE(renSup->adaptiveThreshold);
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_BACK_CULLING_TOKEN);
    if (renSup->backCulling)
	_DK_OUT_TOKEN(DK_A_ACTIVE_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_INACTIVE_TOKEN);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_FADE_ACTIVE_TOKEN);
    if (renSup->fade.active)
        _DK_OUT_TOKEN(DK_A_ACTIVE_TOKEN);
    else
        _DK_OUT_TOKEN(DK_A_INACTIVE_TOKEN);
   
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_FADE_START_TOKEN);
    DK_valueWriteAsciiAnimation(file, renSup->fade.start, renSup->fade.fcv[DK_FCV_FAD_START]);
   
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_FADE_END_TOKEN);
    DK_valueWriteAsciiAnimation(file, renSup->fade.end, renSup->fade.fcv[DK_FCV_FAD_STOP]);
    
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_FADE_COLOUR_TOKEN);
    DK_value3WriteAsciiAnimation(file,
	renSup->fade.colour.r, renSup->fade.colour.g, renSup->fade.colour.b,
	renSup->fade.fcv[DK_FCV_FAD_RED], renSup->fade.fcv[DK_FCV_FAD_GREEN],
	renSup->fade.fcv[DK_FCV_FAD_BLUE]);
    
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_FADE_TRANSP_TOKEN);
    DK_valueWriteAsciiAnimation(file, renSup->fade.transparency, 
	renSup->fade.fcv[DK_FCV_FAD_TRANSP]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_AMBIENCE_TOKEN);
    DK_value3WriteAsciiAnimation(file, renSup->ambiance.r, renSup->ambiance.g,
	renSup->ambiance.b, NULL, NULL, NULL);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_MOTIONBLUR_TOKEN);
    _DK_OUT_SVALUE( renSup->motion_blur);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_SHUTTERSPEED_TOKEN);
    _DK_OUT_FVALUE( renSup->shutter_speed);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_MINMOVEMENT_TOKEN);
    _DK_OUT_FVALUE( renSup->min_movement);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_ONFIELD_TOKEN);
    if (renSup->onfields)
	_DK_OUT_TOKEN(DK_A_YES_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_NO_TOKEN);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_REN_FIRSTFIELD_TOKEN);
    _DK_OUT_SVALUE( renSup->firstfield);

    return(TRUE);
}


/*******************************************************************************

   $$L DK_layoutReadAsciiAnimation

   This function reads the window layout.

   Notes: The layout parameters are ignored.

*******************************************************************************/

static void DK_layoutReadAsciiAnimation
    (
	FILE *file	/* file io pointer */
    )
{
    DK_Vector min, max;
    char *cptr;
    int first = TRUE;

    DK_in_animToken(file);
    while (TRUE)
    {
	if (strcmp(DK_A_LAYOUT_WINDOW_TYPE_TOKEN, DK_animToken) == 0) 
	{
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LAYOUT_CAMERA_TYPE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LAYOUT_CAM_POS_TOKEN, DK_animToken) == 0)
	{
	    DK_vectorReadAsciiAnimation(file, &min);
	}
	else if (strcmp( DK_A_LAYOUT_CAM_INT_TOKEN, DK_animToken) == 0)
	{
	    DK_vectorReadAsciiAnimation(file, &min);
	}
	else if (strcmp( DK_A_LAYOUT_CAM_FOV_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	}
	else if (strcmp( DK_A_LAYOUT_CAM_ASPECT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	}
	else if (strcmp( DK_A_LAYOUT_CAM_NEAR_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	}
	else if (strcmp( DK_A_LAYOUT_MIN_TOKEN, DK_animToken) == 0)
	{
	    DK_vectorReadAsciiAnimation(file, &min);
	}
	else if (strcmp( DK_A_LAYOUT_MAX_TOKEN, DK_animToken) == 0)
	{
	    DK_vectorReadAsciiAnimation(file, &max);
	}
	else if (strcmp( DK_A_LAYOUT_GRID_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LAYOUT_AXIS_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LAYOUT_RULER_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LAYOUT_FIELD_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LAYOUT_XLOCK_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LAYOUT_YLOCK_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LAYOUT_ZLOCK_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LAYOUT_DELTA_TOKEN, DK_animToken) == 0)
	{
	    DK_vectorReadAsciiAnimation(file, NULL);
	}
	else if (strcmp( DK_A_LAYOUT_DISPLAY_TYPE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	}
	else if (strcmp( DK_A_LAYOUT_PICTURE_TOKEN, DK_animToken) == 0)
	{
	    if ((cptr = DK_stringReadAsciiAnimation(file)) != NULL)
	    {
		_DK_FREE(cptr, -1);
	    }
	}
	else if (strcmp( DK_A_LAYOUT_PICTURE_FRAME_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    DK_in_animSValue_na(file);
	    DK_in_animSValue_na(file);
	    DK_in_animSValue_na(file);
	    DK_in_animSValue_na(file);
	    DK_in_animSValue_na(file);
	}
	else if (strcmp( DK_A_LAYOUT_DEPTHCUE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	    DK_in_animFValue_na(file);
	    DK_in_animSValue_na(file);
	    DK_in_animSValue_na(file);
	    DK_in_animSValue_na(file);
	}
	else 
	{
	    if (first)
		DK_in_animToken(file);
	    return;
	}
	DK_in_animToken(file);
	first = FALSE;
    }
}



/*******************************************************************************

   $$P DK_sceneWriteAsciiAnimation

   This routine writes the scene data file.

   Returned Value: TRUE or FALSE

*******************************************************************************/


int DK_sceneWriteAsciiAnimation
    (
	char *comment,		/* user comment */
	float version, 		/* version number */
	DK_Scene *scn		/* scene pointer */
    )
{
    DK_Model *mdl;
    DK_Spline *spl;
    DK_Light *lit;
    char *fileOut;
    FILE *file;

    if (scn == NULL)
    {
	if ( DK_option.verbose == TRUE)
	    fprintf( DK_option.msgFd,"Error, null scene pointer\n");
	return(FALSE);
    }

    if (scn->name == NULL || scn->name[0] == '\0')
    {
	if ( DK_option.verbose == TRUE)
            fprintf( DK_option.msgFd,"Error, no scene filename\n");
	return(FALSE);
    }

    DK_incIndent(0);
    /* get output file */
    if ((fileOut = _DK_CALLOC(char, strlen(scn->name) + 5, -1)) == NULL)
	return(FALSE);
    strcpy(fileOut, scn->name);
    strcat(fileOut, ".scn");

    if ((file = fopen(fileOut,"wb")) == NULL)
    {
	if ( DK_option.verbose == TRUE)
            fprintf( DK_option.msgFd, "Unable to open file %s", fileOut);
        _DK_FREE(fileOut, -1);
        return(FALSE);
    }

    if ( DK_option.verbose == TRUE)
        fprintf( DK_option.msgFd, "Saving scene file '%s'.\n", scn->name);
     
    DK_headerWriteAsciiAnimation(file, comment, version);

    /* write paths */
    if (scn->object_path != NULL || scn->texture_path != NULL ||
	scn->picture_path != NULL || scn->material_lib != NULL ||
	scn->texture_lib != NULL || scn->palette_lib != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_SCENE_PATH_TOKEN);
        DK_pathWriteAsciiAnimation(file, scn);
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SCENE_CLASSIC_SCALING_TOKEN );

    if ( DK_CLASSIC_SCALING ) 
    {
       _DK_OUT_TOKEN( DK_A_YES_TOKEN );
    } 
    else
    {
       _DK_OUT_TOKEN( DK_A_NO_TOKEN );
    }

    /* write start frame */
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SCENE_FRAME_TOKEN);
    _DK_OUT_SVALUE( scn->start);
    /* write end frame */
    _DK_OUT_SVALUE( scn->end);
    /* write current frame */
    _DK_OUT_SVALUE( scn->current);
    /* write frame step */
    _DK_OUT_SVALUE( scn->step);

    /* write pause */
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SCENE_PAUSE_TOKEN);
    _DK_OUT_FVALUE( scn->pause);

    /* write rate */
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SCENE_RATE_TOKEN);
    _DK_OUT_FVALUE( scn->rate);

    DK_renderWriteAsciiAnimation(file, &scn->render);

    /* write camera */
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SCENE_CAMERA_TOKEN);
    DK_cameraWriteAsciiAnimation(file, &scn->camera);

    /* write lights */
    lit = scn->light;
    while (lit != NULL)
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN( DK_A_SCENE_LIGHT_TOKEN);
	DK_lightWriteAsciiAnimation(file, lit);
	lit = lit->next;
    }

    /* write hierarchy */
    mdl = scn->model;
    while (mdl != NULL)
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN( DK_A_SCENE_HIER_TOKEN);
	DK_objectWriteAsciiAnimation(file, mdl);
	mdl = mdl->next;
    }

    /* write spline curves if not a path */
    spl = scn->spline;
    while (spl != NULL)
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN( DK_A_SCENE_SPLINE_TOKEN);
	DK_splineWriteAsciiAnimation(file, spl);
	spl = spl->next;
    }

    if (scn->wave != NULL)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_SCENE_WAVE_TOKEN);
        DK_waveWriteAsciiAnimation(file, scn->wave_filename, scn->wave);
    }

    _DK_OUT_INDENT();

    fflush(file);

    _DK_FREE(fileOut, -1);
    fclose(file);

    /* reset indentation to zero */
    DK_setIndentLength(0);
    DK_incIndent(0);

    return(TRUE);
}

/*******************************************************************************

   $$P DK_sceneReadAsciiAnimation

   This routine reads the scene data file.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_sceneReadAsciiAnimation
    (
	float *fileVersion, 	/* file version */
	DK_Scene *scn		/* scene pointer */
    )
{
    DK_Light *lit, *lit2;
    DK_Spline *spl, *spl2;
    char *fileIn;
    FILE *file;
    int read_animToken, valid, c, errorMsgOut = FALSE;

    if (scn == NULL)
    {
        if ( DK_option.verbose == TRUE)
	    fprintf( DK_option.msgFd,"Error, null scene pointer\n");
	return(FALSE);
    }

    if (scn->name == NULL || scn->name[0] == '\0')
    {
        if ( DK_option.verbose == TRUE)
            fprintf( DK_option.msgFd,"Error, no scene filename\n");
	return(FALSE);
    }

    /* get input file */
    if ((fileIn = _DK_CALLOC(char, strlen(scn->name) + 6, -1)) == NULL)
	return(FALSE);
    strcpy(fileIn, scn->name);
    strcat(fileIn, ".scn");

    /* read header */
    if ((file = fopen(fileIn, "rb")) == NULL)
    {
        if ( DK_option.verbose == TRUE)
            fprintf( DK_option.msgFd, "Unable to open file '%s'.\n", fileIn);
        _DK_FREE(fileIn, -1);
        return(FALSE);
    }
    else 
    {
        _DK_FREE(fileIn, -1);
	DK_in_animToken(file);
	if (strcmp(DK_A_SCENE_ID_TOKEN, DK_animToken) != 0)
	{
            if ( DK_option.verbose == TRUE)
                fprintf( DK_option.msgFd, "Not a scene file\n");
	    fclose(file);
            return(FALSE);
	}
    }

    if ( DK_option.verbose == TRUE)
        fprintf( DK_option.msgFd, "Reading scene file '%s'.\n", scn->name);

    if (DK_headerReadAsciiAnimation(file, fileVersion) == FALSE)
    {
	fclose(file);
        if ( DK_option.verbose == TRUE)
	    fprintf( DK_option.msgFd,"Bad header.\n");
	return(FALSE);
    }

    /* STOP if scene file version is later than DKit release version */
    if (*fileVersion > DK_RELEASE_NUMBER)
    {
        if ( DK_option.verbose == TRUE)
	{
	    fprintf( DK_option.msgFd,"Warning: scene file version %g is later \
than the dkit release version %g\n", *fileVersion, DK_RELEASE_NUMBER);
	    fprintf( DK_option.msgFd,"Attempt to read: An error may occur...\n");
	}
    }

    DK_animFileVersion = *fileVersion;

    valid = DK_in_animToken(file);
    while (valid)
    {
        read_animToken = TRUE;
        if (strcmp( DK_A_SCENE_PATH_TOKEN, DK_animToken) == 0)
	{
	    DK_pathReadAsciiAnimation ( file, scn);
	    read_animToken = FALSE;
	}
        else if (strcmp( DK_A_SCENE_CLASSIC_SCALING_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
	       DK_CLASSIC_SCALING = TRUE; 
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
	       DK_CLASSIC_SCALING = FALSE; 
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: error, \
unknown classic scaling DK_animToken '%s'.\n",DK_animToken);
	}
        else if (strcmp( DK_A_SCENE_FRAME_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    scn->start = DK_animSValue;
	    DK_in_animSValue_na(file);
	    scn->end = DK_animSValue;
	    DK_in_animSValue_na(file);
	    scn->current = DK_animSValue;
	    DK_in_animSValue_na(file);
	    scn->step = DK_animSValue;
	}
        else if (strcmp( DK_A_SCENE_PAUSE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	    scn->pause = (short) DK_animFValue;
	}
        else if (strcmp( DK_A_SCENE_RATE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	    scn->rate = (short) DK_animFValue;
	}
        else if (strcmp( DK_A_SCENE_CAMERA_VISIBLE_TOKEN, DK_animToken) == 0)
	{
	    /* ignore camera visible */
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_SCENE_CAMERA_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    DK_cameraReadAsciiAnimation(file, &scn->camera);
	    read_animToken = FALSE;
	}
        else if (strcmp( DK_A_SCENE_LIGHT_TOKEN, DK_animToken) == 0)
	{
	    /* allocate and insert light */
	    lit = DK_lightAllocate();
	    if (scn->light == NULL)
		scn->light = lit;
	    else
	    {
		lit2 = scn->light;
		while (lit2 != NULL && lit2->next != NULL)
		    lit2 = lit2->next;
		lit2->next = lit;
	    }
	    DK_lightReadAsciiAnimation(file, lit);
	    read_animToken = FALSE;
	}
	else if (strcmp( DK_A_SCENE_SPLINE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    spl = DK_splineAllocate();
	    if (scn->spline == NULL)
		scn->spline = spl;
	    else
	    {
		spl2 = scn->spline;
		while (spl2 != NULL && spl2->next != NULL)
		{
		    spl2 = spl2->next;
		}
		spl2->next = spl;
	    }
	    DK_splineReadAsciiAnimation(file, spl);
	    read_animToken = FALSE;
	}
        else if (strcmp( DK_A_SCENE_HIER_TOKEN, DK_animToken) == 0)
	{
            DK_objectReadAsciiAnimation(file, &scn->model,
						     scn->object_path );
	    read_animToken = FALSE;
	}
	else if (strcmp( DK_A_SCENE_WAVE_TOKEN, DK_animToken) == 0)
	{
	    DK_waveReadAsciiAnimation(file, &scn->wave_filename, &scn->wave);
	    read_animToken = FALSE;
	}
	else if (strcmp( DK_A_SCENE_LAYOUT_TOKEN, DK_animToken) == 0)
	{
	    DK_layoutReadAsciiAnimation(file);
	    read_animToken = FALSE;
	}
	else if (strcmp( DK_A_SCENE_SELECT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	}
    
	/* rendering stuff */
    
        else if (strcmp( DK_A_REN_TYPE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_REN_WIRE_TYPE_TOKEN, DK_animToken) == 0)
	        scn->render.type = DK_REN_WIREFRAME;
	    else if (strcmp(DK_A_REN_HIDDEN_TYPE_TOKEN, DK_animToken) == 0)
	        scn->render.type = DK_REN_HIDDENLINE;
	    else if (strcmp(DK_A_REN_RENDER_TYPE_TOKEN, DK_animToken) == 0)
	        scn->render.type = DK_REN_SOFTIMAGE;
	    else if (strcmp(DK_A_REN_HARDWARE_TYPE_TOKEN, DK_animToken) == 0)
	        scn->render.type = DK_REN_HARDWARE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: error, unknown \
ren type DK_animToken '%s'.\n",DK_animToken);
	}
        else if (strcmp( DK_A_REN_FRAME_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &scn->render.start);
	    DK_in_animSValue(file, &scn->render.end);
	    DK_in_animSValue(file, &scn->render.step);
	}
        else if (strcmp( DK_A_REN_fBUF_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_ACTIVE_TOKEN, DK_animToken) == 0)
	        scn->render.frameBuffer = TRUE;
	    else if (strcmp(DK_A_INACTIVE_TOKEN, DK_animToken) == 0)
	        scn->render.frameBuffer = FALSE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: error, unknown \
fbuffer active DK_animToken '%s'.\n",DK_animToken);
	}
        else if (strcmp( DK_A_REN_OUTFILE_TOKEN, DK_animToken) == 0)
	{
	    scn->render.pictureFileName = DK_stringReadAsciiAnimation(file);
	}
        else if (strcmp( DK_A_REN_RES_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &scn->render.xResolution);
	    DK_in_animSValue(file, &scn->render.yResolution);
	}
        else if (strcmp( DK_A_REN_PICFORMAT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    DK_pictureReadAsciiAnimation(file, &scn->render);
	    read_animToken = FALSE;
	}
        else if (strcmp( DK_A_REN_RATIO_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue(file, &scn->render.pixelRatio);
	}
        else if (strcmp( DK_A_REN_FILTER_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &scn->render.samplingFilter);
	}
        else if (strcmp( DK_A_REN_RDEPTH_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &scn->render.rayMaxDepth);
	}
        else if (strcmp( DK_A_REN_TLEAF_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &scn->render.rayTrianglesPerLeaf);
	}
        else if (strcmp( DK_A_REN_TDEPTH_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &scn->render.rayMaxTreeDepth);
	}
        else if (strcmp( DK_A_REN_PRESCRIPT_TOKEN, DK_animToken) == 0)
	{
	    scn->render.preFrameFile = DK_stringReadAsciiAnimation(file);
	}
        else if (strcmp( DK_A_REN_POSTSCRIPT_TOKEN, DK_animToken) == 0)
	{
	    scn->render.postFrameFile = DK_stringReadAsciiAnimation(file);
	}
        else if (strcmp( DK_A_REN_TIME_STAT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
	        scn->render.statTiming = TRUE;
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
	        scn->render.statTiming = FALSE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: error, unknown \
time stat DK_animToken '%s'.\n",DK_animToken);
	}
        else if (strcmp( DK_A_REN_MEMORY_STAT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
	        scn->render.statMemory = TRUE;
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
	        scn->render.statMemory = FALSE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: error, unknown \
memory stat DK_animToken '%s'.\n",DK_animToken);
	}
        else if (strcmp( DK_A_REN_TREE_STAT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
	        scn->render.statTree = TRUE;
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
	        scn->render.statTree = FALSE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: error, \
unknown tree stat DK_animToken '%s'.\n",DK_animToken);
	}
        else if (strcmp( DK_A_REN_LINE_STAT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
	        scn->render.statLine = TRUE;
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
	        scn->render.statLine = FALSE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: \
error, unknown line stat DK_animToken '%s'.\n",DK_animToken);
	}
        else if (strcmp( DK_A_REN_ANTIALIAS_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_REN_FIXED_TOKEN, DK_animToken) == 0)
	        scn->render.antialiasType = DK_REN_ANTIALIAS_BARTLET;
	    else if (strcmp(DK_A_REN_ADAPTIVE_TOKEN, DK_animToken) == 0)
	        scn->render.antialiasType = DK_REN_ANTIALIAS_ADAPTIVE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: error, unknown \
antialias DK_animToken '%s'.\n",DK_animToken);
	    DK_in_animSValue(file, &scn->render.samplingFilter);
	    DK_in_animSValue(file, &scn->render.maxLevel);
	    DK_in_animFValue(file, &scn->render.adaptiveThreshold);
	}
        else if (strcmp( DK_A_REN_BACK_CULLING_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_ACTIVE_TOKEN, DK_animToken) == 0)
	        scn->render.backCulling = TRUE;
	    else if (strcmp(DK_A_INACTIVE_TOKEN, DK_animToken) == 0)
	        scn->render.backCulling = FALSE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: error, unknown \
back culling DK_animToken '%s'.\n",DK_animToken);
	}
        else if (strcmp( DK_A_REN_FADE_ACTIVE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_ACTIVE_TOKEN, DK_animToken) == 0)
	        scn->render.fade.active = TRUE;
	    else if (strcmp(DK_A_INACTIVE_TOKEN, DK_animToken) == 0)
	        scn->render.fade.active = FALSE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: \
error, unknown fade active DK_animToken '%s'.\n",DK_animToken);
	}

        else if (strcmp( DK_A_REN_FADE_START_TOKEN, DK_animToken) == 0)
	{
	    DK_valueReadAsciiAnimation(file, &scn->render.fade.start,
		&scn->render.fade.fcv[DK_FCV_FAD_START],TRUE);
	    read_animToken = FALSE;
	}

        else if (strcmp( DK_A_REN_FADE_END_TOKEN, DK_animToken) == 0)
	{
	    DK_valueReadAsciiAnimation(file, &scn->render.fade.end, 
		&scn->render.fade.fcv[DK_FCV_FAD_STOP], TRUE);
	    read_animToken = FALSE;
	}

        else if (strcmp( DK_A_REN_FADE_COLOUR_TOKEN, DK_animToken) == 0)
	{
	    DK_value3ReadAsciiAnimation(file, &scn->render.fade.colour.r, 
		&scn->render.fade.colour.g, &scn->render.fade.colour.b,
		&scn->render.fade.fcv[DK_FCV_FAD_RED],
		&scn->render.fade.fcv[DK_FCV_FAD_GREEN],
		&scn->render.fade.fcv[DK_FCV_FAD_BLUE]);
	    read_animToken = FALSE;
	}

        else if (strcmp( DK_A_REN_FADE_TRANSP_TOKEN, DK_animToken) == 0)
	{
	    DK_valueReadAsciiAnimation(file, &scn->render.fade.transparency, 
		&scn->render.fade.fcv[DK_FCV_FAD_TRANSP], TRUE);
	    read_animToken = FALSE;
	}

        else if (strcmp( DK_A_REN_AMBIENCE_TOKEN, DK_animToken) == 0)
	{
	    DK_value3ReadAsciiAnimation(file, 
		&scn->render.ambiance.r, &scn->render.ambiance.g,
		&scn->render.ambiance.b, NULL, NULL, NULL);
	    read_animToken = FALSE;
	}
        else if (strcmp( DK_A_REN_MOTIONBLUR_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &scn->render.motion_blur);
	}
        else if (strcmp( DK_A_REN_SHUTTERSPEED_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue(file, &scn->render.shutter_speed);
	}
        else if (strcmp( DK_A_REN_MINMOVEMENT_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue(file, &scn->render.min_movement);
	}
        else if (strcmp( DK_A_REN_ONFIELD_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
	        scn->render.onfields = TRUE;
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
	        scn->render.onfields = FALSE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: \
error, unknown on field DK_animToken '%s'.\n",DK_animToken);
	}
        else if (strcmp( DK_A_REN_FIRSTFIELD_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &scn->render.firstfield);
	}
	else
	{
	    /* check if EOF, if so then don't print messgae */
	    c = getc(file);
	    if (c != EOF)
	    {
		ungetc(c, file);
		if (errorMsgOut == FALSE &&  DK_option.verbose == TRUE)
	            fprintf( DK_option.msgFd, "DK_SceneReadAsciiAnimation: \
error, unknown DK_animToken '%s'.\n", DK_animToken);
		errorMsgOut = TRUE;
	    }
	}
	valid = TRUE;
	if (read_animToken)
            valid = DK_in_animToken(file);
    }

    fclose(file);

    return(TRUE);
}
