/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyPicture.c,v $
   $Revision: 1.9.14.1 $ $Date: 1995/06/09 22:00:16 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's picture functions,
   its content is sufficient to read, write and manage picture data structure
   into SOFTIMAGE's file format...

   Written by: Laurent Lauzon

   (c) Copyright 1990, 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinyByteswap.h"
#include "tinyPicture.h"

/******************************************************************************/
/*** Static function prototypes ***********************************************/
/******************************************************************************/

static void DK_channelsWriteRaw(  FILE *, short, short, char ** );
static void DK_channelsWritePure(  FILE *, short, short, char ** );
static void DK_channelsWriteMixed(  FILE *, short, short, char ** );

static void DK_channelsReadRaw(  FILE *, short, short, char ** );
static void DK_channelsReadPure(  FILE *, short, short, char ** );
static void DK_channelsReadMixed(  FILE *, short, short, char ** );

/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

DK_Picture *DK_pictureAllocate( void )
{
   DK_Picture    *pic;

   if( pic = (DK_Picture *) calloc( 1, sizeof( DK_Picture ) ) )
   {
      pic->width  = 0;
      pic->height = 0;
      pic->ratio  = 1.0;
      pic->fields =  DK_NO_FIELD;
      pic->red    =  pic->ptrRed   = NULL;
      pic->green  =  pic->ptrGreen = NULL;
      pic->blue   =  pic->ptrBlue  = NULL;
      pic->alpha  =  pic->ptrAlpha = NULL;
      pic->shadow = pic->ptrShadow = NULL;
      pic->depth = pic->ptrDepth = NULL;
      pic->auxiliary1 = pic->ptrAuxiliary1 = NULL;
      pic->auxiliary2 = pic->ptrAuxiliary2 = NULL;
   }
   return( pic );
}

DK_Boolean DK_pictureDispose( DK_Picture **pic )
{
   if( *pic )
   {
      if( (*pic)->red )
         free( (*pic)->red );
      if( (*pic)->green )
         free( (*pic)->green );
      if( (*pic)->blue )
         free( (*pic)->blue );
      if( (*pic)->alpha )
         free( (*pic)->alpha );

      free( *pic );
   }

   return( TRUE );
}

DK_Picture *DK_pictureReadFile( DK_String filename )
{
    FILE			*file;
   DK_Picture		*pic;
    DK_channelPacket	*ptr, *pck;
   int			 i;

   if( ( pic = DK_pictureAllocate() ) == NULL )
      return( NULL );

   if( file = DK_pictureReadFileHeader( filename, pic, &pck ) )
   {
      for( ptr = pck; ptr != NULL; ptr = ptr->next )
      {
         if( ptr->channels &  DK_RED_CHANNEL )
            pic->ptrRed   = pic->red   =
			 (unsigned char *) malloc( pic->height * pic->width );
         if( ptr->channels &  DK_GREEN_CHANNEL )
            pic->ptrGreen = pic->green =
			 (unsigned char *) malloc( pic->height * pic->width );
         if( ptr->channels &  DK_BLUE_CHANNEL )
            pic->ptrBlue  = pic->blue  =
			 (unsigned char *) malloc( pic->height * pic->width );
         if( ptr->channels & DK_ALPHA_CHANNEL )
            pic->ptrAlpha = pic->alpha =
			 (unsigned char *) malloc( pic->height * pic->width );
      }

      for( i = 0; i < pic->height; i++ )
         DK_pictureReadScanline( file, pic, pck );

      DK_channelPacketDispose( &pck );
      fclose( file );
      return( pic );
   }
   return( NULL );
}

DK_Boolean DK_pictureWriteFile
   (
      DK_String filename,
      DK_Picture *pic,
      DK_channelPacket *fmt   /* if != NULL then format used */
   )
{
    DK_channelPacket	*pck = fmt;
   short		 i;
    FILE			*file;

   if( ( file = DK_pictureWriteFileHeader( filename, pic, &pck ) ) != NULL )
   {
      pic->ptrRed   = pic->red;
      pic->ptrGreen = pic->green;
      pic->ptrBlue  = pic->blue;
      pic->ptrAlpha = pic->alpha;
      for( i = 0; i < pic->height; i++ )
         DK_pictureWriteScanline( file, pic, pck );
      fclose(file);
   }
   
   if( fmt == NULL )
      DK_channelPacketDispose( &pck );
   return( TRUE );
}

FILE *DK_pictureWriteFileHeader( DK_String filename, DK_Picture *pic,
			       DK_channelPacket **pck )
{
   char			 buf[ 256 ];
    FILE			*file;
   char		 	 chained;
    DK_channelPacket	*ptr;
   short		 pad = 0;

   if( (*pck) == NULL )
   {
      (*pck) = ( DK_channelPacket *) malloc( sizeof(  DK_channelPacket ) );
      (*pck)->size     = 8;
      (*pck)->type     =  DK_UNSIGNED_INTEGER |  DK_MIXED_RUN_LENGTH;
      (*pck)->channels =  DK_RED_CHANNEL |  DK_GREEN_CHANNEL |  DK_BLUE_CHANNEL;
      if( pic->alpha == NULL )
         (*pck)->next = NULL;
      else
      {
         (*pck)->next =
		 ( DK_channelPacket *) malloc( sizeof(  DK_channelPacket ) );
         (*pck)->next->size     = 8;
         (*pck)->next->type     =  DK_UNSIGNED_INTEGER |  DK_MIXED_RUN_LENGTH;
         (*pck)->next->channels = DK_ALPHA_CHANNEL;
         (*pck)->next->next     = NULL;
      }
   }

   if( ( strlen( filename ) > 4 ) &&
       ( strcmp( ".pic", &filename[ strlen( filename ) - 4 ] ) == 0 ) )
      strcpy( buf, filename );
   else
      sprintf( buf, "%s.pic", filename );

   if( file = fopen( buf, "wb" ) )
   {
      DK_headerWriteBinary( file, NULL );
      DK_String_fwrite( "PICT", 4, 1, file );

      DK_byteswap_fwrite( &pic->width, 2, 1, file );
      DK_byteswap_fwrite( &pic->height, 2, 1, file );
      DK_byteswap_fwrite( &pic->ratio, 4, 1, file );
      DK_byteswap_fwrite( &pic->fields, 2, 1, file );

      DK_byteswap_fwrite( &pad, 2, 1, file );

      for( ptr = *pck; ptr != NULL; ptr = ptr->next )
      {
         chained = (ptr->next != NULL);
         fputc( chained, file );
         fputc( ptr->size, file );
         fputc( ptr->type, file );
         fputc( ptr->channels, file );
      }
   }

   return( file );
}

 FILE *DK_pictureReadFileHeader( DK_String filename, DK_Picture *pic,
			      DK_channelPacket **pck )
{
   char			 buf[ 256 ];
    FILE			*file;
   char		 	 chained;
    DK_channelPacket	*ptr;
   long			 id;
   short		 pad;

   if( ( strlen( filename ) > 4 ) &&
       ( strcmp( ".pic", &filename[ strlen( filename ) - 4 ] ) == 0 ) )
      strcpy( buf, filename );
   else
      sprintf( buf, "%s.pic", filename );

   if( file = fopen( buf, "rb" ) )
   {
      id = DK_headerReadBinary( file, NULL, NULL );
      if( id != 'PICT' )
      {
         fclose( file );
         return( NULL );
      }

      DK_byteswap_fread( &pic->width, 2, 1, file );
      DK_byteswap_fread( &pic->height, 2, 1, file );
      DK_byteswap_fread( &pic->ratio, 4, 1, file );
      DK_byteswap_fread( &pic->fields, 2, 1, file );

      DK_byteswap_fread( &pad, 2, 1, file );

      ptr = NULL;
      do
      {
         if( ptr == NULL )
            ptr = *pck =
		 ( DK_channelPacket *) malloc( sizeof(  DK_channelPacket ) );
         else
         {
            ptr->next =
		 ( DK_channelPacket *) malloc( sizeof(  DK_channelPacket ) );
            ptr = ptr->next;
         }
         ptr->next = NULL;

         chained = fgetc( file );
         ptr->size = fgetc( file );
         ptr->type = fgetc( file );
         ptr->channels = fgetc( file );
      } while( chained != 0 );
   }

   return( file );
}

DK_Boolean DK_pictureReadScanline(  FILE *file, DK_Picture *pic,  DK_channelPacket *pck )
{
    DK_channelPacket	*ptr;
   char			*col[4];
   int			 nbcol;

   for( ptr = pck; ptr != NULL; ptr = ptr->next )
   {
      nbcol = 0;
      if( ptr->channels &  DK_RED_CHANNEL )
         col[ nbcol++ ] = (char *) pic->ptrRed;
      if( ptr->channels &  DK_GREEN_CHANNEL )
         col[ nbcol++ ] = (char *) pic->ptrGreen;
      if( ptr->channels &  DK_BLUE_CHANNEL )
         col[ nbcol++ ] = (char *) pic->ptrBlue;
      if( ptr->channels & DK_ALPHA_CHANNEL )
	 col[ nbcol++ ] = (char *) pic->ptrAlpha;

      switch( ptr->type & 0x0f )
      {
         case DK_UNCOMPRESSED:
	    DK_channelsReadRaw( file, pic->width, nbcol, col );
            break;
         case  DK_PURE_RUN_LENGTH:
	    DK_channelsReadPure( file, pic->width, nbcol, col );
            break;
         case  DK_MIXED_RUN_LENGTH:
	    DK_channelsReadMixed( file, pic->width, nbcol, col );
            break;
      }

      if( ptr->channels &  DK_RED_CHANNEL )
         pic->ptrRed   += pic->width;
      if( ptr->channels &  DK_GREEN_CHANNEL )
         pic->ptrGreen += pic->width;
      if( ptr->channels &  DK_BLUE_CHANNEL )
         pic->ptrBlue  += pic->width;
      if( ptr->channels & DK_ALPHA_CHANNEL )
	 pic->ptrAlpha += pic->width;
   }
   return( TRUE );
}

DK_Boolean DK_pictureWriteScanline(  FILE *file, DK_Picture *pic,  DK_channelPacket *pck )
{
    DK_channelPacket	*ptr;
   char			*col[4];
   int			 nbcol;

   for( ptr = pck; ptr != NULL; ptr = ptr->next )
   {
      nbcol = 0;
      if( ptr->channels &  DK_RED_CHANNEL )
         col[ nbcol++ ] = (char *) pic->ptrRed;
      if( ptr->channels &  DK_GREEN_CHANNEL )
         col[ nbcol++ ] = (char *) pic->ptrGreen;
      if( ptr->channels &  DK_BLUE_CHANNEL )
         col[ nbcol++ ] = (char *) pic->ptrBlue;
      if( ptr->channels & DK_ALPHA_CHANNEL )
	 col[ nbcol++ ] = (char *) pic->ptrAlpha;

      switch( ptr->type & 0x0f )
      {
         case DK_UNCOMPRESSED:
	    DK_channelsWriteRaw( file, pic->width, nbcol, col );
            break;
         case  DK_PURE_RUN_LENGTH:
	    DK_channelsWritePure( file, pic->width, nbcol, col );
            break;
         case  DK_MIXED_RUN_LENGTH:
	    DK_channelsWriteMixed( file, pic->width, nbcol, col );
            break;
      }

      if( ptr->channels &  DK_RED_CHANNEL )
         pic->ptrRed   += pic->width;
      if( ptr->channels &  DK_GREEN_CHANNEL )
         pic->ptrGreen += pic->width;
      if( ptr->channels &  DK_BLUE_CHANNEL )
         pic->ptrBlue  += pic->width;
      if( ptr->channels & DK_ALPHA_CHANNEL )
	 pic->ptrAlpha += pic->width;
   }
   return( TRUE );
}

static void DK_channelsWriteRaw(  FILE *file, short nbPix, short nbChnl, char **chnl )
{
   int   i, j;

   for( i = 0; i < nbPix; i++ )
      for( j = 0; j < nbChnl; j++ )
         putc( (unsigned char) chnl[j][i], file );
}

static void DK_channelsWritePure(  FILE *file, short nbPix, short nbChnl, char **chnl )
{
   unsigned char        col[4];
   int                  i, k, count;
   DK_Boolean           same;

   count = 0;
   col[0] = col[1] = col[2] = col[3] = 0;
   for( k = 0; k < nbPix; k++ )
   {
      if( count == 0 )
      {
         for( i = 0; i < nbChnl; i++ )
            col[i] = chnl[i][k];
         count = 1;
      }
      else
      {
         same = TRUE;
         for( i = 0; i < nbChnl; i++ )
            if( col[i] != chnl[i][k] )
            {
               same = FALSE;
               k -- ;
               break ;
            }

         if( same )
            count++;

         if( !same || (count == 255) )
         {
            putc( (unsigned char) count, file );
            for( i = 0; i < nbChnl; i++ )
               putc( (unsigned char) col[i], file );
            count = 0;
         }
      }
   }

   if( count != 0 )
   {
      putc( (unsigned char) count, file );
      for( i = 0; i < nbChnl; i++ )
         putc( (unsigned char) col[i], file );
   }
}

static void DK_channelsWriteMixed(  FILE *file, short nbPix, short nbChnl, char **chnl )
{
   short                     same, seq_same;
   register long             i, j, k;
   unsigned char             pixel[128][4], col[4];
   unsigned short            count;

   count = 0;
   col[0] = col[1] = col[2] = col[3] = 0;
   for( k = 0; k < nbPix; k++ )
   {
      for( i = 0; i < nbChnl; i++ )
         col[i] = chnl[i][k];

      if( count == 0 )
      {
         pixel[0][0] = col[0];
         pixel[0][1] = col[1];
         pixel[0][2] = col[2];
         pixel[0][3] = col[3];
         count ++;
      }
      else if( count == 1 )
      {
         seq_same = ( col[0] == pixel[0][0] );
         seq_same &= ( col[1] == pixel[0][1] );
         seq_same &= ( col[2] == pixel[0][2] );
         seq_same &= ( col[3] == pixel[0][3] );
         if( !seq_same )
         {
            pixel[count][0] = col[0];
            pixel[count][1] = col[1];
            pixel[count][2] = col[2];
            pixel[count][3] = col[3];
         }
         count ++;
      }
      else if( count > 1 )
      {
         if( seq_same )
         {
            same  = ( col[0] == pixel[0][0] );
            same &= ( col[1] == pixel[0][1] );
            same &= ( col[2] == pixel[0][2] );
            same &= ( col[3] == pixel[0][3] );
         }
         else
         {
            same  = ( col[0] == pixel[count-1][0] );
            same &= ( col[1] == pixel[count-1][1] );
            same &= ( col[2] == pixel[count-1][2] );
            same &= ( col[3] == pixel[count-1][3] );
         }

         if( same ^ seq_same )
         {
           /* sequence is break */
           if( !seq_same )
           {
             /* put (count - 1) different item */
               putc( (unsigned char) ( count - 2), file );
               for( i = 0; i < count -1; i++ )
                  for( j = 0; j < nbChnl; j++ )
                     putc( pixel[i][j], file );
               pixel[0][0] = pixel[1][0] = col[0];
               pixel[0][1] = pixel[1][1] = col[1];
               pixel[0][2] = pixel[1][2] = col[2];
               pixel[0][3] = pixel[1][3] = col[3];
               count = 2;
               seq_same = 1;
           }
           else
           {
             /* put count similar item */
               if( count < 128 )
                  putc( (unsigned char) ( count + 127 ), file );
               else
               {
                  putc( 128, file );
                  DK_byteswap_fwrite( &count, 2, 1, file );
               }
               for( j = 0; j < nbChnl; j++ )
                  putc( pixel[0][j], file );

               pixel[0][0] = col[0];
               pixel[0][1] = col[1];
               pixel[0][2] = col[2];
               pixel[0][3] = col[3];
               count = 1;
           }
         }
         else
         {
            if( !same )
            {
               pixel[count][0] = col[0];
               pixel[count][1] = col[1];
               pixel[count][2] = col[2];
               pixel[count][3] = col[3];
            }
            count ++;
            if( (count == 128) && !seq_same )
            {
             /* put (128) different item */
               putc( 127, file );
               for( i = 0; i < count; i++ )
                  for( j = 0; j < nbChnl; j++ )
                     putc( pixel[i][j], file );
               count = 0;
            }
            if( (count == 65535 ) && seq_same )
            {
               putc( 128, file );
               DK_byteswap_fwrite( &count, 2, 1, file );
               for( j = 0; j < nbChnl; j++ )
                  putc( pixel[0][j], file );
               count = 0;
            }
         }
      }
   }
   if( count )
      if( (count == 1) || (!seq_same) )
      {
       /* put (count) different item */
         putc( (unsigned char) (count - 1), file );
         for( i = 0; i < count; i++ )
            for( j = 0; j < nbChnl; j++ )
               putc( pixel[i][j], file );
      }
      else
      {
         if( count < 128 )
            putc( (unsigned char) ( count + 127 ), file );
         else
         {
            putc( 128, file );
            DK_byteswap_fwrite( &count, 2, 1, file );
         }
         for( j = 0; j < nbChnl; j++ )
            putc( pixel[0][j], file );
      }
}

static void DK_channelsReadRaw(  FILE *file, short nbPix, short nbChnl, char **chnl )
{
   int    i, j;

   for( i = 0; i < nbPix; i++ )
      for( j = 0; j < nbChnl; j++ )
         chnl[j][i] = (unsigned char) getc( file );
}

static void DK_channelsReadPure(  FILE *file, short nbPix, short nbChnl, char **chnl )
{
   unsigned char	count, col[4];
   short		i, j, k, ind;

   ind = 0;
   for( i = nbPix; i > 0; )
   {
      count = (unsigned char) getc( file );

      if( count > nbPix )
         count = nbPix;

      i -= count;

      for( j = 0; j < nbChnl; j++ )
         col[j] = (unsigned char) getc( file );

      for( j = 0; j < nbChnl; j++ )
         for( k = 0; k < count; k++ )
            chnl[j][ind + k] = col[j];

      ind += count;
   }
}

static void DK_channelsReadMixed(  FILE *file, short nbPix, short nbChnl, char **chnl )
{
  long            pixcount, j, k;
  unsigned short          count;
  char		   c;

   for(pixcount = 0; pixcount < nbPix; pixcount += count)
   {
      count = (unsigned char) getc( file );
      if(count >= 128)
      {  /* long run length */
         if(count == 128)
         {
	    DK_byteswap_fread(&count, 2, 1, file);
	    if (feof(file))
	       return;
         }
         else
	   count -= 127;
 
         /* see if we hit garbage -> set eof and return */
         if (pixcount + count > nbPix)
         {
            fseek(file, 0, SEEK_END);
	     return;
         }

         for( j = 0; j < nbChnl; j++ )
         {
	    c = (char) getc( file );
	    memset(chnl[j], c, count);
	    chnl[j] += count;
         }
      }
      else
      {   /* raw sequence */
         count++;

         /* see if we hit garbage -> set eof and return */
         if (pixcount + count > nbPix)
         { /* we hit garbage set eof and return */
           fseek(file, 0, SEEK_END);
	   return;
         }

         for(k = count; k > 0; k--)
         {
 	    for(j = 0; j < nbChnl; j++)
            {
 	       *(chnl[j]) = (char) getc( file );
	       chnl[j]++;
	    }
         }
      }
   }
}

void DK_channelPacketDispose(  DK_channelPacket **pck )
{
    DK_channelPacket	*ptr, *nxt;

   for( ptr = *pck; ptr != NULL; ptr = nxt )
   {
      nxt = ptr->next;
      free( ptr );
   }
}
