/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyNurbs.c,v $
   $Revision: 1.24.14.2 $ $Date: 1995/07/20 22:25:28 $
   Checkin by: $Author: scoy $

   This file contains a subset of the SOFTIMAGE's model functions,
   its content is sufficient to read, write and manage model data structure
   into SOFTIMAGE's file format...

   Written by: Laurent Ruhlmann, 12 mai 1994 
     (c) Copyright 1994 ,  SOFTIMAGE Inc.

    
********************************************************************************
*******************************************************************************/




#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinySceneToken.h"
#include "tinyUtils.h"
#include "tinyNurbs.h"

#define DBUG	0 


static DK_NurbsPoint *DK_nurbsPointsAllocate( int noPoints );
static double *DK_knotsAllocate( short noKnots );
static DK_Boolean nurbsCompCrvBinaryWrite( FILE *file, DK_NurbsCompCrv *cmp );
static DK_NurbsCompCrv* nurbsCompCrvBinaryRead( FILE *file , DK_FileProt *prot);
static void DK_nurbsCrvAsciiReadCVs(FILE *file, DK_NurbsCrv* crv );
static void DK_nurbsCrvAsciiReadKnots(FILE *file, DK_NurbsCrv* crv );
static void DK_nurbsSrfAsciiReadCVs(FILE *file, DK_NurbsSrf* srf);
static void DK_nurbsSrfAsciiReadKnots(FILE *file, DK_NurbsSrf* srf, DK_Boolean readinU);
static void DK_nurbsSrfReadTrims(FILE *file, DK_NurbsSrf* srf);


/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

/******************************************************************************
DK_nurbsCrvAllocate()
	Description:
	Allocates memory for a Nurbs curve. It sets the data to a default 
	value. 
	Arguments: none
	Returnewd value: a pointer ot the nurbs curve.
******************************************************************************/

DK_NurbsCrv *    DK_nurbsCrvAllocate( void )
{
   DK_NurbsCrv  *crv = NULL;

   if ( crv = (DK_NurbsCrv*) calloc( 1, sizeof( DK_NurbsCrv ) )) 
   {
      crv->dir.order    = 0;
      crv->dir.noPoints = 0;
      crv->dir.noKnots  = 0;
      crv->dir.knots    = NULL;
      crv->dir.start    = 0.0;
      crv->dir.end      = 0.0;
      crv->dir.closed   = FALSE;

      crv->uniformity = DK_NRB_UNIFORM;
      crv->rational   = FALSE;
      crv->points     = NULL;
      crv->noStep     = 10;
      crv->closeInfo  = DK_NRB_CLS_NRM;
   }

   return( crv );
}

/******************************************************************************
DK_nurbsSrfAllocate()
	Description:
	Allocates memory for a Nurbs Surface. It sets the data to a default 
	value. 
	Arguments: none
	Returnewd value: a pointer ot the nurbs surface.
******************************************************************************/

DK_NurbsSrf *    DK_nurbsSrfAllocate( void )
{

   DK_NurbsSrf  *srf = NULL;

   if ( srf = (DK_NurbsSrf*) calloc( 1, sizeof( DK_NurbsSrf ) ))
   {
      srf->uDir.order    = 0;
      srf->uDir.noPoints = 0;
      srf->uDir.noKnots  = 0;
      srf->uDir.knots    = NULL;
      srf->uDir.start    = 0.0;
      srf->uDir.end      = 0.0;
      srf->uDir.closed   = FALSE;

      srf->vDir.order    = 0;
      srf->vDir.noPoints = 0;
      srf->vDir.noKnots  = 0;
      srf->vDir.knots    = NULL;
      srf->vDir.start    = 0.0;
      srf->vDir.end      = 0.0;
      srf->vDir.closed   = FALSE;

      srf->noPoints       = 0;
      srf->uniformity     = DK_NRB_UNIFORM;
      srf->rational       = FALSE;
      srf->points         = NULL;
      srf->flag		  = 0;

      srf->autoBorder     = FALSE;
      srf->noTrims        = 0;
      srf->trims          = NULL;

      srf->uNoStep        = 3;
      srf->vNoStep        = 3;

      srf->uCurve 	  = 1;
      srf->vCurve 	  = 1;

      srf->uBottom        = FALSE;
      srf->uTop           = FALSE;
      srf->vBottom        = FALSE;
      srf->vTop           = FALSE;

      srf->uCloseInfo     = DK_NRB_CLS_NRM;		/* normal open/close */
      srf->vCloseInfo     = DK_NRB_CLS_NRM;		/* normal open/close */

      /* Mental Ray stuff */
      srf->approx_type    = DK_PARAMETRIC;
      srf->approx.viewDep = FALSE;
      srf->approx.spacial = 5.0;
      srf->approx.curv_u  = 5.0;
      srf->approx.curv_v  = 5.0;
      srf->approx.recMin  = 1;
      srf->approx.recMax  = 10;
      srf->approx.recursion = FALSE;
   }

   return( srf );

}
/******************************************************************************
DK_nurbsCrvCopy(DK_NurbsCrv)
	Description:
	Does a copy of a Nurbs Curve: memory is allocated, points are copied
	and data values are transferred from the original to the copy.
	Arguments: a pointer to a nurbs curve
	Returnewd value: a pointer ot the new nurbs curve.
******************************************************************************/

DK_NurbsCrv* DK_nurbsCrvCopy( DK_NurbsCrv *crv)
{

   DK_NurbsCrv *cpy;
   int       i;

   if( (cpy = DK_nurbsCrvAllocate()) != NULL )
   {
      cpy->dir.order    = crv->dir.order;
      cpy->dir.noPoints = crv->dir.noPoints;
      if( (cpy->dir.noKnots = crv->dir.noKnots) > 0 )
      {
         cpy->dir.knots = DK_knotsAllocate( cpy->dir.noKnots );
         for( i = 0; i < cpy->dir.noKnots; i++ )
            cpy->dir.knots[i] = crv->dir.knots[i];
      }
      cpy->dir.start    = crv->dir.start;
      cpy->dir.end      = crv->dir.end;
      cpy->dir.closed   = crv->dir.closed;

      cpy->rational      = crv->rational;
      cpy->uniformity    = crv->uniformity;
      cpy->noStep	 = crv->noStep;
      cpy->closeInfo	 = crv->closeInfo;
      if( crv->dir.noPoints > 0 )
      {
         cpy->points = DK_nurbsPointsAllocate( cpy->dir.noPoints );
         for( i = 0; i < cpy->dir.noPoints; i++ )
            cpy->points[i] = crv->points[i];
      }
   }

   return( cpy );

}

DK_NurbsCompCrv* DK_nurbsCompCrvCopy( DK_NurbsCompCrv *comp)
{
   DK_NurbsCompCrv*  cpy = NULL;
   long              j;

   cpy = (DK_NurbsCompCrv*) malloc ( sizeof( DK_NurbsCompCrv ));
   if (cpy != NULL )
   {
      if( (cpy->noCurves = comp->noCurves) > 0 )
      {
         cpy->curves = (DK_NurbsCrv**) calloc( cpy->noCurves,sizeof(DK_NurbsCrv*));

         for( j = 0; j < cpy->noCurves; j++ )
            cpy->curves[j] = DK_nurbsCrvCopy( comp->curves[j] );

      }
      cpy->noPoints = 0;
      cpy->points = NULL;
   }
   return( cpy );

}


/*****************************************************************************
DK_nurbsCrvDispose(DK_NurbsCrv**)
	Description:
	Deletes a Nurbs curve, and its points .
	Arguments: a pointer to a nurbs curve's pointer.
	Returnewd value:  none.
******************************************************************************/

void DK_nurbsCrvDispose( DK_NurbsCrv **crv )
{
   if( *crv != NULL )
   {
      if( (*crv)->dir.knots != NULL )
         _DK_FREE( (*crv)->dir.knots, -1 );

      if( (*crv)->points != NULL )
         _DK_FREE( (*crv)->points, -1 );

      _DK_FREE( *crv, -1 );
   }

}

/******************************************************************************
DK_nurbsSrfDispose(DK_NurbsSrf**)
	Description:
	Deletes a Nurbs surface, and its points .
	Arguments: a pointer to a nurbs surface's pointer.
	Returnewd value:  none.
******************************************************************************/

void          DK_nurbsSrfDispose( DK_NurbsSrf **srf )
{

   int       i;

   if( *srf != NULL )
   {
      if( (*srf)->uDir.knots != NULL )
         _DK_FREE( (*srf)->uDir.knots, -1 );

      if( (*srf)->vDir.knots != NULL )
         _DK_FREE( (*srf)->vDir.knots, -1 );

      if( (*srf)->points != NULL )
         _DK_FREE( (*srf)->points, -1 );

      if( (*srf)->trims )
      {
         for( i = 0; i < (*srf)->noTrims; i++ )
            DK_nurbsCompCrvDispose( &(*srf)->trims[i] );
         _DK_FREE( (*srf)->trims, -1 );
      }

      _DK_FREE( *srf, -1 );
   }

}


void DK_nurbsCompCrvDispose( DK_NurbsCompCrv **cmp)
{

   int       i;

   if( *cmp )
   {
      if( (*cmp)->curves )
      {
         for( i = 0; i < (*cmp)->noCurves; i++ )
            DK_nurbsCrvDispose( &(*cmp)->curves[i] );
         _DK_FREE( (*cmp)->curves, -1 );
      }

      if( (*cmp)->points )
         _DK_FREE( (*cmp)->points, -1 );

      _DK_FREE( *cmp, -1 );
   }

}
/******************************************************************************
DK_nurbsCrvBinaryRead(FILE *file , DK_FileProt *prot )
	Description:
	Reads a Nurbs Curve in binary format. Creates the curve, and gets the 
	data from the file, according to the prototypes.

	Arguments:
	FILE *file: a pointer to the file to read .
	DK_FileProt *prot: a pointer to the prototype.
	Returnewd value:  A pointer to the nurbs curve read.
******************************************************************************/

DK_NurbsCrv*   DK_nurbsCrvBinaryRead( FILE *file , DK_FileProt *prot )
{

   short     tok;
   DK_NurbsCrv  *crv = NULL;
   int       i;
   int       ind = 0;

   if( (crv = DK_nurbsCrvAllocate() ) == NULL )
      return( NULL );

   if( prot->nurbsCrv!= NULL )
      tok = prot->nurbsCrv[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );

   while( tok != 0 )
   {
      #if DBUG
         printf(" read nurbsCrv token %d\n",tok);
      #endif
      switch( tok )
      {
         case 1:        /* order */
            DK_byteswap_fread( &crv->dir.order, sizeof( long ), 1, file );
            break;

         case 2:        /* noPoints */
            DK_byteswap_fread( &crv->dir.noPoints, sizeof( long ), 1, file );
            crv->points = DK_nurbsPointsAllocate( crv->dir.noPoints );
            for( i = 0; i < crv->dir.noPoints; i++ )
	    {
               DK_byteswap_fread( &crv->points[i], sizeof( double ), 4, file );
               DK_byteswap_fread( &crv->points[i].flag, sizeof( long ), 1, file );
	       #if DBUG
	       printf("point # %d:x %f ,y %f, z %f, w %f \n",i,crv->points[i].x,crv->points[i].y,crv->points[i].z,crv->points[i].w);
               #endif
            }
	    #if DBUG
	       printf(" points read: %d\n",i);
	    #endif
            break;

         case 3:        /* closed */
            DK_byteswap_fread( &crv->dir.closed, sizeof( short), 1, file );
            break;

         case 4:        /* uniformity */
            DK_byteswap_fread( &crv->uniformity, sizeof( int), 1, file );
            DK_byteswap_fread( &crv->dir.noKnots, sizeof( long ), 1, file );
	    #if DBUG
	    (void) printf( "read - noKnots = %d\n", crv->dir.noKnots );
	    #endif
            crv->dir.knots = DK_knotsAllocate( crv->dir.noKnots );
            DK_byteswap_fread( crv->dir.knots, sizeof( double ), crv->dir.noKnots, file );
            break;
         case 5:        /* start, end */
            DK_byteswap_fread( &crv->dir.start, sizeof( double ), 1, file );
            DK_byteswap_fread( &crv->dir.end, sizeof( double ), 1, file );
            break;
         case 6:       /* display steps */
	    DK_byteswap_fread( &crv->noStep, sizeof( long ), 1, file );
	    break;

         case 7:       /* closing information */
	    DK_byteswap_fread( &crv->closeInfo, sizeof( int ), 1, file );
	    break;

         default:
            DK_fileSkipField( file, tok );

      }
      if( prot->nurbsCrv!= NULL )
         tok = prot->nurbsCrv[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );

   }

   return( crv );

}  /* DK_nurbsCrvBinaryRead */

/******************************************************************************
DK_nurbsSrfBinaryRead(FILE *file , DK_FileProt *prot )
	Description:
	Reads a Nurbs Surface in binary format. Creates the surface, and gets 
	the data from the file, according to the prototypes.

	Arguments:
	FILE *file: a pointer to the file to read .
	DK_FileProt *prot: a pointer to the prototype.
	Returnewd value:  A pointer to the nurbs surface read.
******************************************************************************/

DK_NurbsSrf*    DK_nurbsSrfBinaryRead( FILE *file , DK_FileProt *prot )
{

   short     tok;
   short      len;
   short     knotNum = 0;
   DK_NurbsSrf  *srf = NULL;
   int       i;
   int       ind = 0;


   if( (srf = DK_nurbsSrfAllocate() ) == NULL )
      return( NULL );

   if( prot->nurbsSrf!= NULL )
      tok = prot->nurbsSrf[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );

#if DBUG
   printf(" dans nurbsSRF: tok %d \n",tok);
   printf(" : tok %d \n",tok);
#endif
   while( tok != 0 )
   {
      #if DBUG
         printf(" read nurbsSRF token %d\n",tok);
      #endif

      switch( tok )
      {
         case 1:        /* order in U */
            DK_byteswap_fread( &srf->uDir.order, sizeof( long ), 1, file );
            DK_byteswap_fread( &srf->vDir.order, sizeof( long ), 1, file );
	    #if DBUG
	       printf(" uDir.order,vDir.order:%d,%d\n",srf->uDir.order,srf->vDir.order);
	       fflush(stdout);
	    #endif
            break;

         case 2:        /* points  */
            DK_byteswap_fread( &srf->uDir.noPoints, sizeof( long ), 1, file );
            DK_byteswap_fread( &srf->vDir.noPoints, sizeof( long ), 1, file );
	    #if DBUG
	       printf(" uDir.noPoints,vDir.noPoints:%d,%d\n",srf->uDir.noPoints,srf->vDir.noPoints);
	       fflush(stdout);
	    #endif
            srf->noPoints = srf->uDir.noPoints * srf->vDir.noPoints;
            srf->points = DK_nurbsPointsAllocate( srf->noPoints );
	    #if DBUG
	       printf(" srf->noPoints:%d\n",srf->noPoints);
	       fflush(stdout);
	    #endif
            for( i = 0; i < srf->noPoints; i++ )
	    {
               DK_byteswap_fread( &srf->points[i], sizeof( double ), 4, file );
               DK_byteswap_fread( &srf->points[i].flag, sizeof( long ), 1, file );
            }
            break;

         case 3:        /* closed */
            DK_byteswap_fread( &srf->uDir.closed, sizeof( short), 1, file );
            DK_byteswap_fread( &srf->vDir.closed, sizeof( short), 1, file );
            break;

         case 4:        /* uniformity */
            DK_byteswap_fread( &srf->uniformity, sizeof( int), 1, file );
            DK_byteswap_fread( &srf->uDir.noKnots, sizeof( long), 1, file );
            srf->uDir.knots = DK_knotsAllocate( srf->uDir.noKnots );

            DK_byteswap_fread( srf->uDir.knots, sizeof( double ),
                                                 srf->uDir.noKnots, file );

            DK_byteswap_fread( &srf->vDir.noKnots, sizeof( long), 1, file );
            srf->vDir.knots = DK_knotsAllocate( srf->vDir.noKnots );
            DK_byteswap_fread( srf->vDir.knots, sizeof( double ),
                                                 srf->vDir.noKnots, file );
            break;

         case 5:        /* start & end */
            DK_byteswap_fread( &srf->uDir.start, sizeof( double ), 1, file );
            DK_byteswap_fread( &srf->uDir.end, sizeof( double ), 1, file );
            DK_byteswap_fread( &srf->vDir.start, sizeof( double ), 1, file );
            DK_byteswap_fread( &srf->vDir.end, sizeof( double ), 1, file );
            break;

         case 6:        /* autoBorder */
            DK_byteswap_fread( &srf->autoBorder, sizeof( short ), 1, file );
            break;

         case 7:        /* trims */
            DK_byteswap_fread( &srf->noTrims, sizeof( int), 1, file );
            if( srf->noTrims > 0 )
            {
               srf->trims = (DK_NurbsCompCrv**) calloc(srf->noTrims, sizeof(DK_NurbsCompCrv*));
               for( i = 0; i < srf->noTrims; i++ )
                   srf->trims[i] = nurbsCompCrvBinaryRead( file ,prot );
            }
            break;

         case 8:        /* u display wireframe, SHADES & rendering step */
            DK_byteswap_fread( &srf->uNoStep, sizeof( long ), 1, file );
            break;
         case 9:        /* v display wireframe, SHADES & rendering step */
            DK_byteswap_fread( &srf->vNoStep, sizeof( long ), 1, file );
            break;
         case 10:        /* u drawing step */
            DK_byteswap_fread( &srf->uCurve, sizeof( long ), 1, file );
            break;
         case 11:        /* v drawing step */
            DK_byteswap_fread( &srf->vCurve, sizeof( long ), 1, file );
            break;

         case 12: /* capping U bottom */
            DK_byteswap_fread( &srf->uBottom, sizeof( short ), 1, file );
            break;

         case 13: /* capping U top */
            DK_byteswap_fread( &srf->uTop, sizeof( short ), 1, file );
            break;

         case 14: /* capping V bottom */
            DK_byteswap_fread( &srf->vBottom, sizeof( short ), 1, file );
            break;

         case 15: /* capping V top */
            DK_byteswap_fread( &srf->vTop, sizeof( short ), 1, file );
            break;

         case 16: /* U closing information */
            DK_byteswap_fread( &srf->uCloseInfo, sizeof( int ), 1, file );
            break;
	 
         case 17: /* V closing information */
            DK_byteswap_fread( &srf->vCloseInfo, sizeof( int ), 1, file );
            break;
	 
	 case 18: /* type approx */
	    DK_byteswap_fread( &srf->approx_type, sizeof(unsigned short), 1, file);
	    break;
	 
	 case 19: /* view Dep */
	    DK_byteswap_fread( &len, sizeof(short), 1, file);
	    if( len )
	       srf->approx.viewDep = TRUE;
	    else
	       srf->approx.viewDep = FALSE;
	    break;
	 
	 case 20: /* spacial */
	    DK_byteswap_fread( &srf->approx.spacial, sizeof(float), 1, file);
	    break;
	 
	 case 21: /* curv_u */
	    DK_byteswap_fread( &srf->approx.curv_u, sizeof(float), 1, file);
	    break;
	 
	 case 22: /* curv_v */
	    DK_byteswap_fread( &srf->approx.curv_v, sizeof(float), 1, file);
	    break;
	 
	 case 23: /* recMin */
	    DK_byteswap_fread( &srf->approx.recMin, sizeof(unsigned short), 1, file);
	    break;
	 
	 case 24: /* recMax */
	    DK_byteswap_fread( &srf->approx.recMax, sizeof(unsigned short), 1, file);
	    break;
	 
	 case 25: /* recursion */
	    DK_byteswap_fread( &len, sizeof(short), 1, file);
	    if( len )
	       srf->approx.recursion = TRUE;
	    else
	       srf->approx.recursion = FALSE;
	    break;

         default:
            DK_fileSkipField( file, tok );

      }
      if( prot->nurbsSrf!= NULL )
         tok = prot->nurbsSrf[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   return( srf );

}  /* DK_nurbsSrfBinaryRead */

/******************************************************************************
DK_nurbsCrvBinaryWrite(FILE *file ,  DK_NurbsCrv *crv )
	Description:
	Writes a Nurbs curve in binary format into a file.

	Arguments:
	FILE *file: a pointer to the file to read .
  	 DK_NurbsCrv *crv: a pointer to the curve to write.	

	Returnewd value: True if successfull.

******************************************************************************/

DK_Boolean DK_nurbsCrvBinaryWrite ( FILE *file , DK_NurbsCrv *crv )
{
   int           pos;
   short         tok;
   int           i;
   DK_Boolean    ok = TRUE;
   DK_NurbsPoint pt;
   double*       knots = NULL;
   long 	 knotNum = 0;

  
  /* 1 order */
   ok &= ( DK_byteswap_fwrite( &crv->dir.order, sizeof( long ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Crv token 1\n");
   #endif


  /* 2 points */
   ok &= ( DK_byteswap_fwrite( &crv->dir.noPoints, sizeof( long ), 1, file ) == 1 );
   for( i = 0; i < crv->dir.noPoints; i++ )
   {
      pt.x = crv->points[i].x;
      pt.y = crv->points[i].y;
      pt.z = crv->points[i].z;
      pt.w = crv->points[i].w;

      ok &= ( DK_byteswap_fwrite( &pt, sizeof( double ), 4, file ) == 4 );
      ok &= ( DK_byteswap_fwrite( &crv->points[i].flag, sizeof( long ), 1, file ) == 1 );
   }
   #if DBUG
   printf(" write Nurbs Crv token 2, %d points\n", i);
   #endif

  /* 3 closed */
   ok &= ( DK_byteswap_fwrite( &crv->dir.closed, sizeof( short ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Crv token 3\n");
  (void) printf( "written - noKnots = %d\n", crv->dir.noKnots );
   #endif
  /* 4 uniformity */
   ok &= ( DK_byteswap_fwrite( &crv->uniformity, sizeof( int ), 1, file ) == 1 );

   /* attention aux fichiers IGES */
   if ( crv->dir.noKnots == crv->dir.noPoints + ((crv->dir.closed) ? 1: (crv->dir.order - 2)))
   {
      ok &= ( DK_byteswap_fwrite( &crv->dir.noKnots, sizeof( long ), 1, file ) == 1 );
      ok &= ( DK_byteswap_fwrite( crv->dir.knots, sizeof( double ), crv->dir.noKnots, file ) == crv->dir.noKnots );
   }
   else
   {
      knotNum = crv->dir.noKnots-2;
      ok &= ( DK_byteswap_fwrite( &knotNum, sizeof( long ), 1, file ) == 1 );
      knots = DK_knotsAllocate( crv->dir.noKnots-2 );
      /* transfer the knots */
      for ( i= 1; i<crv->dir.noKnots-1; i++)
	 knots[i-1] = crv->dir.knots[i];
      ok &= ( DK_byteswap_fwrite( knots, sizeof( double ), crv->dir.noKnots-2, file ) == crv->dir.noKnots-2 );
      #if DBUG
      printf("number of knots written: %d\n", knotNum);
      #endif
   }
   #if DBUG
   printf(" write Nurbs Crv token 4\n");
   #endif
  /* 5 start,end  */
   ok &= ( DK_byteswap_fwrite( &crv->dir.start, sizeof( double ), 1, file ) == 1 );
   ok &= ( DK_byteswap_fwrite( &crv->dir.end, sizeof( double ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Crv token 5\n");
   #endif

   /* 6 display steps */
   ok &= ( DK_byteswap_fwrite( &crv->noStep , sizeof( long ), 1, file ) == 1 );

   /* 7 closing information */
   ok &= ( DK_byteswap_fwrite( &crv->closeInfo , sizeof( int ), 1, file ) == 1 );

   #if DBUG
   printf(" write Nurbs Crv token 6\n");
   #endif

  /* end of write */

   return( ok );

}  /* DK_nurbsCrvBinaryWrite */


/******************************************************************************
DK_nurbsSrfBinaryWrite(FILE *file ,  DK_NurbsSrf *srf)
	Description:
	Writes a Nurbs surface in binary format into a file.

	Arguments:
	FILE *file: a pointer to the file to read .
  	 DK_NurbsSrf *srf: a pointer to the surface to write.	

	Returnewd value: True if successfull.

******************************************************************************/

DK_Boolean DK_nurbsSrfBinaryWrite ( FILE *file , DK_NurbsSrf *srf )
{

   int           pos, pos1;
   short         tok;
   short          len;
   int           i;
   DK_Boolean    ok = TRUE;
   DK_NurbsPoint pt;
   double*       uKnots = NULL;
   double*       vKnots = NULL;
   long          uKnotNum = 0;
   long          vKnotNum = 0;

   /* 1 orders (UV) */
   ok &= ( DK_byteswap_fwrite( &srf->uDir.order, sizeof( long ), 1, file ) == 1 );
   ok &= ( DK_byteswap_fwrite( &srf->vDir.order, sizeof( long ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Srf token 1\n");
   #endif

   /* 2 points (uNoPoints, vNoPoints, points...) */
   ok &= ( DK_byteswap_fwrite( &srf->uDir.noPoints, sizeof( long ), 1, file ) == 1 );
   ok &= ( DK_byteswap_fwrite( &srf->vDir.noPoints, sizeof( long ), 1, file ) == 1 );
   for( i = 0; i < srf->noPoints; i++ )
   {
      pt.x = srf->points[i].x;
      pt.y = srf->points[i].y;
      pt.z = srf->points[i].z;
      pt.w = srf->points[i].w;

      ok &= ( DK_byteswap_fwrite( &pt, sizeof( double ), 4, file ) == 4 );
      ok &= ( DK_byteswap_fwrite( &srf->points[i].flag, sizeof( long ), 1, file ) == 1 );
   }
   #if DBUG
   printf(" write Nurbs Srf token 2: %d points\n",srf->noPoints);
   #endif

  /* 3 closed */
   ok &= ( DK_byteswap_fwrite( &srf->uDir.closed, sizeof( short), 1, file ) == 1 );
   ok &= ( DK_byteswap_fwrite( &srf->vDir.closed, sizeof( short), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Srf token 3\n");
   #endif

  /* 4 uniformity */
   ok &= ( DK_byteswap_fwrite( &srf->uniformity, sizeof( int), 1, file ) == 1 );
   if (srf->uDir.noKnots == srf->uDir.noPoints +
                            ((srf->uDir.closed) ? 1: (srf->uDir.order - 2)))
   {
      ok &= ( DK_byteswap_fwrite( &srf->uDir.noKnots, sizeof( long), 1, file ) == 1 );
      ok &= ( DK_byteswap_fwrite( srf->uDir.knots, sizeof( double ),
                            srf->uDir.noKnots, file ) == srf->uDir.noKnots );
   }
   else
   {
      uKnotNum = srf->uDir.noKnots-2;
      ok &= ( DK_byteswap_fwrite( &uKnotNum, sizeof( long ), 1, file ) == 1 );
      uKnots = DK_knotsAllocate( srf->uDir.noKnots-2 );
      for ( i= 1; i < srf->uDir.noKnots-1; i++)
         uKnots[i-1] = srf->uDir.knots[i];
      ok &= ( DK_byteswap_fwrite( uKnots, sizeof( double ), uKnotNum, file ) == uKnotNum);
      #if DBUG
      printf("number of knots written in u : %d\n", uKnotNum);
      #endif

   }


   if (srf->vDir.noKnots == srf->vDir.noPoints +
                            ((srf->vDir.closed) ? 1: (srf->vDir.order - 2)))
   {
      ok &= ( DK_byteswap_fwrite( &srf->vDir.noKnots, sizeof( long), 1, file ) == 1 );
      ok &= ( DK_byteswap_fwrite( srf->vDir.knots, sizeof( double ),
                            srf->vDir.noKnots, file ) == srf->vDir.noKnots );
   }
   else
   {
      vKnotNum = srf->vDir.noKnots-2;
      ok &= ( DK_byteswap_fwrite( &vKnotNum, sizeof( long ), 1, file ) == 1 );
      vKnots = DK_knotsAllocate( srf->vDir.noKnots-2 );
      for ( i= 1; i < srf->vDir.noKnots-1; i++)
         vKnots[i-1] = srf->vDir.knots[i];
      ok &= ( DK_byteswap_fwrite( vKnots, sizeof( double ), vKnotNum, file ) == vKnotNum);
      #if DBUG
      printf("number of knots written in v : %d\n", vKnotNum);
      #endif

   }

   #if DBUG
   printf(" write Nurbs Srf token 4\n");
   #endif

  /* 5 start & end */
   ok &= ( DK_byteswap_fwrite( &srf->uDir.start, sizeof( double ), 1, file ) == 1 );
   ok &= ( DK_byteswap_fwrite( &srf->uDir.end,   sizeof( double ), 1, file ) == 1 );
   ok &= ( DK_byteswap_fwrite( &srf->vDir.start, sizeof( double ), 1, file ) == 1 );
   ok &= ( DK_byteswap_fwrite( &srf->vDir.end,   sizeof( double ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Srf token 5\n");
   #endif

 
  /* 6 autoBorder */
   ok &= ( DK_byteswap_fwrite( &srf->autoBorder, sizeof( short ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Srf token 6\n");
   #endif

  /* 7 trims */
   ok &= ( DK_byteswap_fwrite( &srf->noTrims, sizeof( int ), 1, file ) == 1 );
   for( i = 0; i < srf->noTrims; i++ )
      nurbsCompCrvBinaryWrite( file, srf->trims[i] );
   #if DBUG
   printf(" write Nurbs Srf token 7\n");
   #endif

  /* 8 u display wireframe, SHADES & rendering step */
   ok &= ( DK_byteswap_fwrite( &srf->uNoStep, sizeof( long ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Srf token 8\n");
   #endif

  /* 9  v display wireframe, SHADES & rendering step */
   ok &= ( DK_byteswap_fwrite( &srf->vNoStep, sizeof( long ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Srf token 9\n");
   #endif

  /* 10 u drawing step */
   ok &= ( DK_byteswap_fwrite( &srf->uCurve, sizeof( long ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Srf token 10\n");
   #endif

  /* 11 v drawing step */
   ok &= ( DK_byteswap_fwrite( &srf->vCurve, sizeof( long ), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Srf token 11\n");
   #endif

   /* 12: capping U bottom */
   DK_byteswap_fwrite( &srf->uBottom, sizeof( short ), 1, file );

   /* 13: capping U top */
   DK_byteswap_fwrite( &srf->uTop, sizeof( short ), 1, file );

   /* 14: capping V bottom */
   DK_byteswap_fwrite( &srf->vBottom, sizeof( short ), 1, file );

   /* 15: capping V top */
   DK_byteswap_fwrite( &srf->vTop, sizeof( short ), 1, file );

   /* 16: U closing information */
   DK_byteswap_fwrite( &srf->uCloseInfo, sizeof( int ), 1, file );

   /* 17: V closing information */
   DK_byteswap_fwrite( &srf->vCloseInfo, sizeof( int ), 1, file );

   /* 18: approx_type */
   DK_byteswap_fwrite( &srf->approx_type, sizeof(unsigned short), 1, file);

   /* 19: viewDep */
   len = (short )srf->approx.viewDep;
   DK_byteswap_fwrite( &len, sizeof(short), 1, file);

   /* 20: spacial */
   DK_byteswap_fwrite( &srf->approx.spacial, sizeof(float), 1, file);

   /* 21: curv_u */
   DK_byteswap_fwrite( &srf->approx.curv_u, sizeof(float), 1, file);

   /* 22: curv_v */
   DK_byteswap_fwrite( &srf->approx.curv_v, sizeof(float), 1, file);

   /* 23: recMin */
   DK_byteswap_fwrite( &srf->approx.recMin, sizeof(unsigned short), 1, file);

   /* 24: recMax */
   DK_byteswap_fwrite( &srf->approx.recMax, sizeof(unsigned short), 1, file);

   /* 25: recursion */
   len = (short )srf->approx.recursion;
   DK_byteswap_fwrite( &len, sizeof(short), 1, file);


  /* end */

   return( ok );


}  /* DK_nurbsSrfBinaryWrite */



/******************************************************************************
DK_nurbsCrvAsciiWrite(FILE *file ,  DK_NurbsCrv *crv, DK_String space )
	Description:
	Writes a Nurbs curve in ascii format into a file.

	Arguments:
	FILE *file: a pointer to the file to read .
  	DK_NurbsCrv *crv: a pointer to the curve to write.	
        DK_String space: some spacing character for alignment purposes.

	Returnewd value: True if successfull.

******************************************************************************/

DK_Boolean DK_nurbsCrvAsciiWrite( FILE * file, DK_NurbsCrv* crv, DK_String space)
{

   int               i, j;
   DK_NurbsPoint        *pnt;


   fprintf( file, "\n%sNurbs_Curve\n", space );
   fprintf( file, "%s{\n", space );

   if (crv->dir.closed) 
      fprintf( file, "%s   closed\n", space );
   else
      fprintf( file, "%s   not_closed\n", space );

   if (crv->rational) 
      fprintf( file, "%s   rational\n", space );
   else
      fprintf( file, "%s   not_rational\n", space );
   fprintf( file, "%s   uniformity %d\n", space, crv->uniformity );
   fprintf( file, "%s   noStep     %d\n", space, crv->noStep);
   fprintf( file, "%s   closeInfo  %d\n", space, crv->closeInfo);

   /* Dir */
   fprintf( file, "%s   order      %d\n", space, crv->dir.order );
   fprintf( file, "%s   noPoints   %d\n", space, crv->dir.noPoints );
   fprintf( file, "%s   noKnots    %d\n", space, crv->dir.noKnots );

   /* knots */
   fprintf( file, "%s   start      %f\n", space, crv->dir.start );
   fprintf( file, "%s   end        %f\n", space, crv->dir.end );
   fprintf( file, "\n%s   knots\n",space);
   fprintf( file, "%s   {\n",space);
   for( i = 0; i < crv->dir.noKnots; i++ )
   {
      fprintf( file, "%s        [%1d] value %f\n",space, i, crv->dir.knots[i] );
   }
   fprintf( file, "%s   }\n", space );




  /* Points */
   fprintf( file, "\n%s  Vertices\n", space );
   fprintf( file, "%s  {\n", space );

   for( i = 0, pnt = crv->points; i < crv->dir.noPoints; i++, pnt++ )
       fprintf( file, "%s    [%1d]  position   %*.*f  %*.*f  %*.*f %*.*f\n",
                space, i,  DK_FM,  DK_FD, pnt->x,
                           DK_FM,  DK_FD, pnt->y,
                           DK_FM,  DK_FD, pnt->z,
                           DK_FM,  DK_FD, pnt->w );

   fprintf( file, "%s  }\n", space );

   fprintf( file, "%s}\n", space );


   return( TRUE );
}   /* DK_nurbsCrvAsciiWrite */


/******************************************************************************
DK_nurbsCrvAsciiRead(FILE *file )  
	Description:
	Reads a Nurbs curve in ascii format from a file.

	Arguments:
	FILE *file: a pointer to the file to read from.

	Returnewd value: a pointer to the read Nurbs curve.
******************************************************************************/


DK_NurbsCrv* DK_nurbsCrvAsciiRead( FILE * file )
{

   DK_NurbsCrv*  crv =NULL;
   char          str[ 80 ],
                 type[ 80 ];

   if ( (crv = DK_nurbsCrvAllocate()) == NULL )
      return ( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( crv);


   do
   {
      fscanf( file, "%s", str );

      if( strcmp( "order", str ) == 0 )
         fscanf(file, "%d", &crv->dir.order);
      else if( strcmp( "noPoints", str ) == 0 )
         fscanf(file,"%d", &crv->dir.noPoints);
      else if( strcmp( "noKnots", str ) == 0 )
         fscanf(file, "%d", &crv->dir.noKnots);
      else if( strcmp( "start", str ) == 0 )
         fscanf(file,"%f", &crv->dir.start);
      else if( strcmp( "end", str ) == 0 )
      {
         fscanf(file,"%s", str);
	 crv->dir.end = atof( str );
      }
      else if( strcmp( "closed", str ) == 0 )
         crv->dir.closed = TRUE;
      else if( strcmp( "not_closed", str ) == 0 )
         crv->dir.closed = FALSE;
      else if( strcmp( "uniformity", str ) == 0 )
         fscanf(file,"%d", &crv->uniformity);
      else if( strcmp( "noStep", str ) == 0 )
         fscanf(file,"%d", &crv->noStep);
      else if( strcmp( "closeInfo", str ) == 0 )
         fscanf(file,"%d", &crv->closeInfo);
      else if( strcmp( "rational", str ) == 0 )
         crv->rational = TRUE;
      else if( strcmp( "not_rational", str ) == 0 )
         crv->rational = FALSE;
      else if ( strcmp( "Vertices",str) == 0)
         DK_nurbsCrvAsciiReadCVs(file,crv);
      else if ( strcmp( "knots",str) == 0)
         DK_nurbsCrvAsciiReadKnots(file,crv);
   } while( strcmp( "}", str ) != 0 );

   return (crv);

}   /* DK_nurbsCrvAsciiRead */

/******************************************************************************
DK_nurbsSrfAsciiWrite(FILE *file ,  DK_NurbsSrf *srf, DK_String space )
	Description:
	Writes a Nurbs surface in ascii format into a file.

	Arguments:
	FILE *file: a pointer to the file to write.
  	DK_NurbsCrv *crv: a pointer to the curve to write.	
        DK_String space: some spacing character for alignment purposes.

	Returnewd value: True if successfull.

******************************************************************************/

DK_Boolean DK_nurbsSrfAsciiWrite( FILE * file, DK_NurbsSrf* srf, DK_String space)
{
   DK_NurbsPoint     *pnt;
   DK_Boolean	     ok = TRUE;
   int               i, j;
   char              sspace[80];

   sprintf( sspace, "%s  ", space );

   fprintf( file, "\n%sNurbs_Surface\n", space );
   fprintf( file, "%s{\n", space );

   if (srf->rational) 
      fprintf( file, "%s  rational\n", space );
   else 
      fprintf( file, "%s  not_rational\n", space );

   fprintf( file, "%s  uniformity   %d\n", space, srf->uniformity );

  /* UDir */
   fprintf( file, "\n%s  U_Dir\n" ,space );
   fprintf( file, "%s  {\n", space );
   fprintf( file, "%s    noPoints   %d\n", space, srf->uDir.noPoints );
   fprintf( file, "%s    order      %d\n", space, srf->uDir.order );
   fprintf( file, "%s    NoStep     %d\n", space, srf->uNoStep);
   fprintf( file, "%s    Curve      %d\n", space, srf->uCurve);
   if( srf->uBottom )
      fprintf( file, "%s    Bottom\n", space );
   if( srf->uTop )
      fprintf( file, "%s    Top\n", space );
   fprintf( file, "%s    CloseInfo  %d\n", space, srf->uCloseInfo);
   fprintf( file, "%s    start      %g\n", space, srf->uDir.start );
   fprintf( file, "%s    end        %g\n", space, srf->uDir.end );
   if ( srf->uDir.closed) 
      fprintf( file, "%s    closed\n", space );
   else 
      fprintf( file, "%s    not_closed\n", space );
   fprintf( file, "%s    noKnots    %d\n\n", space, srf->uDir.noKnots );
   for( i = 0; i < srf->uDir.noKnots; i++ )
   {
      fprintf( file, "%s       [%1d] value %f\n",space, i, srf->uDir.knots[i] );
   }
   fprintf( file, "%s  }\n", space );

  /* VDir */
   fprintf( file, "\n%s  V_Dir\n" , space );
   fprintf( file, "%s  {\n", space );
   fprintf( file, "%s    noPoints   %d\n", space, srf->vDir.noPoints );
   fprintf( file, "%s    order      %d\n", space, srf->vDir.order );
   fprintf( file, "%s    NoStep     %d\n", space, srf->vNoStep);
   fprintf( file, "%s    Curve      %d\n", space, srf->vCurve);
   if( srf->vBottom )
      fprintf( file, "%s    Bottom\n", space );
   if( srf->vTop )
      fprintf( file, "%s    Top\n", space );
   fprintf( file, "%s    CloseInfo  %d\n", space, srf->vCloseInfo);
   fprintf( file, "%s    start      %g\n", space, srf->vDir.start );
   fprintf( file, "%s    end        %g\n", space, srf->vDir.end );
   if ( srf->vDir.closed) 
      fprintf( file, "%s    closed\n", space );
   else 
      fprintf( file, "%s    not_closed\n", space );
   fprintf( file, "%s    noKnots    %d\n\n", space, srf->vDir.noKnots );
   for( i = 0; i < srf->vDir.noKnots; i++ )
   {
      fprintf( file, "%s       [%1d] value %f\n",space, i, srf->vDir.knots[i] );
   }
   fprintf( file, "%s  }\n", space );

   fprintf( file, "%s  approx_type %hd\n", space, srf->approx_type);
   fprintf( file, "%s  viewDep	%d\n", space, srf->approx.viewDep);
   fprintf( file, "%s  spacial	%*.*f\n", space, DK_FM, DK_FD, srf->approx.spacial);
   fprintf( file, "%s  curv_u	%*.*f\n", space, DK_FM, DK_FD, srf->approx.curv_u);
   fprintf( file, "%s  curv_v	%*.*f\n", space, DK_FM, DK_FD, srf->approx.curv_v);
   fprintf( file, "%s  recMin	%hd\n", space, srf->approx.recMin);
   fprintf( file, "%s  recMax	%hd\n", space, srf->approx.recMax);
   fprintf( file, "%s  recursion	%d\n", space, srf->approx.recursion);

  /* Points */
   fprintf( file, "\n%s  noPoints %d\n", space, srf->noPoints );
   fprintf( file, "\n%s  Vertices\n", space );
   fprintf( file, "%s  {\n", space );

   for( i = 0, pnt = srf->points; i < srf->noPoints; i++, pnt++ )
       fprintf( file, "%s    [%1d]  position   %*.*f  %*.*f  %*.*f %*.*f\n",
                space, i,  DK_FM,  DK_FD, pnt->x,
                           DK_FM,  DK_FD, pnt->y,
                           DK_FM,  DK_FD, pnt->z,
                           DK_FM,  DK_FD, pnt->w );

   fprintf( file, "%s  }\n", space );

   /* composite curves */
   fprintf( file, "\n%s  Number_of_trims: %d\n", space, srf->noTrims);
   for ( i=0; i< srf->noTrims; i++)
   {
       fprintf( file, "Trim_Curve # %d\n",i);
       if ( srf->trims[i]->boundary ) 
	  fprintf( file,"Curve_is_a_Boundary\n");
       for ( j=0; j<srf->trims[i]->noCurves;j++)
       {
	  fprintf( file, "Curve_number: %d\n",j);
	  ok = DK_nurbsCrvAsciiWrite(file, srf->trims[i]->curves[j],space);
       }
   }
   fprintf( file, "%s}\n", space );
   return( TRUE );


}  /* DK_nurbsSrfAsciiWrite */

/******************************************************************************
DK_nurbsSrfAsciiRead(FILE *file )  
	Description:
	Reads a Nurbs surface in ascii format from a file.

	Arguments:
	FILE *file: a pointer to the file to read from.

	Returnewd value: a pointer to the read Nurbs surface.
******************************************************************************/

DK_NurbsSrf* DK_nurbsSrfAsciiRead( FILE * file )
{
   DK_NurbsSrf  *srf = NULL;
   int		len;
   char          str[ 80 ],
                 str1[ 80 ],
                 type[ 80 ];

   if ( (srf = DK_nurbsSrfAllocate()) == NULL )
      return ( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( srf);
   do
   {
      fscanf( file, "%s", str );

      if( strcmp( "U_Dir", str ) == 0 )
          DK_nurbsSrfAsciiReadKnots(file,srf,TRUE);	 
      if( strcmp( "V_Dir", str ) == 0 )
          DK_nurbsSrfAsciiReadKnots(file,srf,FALSE);	 
      else if( strcmp( "noPoints", str ) == 0 )
         fscanf(file,"%d", &srf->noPoints);
      else if( strcmp( "uniformity", str ) == 0 )
         fscanf(file,"%d", &srf->uniformity);
      else if( strcmp( "rational", str ) == 0 )
         srf->rational = TRUE;
      else if( strcmp( "not_rational", str ) == 0 )
         srf->rational = FALSE;
      else if ( strcmp( "Vertices",str) == 0)
         DK_nurbsSrfAsciiReadCVs(file,srf);
      else if( strcmp( "approx_type", str ) == 0 )
         fscanf( file, "%hd", &srf->approx_type);
      else if( strcmp( "viewDep", str ) == 0 )
      {
         fscanf( file, "%d", &len);
	 if( len )
	    srf->approx.viewDep = TRUE;
	 else
	    srf->approx.viewDep = FALSE;
      }
      else if( strcmp( "spacial", str ) == 0 )
         fscanf( file, "%f", &srf->approx.spacial);
      else if( strcmp( "curv_u", str ) == 0 )
         fscanf( file, "%f", &srf->approx.curv_u);
      else if( strcmp( "curv_v", str ) == 0 )
         fscanf( file, "%f", &srf->approx.curv_v);
      else if( strcmp( "recMin", str ) == 0 )
         fscanf( file, "%hd", &srf->approx.recMin);
      else if( strcmp( "recMax", str ) == 0 )
         fscanf( file, "%hd", &srf->approx.recMax);
      else if( strcmp( "recursion", str ) == 0 )
      {
         fscanf( file, "%d", &len);
	 if( len )
	    srf->approx.recursion = TRUE;
	 else
	    srf->approx.recursion = FALSE;
      }
      else if ( strcmp( "Number_of_trims:",str) == 0)
      {
	 fscanf( file, "%d", &srf->noTrims);
	 #if DBUG
	 printf("# of trimmed curves : %d\n",srf->noTrims);
	 #endif
	 if ( srf->noTrims > 0 )
	    DK_nurbsSrfReadTrims(file,srf);
      }
   } while( strcmp( "}", str ) != 0 );


   return (srf);

} /* DK_nurbsSrfAsciiRead */

/****************************************************************/
/*                 static functions                             */
/****************************************************************/

static DK_Boolean nurbsCompCrvBinaryWrite( FILE *file, DK_NurbsCompCrv *cmp )
{


   int       pos, pos1;
   short     tok;
   int       i;
   DK_Boolean   ok = TRUE;

  /* 1 Boundary */
   ok &= ( DK_byteswap_fwrite( &cmp->boundary, sizeof( short), 1, file ) == 1 );
   #if DBUG
   printf(" write Nurbs Comp Crv token 1 : ");
   printf("boundary: %d\n",cmp->boundary);
   #endif

  /* 2 Number of Curves */
   ok &= ( DK_byteswap_fwrite( &cmp->noCurves, sizeof( int), 1, file ) == 1 );

   for( i = 0; i < cmp->noCurves; i++ )
      DK_nurbsCrvBinaryWrite( file, cmp->curves[i] );
   #if DBUG
   printf(" write Nurbs Comp Crv token 2\n");
   #endif

  /* end */

   return( ok );

}   /* DK_nurbsCompCrvBinaryWrite */

static DK_NurbsPoint *DK_nurbsPointsAllocate( int noPoints )
{
   DK_NurbsPoint        *points, defPoint;
   int               i;

   points = (DK_NurbsPoint*) malloc( noPoints*sizeof(DK_NurbsPoint));

   if ( points != NULL) 
   {
      defPoint.x = defPoint.y = defPoint.z = 0.0; defPoint.w = 1.0;
      defPoint.flag = 0;
      for( i = 0; i < noPoints; i++ )
         points[i] = defPoint;
   }
#if DBUG
printf(" %d points allocated \n",noPoints);
#endif

   return( points );
}


static double *DK_knotsAllocate( short noKnots )
{
   double    *knots = NULL;

   knots = (double*) calloc ( noKnots, sizeof(double) );
   return( knots );
}

static void DK_nurbsCrvAsciiReadCVs(FILE *file, DK_NurbsCrv* crv )
{
   char          str[ 80 ];
   int           i = 0;
   float         x,y,z,w;

   if (( crv->points = DK_nurbsPointsAllocate( crv->dir.noPoints)) ==
NULL )
      return;

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return;
   do
   {
      fscanf( file, "%s", str );

      if( str[0] == '[' )
      {
         i++;
      }
      else if( strcmp( "position", str ) == 0 )
      {
          fscanf( file, "%f %f %f %f", &x,&y,&z,&w);
          crv->points[i-1].x = x;
          crv->points[i-1].y = y;
          crv->points[i-1].z = z;
          crv->points[i-1].w = w;
          #if DBUG
          printf("point read <%f, %f, %f, %f>\n",x,y,z,w);
          #endif
      }
   }  while( strcmp( "}", str ) != 0 );

}


static void DK_nurbsCrvAsciiReadKnots(FILE *file, DK_NurbsCrv* crv )
{
   char          str[ 80 ];
   int           i = 0;
   float         knot ;

  crv->dir.knots = DK_knotsAllocate(crv->dir.noKnots);
  if ( crv->dir.knots == NULL )
     return;

  fscanf( file, "%s", str );
  if( strcmp( str, "{" ) != 0 )
    return;

   do
   {
      fscanf( file, "%s", str );

      if( str[0] == '[' )
      {
         i++;
      }
      else if( strcmp( "value", str ) == 0 )
      {
          fscanf( file, "%f", &knot );
          crv->dir.knots[i-1] = knot;
          #if DBUG
          printf("knot #%d read <%f>\n",i,knot);
          #endif

      }

   }  while( strcmp( "}", str ) != 0 );

   #if DBUG
   printf("Curve data: \n");
   printf(" noPoints: %d, order: %d, start: %f, end: %f, closed:%d, noKnots: %d\n",crv->dir.noPoints,crv->dir.order, crv->dir.start,crv->dir.end,crv->dir.closed, crv->dir.noKnots);
   #endif

}  /* DK_nurbsCrvAsciiReadKnots */


static DK_NurbsCompCrv* nurbsCompCrvBinaryRead( FILE *file , DK_FileProt
*prot)
{

   short     tok;
   DK_NurbsCompCrv *cmp = NULL;
   int       i;
   int       ind = 0;

   if ( (cmp = (DK_NurbsCompCrv*) malloc(  sizeof( DK_NurbsCompCrv ) )) == NULL )
      return( NULL );

   cmp->boundary = 0;
   cmp->noCurves = 0;
   cmp->curves = NULL;
   cmp->noPoints = 0;
   cmp->points = NULL;

   if( prot->nurbsCompCrv!= NULL )
      tok = prot->nurbsCompCrv[ ind ++ ];
    else
      DK_byteswap_fread( &tok, 2, 1, file );

   while( tok != 0 )
   {
#if DBUG
   printf(" dans nurbs comp curve: tok %d \n",tok);
#endif

      switch( tok )
      {
        case 1:         /* boundary */
            DK_byteswap_fread( &cmp->boundary, sizeof( short), 1, file );
	    break;
         case 2:        /* curves */
            DK_byteswap_fread( &cmp->noCurves, sizeof( int), 1, file );
            cmp->curves = (DK_NurbsCrv**) calloc( cmp->noCurves, sizeof(DK_NurbsCrv*));
            for( i = 0; i < cmp->noCurves; i++ )
	    {
               #if DBUG
                  printf(" processing composite curve #%d\n",i);
               #endif
               cmp->curves[i] = DK_nurbsCrvBinaryRead( file, prot  );
            }
            break;


         default:
            DK_fileSkipField( file, tok );

      }
      if( prot->nurbsCompCrv != NULL )
         tok = prot->nurbsCompCrv[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );

   }

   return( cmp );

}



static void DK_nurbsSrfAsciiReadCVs(FILE *file, DK_NurbsSrf* srf)
{
   char          str[ 80 ];
   int           i = 0;
   float         x,y,z,w;

   if (( srf->points = DK_nurbsPointsAllocate( srf->noPoints)) ==
NULL )
      return;

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return;
   do
   {
      fscanf( file, "%s", str );

      if( str[0] == '[' )
      {
         i++;
      }
      else if( strcmp( "position", str ) == 0 )
      {
          fscanf( file, "%f %f %f %f", &x,&y,&z,&w);
          srf->points[i-1].x = x;
          srf->points[i-1].y = y;
          srf->points[i-1].z = z;
          srf->points[i-1].w = w;
          #if DBUG
          printf("point read <%f, %f, %f, %f>\n",x,y,z,w);
          #endif
      }
   }  while( strcmp( "}", str ) != 0 );

}   /* DK_nurbsSrfAsciiReadCVs */



static void DK_nurbsSrfAsciiReadKnots(FILE *file, DK_NurbsSrf* srf, DK_Boolean readInU )
{
   char          str[ 80 ];
   int           i = 0;
   float         knot ;


   if ( readInU )
   {

      fscanf( file, "%s", str );
      if( strcmp( str, "{" ) != 0 )
         return;

      do
      {
         fscanf( file, "%s", str );

         if( str[0] == '[' )
         {
             i++;
         }
	 else if (strcmp("noPoints",str ) == 0 )
	     fscanf( file,"%d",&srf->uDir.noPoints);
	 else if (strcmp("order",str ) == 0 )
	     fscanf( file,"%d",&srf->uDir.order);
	 else if (strcmp("NoStep",str ) == 0 )
	     fscanf( file,"%d",&srf->uNoStep);
	 else if (strcmp("Curve",str ) == 0 )
	     fscanf( file,"%d",&srf->uCurve);
	 else if (strcmp("Bottom",str ) == 0 )
	     srf->uBottom = TRUE;
	 else if (strcmp("Top",str ) == 0 )
	     srf->uTop = TRUE;
	 else if (strcmp("CloseInfo",str ) == 0 )
	     fscanf( file,"%d",&srf->uCloseInfo);
	 else if (strcmp("start",str ) == 0 )
	 {
	     fscanf( file,"%g",&knot);
	     srf->uDir.start = knot;
         }
	 else if (strcmp("end",str ) == 0 )
	 {
	     fscanf( file,"%g",&knot);
	     srf->uDir.end = knot ;
         }
	 else if (strcmp("closed",str ) == 0 )
	     srf->uDir.closed = TRUE;
	 else if (strcmp("not_closed",str ) == 0 )
	     srf->uDir.closed = FALSE;
	 else if (strcmp("noKnots",str ) == 0 )
         {
	    fscanf( file,"%d",&srf->uDir.noKnots);
	    srf->uDir.knots = DK_knotsAllocate(srf->uDir.noKnots);
	    if ( srf->uDir.knots == NULL )
		return;
            for ( i =0; i<srf->uDir.noKnots; i++)
	    {
	       fscanf( file, "%s", str);
	       fscanf( file, "%s", str);
	       fscanf( file, "%f", &knot );
	       srf->uDir.knots[i] = knot;
	       #if DBUG
	       printf("knot #%d read <%f>\n",i,knot);
	       #endif
	    }

         }

      }  while( strcmp( "}", str ) != 0 );

      #if DBUG
      printf(" U dir: noPoints: %d, order: %d, start: %f, end: %f, closed:%d, noKnots: %d, uNoStep: %d , uCurve %d \n",srf->uDir.noPoints,srf->uDir.order, srf->uDir.start,srf->uDir.end,srf->uDir.closed, srf->uDir.noKnots, srf->uNoStep, srf->uCurve);
      #endif
   }
   else
   {

      fscanf( file, "%s", str );
      if( strcmp( str, "{" ) != 0 )
         return;

      do
      {
	 fscanf( file, "%s", str );

	 if( str[0] == '[' )
	 {
	    i++;
	 }
         else if (strcmp("noPoints",str ) == 0 )
             fscanf( file,"%d",&srf->vDir.noPoints);
         else if (strcmp("order",str ) == 0 )
             fscanf( file,"%d",&srf->vDir.order);
	 else if (strcmp("NoStep",str ) == 0 )
	     fscanf( file,"%d",&srf->vNoStep);
	 else if (strcmp("Curve",str ) == 0 )
	     fscanf( file,"%d",&srf->vCurve);
	 else if (strcmp("Bottom",str ) == 0 )
	     srf->vBottom = TRUE;
	 else if (strcmp("Top",str ) == 0 )
	     srf->vTop = TRUE;
	 else if (strcmp("CloseInfo",str ) == 0 )
	     fscanf( file,"%d",&srf->vCloseInfo);
         else if (strcmp("start",str ) == 0 )
	 {
             fscanf( file,"%g",&knot);
	     srf->vDir.start = knot;
	 }
         else if (strcmp("end",str ) == 0 )
	 {
             fscanf( file,"%g",&knot);
	     srf->vDir.end = knot;
	 }
         else if (strcmp("closed",str ) == 0 )
             srf->vDir.closed = TRUE;
         else if (strcmp("not_closed",str ) == 0 )
             srf->vDir.closed = FALSE;
         else if (strcmp("noKnots",str ) == 0 )
	 {
             fscanf( file,"%d",&srf->vDir.noKnots);
	     srf->vDir.knots = DK_knotsAllocate(srf->vDir.noKnots);
	     if ( srf->vDir.knots == NULL )
	        return;
             for ( i=0; i<srf->vDir.noKnots;i++)
	     {
		fscanf( file, "%s", str);
		fscanf( file, "%s", str);
		fscanf( file, "%f", &knot );
		srf->vDir.knots[i] = knot;
		#if DBUG
		printf("knot #%d read <%f>\n",i,knot);
		#endif
	     }

         }

      }  while( strcmp( "}", str ) != 0 );

      #if DBUG
      printf(" V dir: noPoints: %d, order: %d, start: %f, end: %f, closed:%d, noKnots: %d, vNoStep: %d , vCurve %d \n",srf->vDir.noPoints,srf->vDir.order, srf->vDir.start,srf->vDir.end,srf->vDir.closed, srf->vDir.noKnots, srf->vNoStep, srf->vCurve);
      #endif
   }




}  /* DK_nurbsSrfAsciiReadKnots */

static void DK_nurbsSrfReadTrims(FILE *file, DK_NurbsSrf* srf)
{
   char          str[ 80 ];
   int           i = 0;
   int           j = 0;
   long		trim = 0;
   int 		lu;



   srf->trims = (DK_NurbsCompCrv**) calloc(srf->noTrims, sizeof(DK_NurbsCompCrv*));
   for( i = 0; i < srf->noTrims; i++ )
   {
      /* read Trim_Curve */
      fscanf( file, "%s", str );

      /* read #*/
      fscanf( file, "%s", str );

      /* read trim curve number */
      fscanf( file, "%d", &lu);

      if ( (srf->trims[i] = (DK_NurbsCompCrv*) 
		malloc(  sizeof( DK_NurbsCompCrv ) )) != NULL)
      {
	 srf->trims[i]->boundary = 0;
	 srf->trims[i]->noCurves = 0;
	 srf->trims[i]->noPoints = 0;
	 srf->trims[i]->curves = NULL;
	 srf->trims[i]->points = NULL;


	 #if DBUG
	 printf("processing trim # %d\n",i);
	 #endif
	 fscanf( file, "%s", str );
	 if( strcmp( str, "Curve_is_a_Boundary" ) == 0 )
	 {
	     srf->trims[i]->boundary = 1;  
	     #if DBUG
	     printf("This is a boundary curve \n");
	     #endif
	     fscanf( file, "%s", str );
             if( strcmp( str, "Curve_number:" ) == 0 )
	     {
	       fscanf( file, "%d", &lu);
	       srf->trims[i]->noCurves = lu + 1;
	       #if DBUG
	       printf("number of curves: %d\n",lu);
	       #endif
	    }
	 }
         else if( strcmp( str, "Curve_number:" ) == 0 )
	 {
	    fscanf( file, "%d", &lu);
	    srf->trims[i]->noCurves = lu + 1;
	     #if DBUG
	     printf("number of curves: %d\n",lu);
	     #endif
         }
	 /* read the Nurbs_Curve */
         srf->trims[i]->curves = (DK_NurbsCrv**) 
			 calloc( srf->trims[i]->noCurves, sizeof(DK_NurbsCrv*));

	 for ( j=0; j<srf->trims[i]->noCurves;j++)
	 {
	    fscanf( file, "%s",str);
	    if( strcmp( str, "Nurbs_Curve" ) == 0 )
	    srf->trims[i]->curves[j] = DK_nurbsCrvAsciiRead(file );
         }
      }
   }
} 
