/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyModel.c,v $
   $Revision: 1.64.14.1 $ $Date: 1995/06/09 22:00:13 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's model functions,
   its content is sufficient to read, write and manage model data structure
   into SOFTIMAGE's file format...

   Written by: Laurent Lauzon

   (c) Copyright 1990, 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinyModel.h"
#include "tinySceneToken.h"
#include "tinyUtils.h"
#include "tinyNurbs.h"

/******************************************************************************/
/*** Static function prototypes ***********************************************/
/******************************************************************************/

static DK_Boolean DK_modelWriteAsciiFile( DK_String, DK_Model * );
static DK_Boolean DK_modelWriteBinaryFile( DK_String filename, DK_Model *mdl );
static DK_Boolean DK_modelWriteAsciiNode( FILE *, DK_Model *, DK_String,
					  DK_Boolean );
static DK_Model *DK_modelReadAsciiFile( DK_String );
static DK_Model *DK_modelReadBinaryFile( DK_String );

#ifndef _WIN32 /* These are declared in tinyModel.h for the _WIN32 case */
static DK_Model *DK_modelReadAsciiNode( FILE *, DK_Model * );
static DK_Model *DK_modelReadBinaryNode( FILE *, DK_Model *, DK_FileProt * );
static DK_Boolean DK_modelWriteBinaryNode( FILE *, DK_Model *, DK_FileProt *,
					   DK_Boolean );
#endif


static int 	DK_shapefcvWriteAsciiAnimation( FILE *, DK_Fcurve *,
						DK_Fcurve **, char *, int );
static int 	DK_shapefcvReadAsciiAnimation( FILE *, DK_Fcurve **,
					       DK_Fcurve ***, char **, int * );
static int 	DK_modelNodeWriteAsciiAnimation( FILE *, DK_Model * );
static int 	DK_modelWriteAsciiAnimation( FILE *, DK_Model * );
static int 	DK_modelReadAsciiAnimation( FILE *, DK_Model * );

/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

DK_Model *DK_modelAllocate( void )
{
   DK_Model *mdl;
   int i;

   if( mdl = (DK_Model *) calloc( 1, sizeof( DK_Model ) ) )
   {
      mdl->filename 	 = NULL;
      mdl->name          = NULL;
      mdl->prefix        = NULL;
      mdl->type          = DK_MDL_NILL;
      mdl->definition    = NULL;
      mdl->Kdefinition    = NULL;

      mdl->scaling.x     = mdl->scaling.y     = mdl->scaling.z     = 1.0;
      mdl->rotation.x    = mdl->rotation.y    = mdl->rotation.z    = 0.0;
      mdl->translation.x = mdl->translation.y = mdl->translation.z = 0.0;

      mdl->pos_constraint_name = NULL;
      mdl->pos_constraint_type = 0;

      mdl->path = NULL;
      mdl->path_name 	 = NULL;

      mdl->shape_filename = NULL;
      mdl->shape_interp_type = DK_LINEAR_SHAPE_INTERP;
      mdl->nbshape    = 0;
      mdl->shapes     = NULL;

      /* clusters */
      mdl->cluster 	= NULL;
      /* klusters */
      mdl->klusters 	= NULL;

      mdl->currentMaterial = 0;
      mdl->nbMaterials   = 0;
      mdl->materials     = NULL;
      mdl->nbTextures    = 0;
      mdl->textures      = NULL;
      mdl->nbTextures3D  = 0;
      mdl->textures3D    = NULL;
 
      mdl->nbnodelat     = 0;
      mdl->nodelat       = NULL;
      mdl->nbtreelat     = 0;
      mdl->treelat       = NULL;
      mdl->curlatdef = 0;

      mdl->parent        = NULL;
      mdl->sibling       = NULL;
      mdl->child         = NULL;

      /* dynamics, envelope, inverse kinematics */
      mdl->algotype 	 = DK_ASTD;
      mdl->envtype 	 = DK_ENONE;
      mdl->dcinfo    	 = NULL;
      mdl->dchnvar    	 = NULL;
      mdl->dmatter    	 = NULL;
      mdl->djntinfo    	 = NULL;
      mdl->ikvar    	 = NULL;
      mdl->envdata    	 = NULL;
      mdl->init_srt      = NULL;

      /* fitting */
      mdl->nodefit 	 = NULL;
      mdl->brchfit 	 = NULL;
      mdl->nodepfit 	 = NULL;
      mdl->brchpfit 	 = NULL;

      mdl->sub_active = 0;
      mdl->sub_model_name = NULL;
      mdl->sub_start = 0;
      mdl->sub_end = 0;
      mdl->sub_inmemory = 0;
      mdl->sub_pre_script_file = NULL;
      mdl->sub_post_script_file = NULL;

      mdl->dcinfo = NULL;
      mdl->dchnvar = NULL;
      mdl->dmatter = NULL;
      mdl->djntinfo = NULL;
      mdl->algotype = DK_ASTD;

      mdl->ikvar = NULL;

      mdl->envtype = DK_ENONE;
      mdl->envdata = NULL;
      mdl->init_srt = NULL;

      mdl->parent = NULL;
      mdl->sibling = NULL;
      mdl->child = NULL;

      /* display */
      mdl->wire_colour = DK_DEFAULT_WIRE_COLOUR;
      mdl->line_type   = DK_VISIBLE;
      mdl->selected    = FALSE;
      mdl->bbox = FALSE;
      _DK_VectorInit(mdl->bbox_min, 0.0, 0.0, 0.0);
      _DK_VectorInit(mdl->bbox_max, 0.0, 0.0, 0.0);
      mdl->schem_posx  = 0.0;
      mdl->schem_posy  = 0.0;
      mdl->schem_coll  = 0;
      mdl->schem_userp = 0;

      mdl->constr_list   = NULL;
      for (i = 0; i < DK_FCV_MDL_NUM; i++)
      {
	  mdl->fcv[i] = NULL;
	  mdl->active_fcv[i].active = FALSE;
	  mdl->active_fcv[i].fcv = NULL;
      }
      mdl->wfcv = NULL;
      mdl->active_wfcv = NULL;

      mdl->userDataList = NULL;

      mdl->next          = NULL;

      /* Render Setup */
      mdl->visibilityFlags = DK_PRIMARY_VIS | DK_TRANSMIT_VIS | DK_SHADOW_VIS;
      mdl->motionBlurFlags = DK_MB_NONE;
   }

   return( mdl );
}

void DK_modelDispose( DK_Model **mdl )
{
   if( mdl )
   {
      DK_Model *model = *mdl;

      while( model )
      {
         DK_Model *save = model;

         _DK_FREE( model->filename, -1 );
         _DK_FREE( model->name, -1 );
         _DK_FREE( model->prefix, -1 );

         switch( model->type )
         {
            case DK_MDL_NILL:
            case DK_MDL_JNT:
               break;
            case DK_MDL_FACE: {
               DK_Face *fac = (DK_Face *) model->definition;
               DK_faceDispose( &fac );
            } break;
            case DK_MDL_PTCH: {
               DK_Patch *ptch = (DK_Patch *) model->definition;
               DK_patchDispose( &ptch );
            } break;
            case DK_MDL_MESH: {
               DK_Mesh *msh = (DK_Mesh *) model->definition;
               DK_meshDispose( &msh );
            } break;
            case DK_MDL_SPLN: {
               DK_Spline *spl = (DK_Spline *) model->definition;
               DK_splineDispose( &spl );
            } break;
            case DK_MDL_BALL: {
               DK_Metaball *ball = (DK_Metaball *) model->definition;
               DK_metaballDispose( &ball );
            } break;
            case DK_MDL_META: {
               DK_MetaSystem *meta = (DK_MetaSystem *) model->definition;
               DK_metaSystemDispose( &meta );
            } break;
            case DK_MDL_CRV: {
               DK_NurbsCrv *crv = (DK_NurbsCrv*) model->definition;
               DK_nurbsCrvDispose( &crv);
            } break;
            case DK_MDL_SRF: {
               DK_NurbsSrf *srf = (DK_NurbsSrf*) model->definition;
               DK_nurbsSrfDispose( &srf);
            } break;
         }

         _DK_FREE( model->pos_constraint_name, -1 );
         if( model->path )       DK_splineDispose( &model->path );
         _DK_FREE( model->path_name, -1 );
         _DK_FREE( model->shape_filename, -1 );
         if( model->materials )  DK_materialDispose( &model->materials );
         if( model->textures )   DK_textureDispose( &model->textures );
         if( model->textures3D ) DK_texture3DDispose( &model->textures3D );
         if( model->nodelat )    DK_latticeDispose( &model->nodelat );
         if( model->treelat )    DK_latticeDispose( &model->treelat );
         if( model->nodefit )    DK_spl_fittingDispose( &model->nodefit );
         if( model->brchfit )    DK_spl_fittingDispose( &model->brchfit );
         if( model->nodepfit )   DK_ptch_fittingDispose( &model->nodepfit );
         if( model->brchpfit )   DK_ptch_fittingDispose( &model->brchpfit );
         if( model->cluster )   DK_clusterDispose( &model->cluster );
         if( model->klusters )   DK_klusterDispose( &model->klusters );
         _DK_FREE( model->sub_model_name, -1 );
         _DK_FREE( model->sub_pre_script_file, -1 );
         _DK_FREE( model->sub_post_script_file, -1 );
         if( model->dcinfo )     DK_dynaConstrInfoDispose( &model->dcinfo );
         if( model->dchnvar )    DK_dynaChnVarDispose( &model->dchnvar );
         if( model->dmatter )    DK_dynaMatterInfoDispose( &model->dmatter );
         if( model->djntinfo )   DK_dynaJntInfoDispose( &model->djntinfo );
         if( model->ikvar )      DK_invKinDispose( &model->ikvar );
         if( model->envdata )    DK_envelopDataDispose( &model->envdata );
	 _DK_FREE( model->init_srt, -1 );
         if( model->child )      DK_modelDispose( &model->child );
         if( model->sibling )    DK_modelDispose( &model->sibling );

	 DK_constraintDispose( &model->constr_list );
         DK_fcurveArrayDispose( model->fcv, DK_FCV_MDL_NUM );
         DK_actFcurveArrayDispose( model->active_fcv, DK_FCV_MDL_NUM );

         if( model->wfcv )
         {
            DK_fcurveArrayDispose( model->wfcv, model->nbshape );
            DK_actFcurveArrayDispose( model->active_wfcv, model->nbshape );
            _DK_FREE( model->wfcv, -1 );
            _DK_FREE( model->active_wfcv, -1 );
         }

         if( model->userDataList != NULL )
         {
            DK_UserNode       *ptr, *next;

            for( ptr = model->userDataList; ptr != NULL; ptr = next )
            {
               next = ptr->next;
               if( ptr->data != NULL )
                  _DK_FREE( ptr->data, -1 );
               _DK_FREE( ptr, -1 );
            }
         }

         model = model->next;
         free( save );
      }

      *mdl = NULL;
   }
}

DK_Boolean	DK_modelCreateParentship( DK_Model *fth, DK_Model *mdl )
{
   DK_Model	*ptr;

  /*********************************
   check if mdl already has a parent
   *********************************/

   if( mdl->parent != NULL )
      return( FALSE );

  /**************************************************************
   Check if mdl is not in the same hierarchy as fth.
   Since mdl doesn't have a parent, the only way to it to be in
   the same hierarchy of fth is to be the root of this hierarchy.
   **************************************************************/

   for( ptr = fth; ptr != NULL; ptr = ptr->parent )
      if( ptr == mdl )
         return( FALSE );

  /**********************
   OK to create hierarchy
   **********************/

   if( fth->child == NULL )
      fth->child = mdl;
   else
   {
      for( ptr = fth->child; ptr->sibling != NULL; ptr = ptr->sibling )
         /* NOP */ ;
         ptr->sibling = mdl;
   }
   mdl->parent = fth;

   return( TRUE );
}

DK_Boolean DK_modelCut( DK_Model *mdl )
{
   DK_Model	*ptr, *fth;

   if( (fth = mdl->parent) != NULL )
   {
      if( fth->child == mdl )
         fth->child = mdl->sibling;
      else
      {
         for( ptr = fth->child; (ptr != NULL) && (ptr->sibling != mdl);
                                                            ptr = ptr->sibling )
            /* NOP */ ;

         if( ptr == NULL )
            return( FALSE );

         ptr->sibling = mdl->sibling;
      }

      mdl->sibling = NULL;
      mdl->parent = NULL;
   }

   return( TRUE );
}

DK_Model *DK_modelNextNode
    ( 
	DK_Model *mdl
    )
{
   if( mdl->child != NULL )
      return( mdl->child );

   if( mdl->sibling != NULL )
      return( mdl->sibling );

   while( (mdl->sibling == NULL) && (mdl->parent != NULL) )
      mdl = mdl->parent;

   if( mdl->sibling != NULL )
      return( mdl->sibling );

   return( NULL );
}

DK_Model *DK_modelNodeFindByName
    ( 
	DK_Model *mdl,
	DK_String  name
    )
{
   DK_Model   *ptr;

   ptr = mdl;
   while( ptr && (ptr->name == NULL || strcmp( name, ptr->name )) )
   {
      if( ptr->child )
         ptr = ptr->child;
      else if( ptr->sibling )
         ptr = ptr->sibling;
      else
      {
         while( ptr && !ptr->sibling )
            ptr = ptr->parent;
         if( ptr && ptr->sibling )
            ptr = ptr->sibling;
      }
   }
   return( ptr );
}

DK_Model *DK_modelReadFile( DK_String filename )
{
   char		 buf[256];
   long          id, len;
    FILE		*file;

   if( (len = strlen( filename )) > 4 &&
       strcmp( &filename[len-4], ".hrc" ) == 0 )
      strcpy( buf, filename );
   else
      sprintf( buf, "%s.hrc", filename );

   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( NULL );

   DK_byteswap_fread( &id, 4, 1, file );
   fclose( file );

   if( id == 'HRCH' )
   {
      return( DK_modelReadAsciiFile( buf ) );
   }
   else if( id == DK_BINARY_MAGIC_NUMBER )
   {
      return( DK_modelReadBinaryFile( buf ) );
   }
   else
      return( NULL );
}

DK_Boolean DK_modelWriteFile( DK_String filename, DK_Model *mdl , DK_FileFormat fmt )
{
   char new_name[ 256 ];
   long len;

   if( (len = strlen( filename )) > 4 &&
       strcmp( &filename[len-4], ".hrc" ) == 0 )
      strcpy( new_name, filename );
   else
      sprintf( new_name, "%s.hrc", filename );

   if( fmt ==  DK_FIL_ASCII )
      return( DK_modelWriteAsciiFile( new_name, mdl ) );
   else if( fmt ==  DK_FIL_BINARY )
      return( DK_modelWriteBinaryFile( new_name, mdl ) );
   else
      return( FALSE );
}


/******************************************************************************/
/*** Static functions *********************************************************/
/******************************************************************************/

static DK_Boolean DK_modelWriteAsciiFile( DK_String filename, DK_Model *mdl )
{
   FILE		*file;
   DK_Boolean    success;

   if( ( file = fopen( filename, "wb" ) ) == NULL )
      return( FALSE );

   success = DK_headerWriteAscii( file, "HRCH" );
   success &= DK_modelWriteAsciiNode( file, mdl, "", FALSE );
   /* Verify if file was saved correctly */
   fflush( file );
   success &= (ferror( file ) == 0 );

   fclose( file );
   return( success );
}

DK_Boolean DK_fdmodelWriteBinaryFile( FILE *fd, DK_Model *mdl )
{
   DK_FileProt	*prot;
   DK_Boolean	 success;

   DK_headerWriteBinary( fd, NULL );
   DK_prototypeInitModel( &prot, TRUE );
   DK_prototypeWrite( fd, prot );
   DK_String_fwrite( "HRCH", 4, 1, fd );
   success = DK_modelWriteBinaryNode( fd, mdl, prot, FALSE );

   /* Verify if file was saved correctly */

   fflush( fd );
   success &= (ferror( fd ) == 0 );
   DK_prototypeDispose( &prot );
   return( success );
}

static DK_Boolean DK_modelWriteBinaryFile( DK_String filename, DK_Model *mdl )
{
   FILE	       *file;
   DK_Boolean  success;

   if( ( file = fopen( filename, "wb" ) ) == NULL )
      return( FALSE );

   success = DK_fdmodelWriteBinaryFile( file, mdl );

   fclose( file );
   return( success );
}

static DK_Boolean DK_modelWriteAsciiNode(  FILE *file,
			    DK_Model *mdl,
			    DK_String space,
			    DK_Boolean sibling )
{
   char			sspace[ 80 ];
   DK_Face		*fac;
   DK_Mesh		*msh;
   DK_Patch		*ptch;
   DK_Spline 		*spl;
   DK_Material		*mtrl;
   DK_Texture		*txt;
   DK_Texture3D		*txt3D;
   DK_Metaball		*ball;
   DK_MetaSystem 	*meta;
   DK_NurbsCrv		*crv = NULL;
   DK_NurbsSrf		*srf = NULL;

   int			 i;
   
   if ( sibling )
   {
      /* Do nothing for now. */
   }

   sprintf( sspace, "%s  ", space );

   fprintf( file, "\n%smodel\n", space );
   fprintf( file, "%s{\n", space );
   fprintf( file, "%sname         \"%s\"\n", sspace, mdl->name );
   fprintf( file, "%sscaling      %*.*f %*.*f %*.*f\n", sspace,
             DK_FM,  DK_FD, mdl->scaling.x,  DK_FM,  DK_FD, mdl->scaling.y,  DK_FM,  DK_FD, mdl->scaling.z );
   fprintf( file, "%srotation     %*.*f %*.*f %*.*f\n", sspace,
                                   DK_FM,  DK_FD, _DK_Rad2Deg( mdl->rotation.x ),
                                   DK_FM,  DK_FD, _DK_Rad2Deg( mdl->rotation.y ),
                                   DK_FM,  DK_FD, _DK_Rad2Deg( mdl->rotation.z ) );
   fprintf( file, "%stranslation  %*.*f %*.*f %*.*f\n", sspace,
             DK_FM,  DK_FD, mdl->translation.x,  DK_FM,  DK_FD, mdl->translation.y,  DK_FM,  DK_FD, mdl->translation.z );


   switch( mdl->type )
   {
      case DK_MDL_NILL :
      case DK_MDL_JNT :
	 break;
      case DK_MDL_FACE:
         fac = (DK_Face *) mdl->definition;
         DK_faceWriteAscii( file, fac, sspace );
         break;
      case DK_MDL_PTCH:
         ptch = (DK_Patch *) mdl->definition;
         DK_patchWriteAscii( file, ptch, sspace );
         break;
      case DK_MDL_MESH:
         msh = (DK_Mesh *) mdl->definition;
         DK_meshWriteAscii( file, msh, sspace );
         break;
      case DK_MDL_SPLN:
         spl = (DK_Spline *) mdl->definition;
         DK_splineWriteAscii( file, spl, sspace );
         break;
      case DK_MDL_BALL:
         ball = (DK_Metaball *) mdl->definition;
         DK_metaballWriteAscii( file, ball, sspace );
         break;
      case DK_MDL_META:
         meta = (DK_MetaSystem *) mdl->definition;
         DK_metaSystemWriteAscii( file, meta, sspace );
         break;
      case DK_MDL_CRV:
	 crv = (DK_NurbsCrv *) mdl->definition;
	 DK_nurbsCrvAsciiWrite( file, crv, sspace);
	 break;
      case DK_MDL_SRF:
	 srf = (DK_NurbsSrf *) mdl->definition;
	 DK_nurbsSrfAsciiWrite( file, srf, sspace);
         break;
   } 

   if ( DK_VERSION >= DK_v2_6 && mdl->klusters != NULL )
      DK_clustersWriteAscii( file, mdl, sspace );

   for( i = 0, mtrl = mdl->materials; i < mdl->nbMaterials; i++, mtrl = mtrl->next )
      DK_materialWriteAscii( file, mtrl, sspace, i );

   for( i = 0, txt = mdl->textures; i < mdl->nbTextures; i++, txt = txt->next )
      DK_textureWriteAscii( file, txt, sspace, i );

   for( i = 0, txt3D = mdl->textures3D; i < mdl->nbTextures3D; i++, txt3D = txt3D->next )
      DK_texture3DWriteAscii( file, txt3D, sspace, i );

   if ( mdl->algotype != DK_ASTD || mdl->dmatter != NULL || mdl->envtype != DK_ENONE )
   {
      DK_modelActorDataWriteAscii( file, mdl, sspace );
   }

   if( DK_VERSION >= DK_v3_0 )
   {
      if ( mdl->init_srt != NULL )
      {
	 fprintf( file, "%sinit_scl  %*.*f %*.*f %*.*f\n", sspace,
		  DK_FM, DK_FD, mdl->init_srt->scl.x,
		  DK_FM, DK_FD, mdl->init_srt->scl.y,
		  DK_FM, DK_FD, mdl->init_srt->scl.z );
	 fprintf( file, "%sinit_rot  %*.*f %*.*f %*.*f\n", sspace,
		  DK_FM, DK_FD, mdl->init_srt->rot.x,
		  DK_FM, DK_FD, mdl->init_srt->rot.y,
		  DK_FM, DK_FD, mdl->init_srt->rot.z );
	 fprintf( file, "%sinit_trs  %*.*f %*.*f %*.*f\n", sspace,
		  DK_FM, DK_FD, mdl->init_srt->trs.x,
		  DK_FM, DK_FD, mdl->init_srt->trs.y,
		  DK_FM, DK_FD, mdl->init_srt->trs.z );
      }
   }

   if( DK_VERSION >= DK_v2_6 )
   {
      DK_UserNode    *node;

      for( node = mdl->userDataList; node != NULL; node = node->next )
         DK_userDataWriteAscii( file, node, sspace );
   }

   if( mdl->child )
      DK_modelWriteAsciiNode( file, mdl->child, sspace, TRUE );

   fprintf( file, "%s}\n", space );

   if( mdl->sibling )
      DK_modelWriteAsciiNode( file, mdl->sibling, space, TRUE );

   return( TRUE );
}

#ifndef _WIN32
static DK_Boolean DK_modelWriteBinaryNode(  FILE *file,
			     DK_Model *mdl,
			     DK_FileProt *prot,
			     DK_Boolean writesibling )
#else /* Win32 version is global */
DK_Boolean DK_modelWriteBinaryNode(  FILE *file,
			     DK_Model *mdl,
			     DK_FileProt *prot,
			     DK_Boolean writesibling )
#endif

{
   short		 ind;
   short		 nb;
   short		 child;
   short		 sibling;
   short		 external;
   DK_Material		*mtrl;
   DK_Texture		*txt;
   DK_Texture3D		*txt3D;
   DK_Face		*fac;
   DK_Patch		*ptch;
   DK_Mesh		*msh;
   DK_Spline 		*spl;
   DK_Metaball		*ball;
   DK_MetaSystem 	*meta;
   DK_UserNode		*dataPtr;
   DK_NurbsCrv*		crv = NULL;
   DK_NurbsSrf*		srf = NULL;


   for( ind = 0; prot->model[ ind ] != 0; ind++ )
   {
      switch( prot->model[ ind ] )
      {
         case 1:	/* name */
            if( DK_stringWriteBinary( file, mdl->name ) == FALSE )
               return( FALSE );
            break;

         case 2:	/* type */
            nb = (short) mdl->type;
            if( DK_byteswap_fwrite( &nb, 2, 1, file ) != 1 )
               return( FALSE );
            break;

         case 3:	/* definition */
            switch( mdl->type )
            {
               case DK_MDL_NILL:
                  break;
               case DK_MDL_MESH:
                  msh = (DK_Mesh *) mdl->definition;
                  if( ! DK_meshWriteBinary( file, msh, prot ) )
                     return( FALSE );
                  break;
               case DK_MDL_FACE:
                  fac = (DK_Face *) mdl->definition;
                  if( ! DK_faceWriteBinary( file, fac, prot ) )
                     return( FALSE );
                  break;
               case DK_MDL_PTCH:
                  ptch = (DK_Patch *) mdl->definition;
                  if( ! DK_patchWriteBinary( file, ptch, prot ) )
                     return( FALSE );
                  break;
	       case DK_MDL_SPLN:
		  spl = (DK_Spline *) mdl->definition;
		  if ( ! DK_splineWriteBinary( file, spl, prot))
		      return(FALSE);
		  break;
	       case DK_MDL_BALL:
		  ball = (DK_Metaball *) mdl->definition;
		  if ( ! DK_metaballWriteBinary( file, ball, prot))
		      return(FALSE);
		  break;
	       case DK_MDL_META:
		  meta = (DK_MetaSystem *) mdl->definition;
		  if ( ! DK_metaSystemWriteBinary( file, meta, prot))
		      return(FALSE);
		  break;
	       case DK_MDL_CRV:
		  crv = (DK_NurbsCrv*) mdl->definition;
		  if ( ! DK_nurbsCrvBinaryWrite( file, crv))
		      return(FALSE);
		  break;
	       case DK_MDL_SRF:
		  srf = (DK_NurbsSrf*) mdl->definition;
		  if ( ! DK_nurbsSrfBinaryWrite( file, srf ))
		      return(FALSE);
		  break;
            }
            break;

         case 4: 	/* scaling */
            if( DK_Vector_fwrite( &mdl->scaling, sizeof( DK_Vector ), 1, file ) != 1 )
               return( FALSE );
            break;

         case 5: 	/* rotation */
            if( DK_Vector_fwrite( &mdl->rotation, sizeof( DK_Vector ), 1, file ) != 1 )
               return( FALSE );
            break;

         case 6: 	/* translation */
            if( DK_Vector_fwrite( &mdl->translation, sizeof( DK_Vector ), 1, file ) != 1 )
               return( FALSE );
            break;

         case 7:	/* materials list */
            if( DK_byteswap_fwrite( &mdl->nbMaterials, 2, 1, file ) != 1 )
               return( FALSE );
            for( mtrl = mdl->materials; mtrl != NULL; mtrl = mtrl->next )
            {
               external = FALSE;
               if( DK_byteswap_fwrite( &external, 2, 1, file ) != 1 )
                  return( FALSE );
               
               /* Turns out that 4th param is useless since all  */
               /* local textures are to be saved.  v1.12 of      */
               /* tinyMaterial.c rendered the 4th param useless. */

               if( ! DK_materialWriteBinary( file, mtrl, prot ) )
                  return( FALSE );
            }
            break;

         case 8:	/* textures list */
            if( DK_byteswap_fwrite( &mdl->nbTextures, 2, 1, file ) != 1 )
               return( FALSE );
            for( txt = mdl->textures; txt != NULL; txt = txt->next )
            {
               external = FALSE;
               if( DK_byteswap_fwrite( &external, 2, 1, file ) != 1 )
                  return( FALSE );
               if( ! DK_textureWriteBinary( file, txt, prot )  )
                  return( FALSE );
            }
            break;

         case 9:	/* 3D textures list */
            if( DK_byteswap_fwrite( &mdl->nbTextures3D, 2, 1, file ) != 1 )
               return( FALSE );
            for( txt3D = mdl->textures3D; txt3D != NULL; txt3D = txt3D->next )
            {
               external = FALSE;
               if( DK_byteswap_fwrite( &external, 2, 1, file ) != 1 )
                  return( FALSE );
               if( ! DK_texture3DWriteBinary( file, txt3D, prot )  )
                  return( FALSE );
            }
            break;

         case 10:	/* child */
            child = (mdl->child != NULL);
            if( DK_byteswap_fwrite( &child, 2, 1, file ) != 1 )
               return( FALSE );
            if( child )
               if( ! DK_modelWriteBinaryNode( file, mdl->child, prot, TRUE ) )
                  return( FALSE );
            break;

         case 11:	/* next sibling */
            sibling = (short) (writesibling && (mdl->sibling != NULL));
            if( DK_byteswap_fwrite( &sibling, 2, 1, file ) != 1 )
               return( FALSE );
            if( sibling )
               if( ! DK_modelWriteBinaryNode( file, mdl->sibling, prot, TRUE ) )
                  return( FALSE );
            break;

	 case 12:
	    nb = mdl->shape_interp_type;
	    DK_byteswap_fwrite( &nb, 2, 1, file );
	    break;
	 case 13:
	    break;
	 case 14: /*  physical properties (dynamic matter) info */
	    nb = (short) ( mdl->dmatter != NULL );
	    DK_byteswap_fwrite( &nb, 2, 1, file );
	    if ( mdl->dmatter != NULL )
	    {
               external = FALSE;
	       DK_byteswap_fwrite( &external, 2, 1, file );
	       DK_dynamatterWriteBinaryChunk( file,  mdl->dmatter  );
	    }
	    break;

	 case 15:
	    break;
 	 case 16: /* algo type */
	    nb = (short) mdl->algotype;
	    DK_byteswap_fwrite( &nb, 2, 1, file );
	    break;

 	 case 17: /* envelope info */
	    nb = (short) mdl->envtype;
	    DK_byteswap_fwrite( &nb, 2, 1, file );

	    nb = (short) ( mdl->envdata != NULL );
	    DK_byteswap_fwrite( &nb, 2, 1, file );
	    if ( mdl->envdata != NULL )
	       DK_envelopWriteBinaryChunk( file, mdl->envdata );
	    break;

 	 case 18: /* dynamic  DK_options data (only for space allocation) */
	    nb = (short) ( mdl->dchnvar != NULL );
	    DK_byteswap_fwrite( &nb, 2, 1, file );
	    break;

 	 case 19: /* joint dynamic properties info */
	    nb = (short) ( mdl->djntinfo != NULL );
	    DK_byteswap_fwrite( &nb, 2, 1, file );
	    if ( mdl->djntinfo != NULL )
	    {
	       DK_dynajntpropWriteBinaryChunk( file, mdl->djntinfo );
	    }
	    break;

 	 case 20: /* inverse kinematic data */
	    nb = (short) ( mdl->ikvar != NULL );
	    DK_byteswap_fwrite( &nb, 2, 1, file );
	    if ( mdl->ikvar != NULL )
	    {
	       DK_invKinWriteBinaryChunk( file, mdl->ikvar );
	    }
	    break;
   	case 21: /* cluster data */
	   nb = DK_clusterCount( mdl );
	   DK_byteswap_fwrite( &nb, 2, 1, file );
	   if ( nb > 0 )
	   {
	      DK_clustersWriteBinaryChunk( file, mdl );
	   }
	   break;

   	case 22: /* user data */
           nb = 0;
           for( dataPtr = mdl->userDataList; dataPtr; dataPtr = dataPtr->next )
              nb++;
           DK_byteswap_fwrite( &nb, 2, 1, file );
           for( dataPtr = mdl->userDataList; dataPtr; dataPtr = dataPtr->next )
           {
              DK_byteswap_fwrite(  dataPtr->type, 4, 1, file );
              DK_byteswap_fwrite( &dataPtr->size, 4, 1, file );
              DK_byteswap_fwrite(  dataPtr->data, 1, dataPtr->size, file );
           }
	   break;

        case 23: 	/* visibilityFlags */
	   DK_byteswap_fwrite( &mdl->visibilityFlags, sizeof(short), 1, file);
	   break;

        case 24: 	/* motionBlurFlags */
	   DK_byteswap_fwrite( &mdl->motionBlurFlags, sizeof(short), 1, file);
	   break;

	case 25: /* rigid env. init. transfos. */
	   nb = (short) ( mdl->init_srt != NULL );
	   DK_byteswap_fwrite( &nb, 2, 1, file );
	   if ( mdl->init_srt != NULL )
	   {
	      DK_Vector_fwrite( &mdl->init_srt->scl, sizeof( DK_Vector ), 1, file );
	      DK_Vector_fwrite( &mdl->init_srt->rot, sizeof( DK_Vector ), 1, file );
	      DK_Vector_fwrite( &mdl->init_srt->trs, sizeof( DK_Vector ), 1, file );
	   }
	   break;

         default:
	    if ( DK_option.verbose == TRUE)
                fprintf( DK_option.msgFd, "Error: %s: %hd: illegal token: %hd\n",
		     __FILE__, __LINE__, prot->model[ ind ] );
            break;
      }
   }
   return( TRUE );
}

static DK_Model *DK_modelReadAsciiFile( DK_String filename )
{
   char          str[ 80 ];
   FILE		*file;
   DK_Model	*mdl = NULL;
   long		 id;
   float	 version;

   if( ( file = fopen( filename, "rb" ) ) == NULL )
      return( NULL );

   if( (id = DK_headerReadAscii( file, &version )) != 'HRCH' )
      return( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str , "model" ) == 0 )
   {
      mdl = DK_modelReadAsciiNode( file, NULL );
      mdl->filename = strdup( filename );
   } 

   fclose( file );

   return( mdl );
}

DK_Model *DK_fdmodelReadBinaryFile( FILE *fd )
{
   float	 version;
   long		 id;
   DK_FileProt	*prot = NULL;
   DK_Model	*mdl = NULL;

   id = DK_headerReadBinary( fd, &version, NULL );
   if( id == 'PROT' )
   {
      prot = DK_prototypeRead( fd );
      DK_byteswap_fread( &id, 4, 1, fd );
   }
   if( id == 'HRCH' )
   {
      mdl = DK_modelReadBinaryNode( fd, NULL, prot );
   }

   DK_prototypeDispose( &prot );

   return( mdl );
}

static DK_Model *DK_modelReadBinaryFile( DK_String filename )
{
   FILE		*file;
   float	 version;
   long		 id;
   DK_FileProt	*prot = NULL;
   DK_Model	*mdl = NULL;

   if( ( file = fopen( filename, "rb" ) ) == NULL )
      return( NULL );

   mdl = DK_fdmodelReadBinaryFile( file );

   if ( mdl )
      mdl->filename = strdup( filename );

   fclose( file );

   return( mdl );
}

#ifndef _WIN32
static DK_Model *DK_modelReadBinaryNode(  FILE *file, DK_Model *parent, DK_FileProt *prot )
#else /* Win32 version is global */
DK_Model *DK_modelReadBinaryNode(  FILE *file, DK_Model *parent, DK_FileProt *prot )
#endif
{
   DK_Model	*mdl;
   short	 external;
   short	 i, nb, tok, ind = 0;
   DK_Material	*mtrl  = NULL;
   DK_Texture	*txt   = NULL;
   DK_Texture3D	*txt3D = NULL;
   DK_String	 name;
   DK_UserNode           *dataPtr, *tail = NULL;


   if( (mdl = DK_modelAllocate()) == NULL )
      return( NULL );

   mdl->parent = parent;

   if( (prot != NULL) && (prot->model != NULL) )
      tok = prot->model[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );
   while( tok != 0 /* End of model definition */ )
   {
      switch( tok )
      {
         case 1:  /* name of model node */
            mdl->name = DK_stringReadBinary( file );
            break;

         case 2: /* type of model node */
            DK_byteswap_fread( &nb, 2, 1, file );
            mdl->type = (DK_ModelType) nb;
            break;

         case 3: /* definition */
            switch( mdl->type )
            {
               case DK_MDL_NILL:
               case DK_MDL_LINK:
               case DK_MDL_JNT:
               case DK_MDL_PMSH:
                  break;

               case DK_MDL_MESH:
                  mdl->definition = (DK_Pointer) DK_meshReadBinary( file, prot );
                  break;
               case DK_MDL_FACE:
                  mdl->definition = (DK_Pointer) DK_faceReadBinary( file, prot );
                  break;
               case DK_MDL_PTCH:
                  mdl->definition = (DK_Pointer) DK_patchReadBinary( file, prot );
                  break;
               case DK_MDL_SPLN:
                  mdl->definition = (DK_Pointer) DK_splineReadBinary( file, prot );
                  break;
               case DK_MDL_BALL:
                  mdl->definition = (DK_Pointer) DK_metaballReadBinary( file, prot );
                  break;
               case DK_MDL_META:
                  mdl->definition = (DK_Pointer) DK_metaSystemReadBinary( file, prot );
                  break;
               case DK_MDL_CRV:
                  mdl->definition = (DK_Pointer) DK_nurbsCrvBinaryRead( file, prot );
                  break;
               case DK_MDL_SRF:
                  mdl->definition = (DK_Pointer) DK_nurbsSrfBinaryRead( file, prot );
                  break;

               default:
                  DK_fileSkipField( file, 0xffff );

            }
            break;

         case 4: /* scaling */
            DK_Vector_fread( &mdl->scaling, sizeof( DK_Vector ), 1, file );
            break;

         case 5: /* rotation */
            DK_Vector_fread( &mdl->rotation, sizeof( DK_Vector ), 1, file );
            break;

         case 6: /* translation */
            DK_Vector_fread( &mdl->translation, sizeof( DK_Vector ), 1, file );
            break;

         case 7: /* DK_material list */
            DK_byteswap_fread( &mdl->nbMaterials, 2, 1, file );
            for( i = 0; i < mdl->nbMaterials; i++ )
            {
               DK_byteswap_fread( &external, 2, 1, file );
               if( external )  /* extern  DK_material */
               {
                  name = DK_stringReadBinary( file );
                  _DK_FREE( name, -1 );
               }
               else
                  if( mtrl == NULL )
                     mtrl = mdl->materials = DK_materialReadBinary( file, prot );
                  else
                  {
                     mtrl->next = DK_materialReadBinary( file, prot );
                     mtrl = mtrl->next;
                  }
            }
            break;

         case 8: /* texture list */
            DK_byteswap_fread( &mdl->nbTextures, 2, 1, file );
            for( i = 0; i < mdl->nbTextures; i++ )
            {
               DK_byteswap_fread( &external, 2, 1, file );
               if( external )  /* extern  DK_material */
               {
                  name = DK_stringReadBinary( file );
                  _DK_FREE( name, -1 );
               }
               else
                  if( txt == NULL )
                     txt = mdl->textures = DK_textureReadBinary( file, prot );
                  else
                  {
                     txt->next = DK_textureReadBinary( file, prot );
                     txt = txt->next;
                  }
            }
            break;

         case 9: /* 3D texture list */
            DK_byteswap_fread( &mdl->nbTextures3D, 2, 1, file );
            for( i = 0; i < mdl->nbTextures3D; i++ )
            {
               DK_byteswap_fread( &external, 2, 1, file );
               if( external )  /* extern  DK_material */
               {
                  name = DK_stringReadBinary( file );
                  _DK_FREE( name, -1 );
               }
               else
               {
                  if( txt3D == NULL )
                     txt3D = mdl->textures3D = DK_texture3DReadBinary( file, prot);
                  else
                  {
                     txt3D->next = DK_texture3DReadBinary( file, prot );
                     txt3D = txt3D->next;
                  }
               }
            }
            break;

         case 10: /* child */
            DK_byteswap_fread( &nb, 2, 1, file );
            if( nb != 0 )
               mdl->child = DK_modelReadBinaryNode( file, mdl, prot );
            break;

         case 11: /* brother */
            DK_byteswap_fread( &nb, 2, 1, file );
            if( nb != 0 )
               mdl->sibling = DK_modelReadBinaryNode( file, parent, prot );
            break;

         case 12:
            DK_byteswap_fread( &nb, 2, 1, file );
	    mdl->shape_interp_type = nb;
            break;
         case 13: /* old algo type & ikin & dynamic info - OBSOLETE after 2.16*/
            DK_byteswap_fread( &nb, 2, 1, file );
            mdl->algotype  = (DK_AlgoType) nb;

            switch ( mdl->algotype )
            {
               case DK_AIKIN:
                  mdl->ikvar = DK_invKinReadBinaryChunk( file );
                  break;
               case DK_ADYNA:
                  DK_odynamicReadBinaryChunk( file, mdl );
		  mdl->algotype = DK_AIKIN;
		  break;
	       case DK_ADYNMDL:
                  DK_odynamicReadBinaryChunk( file, mdl );
		  mdl->algotype = DK_ASTD;
                  break;

               default:
                  DK_fileSkipField( file, 0xffff );

            }
	    break;

         case 14: /* physical properties (dynamic matter) info */
            DK_byteswap_fread( &nb, 2, 1, file );
            if ( nb != 0 )
            {
               DK_byteswap_fread( &external, 2, 1, file );       /* extern */
               mdl->dmatter = DK_dynamatterReadBinaryChunk( file, prot );
            }
            break;

         case 15: /* old envelope info - OBSOLETE after 2.16 */
            DK_byteswap_fread( &nb, 2, 1, file );
            mdl->envtype = (DK_EnvelopeType)nb;

            DK_byteswap_fread( &nb, 2, 1, file );
            if ( nb != 0 )
               mdl->envdata = DK_oEnvelopeReadBinaryChunk( file );
            break;

	 case 16: /* algotype (v2.2 and up) */
            DK_byteswap_fread( &nb, 2, 1, file );
            mdl->algotype = (DK_AlgoType)nb;
	    break;

	 case 17: /* envelope info (v2.2 and up) */
            DK_byteswap_fread( &nb, 2, 1, file );
            mdl->envtype = (DK_EnvelopeType)nb;

            DK_byteswap_fread( &nb, 2, 1, file );
            if ( nb != 0 )
               mdl->envdata = DK_envelopReadBinaryChunk( file, mdl, prot );
	    break;

	 case 18: /* dynamic  DK_options info (for space allocation only) */
            DK_byteswap_fread( &nb, 2, 1, file );
            if ( nb != 0 )
	        mdl->dchnvar = DK_dynaChnVarAllocate();
	    break;

	 case 19: /* joint dynamic properties */
            DK_byteswap_fread( &nb, 2, 1, file );
            if ( nb != 0 )
            {
	       DK_dynajntpropReadBinaryChunk( file, mdl, prot );
	    }
	    break;

	 case 20: /* inverse kinematic data   */
            DK_byteswap_fread( &nb, 2, 1, file );
            if ( nb != 0 )
            {
	       mdl->ikvar = DK_invKinReadBinaryChunk( file );
	    }
	    break;
	 case 21: /* cluster data */
            DK_byteswap_fread( &nb, 2, 1, file );
            if ( nb != 0 )
            {
	       for ( i=0; i<nb; i++ )
		  DK_clustersReadBinaryChunk( file, mdl, prot );
	    }
	    break;

	 case 22: /* user data */
            DK_byteswap_fread( &nb, 2, 1, file );
            for( i = 0; i < nb; i++ )
            {
               dataPtr = _DK_CALLOC( DK_UserNode, 1, -1 );
               DK_byteswap_fread(  dataPtr->type, 4, 1, file );
               DK_byteswap_fread( &dataPtr->size, 4, 1, file );
               dataPtr->data = _DK_MALLOC( char, dataPtr->size, -1 );
               DK_byteswap_fread(  dataPtr->data, 1, dataPtr->size, file );

               if( tail != NULL )
                  tail->next = dataPtr;
               else
                  mdl->userDataList = dataPtr;

               tail = dataPtr;
            }
	    break;
	 
         case 23: /* visibilityFlags */
            DK_byteswap_fread( &mdl->visibilityFlags, sizeof(short), 1, file);
            break;

         case 24: /* motionBlurFlags */
            DK_byteswap_fread( &mdl->motionBlurFlags, sizeof(short), 1, file);
            break;

	 case 25: /* rigid env. init. transfos. */
            DK_byteswap_fread( &nb, 2, 1, file );
            if ( nb != 0 )
            {
	       mdl->init_srt = _DK_CALLOC( DK_SRT_Vect, 1, -1 );
	       DK_Vector_fread( &mdl->init_srt->scl, sizeof( DK_Vector ), 1, file );
	       DK_Vector_fread( &mdl->init_srt->rot, sizeof( DK_Vector ), 1, file );
	       DK_Vector_fread( &mdl->init_srt->trs, sizeof( DK_Vector ), 1, file );
	    }
	    break;


         default:
            DK_fileSkipField( file, tok );

      }

      if( (prot) && (prot->model != NULL) )
         tok = prot->model[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   return( mdl );
}

#ifndef _WIN32
static DK_Model *DK_modelReadAsciiNode(  FILE *file, DK_Model *parent )
#else /* Win32 version is global */
DK_Model *DK_modelReadAsciiNode(  FILE *file, DK_Model *parent )
#endif
{
   char		 str[ 80 ];
   DK_Model	*mdl,
		*child = NULL;
   DK_Material	*mtrl = NULL;
   DK_Texture	*txt = NULL;
   DK_Texture3D	*txt3D = NULL;
   int		 readstr = 1;
   DK_UserNode	*tail = NULL;
   

   if( (mdl = DK_modelAllocate()) == NULL )
      return( NULL );

   mdl->parent = parent;

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( mdl );

   do
   {
      if ( readstr )
	 fscanf( file, "%s", str );
      readstr = 1;
      if( strcmp( "name", str ) == 0 )
         mdl->name = DK_stringReadAscii( file );
      else if( strcmp( "scaling", str ) == 0 )
         fscanf( file, "%f %f %f", &mdl->scaling.x, &mdl->scaling.y,
                                                    &mdl->scaling.z );
      else if( strcmp( "rotation", str ) == 0 )
      {
         fscanf( file, "%f %f %f", &mdl->rotation.x, &mdl->rotation.y,
                                                     &mdl->rotation.z );
         mdl->rotation.x = _DK_Deg2Rad( mdl->rotation.x );
         mdl->rotation.y = _DK_Deg2Rad( mdl->rotation.y );
         mdl->rotation.z = _DK_Deg2Rad( mdl->rotation.z );
      }
      else if( strcmp( "translation", str ) == 0 )
         fscanf( file, "%f %f %f", &mdl->translation.x, &mdl->translation.y,
                                                        &mdl->translation.z);
      else if( strcmp( "mesh", str ) == 0 )
      {
         mdl->type = DK_MDL_MESH;
         mdl->definition = (DK_Pointer) DK_meshReadAscii( file );
      }
      else if( strcmp( "patch", str ) == 0 )
      {
         mdl->type = DK_MDL_PTCH;
         mdl->definition = (DK_Pointer) DK_patchReadAscii( file );
      }
      else if( strcmp( "face", str ) == 0 )
      {
         mdl->type = DK_MDL_FACE;
         mdl->definition = (DK_Pointer) DK_faceReadAscii( file );
      }
      else if( strcmp( "NURBS_face:", str ) == 0 )
      {
         mdl->type = DK_MDL_FACE;
         mdl->definition = (DK_Pointer) DK_nurbsFaceReadAscii( file );
      }
      else if( strcmp( "spline", str ) == 0 )
      {
         mdl->type = DK_MDL_SPLN;
         mdl->definition = (DK_Pointer) DK_splineReadAscii( file, NULL, NULL );
      }
      else if( strcmp( "metaball", str ) == 0 )
      {
         mdl->type = DK_MDL_BALL;
         mdl->definition = (DK_Pointer) DK_metaballReadAscii( file );
      }
      else if( strcmp( "metasystem", str ) == 0 )
      {
         mdl->type = DK_MDL_META;
         mdl->definition = (DK_Pointer) DK_metaSystemReadAscii( file );
      }
      else if(strcmp( "Nurbs_Curve", str ) == 0 )
      {
	  mdl->type = DK_MDL_CRV;
	   mdl->definition = (DK_Pointer) DK_nurbsCrvAsciiRead( file);
      }
      else if(strcmp( "Nurbs_Surface", str ) == 0 )
      {
	  mdl->type = DK_MDL_SRF;
	   mdl->definition = (DK_Pointer) DK_nurbsSrfAsciiRead( file);
      }
      else if ( strcmp( DK_CLUSTER_TOKEN, str ) == 0 )
      {
	 DK_clustersReadAscii( file, mdl );
      }
      else if( strcmp( "material", str ) == 0 )
      {
         if( mtrl == NULL )
            mtrl = mdl->materials = DK_materialReadAscii( file );
         else
         {
            mtrl->next = DK_materialReadAscii( file );
            mtrl = mtrl->next;
         }
         mdl->nbMaterials++;
      }
      else if( strcmp( "texture", str ) == 0 )
      {
         if( txt == NULL )
            txt = mdl->textures = DK_textureReadAscii( file );
         else
         {
            txt->next = DK_textureReadAscii( file );
            txt = txt->next;
         }
         mdl->nbTextures++;
      }
      else if( strcmp( "texture3D", str ) == 0 )
      {
         if( txt3D == NULL )
            txt3D = mdl->textures3D = DK_texture3DReadAscii( file );
         else
         {
            txt3D->next = DK_texture3DReadAscii( file );
            txt3D = txt3D->next;
         }
         mdl->nbTextures3D++;
      }
      else if( strcmp( "actor_data", str ) == 0 )
      {
	 DK_modelActorDataReadAscii( file, mdl, str );
	 readstr = FALSE;
      }
      else if( strcmp( "init_scl", str ) == 0 )
      {
	 if ( mdl->init_srt == NULL )
	    mdl->init_srt = _DK_CALLOC( DK_SRT_Vect, 1, -1 );
         fscanf( file, "%f %f %f", &mdl->init_srt->scl.x,
				   &mdl->init_srt->scl.y,
				   &mdl->init_srt->scl.z );
      }
      else if( strcmp( "init_rot", str ) == 0 )
      {
	 if ( mdl->init_srt == NULL )
	    mdl->init_srt = _DK_CALLOC( DK_SRT_Vect, 1, -1 );
         fscanf( file, "%f %f %f", &mdl->init_srt->rot.x,
				   &mdl->init_srt->rot.y,
				   &mdl->init_srt->rot.z );
      }
      else if( strcmp( "init_trs", str ) == 0 )
      {
	 if ( mdl->init_srt == NULL )
	    mdl->init_srt = _DK_CALLOC( DK_SRT_Vect, 1, -1 );
         fscanf( file, "%f %f %f", &mdl->init_srt->trs.x,
				   &mdl->init_srt->trs.y,
				   &mdl->init_srt->trs.z );
      }
      else if( strcmp( "userData", str ) == 0 )
      {
	 DK_UserNode	*node;

         node = DK_userDataReadAscii( file );
         if( tail != NULL )
            tail->next = node;
         else
            mdl->userDataList = node;

         tail = node;
      }
      else if( strcmp( "model", str ) == 0 )
      {
         if( child == NULL )
            child = mdl->child = DK_modelReadAsciiNode( file, NULL );
         else
         {
            child->sibling = DK_modelReadAsciiNode( file, NULL );
            child = child->sibling;
         }
      }
   } while( strcmp( "}", str ) != 0 );

   return( mdl );
}

/*******************************************************************************

   $$L DK_shapefcvWriteAsciiAnimation

   This function writes an object's shape

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_shapefcvWriteAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Fcurve *fcv,		/* shape fcurve pointer */
        DK_Fcurve **wfcv,		/* shape weights fcurve pointer */
	char *name,		/* shape file name */
	int nbwshp		/* number of shape weights */
    )
{
    int i;

    DK_incIndent(1);
    if (name != NULL) 
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN(DK_A_SHAPE_NAME_TOKEN);
        _DK_OUT_STRING(name);
    }

    if (fcv != NULL) 
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN(DK_A_SHAPE_FCV_TOKEN);
        DK_fcurveWriteAsciiAnimation(file, fcv);
    }

    if (nbwshp > 1 && wfcv != NULL)
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN(DK_A_SHAPE_WEIGHT_TOKEN);
	_DK_OUT_SVALUE(nbwshp);
	_DK_OUT_INDENT();
	for ( i = 0; i < nbwshp; i++)
            DK_fcurveWriteAsciiAnimation(file, wfcv[i]);
    }
    DK_incIndent(-1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_shapefcvReadAsciiAnimation

   This function reads an object's shape

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_shapefcvReadAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Fcurve **fcv,		/* shape fcurve pointer */
	DK_Fcurve ***wfcv,		/* shape fcurve pointer */
	char **name,		/* shape file name */
	int *nbwshp		/* number of shape weights */
    )
{
    char *s = NULL;
    int i;
    DK_Fcurve *tfcv;

    DK_in_animToken(file);
    while (TRUE)
    {
        if (strcmp( DK_A_SHAPE_NAME_TOKEN, DK_animToken) == 0)
	{
	    s = DK_stringReadAsciiAnimation(file);
            DK_in_animToken(file);
    
	    if (s != NULL )
	    {
	        if (name == NULL) 
	        {
		    _DK_FREE(s, -1);
		}
		else 
		    *name = s;
	    }
	}
        else if (strcmp( DK_A_SHAPE_FCV_TOKEN, DK_animToken) == 0)
	{
            DK_in_animToken(file);
	    DK_fcurveReadAsciiAnimation(file, fcv);
        }

        else if (strcmp(DK_A_SHAPE_WEIGHT_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animSValue_na(file);
	    *nbwshp = (int) DK_animSValue;
	    if (*wfcv != NULL)
		_DK_FREE(*wfcv, -1);
	    *wfcv = _DK_MALLOC(DK_Fcurve *, sizeof(DK_Fcurve *) * *nbwshp, -1);
	    DK_in_animToken(file);
	    for (i = 0; i < *nbwshp; i++)
	    {
	        tfcv = NULL;
	        DK_fcurveReadAsciiAnimation(file, &tfcv);
	        if (*wfcv != NULL)
	            (*wfcv)[i] = tfcv;
	    }
        }
	else 
	{
	    return(TRUE);
	}
    }
}

/*******************************************************************************

   $$L DK_modelNodeWriteAsciiAnimation

   This function writes a model.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_modelNodeWriteAsciiAnimation
    (
	 FILE *file,  	/* file io pointer */
	DK_Model *mdl	/* model pointer */
    )
{
   DK_Texture *txt;
   DK_Material *mat;
   DK_Spl_fitting *sf;
   DK_Ptch_fitting *pf;
   int i;
   DK_DynaConstrInfo *di;

   if (mdl == NULL)
       return(FALSE);

    DK_incIndent(1);

    _DK_OUT_INDENT();
    _DK_OUT_STRING(mdl->name);

    if( !DK_WRITE_ANIMATION )
    {
       _DK_OUT_INDENT();
       _DK_OUT_TOKEN( DK_A_MDL_BBOX_TOKEN);
       _DK_OUT_TOKEN(DK_A_INACTIVE_TOKEN);

       _DK_OUT_INDENT();
       _DK_OUT_TOKEN(DK_A_MDL_WIRE_COLOUR_TOKEN);
       _DK_OUT_SVALUE( mdl->wire_colour);

       _DK_OUT_INDENT();
       _DK_OUT_TOKEN(DK_A_MDL_LINE_TYPE_TOKEN);
       _DK_OUT_SVALUE( mdl->line_type);

       _DK_OUT_INDENT();
       _DK_OUT_TOKEN(DK_A_MDL_SELECTED_TOKEN);
       switch (mdl->selected)
       {
           case  DK_UNSELECTED:
               _DK_OUT_TOKEN(DK_A_UNSELECTED_TOKEN);
               break;
           case  DK_NODE_SELECTED:
               _DK_OUT_TOKEN(DK_A_NODE_SELECTED_TOKEN);
               break;
           case  DK_BRANCH_SELECTED:
               _DK_OUT_TOKEN(DK_A_BRANCH_SELECTED_TOKEN);
               break;
           case  DK_INHERIT_SELECTED:
               _DK_OUT_TOKEN(DK_A_INHERIT_SELECTED_TOKEN);
               break;
       }

       DK_incIndent(-1);
       return(TRUE);

    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_MDL_SCL_TOKEN);
    DK_value3WriteAsciiAnimation(file, mdl->scaling.x, mdl->scaling.y, mdl->scaling.z,
	mdl->fcv[DK_FCV_MDL_SCLX], mdl->fcv[DK_FCV_MDL_SCLY], mdl->fcv[DK_FCV_MDL_SCLZ]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_MDL_ROT_TOKEN);
    DK_value3WriteAsciiAnimation(file, mdl->rotation.x, mdl->rotation.y, mdl->rotation.z,
	mdl->fcv[DK_FCV_MDL_ROTX], mdl->fcv[DK_FCV_MDL_ROTY], mdl->fcv[DK_FCV_MDL_ROTZ]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_MDL_TRANS_TOKEN);
    DK_positionWriteAsciiAnimation(file, &mdl->translation,
	mdl->fcv[DK_FCV_MDL_POS], mdl->path,
	mdl->pos_constraint_name, mdl->pos_constraint_type,
	mdl->fcv[DK_FCV_MDL_EPOSX], mdl->fcv[DK_FCV_MDL_EPOSY],
	mdl->fcv[DK_FCV_MDL_EPOSZ],
	mdl->path_name);

    if (mdl->nbnodelat > 0) 
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_MDL_NLAT_TOKEN);
	DK_latticeWriteAsciiAnimation(file, mdl->nbnodelat, mdl->nodelat, 
		mdl->fcv[DK_FCV_MDL_NLAT]);
    }

    if (mdl->nbtreelat > 0) 
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_MDL_TLAT_TOKEN);
	DK_latticeWriteAsciiAnimation(file, mdl->nbtreelat, mdl->treelat,
		mdl->fcv[DK_FCV_MDL_TLAT]);
    }

    if (mdl->materials != NULL) 
    {
	i = 0;
	mat = mdl->materials;
	while (mat != NULL)
	{
            _DK_OUT_INDENT();
            _DK_OUT_TOKEN(DK_A_MDL_MATERIAL_INDEX_TOKEN);
	    _DK_OUT_SVALUE(i);

            _DK_OUT_INDENT();
            _DK_OUT_TOKEN(DK_A_MDL_MAT_TOKEN);
	    DK_materialWriteAsciiAnimation(file, mat);

            if (mat->textures != NULL) 
	    {
	        for (txt = mat->textures; txt; txt = txt->next) 
	        {
                    _DK_OUT_INDENT();
		    _DK_OUT_TOKEN( DK_A_MDL_LTXT_TOKEN);
		    DK_textureWriteAsciiAnimation(file, txt);
	        }
	    }

	    if (mat->textures3D != NULL) 
	    {
                _DK_OUT_INDENT();
	        _DK_OUT_TOKEN( DK_A_MDL_LTXT3D_TOKEN);
	        DK_texture3DWriteAsciiAnimation(file, mat->textures3D);
	    }
	    mat = mat->next;
	    i++;
        }
    }

    if (mdl->textures != NULL) 
    {
	for (txt = mdl->textures; txt; txt = txt->next) 
	{
	    _DK_OUT_INDENT();
	    _DK_OUT_TOKEN( DK_A_MDL_TXT_TOKEN);
	    DK_textureWriteAsciiAnimation(file, txt);
	}
    }
    if (mdl->textures3D != NULL) 
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN( DK_A_MDL_TXT3D_TOKEN);
	DK_texture3DWriteAsciiAnimation(file, mdl->textures3D);
    }

    if (mdl->nbshape > 1)
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_MDL_SHAPE_TOKEN);
        DK_shapefcvWriteAsciiAnimation(file, mdl->fcv[DK_FCV_MDL_SHAPE], mdl->wfcv, 
		mdl->shape_filename, mdl->nbshape);
    }

    /* write clusters */
    if (mdl->cluster != NULL) 
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_MDL_CLUSTER_TOKEN);
        DK_clusterWriteAsciiAnimation(file, mdl->cluster, mdl->klusters);

	/* write cluster file */
	if (mdl->cluster->filename != NULL)
	    DK_clusterWriteAsciiFile(mdl->cluster->filename, mdl);
    }

    if ((sf = mdl->nodefit) != NULL) 
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_MDL_NFIT_TOKEN);
	DK_splineFitWriteAsciiAnimation(file, sf);
    }

    if ((sf = mdl->brchfit) != NULL) 
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_MDL_TFIT_TOKEN);
	DK_splineFitWriteAsciiAnimation(file, sf);
    }

    if ((pf = mdl->nodepfit) != NULL) 
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_MDL_PNFIT_TOKEN);
	DK_patchFitWriteAsciiAnimation(file, pf);
    }

    if ((pf = mdl->brchpfit) != NULL) 
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_MDL_PTFIT_TOKEN);
	DK_patchFitWriteAsciiAnimation(file, pf);
    }

    if (mdl->shape_interp_type != DK_NULL_SHAPE_INTERP)
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_MDL_SHAPE_INTERP_TOKEN);
        switch (mdl->shape_interp_type)
        {
            case DK_LINEAR_SHAPE_INTERP:
	        _DK_OUT_TOKEN(DK_A_LINEAR_SHAPE_INTERP_TOKEN);
	        break;
            case DK_CARDINAL_SHAPE_INTERP:
	        _DK_OUT_TOKEN(DK_A_CARDINAL_SHAPE_INTERP_TOKEN);
	        break;
            case DK_WEIGHT_SHAPE_INTERP:
	        _DK_OUT_TOKEN(DK_A_WEIGHT_SHAPE_INTERP_TOKEN);
	        break;
        }
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_MDL_BBOX_TOKEN);
    if (mdl->bbox == TRUE)
    {
        _DK_OUT_TOKEN(DK_A_ACTIVE_TOKEN);
        DK_vectorWriteAsciiAnimation(file, &mdl->bbox_min);
        DK_vectorWriteAsciiAnimation(file, &mdl->bbox_max);
    }
    else
        _DK_OUT_TOKEN(DK_A_INACTIVE_TOKEN);

   /* write dynamic chain animation info  */
   if ( mdl->dchnvar != NULL)
   {
      _DK_OUT_INDENT();
      _DK_OUT_TOKEN( DK_A_MDL_DYNASETUP_TOKEN );
      DK_dynachnvarWriteAsciiChunk( file, mdl->dchnvar);
   }

   /* write dynamic constraints */
   di = mdl->dcinfo;
   while ( di != NULL )
   {
      _DK_OUT_INDENT();
      _DK_OUT_TOKEN( DK_A_MDL_DYNACONST_TOKEN );
      DK_dynaConstraintWriteAsciiChunk( file, di );
      di = di->next;
   }

    if (mdl->fcv[DK_FCV_MDL_HIDDEN] != NULL)
    {
	/* the value of 0.0 is a dummy value and would not be written out */
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_MDL_HID_TOKEN);
        DK_valueWriteAsciiAnimation(file, 0.0, mdl->fcv[DK_FCV_MDL_HIDDEN]);
    }

    if ( mdl->fcv[DK_FCV_MDL_ALGO] != NULL )
    {
       _DK_OUT_INDENT();
       _DK_OUT_TOKEN(DK_A_MDL_ALGO_TOKEN);
       DK_valueWriteAsciiAnimation(file, 0.0, mdl->fcv[DK_FCV_MDL_ALGO]);
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_MDL_WIRE_COLOUR_TOKEN);
    _DK_OUT_SVALUE( mdl->wire_colour);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_MDL_LINE_TYPE_TOKEN);
    _DK_OUT_SVALUE( mdl->line_type);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_MDL_SELECTED_TOKEN);
    switch (mdl->selected)
    {
	case  DK_UNSELECTED:
	    _DK_OUT_TOKEN(DK_A_UNSELECTED_TOKEN);
	    break;
	case  DK_NODE_SELECTED:
	    _DK_OUT_TOKEN(DK_A_NODE_SELECTED_TOKEN);
	    break;
	case  DK_BRANCH_SELECTED:
	    _DK_OUT_TOKEN(DK_A_BRANCH_SELECTED_TOKEN);
	    break;
	case  DK_INHERIT_SELECTED:
	    _DK_OUT_TOKEN(DK_A_INHERIT_SELECTED_TOKEN);
	    break;
    }

    DK_incIndent(-1);
    return(TRUE);
}

/*******************************************************************************

   $$L DK_modelReadAsciiAnimation

   This function reads a model.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_modelReadAsciiAnimation
    (
	 FILE *file,	/* file io pointer */
	DK_Model *root	/* model pointer */
    )
{
    DK_Texture3D *txt3D, *txt3D2;
    DK_Material *mat, *last_mat;
    char *name;
    int first = TRUE;
    int i, j;
    DK_Texture *txt, *txt2;
    DK_Model *mdl;

    name = DK_stringReadAsciiAnimation(file);

    if ((mdl = DK_modelNodeFindByName(root, name)) == NULL)
    {
	if ( DK_option.verbose == TRUE)
	    fprintf( DK_option.msgFd, "DK_ModelReadAsciiAnimation: \
error, cannot find model '%s'.\n", name);
	mdl = root;
    }
    _DK_FREE(name, -1);

    DK_in_animToken(file);
    while (TRUE)
    {
        if (strcmp(DK_A_MDL_SCL_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL) 
    	    {
                DK_value3ReadAsciiAnimation(file, &mdl->scaling.x, &mdl->scaling.y,
			&mdl->scaling.z, &mdl->fcv[DK_FCV_MDL_SCLX], 
			&mdl->fcv[DK_FCV_MDL_SCLY], &mdl->fcv[DK_FCV_MDL_SCLZ]);
            }
            else
                DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
        }
        else if (strcmp(DK_A_MDL_ROT_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL) 
    	    {
                DK_value3ReadAsciiAnimation(file, &mdl->rotation.x, &mdl->rotation.y,
			&mdl->rotation.z, &mdl->fcv[DK_FCV_MDL_ROTX],
			&mdl->fcv[DK_FCV_MDL_ROTY], &mdl->fcv[DK_FCV_MDL_ROTZ]);
            }
            else
                DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
        }
        else if (strcmp(DK_A_MDL_TRANS_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL) 
    	    {
                DK_positionReadAsciiAnimation(file, &mdl->translation,
		    &mdl->fcv[DK_FCV_MDL_POS], &mdl->path, 
		    &mdl->pos_constraint_name,
		    &mdl->pos_constraint_type,
		    &mdl->fcv[DK_FCV_MDL_EPOSX], &mdl->fcv[DK_FCV_MDL_EPOSY],
		    &mdl->fcv[DK_FCV_MDL_EPOSZ],
		    &mdl->path_name);
            }
            else
                DK_positionReadAsciiAnimation(file, NULL, NULL, NULL,
    	            NULL, NULL, NULL, NULL, NULL, NULL);
        }
        else if (strcmp(DK_A_MDL_NLAT_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animToken(file);
            if (mdl != NULL) 
                DK_latticeReadAsciiAnimation(file, FALSE, &mdl->nbnodelat, &mdl->nodelat, 
			&mdl->fcv[DK_FCV_MDL_NLAT]);
            else
                DK_latticeReadAsciiAnimation(file, FALSE, NULL, NULL, NULL);
        }
        else if (strcmp(DK_A_MDL_TLAT_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animToken(file);
            if (mdl != NULL) 
                DK_latticeReadAsciiAnimation(file, TRUE, &mdl->nbtreelat, &mdl->treelat, 
			&mdl->fcv[DK_FCV_MDL_TLAT]);
            else
                DK_latticeReadAsciiAnimation(file, TRUE, NULL, NULL, NULL);
        }
        else if (strcmp(DK_A_MDL_MATERIAL_INDEX_TOKEN, DK_animToken) == 0)
        {
            DK_in_animSValue_na(file);
            if (mdl != NULL) 
		mdl->currentMaterial = DK_animSValue;
    	    DK_in_animToken(file);
        }
        else if (strcmp(DK_A_MDL_MAT_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL)
    	    {
		/* find DK_material pointer */
		last_mat = mat = mdl->materials;
		i = mdl->currentMaterial;
		while (i > 0 && mat != NULL)
		{
		    last_mat = mat;
		    mat = mat->next;
		    i--;
		}
		/* if not found then allocate DK_material structure */
		if (mat == NULL)
		{
		    for (j = 0; j <= i; j++)
		    {
		        mat = DK_materialAllocate();
		        mat->next = NULL;
		        if (mdl->materials == NULL)
			    mdl->materials = mat;
			else
		            last_mat->next = mat;
		        last_mat = mat;
		    }
		}
                DK_materialReadAsciiAnimation(file, mat);
            }
            else
                DK_materialReadAsciiAnimation(file, NULL);
        }
        else if (strcmp(DK_A_MDL_LTXT3D_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL && mdl->materials != NULL) 
    	    {
		txt3D = NULL;
		i = mdl->currentMaterial;
		mat = mdl->materials;
		while (i > 0 && mat != NULL)
		    mat = mat->next;
                if (mat != NULL)
		{
		    txt3D = DK_texture3DAllocate();
		    if (txt3D != NULL)
		    {
			txt3D->next = NULL;
			if (mat->textures3D == NULL)
			    /* first one */
			    mat->textures3D = txt3D;
			else
			{
			    /* add to the end of the list */
			    txt3D2 = mat->textures3D;
			    while (txt3D2 != NULL && txt3D2->next != NULL)
				txt3D2 = txt3D2->next;
			    txt3D2->next = txt3D;
			}
		    }
		}
                DK_texture3DReadAsciiAnimation(file, txt3D);
            }
            else
                DK_texture3DReadAsciiAnimation(file, NULL);
        }
        else if (strcmp(DK_A_MDL_LTXT_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL && mdl->materials != NULL) 
    	    {
		txt = NULL;
		i = mdl->currentMaterial;
		mat = mdl->materials;
		while (i > 0 && mat != NULL)
		    mat = mat->next;
    	        if (mat != NULL)
		{
		    txt = DK_textureAllocate();
		    txt->next = NULL;
		    if (mat->textures == NULL)
			/* first txt */
			mat->textures = txt;
		    else
		    {
			/* add texture to the end */
			txt2 = mat->textures;
			while (txt2 != NULL && txt2->next != NULL)
			    txt2 = txt2->next;
			txt2->next = txt;
		    }
		}
                DK_textureReadAsciiAnimation(file, txt);
            }
            else
                DK_textureReadAsciiAnimation(file, NULL);
        }
        else if (strcmp(DK_A_MDL_TXT3D_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL) 
	    {
		txt3D = DK_texture3DAllocate();
		if (txt3D != NULL)
		{
		    txt3D->next = NULL;
		    if (mdl->textures3D == NULL)
			/* first one */
			mdl->textures3D = txt3D;
		    else
		    {
			/* add to the end of the list */
			txt3D2 = mdl->textures3D;
			while (txt3D2 != NULL && txt3D2->next != NULL)
			    txt3D2 = txt3D2->next;
			txt3D2->next = txt3D;
		    }
		}
                DK_texture3DReadAsciiAnimation(file, txt3D);
	    }
            else
                DK_texture3DReadAsciiAnimation(file, NULL);
        }
        else if (strcmp(DK_A_MDL_TXT_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL) 
    	    {
		txt = DK_textureAllocate();
		txt->next = NULL;
		if (mdl->textures == NULL)
		    /* first txt */
		    mdl->textures = txt;
		else
		{
		    /* add texture to the end */
		    txt2 = mdl->textures;
		    while (txt2 != NULL && txt2->next != NULL)
			txt2 = txt2->next;
		    txt2->next = txt;
		}
                DK_textureReadAsciiAnimation(file, txt);
            }
            else
                DK_textureReadAsciiAnimation(file, NULL);
        }
        else if (strcmp(DK_A_MDL_SHAPE_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL)
            {
                DK_shapefcvReadAsciiAnimation(file, &mdl->fcv[DK_FCV_MDL_SHAPE], 
		    &mdl->wfcv, &mdl->shape_filename, &mdl->nbshape);
                DK_modelReadShapeBinaryFile( mdl );
            }
            else
                DK_shapefcvReadAsciiAnimation(file, NULL, NULL, NULL, NULL);
        }
        else if (strcmp(DK_A_MDL_CLUSTER_TOKEN, DK_animToken) == 0)
        {
            DK_clusterReadAsciiAnimation(file, mdl);
        }
        else if (strcmp(DK_A_MDL_NFIT_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animToken(file);
            if (mdl != NULL) 
    	        DK_splineFitReadAsciiAnimation(file, &mdl->nodefit);
            else
    	        DK_splineFitReadAsciiAnimation(file, NULL);
        }
        else if (strcmp(DK_A_MDL_TFIT_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animToken(file);
            if (mdl != NULL) 
    	        DK_splineFitReadAsciiAnimation(file, &mdl->brchfit);
            else
    	        DK_splineFitReadAsciiAnimation(file, NULL);
        }
        else if (strcmp(DK_A_MDL_PNFIT_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animToken(file);
            if (mdl != NULL) 
    	        DK_patchFitReadAsciiAnimation(file, &mdl->nodepfit);
            else
    	        DK_patchFitReadAsciiAnimation(file, NULL);
        }
        else if (strcmp(DK_A_MDL_PTFIT_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animToken(file);
            if (mdl != NULL) 
    	        DK_patchFitReadAsciiAnimation(file, &mdl->brchpfit);
            else
    	        DK_patchFitReadAsciiAnimation(file, NULL);
        }
        else if (strcmp(DK_A_MDL_SHAPE_INTERP_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animToken(file);
            if (mdl != NULL)
	    {
	        if (strcmp(DK_A_LINEAR_SHAPE_INTERP_TOKEN, DK_animToken) == 0)
		    mdl->shape_interp_type = DK_LINEAR_SHAPE_INTERP;
	        else if (strcmp(DK_A_CARDINAL_SHAPE_INTERP_TOKEN, DK_animToken) == 0)
		    mdl->shape_interp_type = DK_CARDINAL_SHAPE_INTERP;
	        else if (strcmp(DK_A_WEIGHT_SHAPE_INTERP_TOKEN, DK_animToken) == 0)
		    mdl->shape_interp_type = DK_WEIGHT_SHAPE_INTERP;
	        else if ( DK_option.verbose == TRUE)
		    fprintf( DK_option.msgFd, "DK_ModelReadAsciiAnimation: \
error, unknown shape interp type '%s'.\n", DK_animToken);
	    
	    }
    	    DK_in_animToken(file);
        }
        else if (strcmp(DK_A_MDL_BBOX_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animToken(file);
            if (mdl != NULL)
	    {
	        if (strcmp(DK_A_ACTIVE_TOKEN, DK_animToken) == 0)
		{
		    mdl->bbox = TRUE;
                    DK_vectorReadAsciiAnimation(file, &(mdl->bbox_min));
                    DK_vectorReadAsciiAnimation(file, &(mdl->bbox_max));
		}
	        else if (strcmp(DK_A_INACTIVE_TOKEN, DK_animToken) == 0)
		    mdl->bbox = FALSE;
	        else if ( DK_option.verbose == TRUE)
	            fprintf( DK_option.msgFd, "DK_ModelReadAsciiAnimation: \
error, unknown mdl bbox DK_animToken '%s'.\n",DK_animToken);
	    }
	    else if (strcmp(DK_A_ACTIVE_TOKEN, DK_animToken) == 0)
	    {
                DK_vectorReadAsciiAnimation(file, NULL);
                DK_vectorReadAsciiAnimation(file, NULL);
	    }
    	    DK_in_animToken(file);
	}
        else if (strcmp(DK_A_MDL_DYNASETUP_TOKEN, DK_animToken) == 0)
	{
            DK_dynachnvarReadAsciiChunk( file, mdl );
	}
        else if (strcmp(DK_A_MDL_DYNACONST_TOKEN, DK_animToken) == 0)
	{
            DK_dynaConstraintReadAsciiChunk( file, mdl );
	}
        else if (strcmp(DK_A_MDL_HID_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL) 
                DK_valueReadAsciiAnimation(file, NULL, &mdl->fcv[DK_FCV_MDL_HIDDEN], TRUE);
            else
                DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
        }
        else if (strcmp(DK_A_MDL_ALGO_TOKEN, DK_animToken) == 0)
        {
            if (mdl != NULL) 
                DK_valueReadAsciiAnimation(file, NULL, &mdl->fcv[DK_FCV_MDL_ALGO], TRUE);
            else
                DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
        }
        else if (strcmp(DK_A_MDL_WIRE_COLOUR_TOKEN, DK_animToken) == 0)
        {
            DK_in_animSValue_na(file);
            if (mdl != NULL)
    	        mdl->wire_colour = DK_animSValue;
    	    DK_in_animToken(file);
        }
        else if (strcmp(DK_A_MDL_WIRE_COLOUR_INDEX_TOKEN, DK_animToken) == 0)
        {
            DK_in_animSValue_na(file);
            DK_in_animSValue_na(file);
	    /* forget the two indexes for the moment */
            if (mdl != NULL)
    	        mdl->wire_colour = DK_DEFAULT_WIRE_COLOUR;
    	    DK_in_animToken(file);
        }
        else if (strcmp(DK_A_MDL_COLLAPSE_TOKEN, DK_animToken) == 0)
        {
            DK_in_animSValue_na(file);
	    /* forget the value for the moment */
    	    DK_in_animToken(file);
        }
        else if (strcmp(DK_A_MDL_SCHEMPOS_TOKEN, DK_animToken) == 0)
        {
            DK_in_animFValue_na(file);
            DK_in_animFValue_na(file);
	    /* forget the values for the moment */
    	    DK_in_animToken(file);
        }
        else if (strcmp(DK_A_MDL_LINE_TYPE_TOKEN, DK_animToken) == 0)
        {
            DK_in_animSValue_na(file);
            if (mdl != NULL)
    	        mdl->line_type = DK_animSValue;
    	    DK_in_animToken(file);
        }
        else if (strcmp(DK_A_MDL_SELECTED_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animToken(file);
	    if (mdl != NULL) 
	    {
		if (strcmp(DK_A_UNSELECTED_TOKEN, DK_animToken) == 0)
		    mdl->selected =  DK_UNSELECTED;
		else if (strcmp(DK_A_NODE_SELECTED_TOKEN, DK_animToken) == 0)
		    mdl->selected =  DK_NODE_SELECTED;
		else if (strcmp(DK_A_BRANCH_SELECTED_TOKEN, DK_animToken) == 0)
		    mdl->selected =  DK_BRANCH_SELECTED;
		else if (strcmp(DK_A_INHERIT_SELECTED_TOKEN, DK_animToken) == 0)
		    mdl->selected =  DK_INHERIT_SELECTED;
	        else if ( DK_option.verbose == TRUE)
		    fprintf( DK_option.msgFd, "DK_ModelReadAsciiAnimation: \
error, unknown select animToken '%s'.\n", DK_animToken);
	    }
	    DK_in_animToken(file);
        }
        else
        {
	    if (first)
		DK_in_animToken(file);
            return(TRUE);
        }
	first = FALSE;
    }
}

/*******************************************************************************

   $$L DK_modelWriteAsciiAnimation

   This function writes a model hierarchy into the scene file.

   Returned Value: DK_Model *

*******************************************************************************/

static int DK_modelWriteAsciiAnimation
    (
	 FILE *file,
	DK_Model *mdl
    )
{
    int rtn = FALSE;

    /* write object */
    while (mdl != NULL)
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN(DK_A_OBJ_NODE_TOKEN);

	rtn |= DK_modelNodeWriteAsciiAnimation(file, mdl);
	mdl = DK_modelNextNode(mdl);
    }
    return(rtn);
}


/*******************************************************************************

   $$L DK_objectWriteAsciiAnimation

   This function writes an object hierarchy.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_objectWriteAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Model *mdl		/* root model pointer */
    )
{
    int rtn;
    char string[DK_MAXLINE];
    char *tmp;

    if (mdl->filename != NULL)
    {
	/* write out model file */
	DK_modelWriteFile( mdl->filename, mdl,  DK_FIL_BINARY);

	DK_incIndent(1);

	if ( DK_VERSION == DK_v2_5                         &&
	     (tmp = strrchr( mdl->filename, '.' )) != NULL &&
	     strcmp( tmp, ".hrc" ) == 0 )
	{
	   *tmp = '\0';
	}
	else
	   tmp = NULL;
	_DK_OUT_INDENT();
        _DK_OUT_STRING( mdl->filename );
	if ( tmp )
	   *tmp = '.';
    
	/* object substitution */
	if (mdl->sub_active == TRUE)
	{
	    _DK_OUT_INDENT();
	    _DK_OUT_TOKEN(DK_A_OBJ_SUBSTITUTION_TOKEN);
	    _DK_OUT_STRING(mdl->sub_model_name);
	    _DK_OUT_SVALUE( mdl->sub_start);
	    _DK_OUT_SVALUE( mdl->sub_end);
	    if (mdl->sub_inmemory == TRUE)
		_DK_OUT_TOKEN(DK_A_YES_TOKEN);
	    else
		_DK_OUT_TOKEN(DK_A_NO_TOKEN);

	    if (mdl->sub_pre_script_file != NULL)
	    {
	        _DK_OUT_INDENT();
	        _DK_OUT_TOKEN(DK_A_OBJ_SUBSTITUTION_PRESCRIPT_TOKEN);
	        _DK_OUT_STRING(mdl->sub_pre_script_file);
	    }
	    if (mdl->sub_post_script_file != NULL)
	    {
	        _DK_OUT_INDENT();
	        _DK_OUT_TOKEN(DK_A_OBJ_SUBSTITUTION_POSTSCRIPT_TOKEN);
	        _DK_OUT_STRING(mdl->sub_post_script_file);
	    }
	}

        /* write object */
	rtn = DK_modelWriteAsciiAnimation(file, mdl);
        DK_incIndent(-1);
    }
    else
    {
	/* write a null string */
	string[0] = '\0';
        _DK_OUT_STRING(string);
    }

    return(rtn);
}

/*******************************************************************************

   $$L DK_objectReadAsciiAnimation

   This function allocates and reads an object hierarchy.
   Also, the object is entered into the graflist.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_objectReadAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Model **mdl_head,	/* model list pointer */
	char	*object_path	/* object path, where to read object */
    )
{
    DK_SVALUE start, end;
    char *name = NULL, *filename = NULL, *prescript = NULL, *postscript = NULL;
    int inmemory, substitution = FALSE, valid = TRUE;
    DK_Model *mdl, *last_mdl;

    filename = DK_stringReadAsciiAnimation(file);

    if( filename[0] != '/' )
    {
       char	*fullFilename;
       int	len;

       len = strlen( filename ) + strlen( object_path ) + 2;
       fullFilename = ( char * )malloc( len );
       strcpy( fullFilename, object_path );
       strcat( fullFilename, "/" );
       strcat( fullFilename, filename );
       if( (mdl = DK_modelReadFile( fullFilename )) != NULL )
       {
          _DK_FREE( filename, -1 );
          filename = fullFilename;
       }
       else
       {
          _DK_FREE( fullFilename, -1 );
          mdl = DK_modelReadFile( filename );
       }
    }
    else
       mdl = DK_modelReadFile( filename );

    if (mdl == NULL)
    {
	if ( DK_option.verbose == TRUE)
	    fprintf( DK_option.msgFd, "ObjectReadAsciiAnimation: \
error, cannot read model file '%s'.\n", filename);
	mdl = DK_modelAllocate();
	valid = FALSE;
	_DK_FREE(filename, -1);
    }
    else
    {
        /* insert the model hierarchy in the end of the model list */
	last_mdl = *mdl_head;
	while (last_mdl != NULL && last_mdl->next != NULL)
	    last_mdl = last_mdl->next;
	if (*mdl_head == NULL)
	    *mdl_head = mdl;
	else
	    last_mdl->next = mdl;
	mdl->filename = filename;
    }

    /* read each node */
    DK_in_animToken(file);
    while (TRUE)
    {
	if (strcmp(DK_A_OBJ_NODE_TOKEN, DK_animToken) == 0)
	{
	    DK_modelReadAsciiAnimation(file, mdl);
	}
	else if (strcmp( DK_A_OBJ_SUBSTITUTION_TOKEN, DK_animToken) == 0)
	{
	    /* get substitution parameters but don't deal with it until
	       all the object's shapes are read in (if any) */
	    substitution = TRUE;
	    if (name != NULL)
		_DK_FREE(name, -1);
	    name = DK_stringReadAsciiAnimation(file);
	    DK_in_animSValue(file, &start);
	    DK_in_animSValue(file, &end);
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
		inmemory = TRUE;
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
		inmemory = FALSE;
	    else 
	    {
	        if ( DK_option.verbose == TRUE)
		    fprintf( DK_option.msgFd, "ObjectReadAsciiAnimation: \
error, unknown incore animToken '%s'.\n", DK_animToken);
		substitution = FALSE;
	    }
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_OBJ_SUBSTITUTION_PRESCRIPT_TOKEN, DK_animToken) == 0)
	{
	    if (prescript != NULL)
		_DK_FREE(prescript, -1);
	    prescript = DK_stringReadAsciiAnimation(file);
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_OBJ_SUBSTITUTION_POSTSCRIPT_TOKEN, DK_animToken) == 0)
	{
	    if (postscript != NULL)
		_DK_FREE(postscript, -1);
	    postscript = DK_stringReadAsciiAnimation(file);
	    DK_in_animToken(file);
	}
	else
	{
	    /* get substitution files */
	    if (substitution == TRUE && mdl != NULL)
	    {
		mdl->sub_active = TRUE;
		mdl->sub_start = (int) start;
		mdl->sub_end = (int) end;
		mdl->sub_inmemory = inmemory;
		mdl->sub_model_name = name;
		mdl->sub_pre_script_file = prescript;
		mdl->sub_post_script_file = postscript;
	    }
	    /* no substitution, so free name, prescript, and 
		postscript strings */
	    else if (name != NULL)
	    {
		_DK_FREE(name, -1);
		if (prescript != NULL)
		    _DK_FREE(prescript, -1);
		if (postscript != NULL)
		    _DK_FREE(postscript, -1);
	    }
	    if (valid == FALSE)
		DK_modelDispose(&mdl);

	    return(TRUE);
	}
    }
}

DK_Boolean DK_modelAttachUserData( DK_Model *mdl, DK_String type, int size, char *data )
{
   DK_UserNode *node, *tail;

   if( (node = _DK_CALLOC( DK_UserNode, 1, -1 )) == NULL )
      return( FALSE );

   strncpy ( node->type, type, 4 );
   node->size = size;
   node->data = data;

   if( mdl->userDataList != NULL )
   {
      for( tail = mdl->userDataList; tail->next != NULL; tail = tail->next )
         /* NO_OP */;
      tail->next = node;
   }
   else
      mdl->userDataList = node;

   return( TRUE );
}

DK_UserNode *DK_modelSearchUserData( DK_Model *mdl, DK_String type )
{
   DK_UserNode *node;

   for( node = mdl->userDataList; node != NULL; node = node->next )
      if( strncmp( node->type, type, 4 ) == 0 )
         return( node );

   return( NULL );
}

void DK_modelDisposeUserData( DK_Model *mdl, DK_String type )
{
   DK_UserNode *node, *prev;

   for( prev = NULL, node = mdl->userDataList; node != NULL;
					 prev = node, node = node->next )
   {
      if( strncmp( node->type, type, 4 ) == 0 )
      {
         if( prev != NULL )
            prev->next = node->next;
         else
            mdl->userDataList = node->next;

         _DK_FREE( node->data, -1 );
         _DK_FREE( node, -1 );
      }
#ifdef _WIN32
      /* This appears to be needed in the IRIX case also because DK_FREE
         sets node to NULL which is then dereferenced in the 'for'
	 statement. On Windows NT this produces an access violation. */
      if (node == NULL)
         break;
#endif
   }
}

int DK_fdmodelReadShapeBinaryFile( FILE *file, DK_Model *mdl )
{
   float         version;
   long          id, i;
   short	 nbShapes;
   DK_FileProt  *prot = NULL;
   DK_Boolean	 same;
   DK_Pointer	*shapes, bidon;

   id = DK_headerReadBinary( file, &version, NULL );
   if( id == 'PROT' )
   {
      prot = DK_prototypeRead( file );
      DK_byteswap_fread( &id, 4, 1, file );
   }

   DK_byteswap_fread( &nbShapes, 2, 1, file );

   if ( nbShapes < 0 )
   {

      if ( nbShapes == -2 )  /* nb == -2 : shpAnimMode + orgdef */
      {
         DK_byteswap_fread( &nbShapes, 2, 1, file ); /* dummy (shpAnimMode) */
      }

      /* For the moment, we don't want to use "orgdef"    */
      /* in the DKit, so we read it but it is never used. */
      switch ( mdl->type )
      {
	 case DK_MDL_MESH:
	    mdl->definition = (DK_Pointer) DK_meshReadBinary( file, prot );
	    break;
	 case DK_MDL_FACE:
	    mdl->definition = (DK_Pointer) DK_faceReadBinary( file, prot );
	    break;
	 case DK_MDL_PTCH:
	    mdl->definition = (DK_Pointer) DK_patchReadBinary( file, prot );
	    break;
	 case DK_MDL_SPLN:
	    mdl->definition = (DK_Pointer) DK_splineReadBinary( file, prot );
	    break;
	 case DK_MDL_CRV:
	    mdl->definition = (DK_Pointer) DK_nurbsCrvBinaryRead( file, prot );
	    break;
	 case DK_MDL_SRF:
	    mdl->definition = (DK_Pointer) DK_nurbsSrfBinaryRead( file, prot );
	    break;

         default:
            DK_fileSkipField( file, 0xffff );

      }

      DK_byteswap_fread( &nbShapes, 2, 1, file );
   }

   shapes = _DK_CALLOC( DK_Pointer, nbShapes, -1 );
   same = TRUE;

   switch( mdl->type )
   {
      case DK_MDL_MESH:
         if( id != 'MESH' )
            same = FALSE;
         else
            for( i = 0; i < nbShapes; i++ )
               shapes[i] = (DK_Pointer) DK_meshReadBinary( file, prot );
         break;
      case DK_MDL_FACE:
         if( id != 'FACE' )
            same = FALSE;
         else
            for( i = 0; i < nbShapes; i++ )
               shapes[i] = (DK_Pointer) DK_faceReadBinary( file, prot );
         break;
      case DK_MDL_PTCH:
         if( id != 'PTCH' )
            same = FALSE;
         else
            for( i = 0; i < nbShapes; i++ )
               shapes[i] = (DK_Pointer) DK_patchReadBinary( file, prot );
         break;
      case DK_MDL_SPLN:
         if( id != 'SPLN' )
            same = FALSE;
         else
            for( i = 0; i < nbShapes; i++ )
               shapes[i] = (DK_Pointer) DK_splineReadBinary( file, prot );
         break;
      case DK_MDL_CRV:
         if( id != 'CRV' )
            same = FALSE;
         else
            for( i = 0; i < nbShapes; i++ )
               shapes[i] = (DK_Pointer) DK_nurbsCrvBinaryRead( file, prot );
         break;
      case DK_MDL_SRF:
         if( id != 'SRF' )
            same = FALSE;
         else
            for( i = 0; i < nbShapes; i++ )
               shapes[i] = (DK_Pointer) DK_nurbsSrfBinaryRead( file, prot );
         break;
   }

   if( same )
   {
      mdl->shapes = shapes;
      mdl->nbshape = nbShapes;
   }
   else
      _DK_FREE( shapes, -1 );

   DK_prototypeDispose( &prot );
   return( TRUE );
}


int DK_modelReadShapeBinaryFile( DK_Model *mdl )
{
   char          buf[256];
   FILE         *file;

   sprintf( buf, "%s.shp", mdl->shape_filename );
   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( FALSE );

   DK_fdmodelReadShapeBinaryFile( file, mdl );

   fclose( file );

   return( TRUE );
}

