/***************************************************************************************************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyMetaball.c,v $
   $Revision: 1.9.14.1 $ $Date: 1995/06/09 22:00:10 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's metaball functions,
   its content is sufficient to read, write and manage mesh data structure
   into SOFTIMAGE's file format...

   Written by: Laurent Lauzon

   (c) Copyright 1992, 1993 SOFTIMAGE Inc.

***************************************************************************************************************************************************************/

#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinyMetaball.h"


/******************************************************************************/
/*** Static function prototypes ***********************************************/
/******************************************************************************/

static DK_MetaGroup *DK_metaGroupReadAscii( FILE *, DK_MetaSystem * );

/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

DK_Metaball *DK_metaballAllocate( void )
{
   DK_Metaball *ball;

   if( ball = (DK_Metaball *) calloc( 1, sizeof( DK_Metaball ) ) )
   {
      ball->centerMat[0][0] = 1.0;
      ball->centerMat[0][1] = 0.0;
      ball->centerMat[0][2] = 0.0;
      ball->centerMat[0][3] = 0.0;

      ball->centerMat[1][0] = 0.0;
      ball->centerMat[1][1] = 1.0;
      ball->centerMat[1][2] = 0.0;
      ball->centerMat[1][3] = 0.0;

      ball->centerMat[2][0] = 0.0;
      ball->centerMat[2][1] = 0.0;
      ball->centerMat[2][2] = 1.0;
      ball->centerMat[2][3] = 0.0;

      ball->centerMat[3][0] = 0.0;
      ball->centerMat[3][1] = 0.0;
      ball->centerMat[3][2] = 0.0;
      ball->centerMat[3][3] = 1.0;

      ball->weight = 2.0;
      ball->ratio  = 0.5;

      ball->exponent[0] = 2.0;
      ball->exponent[1] = 2.0;
      ball->exponent[2] = 2.0;
   }

   return( ball );
}

void DK_metaballDispose( 
   DK_Metaball **ball 
)
{
   if( *ball != NULL )
   {
      free( *ball );
      *ball = NULL;
   }
}

DK_Boolean DK_metaballWriteBinary( 
   FILE        *file, 
   DK_Metaball *ball, 
   DK_FileProt *prot 
)
{
   long	i;
   int	nb;

   for( i = 0; i < prot->nbMetaball; i++ )
   {
      switch( prot->metaball[ i ] )
      {
         case 0:	/* end of metaball definition */
            break;

         case 4:	/* ratio: derivated from weight */
            if( DK_byteswap_fwrite( &ball->ratio, 8, 1, file ) != 1 )
               return( FALSE );
            break;

         case 5:	/* weight */
            if( DK_byteswap_fwrite( &ball->weight, 8, 1, file ) != 1 )
               return( FALSE );
            break;

         case 6:	/* exponent */
            if( DK_3doubles_fwrite( ball->exponent, 24, 1, file ) != 1 )
               return( FALSE );
            break;

         case 7:	/* centerMat */
            if( DK_Matrix_fwrite( ball->centerMat, 64, 1, file ) != 1 )
               return( FALSE );
            break;

         default:
	    if (DK_option.verbose == TRUE)
                fprintf(
		   DK_option.msgFd,  "Error: %s %d): illegal token: %hd\n",
		   __FILE__, __LINE__, prot->mesh[i] 
	        );
            break;
      }
   }

   return( TRUE );
}

DK_Metaball *DK_metaballReadBinary( 
   FILE        *file, 
   DK_FileProt *prot 
)
{
   DK_Metaball *ball;
   short    nb, tok, ind = 0;

   if( (ball = DK_metaballAllocate()) == NULL )
      return( NULL );

   if( (prot != NULL) && (prot->metaball != NULL) )
      tok = prot->metaball[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );

   while( tok != 0 ) /* End of metaball definition */ 
   {
      switch( tok )
      {
         case 1:        /* OBSOLETE: nbcurve */
            DK_byteswap_fread( &nb, 4, 1, file );
            break;

         case 2:        /* OBSOLETE: nbsub */
            DK_byteswap_fread( &nb, 4, 1, file );
            break;

         case 4:        /* ratio: derivated from weight */
            DK_byteswap_fread( &ball->ratio, 8, 1, file );
            break;

         case 5:        /* weight */
            DK_byteswap_fread( &ball->weight, 8, 1, file );
            break;

         case 6:        /* exponent */
            DK_3doubles_fread( ball->exponent, 24, 1, file );
            break;

         case 7:        /* centerMat */
            DK_Matrix_fread( ball->centerMat, 64, 1, file );
            break;

         default:
            DK_fileSkipField( file, tok );
      }

      if( (prot != NULL) && (prot->metaball != NULL) )
         tok = prot->metaball[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   return( ball );
}

DK_Boolean	DK_metaballWriteAscii( 
   FILE        *file, 
   DK_Metaball *ball, 
   DK_String    space 
)
{
   char		 sspace[80];
   int		 i, j;

   sprintf( sspace, "%s  ", space );

   fprintf( file, "\n%smetaball\n", space );
   fprintf( file, "%s{\n", space );
   
   fprintf( file, "%sweight    %*.*f\n", sspace, DK_FM, DK_FD, ball->weight );
   fprintf( file, "%sratio     %*.*f\n", sspace, DK_FM, DK_FD, ball->ratio );
   fprintf( file, "%sexponent  %*.*f  %*.*f  %*.*f\n", sspace,
					 DK_FM, DK_FD, ball->exponent[0],
					 DK_FM, DK_FD, ball->exponent[1],
					 DK_FM, DK_FD, ball->exponent[2] );
   fprintf( file, "%scenterMat %*.*f  %*.*f  %*.*f  %*.*f\n", sspace,
					 DK_FM, DK_FD, ball->centerMat[0][0],
					 DK_FM, DK_FD, ball->centerMat[0][1],
					 DK_FM, DK_FD, ball->centerMat[0][2],
					 DK_FM, DK_FD, ball->centerMat[0][3]  );
   fprintf( file, "%s          %*.*f  %*.*f  %*.*f  %*.*f\n", sspace,
					 DK_FM, DK_FD, ball->centerMat[1][0],
					 DK_FM, DK_FD, ball->centerMat[1][1],
					 DK_FM, DK_FD, ball->centerMat[1][2],
					 DK_FM, DK_FD, ball->centerMat[1][3]  );
   fprintf( file, "%s          %*.*f  %*.*f  %*.*f  %*.*f\n", sspace,
					 DK_FM, DK_FD, ball->centerMat[2][0],
					 DK_FM, DK_FD, ball->centerMat[2][1],
					 DK_FM, DK_FD, ball->centerMat[2][2],
					 DK_FM, DK_FD, ball->centerMat[2][3]  );
   fprintf( file, "%s          %*.*f  %*.*f  %*.*f  %*.*f\n", sspace,
					 DK_FM, DK_FD, ball->centerMat[3][0],
					 DK_FM, DK_FD, ball->centerMat[3][1],
					 DK_FM, DK_FD, ball->centerMat[3][2],
					 DK_FM, DK_FD, ball->centerMat[3][3]  );

   fprintf( file, "%s}\n", space );
   return( TRUE );
}

DK_Metaball *DK_metaballReadAscii( 
   FILE *file 
)
{
   DK_Metaball	*ball = NULL;
   char		 str[ 80 ];

   if( (ball = DK_metaballAllocate()) == NULL )
      return( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( ball );

   do
   {
      fscanf( file, "%s", str );

      if( strcmp( "weight", str ) == 0 )
         fscanf( file, "%lf", &ball->weight );
      else if( strcmp( "ratio", str ) == 0 )
         fscanf( file, "%lf", &ball->ratio );
      else if( strcmp( "exponent", str ) == 0 )
         fscanf( file, "%lf %lf %lf",	&ball->exponent[0],
					&ball->exponent[1],
					&ball->exponent[2]  );
      else if( strcmp( "centerMat", str ) == 0 )
         fscanf( file, "%f %f %f %f  %f %f %f %f  %f %f %f %f  %f %f %f %f",
		&ball->centerMat[0][0], &ball->centerMat[0][1],
		&ball->centerMat[0][2], &ball->centerMat[0][3],
		&ball->centerMat[1][0], &ball->centerMat[1][1],
		&ball->centerMat[1][2], &ball->centerMat[1][3],
		&ball->centerMat[2][0], &ball->centerMat[2][1],
		&ball->centerMat[2][2], &ball->centerMat[2][3],
		&ball->centerMat[3][0], &ball->centerMat[3][1],
		&ball->centerMat[3][2], &ball->centerMat[3][3] );

   } while( strcmp( str, "}" ) != 0 );

   return( ball );
}

DK_MetaSystem *DK_metaSystemAllocate( void )
{
   DK_MetaSystem *sys;

   if( sys = (DK_MetaSystem *) calloc( 1, sizeof( DK_MetaSystem ) ) )
   {
      sys->wfResolution = 0.5;
      sys->renResolution = 0.5;
      sys->showSurface = FALSE;
      sys->groupList = NULL;
      sys->unblendList = NULL;
   }

   return( sys );
}

void DK_metaSystemDispose( 
   DK_MetaSystem **sys 
)
{
   if( *sys != NULL )
   {
      while( (*sys)->groupList != NULL )
         DK_metaGroupDispose( *sys, &(*sys)->groupList );

      while( (*sys)->unblendList )
         DK_metaGroupsBlend( *sys, (*sys)->unblendList->groups[0],
			           (*sys)->unblendList->groups[1] );

      free( *sys );
      *sys = NULL;
   }
}

DK_Boolean DK_metaSystemWriteBinary( 
   FILE          *file, 
   DK_MetaSystem *sys, 
   DK_FileProt   *prot 
)
{
   long			i, j;
   short		nb;
   DK_MetaLink		*lnk;
   DK_MetaGroup		*group;
   DK_MetaUnblend	*unblend;

   for( i = 0; i < prot->nbMetaSystem; i++ )
   {
      switch( prot->metaSystem[ i ] )
      {
         case 0:        /* end of metaSystem definition */
            break;

         case 1:        /* wireframe Resolution */
            DK_byteswap_fwrite( &sys->wfResolution, 4, 1, file );
            break;

         case 2:        /* rendering Resolution */
            DK_byteswap_fwrite( &sys->renResolution, 4, 1, file );
            break;

         case 3:        /* show surface */
            DK_byteswap_fwrite( &sys->showSurface, 2, 1, file );
            break;

         case 4:        /* goups list: nb, groups[ nb ] */
            /* count nb of groups */
            nb = 0;
            for( group = sys->groupList; group != NULL; group = group->next )
            {
               nb++;
            }

            DK_byteswap_fwrite( &nb, 2, 1, file );

            for( group = sys->groupList; group != NULL; group = group->next )
               for( j = 0; j < prot->nbMetaGroup; j++ )
               {
                  switch( prot->metaGroup[ j ] )
                  {
                     case 0:        /* end of metaGroup definition */
                        break;
   
                     case 1:        /* name */
                        DK_stringWriteBinary( file, group->name );
                        break;

                     case 2:        /* metaballs list: nb, metaballs[ nb ] */
                        /* count nb of groups */
                        nb = 0;
                        for( lnk = group->prims; lnk != NULL; lnk = lnk->next )
                           nb++;
                        DK_byteswap_fwrite( &nb, 2, 1, file );
                        for( lnk = group->prims; lnk != NULL; lnk = lnk->next )
                           DK_stringWriteBinary( file, lnk->prim );
                        break;
                  }
               }
            break;

         case 5:        /* unblendings list: nb, unblending[ nb ]  */
            nb = 0;
            for( unblend = sys->unblendList; unblend != NULL;
						 unblend = unblend->next )
               nb++;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            for( unblend = sys->unblendList; unblend != NULL;
						 unblend = unblend->next )
               for( j = 0; j < prot->nbMetaUnblend; j++ )
               {
                  switch( prot->metaUnblend[ j ] )
                  {
                     case 0:        /* end of metaUnblend definition */
                        break;

                     case 1:        /* names */
                        DK_stringWriteBinary( file, unblend->groups[0] );
                        DK_stringWriteBinary( file, unblend->groups[1] );
                        break;
                  }
               }
            break;
      }
   }

   return( TRUE );
}

DK_MetaSystem *DK_metaSystemReadBinary( 
   FILE        *file, 
   DK_FileProt *prot 
)
{
   DK_MetaSystem	*sys;
   long			i, j, ind = 0, ind1;
   short		nb, nbGroups, nbPrims;
   DK_MetaLink		*lnk;
   DK_MetaGroup		*group;
   DK_MetaUnblend	*unblend;
   short		 tok;
   char			*name, *name1;

   if( (sys = DK_metaSystemAllocate()) == NULL )
      return( NULL );

   if( (prot != NULL) && (prot->metaSystem != NULL) )
      tok = prot->metaSystem[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );

   while( tok != 0 /* End of metaball definition */ )
   {
      switch( tok )
      {
         case 1:        /* wireframe Resolution */
            DK_byteswap_fread( &sys->wfResolution, 4, 1, file );
            break;

         case 2:        /* rendering Resolution */
            DK_byteswap_fread( &sys->renResolution, 4, 1, file );
            break;

         case 3:        /* show surface */
            DK_byteswap_fread( &sys->showSurface, 2, 1, file );
            break;

         case 4:        /* goups list: nb, groups[ nb ] */
            DK_byteswap_fread( &nbGroups, 2, 1, file );

            for( i = 0; i < nbGroups; i++ )
            {
               group = DK_metaGroupAllocate( sys );
               ind1 = 0;

               if( (prot != NULL) && (prot->metaGroup != NULL) )
                  tok = prot->metaGroup[ ind1 ++ ];
               else
                  DK_byteswap_fread( &tok, 2, 1, file );

               while( tok != 0 /* End of metaball definition */ )
               {
                  switch( tok )
                  {
                     case 1:        /* name */
                        group->name = DK_stringReadBinary( file );
                        break;

                     case 2:        /* metaballs list: nb, metaballs[ nb ] */
                        /* count nb of primitives */
                        DK_byteswap_fread( &nbPrims, 2, 1, file );
                        for( j = 0; j < nbPrims; j++ )
                        {
                           name = DK_stringReadBinary( file );
                           DK_metaGroupAddPrimitive( group, name );
                        }
                        break;

                     default:
                        DK_fileSkipField( file, tok );

                  }

                  if( (prot != NULL) && (prot->metaGroup != NULL) )
                     tok = prot->metaGroup[ ind1 ++ ];
                  else
                     DK_byteswap_fread( &tok, 2, 1, file );
               }
            }
            break;

         case 5:        /* unblendings list: nb, unblending[ nb ]  */
            DK_byteswap_fread( &nb, 2, 1, file );

            for( i = 0; i < nb; i++ )
            {
               ind1 = 0;

               if( (prot != NULL) && (prot->metaUnblend != NULL) )
                  tok = prot->metaUnblend[ ind1 ++ ];
               else
                  DK_byteswap_fread( &tok, 2, 1, file );

               while( tok != 0 /* End of metaball definition */ )
               {
                  switch( tok )
                  {
                     case 1:        /* names */
                        name  = DK_stringReadBinary( file );
                        name1 = DK_stringReadBinary( file );
                        break;

                     default:
                        DK_fileSkipField( file, tok );

                  }
                  if( (prot != NULL) && (prot->metaUnblend != NULL) )
                     tok = prot->metaUnblend[ ind1 ++ ];
                  else
                     DK_byteswap_fread( &tok, 2, 1, file );
               }
               DK_metaGroupsUnblend( sys, name, name1 );
            }
            break;


         default:
            DK_fileSkipField( file, tok );

       }
       if( (prot != NULL) && (prot->metaSystem != NULL) )
          tok = prot->metaSystem[ ind ++ ];
       else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   return( sys );
}

DK_Boolean DK_metaSystemWriteAscii( 
   FILE          *file, 
   DK_MetaSystem *sys, 
   DK_String      space 
)
{
   long		   i, j;
   DK_MetaLink	  *lnk;
   DK_MetaGroup	  *group;
   DK_MetaUnblend *unblend;
   char            sspace[80];


   sprintf( sspace, "%s  ", space );

   fprintf( file, "\n%smetasystem\n", space );
   fprintf( file, "%s{\n", space );

   fprintf( 
      file, "%s  wfResolution   %*.*f\n", 
      space, DK_FM, DK_FD, sys->wfResolution 
   );

   fprintf( 
      file, "%s  renResolution   %*.*f\n", 
      space, DK_FM, DK_FD, sys->renResolution 
   );

   if( sys->showSurface )
      fprintf( file, "%sshowSurface\n", space );
   
   for( group = sys->groupList; group != NULL; group = group->next )
   {
      fprintf( file, "\n%s  group\n", space );
      fprintf( file, "%s  {\n", space );
      fprintf( file, "%s    name     \"%s\"\n", space, group->name );
      for( lnk = group->prims; lnk != NULL; lnk = lnk->next )
         fprintf( file, "%s    primitive  \"%s\"\n", space, lnk->prim );
      fprintf( file, "%s  }\n", space );
   }

   for( unblend = sys->unblendList; unblend != NULL; unblend = unblend->next )
      fprintf( file, "\n%s  unblend \"%s\" \"%s\"\n",
			       space, unblend->groups[0], unblend->groups[1] );

   fprintf( file, "%s}\n", space );
   return( TRUE );
}

DK_MetaSystem *DK_metaSystemReadAscii( 
   FILE *file 
)
{
   DK_MetaSystem	*sys = NULL;
   DK_String		group1, group2; 
   char			str[ 80 ];
   

   if( (sys = DK_metaSystemAllocate()) == NULL )
      return( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( sys );

   do
   {
      fscanf( file, "%s", str );

      if( strcmp( "wfResolution", str ) == 0 )
         fscanf( file, "%f", &sys->wfResolution );

      else if( strcmp( "renResolution", str ) == 0 )
         fscanf( file, "%f", &sys->renResolution );

      else if( strcmp( "showSurface", str ) == 0 )
         sys->showSurface = TRUE;

      else if( strcmp( "group", str ) == 0 )
         DK_metaGroupReadAscii( file, sys );

      else if( strcmp( "unblend", str ) == 0 )
      {
         group1 = DK_stringReadAscii( file );
         group2 = DK_stringReadAscii( file );
         DK_metaGroupsUnblend( sys, group1, group2 );
      }

   } while( strcmp( str, "}" ) != 0 );

   return( sys );
}

static DK_MetaGroup *DK_metaGroupReadAscii( FILE *file, DK_MetaSystem *sys )
{
   DK_MetaGroup		*group;
   char			str[ 80 ];
   DK_String		ball;

   if( (group = DK_metaGroupAllocate( sys )) == NULL )
      return( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( group );

   do
   {
      fscanf( file, "%s", str );

      if( strcmp( "name", str ) == 0 )
         group->name = DK_stringReadAscii( file );

      else if( strcmp( "primitive", str ) == 0 )
      {
         ball = DK_stringReadAscii( file );
         DK_metaGroupAddPrimitive( group, ball );
      }

   } while( strcmp( str, "}" ) != 0 );

   return( group );

}


DK_MetaGroup *DK_metaGroupAllocate( 
   DK_MetaSystem *sys 
)
{
   DK_MetaGroup *group, *prev;

   if( group = (DK_MetaGroup *) calloc( 1, sizeof( DK_MetaGroup ) ) )
   {
      group->name = NULL;
      group->prims = NULL;
      group->next = NULL;
      group->prev = NULL;

      if( sys->groupList == NULL )
         sys->groupList = group;
      else
      {
         for( prev = sys->groupList; prev->next != NULL;  prev = prev->next )
            /* _NOP() */ ;
         prev->next = group;
         group->prev = prev;
      }

      
   }

   return( group );
}

void DK_metaGroupDispose( 
   DK_MetaSystem  *sys, 
   DK_MetaGroup  **group 
)
{
   if( *group != NULL )
   {
      DK_MetaGroup *groupListBackup = NULL;
      DK_Boolean groupListSet = FALSE;

      while( (*group)->prims != NULL )
         DK_metaGroupRemovePrimitive( *group, (*group)->prims->prim );

      if( (*group)->prev ) 
      {
         (*group)->prev->next = (*group)->next;
      }
      else 
      {
         /* 
         *** Fact: if only 1 MetaGroup, &(sys->groupList) == (*group)
         *** Without this, (*group) is modified if sys->groupList 
         *** is assigned directly.  Thus, (*group) becomes a dangling ptr
         *** and is never freed up.
         */
         groupListSet = TRUE;
         groupListBackup = (*group)->next;
      }

      if( (*group)->next )
         (*group)->next->prev = (*group)->prev;
      
      free( *group );
      *group = NULL;
   
      if ( groupListSet ) {
         sys->groupList = groupListBackup;
      }
      
   }
}

DK_Boolean DK_metaGroupAddPrimitive( 
   DK_MetaGroup *group, 
   DK_String     ball 
)
{
   DK_MetaLink	*prev, *lnk;

   if( lnk = (DK_MetaLink *) calloc( 1, sizeof( DK_MetaLink ) ) )
   {
      lnk->next  = NULL;
      lnk->prev  = NULL;
      lnk->prim  = strdup( ball );

      if( group->prims == NULL )
         group->prims = lnk;
      else
      {
         for( prev = group->prims; prev->next != NULL;  prev = prev->next )
	    /* _NOP() */ ;
         prev->next = lnk;
         lnk->prev = prev;
      }

      return( TRUE );
   }
   return( FALSE );
}

DK_Boolean DK_metaGroupRemovePrimitive( 
   DK_MetaGroup *group, 
   DK_String     ball 
)
{
   DK_MetaLink  *lnk;

   for( lnk = group->prims; lnk != NULL;  lnk = lnk->next )
      if( strcmp( lnk->prim, ball ) == 0 )
      {
         if( lnk->prev != NULL )
            lnk->prev->next = lnk->next;
         else 
            group->prims = lnk->next;

         if( lnk->next != NULL )
            lnk->next->prev = lnk->prev;

         if( lnk->prim != NULL )
            free( lnk->prim );

         free( lnk );
         return( TRUE );
      }
   return( FALSE );
}

DK_Boolean DK_metaGroupsBlend( 
   DK_MetaSystem *sys, 
   DK_String      group0, 
   DK_String      group1 
)
{
   DK_MetaUnblend	*unblend;

   /*********************************************************************/
   /* Group are blending together if there is no unlink node with them, */
   /* so search and remove a such node					*/
   /*********************************************************************/

   for( unblend = sys->unblendList; unblend != NULL; unblend = unblend->next )
      if( ( (strcmp( unblend->groups[0], group0 ) == 0) &&
            (strcmp( unblend->groups[1], group1 ) == 0)    ) ||
          ( (strcmp( unblend->groups[0], group1 ) == 0) &&
            (strcmp( unblend->groups[1], group0 ) == 0)    )  )
      {
         if( unblend->prev != NULL )
            unblend->prev->next = unblend->next;
         else
            sys->unblendList = unblend->next;

         if( unblend->next != NULL )
            unblend->next->prev = unblend->prev;

         if( unblend->groups[0] != NULL )
            free( unblend->groups[0] );

         if( unblend->groups[1] != NULL )
            free( unblend->groups[1] );

         free( unblend );

         return( TRUE );
      }

   return( TRUE );
}

DK_Boolean DK_metaGroupsUnblend( 
   DK_MetaSystem *sys, 
   DK_String      group0, 
   DK_String      group1 
)
{
   DK_MetaUnblend	*prev, *unblend;

   for( prev = sys->unblendList; prev != NULL; prev = prev->next )
   {
      if( ( (strcmp( unblend->groups[0], group0 ) == 0) &&
            (strcmp( unblend->groups[1], group1 ) == 0)    ) ||
          ( (strcmp( unblend->groups[0], group1 ) == 0) &&
            (strcmp( unblend->groups[1], group0 ) == 0)    )  )
         return( TRUE );

      if( prev->next == NULL )
         break;
   }

   if( unblend = (DK_MetaUnblend *) calloc( 1, sizeof( DK_MetaUnblend ) ) )
   {
      unblend->groups[0] = group0;
      unblend->groups[1] = group1;
      unblend->next     = NULL;
      unblend->prev     = NULL;

      if( prev == NULL )
         sys->unblendList = unblend;
      else
      {
         prev->next = unblend;
         unblend->prev = prev;
      }
      return( TRUE );
   }

   return( FALSE );
}
