/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyMesh.c,v $
   $Revision: 1.9.14.1 $ $Date: 1995/06/09 22:00:09 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's mesh functions,
   its content is sufficient to read, write and manage mesh data structure
   into SOFTIMAGE's file format...

   Written by: Laurent Lauzon

   (c) Copyright 1990, 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinyMesh.h"


/******************************************************************************/
/*** Static function prototypes ***********************************************/
/******************************************************************************/

static void polygonNodeReadAscii(  FILE *, DK_Polygon * );

/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

DK_Mesh *DK_meshAllocate( void )
{
   DK_Mesh *msh;

   if( msh = (DK_Mesh *) calloc( 1, sizeof( DK_Mesh ) ) )
   {
      msh->nbVertices = 0;
      msh->vertices   = NULL;
      msh->nbPolygons = 0;
      msh->polygons   = NULL;
      msh->nbEdges    = 0;
      msh->edges      = NULL;
      msh->discAngle  = 60.0;
      msh->flag       = 0;
   }

   return( msh );
}

void DK_meshDispose( DK_Mesh **msh )
{
   DK_Polygon	*pol;
   int		i;

   if( *msh != NULL )
   {
      if( (*msh)->vertices != NULL )
         free( (*msh)->vertices );

      if( (*msh)->polygons != NULL )
      {
         for( i = 0, pol = (*msh)->polygons; i < (*msh)->nbPolygons; i++, pol++)
            if( pol->nodes != NULL )
               free( pol->nodes );

         free( (*msh)->polygons );
      }

      if( (*msh)->edges != NULL )
         free( (*msh)->edges );

      free( (*msh ) );
      *msh = NULL;
   }
}

DK_Edge *DK_edgeAllocate( long nb )
{
   DK_Edge   *edg, *ptr;
   int       i;

   if( edg = (DK_Edge *) calloc( nb + 1, sizeof( DK_Edge ) ) )
   {
      for( i = 0, ptr = edg; i <= nb; i++, ptr++ )
      {
         ptr->vertices[0] = -1;
         ptr->vertices[1] = -1;
         ptr->polygons[0] = -1;
         ptr->polygons[1] = -1;
         ptr->flag        = 0x0;
      }
   }
   return( edg );
}

DK_Polygon *DK_polygonAllocate( long nb )
{
   DK_Polygon   *pol, *ptr;
   int       i;

   if( pol = (DK_Polygon *) calloc( nb, sizeof( DK_Polygon ) ) )
   {
      for( i = 0, ptr = pol; i < nb; i++, ptr++ )
      {
         ptr->normal.x = ptr->normal.y = ptr->normal.z = 0.0;
         ptr->material = 0;
         ptr->nbNodes  = 0;
         ptr->nodes    = NULL;
         ptr->flag     = 0x0;
      }
   }
   return( pol );
}

DK_Vertex *DK_vertexAllocate( long nb )
{
   DK_Vertex   *vrt, *ptr;
   int       i;

   if( vrt = (DK_Vertex *) calloc( nb, sizeof( DK_Vertex ) ) )
   {
      for( i = 0, ptr = vrt; i < nb; i++, ptr++ )
      {
         ptr->position.x = ptr->position.y = ptr->position.z = 0.0;
         ptr->flag       = 0x0;
      }
   }
   return( vrt );
}


void DK_polygonInitializeVertex( DK_Polygon *pol, int nbn, ... )
{
   va_list           ap;
   int               i;

   pol->nbNodes = nbn;
   pol->nodes = (DK_PolyNode *) calloc( pol->nbNodes, sizeof( DK_PolyNode ) );
   va_start( ap, nbn );
   for( i = 0; i < nbn; i++ )
      pol->nodes[i].vertex = va_arg( ap, int );
   va_end( ap );
}

DK_Boolean DK_meshWriteBinary(  FILE *file, DK_Mesh *msh, DK_FileProt *prot )
{
   long      i, j, k, l, m;
   short     sNb;
   DK_Vertex   *vrt;
   DK_Polygon  *pol;
   DK_Edge     *edg;

   for( i = 0; i < prot->nbMesh; i++ )
   {
      switch( prot->mesh[ i ] )
      {
         case 0:	/* end of mesh definition */
            break;

         case 2:	/* discontinuity angle */
            if( DK_byteswap_fwrite( &msh->discAngle, 4, 1, file ) != 1 )
               return( FALSE );
            break;

         case 3:	/* vertices list (short) */
         case 7:	/* vertices list (long) */
            if( prot->mesh[ i ] == 3 )
            {
               sNb = msh->nbVertices;
               if( DK_byteswap_fwrite( &sNb, 2, 1, file ) != 1 )
                  return( FALSE );
            }
            else  /* 7 */
            {
               if( DK_byteswap_fwrite( &msh->nbVertices, 4, 1, file ) != 1 )
                  return( FALSE );
            }
            for( j = 0, vrt = msh->vertices; j < msh->nbVertices; j++, vrt++ )
               for( k = 0; k < prot->nbVertex; k++ )
               {
                  switch( prot->vertex[ k ] )
                  {
                     case 0:	/* end of vertex definition */
                        break;

                     case 1:	/* 1: position */
               	        if( DK_Vector_fwrite( &vrt->position, sizeof( DK_Vector ), 1, file )
                                                                          != 1 )
                           return( FALSE );
                        break;

                     case 4:	/* 4: flag */
                        if( DK_byteswap_fwrite( &vrt->flag, 2, 1, file ) != 1 )
                           return( FALSE );
                        break;

                     default:
			if(  DK_option.verbose == TRUE )
                            fprintf(  DK_option.msgFd, 
				"Error: %s %d): illegal token: %hd\n",
                                __FILE__, __LINE__, prot->vertex[ k ] );
                        break;
                  }
               }
            break;

         case 4:	/* polygons list (short) */
         case 8:	/* polygons list (long) */
            if( prot->mesh[ i ] == 4 )
            {
               sNb = msh->nbPolygons;
               if( DK_byteswap_fwrite( &sNb, 2, 1, file ) != 1 )
                  return( FALSE );
            }
            else  /* 8 */
            {
               if( DK_byteswap_fwrite( &msh->nbPolygons, 4, 1, file ) != 1 )
                  return( FALSE );
            }
            for( j = 0, pol = msh->polygons; j < msh->nbPolygons; j++, pol++ )
               for( k = 0; k < prot->nbPolygon; k++ )
               {
                  switch( prot->polygon[ k ] )
                  {
                     case 0:	/* end of polygon definition  */
                        break;

                     case 2:	/* nodes list */
                        if( DK_byteswap_fwrite( &pol->nbNodes, 2, 1, file ) != 1 )
                           return( FALSE );
                        for( l = 0; l < pol->nbNodes; l++ )
                           for( m = 0; m < prot->nbPolNode; m++ )
                           {
                              switch( prot->polNode[ m ] )
                              {
                                 case 0:	/* end of node definition */
                                    break;

                                 case 1:	/* vertex index (short) */
                                    sNb = pol->nodes[l].vertex;
                                    if( DK_byteswap_fwrite( &sNb, 2, 1, file ) != 1 )
                                       return( FALSE );
                                    break;

                                 case 2:	/* normal */
                                    if( DK_Vector_fwrite( &pol->nodes[l].normal,
                                              sizeof( DK_Vector ), 1, file ) != 1 )
                                       return( FALSE );
                                    break;

                                 case 4:        /* uvT Texture coord.*/
                                    if( DK_byteswap_fwrite( &pol->nodes[l].uTxt,
                                                4, 1, file ) != 1 )
                                       return( FALSE );
                                    if( DK_byteswap_fwrite( &pol->nodes[l].vTxt,
                                                4, 1, file ) != 1 )
                                       return( FALSE );
                                    break;


                                 case 3:	/* vertex index (long) */
                                    if( DK_byteswap_fwrite( &pol->nodes[l].vertex,
                                                4, 1, file ) != 1 )
                                       return( FALSE );
                                    break;

                                 default:
				    if ( DK_option.verbose == TRUE)
                                        fprintf( DK_option.msgFd, 
					"Error: %s %d: illegal token: %hd\n",
                                       __FILE__, __LINE__, prot->polNode[ m ] );
                                    break;
                              }
                           }
                        break;

                     case 3:	/* material index */
                        if( DK_byteswap_fwrite( &pol->material, 2, 1, file ) != 1 )
                           return( FALSE );
                        break;

		     case 4: /* flag */
			if( DK_byteswap_fwrite( &pol->flag, 2, 1, file ) != 1 )
			   return( FALSE );
			break;
			
                     default:
			if ( DK_option.verbose == TRUE)
                            fprintf( DK_option.msgFd,
				 "Error: %s %d): illegal token: %hd\n",
                                 __FILE__, __LINE__, prot->polygon[ k ] );
                        break;
                  }
               }
            break;

         case 5:	/* edge list (short) */
         case 9:	/* edge list (long) */
            if( prot->mesh[ i ] == 5 )
            {
               sNb = msh->nbEdges;
               if( DK_byteswap_fwrite( &sNb, 2, 1, file ) != 1 )
                  return( FALSE );
            }
            else  /* 9 */
            {
               if( DK_byteswap_fwrite( &msh->nbEdges, 4, 1, file ) != 1 )
                  return( FALSE );
            }
            for( j = 1, edg = &msh->edges[1]; j <= msh->nbEdges; j++, edg++ )
               for( k = 0; k < prot->nbEdge; k++ )
               {
                  switch( prot->edge[ k ] )
                  {
                     case 0:	/* end of edge definition */
                        break;

                     case 1:	/* vertex indexes (short) */
                        sNb = edg->vertices[0];
                        if( DK_byteswap_fwrite( &sNb, 2, 1, file ) != 1 )
                           return( FALSE );
                        sNb = edg->vertices[1];
                        if( DK_byteswap_fwrite( &sNb, 2, 1, file ) != 1 )
                           return( FALSE );
                        break;

                     case 3:	/* flag */
                        if( DK_byteswap_fwrite( &edg->flag, 2, 1, file ) != 1 )
                           return( FALSE );
                        break;

                     case 4:	/* vertex indexes (long) */
                        if( DK_byteswap_fwrite( &edg->vertices[0], 4, 1, file ) != 1 )
                           return( FALSE );
                        if( DK_byteswap_fwrite( &edg->vertices[1], 4, 1, file ) != 1 )
                           return( FALSE );
                        break;

                     default:
			if ( DK_option.verbose == TRUE)
                            fprintf( DK_option.msgFd,
				 "Error: %s %d): illegal token: %hd\n",
                                 __FILE__, __LINE__, prot->edge[ k ] );
                        break;
                  }
               }
            break;

         case 6:	/* flag */
            if( DK_byteswap_fwrite( &msh->flag, 2, 1, file ) != 1 )
               return( FALSE );
            break;

         default:
	    if ( DK_option.verbose == TRUE)
                fprintf( DK_option.msgFd,  "Error: %s %d): illegal token: %hd\n",
                                            __FILE__, __LINE__, prot->mesh[i] );
            break;
      }
   }

   return( TRUE );
}

DK_Mesh *DK_meshReadBinary(  FILE *file, DK_FileProt *prot )
{
   DK_Mesh *msh;
   short    nb, tok, ind = 0, ind1, ind2;
   long     i, j;
   DK_Vertex  *vrt;
   DK_Polygon *pol;
   DK_Edge    *edg;

   if( (msh = DK_meshAllocate()) == NULL )
      return( NULL );

   if( (prot != NULL) && (prot->mesh != NULL) )
      tok = prot->mesh[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );
   while( tok != 0 /* End of mesh definition */ )
   {
      switch( tok )
      {
         case 1: /* faceted */
            DK_byteswap_fread( &nb, 2, 1, file );
            if( nb != 0 )
               msh->flag |= DK_MSH_FACETED;
            break;

         case 2: /* discontinuity angle */
            DK_byteswap_fread( &msh->discAngle, 4, 1, file );
            break;

         case 3: /* vertices list (short) */
         case 7: /* vertices list (long) */
            if( tok == 3 )
            {
               DK_byteswap_fread( &nb, 2, 1, file );
               msh->nbVertices = nb;
            }
            else   /* 7 */
               DK_byteswap_fread( &msh->nbVertices, 4, 1, file );
            vrt = msh->vertices = DK_vertexAllocate( msh->nbVertices );
            for( i = 0; i < msh->nbVertices; i++, vrt++ )
            {
               ind1 = 0;
               if( (prot != NULL) && (prot->vertex != NULL) )
                  tok = prot->vertex[ ind1 ++ ];
               else
                  DK_byteswap_fread( &tok, 2, 1, file );
               while( tok != 0 /* End of vertex definition */ )
               {
                  switch( tok )
                  {
                     case 1: /* position */
                        DK_Vector_fread( &vrt->position, sizeof( DK_Vector ), 1, file );
                        break;
                     case 4: /* flag */
                        DK_byteswap_fread( &vrt->flag, 2, 1, file );
                        break;

                     default:
                        DK_fileSkipField( file, tok );
                  }
                  if( (prot != NULL) && (prot->vertex != NULL) )
                     tok = prot->vertex[ ind1 ++ ];
                  else
                     DK_byteswap_fread( &tok, 2, 1, file );
               }
            }
            break;

         case 4: /* polygons list (short) */
         case 8: /* polygons list (long) */
            if( tok == 4 )
            {
               DK_byteswap_fread( &nb, 2, 1, file );
               msh->nbPolygons = nb;
            }
            else  /* 8 */
               DK_byteswap_fread( &msh->nbPolygons, 4, 1, file );
            pol = msh->polygons = DK_polygonAllocate( msh->nbPolygons );
            for( i = 0; i < msh->nbPolygons; i++, pol++ )
            {
               ind1 = 0;
               if( (prot != NULL) && (prot->polygon != NULL) )
                  tok = prot->polygon[ ind1 ++ ];
               else
                  DK_byteswap_fread( &tok, 2, 1, file );
               while( tok != 0 /* End of polygon definition */ )
               {
                  switch( tok )
                  {
                     case 1: /* normal polygons */
                        DK_Vector_fread( &pol->normal, sizeof( DK_Vector ), 1, file );
                        break;
                     case 2: /* nodes list */
                        DK_byteswap_fread( &pol->nbNodes, 2, 1, file );
                        pol->nodes = (DK_PolyNode *) calloc( pol->nbNodes, sizeof( DK_PolyNode ) );
                        for( j = 0; j < pol->nbNodes; j++ )
                        {
                           ind2 = 0;
                           if( (prot != NULL) && (prot->polNode != NULL) )
                              tok = prot->polNode[ ind2 ++ ];
                           else
                              DK_byteswap_fread( &tok, 2, 1, file );
                           while( tok != 0 /* End of node definition */ )
                           {
                              switch( tok )
                              {
                                 case 1: /* vertex index (short)  */
                                    DK_byteswap_fread( &nb, 2, 1, file );
                                    pol->nodes[j].vertex = nb;
                                    break;

                                 case 2: /* normal */
                                    DK_Vector_fread( &pol->nodes[j].normal,
                                                    sizeof( DK_Vector ), 1, file );
                                    break;

                                 case 3: /* vertex index (long)  */
                                    DK_byteswap_fread( &pol->nodes[j].vertex, 4, 1, file );
                                    break;

                                 case 4: /* uv Texture coord */
                                    DK_byteswap_fread( &pol->nodes[j].uTxt, 4, 1, file );
                                    DK_byteswap_fread( &pol->nodes[j].vTxt, 4, 1, file );
                                    break;

                                 default:
                                   DK_fileSkipField( file, tok );
                              }
                              if( (prot != NULL) && (prot->polNode != NULL) )
                                 tok = prot->polNode[ ind2 ++ ];
                              else
                                 DK_byteswap_fread( &tok, 2, 1, file );
                           }
                        }
                        break;

                     case 3: /* material */
                        DK_byteswap_fread( &pol->material, 2, 1, file );
                        break;
                     case 4: /* flag */
                        DK_byteswap_fread( &pol->flag, 2, 1, file );
                        break;

                     default:
                        DK_fileSkipField( file, tok );
                  }
                  if( (prot != NULL) && (prot->polygon != NULL) )
                     tok = prot->polygon[ ind1 ++ ];
                  else
                     DK_byteswap_fread( &tok, 2, 1, file );
               }
            }
            break;

         case 5: /* edges list (short) */
         case 9: /* edges list (long) */
            if( tok == 5 )
            {
               DK_byteswap_fread( &nb, 2, 1, file );
               msh->nbEdges = nb;
            }
            else  /* 9 */
               DK_byteswap_fread( &msh->nbEdges, 4, 1, file );
            msh->edges = DK_edgeAllocate( msh->nbEdges );
            for( i = 1, edg = &msh->edges[1]; i <= msh->nbEdges; i++, edg++ )
            {
               ind1 = 0;
               if( (prot != NULL) && (prot->edge != NULL) )
                  tok = prot->edge[ ind1 ++ ];
               else
                  DK_byteswap_fread( &tok, 2, 1, file );
               while( tok != 0 /* End of edge definition */ )
               {
                  switch( tok )
                  {
                     case 1: /* vertex (short) */
                        DK_byteswap_fread( &nb, 2, 1, file );
                        edg->vertices[0] = nb;
                        DK_byteswap_fread( &nb, 2, 1, file );
                        edg->vertices[1] = nb;
                        break;

                     case 3: /* flag */
                        DK_byteswap_fread( &edg->flag, 2, 1, file );
			break;

                     case 4: /* vertex (long) */
                        DK_byteswap_fread( &edg->vertices[0], 4, 1, file );
                        DK_byteswap_fread( &edg->vertices[1], 4, 1, file );
                        break;

                     default:
                        DK_fileSkipField( file, tok );
                  }
                  if( (prot != NULL) && (prot->edge != NULL) )
                     tok = prot->edge[ ind1 ++ ];
                  else
                     DK_byteswap_fread( &tok, 2, 1, file );
               }
            }
            break;

         case 6: /* flag */
            DK_byteswap_fread( &msh->flag, 2, 1, file );
            break;


         default:
            DK_fileSkipField( file, tok );

      }

      if( (prot != NULL) && (prot->mesh != NULL) )
         tok = prot->mesh[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   return( msh );
}

DK_Boolean	DK_meshWriteAscii(  FILE *file, DK_Mesh *msh, DK_String space )
{
   char		 sspace[80];
   DK_Vertex	*vrt;
   DK_Polygon	*pol;
   DK_Edge		*edg;
   int		 i, j;

   sprintf( sspace, "%s  ", space );

   fprintf( file, "\n%smesh\n", space );
   fprintf( file, "%s{\n", space );
   
   if( msh->flag & (DK_MSH_FACETED | DK_MSH_PROCESS_NORMAL | DK_MSH_AVERAGED_NORMAL) )
   {
      fprintf( file, "%sflag    ( ", sspace );
      if( msh->flag & DK_MSH_FACETED )
         fprintf( file, "FACETED " );
      if( msh->flag & DK_MSH_PROCESS_NORMAL )
         fprintf( file, "PROCESS " );
      if( msh->flag & DK_MSH_AVERAGED_NORMAL )
         fprintf( file, "AVERAGED " );
      fprintf( file, ")\n" );
   }

   fprintf( file, "%sdiscontinuity  %*.*f\n", sspace,  DK_FM,  DK_FD, msh->discAngle );

   if( msh->nbVertices )
   {
      fprintf( file, "\n%svertices   %d\n", sspace, msh->nbVertices );
      fprintf( file, "%s{\n", sspace );
      for( i = 0, vrt = msh->vertices; i < msh->nbVertices; i++, vrt++ )
      {
         fprintf( file, "%s  [%1d] position  ", sspace, i );
         fprintf( file, "%*.*f  %*.*f  %*.*f\n",  DK_FM,  DK_FD, vrt->position.x,
                         DK_FM,  DK_FD, vrt->position.y,  DK_FM,  DK_FD, vrt->position.z );
         if( vrt->flag & DK_VRT_SELECTED )
            fprintf( file, "%s      flag     ( SELECTED )\n", sspace );
      }
      fprintf( file, "%s}\n", sspace );
   }

   if( msh->nbPolygons )
   {
      fprintf( file, "\n%spolygons   %d\n", sspace, msh->nbPolygons );
      fprintf( file, "%s{\n", sspace );
      for( i = 0, pol = msh->polygons; i < msh->nbPolygons; i++, pol++ )
      {
         fprintf( file, "%s  [%1d] nodes  %hd\n", sspace, i, pol->nbNodes );
         fprintf( file, "%s      {\n", sspace );
         for( j = 0; j < pol->nbNodes; j++ )
         {
            fprintf( file, "%s        [%1d] vertex  %d\n", sspace, j,
                                                         pol->nodes[j].vertex );
            if( DK_NORMAL )
            {
               fprintf( file, "%s            normal  %*.*f  %*.*f  %*.*f\n",
                       sspace,  DK_FM,  DK_FD, pol->nodes[j].normal.x,  DK_FM,  DK_FD,
                       pol->nodes[j].normal.y,  DK_FM,  DK_FD, pol->nodes[j].normal.z );
            }

            fprintf( file, "%s            uvTexture  %*.*f  %*.*f\n",
			sspace,  DK_FM,  DK_FD, pol->nodes[j].uTxt,
				 DK_FM,  DK_FD, pol->nodes[j].vTxt );

         }
         fprintf( file, "%s      }\n", sspace );

         fprintf( file, "%s      material  %hd\n", sspace, pol->material );
         if( pol->flag & DK_POL_SELECTED )
            fprintf( file, "%s      flag      ( SELECTED )\n", sspace );
   
      }
      fprintf( file, "%s}\n", sspace );
   }

   if( msh->nbEdges )
   {
      fprintf( file, "\n%sedges   %d\n", sspace, msh->nbEdges );
      fprintf( file, "%s{\n", sspace );
      for( i = 1, edg = &msh->edges[1]; i <= msh->nbEdges; i++, edg++ )
      {
         fprintf( file, "%s  [%1d] vertices  %d  %d\n", sspace, i,
                                          edg->vertices[0], edg->vertices[1] );
         if( edg->flag & (DK_EDG_SELECTED | DK_EDG_USER_VISIBLE | DK_EDG_USER_HIDDEN |
                          DK_EDG_USER_HARD) )
         {
            fprintf( file, "%s      flag    ( ", sspace );
            if( edg->flag & DK_EDG_SELECTED )
               fprintf( file, "SELECTED " );
            if( edg->flag & DK_EDG_USER_VISIBLE )
               fprintf( file, "VISIBLE " );
            if( edg->flag & DK_EDG_USER_HIDDEN )
               fprintf( file, "HIDDEN " );
            if( edg->flag & DK_EDG_USER_HARD )
               fprintf( file, "HARD " );
            fprintf( file, ")\n" );
         }
      }
      fprintf( file, "%s}\n", sspace );
   }

   fprintf( file, "%s}\n", space );
   return( TRUE );
}

DK_Mesh *DK_meshReadAscii(  FILE *file )
{
   DK_Mesh		*msh = NULL;
   DK_Vertex	*vrt = NULL;
   DK_Polygon	*pol = NULL;
   DK_Edge		*edg = NULL;
   char		 str[ 80 ], str2[ 80 ], str3[ 80 ];

   if( (msh = DK_meshAllocate()) == NULL )
      return( NULL );

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( msh );

   do
   {
      fscanf( file, "%s", str );

      if( strcmp( "flag", str ) == 0 )
      {
         fscanf( file, "%s", str2 );
         if( strcmp( str2, "(" ) != 0 )
            return( msh );
         do
         {
            fscanf( file, "%s", str2 );

            if( strcmp( "FACETED", str2 ) == 0 )
               msh->flag |= DK_MSH_FACETED;
            else if( strcmp( "PROCESS", str2 ) == 0 )
               msh->flag |= DK_MSH_PROCESS_NORMAL;
            else if( strcmp( "AVERAGED", str2 ) == 0 )
               msh->flag |= DK_MSH_AVERAGED_NORMAL;
         } while( strcmp( str2, ")" ) != 0 );
      }
      else if( strcmp( "discontinuity", str ) == 0 )
         fscanf( file, "%f", &msh->discAngle );
      else if( strcmp( "vertices", str ) == 0 )
      {
         fscanf( file, "%d", &msh->nbVertices );
         msh->vertices = DK_vertexAllocate( msh->nbVertices );
         do
         {
            fscanf( file, "%s", str2 );

            if( str2[0] == '[' )
            {
               if( vrt == NULL )
                  vrt = msh->vertices;
               else
                  vrt++;
            }
            else if( strcmp( str2, "position" ) == 0 )
               fscanf( file, "%f %f %f", &vrt->position.x, &vrt->position.y,
                                                           &vrt->position.z );
            else if( strcmp( str2, "flag" ) == 0 )
            {
               fscanf( file, "%s", str3 );
               if( strcmp( str3, "(" ) != 0 )
                  return( msh );
               do
               {
                  fscanf( file, "%s", str3 );

                  if( strcmp( "SELECTED", str3 ) == 0 )
                     vrt->flag |= DK_VRT_SELECTED;
               } while( strcmp( str3, ")" ) != 0 );
            }

         } while( strcmp( "}", str2 ) != 0 );
      }
      else if( strcmp( "polygons", str ) == 0 )
      {
         fscanf( file, "%d", &msh->nbPolygons );
         msh->polygons = DK_polygonAllocate( msh->nbPolygons );
         do
         {
            fscanf( file, "%s", str2 );

            if( str2[0] == '[' )
            {
               if( pol == NULL )
                  pol = msh->polygons;
               else
                  pol++;
            }
            else if( strcmp( str2, "nodes" ) == 0 )
               polygonNodeReadAscii( file, pol );
            else if( strcmp( str2, "material" ) == 0 )
               fscanf( file, "%hd", &pol->material );
            else if( strcmp( str2, "flag" ) == 0 )
            {
               fscanf( file, "%s", str3 );
               if( strcmp( str3, "(" ) != 0 )
                  return( msh );
               do
               {
                  fscanf( file, "%s", str3 );

                  if( strcmp( "SELECTED", str3 ) == 0 )
                     pol->flag |= DK_POL_SELECTED;
               } while( strcmp( str3, ")" ) != 0 );
            }
         } while( strcmp( "}", str2 ) != 0 );
      }
      else if( strcmp( "edges", str ) == 0 )
      {
         fscanf( file, "%d", &msh->nbEdges );
         msh->edges = DK_edgeAllocate( msh->nbEdges );
         do
         {
            fscanf( file, "%s", str2 );

            if( str2[0] == '[' )
            {
               if( edg == NULL )
                  edg = &msh->edges[1];
               else
                  edg++;
            }
            else if( strcmp( str2, "vertices" ) == 0 )
               fscanf( file, "%d %d", &edg->vertices[0], &edg->vertices[1] );
            else if( strcmp( str2, "flag" ) == 0 )
            {
               fscanf( file, "%s", str3 );
               if( strcmp( str3, "(" ) != 0 )
                  return( msh );
               do
               {
                  fscanf( file, "%s", str3 );

                  if( strcmp( "SELECTED", str3 ) == 0 )
                     edg->flag |= DK_EDG_SELECTED;
                  else if( strcmp( "VISIBLE", str3 ) == 0 )
                     edg->flag |= DK_EDG_USER_VISIBLE;
                  else if( strcmp( "HIDDEN", str3 ) == 0 )
                     edg->flag |= DK_EDG_USER_HIDDEN;
                  else if( strcmp( "HARD", str3 ) == 0 )
                     edg->flag |= DK_EDG_USER_HARD;
               } while( strcmp( str3, ")" ) != 0 );
            }
         } while( strcmp( "}", str2 ) != 0 );
      }
   } while( strcmp( str, "}" ) != 0 );

   return( msh );
}

static void polygonNodeReadAscii(  FILE *file, DK_Polygon *pol )
{
   char		 str[ 80 ];
   int		 i;

   fscanf( file, "%hd", &pol->nbNodes );
   pol->nodes = (DK_PolyNode *) calloc( pol->nbNodes, sizeof( DK_PolyNode ) );
   for( i = 0; i < pol->nbNodes; i++ )
   {
      pol->nodes[i].vertex = -1;
      pol->nodes[i].normal.x = 0.0;
      pol->nodes[i].normal.y = 0.0;
      pol->nodes[i].normal.z = 0.0;
      pol->nodes[i].uTxt = pol->nodes[i].vTxt = 0.0;
   }

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return;

   i = -1;
   do
   {
      fscanf( file, "%s", str );

      if( str[0] == '[' )
         i++;
      else if( strcmp( "vertex", str ) == 0 )
         fscanf( file, "%d", &pol->nodes[i].vertex );
      else if( strcmp( "normal", str ) == 0 )
         fscanf( file, "%f %f %f", &pol->nodes[i].normal.x,
                       &pol->nodes[i].normal.y, &pol->nodes[i].normal.z );
      else if( strcmp( "uvTexture", str ) == 0 )
         fscanf( file, "%f %f", &pol->nodes[i].uTxt, &pol->nodes[i].vTxt );

   } while( strcmp( str, "}" ) != 0 );
}
