/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyMaterial.c,v $
   $Revision: 1.24.14.1 $ $Date: 1995/06/09 22:00:06 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's material functions,
   its content is sufficient to read, write and manage material data structure
   into SOFTIMAGE's file format...

   Written by: Laurent Lauzon

   (c) Copyright 1990, 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinyMaterial.h"
#include "tinyFcurve.h"
#include "tinySceneToken.h"
#include "tinyUtils.h"
#include "tinyMental.h"

DK_String  DK_MaterialTypeName[] =
   {
      "--------",
      "CONSTANT",
      "FLAT",
      "LAMBERT",
      "PHONG",
      "BLINN",
      "SHADOW"
   };

/******************************************************************************/
/*** Static function prototypes ***********************************************/
/******************************************************************************/

static DK_MaterialType	DK_materialGetTypeByName( DK_String );
static DK_Boolean	DK_materialWriteAsciiFile( DK_String, DK_Material * );
static DK_Boolean	DK_materialWriteBinaryFile( DK_String, DK_Material * );
static DK_Material	*DK_materialReadAsciiFile( DK_String );
static DK_Material	*DK_materialReadBinaryFile( DK_String );

/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

DK_Material *DK_materialAllocate( void )
{
   DK_Material *mtrl;
   int		i;

   if( mtrl = (DK_Material *) calloc( 1, sizeof( DK_Material ) ) )
   {
      mtrl->name         = NULL;
      mtrl->prefix       = NULL;
      mtrl->type         = DK_MAT_PHONG;
      mtrl->ambient.r    = mtrl->ambient.g  = mtrl->ambient.b = 0.5;
      mtrl->diffuse.r    = mtrl->diffuse.g  = mtrl->diffuse.b = 0.75;
      mtrl->specular.r   = mtrl->specular.g = mtrl->specular.b = 1.0;
      mtrl->attenuate.r   = mtrl->attenuate.g = mtrl->attenuate.b = 0.0;
      mtrl->exponent     = 20.0;
      mtrl->reflectivity = 0.0;
      mtrl->reflectivityJitter = 0.0;
      mtrl->transparency = 0.0;
      mtrl->transparencyJitter = 0.0;
      mtrl->refracIndex  = 1.0;
      mtrl->motiontransp = 0.5;
      mtrl->nbTextures   = 0;
      mtrl->textures     = NULL;
      mtrl->nbTextures3D = 0;
      mtrl->textures3D   = NULL;
      mtrl->glow = FALSE;
      mtrl->coc = 0.0;
      for( i = 0; i < DK_FCV_MAT_NUM; i ++ )
      {
	 mtrl->fcv[i]		= NULL;
	 mtrl->active_fcv[i].active = FALSE;
	 mtrl->active_fcv[i].fcv= NULL;
      }

      mtrl->staticblur = FALSE;
      mtrl->width = 1.0;
      mtrl->decay.value = 3.0;
      for (i = 0; i < DK_DECAY_TAB_SIZE; i++)
      {
	 mtrl->decay.table[i][0] = 0.0;
	 mtrl->decay.table[i][1] = 0.0;
      }
      mtrl->vol_shader	 = NULL;
      mtrl->mat_shader	 = NULL;
      mtrl->shdw_shader	 = NULL;
      
      mtrl->next         = NULL;

      mtrl->constr_list	 = NULL;

      mtrl->schem_posx = 0.0;
      mtrl->schem_posy = 0.0;
      mtrl->schem_userppos = 0;
   }

   return( mtrl );
}

void DK_materialDispose( DK_Material **mtrl )
{
   if( mtrl )
   {
      DK_Material *material = *mtrl;

      while( material )
      {
         DK_Material *save = material;

         _DK_FREE( material->name, -1 );
         _DK_FREE( material->prefix, -1 );
         if( material->textures )
             DK_textureDispose( &material->textures );
         if( material->textures3D )
             DK_texture3DDispose( &material->textures3D );
         DK_fcurveArrayDispose( material->fcv, DK_FCV_MAT_NUM );
         DK_actFcurveArrayDispose( material->active_fcv, DK_FCV_MAT_NUM );
         DK_udfDispose( &material->mat_shader );
         DK_udfDispose( &material->vol_shader );
         DK_udfDispose( &material->shdw_shader );
	 DK_constraintDispose( &material->constr_list );
         material = material->next;
         free( save );
      }

      *mtrl = NULL;
   }
}

DK_Material *DK_materialReadBinary(  FILE *file, DK_FileProt *prot )
{
   DK_Material	*mtrl;
   short	 ext, i, nb, tok, ind = 0;
   DK_Texture	*txt;
   DK_Texture3D	*txt3D;
   DK_Colour24	 col;
   float	 value;
   DK_String	 name;
   DK_Boolean	 tmpb;
   float	 tmpf;

   if( (mtrl = DK_materialAllocate()) == NULL )
      return( NULL );

   if( (prot != NULL) &&( prot->material != NULL) )
      tok = prot->material[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );
   while( tok != 0 /* End of material definition */ )
   {
      switch( tok )
      {
         case 1: /* material name */
            mtrl->name = DK_stringReadBinary( file );
            break;

         case 2: /* shading DK_model */
            DK_byteswap_fread( &nb, 2, 1, file );
            mtrl->type = (DK_MaterialType) nb;
            break;

         case 3: /* ambient color */
            DK_RGB_fread( &mtrl->ambient, sizeof( DK_Colour24 ), 1, file );
            break;

         case 4: /* diffuse color */
            DK_RGB_fread( &mtrl->diffuse, sizeof( DK_Colour24 ), 1, file );
            break;

         case 5: /* specular color */
            DK_RGB_fread( &mtrl->specular, sizeof( DK_Colour24 ), 1, file );
            break;

         case 6: /* specular exponent */
            DK_byteswap_fread( &mtrl->exponent, 4, 1, file );
            break;

         case 7: /* transparency */
            DK_byteswap_fread( &mtrl->transparency, 4, 1, file );
            break;

         case 8: /* transparency jitter */
            DK_byteswap_fread( &value, 4, 1, file );
            break;

         case 9: /* reflectivity color OLD WAY */
            DK_RGB_fread( &col, sizeof( DK_Colour24 ), 1, file );
            mtrl->reflectivity = ( col.r + col.g + col.b ) / 3.0;
            break;

         case 10: /* reflectivity jitter */
            DK_byteswap_fread( &value, 4, 1, file );
            break;

         case 11: /* refractive index */
            DK_byteswap_fread( &mtrl->refracIndex, 4, 1, file );
            break;

         case 13: /* reflectivity value */
            DK_byteswap_fread( &mtrl->reflectivity, 4, 1, file );
            break;

         case 14: /* glow */
            DK_byteswap_fread( &nb, 2, 1, file );
	    mtrl->glow = nb;
            break;

         case 15: /* coc */
            DK_byteswap_fread( &value, 4, 1, file );
	    mtrl->coc = value;
            break;

         case 16: /* local textures list */
            DK_byteswap_fread( &mtrl->nbTextures, 2, 1, file );
            txt = NULL;
            for( i = 0; i < mtrl->nbTextures; i++ )
            {
               DK_byteswap_fread( &ext, 2, 1, file );
               if( ext )  /* extern  texture */
               {
                  name = DK_stringReadBinary( file );
                  free( name );
               }
               else if( txt == NULL )
                  txt = mtrl->textures = DK_textureReadBinary( file, prot );
               else
               {
                  txt->next = DK_textureReadBinary( file, prot );
                  txt = txt->next;
               }
            }
            break;

         case 17: /* local 3D textures list */
            DK_byteswap_fread( &mtrl->nbTextures3D, 2, 1, file );
            txt3D = NULL;
            for( i = 0; i < mtrl->nbTextures3D; i++ )
            {
               DK_byteswap_fread( &ext, 2, 1, file );
               if( ext )  /* extern  texture */
               {
                  name = DK_stringReadBinary( file );
                  free( name );
               }
               else if( txt3D == NULL )
                  txt3D = mtrl->textures3D = DK_texture3DReadBinary( file, prot );
               else
               {
                  txt3D->next = DK_texture3DReadBinary( file, prot );
                  txt3D = txt3D->next;
               }
            }
            break;

	 case 18: /* static blur */
	    DK_byteswap_fread( &mtrl->staticblur, 2, 1, file );
	    break;

	 case 19: /* static blur width */
	    DK_byteswap_fread( &mtrl->width, 4, 1, file );
	    break;

	 case 20: /* static blur decay */
	    DK_byteswap_fread( &mtrl->decay.value, 4, 1, file );
	    break;

         case 21: /* 21: material surface shaders */
            mtrl->mat_shader = DK_udfReadBinary( file, prot );
            break;

         case 22: /* 22: volume shaders */
            mtrl->vol_shader = DK_udfReadBinary( file, prot );
            break;

         case 23: /* 23: shadow shaders */
            mtrl->shdw_shader = DK_udfReadBinary( file, prot );
            break;

         default:
            DK_fileSkipField( file, tok );
      }
      if( (prot != NULL) &&( prot->material != NULL) )
         tok = prot->material[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   return( mtrl );
}

DK_Boolean DK_materialWriteAscii(  FILE *file, DK_Material *mtrl, DK_String space, short num )
{
   char		 sspace[ 256 ];
   int		 i;
   DK_Texture	*txt;
   DK_Texture3D	*tx3D;

   sprintf( sspace, "%s  ", space );

   fprintf( file, "\n%smaterial     [%1hd]\n", space, num );
   fprintf( file, "%s{\n", space );
   fprintf( file, "%s  name          \"%s\"\n", space, mtrl->name );
   fprintf( file, "%s  type          %s\n", space,
                   DK_MaterialTypeName[ mtrl->type ] );
   if( mtrl->type == DK_MAT_LAMBERT || mtrl->type == DK_MAT_PHONG )
      fprintf( file, "%s  ambient       %*.*f  %*.*f  %*.*f\n", space,  DK_FM,  DK_FD,
            mtrl->ambient.r,  DK_FM,  DK_FD, mtrl->ambient.g,  DK_FM,  DK_FD, mtrl->ambient.b );
   if( mtrl->type != DK_MAT_SHADOW )
      fprintf( file, "%s  diffuse       %*.*f  %*.*f  %*.*f\n", space,  DK_FM,  DK_FD,
            mtrl->diffuse.r,  DK_FM,  DK_FD, mtrl->diffuse.g,  DK_FM,  DK_FD, mtrl->diffuse.b );
   if( mtrl->type != DK_MAT_SHADOW )
   {
      fprintf( file, "%s  specular      %*.*f  %*.*f  %*.*f\n", space,  DK_FM,  DK_FD,
         mtrl->specular.r,  DK_FM,  DK_FD, mtrl->specular.g,  DK_FM,  DK_FD, mtrl->specular.b );
      fprintf( file, "%s  exponent      %*.*f\n", space,  DK_FM,  DK_FD,
                                                   mtrl->exponent );
   }
   fprintf( file, "%s  reflectivity  %*.*f\n", space,  DK_FM,  DK_FD,
                                                 mtrl->reflectivity );
   fprintf( file, "%s  transparency  %*.*f\n", space,  DK_FM,  DK_FD,
                                               mtrl->transparency );
   fprintf( file, "%s  refracIndex   %*.*f\n", space,  DK_FM,  DK_FD,
                                               mtrl->refracIndex );
   fprintf( file, "%s  glow   %hd\n", space, mtrl->glow );
   fprintf( file, "%s  coc   %*.*f\n", space,  DK_FM,  DK_FD, mtrl->coc );

   if( mtrl->staticblur )
   {
      fprintf( file, "%s  staticBlur   %hd\n", space, mtrl->staticblur );
   
      fprintf( file, "%s  blurWidht   %*.*f\n", space,  DK_FM,  DK_FD, 
					    mtrl->width );
      fprintf( file, "%s  blurDecay   %*.*f\n", space,  DK_FM,  DK_FD, 
					    mtrl->decay.value );
   }

   if ( mtrl->mat_shader )
   {
     DK_udfWriteAscii( file, mtrl->mat_shader, space, "mat_shader");
   }
   if ( mtrl->vol_shader )
   {
     DK_udfWriteAscii( file, mtrl->vol_shader, space, "vol_shader" );
   }
   if ( mtrl->shdw_shader )
   {
     DK_udfWriteAscii( file, mtrl->shdw_shader, space, "shdw_shader" );
   }

   for( i = 0, txt = mtrl->textures; i < mtrl->nbTextures; i++, txt = txt->next)
      DK_textureWriteAscii( file, txt, sspace, i );
   for( i = 0, tx3D = mtrl->textures3D; i < mtrl->nbTextures3D;
                                                         i++, tx3D = tx3D->next)
      DK_texture3DWriteAscii( file, tx3D, sspace, i );
      
   fprintf( file, "%s}\n", space );
   return( TRUE );
}

DK_Boolean DK_materialWriteBinary(  FILE *file,
			     DK_Material *mtrl,
			     DK_FileProt *prot )
{
   short		 i, nb, ext = 0;
   DK_Texture		*txt;
   DK_Texture3D		*txt3D;

   for( i = 0; i < prot->nbMaterial; i++ )
   {
      switch( prot->material[ i ] )
      {
         case 0:	/* end of material definition */
            break;

         case 1:	/* material name */
            DK_stringWriteBinary( file, mtrl->name );
            break;

         case 2:	/* shading DK_model */
            nb = mtrl->type;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 3:	/* ambient color */
            DK_RGB_fwrite( &mtrl->ambient, sizeof( DK_Colour24 ), 1, file );
            break;

         case 4:	/* diffuse color */
            DK_RGB_fwrite( &mtrl->diffuse, sizeof( DK_Colour24 ), 1, file );
            break;

         case 5:	/* specular color */
            DK_RGB_fwrite( &mtrl->specular, sizeof( DK_Colour24 ), 1, file );
            break;

         case 6:	/* specular exponent */
            DK_byteswap_fwrite( &mtrl->exponent, 4, 1, file );
            break;

         case 7:	/* transparency */
            DK_byteswap_fwrite( &mtrl->transparency, 4, 1, file );
            break;

         case 13:	/* reflectivity */
            DK_byteswap_fwrite( &mtrl->reflectivity, 4, 1, file );
            break;

         case 11:	/* refractive index */
            DK_byteswap_fwrite( &mtrl->refracIndex, 4, 1, file );
            break;

         case 14: /* glow */
            nb = mtrl->glow;
            DK_byteswap_fwrite( &nb, 2, 1, file );
            break;

         case 15: /* coc */
            DK_byteswap_fwrite( &mtrl->coc, 4, 1, file );
            break;
         case 16:	/* local texture list */
            DK_byteswap_fwrite( &mtrl->nbTextures, 2, 1, file );
            if ( mtrl->nbTextures ) 
            {
              for( txt = mtrl->textures; txt != NULL; txt = txt->next )
              {
                  DK_byteswap_fwrite( &ext, 2, 1, file );
                  DK_textureWriteBinary( file, txt, prot );
              }
            }
            break;

         case 17:	/* 17 : local 3Dtexture list */
            DK_byteswap_fwrite( &mtrl->nbTextures3D, 2, 1, file );
            if ( mtrl->nbTextures3D ) 
            {
              for( txt3D =mtrl->textures3D; txt3D != NULL; txt3D = txt3D->next)
              {
                 DK_byteswap_fwrite( &ext, 2, 1, file );
                 DK_texture3DWriteBinary( file, txt3D, prot );
              }
            }
            break;
	    
	
	case 18:    /* 18: static blur */
	   DK_byteswap_fwrite( &mtrl->staticblur, 2, 1, file );
	    break;
	    
	
	case 19:    /* 19: static blur width */
	    DK_byteswap_fwrite( &mtrl->width, 4, 1, file );
	    break;
	
	
	case 20:    /* 20: static blur decay value */
	    DK_byteswap_fwrite( &mtrl->decay.value, 4, 1, file );
	    break;
	
	
	case 21:    /* 21: udf Shaders */
	   DK_udfWriteBinary( file, prot, mtrl->mat_shader );
	   break;

	 case 22:   /* 22: udf Shaders */
	   DK_udfWriteBinary( file, prot, mtrl->vol_shader );
	   break;
	 
	 case 23:   /* 23: udf Shaders */
	   DK_udfWriteBinary( file, prot , mtrl->shdw_shader );
	   break;

         default:
	    if ( DK_option.verbose == TRUE)
                fprintf(  DK_option.msgFd, "Error: %s %d): illegal token: %hd\n",
                                      __FILE__, __LINE__, prot->material[ i ] );
            break;
      }
   }

   return( TRUE );
}

DK_Material *DK_materialReadAscii(  FILE *file )
{
   DK_Material	*mtrl;
   char		 str[ 80 ], str2[ 80 ];
   DK_Texture	*txt = NULL;
   DK_Texture3D	*txt3D = NULL;

   if( (mtrl = DK_materialAllocate()) == NULL )
      return( NULL );

   /* get rid of number */
   fscanf( file, "%s", str );

   fscanf( file, "%s", str );
   if( strcmp( str, "{" ) != 0 )
      return( mtrl );

   do
   {
      fscanf( file, "%s", str );
      if( strcmp( str, "name" ) == 0 )
         mtrl->name = DK_stringReadAscii( file );
      else if( strcmp( str, "type" ) == 0 )
      {
         fscanf( file, "%s", str2 );
         mtrl->type = DK_materialGetTypeByName( str2 );
      }
      else if( strcmp( str, "ambient" ) == 0 )
         fscanf( file, "%f %f %f", &mtrl->ambient.r, &mtrl->ambient.g,
                                                     &mtrl->ambient.b );
      else if( strcmp( str, "diffuse" ) == 0 )
         fscanf( file, "%f %f %f", &mtrl->diffuse.r, &mtrl->diffuse.g,
                                                     &mtrl->diffuse.b );
      else if( strcmp( str, "specular" ) == 0 )
         fscanf( file, "%f %f %f", &mtrl->specular.r, &mtrl->specular.g,
                                                      &mtrl->specular.b );
      else if( strcmp( str, "exponent" ) == 0 )
         fscanf( file, "%f", &mtrl->exponent );
      else if( strcmp( str, "reflectivity" ) == 0 )
         fscanf( file, "%f", &mtrl->reflectivity );
      else if( strcmp( str, "transparency" ) == 0 )
         fscanf( file, "%f", &mtrl->transparency );
      else if( strcmp( str, "refracIndex" ) == 0 )
         fscanf( file, "%f", &mtrl->refracIndex );
      else if( strcmp( str, "glow" ) == 0 )
         fscanf( file, "%d", &mtrl->glow );
      else if( strcmp( str, "coc" ) == 0 )
         fscanf( file, "%f", &mtrl->coc );
      else if( strcmp( str, "staticBlur" ) == 0 )
         fscanf( file, "%d", &mtrl->staticblur );
      else if( strcmp( str, "blurWidht" ) == 0 )
         fscanf( file, "%f", &mtrl->width );
      else if( strcmp( str, "blurDecay" ) == 0 )
         fscanf( file, "%f", &mtrl->decay.value );
      else if( strcmp( str, "mat_shader" ) == 0 )
	 mtrl->mat_shader = DK_udfReadAscii( file );
      else if( strcmp( str, "vol_shader" ) == 0 )
	 mtrl->vol_shader = DK_udfReadAscii( file );
      else if( strcmp( str, "shdw_shader" ) == 0 )
	 mtrl->shdw_shader = DK_udfReadAscii( file );
      else if( strcmp( str, "texture" ) == 0 )
      {
         if( txt == NULL )
            txt = mtrl->textures = DK_textureReadAscii( file );
         else
         {
            txt->next = DK_textureReadAscii( file );
            txt = txt->next;
         }
         mtrl->nbTextures++;
      }
      else if( strcmp( "texture3D", str ) == 0 )
      {
         if( txt3D == NULL )
            txt3D = mtrl->textures3D = DK_texture3DReadAscii( file );
         else
         {
            txt3D->next = DK_texture3DReadAscii( file );
            txt3D = txt3D->next;
         }
         mtrl->nbTextures3D++;
      }
   } while( strcmp( "}", str ) != 0 );

   return( mtrl );
}


static DK_MaterialType DK_materialGetTypeByName( DK_String name )
{
   DK_MaterialType	 type = DK_MAT_CONSTANT;
   int		 i;


   for( i = 0; i < 7; i++ )
      if( strcmp( name, DK_MaterialTypeName[ i ] ) == 0 )
         type = (DK_MaterialType) i;

   return( type );
}

DK_Material *DK_materialReadFile( DK_String filename )
{
   char		 buf[256];
   long          id;
    FILE		*file;

   sprintf( buf, "%s.mtr", filename );
   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( NULL );

   DK_byteswap_fread( &id, 4, 1, file );
   fclose( file );

   if( id == 'MTRL' )
      return( DK_materialReadAsciiFile( filename ) );
   else if( id == DK_BINARY_MAGIC_NUMBER || id == DK_MAGIC_MAT_BIN )
      return( DK_materialReadBinaryFile( filename ) );
   else
      return( NULL );
}

DK_Boolean DK_materialWriteFile( DK_String filename, DK_Material *mtrl , DK_FileFormat fmt )
{
   if( fmt ==  DK_FIL_ASCII )
      return( DK_materialWriteAsciiFile( filename, mtrl ) );
   else if( fmt ==  DK_FIL_BINARY )
      return( DK_materialWriteBinaryFile( filename, mtrl ) );
   else
      return( FALSE );
}

static DK_Boolean DK_materialWriteAsciiFile( DK_String filename, DK_Material *mtrl )
{
    FILE		*file;
   char		buf[256];
   DK_Boolean      success;

   sprintf( buf, "%s.mtr", filename );
   if( ( file = fopen( buf, "wb" ) ) == NULL )
      return( FALSE );

   success = DK_headerWriteAscii( file, "MTRL" );
   success &= DK_materialWriteAscii( file, mtrl, "", 0 );
   /* Verify if file was saved correctly */
   fflush( file );
   success = (ferror( file ) == 0 );

   fclose( file );
   return( success );
}

DK_Boolean DK_fdmaterialWriteBinaryFile( FILE *file, DK_Material *mtrl )
{
   char		 buf[256];
   DK_FileProt	*prot;
   DK_Boolean	 success;

   DK_newheaderWriteBinary( file, DK_MAGIC_MAT_BIN, NULL );
   DK_prototypeInitMaterial( &prot, TRUE );
    /* Because those are included in material they have to be
     * initialized in here too! -MKL-
     */
    if( prot->texture3D == NULL )
       DK_prototypeInitTexture3D( &prot, FALSE );
    if( prot->texture == NULL )
       DK_prototypeInitTexture( &prot, FALSE );
   
   DK_prototypeWrite( file, prot );
   DK_String_fwrite( "MTRL", 4, 1, file );
   success = DK_materialWriteBinary( file, mtrl, prot );

   DK_prototypeDispose( &prot );
   return( success );
}

static DK_Boolean DK_materialWriteBinaryFile( DK_String filename, DK_Material *mtrl )
{
   FILE		*file;
   char		 buf[256];
   DK_Boolean	 success;

   sprintf( buf, "%s.mtr", filename );
   if( ( file = fopen( buf, "wb" ) ) == NULL )
      return( FALSE );

   DK_fdmaterialWriteBinaryFile( file, mtrl );

   /* Verify if file was saved correctly */
   fflush( file );
   success = (ferror( file ) == 0 );

   fclose( file );
   return( success );
}

static DK_Material *DK_materialReadAsciiFile( DK_String filename )
{
   char		 str[ 80 ];
   FILE		*file;
   DK_Material	*mtrl;
   long		 id;
   float	 version = 0.0;

   sprintf( str, "%s.mtr", filename );
   if( ( file = fopen( str, "rb" ) ) == NULL )
      return( NULL );

   if( (id = DK_headerReadAscii( file, &version )) != 'MTRL' )
      return( NULL );
   fscanf( file, "%s", str );
   if( strcmp( str , "material" ) == 0 )
      mtrl = DK_materialReadAscii( file );

   fclose( file );

   return( mtrl );
}

DK_Material *DK_fdmaterialReadBinaryFile( FILE *file )
{
   float	 version = 0.0;
   long		 id;
   DK_FileProt	*prot;
   DK_Material	*mtrl;

   id = DK_headerReadBinary( file, &version, NULL );
   if( id == 'PROT' )
   {
      prot = DK_prototypeRead( file );
      DK_byteswap_fread( &id, 4, 1, file );
   }

   if( id == 'MTRL' )
      mtrl = DK_materialReadBinary( file, prot );

   DK_prototypeDispose( &prot );

   return( mtrl );
}

static DK_Material *DK_materialReadBinaryFile( DK_String filename )
{
   char		 buf[256];
   FILE		*file;
   DK_Material	*mtrl;

   sprintf( buf, "%s.mtr", filename );
   if( ( file = fopen( buf, "rb" ) ) == NULL )
      return( NULL );

   mtrl = DK_fdmaterialReadBinaryFile( file );

   fclose( file );
   return( mtrl );
}

/*******************************************************************************

   $$L DK_materialReadAsciiAnimation

   This function reads a material from old v2.5 scenes.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_materialReadAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Material *mat	/* material pointer */
    )
{
    char *cptr;
    int first = TRUE;

    DK_in_animToken(file);
    while (TRUE)
    {
        if (strcmp(DK_A_MAT_NAME_TOKEN, DK_animToken) == 0)
	{
	    cptr = DK_stringReadAsciiAnimation(file);
	    if (cptr != NULL && mat != NULL)
		mat->name = cptr;
	    DK_in_animToken(file);
	}
        else if (strcmp(DK_A_MAT_AMB_TOKEN, DK_animToken) == 0)
	{
	    if (mat != NULL)
	        DK_value3ReadAsciiAnimation(file, 
		    &mat->ambient.r, &mat->ambient.g, &mat->ambient.b,
		    &mat->fcv[DK_FCV_MAT_AMB_R], &mat->fcv[DK_FCV_MAT_AMB_G],
		    &mat->fcv[DK_FCV_MAT_AMB_B]);
	    else
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
        }
        else if (strcmp(DK_A_MAT_DIFF_TOKEN, DK_animToken) == 0)
	{
	    if (mat != NULL)
	        DK_value3ReadAsciiAnimation(file,
		    &mat->diffuse.r, &mat->diffuse.g, &mat->diffuse.b,
		    &mat->fcv[DK_FCV_MAT_DIFF_R], &mat->fcv[DK_FCV_MAT_DIFF_G],
		    &mat->fcv[DK_FCV_MAT_DIFF_B]);
	    else
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
        }
        else if (strcmp(DK_A_MAT_SPEC_TOKEN, DK_animToken) == 0)
	{
	    if (mat != NULL)
	    {
	        DK_value3ReadAsciiAnimation(file,
		    &mat->specular.r, &mat->specular.g, &mat->specular.b,
		    &mat->fcv[DK_FCV_MAT_SPEC_R], &mat->fcv[DK_FCV_MAT_SPEC_G],
		    &mat->fcv[DK_FCV_MAT_SPEC_B]);
	        DK_valueReadAsciiAnimation(file, &mat->exponent, &mat->fcv[DK_FCV_MAT_EXPN],
		    FALSE);
	    }
	    else
	    {
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
	        DK_valueReadAsciiAnimation(file, NULL, NULL, FALSE);
	    }
        }
        else if (strcmp(DK_A_MAT_ATTEN_TOKEN, DK_animToken) == 0)
	{
	    if (mat != NULL)
	        DK_value3ReadAsciiAnimation(file,
		    &mat->attenuate.r, &mat->attenuate.g, &mat->attenuate.b,
		    &mat->fcv[DK_FCV_MAT_ATTEN_R], &mat->fcv[DK_FCV_MAT_ATTEN_G],
		    &mat->fcv[DK_FCV_MAT_ATTEN_B]);
	    else
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp(DK_A_MAT_TRANSP_TOKEN, DK_animToken) == 0)
	{
	    if (mat != NULL) 
	    {
	        DK_valueReadAsciiAnimation(file, &mat->transparency, 
			&mat->fcv[DK_FCV_MAT_TRSP], TRUE);
	        DK_valueReadAsciiAnimation(file, &mat->transparencyJitter, 
			&mat->fcv[DK_FCV_MAT_TRSP_J], FALSE);
	    }
	    else
	    {
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	        DK_valueReadAsciiAnimation(file, NULL, NULL, FALSE);
	    }
        }
	else if (strcmp(DK_A_MAT_REFL_TOKEN, DK_animToken) == 0)
	{
	    if (mat != NULL)
	    {
	        DK_valueReadAsciiAnimation(file, &mat->reflectivity, 
			&mat->fcv[DK_FCV_MAT_REFL], TRUE);
	        DK_valueReadAsciiAnimation(file, &mat->reflectivityJitter,
			&mat->fcv[DK_FCV_MAT_REFL_J], FALSE);
	    }
	    else
	    {
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	        DK_valueReadAsciiAnimation(file, NULL, NULL, FALSE);
	    }
	}
        else if (strcmp(DK_A_MAT_RIND_TOKEN, DK_animToken) == 0)
	{
	    if (mat != NULL)
	        DK_valueReadAsciiAnimation(file, &mat->refracIndex,
			&mat->fcv[DK_FCV_MAT_RIND], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
        }
        else if (strcmp(DK_A_MAT_MTRANSP_TOKEN, DK_animToken) == 0)
	{
	    if (mat != NULL)
	        DK_valueReadAsciiAnimation(file, &mat->motiontransp, NULL, TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
	else
	{
	    if (first)
		DK_in_animToken(file);
	    return(TRUE);
	}
	first = FALSE;
    }
}

/*******************************************************************************

   $$L DK_materialWriteAsciiAnimation

   This function writes a material.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_materialWriteAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Material *mat	/* material pointer */
    )
{
    if (mat == NULL)
	return(FALSE);

    DK_incIndent(1);
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_MAT_NAME_TOKEN);
    _DK_OUT_STRING(mat->name);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_MAT_AMB_TOKEN);
    DK_value3WriteAsciiAnimation(file, mat->ambient.r, mat->ambient.g, mat->ambient.b,
	mat->fcv[DK_FCV_MAT_AMB_R], mat->fcv[DK_FCV_MAT_AMB_G], 
	mat->fcv[DK_FCV_MAT_AMB_B]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_MAT_DIFF_TOKEN);
    DK_value3WriteAsciiAnimation(file, mat->diffuse.r, mat->diffuse.g, mat->diffuse.b,
	mat->fcv[DK_FCV_MAT_DIFF_R], mat->fcv[DK_FCV_MAT_DIFF_G], 
	mat->fcv[DK_FCV_MAT_DIFF_B]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_MAT_SPEC_TOKEN);
    DK_value3WriteAsciiAnimation(file, mat->specular.r, mat->specular.g, mat->specular.b,
	mat->fcv[DK_FCV_MAT_SPEC_R], mat->fcv[DK_FCV_MAT_SPEC_G], 
	mat->fcv[DK_FCV_MAT_SPEC_B]);
    DK_valueWriteAsciiAnimation(file, mat->exponent, mat->fcv[DK_FCV_MAT_EXPN]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_MAT_ATTEN_TOKEN);
    DK_value3WriteAsciiAnimation(file,mat->attenuate.r, mat->attenuate.g, mat->attenuate.b,
	mat->fcv[DK_FCV_MAT_ATTEN_R], mat->fcv[DK_FCV_MAT_ATTEN_G], 
	mat->fcv[DK_FCV_MAT_ATTEN_B]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_MAT_TRANSP_TOKEN);
    DK_valueWriteAsciiAnimation(file, mat->transparency, mat->fcv[DK_FCV_MAT_TRSP]);
    DK_valueWriteAsciiAnimation(file, mat->transparencyJitter, mat->fcv[DK_FCV_MAT_TRSP_J]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_MAT_REFL_TOKEN);
    DK_valueWriteAsciiAnimation(file, mat->reflectivity, mat->fcv[DK_FCV_MAT_REFL]);
    DK_valueWriteAsciiAnimation(file, mat->reflectivityJitter, mat->fcv[DK_FCV_MAT_REFL_J]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_MAT_RIND_TOKEN);
    DK_valueWriteAsciiAnimation(file, mat->refracIndex, mat->fcv[DK_FCV_MAT_RIND]);

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_MAT_MTRANSP_TOKEN );
    DK_valueWriteAsciiAnimation(file, mat->motiontransp, NULL);

    DK_incIndent(-1);
    return(TRUE);
}

