/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyLight.c,v $
   $Revision: 1.21.14.1 $ $Date: 1995/06/09 22:00:04 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's light functions.

   Written by: Colin Hui

   (c) Copyright 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinySoftType.h"
#include "tinyLight.h"
#include "tinySceneToken.h"
#include "tinyUtils.h"

/*******************************************************************************

   $$L DK_lightAllocate

   This function allocates a light structure.

   Returned Value: DK_Light structure

*******************************************************************************/

DK_Light *DK_lightAllocate( void )
{
   DK_Light *lit;
   int i;

   if( lit = (DK_Light *) calloc( 1, sizeof( DK_Light ) ) )
   {
      lit->name  = NULL;
      lit->prefix  = NULL;
      lit->type = DK_LIGHT_POINT;

      _DK_VectorInit(lit->position, 0.0, 0.0, 0.0);
      lit->colour.r = 1.0;
      lit->colour.g = 1.0;
      lit->colour.b = 1.0;
      lit->startFalloff = 10000.0;
      lit->endFalloff = 10000.0;
      lit->shadowFactor = 0.75;

      lit->pos_constraint_name = NULL;
      lit->pos_constraint_type = 0;
      lit->int_constraint_name = NULL;
      lit->int_constraint_type = 0;
      lit->pos_path = NULL;
      lit->int_path = NULL;
      lit->pos_path_name = NULL;
      lit->int_path_name = NULL;

      _DK_VectorInit(lit->spotInterest, 0.0, 0.0, 0.0);
      lit->spotConeAngle = 0.523598;	/* 30 degrees */
      lit->spotSpreadAngle = 0.087266;	/* 5 degrees */

      lit->shadowType = DK_LIGHT_NOSHADOW;
      lit->shadowMapRes = 500;
      lit->shadowFilterSize = 10;
      lit->shadowFilterStep = 2;
      lit->shadowFadeStart = 0.5;
      lit->shadowFadeRate = 1.0;

      /* display */
      lit->wire_colour = DK_DEFAULT_WIRE_COLOUR;
      lit->line_type = DK_VISIBLE;
      lit->selected = FALSE;
      lit->hidden = FALSE;
      lit->schem_posx = 0.0;
      lit->schem_posy = 0.0;
      lit->schem_intx = 0.0;
      lit->schem_inty = 0.0;
      lit->schem_userppos = 0;
      lit->schem_userpint = 0;

      lit->constr_list   = NULL;
      for (i = 0; i < DK_FCV_LIG_NUM; i++)
      {
	  lit->fcv[i] = NULL;
	  lit->active_fcv[i].active = FALSE;
	  lit->active_fcv[i].fcv = NULL;
      }

      lit->selective = FALSE;
      lit->inc_exc = TRUE;

      lit->sun = NULL;

      /*---- Initialize the light shader. */
      lit->shader = NULL;

      lit->next  = NULL;

      /*********************/
      /* Area light fields */
      /*********************/
      lit->area_light = FALSE;
      lit->area_type = DK_AREA_NONE;

      lit->area_u_sample = 3;
      lit->area_v_sample = 3;

      for (i = 0; i < 4; i++)
      {
	 _DK_VectorInit(lit->area_geo.rect.vertex[i], 0.0, 0.0, 0.0);
      }
      lit->area_geo.sphere.radius = 0.0;
      _DK_VectorInit(lit->area_geo.disk.dir, 0.0, 0.0, 0.0);
      lit->area_geo.disk.radius = 0.0;

      _DK_VectorInit(lit->area_scale, 1.0, 1.0, 1.0);

      _DK_VectorInit(lit->area_rot, 0.0, 0.0, 0.0);

      lit->area_local = FALSE;
      lit->area_selected = FALSE;

      /*******************************************************/
      /* Initialize lit->area_rot_mat to the identity matrix */
      /*******************************************************/
      lit->area_rot_mat[0][0] = 1.0;
      lit->area_rot_mat[0][1] = 0.0;
      lit->area_rot_mat[0][2] = 0.0;
      lit->area_rot_mat[0][3] = 0.0;
      lit->area_rot_mat[1][0] = 0.0;
      lit->area_rot_mat[1][1] = 1.0;
      lit->area_rot_mat[1][2] = 0.0;
      lit->area_rot_mat[1][3] = 0.0;
      lit->area_rot_mat[2][0] = 0.0;
      lit->area_rot_mat[2][1] = 0.0;
      lit->area_rot_mat[2][2] = 1.0;
      lit->area_rot_mat[2][3] = 0.0;
      lit->area_rot_mat[3][0] = 0.0;
      lit->area_rot_mat[3][1] = 1.0;
      lit->area_rot_mat[3][2] = 0.0;
      lit->area_rot_mat[3][3] = 0.0;
   }

   return( lit );
}

/*******************************************************************************

   $$L DK_lightDispose

   This function disposes a light structure.

   Returned Value: None

*******************************************************************************/

void DK_lightDispose
    (
	DK_Light **lit
    )
{
    if( lit )
    {
        DK_Light *light = *lit;

        while( light )
        {
           DK_Light *save = light;

           _DK_FREE( light->name, -1 );
           _DK_FREE( light->prefix, -1 );

           _DK_FREE( light->pos_constraint_name, -1 );
           _DK_FREE( light->int_constraint_name, -1 );
           if( light->pos_path )
               DK_splineDispose( &light->pos_path );
           if( light->int_path )
               DK_splineDispose( &light->int_path );
           _DK_FREE( light->pos_path_name, -1 );
           _DK_FREE( light->int_path_name, -1 );

	   DK_constraintDispose( &light->constr_list );
           DK_fcurveArrayDispose( light->fcv, DK_FCV_LIG_NUM );
           DK_actFcurveArrayDispose( light->active_fcv, DK_FCV_LIG_NUM );
           DK_udfDispose( &light->shader );
           light = light->next;
           free( save );
        }

        *lit = NULL;
    }
}

/*******************************************************************************

   $$L DK_lightWriteAsciiAnimation

   This function writes a light.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_lightWriteAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Light *lit 		/* light pointer */
    )
{
   int i,j;

    if (lit == NULL)
	return(FALSE);

    DK_incIndent(1);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_NAME_TOKEN);
    _DK_OUT_STRING(lit->name);

    /* write light data */
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_TYPE_TOKEN);
    switch (lit->type)
    {
	case DK_LIGHT_POINT:
	    _DK_OUT_TOKEN(DK_A_LIGHT_POINT_TYPE_TOKEN);
	    break;
	case DK_LIGHT_SPOT:
	    _DK_OUT_TOKEN(DK_A_LIGHT_SPOT_TYPE_TOKEN);
	    break;
	case DK_LIGHT_INFINITE:
	    _DK_OUT_TOKEN(DK_A_LIGHT_INFINITE_TYPE_TOKEN);
	    break;
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_COLOUR_TOKEN);
    DK_value3WriteAsciiAnimation(file, lit->colour.r, lit->colour.g, lit->colour.b,
	lit->fcv[DK_FCV_LIG_COL_R], lit->fcv[DK_FCV_LIG_COL_G], 
	lit->fcv[DK_FCV_LIG_COL_B]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_FSTART_TOKEN);
    DK_valueWriteAsciiAnimation(file, lit->startFalloff, lit->fcv[DK_FCV_LIG_START]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_FEND_TOKEN);
    DK_valueWriteAsciiAnimation(file, lit->endFalloff, lit->fcv[DK_FCV_LIG_END]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_SFACT_TOKEN);
    DK_valueWriteAsciiAnimation(file, lit->shadowFactor, lit->fcv[DK_FCV_LIG_SHADOW]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_POS_TOKEN);
    DK_positionWriteAsciiAnimation(file, &lit->position, lit->fcv[DK_FCV_LIG_POS], 
	lit->pos_path,
	lit->pos_constraint_name, lit->pos_constraint_type, 
	lit->fcv[DK_FCV_LIG_EPOSX], lit->fcv[DK_FCV_LIG_EPOSY],
	lit->fcv[DK_FCV_LIG_EPOSZ],
	lit->pos_path_name);

    if (lit->type == DK_LIGHT_SPOT) 
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_LIGHT_INT_TOKEN);
        DK_positionWriteAsciiAnimation(file, &lit->spotInterest, 
		lit->fcv[DK_FCV_LIG_INT],
		lit->int_path, lit->int_constraint_name, 
		lit->int_constraint_type, lit->fcv[DK_FCV_LIG_EINTX],
		lit->fcv[DK_FCV_LIG_EINTY], lit->fcv[DK_FCV_LIG_EINTZ],
		lit->int_path_name);

        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_LIGHT_CONE_TOKEN);
        DK_valueWriteAsciiAnimation(file, lit->spotConeAngle, 
		lit->fcv[DK_FCV_LIG_CONE]);
    
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_LIGHT_SPRD_TOKEN);
        DK_valueWriteAsciiAnimation(file, lit->spotSpreadAngle, 
		lit->fcv[DK_FCV_LIG_SPREAD]);
    }

    if (lit->fcv[DK_FCV_LIG_HIDDEN] != NULL)
    {
	/* the 0.0 value is just a dummy value */
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_LIGHT_HIDDEN_TOKEN);
        DK_valueWriteAsciiAnimation(file, 0.0, lit->fcv[DK_FCV_LIG_HIDDEN]);
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_SHADOW_TYPE_TOKEN);
    _DK_OUT_SVALUE( lit->shadowType);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_SHADOW_MAP_RES_TOKEN);
    _DK_OUT_SVALUE( lit->shadowMapRes);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_SHADOW_FILTER_SIZE_TOKEN);
    _DK_OUT_SVALUE( lit->shadowFilterSize);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_SHADOW_FILTER_STEP_TOKEN);
    _DK_OUT_SVALUE( lit->shadowFilterStep);

    /* no fcurve yet */
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_SHADOW_FADE_START_TOKEN);
    DK_valueWriteAsciiAnimation(file, lit->shadowFadeStart, NULL);

    /* no fcurve yet */
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_LIGHT_SHADOW_FADE_RATE_TOKEN);
    DK_valueWriteAsciiAnimation(file, lit->shadowFadeRate, NULL);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_LIGHT_WIRE_COLOUR_TOKEN);
    _DK_OUT_SVALUE( lit->wire_colour);
	 
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_LIGHT_LINE_TYPE_TOKEN);
    _DK_OUT_SVALUE( lit->line_type);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_LIGHT_SELECTED_TOKEN);
    switch (lit->selected)
    {
	case  DK_UNSELECTED:
	    _DK_OUT_TOKEN(DK_A_UNSELECTED_TOKEN);
		break;
	case  DK_NODE_SELECTED:
	    _DK_OUT_TOKEN(DK_A_NODE_SELECTED_TOKEN);
	    break;
	case  DK_BRANCH_SELECTED:
	    _DK_OUT_TOKEN(DK_A_BRANCH_SELECTED_TOKEN);
	    break;
	case  DK_INHERIT_SELECTED:
	    _DK_OUT_TOKEN(DK_A_INHERIT_SELECTED_TOKEN);
	    break;
    }


   /************************/
   /* Data for area lights */
   /************************/

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_BOOLEAN );
   _DK_OUT_SVALUE( lit->area_light );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_TYPE );
   _DK_OUT_SVALUE( lit->area_type );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_U_SAMPLE );
   _DK_OUT_SVALUE( lit->area_u_sample );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_V_SAMPLE );
   _DK_OUT_SVALUE( lit->area_v_sample );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_GEO );
   switch(lit->area_type) {
      case DK_AREA_RECT: {
	 for (i = 0; i < 4; i ++) {
	    DK_value3WriteAsciiAnimation(file,lit->area_geo.rect.vertex[i].x,
	       lit->area_geo.rect.vertex[i].y,
	       lit->area_geo.rect.vertex[i].z,NULL,NULL,NULL);
	 }
      } break;

      case DK_AREA_SPHERE: {
	 DK_valueWriteAsciiAnimation(file, lit->area_geo.sphere.radius, NULL);
      } break;

      case DK_AREA_DISK: {
	 DK_value3WriteAsciiAnimation(file,lit->area_geo.disk.dir.x,
	    lit->area_geo.disk.dir.y,lit->area_geo.disk.dir.z,NULL,NULL,NULL);

	 DK_valueWriteAsciiAnimation(file, lit->area_geo.disk.radius, NULL);
      } break;
   }

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_LOCAL_BOOLEAN );
   _DK_OUT_SVALUE( lit->area_local );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_SCALE );
   DK_value3WriteAsciiAnimation(file,lit->area_scale.x,lit->area_scale.y,
      lit->area_scale.z,lit->fcv[DK_FCV_LIG_ASCL_X],lit->fcv[DK_FCV_LIG_ASCL_Y],
      lit->fcv[DK_FCV_LIG_ASCL_Z]);

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_ROTATION );
   DK_value3WriteAsciiAnimation(file,lit->area_rot.x,lit->area_rot.y,
      lit->area_rot.z,lit->fcv[DK_FCV_LIG_AROT_X],lit->fcv[DK_FCV_LIG_AROT_Y],
      lit->fcv[DK_FCV_LIG_AROT_Z]);

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_SELECTED );
   _DK_OUT_SVALUE( lit->area_selected );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_LIGHT_AREA_ROTATION_MAT );
   for (i = 0; i < 4; i++)
   for (j = 0; j < 4; j++)
   DK_valueWriteAsciiAnimation(file, lit->area_rot_mat[i][j], NULL);


    DK_incIndent(-1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_lightReadAsciiAnimation

   This function allocates and reads a light.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_lightReadAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Light *lit		/* light pointer */
    )
{
    char *lname = NULL;
    DK_SVALUE sv;
    int first = TRUE;

    DK_in_animToken(file);
    while (TRUE)
    {
        if (strcmp( DK_A_LIGHT_NAME_TOKEN, DK_animToken) == 0)
	{
	    lname = DK_stringReadAsciiAnimation(file);
	    if (lit != NULL)
		lit->name = lname;
	    else
		_DK_FREE(lname, -1);
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_LIGHT_TYPE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_LIGHT_POINT_TYPE_TOKEN, DK_animToken) == 0)
		sv = DK_LIGHT_POINT;
	    else if (strcmp(DK_A_LIGHT_SPOT_TYPE_TOKEN, DK_animToken) == 0)
		sv = DK_LIGHT_SPOT;
	    else if (strcmp(DK_A_LIGHT_INFINITE_TYPE_TOKEN, DK_animToken) == 0)
		sv = DK_LIGHT_INFINITE;
	    else if ( DK_option.verbose == TRUE)
		fprintf( DK_option.msgFd, "DK_LightReadAsciiAnimation: \
error, unknown light type DK_animToken '%s'.\n", DK_animToken);
	    DK_in_animToken(file);

	    if (lit != NULL)
	        lit->type = sv;
	}
        else if (strcmp( DK_A_LIGHT_COLOUR_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
	        DK_value3ReadAsciiAnimation(file, 
		    &lit->colour.r, &lit->colour.g, &lit->colour.b,
		    &lit->fcv[DK_FCV_LIG_COL_R], &lit->fcv[DK_FCV_LIG_COL_G],
		    &lit->fcv[DK_FCV_LIG_COL_B]);
	    else
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp( DK_A_LIGHT_FSTART_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
	        DK_valueReadAsciiAnimation(file, &lit->startFalloff,
		    &lit->fcv[DK_FCV_LIG_START], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp( DK_A_LIGHT_FEND_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
	        DK_valueReadAsciiAnimation(file, &lit->endFalloff,
			&lit->fcv[DK_FCV_LIG_END], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp( DK_A_LIGHT_SFACT_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
	        DK_valueReadAsciiAnimation(file, &lit->shadowFactor, 
			&lit->fcv[DK_FCV_LIG_SHADOW], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp( DK_A_LIGHT_POS_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
	    {
	        DK_positionReadAsciiAnimation(file, &lit->position,
		    &lit->fcv[DK_FCV_LIG_POS], &lit->pos_path, 
		    &lit->pos_constraint_name, 
		    &lit->pos_constraint_type,
		    &lit->fcv[DK_FCV_LIG_EPOSX], &lit->fcv[DK_FCV_LIG_EPOSY],
		    &lit->fcv[DK_FCV_LIG_EPOSZ],
		    &lit->pos_path_name);
	    }
	    else
	        DK_positionReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL,
		    NULL, NULL, NULL, NULL);
	}
        else if (strcmp( DK_A_LIGHT_INT_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
                DK_positionReadAsciiAnimation(file, &lit->spotInterest,
	            &lit->fcv[DK_FCV_LIG_INT], &lit->int_path, 
		    &lit->int_constraint_name, 
		    &lit->int_constraint_type,
		    &lit->fcv[DK_FCV_LIG_EINTX], &lit->fcv[DK_FCV_LIG_EINTY],
		    &lit->fcv[DK_FCV_LIG_EINTZ],
		    &lit->int_path_name);
	    else
	        DK_positionReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL,
		    NULL, NULL, NULL, NULL);
	}
        else if (strcmp( DK_A_LIGHT_CONE_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
                DK_valueReadAsciiAnimation(file, &lit->spotConeAngle,
			&lit->fcv[DK_FCV_LIG_CONE], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp( DK_A_LIGHT_SPRD_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
                DK_valueReadAsciiAnimation(file, &lit->spotSpreadAngle, 
			&lit->fcv[DK_FCV_LIG_SPREAD], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp( DK_A_LIGHT_HIDDEN_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
                DK_valueReadAsciiAnimation(file, NULL, &lit->fcv[DK_FCV_LIG_HIDDEN], TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp( DK_A_LIGHT_SHADOW_TYPE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (lit != NULL)
		lit->shadowType = (DK_ShadowType) DK_animSValue;
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_LIGHT_SHADOW_MAP_RES_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (lit != NULL)
		lit->shadowMapRes = (int) DK_animSValue;
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_LIGHT_SHADOW_FILTER_SIZE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (lit != NULL)
		lit->shadowFilterSize = (int) DK_animSValue;
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_LIGHT_SHADOW_FILTER_STEP_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (lit != NULL)
		lit->shadowFilterStep = (int) DK_animSValue;
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_LIGHT_SHADOW_FADE_START_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
                DK_valueReadAsciiAnimation(file, &lit->shadowFadeStart, NULL, TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp( DK_A_LIGHT_SHADOW_FADE_RATE_TOKEN, DK_animToken) == 0)
	{
	    if (lit != NULL)
                DK_valueReadAsciiAnimation(file, &lit->shadowFadeRate, NULL, TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
	/* not in use, so read into dummy */
        else if (strcmp( DK_A_LIGHT_DECAY_TOKEN, DK_animToken) == 0)
	{
	    DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp( DK_A_LIGHT_WIRE_COLOUR_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (lit != NULL)
		lit->wire_colour = DK_animSValue;
	    DK_in_animToken(file);
	}
        else if (strcmp(DK_A_LIGHT_WIRE_COLOUR_INDEX_TOKEN, DK_animToken) == 0)
        {
            DK_in_animSValue_na(file);
            DK_in_animSValue_na(file);
	    /* forget the two indexes for the moment */
            if (lit != NULL)
    	        lit->wire_colour = DK_DEFAULT_WIRE_COLOUR;
    	    DK_in_animToken(file);
        }
	else if (strcmp( DK_A_LIGHT_LINE_TYPE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (lit != NULL)
		lit->line_type = DK_animSValue;
	    DK_in_animToken(file);
	}
	else if (strcmp( DK_A_LIGHT_SELECTED_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (lit != NULL)
	    {
		if (strcmp(DK_A_UNSELECTED_TOKEN, DK_animToken) == 0)
		    lit->selected =  DK_UNSELECTED;
		else if (strcmp(DK_A_NODE_SELECTED_TOKEN, DK_animToken) == 0)
		    lit->selected =  DK_NODE_SELECTED;
		else if (strcmp(DK_A_BRANCH_SELECTED_TOKEN, DK_animToken) == 0)
		    lit->selected =  DK_BRANCH_SELECTED;
		else if (strcmp(DK_A_INHERIT_SELECTED_TOKEN, DK_animToken) == 0)
		    lit->selected =  DK_INHERIT_SELECTED;
		else if ( DK_option.verbose == TRUE)
		    fprintf( DK_option.msgFd, "DK_LightReadAsciiAnimation: \
error, unknown select DK_animToken '%s'.\n", DK_animToken);
	    }
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_LIGHT_AREA_BOOLEAN, DK_animToken) == 0) {
            DK_in_animSValue_na(file);
            if (lit != NULL) lit->area_light = DK_animSValue;
            DK_in_animToken(file);
        }
        else if (strcmp( DK_A_LIGHT_AREA_TYPE, DK_animToken) == 0) {
            DK_in_animSValue_na(file);
            if (lit != NULL) lit->area_type = DK_animSValue;
            DK_in_animToken(file);
        }
        else if (strcmp( DK_A_LIGHT_AREA_U_SAMPLE, DK_animToken) == 0) {
            DK_in_animSValue_na(file);
            if (lit != NULL) lit->area_u_sample = DK_animSValue;
            DK_in_animToken(file);
        }
        else if (strcmp( DK_A_LIGHT_AREA_V_SAMPLE, DK_animToken) == 0) {
            DK_in_animSValue_na(file);
            if (lit != NULL) lit->area_v_sample = DK_animSValue;
            DK_in_animToken(file);
        }
        else if (strcmp( DK_A_LIGHT_AREA_GEO, DK_animToken) == 0) {
            switch(lit->area_type) {
               case DK_AREA_RECT: {
                  int i;

                  for (i = 0; i < 4; i ++)
		  if (lit != NULL)
                     DK_value3ReadAsciiAnimation(file,
			&lit->area_geo.rect.vertex[i].x,
                        &lit->area_geo.rect.vertex[i].y,
                        &lit->area_geo.rect.vertex[i].z,NULL,NULL,NULL);
                  else DK_value3ReadAsciiAnimation(file,NULL,NULL,NULL,NULL,
		     NULL,NULL);
               } break;

               case DK_AREA_SPHERE: {
		  if (lit != NULL) DK_valueReadAsciiAnimation(file,
		     &lit->area_geo.sphere.radius,NULL,TRUE);
                  else DK_valueReadAsciiAnimation(file,NULL,NULL,TRUE);
               } break;

               case DK_AREA_DISK: {
		  if (lit != NULL)
                     DK_value3ReadAsciiAnimation(file,
			&lit->area_geo.disk.dir.x,
                        &lit->area_geo.disk.dir.y,
                        &lit->area_geo.disk.dir.z,NULL,NULL,NULL);
                  else DK_value3ReadAsciiAnimation(file,NULL,NULL,NULL,NULL,
		     NULL,NULL);

		  if (lit != NULL) DK_valueReadAsciiAnimation(file,
			&lit->area_geo.disk.radius,NULL,TRUE);
                  else DK_valueReadAsciiAnimation(file,NULL,NULL,TRUE);
               } break;
            }
        }
        else if (strcmp( DK_A_LIGHT_AREA_LOCAL_BOOLEAN, DK_animToken) == 0) {
            DK_in_animSValue_na(file);
            if (lit != NULL) lit->area_local = DK_animSValue;
            DK_in_animToken(file);
        }
        else if (strcmp( DK_A_LIGHT_AREA_SCALE, DK_animToken) == 0) {
            if (lit != NULL)
               DK_value3ReadAsciiAnimation(file,&lit->area_scale.x,
                  &lit->area_scale.y,&lit->area_scale.z,
		  &lit->fcv[DK_FCV_LIG_ASCL_X],&lit->fcv[DK_FCV_LIG_ASCL_Y],
		  &lit->fcv[DK_FCV_LIG_ASCL_Z]);
            else DK_value3ReadAsciiAnimation(file,NULL,NULL,NULL,NULL,NULL,
	       NULL);
        }
        else if (strcmp( DK_A_LIGHT_AREA_ROTATION, DK_animToken) == 0) {
            if (lit != NULL)
               DK_value3ReadAsciiAnimation(file,&lit->area_rot.x,
                  &lit->area_rot.y,&lit->area_rot.z,
		  &lit->fcv[DK_FCV_LIG_AROT_X],&lit->fcv[DK_FCV_LIG_AROT_Y],
		  &lit->fcv[DK_FCV_LIG_AROT_Z]);
            else DK_value3ReadAsciiAnimation(file,NULL,NULL,NULL,NULL,NULL,
	       NULL);
        }
        else if (strcmp( DK_A_LIGHT_AREA_SELECTED, DK_animToken) == 0) {
            DK_in_animSValue_na(file);
            if (lit != NULL) lit->area_selected = DK_animSValue;
            DK_in_animToken(file);
        }
        else if (strcmp( DK_A_LIGHT_AREA_ROTATION_MAT, DK_animToken) == 0) {
            int i,j;

            for (i = 0; i < 4; i++)
            for (j = 0; j < 4; j++)
            if (lit != NULL) DK_valueReadAsciiAnimation(file,
	       &lit->area_rot_mat[i][j],NULL,TRUE);
	    else DK_valueReadAsciiAnimation(file,NULL,NULL,TRUE);
        }

	else
	{
	    if (first)
		DK_in_animToken(file);

	    return(TRUE);
	}
	first = FALSE;
    }
}

