/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyLattice.c,v $
   $Revision: 1.8.14.1 $ $Date: 1995/06/09 22:00:02 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's DK_lattice functions.

   Written by: Colin Hui

   (c) Copyright 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinySoftType.h"
#include "tinySceneToken.h"
#include "tinyUtils.h"
#include "tinyLattice.h"

/*******************************************************************************

   $$L DK_latticeAllocate

   This function allocates a DK_lattice structure.

   Returned Value: DK_Lattice structure

*******************************************************************************/

DK_Lattice *DK_latticeAllocate( void )
{
   DK_Lattice *lat;

   if( lat = (DK_Lattice *) calloc( 1, sizeof( DK_Lattice ) ) )
   {
      lat->active 		= FALSE;
      lat->nbx			= 0;
      lat->nby			= 0;
      lat->nbz			= 0;
      lat->interpx		= DK_LAT_LINEAR;
      lat->interpy		= DK_LAT_LINEAR;
      lat->interpz		= DK_LAT_LINEAR;
      lat->array		= NULL;
      lat->flag			= NULL;
      _DK_VectorInit(lat->min, 0.0, 0.0, 0.0);
      _DK_VectorInit(lat->max, 0.0, 0.0, 0.0);
      _DK_VectorInit(lat->scl, 0.0, 0.0, 0.0);
      lat->next          	= NULL;
   }

   return( lat );
}

/*******************************************************************************

   $$L DK_latticeDispose

   This function disposes a DK_lattice structure.

   Returned Value: None

*******************************************************************************/

void DK_latticeDispose
    (
	DK_Lattice **lat
    )
{
    if( lat )
    {
        DK_Lattice *DK_lattice = *lat;

        while( DK_lattice )
        {
           DK_Lattice *save = DK_lattice;

           _DK_FREE( DK_lattice->array, -1 );
           _DK_FREE( DK_lattice->flag, -1 );
           DK_lattice = DK_lattice->next;
           free( save );
        }

        *lat = NULL;
    }
}


/*******************************************************************************

   $$L DK_latticeWriteAsciiAnimation

   This function writes a DK_lattice.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_latticeWriteAsciiAnimation
    (
	 FILE *file,	/* file io pointer */
	short nb,	/* number of DK_lattices */
	DK_Lattice *lat,	/* DK_lattice data pointer */
	DK_Fcurve *fcv	/* DK_lattice fcurve */
    )
{
    int i, j, total;
    DK_Vector *v;
    DK_Lattice *lptr;

    if (nb <= 0 || lat == NULL)
	return(FALSE);

    DK_incIndent(1);
    lptr = lat;
    for (j = 0; j < nb; j++) 
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_LATTICE_ACTIVE_TOKEN);
        if (lptr->active)
            _DK_OUT_TOKEN(DK_A_YES_TOKEN);
	else
            _DK_OUT_TOKEN(DK_A_NO_TOKEN);

	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_LATTICE_NBARRAY_TOKEN);
        _DK_OUT_SVALUE( lptr->nbx);
        _DK_OUT_SVALUE( lptr->nby);
        _DK_OUT_SVALUE( lptr->nbz);
    
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_LATTICE_INTERP_TOKEN);
        _DK_OUT_SVALUE( lptr->interpx);
        _DK_OUT_SVALUE( lptr->interpy);
        _DK_OUT_SVALUE( lptr->interpz);
    
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_LATTICE_MIN_TOKEN);
        DK_vectorWriteAsciiAnimation(file, &lptr->min);
    
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_LATTICE_MAX_TOKEN);
        DK_vectorWriteAsciiAnimation(file, &lptr->max);
    
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN(DK_A_LATTICE_SCL_TOKEN);
        DK_vectorWriteAsciiAnimation(file, &lptr->scl);
    
        total = lptr->nbx * lat->nby * lat->nbz;
        if (total > 0) 
	{
	    _DK_OUT_INDENT();
            _DK_OUT_TOKEN( DK_A_LATTICE_KEYS_TOKEN);
            for (i = 0, v = lptr->array; i < total; i++, v++)
	        DK_vectorWriteAsciiAnimation(file, v);
        }
	_DK_OUT_NEWLINE();
	lptr = lptr->next;
    }

    if (fcv != NULL) 
    {
	_DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_LATTICE_FCV_TOKEN);
        DK_fcurveWriteAsciiAnimation(file, fcv);
    }
    DK_incIndent(-1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_latticeReadAsciiAnimation

   This function allocates and reads a DK_lattice.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_latticeReadAsciiAnimation
    (
	 FILE *file, 		/* file io pointer */
	int tree,		/* true if tree DK_lattice, else node DK_lattice */
	short *nb,		/* number of DK_lattices */
	DK_Lattice **head_lat,	/* DK_lattice pointer */
	DK_Fcurve **fcv		/* DK_lattice fcurve */
    )
{

    int i, total, read_animToken, first = TRUE;
    DK_Vector *v;
    DK_Lattice *lptr, *last_lptr = NULL, lat;
    short *sptr;

    if (head_lat != NULL && *head_lat != NULL)
    {
        last_lptr = *head_lat;
	while (last_lptr->next != NULL)
	    last_lptr = last_lptr->next;
    }

    lat.active = FALSE;
    lat.nbx = lat.nby = lat.nbz = 0;
    while (TRUE)
    {
   	read_animToken = TRUE;
        if (strcmp( DK_A_LATTICE_ACTIVE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
		lat.active = TRUE;
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
		lat.active = FALSE;
	    else if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd, "DK_LatticeReadAsciiAnimation: error, \
unknown active DK_animToken '%s'.\n", DK_animToken);
	}
        else if (strcmp( DK_A_LATTICE_NBARRAY_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue(file, &lat.nbx);
	    DK_in_animSValue(file, &lat.nby);
	    DK_in_animSValue(file, &lat.nbz);
	}
        else if (strcmp( DK_A_LATTICE_INTERP_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    lat.interpx = DK_animSValue;
	    DK_in_animSValue_na(file);
	    lat.interpy = DK_animSValue;
	    DK_in_animSValue_na(file);
	    lat.interpz = DK_animSValue;
	}
	else if (strcmp( DK_A_LATTICE_MIN_TOKEN, DK_animToken) == 0)
	{
	    DK_vectorReadAsciiAnimation(file, &lat.min);
	}
	else if (strcmp( DK_A_LATTICE_MAX_TOKEN, DK_animToken) == 0)
	{
	    DK_vectorReadAsciiAnimation(file, &lat.max);
	}
	else if (strcmp( DK_A_LATTICE_SCL_TOKEN, DK_animToken) == 0)
	{
	    DK_vectorReadAsciiAnimation(file, &lat.scl);
	}
        else if (strcmp( DK_A_LATTICE_KEYS_TOKEN, DK_animToken) == 0)
	{
	    total = lat.nbx * lat.nby * lat.nbz;
	    if (total > 0)
	    {
		lat.array = _DK_MALLOC(DK_Vector, sizeof(DK_Vector) * total, -1);
		lat.flag = _DK_MALLOC(short, sizeof(short) * total, -1);
	        if (lat.flag == NULL || lat.array == NULL) 
		{
		    if (lat.array != NULL)
		    {
		        _DK_FREE(lat.array, -1);
		    }
		    if (lat.flag != NULL)
		    {
		        _DK_FREE(lat.flag, -1);
		    }
		}
		if (lat.array != NULL && lat.flag != NULL)
		{
		    for (i = 0, sptr = lat.flag, v = lat.array;
		        i < total; i++, v++, sptr++) 
		    {
		        DK_vectorReadAsciiAnimation(file, v);
		        *sptr = FALSE;
		    }
		    (*nb)++;
	            /* allocate DK_lattice */
	            lptr = DK_latticeAllocate();
	            if (*head_lat == NULL)
		        *head_lat = lptr;
	            else
		        last_lptr->next = lptr;
	            last_lptr = lptr;

		    lptr->array = lat.array;
		    lptr->flag = lat.flag;
		    lptr->active = lat.active;
		    lptr->nbx = lat.nbx;
		    lptr->nby = lat.nby;
		    lptr->nbz = lat.nbz;
		    lptr->interpx = lat.interpx;
		    lptr->interpy = lat.interpy;
		    lptr->interpz = lat.interpz;
		    lptr->min = lat.min;
		    lptr->max = lat.max;
		    lptr->scl = lat.scl;

		    lat.active = FALSE;
		    lat.nbx = lat.nby = lat.nbz = 0;
		    lat.array = NULL;
		    lat.flag = NULL;
		}
	        else 
		{
		    for (i = 0; i < total; i++)
		        DK_vectorReadAsciiAnimation(file, NULL);
	        }
	    }
	}
        else if (strcmp( DK_A_LATTICE_FCV_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
            DK_fcurveReadAsciiAnimation(file, fcv);
	    read_animToken = FALSE;
	}
	else
	{
	    if (first)
		DK_in_animToken(file);

	    return(TRUE);
	}
	if (read_animToken)
	    DK_in_animToken(file);
	first = FALSE;
    }
}

