/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyFitting.c,v $
   $Revision: 1.12.14.1 $ $Date: 1995/06/09 22:00:01 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's fitting functions.

   Written by: Colin Hui

   (c) Copyright 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinySoftType.h"
#include "tinySceneToken.h"
#include "tinyUtils.h"
#include "tinyFitting.h"
#include "tinySpline.h"

/*******************************************************************************

   $$L DK_spl_fittingAllocate

   This function allocates a DK_spline fitting structure.

   Returned Value: DK_Spl_fitting structure

*******************************************************************************/

DK_Spl_fitting *DK_spl_fittingAllocate( void )
{
   DK_Spl_fitting *sf;
   int i;

   if( sf = (DK_Spl_fitting *) calloc( 1, sizeof( DK_Spl_fitting ) ) )
   {
      sf->active = TRUE;
      sf->spl = NULL;
      sf->name = NULL;
      _DK_VectorInit(sf->scal, 1.0, 1.0, 1.0);
      sf->roll = 0.0;
      _DK_VectorInit(sf->trans, 0.0, 0.0, 0.0);
      for (i = 0; i < DK_FCV_SFIT_NUM; i++)
      {
	  sf->fcv[i] = NULL;
	  sf->active_fcv[i].active = FALSE;
	  sf->active_fcv[i].fcv = NULL;
      }
   }

   return( sf );
}

/*******************************************************************************

   $$L DK_ptch_fittingAllocate

   This function allocates a patch fitting structure.

   Returned Value: DK_Ptch_fitting structure

*******************************************************************************/

DK_Ptch_fitting *DK_ptch_fittingAllocate( void )
{
   DK_Ptch_fitting *pf;
   int i;

   if( pf = (DK_Ptch_fitting *) calloc( 1, sizeof( DK_Ptch_fitting ) ) )
   {
      pf->active = TRUE;
      pf->name = NULL;
      _DK_VectorInit(pf->scal, 1.0, 1.0, 1.0);
      _DK_VectorInit(pf->rot, 0.0, 0.0, 0.0);
      _DK_VectorInit(pf->trans, 0.0, 0.0, 0.0);
      for (i = 0; i < DK_FCV_SFIT_NUM; i++)
      {
	  pf->fcv[i] = NULL;
	  pf->active_fcv[i].active = FALSE;
	  pf->active_fcv[i].fcv = NULL;
      }
   }

   return( pf );
}

/*******************************************************************************

   $$L DK_spl_fittingDispose

   This function disposes a DK_spl_fitting structure.

   Returned Value: None

*******************************************************************************/

void DK_spl_fittingDispose
    (
	DK_Spl_fitting **sf
    )
{
    if (sf != NULL && *sf != NULL)
    {
        if( (*sf)->spl )
            DK_splineDispose( &(*sf)->spl );
        _DK_FREE( (*sf)->name, -1 );
        DK_fcurveArrayDispose( (*sf)->fcv, DK_FCV_SFIT_NUM );
        DK_actFcurveArrayDispose( (*sf)->active_fcv, DK_FCV_SFIT_NUM );
        _DK_FREE(*sf, -1);
    }
}

/*******************************************************************************

   $$L DK_ptch_fittingDispose

   This function disposes a DK_ptch_fitting structure.

   Returned Value: None

*******************************************************************************/

void DK_ptch_fittingDispose
    (
	DK_Ptch_fitting **pf
    )
{
    if (pf != NULL && *pf != NULL)
    {
        _DK_FREE( (*pf)->name, -1 );
        DK_fcurveArrayDispose( (*pf)->fcv, DK_FCV_PFIT_NUM );
        DK_actFcurveArrayDispose( (*pf)->active_fcv, DK_FCV_PFIT_NUM );
        _DK_FREE(*pf, -1);
    }
}

/*******************************************************************************

   $$L DK_splineFitWriteAsciiAnimation

   This function writes a DK_spline fit, i.e. translation, scale, roll fcurves
   and the DK_spline path.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_splineFitWriteAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Spl_fitting *sf		/* DK_spline fit pointer */
    )
{
    if (sf == NULL) 
	return(FALSE);

    DK_incIndent(1);
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SFIT_ACTIVE_TOKEN);
    if (sf->active)
	_DK_OUT_TOKEN(DK_A_YES_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_NO_TOKEN);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SFIT_SCL_TOKEN);
    DK_value3WriteAsciiAnimation(file, sf->scal.x, sf->scal.y, sf->scal.z,
	sf->fcv[DK_FCV_SFIT_SCLX], sf->fcv[DK_FCV_SFIT_SCLY], sf->fcv[DK_FCV_SFIT_SCLZ]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SFIT_ROLL_TOKEN);
    DK_valueWriteAsciiAnimation(file, sf->roll, sf->fcv[DK_FCV_SFIT_ROLL]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_SFIT_TRN_TOKEN);
    DK_value3WriteAsciiAnimation(file, sf->trans.x, sf->trans.y, sf->trans.z,
	sf->fcv[DK_FCV_SFIT_TRNX], sf->fcv[DK_FCV_SFIT_TRNY], sf->fcv[DK_FCV_SFIT_TRNZ]);

    if ( sf->name != NULL )
    {
       _DK_OUT_INDENT();
       _DK_OUT_TOKEN( DK_A_SFIT_MODEL_TOKEN );

	DK_incIndent(1);
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN(DK_A_SPLINE_NAME_TOKEN);
        _DK_OUT_STRING(sf->name);
	DK_incIndent(-1);
    }
    else
    {
       _DK_OUT_INDENT();
       _DK_OUT_TOKEN( DK_A_SFIT_SPL_TOKEN );
       DK_splineWriteAsciiAnimation(file, sf->spl);
    }

    DK_incIndent(-1);
    return(TRUE);
}

/*******************************************************************************

   $$L DK_splineFitReadAsciiAnimation

   This function allocates and reads a DK_spline fit, i.e. translation, 
   scale, roll fcurves and the DK_spline path.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_splineFitReadAsciiAnimation
    (
	 FILE *file, 		/* file io pointer */
	DK_Spl_fitting **sf	/* DK_spline fit pointer */
    )
{
    int first = TRUE;
    DK_String name;

    if (sf != NULL) 
    {
	if ((*sf = DK_spl_fittingAllocate()) == NULL &&  DK_option.verbose == TRUE)
	    fprintf( DK_option.msgFd, 
		"DK_SplineFitReadAsciiAnimation: cannot allocate.\n");
	(*sf)->spl = NULL;
    }

    while (TRUE)
    {
        if (strcmp( DK_A_SFIT_ACTIVE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (sf != NULL && *sf != NULL)
	    {
                if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
                    (*sf)->active = TRUE;
                else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
                    (*sf)->active = FALSE;
                else if ( DK_option.verbose == TRUE)
                    fprintf( DK_option.msgFd, "DK_SplineFitReadAsciiAnimation: error, \
unknown active DK_animToken '%s'.\n", DK_animToken);
	    }
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_SFIT_SCL_TOKEN, DK_animToken) == 0)
	{
	    if (sf != NULL && *sf != NULL)
	        DK_value3ReadAsciiAnimation(file, &(*sf)->scal.x, &(*sf)->scal.y,
			&(*sf)->scal.z, &(*sf)->fcv[DK_FCV_SFIT_SCLX],
			&(*sf)->fcv[DK_FCV_SFIT_SCLY], &(*sf)->fcv[DK_FCV_SFIT_SCLZ]);
	    else
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
	}
        else if (strcmp( DK_A_SFIT_ROLL_TOKEN, DK_animToken) == 0)
	{
	    if (sf != NULL && *sf != NULL)
	        DK_valueReadAsciiAnimation(file, &(*sf)->roll, &(*sf)->fcv[DK_FCV_SFIT_ROLL],
			TRUE);
	    else
	        DK_valueReadAsciiAnimation(file, NULL, NULL, TRUE);
	}
        else if (strcmp( DK_A_SFIT_TRN_TOKEN, DK_animToken) == 0)
	{
	    if (sf != NULL && *sf != NULL)
	        DK_value3ReadAsciiAnimation(file, &(*sf)->trans.x, &(*sf)->trans.y,
			&(*sf)->trans.z, &(*sf)->fcv[DK_FCV_SFIT_TRNX],
			&(*sf)->fcv[DK_FCV_SFIT_TRNY], &(*sf)->fcv[DK_FCV_SFIT_TRNZ]);
	    else
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, NULL, NULL);
	}
	/* old scene file format (before v2.5) */
        else if (strcmp( DK_A_SFIT_SPL_TOKEN, DK_animToken) == 0)
	{
	    if (DK_in_animToken(file) == FALSE)
		return(FALSE);
	    if (sf != NULL && *sf != NULL)
	    {
		(*sf)->spl = DK_splineAllocate();
	        DK_splineReadAsciiAnimation(file, (*sf)->spl);
	    }
	    else
	        DK_splineReadAsciiAnimation(file, NULL);
	}
	/* new v2.5 scene file format */
        else if (strcmp( DK_A_SFIT_MODEL_TOKEN, DK_animToken) == 0)
	{
	    if (DK_in_animToken(file) == FALSE)
		return(FALSE);
	    name = DK_stringReadAsciiAnimation(file);
	    if (sf != NULL && *sf != NULL) 
		(*sf)->name = name;
	    else if (name != NULL)
	    {
		_DK_FREE(name, -1);
	    }
	    if (DK_in_animToken(file) == FALSE)
	        return(FALSE);
	}
	else
	{
	    if (first)
	    {
		DK_in_animToken(file);
		if (*sf != NULL)
		    DK_spl_fittingDispose(sf);
	    }
	    return(TRUE);
	}
	first = FALSE;
    }

}

/*******************************************************************************

   $$L DK_patchFitWriteAsciiAnimation

   This function writes a patch fit, i.e. translation, 
   scale, rotate fcurves and the patch DK_model name.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_patchFitWriteAsciiAnimation
    (
	 FILE *file,		/* file io pointer */
	DK_Ptch_fitting *pf	/* patch fit pointer */
    )
{
    if (pf == NULL) 
	return(FALSE);

    DK_incIndent(1);
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_PFIT_ACTIVE_TOKEN);
    if (pf->active)
	_DK_OUT_TOKEN(DK_A_YES_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_NO_TOKEN);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_PFIT_SCL_TOKEN);
    DK_value3WriteAsciiAnimation(file, pf->scal.x, pf->scal.y, pf->scal.z,
	pf->fcv[DK_FCV_PFIT_SCLX], pf->fcv[DK_FCV_PFIT_SCLY], pf->fcv[DK_FCV_PFIT_SCLZ]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_PFIT_ROT_TOKEN);
    DK_value3WriteAsciiAnimation(file, pf->rot.x, pf->rot.y, pf->rot.z,
	pf->fcv[DK_FCV_PFIT_ROTX], pf->fcv[DK_FCV_PFIT_ROTY], pf->fcv[DK_FCV_PFIT_ROTZ]);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_PFIT_TRN_TOKEN);
    DK_value3WriteAsciiAnimation(file, pf->trans.x, pf->trans.y, pf->trans.z,
	pf->fcv[DK_FCV_PFIT_TRNX], pf->fcv[DK_FCV_PFIT_TRNY], pf->fcv[DK_FCV_PFIT_TRNZ]);

    if (pf->name != NULL) 
    {
        _DK_OUT_INDENT();
        _DK_OUT_TOKEN( DK_A_PFIT_MODEL_TOKEN);
        _DK_OUT_STRING(pf->name);
    }
    else if ( DK_option.verbose == TRUE)
	fprintf( DK_option.msgFd,"Null DK_model name.\n");

    DK_incIndent(-1);
    return(TRUE);
}

/*******************************************************************************

   $$L DK_patchFitReadAsciiAnimation

   This function allocates and reads a patch fit, i.e. translation, 
   scale, rotate fcurves and the patch DK_model name.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_patchFitReadAsciiAnimation
    (
	 FILE *file, 		/* file io pointer */
	DK_Ptch_fitting **pf	/* patch fit pointer */
    )
{
    char *name;
    int first = TRUE;

    if (pf != NULL) 
    {
	if ((*pf = DK_ptch_fittingAllocate()) == NULL &&  DK_option.verbose == TRUE)
	    fprintf( DK_option.msgFd, 
		"DK_PatchFitReadAsciiAnimation: cannot allocate.\n");
	(*pf)->name = NULL;
    }

    while (TRUE)
    {
        if (strcmp( DK_A_PFIT_ACTIVE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (pf != NULL && *pf != NULL)
	    {
                if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
                    (*pf)->active = TRUE;
                else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
                    (*pf)->active = FALSE;
                else if ( DK_option.verbose == TRUE)
                    fprintf( DK_option.msgFd, "DK_PatchFitReadAsciiAnimation: error, \
unknown active DK_animToken '%s'.\n", DK_animToken);
	    }
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_PFIT_SCL_TOKEN, DK_animToken) == 0)
	{
	    if (pf != NULL && *pf != NULL)
	        DK_value3ReadAsciiAnimation(file, &(*pf)->scal.x, &(*pf)->scal.y,
			&(*pf)->scal.z, &(*pf)->fcv[DK_FCV_PFIT_SCLX],
			&(*pf)->fcv[DK_FCV_PFIT_SCLY], &(*pf)->fcv[DK_FCV_PFIT_SCLZ]);
	    else
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, 
			NULL, NULL);
	}
        else if (strcmp( DK_A_PFIT_ROT_TOKEN, DK_animToken) == 0)
	{
	    if (pf != NULL && *pf != NULL)
	        DK_value3ReadAsciiAnimation(file, &(*pf)->rot.x, &(*pf)->rot.y,
			&(*pf)->rot.z, &(*pf)->fcv[DK_FCV_PFIT_ROTX],
			&(*pf)->fcv[DK_FCV_PFIT_ROTY], &(*pf)->fcv[DK_FCV_PFIT_ROTZ]);
	    else
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, 
			NULL, NULL);
	}
        else if (strcmp( DK_A_PFIT_TRN_TOKEN, DK_animToken) == 0)
	{
	    if (pf != NULL && *pf != NULL)
	        DK_value3ReadAsciiAnimation(file, &(*pf)->trans.x, &(*pf)->trans.y,
			&(*pf)->trans.z, &(*pf)->fcv[DK_FCV_PFIT_TRNX],
			&(*pf)->fcv[DK_FCV_PFIT_TRNY], &(*pf)->fcv[DK_FCV_PFIT_TRNZ]);
	    else
	        DK_value3ReadAsciiAnimation(file, NULL, NULL, NULL, NULL, 
			NULL, NULL);
	}
	else if (strcmp( DK_A_PFIT_MODEL_TOKEN, DK_animToken) == 0)
	{
	    name = DK_stringReadAsciiAnimation(file);
	    if (pf != NULL && *pf != NULL) 
		(*pf)->name = name;
	    else if (name != NULL)
	    {
		_DK_FREE(name, -1);
	    }
	    if (DK_in_animToken(file) == FALSE)
	        return(FALSE);
	}
	else
	{
	    if (first)
	    {
		DK_in_animToken(file);
		if (*pf != NULL)
		    DK_ptch_fittingDispose(pf);
	    }
	    return(TRUE);
	}
	first = FALSE;
    }
}
