/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyFcurve.c,v $
   $Revision: 1.15.14.1 $ $Date: 1995/06/09 21:59:58 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's fcurve functions.

   Written by: Colin Hui

   (c) Copyright 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinySoftType.h"
#include "tinySceneToken.h"
#include "tinyFcurve.h"
#include "tinyUtils.h"

static int DK_fcvkeyWriteAsciiAnimation (  FILE *, DK_Fcvkey *);
static int DK_fcvkeyReadAsciiAnimation (  FILE *, DK_Fcvkey *);

/*******************************************************************************

   $$L DK_fcurveAllocate

   This function allocates a fcurve structure.

   Returned Value: DK_Fcurve structure

*******************************************************************************/

DK_Fcurve *DK_fcurveAllocate( void )
{
   DK_Fcurve *fcv;

   if( fcv = (DK_Fcurve *) calloc( 1, sizeof( DK_Fcurve ) ) )
   {
      fcv->type 		= DK_FCV_TYPE_DEFAULT;
      fcv->keys          	= NULL;
      fcv->size          	= 0;
      fcv->pre_extrap          	= 0;
      fcv->post_extrap          = 0;
      fcv->slopeFlag            = 0;
      fcv->interp               = DK_CONSTANT_INTERP;
   }

   return( fcv );
}

/*******************************************************************************

   $$L DK_fcvkeyAllocate

   This function allocates a fcurve key array.

   Returned Value: DK_Fcvkey structure

*******************************************************************************/

DK_Fcvkey *DK_fcvkeyAllocate 
    (
	int n
    )
{
    DK_Fcvkey *DK_fcvkey;

    if (DK_fcvkey = (struct DK_fcvkey *) calloc(1, sizeof( struct DK_fcvkey )
		        * _DK_BLOCK_SIZE(n, DK_KEY_MASK)))
    {
	DK_fcvkey->time 	= 0.0;
	DK_fcvkey->value 	= 0.0;
	DK_fcvkey->rslope 	= 0.0;
	DK_fcvkey->lslope 	= 0.0;
	DK_fcvkey->slopeFlag 	= 0;
	DK_fcvkey->interp 	= DK_CONSTANT_INTERP;
	DK_fcvkey->keyType 	= DK_FREEFORM;
	DK_fcvkey->spIndex  	= -1;
    }
    
    return(DK_fcvkey);
}


/*******************************************************************************

   $$L DK_fcurveArrayDispose

   This function disposes an array of activation fcurve structures.
   (The array itself is not freed.)

   Returned Value: None

*******************************************************************************/

void DK_actFcurveArrayDispose
    (
        DK_actFcurve fcv[],
        int n
    )
{
    if (fcv != NULL)
    {
        int i;

        for( i = 0; i < n; ++i )
        {
           if( fcv[i].fcv ) DK_fcurveDispose( &fcv[i].fcv );
        }
    }
}

/*******************************************************************************

   $$L DK_fcurveArrayDispose

   This function disposes an array of fcurve structures.
   (The array itself is not freed.)

   Returned Value: None

*******************************************************************************/

void DK_fcurveArrayDispose
    (
        DK_Fcurve *fcv[],
        int n
    )
{
    if (fcv != NULL)
    {
        int i;

        for( i = 0; i < n; ++i )
        {
           if( fcv[i] ) DK_fcurveDispose( &fcv[i] );
        }
    }
}

/*******************************************************************************

   $$L DK_fcurveDispose

   This function disposes a fcurve structure.

   Returned Value: None

*******************************************************************************/

void DK_fcurveDispose
    (
	DK_Fcurve **fcv
    )
{
    if (fcv != NULL && *fcv)
    {
	DK_fcvkeyDispose(&(*fcv)->keys);
	_DK_FREE(*fcv, -1);
    }
}

/*******************************************************************************

   $$L DK_fcvkeyDispose

   This function disposes a fcurve key array.

   Returned Value: None

*******************************************************************************/

void DK_fcvkeyDispose
    (
	DK_Fcvkey **DK_fcvkey
    )
{
    if (DK_fcvkey != NULL && *DK_fcvkey != NULL)
    {
	_DK_FREE(*DK_fcvkey, -1);
    }
}

/*******************************************************************************

   DK_fcurveCopy()

   Copies an function curve.  If the returned function curve doesn't
   exist yet, one will be allocated.

   Arguments:
       DK_Fcurve *from_fcv	:  original function curve
       DK_Fcurve **to_fcv	:  where to copy the function curve

*******************************************************************************/

void DK_fcurveCopy
   (
      DK_Fcurve	*from_fcv,
      DK_Fcurve	**to_fcv
   )
{
   DK_Fcvkey	*keys;

   if ( from_fcv == NULL )
      return;

   if ( to_fcv == NULL )
      return;

   if ( *to_fcv == NULL )
   {
      *to_fcv = DK_fcurveAllocate();
      if ( *to_fcv == NULL )
	 return;
   }

   if ( from_fcv->size > 0 )
   {
      if ( (*to_fcv)->size != from_fcv->size )
      {
	 if ( (*to_fcv)->size > 0 )
	    DK_fcvkeyDispose( &(*to_fcv)->keys );

	 keys = DK_fcvkeyAllocate( from_fcv->size );
      }

      memcpy( keys, from_fcv->keys, from_fcv->size * sizeof( DK_Fcvkey ) );
   }
   else
   {
      keys = NULL;
   }

   *(*to_fcv) = *from_fcv;
   (*to_fcv)->keys = keys;
}

/*******************************************************************************

   DK_fcurveValue()

   Return value of function curve at specified time (seconds).
   The fcurve can either be an integer or float fcurve. This is
   determine by the fcurve type.

   Arguments:
       fcurve *fnc       :  function curve
       float time        :  time at which value is returned

*******************************************************************************/


float DK_fcurveValue
    (
	DK_Fcurve *fnc,		/* fcurve pointer */
	float time		/* time at which value is returned */
    )
{
    float value;

    switch (fnc->type)
    {
        case DK_FCV_TYPE_MDL_HIDDEN:
        case DK_FCV_TYPE_LIG_HIDDEN:
        case DK_FCV_TYPE_CAM_HIDDEN:
        case DK_FCV_TYPE_MDL_ALGO:
	    value = (float) DK_fcurveIntegerValue(fnc, time, TRUE);
	    break;
	default:
	    value = DK_fcurveFloatValue(fnc, time);
	    break;
    }
    return(value);
}

#define DK_FcurveStepValue(value) \
    if (step == TRUE)\
    {\
	if (value > (float) DK_FCURVE_FALSE)\
	    return(DK_FCURVE_TRUE);\
	else\
	    return(DK_FCURVE_FALSE);\
    }\
    else\
	return((int) (value + 0.5));

/*******************************************************************************

   DK_fcurveIntegerValue()

   This function will return a boolean fcurve value if it is a step
   function, i.e.  if the fcurve value is <= DK_FCURVE_FALSE 
   then DK_FCURVE_FALSE is returned or else DK_FCURVE_TRUE is returned.
   If it is not a step function, the value returned is rounded off to
   the nearest integer.

   Arguments:
       fcurve *fnc       :  function curve
       float time        :  time at which value is returned

*******************************************************************************/

int DK_fcurveIntegerValue
    (
	DK_Fcurve *fnc,		/* fcurve pointer */
	float time,		/* time at which value is returned */
	int step		/* True if it is a step func */
    )
{
   struct DK_fcvkey  *key;
   float t, range;

   /* function has no keys */
   if( fnc == NULL || fnc->size == 0)
       return( DK_FCURVE_FALSE );

   /* time is before first key */
    if( time < fnc->keys[0].time ) 
    {
        switch( fnc->pre_extrap ) 
	{
            case  DK_CONSTANT_EXTRAP:
            case  DK_GRADIENT_EXTRAP:
		DK_FcurveStepValue(fnc->keys[0].value);
	        break;
            case  DK_CYCLE_EXTRAP:
                if( fnc->size == 1 )
		{
		    DK_FcurveStepValue(fnc->keys[0].value);
		}
                else 
		{
		    /* find the time within the fcurve range, 
			then interpolate */
        	    range = fnc->keys[fnc->size-1].time - fnc->keys[0].time;
		    t = fnc->keys[0].time;
		    while (time < t)
		        time = time + range;
	        }
	        break;
            /* illegal extrapolation */
            default:
		if ( DK_option.verbose == TRUE)
                    fprintf( DK_option.msgFd, "Error - fcurvePreValue: \
illegal extrapolation mode.\n");
                return(DK_FCURVE_FALSE);
	        break;
        }
    }


   /* time is after last key */
   if( time >= fnc->keys[fnc->size-1].time ) 
   {
        switch( fnc->post_extrap ) 
	{
            case  DK_CONSTANT_EXTRAP:
            case  DK_GRADIENT_EXTRAP:
		DK_FcurveStepValue(fnc->keys[fnc->size - 1].value);
	        break;
            case  DK_CYCLE_EXTRAP:
                if( fnc->size == 1 )
		{
		    DK_FcurveStepValue(fnc->keys[0].value);
		}
                else 
		{
		    /* find the time within the fcurve range,
		       then interpolate */
        	    range = fnc->keys[fnc->size-1].time - fnc->keys[0].time;
		    t = fnc->keys[fnc->size - 1].time;
		    while (time > t)
		        time = time - range;
	        }
	        break;
            /* illegal extrapolation */
            default:
		if ( DK_option.verbose == TRUE)
                    fprintf( DK_option.msgFd, "Error - fcurvePostValue: \
illegal extrapolation mode\n");
                return(DK_FCURVE_FALSE);
	        break;
        }
    }

    for( key = &fnc->keys[1]; time >= key->time; key++ )
	/* if time is equal to the key then just return the key value 
	   this is valid since hermite goes thru the key points */
	if (time == key->time)
	{
	    DK_FcurveStepValue(key->value);
	}

    key--;
    DK_FcurveStepValue(key->value);
}

/*******************************************************************************

   DK_fcurveFloatValue()

   Return value of function curve at specified time (seconds).

   Arguments:
       fcurve *fnc       :  function curve
       float time        :  time at which value is returned

*******************************************************************************/

float DK_fcurveFloatValue
    (
	DK_Fcurve *fnc,		/* fcurve pointer */
	float time		/* time at which value is returned */
    )
{
   struct DK_fcvkey  *key, *key2;
   float t0, v0, d0, t1, v1, d1, t, t2, value, slope, range;
   float new_value, t3_T2, _2t3_T3, _3t2_T2, t2_T, _1_T, _1_T2;

   /* function has no keys */
   if( fnc == NULL || fnc->size == 0)
       return( 0.0 );

   /* time is before first key */
    if( time < fnc->keys[0].time ) 
    {
        switch( fnc->pre_extrap ) 
	{
            case  DK_CONSTANT_EXTRAP:
                return( fnc->keys[0].value );
	        break;
            case  DK_GRADIENT_EXTRAP:
	        if (fnc->size == 1)
                   return( fnc->keys[0].value );
                key = &fnc->keys[0];
                key2 = &fnc->keys[1];
	        t = key2->time - key->time;
                if (fnc->interp ==  DK_LINEAR_INTERP && t != 0.0) 
                    slope = ( key2->value - key->value ) / t;
                else
                    slope = key->lslope;
                value = key->value - (key->time - time) * slope;
                return(value);
	        break;
            case  DK_CYCLE_EXTRAP:
                if( fnc->size == 1 )
                   return( fnc->keys[0].value );
                else 
		{
		    /* find the time within the fcurve range, 
			then interpolate */
        	    range = fnc->keys[fnc->size-1].time - fnc->keys[0].time;
		    t = fnc->keys[0].time;
		    while (time < t)
		        time = time + range;
	        }
	        break;
            /* illegal extrapolation */
            default:
		if ( DK_option.verbose == TRUE)
                    fprintf( DK_option.msgFd, "Error - fcurvePreValue: \
illegal extrapolation mode.\n");
                return(0.0);
	        break;
        }
    }


   /* time is after last key */
   if( time >= fnc->keys[fnc->size-1].time ) 
   {
        switch( fnc->post_extrap ) 
	{
            case  DK_CONSTANT_EXTRAP:
                return( fnc->keys[fnc->size-1].value );
	        break;
            case  DK_GRADIENT_EXTRAP:
	        if (fnc->size == 1)
                   return( fnc->keys[0].value );
                key = &fnc->keys[fnc->size - 2];
                key2 = &fnc->keys[fnc->size - 1];
	        t = key2->time - key->time;
                if (fnc->interp ==  DK_LINEAR_INTERP && t != 0.0) 
                    slope = ( key2->value - key->value ) / t;
                else
                    slope = key2->rslope;
                value = key2->value + (time - key2->time) * slope;
                return(value);
	        break;
            case  DK_CYCLE_EXTRAP:
                if( fnc->size == 1 )
                   return( fnc->keys[0].value );
                else 
		{
		    /* find the time within the fcurve range,
		       then interpolate */
        	    range = fnc->keys[fnc->size-1].time - fnc->keys[0].time;
		    t = fnc->keys[fnc->size - 1].time;
		    while (time > t)
		        time = time - range;
	        }
	        break;
            /* illegal extrapolation */
            default:
		if ( DK_option.verbose == TRUE)
                    fprintf( DK_option.msgFd, "Error - fcurvePostValue: \
illegal extrapolation mode\n");
                return(0.0);
	        break;
        }
    }

    for( key = &fnc->keys[1]; time >= key->time; key++ )
	/* if time is equal to the key then just return the key value 
	   this is valid since hermite goes thru the key points */
	if (time == key->time)
	    return(key->value);

    t1 = key->time;
    v1 = key->value;
    d1 = key->lslope;

    key--;
    t0 = key->time;
    v0 = key->value;
    d0 = key->rslope;

    switch( key->interp ) 
    {
        case  DK_CONSTANT_INTERP:
            return( v0 );
	    break;
        case  DK_LINEAR_INTERP:
            return( v0 + ( v1 - v0 ) * (time - t0) / (t1 - t0) );
	    break;
        case  DK_HERMITE_INTERP:
            t = time - t0;
            _1_T = 1.0 / (t1 - t0);
            _1_T2 = _1_T * _1_T;
            t2 = t * t;
            t2_T = t2 * _1_T;
            t3_T2 = t2 * t * _1_T2;
            _2t3_T3 = 2.0 * t3_T2 * _1_T;
            _3t2_T2 = 3.0 * t2 * _1_T2;
    
            new_value =  v0 * ( _2t3_T3 - _3t2_T2 + 1 )  + 
		         v1 * ( - _2t3_T3 + _3t2_T2 ) +
                         d0 * ( t3_T2 - t2_T - t2_T + t ) + 
		         d1 * ( t3_T2 - t2_T ) ;
	    return(new_value);
	    break;
        default:
	    return(0.0);
	    break;
    }
    return(0.0);
}


/*******************************************************************************

   $$L DK_fcvkeyWriteAsciiAnimation

   This function writes a fcurve key.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_fcvkeyWriteAsciiAnimation
    (
	 FILE *file,	/* file io pointer */
	DK_Fcvkey *DK_fcvkey  /* fkey pointer */
    )
{
    if (DK_fcvkey == NULL)
	return(FALSE);

    DK_incIndent(1);
    _DK_OUT_INDENT();

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_FCV_KEY_TIME_TOKEN);
    _DK_OUT_FVALUE(DK_fcvkey->time);

    _DK_OUT_4SPACES();
    _DK_OUT_TOKEN( DK_A_FCV_KEY_VALUE_TOKEN);
    _DK_OUT_FVALUE(DK_fcvkey->value);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_FCV_KEY_LSLOPE_TOKEN);
    _DK_OUT_FVALUE(DK_fcvkey->lslope);

    _DK_OUT_4SPACES();
    _DK_OUT_TOKEN( DK_A_FCV_KEY_RSLOPE_TOKEN);
    _DK_OUT_FVALUE(DK_fcvkey->rslope);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_FCV_KEY_INTERP_TOKEN);
    switch (DK_fcvkey->interp)
    {
	case  DK_CONSTANT_INTERP:
	    _DK_OUT_TOKEN( DK_A_CONSTANT_INTERP_TOKEN);
	    break;
	case  DK_LINEAR_INTERP:
	    _DK_OUT_TOKEN( DK_A_LINEAR_INTERP_TOKEN);
	    break;
	case  DK_HERMITE_INTERP:
	    _DK_OUT_TOKEN( DK_A_HERMITE_INTERP_TOKEN);
	    break;
    }

    _DK_OUT_4SPACES();
    _DK_OUT_TOKEN( DK_A_FCV_KEY_SFLAG_TOKEN);
    _DK_OUT_SVALUE( DK_fcvkey->slopeFlag);

    if (DK_fcvkey->keyType == DK_KEYPATH)
    {
        _DK_OUT_4SPACES();
        _DK_OUT_TOKEN( DK_A_FCV_KEY_SPINDEX_TOKEN);
        _DK_OUT_SVALUE( DK_fcvkey->spIndex);
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_FCV_KEY_KTYPE_TOKEN);
    switch (DK_fcvkey->keyType)
    {
	case  DK_FREEFORM:
	    _DK_OUT_TOKEN(DK_A_FCV_FREEFORM_TOKEN);
	    break;
	case  DK_KEYFRAME:
	    _DK_OUT_TOKEN(DK_A_FCV_KEYFRAME_TOKEN);
	    break;
	case  DK_KEYPATH:
	    _DK_OUT_TOKEN(DK_A_FCV_KEYPATH_TOKEN);
	    break;
	case  DK_RAWDATA:
	    _DK_OUT_TOKEN(DK_A_FCV_RAWDATA_TOKEN);
	    break;
    }
    DK_incIndent(-1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_fcvkeyReadAsciiAnimation

   This function reads a fcurve key.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_fcvkeyReadAsciiAnimation
    (
	 FILE *file,	/* file io pointer */
	DK_Fcvkey *DK_fcvkey  /* fkey pointer */
    )
{
    /* initialize key */
    if (DK_fcvkey != NULL)
    {
	DK_fcvkey->time = 0.0;
	DK_fcvkey->value = 0.0;
	DK_fcvkey->rslope = 0.0;
	DK_fcvkey->lslope = 0.0;
	DK_fcvkey->slopeFlag = 0;
	DK_fcvkey->interp =  DK_LINEAR_INTERP;
	DK_fcvkey->keyType =  DK_FREEFORM;
	DK_fcvkey->spIndex = -1;
    }

    while (TRUE)
    {
	if (strcmp( DK_A_FCV_KEY_TIME_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	    if (DK_fcvkey != NULL)
	        DK_fcvkey->time = DK_animFValue;
	}
	else if (strcmp( DK_A_FCV_KEY_VALUE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	    if (DK_fcvkey != NULL)
	        DK_fcvkey->value = DK_animFValue;
	}
	else if (strcmp( DK_A_FCV_KEY_LSLOPE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	    if (DK_fcvkey != NULL)
	        DK_fcvkey->lslope = DK_animFValue;
	}
	else if (strcmp( DK_A_FCV_KEY_RSLOPE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animFValue_na(file);
	    if (DK_fcvkey != NULL)
	        DK_fcvkey->rslope = DK_animFValue;
	}
	else if (strcmp( DK_A_FCV_KEY_INTERP_TOKEN, DK_animToken) == 0)
	{
	    if (DK_in_animToken(file) == FALSE)
		return(FALSE);
	    if (DK_fcvkey != NULL)
	    {
		if (strcmp(DK_A_CONSTANT_INTERP_TOKEN, DK_animToken) == 0)
	            DK_fcvkey->interp =  DK_CONSTANT_INTERP;
		else if (strcmp(DK_A_LINEAR_INTERP_TOKEN, DK_animToken) == 0)
	            DK_fcvkey->interp =  DK_LINEAR_INTERP;
		else if (strcmp(DK_A_HERMITE_INTERP_TOKEN, DK_animToken) == 0)
	            DK_fcvkey->interp =  DK_HERMITE_INTERP;
		else if ( DK_option.verbose == TRUE)
		    fprintf( DK_option.msgFd, "DK_FcvkeyReadAsciiAnimation: \
error, unknown interp DK_animToken '%s'.\n", DK_animToken);
	    }
	}
	else if (strcmp( DK_A_FCV_KEY_SFLAG_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (DK_fcvkey != NULL)
	        DK_fcvkey->slopeFlag = DK_animSValue;
	    if (DK_animFileVersion <= 2.02)
	    {
	        /* terminate the DK_fcvkey read */
	        DK_in_animToken(file);
	        return(TRUE);
	    }
	}
	else if (strcmp( DK_A_FCV_KEY_SPINDEX_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (DK_fcvkey != NULL)
	        DK_fcvkey->spIndex = DK_animSValue;
	}
	else if (strcmp( DK_A_FCV_KEY_KTYPE_TOKEN, DK_animToken) == 0)
	{
	    if (DK_in_animToken(file) == FALSE)
		return(FALSE);
	    if (DK_fcvkey != NULL)
	    {
		if (strcmp(DK_A_FCV_FREEFORM_TOKEN, DK_animToken) == 0)
	            DK_fcvkey->keyType =  DK_FREEFORM;
		else if (strcmp(DK_A_FCV_KEYFRAME_TOKEN, DK_animToken) == 0)
	            DK_fcvkey->keyType =  DK_KEYFRAME;
		else if (strcmp(DK_A_FCV_KEYPATH_TOKEN, DK_animToken) == 0)
	            DK_fcvkey->keyType =  DK_KEYPATH;
		else if (strcmp(DK_A_FCV_RAWDATA_TOKEN, DK_animToken) == 0)
	            DK_fcvkey->keyType =  DK_RAWDATA;
		else if ( DK_option.verbose == TRUE)
		    fprintf( DK_option.msgFd, "DK_FcvkeyReadAsciiAnimation: \
error, unknown keytype DK_animToken '%s'.\n", DK_animToken);
	    }
	    /* terminate the fcvkey read */
	    DK_in_animToken(file);
	    return(TRUE);
	}
	else 
	{
	    return(TRUE);
	}
        if (DK_in_animToken(file) == FALSE)
	    return(FALSE);
    }

}

/*******************************************************************************

   $$L DK_fcurveWriteAsciiAnimation

   This function writes a fcurve.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_fcurveWriteAsciiAnimation
    (
	 FILE *file, 	/* file io pointer */
	DK_Fcurve *fcv 	/* fcurve pointer */
    )
{
    int i;

    if (fcv == NULL)
	return(FALSE);

    DK_incIndent(1);
    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_FCV_TYPE_TOKEN);
    _DK_OUT_TOKEN(  DK_fcv_name[fcv->type].str);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_FCV_PRE_EXP_TOKEN);
    _DK_OUT_SVALUE( fcv->pre_extrap);

    _DK_OUT_4SPACES();
    _DK_OUT_TOKEN( DK_A_FCV_POST_EXP_TOKEN);
    _DK_OUT_SVALUE( fcv->post_extrap);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_FCV_FLAG_TOKEN);
    _DK_OUT_SVALUE( fcv->slopeFlag);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_FCV_INTERP_TOKEN);
    switch (fcv->interp)
    {
	case  DK_CONSTANT_INTERP:
	    _DK_OUT_TOKEN(DK_A_CONSTANT_INTERP_TOKEN);
	    break;
	case  DK_LINEAR_INTERP:
	    _DK_OUT_TOKEN(DK_A_LINEAR_INTERP_TOKEN);
	    break;
	case  DK_HERMITE_INTERP:
	    _DK_OUT_TOKEN(DK_A_HERMITE_INTERP_TOKEN);
	    break;
    }

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN( DK_A_FCV_NBKEYS_TOKEN);
    _DK_OUT_SVALUE( fcv->size);

    for (i = 0; i < fcv->size; i++)
	DK_fcvkeyWriteAsciiAnimation(file, &fcv->keys[i]);

    DK_incIndent(-1);
    return(TRUE);
}

/*******************************************************************************

   $$L DK_fcurveReadAsciiAnimation

   This function allocates and reads a fcurve.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_fcurveReadAsciiAnimation
    (
	 FILE *file, 	/* file io pointer */
	DK_Fcurve **fcv   /* fcurve pointer */
    )
{
    int i, first = TRUE;
    DK_SVALUE size = 0;

    if (fcv != NULL) 
    {
	if (*fcv != NULL)
	    DK_fcurveDispose(fcv);
        *fcv = DK_fcurveAllocate();
    }

    while (TRUE)
    {
        if (strcmp( DK_A_FCV_TYPE_TOKEN, DK_animToken) == 0)
	{
            if (DK_in_animToken(file) == FALSE)
	        return(FALSE);
	    if (fcv != NULL && *fcv != NULL)
	    {
		for (i = 0; i < DK_FCV_NUM; i++)
		    if (strcmp( DK_fcv_name[i].str, DK_animToken) == 0)
		    {
			 (*fcv)->type = (short) i;
			 break;
		    }
	    }
	}
        else if (strcmp( DK_A_FCV_NBKEYS_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
  	    size = DK_animSValue;
            if (DK_in_animToken(file) == FALSE)
	        return(FALSE);
	    if (fcv != NULL && *fcv != NULL) 
	    {
	        (*fcv)->size = size;
		if ((*fcv)->size > 0)
		    (*fcv)->keys = DK_fcvkeyAllocate((*fcv)->size);
		else
		    (*fcv)->keys = NULL;
	        for (i = 0; i < (*fcv)->size; i++) 
		{
		    DK_fcvkeyReadAsciiAnimation(file, &(*fcv)->keys[i]);
	        }
	    }
	    else 
	    {
	        for (i = 0; i < size; i++)
		    DK_fcvkeyReadAsciiAnimation(file, NULL);
	    }
	    return(TRUE);
	}
        else if (strcmp( DK_A_FCV_PRE_EXP_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (fcv != NULL && *fcv != NULL)
	        (*fcv)->pre_extrap = DK_animSValue;
	}
        else if (strcmp( DK_A_FCV_POST_EXP_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (fcv != NULL && *fcv != NULL)
	        (*fcv)->post_extrap = DK_animSValue;
	}
        else if (strcmp( DK_A_FCV_FLAG_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animSValue_na(file);
	    if (fcv != NULL && *fcv != NULL)
	        (*fcv)->slopeFlag = DK_animSValue;
	}
        else if (strcmp( DK_A_FCV_INTERP_TOKEN, DK_animToken) == 0)
	{
            if (DK_in_animToken(file) == FALSE)
	        return(FALSE);
	    if (fcv != NULL && *fcv != NULL)
	    {
		if (strcmp(DK_A_CONSTANT_INTERP_TOKEN, DK_animToken) == 0)
	            (*fcv)->interp =  DK_CONSTANT_INTERP;
		else if (strcmp(DK_A_LINEAR_INTERP_TOKEN, DK_animToken) == 0)
	            (*fcv)->interp =  DK_LINEAR_INTERP;
		else if (strcmp(DK_A_HERMITE_INTERP_TOKEN, DK_animToken) == 0)
	            (*fcv)->interp =  DK_HERMITE_INTERP;
		else if ( DK_option.verbose == TRUE)
		    fprintf( DK_option.msgFd, "DK_FcurveReadAsciiAnimation: \
error, unknown interp DK_animToken '%s'.\n", DK_animToken);
	     }
	}
	else 
	{
 	    if (first)
		DK_in_animToken(file);
	    if (*fcv != NULL && ((*fcv)->size <= 0 || (*fcv)->keys == NULL))
	    {
		DK_fcurveDispose(fcv);
		*fcv = NULL;
	    }
	    return(TRUE);
	}
        if (DK_in_animToken(file) == FALSE)
	    return(FALSE);
	first = FALSE;
    }

}
