/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyFace.c,v $
   $Revision: 1.12.14.1 $ $Date: 1995/06/09 21:59:56 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's face functions,
   its content is sufficient to read, write and manage face data structure
   into SOFTIMAGE's file format...

   Written by: Laurent Lauzon

   (c) Copyright 1990, 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinyFace.h"

extern DK_String	DK_BezConstraintName[];
extern DK_String	DK_BezSegmentName[];
extern DK_String	DK_SplineTypeName[];

#define DBUG   0 

/******************************************************************************/
/*** Static function prototypes ***********************************************/
/******************************************************************************/

/******************************************************************************/
/*** Global functions *********************************************************/
/******************************************************************************/

DK_Face *DK_faceAllocate( void )
{
   DK_Face *fac;

   if( fac = (DK_Face *) calloc( 1, sizeof( DK_Face ) ) )
   {
      fac->exterior = NULL;
      fac->nbHoles  = 0;
      fac->holes    = NULL;
      fac->crv      = NULL;
      fac->isSpline = TRUE;
   }

   return( fac );
}

void DK_faceDispose( DK_Face **fac )
{
   int		 i;

   if( *fac != NULL )
   {
     if ( (*fac)->isSpline)   
     {
	 if( (*fac)->exterior )
	    DK_splineDispose( &(*fac)->exterior );
	 for( i = 0; i < (*fac)->nbHoles; i++ )
	    DK_splineDispose( &(*fac)->holes[i] );

         /* +SA 4/21/94 to plug memory leak */
         if ((*fac)->holes)
           free ((*fac)->holes);

	 free( (*fac ) );
	 *fac = NULL;
      }
      else
      {
	 /* delete a NURBS face and its holes*/
	 if ( (*fac)->crv )
         {
            for( i = 0; i < (*fac)->nbHoles; i++ )
	       DK_nurbsCrvDispose(  &(*fac)->crv[i] );
	    free( (*fac ) );
	    *fac = NULL;
         }
      }
   }
}

DK_Boolean DK_faceWriteBinary(  FILE *file, DK_Face *fac, DK_FileProt *prot )
{
   short    ind, i;

   for( ind = 0; prot->face[ ind ] != 0; ind++ )
   {
      #if DBUG
      printf(" DK_faceWriteBinary writing token %d\n",prot->face[ ind ]);
      #endif

      switch( prot->face[ ind ] )
      {
         case 1:	/* exterior definition */
	    if ( fac->isSpline )
	       DK_splineWriteBinary( file, fac->exterior, prot );
            break;

         case 2:  	/* holes list */
	    if ( fac->isSpline )
	    {
	       DK_byteswap_fwrite( &fac->nbHoles, 2, 1, file );
	       for( i = 0; i < fac->nbHoles; i++ )
		  DK_splineWriteBinary( file, fac->holes[ i ], prot );
            }
	    break;
	 case 3:  {     /* is Spline boolean */
            short spline = fac->isSpline;
	    DK_byteswap_fwrite( &spline, 2, 1, file);
	    break;
         }
	 case 4:       /* NURBS face curves */
	    if ( !fac->isSpline )
	    {
	       DK_byteswap_fwrite( &fac->nbHoles, 2, 1, file );
	       for( i = 0; i < fac->nbHoles; i++ )
		  DK_nurbsCrvBinaryWrite( file,fac->crv[i]);
	    }
	    break;
      }
   }

   return( TRUE );
}

DK_Face *DK_faceReadBinary(  FILE *file, DK_FileProt *prot )
{
   DK_Face	 *fac = NULL;
   short	 i, ind = 0, tok;
   short 	 spline = 0;

   if( (fac = DK_faceAllocate()) == NULL )
      return( NULL );

   if( (prot != NULL) && (prot->face != NULL) )
      tok = prot->face[ ind ++ ];
   else
      DK_byteswap_fread( &tok, 2, 1, file );
   while( tok != 0 /* End of face definition */ )
   {
      #if DBUG
      printf(" DK_faceReadBinary reading token %d\n",tok);
      fflush( stdout );
      #endif
      switch( tok )
      {
         case 1:  /* exterior definition */
	    if ( fac->isSpline )
	       fac->exterior = DK_splineReadBinary( file, prot );
            break;

         case 2:  /* holes list */
	    if ( fac->isSpline )
	    {
	       DK_byteswap_fread( &fac->nbHoles, 2, 1, file );
	       if( fac->nbHoles > 0 )
		  fac->holes = (DK_Spline **) calloc( fac->nbHoles, sizeof( DK_Spline * ) );
	       for( i = 0; i < fac->nbHoles; i++ )
		  fac->holes[i] = DK_splineReadBinary( file, prot );
            }
	    break;
         case 3:   /* isSpline boolean */
	    DK_byteswap_fread( &spline,2,1, file );
	    fac->isSpline = spline ;
	    break;
	 case 4:
	    if ( !fac->isSpline ) 
	    {
	       DK_byteswap_fread( &fac->nbHoles,  2, 1, file );
	       if( fac->nbHoles > 0 )
		  fac->crv = (DK_NurbsCrv**) calloc( fac->nbHoles, sizeof( DK_NurbsCrv * ) );
	       for( i = 0; i < fac->nbHoles; i++ )
		  fac->crv[i] = DK_nurbsCrvBinaryRead( file, prot);
	    }
	    break;

         default:
            DK_fileSkipField( file, tok );
      }
      if( (prot != NULL) && (prot->face != NULL) )
         tok = prot->face[ ind ++ ];
      else
         DK_byteswap_fread( &tok, 2, 1, file );
   }

   return( fac );
}

DK_Boolean DK_faceWriteAscii(  FILE *file, DK_Face *fac, DK_String space )
{
   DK_Spline	*spl = NULL;
   DK_SplineKey	*key = NULL;
   DK_NurbsCrv  *crv = NULL;
   int		 i, j;
   
   if( fac->isSpline )
   {
      fprintf( file, "\n%sface\n", space );
      fprintf( file, "%s{\n", space );
      spl = fac->exterior;
      fprintf( file, "%s  type       %s\n", space, DK_SplineTypeName[ spl->type ]);
      fprintf( file, "%s  nbKeys     %hd\n", space, spl->nbKeys );
      fprintf( file, "%s  tension    %*.*f\n", space,  DK_FM,  DK_FD, spl->tension );
      fprintf( file, "%s  step       %hd\n", space, spl->step );

      if( spl->type ==  DK_SPL_BEZIER )
      {
         fprintf( file, "%s  constraint %s\n", space,
                        DK_BezConstraintName[ spl->constraint ] );
         fprintf( file, "%s  segment    %s\n", space,
                        DK_BezSegmentName[ spl->segment ] );
      }

      fprintf( file, "\n%s  controlPoints\n", space );
      fprintf( file, "%s  {\n", space );
      for( i = 0, key = spl->keys; i < spl->nbKeys; i++, key++ )
      {
         fprintf( file, "%s    [%1d]  position   %*.*f  %*.*f  %*.*f\n", space,
                        i,  DK_FM,  DK_FD, key->position.x,  DK_FM,  DK_FD, key->position.y,
                         DK_FM,  DK_FD, key->position.z );
         if( spl->type ==  DK_SPL_BEZIER )
         {
            fprintf( file, "%s         previous   %*.*f  %*.*f  %*.*f\n", space,
                            DK_FM,  DK_FD, key->previous.x,  DK_FM,  DK_FD, key->previous.y,
                            DK_FM,  DK_FD, key->previous.z );
            fprintf( file, "%s         next       %*.*f  %*.*f  %*.*f\n", space,
                            DK_FM,  DK_FD, key->next.x,  DK_FM,  DK_FD, key->next.y,
                            DK_FM,  DK_FD, key->next.z );
            fprintf( file, "%s         constraint %s\n", space,
                           DK_BezConstraintName[ key->constraint ] );
            fprintf( file, "%s         segment    %s\n", space,
                           DK_BezSegmentName[ key->segment ] );
         }
         if( key->flag &  DK_KEY_SELECTED )
            fprintf( file, "%s         flag    ( SELECTED )\n", space );
      }
      fprintf( file, "%s  }\n", space );

      for( j = 0; j < fac->nbHoles; j++ )
      {
         spl = fac->holes[ j ];
         fprintf( file, "\n%s  hole     [%1hd]\n", space, j );
         fprintf( file, "%s  {\n", space );
         fprintf( file, "%s    type       %s\n", space,
                        DK_SplineTypeName[ spl->type ]);
         fprintf( file, "%s    nbKeys     %hd\n", space, spl->nbKeys );
         fprintf( file, "%s    tension    %*.*f\n", space,  DK_FM,  DK_FD,
                        spl->tension );
         fprintf( file, "%s    step       %hd\n", space, spl->step );

         if( spl->type ==  DK_SPL_BEZIER )
         {
            fprintf( file, "%s    constraint %s\n", space,
                           DK_BezConstraintName[ spl->constraint ] );
            fprintf( file, "%s    segment    %s\n", space,
                           DK_BezSegmentName[ spl->segment ] );
         }

         fprintf( file, "\n%s    controlPoints\n", space );
         fprintf( file, "%s    {\n", space );
         for( i = 0, key = spl->keys; i < spl->nbKeys; i++, key++ )
         {
            fprintf( file, "%s      [%1d]  position   %*.*f  %*.*f  %*.*f\n",
                           space, i,  DK_FM,  DK_FD, key->position.x,  DK_FM,  DK_FD,
                           key->position.y,  DK_FM,  DK_FD, key->position.z );
            if( spl->type ==  DK_SPL_BEZIER )
            {
               fprintf( file, "%s           previous   %*.*f  %*.*f  %*.*f\n",
                              space,  DK_FM,  DK_FD, key->previous.x,  DK_FM,  DK_FD,
                              key->previous.y,  DK_FM,  DK_FD, key->previous.z );
               fprintf( file, "%s           next       %*.*f  %*.*f  %*.*f\n",
                              space,  DK_FM,  DK_FD, key->next.x,  DK_FM,  DK_FD, key->next.y,
                               DK_FM,  DK_FD, key->next.z );
               fprintf( file, "%s           constraint %s\n", space,
                              DK_BezConstraintName[ key->constraint ] );
               fprintf( file, "%s           segment    %s\n", space,
                              DK_BezSegmentName[ key->segment ] );
            }
            if( key->flag &  DK_KEY_SELECTED )
               fprintf( file, "%s           flag    ( SELECTED )\n", space );
         }
         fprintf( file, "%s    }\n", space );
         fprintf( file, "%s  }\n", space );
      }
   }
   else
   {
      #if DBUG
      printf(" DK_faceWriteAscii: writing NURBS_face \n");
      printf(" DK_faceWriteAscii: writing %d holes \n",fac->nbHoles);
      fflush( stdout );
      #endif

      fprintf( file, "\n%sNURBS_face: ", space );
      fprintf( file, "holes %d\n", fac->nbHoles - 1);
      fprintf( file, "%s{\n", space );
      /* NURBS face */
      for ( i = 0 ; i < fac->nbHoles; i++)
      {
	 if ( i == 0 )
	    /* exterieur */
	    fprintf ( file ,"\n%s  exterior   \n", space);
	 else
	    fprintf( file, "\n%s  hole     %d\n", space, i );
	 DK_nurbsCrvAsciiWrite( file, fac->crv[i], space );
      }
   }
   fprintf( file, "%s}\n", space );
   return( TRUE );
}


DK_Face *DK_faceReadAscii(  FILE *file )
{
   DK_Face              *fac;
   int           i;

   if( (fac = DK_faceAllocate()) == NULL )
      return( NULL );

   fac->exterior = DK_splineReadAscii( file, &fac->nbHoles, &fac->holes );

   fac->exterior->close = TRUE;
   for( i = 0; i < fac->nbHoles; i++ )
      fac->holes[i]->close = TRUE;
   return( fac );
}

DK_Face *DK_nurbsFaceReadAscii(  FILE *file )
{
   DK_Face	*fac = NULL;
   int		i = 0;
   int 		holeNumber = 0;
   char         str[ 80 ];
  
   if( (fac = DK_faceAllocate()) == NULL )
      return( NULL );
   /* read the "holes:" string */
   fscanf( file, "%s", str );

   /* NURBS face */
   fac->isSpline = 0;

   /* read the hole number */
   fscanf( file, "%s", str);
   fac->nbHoles = atoi( str );

   /* increment the hole # for the exterior */
   fac->nbHoles++;


   #if DBUG
   printf(" DK_faceReadAscii: reading NURBS_face \n");
   printf(" DK_faceReadAscii: reading %d holes \n",fac->nbHoles);

   fflush( stdout );
   #endif


   fscanf( file, "%s", str);
   if( strcmp( str, "{" ) != 0 )
   {
      /* error */
      DK_faceDispose( &fac );
      return( fac );
   }


   if ( fac->nbHoles > 0 )
   {
      /* allocate memory for holes and exterior */
      fac->crv = (DK_NurbsCrv**) calloc( fac->nbHoles, sizeof( DK_NurbsCrv * ) );
   }

   for ( i = 0 ; i < fac->nbHoles; i++)
   {
       fscanf( file, "%s", str );
       if ( strcmp( str, "hole" ) != 0 &&  strcmp( str, "exterior" ) != 0 )
       {
	  /* error */
	  DK_faceDispose( &fac );
	  return( fac );
       }
       else
       {
	  if ( strcmp( str, "hole" ) == 0 )
	     /* read the hole # */
	     fscanf( file, "%s", str );

	  fscanf( file, "%s", str );
	  if ( strcmp( str, "Nurbs_Curve" ) != 0 )
	  {
	      /* error */
	      DK_faceDispose( &fac );
	      return( fac );
	  }
	  fac->crv[i] = DK_nurbsCrvAsciiRead( file );
       }
   }
   return( fac );
}
