/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyDynamics.c,v $
   $Revision: 1.17.14.1 $ $Date: 1995/06/09 21:59:55 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's dynamic functions.

   Written by: Colin Hui

   (c) Copyright 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinySceneToken.h"
#include "tinyUtils.h"
#include "tinyModel.h"

/*******************************************************************************

   $$L DK_dynaMatterInfoAllocate

   This function allocates a dynamic matter structure.

   Returned Value: DK_DynaMatterInfo structure

*******************************************************************************/

DK_DynaMatterInfo *DK_dynaMatterInfoAllocate( void )
{
   DK_DynaMatterInfo *dm;

   dm = (DK_DynaMatterInfo *) calloc( 1, sizeof( DK_DynaMatterInfo ));

   dm->density    = DK_DYN_DEF_DENSITY;
   dm->mu_s       = DK_DYN_DEF_MU_STATIC;
   dm->mu_k       = DK_DYN_DEF_MU_KINETIC;
   dm->elasticity = DK_DYN_DEF_ELASTICITY;
   dm->rigidity   = DK_DYN_DEF_RIGIDITY;
   dm->plasticity = DK_DYN_DEF_PLASTICITY;
   dm->node_cdlevel = DK_GENERAL;

   return( dm );
}

/*******************************************************************************

   $$L DK_dynaJntInfoAllocate

   This function allocates a dynamic chain variable structure.

   Returned Value: DK_DynaJntInfo structure

*******************************************************************************/

DK_DynaJntInfo *DK_dynaJntInfoAllocate( void )
{
   DK_DynaJntInfo *dj;

   dj = (DK_DynaJntInfo *) calloc( 1, sizeof( DK_DynaJntInfo ));

   dj->friction.x = 4.0 * DK_DYN_DEF_FRICT;
   dj->friction.y = DK_DYN_DEF_FRICT;
   dj->friction.z = DK_DYN_DEF_FRICT;

   return( dj );
}

/*******************************************************************************

   $$L DK_dynaConstrInfoAllocate

   This function allocates a dynamic constraint structure.

   Returned Value: DK_DynaConstrInfo structure

*******************************************************************************/

DK_DynaConstrInfo *DK_dynaConstrInfoAllocate( void )
{
   DK_DynaConstrInfo *di;

   if( di = (DK_DynaConstrInfo *) calloc( 1, sizeof( DK_DynaConstrInfo ) ) )
   {
       di->type = -1;
       di->n = 0;
       di->model_name = (char *)NULL;
       di->obstype = DK_BPLANE;
       di->next = (DK_DynaConstrInfo *)NULL;
   }

   return( di );
}

/*******************************************************************************

   $$L DK_dynaChnVarAllocate

   This function allocates a dynamic chain variable structure.

   Returned Value: DK_DynaChnVars structure

*******************************************************************************/

DK_DynaChnVars *DK_dynaChnVarAllocate( void )
{
   DK_DynaChnVars *di;

   di = (DK_DynaChnVars *) calloc( 1, sizeof( DK_DynaChnVars ));

   di->dorcpos   = TRUE;
   di->dorcori   = FALSE;
   di->dograv    = TRUE;
   di->dofloor   = FALSE;
   di->dowind    = TRUE;

   di->rcposaxe[0] = TRUE;
   di->rcposaxe[1] = TRUE;
   di->rcposaxe[2] = TRUE;
   di->rcoriaxe[0] = TRUE;
   di->rcoriaxe[1] = TRUE;
   di->rcoriaxe[2] = FALSE;
   di->rcposacc    = TRUE;
   di->rcoriacc    = FALSE;

   di->cdetect_level     = DK_BBOX;
   di->ddetect_level     = DK_BBOX;
   di->wind_level        = DK_BBOX;
   di->fan_maxsubdiv     = 1;

   di->geomupdate        = FALSE;
   return( di );
}

/*******************************************************************************

   $$L DK_dynaMatterInfoDispose

   This function disposes a DK_DynaMatterInfo structure.

   Returned Value: None

*******************************************************************************/

void DK_dynaMatterInfoDispose
    (
	DK_DynaMatterInfo **dm
    )
{
    if (dm != NULL)
	_DK_FREE(*dm, -1);
}

/*******************************************************************************

   $$L DK_dynaJntInfoDispose

   This function disposes a DK_DynaJntInfo structure.

   Returned Value: None

*******************************************************************************/

void DK_dynaJntInfoDispose
    (
	DK_DynaJntInfo **dj
    )
{
    if (dj != NULL)
	_DK_FREE(*dj, -1);
}

/*******************************************************************************

   $$L DK_dynaConstrInfoDispose

   This function disposes a dynaConstrInfo structure.

   Returned Value: None

*******************************************************************************/

void DK_dynaConstrInfoDispose
    (
	DK_DynaConstrInfo **di
    )
{
    if (di != NULL)
    {
        DK_DynaConstrInfo *d = *di;

        while ( d )
        {
           DK_DynaConstrInfo *save = d;
           _DK_FREE( d->model_name, -1 );
           d = d->next;
           free( save );
        }

        *di = NULL;
    }
}

/*******************************************************************************

   $$L DK_dynaChnVarDispose

   This function disposes a dynaChnVars structure.

   Returned Value: None

*******************************************************************************/

void DK_dynaChnVarDispose
    (
	DK_DynaChnVars **di
    )
{
    if (di != NULL)
	_DK_FREE(*di, -1);
}

/*******************************************************************************

   $$L shapeLevelAsciiWrite

   - Write a shape level token into a file.

   Returned Value: None.

*******************************************************************************/

static void shapeLevelAsciiWrite
   (
       FILE	*file,
      short	level
   )
{
   switch ( level )
   {
      case DK_GENERAL :
	 _DK_OUT_TOKEN( DK_A_DYNSETUP_GENERAL_TOKEN );
	 break;
      case DK_BPLANE :
	 _DK_OUT_TOKEN( DK_A_DYNSETUP_BPLANE_TOKEN );
	 break;
      case  DK_BSPHERE :
	 _DK_OUT_TOKEN( DK_A_DYNSETUP_BSPHERE_TOKEN );
	 break;
      case DK_BBOX :
	 _DK_OUT_TOKEN( DK_A_DYNSETUP_BBOX_TOKEN );
	 break;
      case DK_SHAPE :
	 _DK_OUT_TOKEN( DK_A_DYNSETUP_SHAPE_TOKEN );
	 break;
   }
}

/*******************************************************************************

   $$L shapeLevelAsciiRead

   - Reads in a shape level token

   Returned Value: resulting shape level.

*******************************************************************************/

static short shapeLevelAsciiRead
   (
       FILE	*file
   )
{
   short	level;

   DK_in_animToken( file );
   if ( strcmp( DK_A_DYNSETUP_BSPHERE_TOKEN, DK_animToken ) == 0 )
   {
      level =  DK_BSPHERE;
   }
   else if ( strcmp( DK_A_DYNSETUP_BBOX_TOKEN, DK_animToken ) == 0 )
   {
      level = DK_BBOX;
   }
   else if ( strcmp( DK_A_DYNSETUP_BPLANE_TOKEN, DK_animToken ) == 0 )
   {
      level = DK_BPLANE;
   }
   else if ( strcmp( DK_A_DYNSETUP_SHAPE_TOKEN, DK_animToken ) == 0 )
   {
      level = DK_SHAPE;
   }
   else if ( strcmp( DK_A_DYNSETUP_GENERAL_TOKEN, DK_animToken ) == 0 )
   {
      level = DK_GENERAL;
   }
   else
   {
      level = DK_BBOX;
      if ( DK_option.verbose == TRUE)
          fprintf( DK_option.msgFd,"Error, unknown shape level token '%s'.", 
		DK_animToken);
   }
   return( level );
}

/*******************************************************************************

   $$L DK_dynamatterWriteBinaryChunk

   - Write dynamic matter data to file.

   Returned Value: None.

*******************************************************************************/

void DK_dynamatterWriteBinaryChunk
   (
       FILE              *file,
      DK_DynaMatterInfo    *p
   )
{
   int	entier;

   /* 1 : density   */
   DK_byteswap_fwrite( &(p->density),    sizeof(float), 1, file );

   /* 2 : roughness (static) */
   DK_byteswap_fwrite( &(p->mu_s),  sizeof(float), 1, file );

   /* 3 : elasticity */
   DK_byteswap_fwrite( &(p->elasticity), sizeof(float), 1, file );

   /* 4 : rigidity */
   DK_byteswap_fwrite( &(p->rigidity),   sizeof(float), 1, file );

   /* 5 : plasticity */
   DK_byteswap_fwrite( &(p->plasticity), sizeof(float), 1, file );

   /* 6 : roughness (kinetic) */
   DK_byteswap_fwrite( &(p->mu_k),  sizeof(float), 1, file );

   /* 7 : node collision detection level */
   entier = (int) p->node_cdlevel;
   DK_byteswap_fwrite( &entier,  sizeof(int), 1, file );
}

/*******************************************************************************

   $$L DK_dynamatterReadBinaryChunk

   - Read dynamic matter data from file.

   Returned Value: pointer to dynamic matter structure

*******************************************************************************/

DK_DynaMatterInfo *DK_dynamatterReadBinaryChunk
   (
       FILE              *file,
      DK_FileProt        *prot
   )
{
   short           tok, ind = 0;
   int		   entier;
   DK_DynaMatterInfo *p;

   p = DK_dynaMatterInfoAllocate();
   if ( p == NULL &&  DK_option.verbose == TRUE)
   {
      fprintf( DK_option.msgFd,"DK_dynamatterReadBinaryChunk : Allocate failed ! \n");
      return( NULL );
   }

   if( prot->physical != NULL )
      tok = prot->physical[ ind ++ ];
   else
      DK_byteswap_fread( &tok, sizeof(short), 1, file );

   while( tok != 0 ) /* End of physical properties definition */
   {
      switch( tok )
      {
         case 1: /* density */
            DK_byteswap_fread( &(p->density),   sizeof(float), 1, file );
            break;
         case 2: /* roughness (static) */
            DK_byteswap_fread( &(p->mu_s), sizeof(float), 1, file );
            break;
         case 3: /* elasticity */
            DK_byteswap_fread( &(p->elasticity),sizeof(float), 1, file );
            break;
         case 4: /* rigidity */
            DK_byteswap_fread( &(p->rigidity),  sizeof(float), 1, file );
            break;
         case 5: /* plasticity */
            DK_byteswap_fread( &(p->plasticity),  sizeof(float), 1, file );
            break;
         case 6: /* roughness (kinetic) */
            DK_byteswap_fread( &(p->mu_k), sizeof(float), 1, file );
            break;
	 case 7: /* node coll. detection level */
            DK_byteswap_fread( &entier, sizeof(int), 1, file );
	    p->node_cdlevel = entier;
	    break;

         default:
            DK_fileSkipField( file, tok );

      }
      if( prot->physical != NULL )
         tok = prot->physical[ ind ++ ];
      else
         DK_byteswap_fread( &tok, sizeof(short), 1, file );
   }

   return( p );
}

/*******************************************************************************

   $$L DK_dynamatterWriteAscii

   - Write physical properties data to file.

   Returned Value: None.

*******************************************************************************/

void DK_dynamatterWriteAscii
   (
       FILE 		*file,
      DK_DynaMatterInfo	*pmat,
      char		*space
   )
{
   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN( DK_A_DYNMATTER_DENSITY_TOKEN );
   fprintf(file,"                 ");
   _DK_OUT_FVALUE( pmat->density );

   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN( DK_A_DYNMATTER_ELASTICITY_TOKEN );
   fprintf(file,"              ");
   _DK_OUT_FVALUE( pmat->elasticity );

   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN( DK_A_DYNMATTER_SROUGHNESS_TOKEN );
   fprintf(file,"        ");
   _DK_OUT_FVALUE( pmat->mu_s );

   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN( DK_A_DYNMATTER_KROUGHNESS_TOKEN );
   fprintf(file,"       ");
   _DK_OUT_FVALUE( pmat->mu_k );

   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN( DK_A_DYNMATTER_RIGIDITY_TOKEN );
   fprintf(file,"                ");
   _DK_OUT_FVALUE( pmat->rigidity );

   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN( DK_A_DYNMATTER_PLASTICITY_TOKEN );
   fprintf(file,"              ");
   _DK_OUT_FVALUE( pmat->plasticity );

   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN( DK_A_DYNMATTER_NCDLEVEL_TOKEN );
   fprintf(file,"  ");
   shapeLevelAsciiWrite( file, pmat->node_cdlevel );

   _DK_OUT_NEWLINE();
}

/*******************************************************************************

   $$L DK_dynamatterReadAscii

   - Read dynamic matter data from file.

   Returned Value: pointer to dynamic matter structure

*******************************************************************************/

DK_DynaMatterInfo *DK_dynamatterReadAscii
   (
       FILE            *file,
      char	      *lastToken
   )
{
   DK_DynaMatterInfo *p;

   p = DK_dynaMatterInfoAllocate();
   if ( p == NULL &&  DK_option.verbose == TRUE)
   {
      fprintf( DK_option.msgFd,"DK_dynamatterReadAscii : Allocate failed ! \n");
      return( NULL );
   }

   DK_in_animToken(file);
   while( TRUE )
   {
      if ( strcmp( DK_A_DYNMATTER_DENSITY_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->density) );
      }
      else if ( strcmp( DK_A_DYNMATTER_ELASTICITY_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->elasticity) );
      }
      else if ( strcmp( DK_A_DYNMATTER_SROUGHNESS_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->mu_s) );
      }
      else if ( strcmp( DK_A_DYNMATTER_KROUGHNESS_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->mu_k) );
      }
      else if ( strcmp( DK_A_DYNMATTER_RIGIDITY_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->rigidity) );
      }
      else if ( strcmp( DK_A_DYNMATTER_PLASTICITY_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->plasticity) );
      }
      else if ( strcmp( DK_A_DYNMATTER_NCDLEVEL_TOKEN, DK_animToken ) == 0 )
      {
	 p->node_cdlevel = shapeLevelAsciiRead( file );
      }
      else
      {
	 strcpy( lastToken, DK_animToken );
	 return( p );
      }

      DK_in_animToken( file );
   }
}

/*******************************************************************************

   $$L DK_dynajntpropWriteBinaryChunk

   - Write dynamic joint properties to file.

   Returned Value: None.

*******************************************************************************/

void DK_dynajntpropWriteBinaryChunk
   (
       FILE 	*file,
      DK_DynaJntInfo *pdjp
   )
{
   /* 1 : friction */
   DK_Vector_fwrite( &(pdjp->friction), 	sizeof(DK_Vector), 1, file );
}


/*******************************************************************************

   $$L DK_dynajntpropReadBinaryChunk

   - Read dynamic joint properties from file.

*******************************************************************************/

void DK_dynajntpropReadBinaryChunk
   (
       FILE		*file,
      struct DK_model	*mdl,
      DK_FileProt          *prot
   )
{
   short        tok, ind = 0;
   DK_DynaJntInfo	*pdjp;

   mdl->djntinfo = DK_dynaJntInfoAllocate();
   pdjp = mdl->djntinfo;
   if ( pdjp == NULL &&  DK_option.verbose == TRUE)
   {
      fprintf( DK_option.msgFd,"DK_dynajntpropReadBinaryChunk : Allocate failed ! \n");
      return;
   }

   if( prot->joint != NULL )
      tok = prot->joint[ ind ++ ];
   else
      DK_byteswap_fread( &tok, sizeof(short), 1, file );

   while( tok != 0 ) /* End of joint properties definition */
   {
      switch( tok )
      {
         case 1: /* friction */
	    DK_Vector_fread( &(pdjp->friction), 	sizeof(DK_Vector), 1, file );
	    break;

         default:
            DK_fileSkipField( file, tok );

      }
      if( prot->joint != NULL )
         tok = prot->joint[ ind ++ ];
      else
         DK_byteswap_fread( &tok, sizeof(short), 1, file );
   }
}

/*******************************************************************************

   $$L DK_dynajntpropWriteAscii

   - Write joint properties data to file.

   Returned Value: None.

*******************************************************************************/

void DK_dynajntpropWriteAscii
   (
       FILE 		*file,
      DK_DynaJntInfo	*pjp,
      char		*space
   )
{
   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN( DK_A_DYNJNTPROP_FRICTION_TOKEN );
   fprintf(file,"\t\t");
   _DK_OUT_FVALUE( pjp->friction.x );
   _DK_OUT_FVALUE( pjp->friction.y );
   _DK_OUT_FVALUE( pjp->friction.z );

   _DK_OUT_NEWLINE();
}

/*******************************************************************************

   $$L DK_dynajntpropReadAscii

   - Read joint properties data from file.

   Returned Value: pointer to joint properties structure

*******************************************************************************/

DK_DynaJntInfo *DK_dynajntpropReadAscii
   (
       FILE	*file,
      char	*lastToken
   )
{
   DK_DynaJntInfo *p;

   p = DK_dynaJntInfoAllocate();
   if ( p == NULL &&  DK_option.verbose == TRUE)
   {
      fprintf( DK_option.msgFd,"DK_dynajntpropReadAscii : Allocate failed ! \n");
      return( NULL );
   }

   DK_in_animToken(file);
   while( TRUE )
   {
      if ( strcmp( DK_A_DYNJNTPROP_FRICTION_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->friction.x) );
	 DK_in_animFValue( file, &(p->friction.y) );
	 DK_in_animFValue( file, &(p->friction.z) );
      }
      else
      {
	 strcpy( lastToken, DK_animToken );
	 return( p );
      }

      DK_in_animToken( file );
   }
}

/*******************************************************************************

   $$L dynaConstrWriteAsciiChunk

   - Write a dynamic constraint to file.

   Returned Value: None.

*******************************************************************************/

int DK_dynaConstraintWriteAsciiChunk
   (
       FILE              *file,
      DK_DynaConstrInfo    *di
   )
{
   DK_incIndent(1);
   _DK_OUT_INDENT();
   /* 1 : constraint type */
   _DK_OUT_TOKEN( DK_A_DYNCONS_TYPE_TOKEN );
   switch ( di->type )
   {
      case DK_DYN_GRAVITY :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_GRAVITY_TOKEN );
	 break;
      case DK_DYN_ROOT_POS_ATTACH :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_POS_ATTACH_TOKEN );
	 break;
      case DK_DYN_ROOT_ORI_ATTACH :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_ORI_ATTACH_TOKEN );
	 break;
      case DK_DYN_IMPULSE :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_IMPULSE_TOKEN );
	 break;
      case DK_DYN_EXT_FORCE :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_EXT_FORCE_TOKEN );
	 break;
      case DK_DYN_EXT_TORQUE :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_EXT_TORQUE_TOKEN );
	 break;
      case DK_DYN_WIND :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_WIND_TOKEN );
	 break;
      case DK_DYN_FAN  :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_FAN_TOKEN );
	 break;
      case DK_DYN_NAIL :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_NAIL_TOKEN );
	 break;
      case DK_DYN_ORIENTATION :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_ORIENTATION_TOKEN );
	 break;
      case DK_DYN_COLLISION :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_COLLISION_TOKEN );
	 break;
      case DK_DYN_DEF_GRAVITY :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_DEF_GRAVITY_TOKEN );
	 break;
      case DK_DYN_DEF_WIND :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_DEF_WIND_TOKEN );
	 break;
      case DK_DYN_DEF_FLOOR :
	 _DK_OUT_TOKEN( DK_A_DYNCONS_DEF_FLOOR_TOKEN );
	 break;
   }


   /* 3 : constraint model name */
   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNCONS_NAME_TOKEN );
   _DK_OUT_SVALUE( di->n );
   if ( di->n != 0 )
   {
      _DK_OUT_4SPACES();
      _DK_OUT_STRING( di->model_name );
   }

   /* 4 : obstacle type */
   if (di->type == DK_DYN_COLLISION)
   {
       _DK_OUT_INDENT();
       _DK_OUT_TOKEN( DK_A_DYNCONS_OBSTACLE_TOKEN );
       shapeLevelAsciiWrite( file, di->obstype );
   }
   DK_incIndent(-1);

   return(TRUE);
}


/*******************************************************************************

   $$P dynaConstrReadAsciiChunk

   - Read a dynamic constraint from file.

   Returned Value: None

*******************************************************************************/

int DK_dynaConstraintReadAsciiChunk
   (
       FILE      *file,
      struct DK_model    *mdl
   )
{
   short           n, obsolete, type, read = FALSE, obstype;
   DK_DynaConstrInfo *di, *di2;
   char		  *name = NULL;

   obstype = DK_BPLANE;  /* default obstacle type for collision constraints */

   DK_in_animToken(file);
   while( TRUE )
   {
      if ( strcmp( DK_A_DYNCONS_TYPE_TOKEN, DK_animToken ) == 0 )
      {
	 read = TRUE;
	 DK_in_animToken( file );
	 if ( strcmp( DK_A_DYNCONS_GRAVITY_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_GRAVITY;
	 else if ( strcmp( DK_A_DYNCONS_POS_ATTACH_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_ROOT_POS_ATTACH;
	 else if ( strcmp( DK_A_DYNCONS_ORI_ATTACH_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_ROOT_ORI_ATTACH;
	 else if ( strcmp( DK_A_DYNCONS_IMPULSE_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_IMPULSE;
	 else if ( strcmp( DK_A_DYNCONS_EXT_FORCE_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_EXT_FORCE;
	 else if ( strcmp( DK_A_DYNCONS_EXT_TORQUE_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_EXT_TORQUE;
	 else if ( strcmp( DK_A_DYNCONS_WIND_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_WIND;
	 else if ( strcmp( DK_A_DYNCONS_FAN_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_FAN;
	 else if ( strcmp( DK_A_DYNCONS_NAIL_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_NAIL;
	 else if ( strcmp( DK_A_DYNCONS_ORIENTATION_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_ORIENTATION;
	 else if ( strcmp( DK_A_DYNCONS_COLLISION_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_COLLISION;
	 else if ( strcmp( DK_A_DYNCONS_DEF_GRAVITY_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_DEF_GRAVITY;
	 else if ( strcmp( DK_A_DYNCONS_DEF_WIND_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_DEF_WIND;
	 else if ( strcmp( DK_A_DYNCONS_DEF_FLOOR_TOKEN, DK_animToken ) == 0 )
	    type = DK_DYN_DEF_FLOOR;
	 else
	 {
	    if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd,
			"Error, unknown constraint type DK_animToken '%s'.\n", 
		    DK_animToken);
	    read = FALSE;
	 }
      }
      else if ( strcmp( DK_A_DYNCONS_PRIO_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &obsolete );
      }
      else if ( strcmp( DK_A_DYNCONS_NAME_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &n );
	 if ( n != 0 )
	    name = DK_stringReadAsciiAnimation(file);
	 else
	    name = NULL;
      }
      else if (strcmp (DK_A_DYNCONS_OBSTACLE_TOKEN, DK_animToken) == 0)
      {
	  obstype = shapeLevelAsciiRead(file);
      }
      else
      {
	 if (read == TRUE)
	 {
	     /* allocate dyna info */
	     if ((di = DK_dynaConstrInfoAllocate()) != NULL)
	     {
		 di2 = mdl->dcinfo;
		 while (di2 != NULL && di2->next != NULL)
		     di2 = di2->next;
		 if (di2 != NULL)
		     di2->next = di;
		 else
		     mdl->dcinfo = di;
		 di->type = type;
		 di->n = n;
		 di->model_name = name;
		 di->obstype = obstype;
	     }
	     else if (name != NULL)
		 _DK_FREE(name, -1);
	 }
	 return(TRUE);
      }

      DK_in_animToken( file );
   }
}

/*******************************************************************************

   $$P DK_dynachnvarWriteAsciiChunk

   - Write dynamic chain (root) data to file.

   Returned Value: None.

*******************************************************************************/

int DK_dynachnvarWriteAsciiChunk
   (
       FILE 		*file,
      DK_DynaChnVars	*dv
   )
{
   DK_incIndent(1);

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_RCPOS_TOKEN );
   _DK_OUT_BOOLEAN( dv->dorcpos );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_RCPOSAXE_TOKEN );
   _DK_OUT_BOOLEAN( dv->rcposaxe[0] );
   _DK_OUT_BOOLEAN( dv->rcposaxe[1] );
   _DK_OUT_BOOLEAN( dv->rcposaxe[2] );

   _DK_OUT_TOKEN( DK_A_DYNSETUP_RCPOSACC_TOKEN );
   _DK_OUT_BOOLEAN( dv->rcposacc );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_RCORI_TOKEN );
   _DK_OUT_BOOLEAN( dv->dorcori );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_RCORIAXE_TOKEN );
   _DK_OUT_BOOLEAN( dv->rcoriaxe[0] );
   _DK_OUT_BOOLEAN( dv->rcoriaxe[1] );
   _DK_OUT_BOOLEAN( dv->rcoriaxe[2] );

   _DK_OUT_TOKEN( DK_A_DYNSETUP_RCORIACC_TOKEN );
   _DK_OUT_BOOLEAN( dv->rcoriacc );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_DOWIND_TOKEN );
   _DK_OUT_BOOLEAN( dv->dowind );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_DOGRAV_TOKEN );
   _DK_OUT_BOOLEAN( dv->dograv );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_CHKCOLL_TOKEN );
   _DK_OUT_BOOLEAN( dv->dofloor );

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_DETECTL_TOKEN );
   shapeLevelAsciiWrite(file, (short) dv->cdetect_level);

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_WINDL_TOKEN );
   shapeLevelAsciiWrite(file, (short) dv->wind_level);

   _DK_OUT_INDENT();
   _DK_OUT_TOKEN( DK_A_DYNSETUP_GEOMUPD_TOKEN );
   _DK_OUT_BOOLEAN( dv->geomupdate );

   DK_incIndent(-1);

   return(TRUE);
}

/*******************************************************************************

   $$P DK_dynachnvarReadAsciiChunk

   - Read dynamic chain (root) data to file.

   Returned Value: None.

*******************************************************************************/

int DK_dynachnvarReadAsciiChunk
   (
       FILE 	*file,
      struct DK_model	*mdl
   )
{
   DK_DynaChnVars   *dv;

   dv = DK_dynaChnVarAllocate();
   if ( dv == NULL )
   {
      if ( DK_option.verbose == TRUE)
          fprintf( DK_option.msgFd, 
		"DK_dynachnvarReadAsciiChunk: Cannot allocate dyna vars\n");
      return(FALSE);
   }
   if (mdl->dchnvar != NULL)
       DK_dynaChnVarDispose(&mdl->dchnvar);
   mdl->dchnvar = dv;

   DK_in_animToken( file );
   while( TRUE )
   {
      if ( strcmp( DK_A_DYNSETUP_RCPOS_TOKEN,DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->dorcpos );
      }
      else if ( strcmp( DK_A_DYNSETUP_RCPOSAXE_TOKEN,DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->rcposaxe[0] );
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->rcposaxe[1] );
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->rcposaxe[2] );
      }
      else if ( strcmp( DK_A_DYNSETUP_RCPOSACC_TOKEN,DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->rcposacc );
      }
      else if ( strcmp( DK_A_DYNSETUP_RCORI_TOKEN,DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->dorcori );
      }
      else if ( strcmp( DK_A_DYNSETUP_RCORIAXE_TOKEN,DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->rcoriaxe[0] );
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->rcoriaxe[1] );
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->rcoriaxe[2] );
      }
      else if ( strcmp( DK_A_DYNSETUP_RCORIACC_TOKEN,DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->rcoriacc );
      }
      else if ( strcmp( DK_A_DYNSETUP_DOWIND_TOKEN,DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->dowind );
      }
      else if ( strcmp( DK_A_DYNSETUP_DOGRAV_TOKEN,DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->dograv );
      }
      else if ( strcmp( DK_A_DYNSETUP_CHKCOLL_TOKEN,DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->dofloor );
      }
      else if ( strcmp( DK_A_DYNSETUP_DETECTL_TOKEN,DK_animToken ) == 0 )
      {
	 dv->cdetect_level = shapeLevelAsciiRead(file);
      }
      else if ( strcmp( DK_A_DYNSETUP_WINDL_TOKEN,DK_animToken ) == 0 )
      {
	 dv->wind_level = shapeLevelAsciiRead(file);
      }
      else if (strcmp ( DK_A_DYNSETUP_GEOMUPD_TOKEN, DK_animToken) == 0)
      {
	 DK_in_animToken( file );
	 _DK_SET_BOOLEAN( DK_animToken, dv->geomupdate );
      }
      else
      {
	 return(TRUE);
      }

      DK_in_animToken( file );
   }
}

/*******************************************************************************

   $$L DK_odynamicWriteBinaryChunk

   - Write dynamic data to file.

   Returned Value: None.

*******************************************************************************/

void DK_odynamicWriteBinaryChunk
   (
       FILE 	   *file,
      struct DK_model *mdl
   )
{
   short	n;
   DK_DynaJntInfo	*pdjp;

   n = (short) ( mdl->djntinfo != NULL );
   DK_byteswap_fwrite( &n, sizeof(short), 1, file );
   if ( mdl->djntinfo != NULL )
   {
      pdjp = mdl->djntinfo;
      DK_Vector_fwrite( &(pdjp->friction), 	sizeof(DK_Vector), 1, file );
   }
}


/*******************************************************************************

   $$L DK_odynamicReadBinaryChunk

   - Read dynamic data from file.

   Returned Value: pointer to...

*******************************************************************************/

void DK_odynamicReadBinaryChunk
   (
       FILE	   *file,
      struct DK_model *mdl
   )
{
   short	jp;
   DK_DynaJntInfo	*pdjp;

   DK_byteswap_fread( &jp, sizeof(short), 1, file );
   if ( jp != 0 )
   {
      mdl->djntinfo = DK_dynaJntInfoAllocate();
      pdjp = mdl->djntinfo;
      DK_Vector_fread( &(pdjp->friction), 	sizeof(DK_Vector), 1, file );
   }
}

