/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyCluster.c,v $
   $Revision: 1.15.14.1 $ $Date: 1995/06/09 21:59:51 $
   Checkin by: $Author: poirierm $

   <DESCRIPTION>

   File supervisor: Super-User

   (c) Copyright 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinySceneToken.h"
#include "tinyModel.h"
#include "tinyUtils.h"

/* static routines */
static int DK_clusterNodeWriteAsciiAnimation( FILE *, DK_ClusterData *, char *);
static int DK_clusterNodeReadAsciiAnimation( FILE *, struct DK_model *);

static int DK_clusterKeyReadAsciiFile( FILE *, DK_ClusterData *, int , int);
static int DK_clusterKeyWriteAsciiFile( FILE *, DK_ClusterKey *, int , int );
static int DK_clusterNodeReadAsciiFile( FILE *, struct DK_model *);
static int DK_clusterNodeWriteAsciiFile( 
   FILE *, DK_ClusterData *, struct DK_model *, char *, int
);

static DK_Kluster *DK_klusterAddToList ( DK_Model *);



/*******************************************************************************

   $$L DK_clusterAllocate

   This function allocates a DK_cluster.

   Returned Value: NULL or DK_cluster pointer

*******************************************************************************/

DK_Cluster *DK_clusterAllocate
    (
    )
{
    DK_Cluster *ptr;

    if ((ptr = _DK_MALLOC(DK_Cluster, sizeof(DK_Cluster), -1)) == NULL)
	return(NULL);

    ptr->filename    = NULL;
    ptr->active = FALSE;

    return(ptr);
}

/*******************************************************************************

   $$L DK_clusterDispose

   This function disposes a cluster.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_clusterDispose
    (
	DK_Cluster **ptr
    )
{
    DK_ClusterData *cptr, *cptr2;

    if (ptr == NULL || *ptr == NULL)
	return(FALSE);

    if ((*ptr)->filename != NULL)
	_DK_FREE((*ptr)->filename, -1);

    _DK_FREE((*ptr), -1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_clusterDataAllocate

   This function allocates a cluster data element.

   Returned Value: NULL or cluster data pointer

*******************************************************************************/

DK_ClusterData *DK_clusterDataAllocate
    (
    )
{
    DK_ClusterData *ptr;

    if ((ptr = _DK_MALLOC(DK_ClusterData, sizeof(DK_ClusterData), -1)) == NULL)
	return(NULL);

    ptr->keys = NULL;
    ptr->fclusterWeights = NULL;
    ptr->active_wfcv = NULL;
    ptr->fcluster = NULL;
    ptr->active_fcv.active = FALSE;
    ptr->active_fcv.fcv = NULL;
    ptr->nbClusterKey = 0;
    ptr->clusterInterp = DK_LINEAR_SHAPE_INTERP;
    ptr->active = FALSE;

    return(ptr);
}

/*******************************************************************************

   $$L DK_clusterDataDispose

   This function disposes a cluster data element.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_clusterDataDispose
    (
	DK_ClusterData **ptr
    )
{
    int i;
    DK_ClusterKey *kptr;

    if (ptr == NULL || *ptr == NULL)
	return(FALSE);

    /* free cluster weights */
    for (i = 0; i < (*ptr)->nbClusterKey; i++)
    {
	DK_fcurveDispose(&(*ptr)->fclusterWeights[i]);
	if ( (*ptr)->active_wfcv && (*ptr)->active_wfcv[i].fcv )
	   DK_fcurveDispose( &(*ptr)->active_wfcv[i].fcv );
    }
    _DK_FREE((*ptr)->fclusterWeights, -1);
    _DK_FREE((*ptr)->active_wfcv, -1);

    /* free interpolation fcurve */
    DK_fcurveDispose(&((*ptr)->fcluster));
    DK_fcurveDispose(&((*ptr)->active_fcv.fcv));

    /* free cluster keys */
    while ((*ptr)->keys != NULL)
    {
        kptr = (*ptr)->keys;
	(*ptr)->keys = (*ptr)->keys->next;
        DK_clusterKeyDispose(&kptr);
    }

    _DK_FREE((*ptr), -1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_clusterKeyAllocate

   This function allocates "nbkey" cluster keys of type "type".

   Returned Value: NULL or cluster key pointer

*******************************************************************************/

DK_ClusterKey *DK_clusterKeyAllocate
    (
	int nbkey,
	int type
    )
{
    DK_ClusterKey *ptr;

    if (nbkey <= 0)
	return(NULL);

    if ((ptr = _DK_MALLOC(DK_ClusterKey, sizeof(DK_ClusterKey), -1)) == NULL)
	return(NULL);

    switch (type)
    {
        case DK_MDL_MESH:
	    ptr->vectors = _DK_MALLOC(DK_Pointer, sizeof(DK_Point) * nbkey, -1);
            break;
        case DK_MDL_PTCH:
        case DK_MDL_CRV:
        case DK_MDL_SRF:
	    ptr->vectors = _DK_MALLOC(DK_Pointer, sizeof(DK_HVector) * nbkey, -1);
            break;
        case DK_MDL_FACE:
	    ptr->vectors = _DK_MALLOC(DK_Pointer, sizeof(DK_Point) * 3 * nbkey, -1);
            break;
        case DK_MDL_SPLN:
	    ptr->vectors = _DK_MALLOC(DK_Pointer, sizeof(DK_Point) * 3 * nbkey, -1);
            break;
    }
    if (ptr->vectors == NULL)
    {
	_DK_FREE(ptr, -1);
	return(NULL);
    }
    ptr->next = NULL;

    return(ptr);
}

/*******************************************************************************

   $$L DK_clusterKeyDispose

   This function disposes a cluster key.

   Returned Value: TRUE or FALSE

*******************************************************************************/


int DK_clusterKeyDispose
    (
	DK_ClusterKey **ptr
    )
{
    if (ptr == NULL || *ptr == NULL)
	return(FALSE);

    if ((*ptr)->vectors != NULL)
	_DK_FREE((*ptr)->vectors, -1);
    _DK_FREE(*ptr, -1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_clusterWeightFcurveAllocate

   This function allocates "nb" cluster weight fcurve pointers.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_clusterWeightFcurveAllocate
    (
	DK_ClusterData *cptr,	/* cluster pointer */
	int nb			/* number of cluster fcurves */
    )
{
    int i;

    if (cptr == NULL)
	return(FALSE);

    if (cptr->fclusterWeights != NULL)
	DK_clusterWeightFcurveDispose(cptr);

    if (nb > 0) 
    {
	if ((cptr->fclusterWeights = 
		_DK_MALLOC(DK_Fcurve *, nb * sizeof(DK_Fcurve *), -1)) == NULL)
	    return(FALSE);
	for (i = 0; i < nb; i++)
	    cptr->fclusterWeights[i] = NULL;
    }
    return(TRUE);
}

/*******************************************************************************

   $$L DK_clusterWeightFcurveDispose

   This function disposes the cluster weight fcurves.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_clusterWeightFcurveDispose
    (
	DK_ClusterData *cptr
    )
{
    int i;

    if (cptr == NULL || cptr->fclusterWeights == NULL)
	return(FALSE);
    
    for (i = 0; i < cptr->nbClusterKey; i++)
	DK_fcurveDispose(&(cptr->fclusterWeights[i]));
    _DK_FREE(cptr->fclusterWeights, -1);
    cptr->fclusterWeights = NULL;
    return(TRUE);
}

/*******************************************************************************

   $$L DK_klusterFindByName

   This function finds the kluster given its name.

   Returned Value: NULL or kluster data pointer

*******************************************************************************/

struct  DK_kluster *DK_klusterFindByName
    (
	struct DK_model *mdl,
	char *name
    )
{
    struct DK_kluster *kptr;

    if (mdl == NULL || name == NULL)
	return(NULL);

    kptr = mdl->klusters;
    while (kptr != NULL)
    {
	if (strcmp(name, kptr->name) == 0)
	    return(kptr);
	kptr = kptr->next;
    }
    return(NULL);
}

/*******************************************************************************

   $$L DK_clusterNodeWriteAsciiAnimation

   This function writes an animation cluster node.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_clusterNodeWriteAsciiAnimation
    (
	FILE *file,		/* file io pointer */
	DK_ClusterData *cptr,	/* cluster node pointer */
	char *name		/* cluster name */
    )
{
    int i;

    if (cptr == NULL)
	return(FALSE);

    DK_incIndent(1);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_CLUSTER_NODE_NAME_TOKEN);
    _DK_OUT_STRING(name);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_CLUSTER_NODE_ACTIVE_TOKEN);
    if (cptr->active == TRUE)
	_DK_OUT_TOKEN(DK_A_YES_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_NO_TOKEN);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_CLUSTER_NODE_INTERP_TOKEN);
    switch (cptr->clusterInterp)
    {
        case DK_LINEAR_SHAPE_INTERP:
	    _DK_OUT_TOKEN(DK_A_LINEAR_SHAPE_INTERP_TOKEN);
	    break;
        case DK_CARDINAL_SHAPE_INTERP:
	    _DK_OUT_TOKEN(DK_A_CARDINAL_SHAPE_INTERP_TOKEN);
	    break;
        case DK_WEIGHT_SHAPE_INTERP:
	    _DK_OUT_TOKEN(DK_A_WEIGHT_SHAPE_INTERP_TOKEN);
	    break;
    }

    if (cptr->fcluster != NULL) 
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN(DK_A_CLUSTER_NODE_FCV_TOKEN);
        DK_fcurveWriteAsciiAnimation(file, cptr->fcluster);
    }

    if (cptr->nbClusterKey > 0 && cptr->fclusterWeights != NULL)
    {
	_DK_OUT_INDENT();
	_DK_OUT_TOKEN(DK_A_CLUSTER_NODE_WEIGHT_TOKEN);
	_DK_OUT_SVALUE(cptr->nbClusterKey);
	_DK_OUT_INDENT();
	for ( i = 0; i < cptr->nbClusterKey; i++)
            DK_fcurveWriteAsciiAnimation(file, cptr->fclusterWeights[i]);
    }
    DK_incIndent(-1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_clusterNodeReadAsciiAnimation

   This function reads an animation cluster node.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_clusterNodeReadAsciiAnimation
    (
	FILE *file,		/* file io pointer */
	struct DK_model *mdl	/* cluster node pointer */
    )
{
    int i, nb = 0;
    DK_Fcurve *tfcv;
    char *name;
    DK_ClusterData *cptr = NULL;
    struct DK_kluster *kptr;

    DK_in_animToken(file);
    while (TRUE)
    {
        if (strcmp( DK_A_CLUSTER_NODE_NAME_TOKEN, DK_animToken) == 0)
	{
	    name = DK_stringReadAsciiAnimation(file);
	    if (name != NULL)
	    {
		if ((kptr = DK_klusterFindByName(mdl, name)) == NULL)
	        {
		    if ( DK_option.verbose == TRUE)
                        fprintf( DK_option.msgFd, "DK_ClusterNodeAsciiReadAnimation: \
cannot find cluster %s.", name);
	        }
		else
		    cptr = kptr->anim;
		_DK_FREE(name, -1);
	    }
            DK_in_animToken(file);
    
	}
        else if (strcmp( DK_A_CLUSTER_NODE_ACTIVE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
	    {
		if (cptr != NULL)
		    cptr->active = TRUE;
	    }
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
	    {
		if (cptr != NULL)
		    cptr->active = FALSE;
	    }
	    else
	    {
		if ( DK_option.verbose == TRUE)
	            fprintf( DK_option.msgFd, "DK_ClusterNodeAsciiReadAnimation: \
error, unknown active token '%s'.", DK_animToken);
	    }
	    DK_in_animToken(file);
	}
        else if (strcmp( DK_A_CLUSTER_NODE_INTERP_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (cptr != NULL)
	    {
		if (strcmp( DK_A_LINEAR_CLUSTER_INTERP_TOKEN, DK_animToken) == 0)
	            cptr->clusterInterp = DK_LINEAR_SHAPE_INTERP;
		else if (strcmp(DK_A_CARDINAL_CLUSTER_INTERP_TOKEN,DK_animToken) == 0)
	            cptr->clusterInterp = DK_CARDINAL_SHAPE_INTERP;
		else if (strcmp(DK_A_WEIGHT_CLUSTER_INTERP_TOKEN, DK_animToken) == 0)
	            cptr->clusterInterp = DK_WEIGHT_SHAPE_INTERP;
		else
		{
		    if ( DK_option.verbose == TRUE)
		        fprintf( DK_option.msgFd, "DK_clusterNodeAsciiReadAnimation: \
error, unknown interp token '%s'.", DK_animToken);
		}
	    }
            DK_in_animToken(file);
    
	}
        else if (strcmp( DK_A_CLUSTER_NODE_FCV_TOKEN, DK_animToken) == 0)
	{
            DK_in_animToken(file);
	    if (cptr != NULL)
	        DK_fcurveReadAsciiAnimation(file, &cptr->fcluster);
	    else
	        DK_fcurveReadAsciiAnimation(file, NULL);
        }

        else if (strcmp(DK_A_CLUSTER_NODE_WEIGHT_TOKEN, DK_animToken) == 0)
        {
	    DK_in_animSValue_na(file);
	    nb = (int) DK_animSValue;
	    DK_in_animToken(file);
	    for (i = 0; i < nb; i++)
	    {
	        tfcv = NULL;
	        DK_fcurveReadAsciiAnimation(file, &tfcv);
		/* test to see if nb > nbClusterKey */
	        if (cptr != NULL && i < cptr->nbClusterKey)
	        {
		    DK_fcurveDispose(&cptr->fclusterWeights[i]);
	            cptr->fclusterWeights[i] = tfcv;
	        }
	    }
        }
	else 
	    return(TRUE);
    }
}

/*******************************************************************************

   $$L DK_clusterWriteAsciiAnimation

   This function writes an animation cluster.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_clusterWriteAsciiAnimation
    (
	FILE *file,		/* file io pointer */
	DK_Cluster *cptr,		/* cluster pointer */
	struct  DK_kluster *kptr	/* kluster pointer */
    )
{
    DK_ClusterData *ptr;

    if (kptr == NULL || cptr == NULL)
	return(FALSE);

    DK_incIndent(1);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_CLUSTER_FILENAME_TOKEN);
    _DK_OUT_STRING(cptr->filename);

    _DK_OUT_INDENT();
    _DK_OUT_TOKEN(DK_A_CLUSTER_ACTIVE_TOKEN);
    if (cptr->active == TRUE)
	_DK_OUT_TOKEN(DK_A_YES_TOKEN);
    else
	_DK_OUT_TOKEN(DK_A_NO_TOKEN);

    while (kptr != NULL)
    {
	ptr = kptr->anim;
	if (ptr != NULL)
	{
	    _DK_OUT_INDENT();
	    _DK_OUT_TOKEN(DK_A_CLUSTER_NODE_TOKEN);
	    DK_clusterNodeWriteAsciiAnimation(file, ptr, kptr->name);
	}
	kptr = kptr->next;
    }
    DK_incIndent(-1);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_clusterReadAsciiAnimation

   This function reads an animation cluster.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_clusterReadAsciiAnimation
    (
	FILE *file,		/* file io pointer */
	struct DK_model *mdl
    )
{
    char *name;
    DK_Cluster *cptr = NULL;

    DK_in_animToken(file);
    while (TRUE)
    {
        if (strcmp( DK_A_CLUSTER_FILENAME_TOKEN, DK_animToken) == 0)
	{
	    name = DK_stringReadAsciiAnimation(file);
	    if (name != NULL)
	    {
		if (mdl != NULL && DK_clusterReadAsciiFile(name, mdl) == TRUE)
		{
		    cptr = mdl->cluster;
		    if (cptr != NULL)
		        cptr->filename = name;
		    else
			_DK_FREE(name, -1);
		}
		else
		    _DK_FREE(name, -1);
	    }
            DK_in_animToken(file);
    
	}
        else if (strcmp( DK_A_CLUSTER_ACTIVE_TOKEN, DK_animToken) == 0)
	{
	    DK_in_animToken(file);
	    if (strcmp(DK_A_YES_TOKEN, DK_animToken) == 0)
	    {
		if (cptr != NULL)
		    cptr->active = TRUE;
	    }
	    else if (strcmp(DK_A_NO_TOKEN, DK_animToken) == 0)
	    {
		if (cptr != NULL)
		    cptr->active = FALSE;
	    }
	    else
	    {
		if ( DK_option.verbose == TRUE)
	            fprintf( DK_option.msgFd, "DK_ClusterAsciiRead: error, \
unknown active token '%s'.", DK_animToken);
	    }
	    DK_in_animToken(file);
    
	}
        else if (strcmp( DK_A_CLUSTER_NODE_TOKEN, DK_animToken) == 0)
	{
	    if (mdl != NULL && cptr != NULL)
	        DK_clusterNodeReadAsciiAnimation(file, mdl);
	    else
	        DK_clusterNodeReadAsciiAnimation(file, NULL);
        }
	else 
	{
	    return(TRUE);
	}
    }
}

/*******************************************************************************

   $$L DK_clusterKeyReadAsciiFile

   This function read a cluster key.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_clusterKeyReadAsciiFile
    (
	FILE *fp,
	DK_ClusterData *cptr,
	int ncv,
	int type
    )
{
    DK_Point *p = NULL;
    DK_HVector *hv = NULL;
    int i;
    DK_ClusterKey *kptr = NULL, *kptr2;
    float x, y, z, w;

    if (fp == NULL)
	return(FALSE);
    
    if (cptr != NULL && (kptr = DK_clusterKeyAllocate(ncv, type)) != NULL)
    {
	/* add to the end of the list */
	if (cptr->keys == NULL)
	    cptr->keys = kptr;
	else
	{
	    kptr2 = cptr->keys;
	    while (kptr2 != NULL && kptr2->next != NULL)
		kptr2 = kptr2->next;
	    kptr2->next = kptr;
	}
        switch (type)
        {
            case DK_MDL_MESH:
	        p = (DK_Point *) kptr->vectors;
                break;
            case DK_MDL_PTCH:
            case DK_MDL_CRV:
            case DK_MDL_SRF:
	        hv = (DK_HVector *) kptr->vectors;
                break;
            case DK_MDL_FACE:
	        p = (DK_Point *) kptr->vectors;
                break;
            case DK_MDL_SPLN:
	        p = (DK_Point *) kptr->vectors;
                break;
        }
    }

    for (i = 0; i < ncv; i++)
    {
	switch (type)
	{
            case DK_MDL_MESH:
		fscanf(fp,"%g %g %g",&x, &y, &z);
		if (p != NULL)
		{
		    p->x = x; p->y = y; p->z = z;
		    p++;
		}
                break;
            case DK_MDL_PTCH:
            case DK_MDL_CRV:
            case DK_MDL_SRF:
		fscanf(fp,"%g %g %g %g",&x, &y, &z, &w);
		if (hv != NULL)
		{
		    hv->x = x; hv->y = y; hv->z = z; hv->w = w;
		    hv++;
		}
                break;
            case DK_MDL_FACE:
            case DK_MDL_SPLN:
		fscanf(fp,"%g %g %g",&x, &y, &z);
		if (p != NULL)
		{
		    p->x = x; p->y = y; p->z = z;
		    p++;
		}
		fscanf(fp,"%g %g %g",&x, &y, &z);
		if (p != NULL)
		{
		    p->x = x; p->y = y; p->z = z;
		    p++;
		}
		fscanf(fp,"%g %g %g",&x, &y, &z);
		if (p != NULL)
		{
		    p->x = x; p->y = y; p->z = z;
		    p++;
		}
                break;
	}
    }
    return(TRUE);
}

/*******************************************************************************

   $$L DK_clusterNodeReadAsciiFile

   This function read a cluster node.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int DK_clusterNodeReadAsciiFile
    (
	FILE *fp,
	struct DK_model *mdl
    )
{
    DK_ClusterData *cptr, *cptr2;
    char str[256];
    int done, ncv, nkeys, i, j, k;
    struct  DK_kluster *kptr;

    if (fp == NULL)
	return(FALSE);
    
    if (mdl != NULL && mdl->cluster == NULL)
    {
	mdl->cluster = DK_clusterAllocate();
	if (mdl->cluster != NULL)
	    mdl->cluster->active = TRUE;
    }

    cptr = DK_clusterDataAllocate();

    done = FALSE;
    while (done == FALSE)
    {
        fscanf(fp,"%s",str);
	if (strcmp(str,DK_A_CLUSTER_NODE_NAME_TOKEN) == 0)
	{
            fscanf(fp,"%s",str);
	    /* find kluster to put anim data */
	    if (cptr != NULL && str[0] != '\0' &&
	       (kptr = DK_klusterFindByName(mdl, (char *) str)) != NULL)
	    {
		if (kptr->anim != NULL)
		    DK_clusterDataDispose(&kptr->anim);
		kptr->anim = cptr;
		ncv = kptr->ncv;
	    }
	}
	else if (strcmp(str,DK_CLUSTER_NODE_NCVS_TOKEN) == 0)
	{
	    /* OBSELETE */
            fscanf(fp,"%d",&ncv);
	}
	else if (strcmp(str,DK_CLUSTER_NODE_NKEYS_TOKEN) == 0)
	{
            fscanf(fp,"%d",&nkeys);
	    if (cptr != NULL && nkeys >= 0)
	    {
		cptr->nbClusterKey = nkeys;
		DK_clusterWeightFcurveAllocate ( cptr, nkeys);
	    }
	}
	else if (strcmp(str,DK_CLUSTER_NODE_INDEXES_TOKEN) == 0)
	{
	    /* OBSELETE */
	    switch (mdl->type)
	    {
		case DK_MDL_FACE:
		    k = ncv * 2;
		    break;
		case DK_MDL_MESH:
		case DK_MDL_PTCH:
		case DK_MDL_SPLN:
		    k = ncv;
		    break;
	    }
	    for (i = 0; i < k; i++)
                fscanf(fp,"%d",&j);
	}
	else if (strcmp(str,DK_CLUSTER_NODE_KEYS_TOKEN) == 0)
	{
	    DK_clusterKeyReadAsciiFile(fp, cptr, ncv, mdl->type);
	}
	else if (strcmp(str,DK_CLUSTER_END_TOKEN) == 0)
	    done = TRUE;
	else
	{
	    if ( DK_option.verbose == TRUE)
	        fprintf( DK_option.msgFd,"DK_ClusterNodeReadAsciiFile: Error, \
unknown token %s",str);
	    done = TRUE;
	}
    }
    return(TRUE);
}

/*******************************************************************************

   $$L DK_clusterReadAsciiFile

   This function read a cluster file.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_clusterReadAsciiFile
    (
	char *filename,
	struct DK_model *mdl
    )
{
    FILE *fp;
    char str[256];
    int i;
    float version;

    if (filename == NULL)
        return(FALSE);

    sprintf( str, "%s.cls", filename );
    if( ( fp = fopen( str, "rb" ) ) == NULL )
        return( FALSE );

    fscanf(fp,"%s",str);
    if (strcmp(str,DK_CLUSTER_ID) != 0)
    {
	if ( DK_option.verbose == TRUE)
	    fprintf( DK_option.msgFd,"DK_ClusterReadAsciiFile: Error file %s \
is not a cluster file", filename);
	fclose(fp);
	return(FALSE);
    }

    /* read header */
    fscanf(fp,"%f",&version);

    /* read cluster node */
    fscanf(fp,"%s",str);
    i= 0;
    while (strcmp(str,DK_CLUSTER_NODE_TOKEN) == 0)
    {
        DK_clusterNodeReadAsciiFile(fp, mdl);
        fscanf(fp,"%s",str);
	if (i++ > 10)
	    break;
    }
    fclose(fp);

    return(TRUE);
}

/*******************************************************************************

   $$L  DK_clusterKeyWriteAsciiFile

   This function writes a cluster key.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int  DK_clusterKeyWriteAsciiFile
    (
	FILE *fp,
	DK_ClusterKey *kptr,
	int ncv,
	int type
    )
{
    DK_Point *p;
    DK_HVector *hv;
    int i;

    if (kptr == NULL || fp == NULL)
	return(FALSE);
    
    fprintf(fp,"\n  %s\n",DK_CLUSTER_NODE_KEYS_TOKEN);
    switch (type)
    {
        case DK_MDL_MESH:
	    p = (DK_Point *) kptr->vectors;
            break;
        case DK_MDL_PTCH:
        case DK_MDL_CRV:
        case DK_MDL_SRF:
	    hv = (DK_HVector *) kptr->vectors;
            break;
        case DK_MDL_FACE:
	    p = (DK_Point *) kptr->vectors;
            break;
        case DK_MDL_SPLN:
	    p = (DK_Point *) kptr->vectors;
            break;
    }
    for (i = 0; i < ncv; i++)
    {
	switch (type)
	{
            case DK_MDL_MESH:
		fprintf(fp,"    %g %g %g\n",p->x, p->y, p->z);
		p++;
                break;
            case DK_MDL_PTCH:
            case DK_MDL_CRV:
            case DK_MDL_SRF:
		fprintf(fp,"    %g %g %g %g\n",hv->x, hv->y, hv->z, hv->w);
		hv++;
                break;
            case DK_MDL_FACE:
            case DK_MDL_SPLN:
		fprintf(fp,"    %g %g %g",p->x, p->y, p->z);
		p++;
		fprintf(fp,"    %g %g %g",p->x, p->y, p->z);
		p++;
		fprintf(fp,"    %g %g %g\n",p->x, p->y, p->z);
		p++;
                break;
	}
    }
    return(TRUE);
}

/*******************************************************************************

   $$L  DK_clusterNodeWriteAsciiFile

   This function writes a cluster node.

   Returned Value: TRUE or FALSE

*******************************************************************************/

static int  DK_clusterNodeWriteAsciiFile
    (
	FILE *fp,
	DK_ClusterData *cptr,
	struct DK_model *mdl,
	char *name,
	int ncv
    )
{
    int i, j;
    DK_ClusterKey *kptr;

    if (cptr == NULL || fp == NULL)
	return(FALSE);
    
    fprintf(fp,"\n%s\n", DK_CLUSTER_NODE_TOKEN);
    fprintf(fp,"  %s %s\n",DK_CLUSTER_NODE_NAME_TOKEN, name);
    fprintf(fp,"  %s %d\n",DK_CLUSTER_NODE_NKEYS_TOKEN, cptr->nbClusterKey);
    kptr = cptr->keys;
    while (kptr != NULL)
    {
	 DK_clusterKeyWriteAsciiFile(fp, kptr, ncv, mdl->type);
	kptr = kptr->next;
    }
    fprintf(fp,"%s\n\n", DK_CLUSTER_END_TOKEN);

    return(TRUE);
}

/*******************************************************************************

   $$L DK_clusterWriteAsciiFile

   This function writes a cluster file.

   Returned Value: TRUE or FALSE

*******************************************************************************/

int DK_clusterWriteAsciiFile
    (
	char *filename,
	struct DK_model *mdl
    )
{
    char buf[256];
    FILE *fp;
    DK_ClusterData *cptr;
    struct DK_kluster *kptr;

    if (mdl == NULL || filename == NULL)
        return(FALSE);

    sprintf( buf, "%s.cls", filename );
    if( ( fp = fopen( buf, "wb" ) ) == NULL )
        return( FALSE );

    /* write version number */
    fprintf(fp,"%s %4.2f\n", DK_CLUSTER_ID, DK_VERSION_NAME[DK_VERSION]);

    /* write cluster node */
    if (mdl->klusters != NULL)
    {
	kptr = mdl->klusters;
        while (kptr != NULL)
        {
	    if ((cptr = kptr->anim) != NULL)
                 DK_clusterNodeWriteAsciiFile(fp, cptr, mdl, kptr->name, kptr->ncv);
            kptr = kptr->next;
        }
    }
    fprintf(fp,"%s\n", DK_CLUSTER_END_TOKEN);
    fclose(fp);

    return(TRUE);
}

/*-------------------DK_KLUSTER ROUTINES---------------------------------------*/

/*******************************************************************************

    $$L DK_klusterAllocate

    Usage      : Allocate space for one kluster.

    [Returned Value : ptr to new space]

    Author     : Rejean Gagne, Feb 92
    Updated by : <Full name>, <Date>, <Update description>

    (c) Copyright 1992 SoftImage Inc.

*******************************************************************************/

DK_Kluster *DK_klusterAllocate
   (
   )
{
   DK_Kluster	*kls;

   if ( ( kls = _DK_MALLOC( DK_Kluster, sizeof( DK_Kluster ), -1 ) ) != NULL )
   {
      kls->name    = NULL;
      kls->ncv     = 0;
      kls->indexes = NULL;
      kls->state   = DK_K_unactive;
      kls->anim	   = NULL;
      kls->next	   = NULL;
   }
   return( kls );
}

/*******************************************************************************

    $$L DK_klusterDispose

    Usage      : Disposes everything associated with one cluster.

    [Returned Value : None]

    Author     : Rejean Gagne, Feb 92

    (c) Copyright 1992 SoftImage Inc.

*******************************************************************************/
void DK_klusterDispose
   (
      DK_Kluster	**kls
   )
{
   DK_Kluster	*prev;
   int	found = FALSE;

   if ( kls == NULL || *kls == NULL)
      return;

   while (*kls != NULL)
   {
      prev = *kls;
      *kls = (*kls)->next;
      DK_clusterDataDispose( &prev->anim);
      _DK_FREE( prev->name, -1 );
      _DK_FREE( prev->indexes, -1 );
      _DK_FREE( prev, -1 );
   }
}


/*******************************************************************************

    $$L  DK_klusterAddToList

    Usage      : Create a new cluster and adds it to the model's cluster list.

    [Returned Value : ptr to the new cluster]

    Author     : Rejean Gagne, Feb 92
    Updated by : <Full name>, <Date>, <Update description>

    (c) Copyright 1992 SoftImage Inc.

*******************************************************************************/
static DK_Kluster *DK_klusterAddToList
   (
      DK_Model	*mdl
   )
{
   DK_Kluster	*newkls = NULL,*prev;

   if ( mdl != NULL )
   {
      newkls = DK_klusterAllocate();
      newkls->next = NULL;

      /* add to the end of the list */
      if ( mdl->klusters == NULL )
      {
	 mdl->klusters = newkls;
      }
      else
      {
	 prev = mdl->klusters;
	 while ( prev->next != NULL )
	    prev = prev->next;
	 
	 prev->next = newkls;
      }
   }
   return( newkls );
}

/*******************************************************************************

    $$P DK_clusterCount

    Usage	: 
    Usage       :

    [Returned Value : Nb of clusters in DK_model]

    Author     : Rejean Gagne, Feb, 92
    Updated by : <Full name>, <Date>, <Update description>

    (c) Copyright 1992 SoftImage Inc.

*******************************************************************************/
int DK_clusterCount
   (
      struct DK_model *mdl
   )
{
   DK_Kluster	*kls;
   int		cnt = 0;

   kls = mdl->klusters;
   while ( kls != NULL )
   {
      cnt++;
      kls = kls->next;
   }
   return( cnt );
}

/*******************************************************************************

   $$L DK_clustersWriteBinaryChunk

   - Write clusters data to file.

   Returned Value: None.

   Author        : Rejean Gagne, February 09, 92
   Updated by    : <UPDATER>, <DATE>, <DESCRIPTION>

   (c) Copyright 1991 Softimage Inc.

*******************************************************************************/

void DK_clustersWriteBinaryChunk
   (
      FILE      *file,
      struct DK_model *mdl
   )
{
   DK_Kluster	*kls;
   int		i, nb;

   /* note : the number of clusters has already been written before this */
   kls = mdl->klusters;

   while ( kls != NULL )
   {
      /* 1 : name   */
      DK_stringWriteBinary( file, kls->name );

      /* 2 : nb of cv's */
      DK_byteswap_fwrite( &(kls->ncv),  sizeof(int), 1, file );

      /* 3 : cv indexes */
      switch (mdl->type)
      {
	 case DK_MDL_FACE:
	     nb = kls->ncv * 2;
	     break;
	 default:
	     nb = kls->ncv;
	     break;
      }
      for( i=0; i< nb; i++ )
      {
	 DK_byteswap_fwrite( &(kls->indexes[i]), sizeof(int), 1, file );
      }

      /* 4 : state */
      DK_byteswap_fwrite( &(kls->state),  sizeof(int), 1, file );

      kls = kls->next;
   }
}

/*******************************************************************************

   $$L DK_clustersReadBinaryChunk

   - Read  DK_clusters data from file.

   Returned Value: None.

   Author        : Rejean Gagne, February 09, 92
   Updated by    : <UPDATER>, <DATE>, <DESCRIPTION>

   (c) Copyright 1991 Softimage Inc.

*******************************************************************************/

void DK_clustersReadBinaryChunk
   (
      FILE      	*file,
      struct DK_model	*mdl,
      DK_FileProt         *prot
   )
{
   short	tok,ind = 0;
   int		*pind,ncv,i,state, nb;
   DK_Kluster	*kls;
   char		*name;

   /* note : the number of clusters has already been read before this */

   if( prot->cluster != NULL )
      tok = prot->cluster[ ind ++ ];
   else
      DK_byteswap_fread( &tok, sizeof(short), 1, file );

   while( tok != 0 ) /* End of cluster data definition */
   {
      switch( tok )
      {
         case 1: /* name */
	    /* we create and add a new cluster to the DK_model */
	    kls  =  DK_klusterAddToList( mdl );

	    name = (char *) DK_stringReadBinary( file );
	    if ( kls != NULL )
	       kls->name = name;
            break;
         case 2: /* nb of cvs */
            DK_byteswap_fread( &(ncv), sizeof(int), 1, file );
	    if ( kls != NULL )
	       kls->ncv = ncv;
            break;
         case 3: /* indexes */
	    switch (mdl->type)
	    {
		case DK_MDL_FACE:
		    nb = ncv * 2;
		    break;
		default:
		    nb = ncv;
		    break;
	    }
	    pind = _DK_MALLOC( int, sizeof( int ) * nb, -1);
	    if ( kls != NULL && pind != NULL )
	    {
	       kls->indexes = pind;
	       for ( i = 0; i < nb; i++ )
		  DK_byteswap_fread( &(kls->indexes[i]), sizeof(int), 1, file );
	    }
            break;
	 case 4: /* state */
            DK_byteswap_fread( &(state), sizeof(int), 1, file );
	    if ( kls != NULL )
	       kls->state = state;
	    break;


         default:
            DK_fileSkipField( file, tok );

      }
      if( prot->cluster != NULL )
         tok = prot->cluster[ ind ++ ];
      else
         DK_byteswap_fread( &tok, sizeof(short), 1, file );
   }

   return;
}

/*******************************************************************************

   $$L DK_clustersWriteAscii

   - Write clusters data to ascii file.

   Returned Value: None.

   Author        : Rejean Gagne, February 27, 92
   Updated by    : <UPDATER>, <DATE>, <DESCRIPTION>

   (c) Copyright 1992 Softimage Inc.

*******************************************************************************/

void DK_clustersWriteAscii
   (
      FILE         *file,
      struct DK_model *mdl,
      char	   *space
   )
{
   DK_Kluster	*kls;
   int		i, nb;
   char         sspace[80];

   /* note : the number of clusters has already been written before this */
   kls = mdl->klusters;

   fprintf( file, "\n%s%s\n%s{", space, DK_CLUSTER_TOKEN, space );
   sprintf( sspace, "%s  ", space );

   while ( kls != NULL )
   {
      /* 1 : name   */
      fprintf( file, "\n%s%s \"%s\"", sspace, DK_CLUSTER_NAME_TOKEN, kls->name );

      /* 2 : nb of cv's */
      fprintf( file, "\n%s%s %d", sspace, DK_CLUSTER_NCV_TOKEN, kls->ncv );

      /* 3 : cv indexes */
      switch (mdl->type)
      {
	 case DK_MDL_FACE:
	     nb = kls->ncv * 2;
	     break;
	 default:
	     nb = kls->ncv;
	     break;
      }
      fprintf( file, "\n%s%s", sspace, DK_CLUSTER_VERT_TOKEN );
      fprintf( file, "\n%s{\n%s  ", sspace, sspace );
      for( i=0; i< nb; i++ )
	 fprintf( file, "%d ", kls->indexes[i] );
      fprintf( file, "\n%s}", sspace );

      /* 4 : state */
      fprintf( file, "\n%s%s %d", sspace, DK_CLUSTER_STATE_TOKEN, kls->state );

      kls = kls->next;

      if ( kls != NULL )
	 fprintf( file, "\n" );
   }

   fprintf( file, "\n%s}\n", space );
}

/*******************************************************************************

   $$L DK_clustersReadAscii

   - Write clusters data from ascii file.

   Returned Value: None.

   Author        : Rejean Gagne, February 27, 92
   Updated by    : <UPDATER>, <DATE>, <DESCRIPTION>

   (c) Copyright 1992 Softimage Inc.

*******************************************************************************/

void DK_clustersReadAscii
   (
      FILE	*file,
      DK_Model	*mdl
   )
{
   DK_Kluster	*kls;
   long		ncv,ind,state;
   int		*pind,i,nb;
   char		*name;
   short 	typ;

   DK_in_animToken( file );
   while ( TRUE )
   {
      if ( strcmp( DK_CLUSTER_NAME_TOKEN, DK_animToken ) == 0 )
      {
	 kls = DK_klusterAddToList( mdl );

	 name = DK_stringReadAscii( file );
	 if ( kls != NULL )
	    kls->name = name;
      }
      else if ( strcmp( DK_CLUSTER_NCV_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animLValue( file, &ncv );
	 if ( kls != NULL )
	    kls->ncv = (int) ncv;
      }
      else if ( strcmp( DK_CLUSTER_VERT_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animToken( file );	/* read the "{" */

	 switch (mdl->type)
	 {
	     case DK_MDL_FACE:
		 nb = ncv * 2;
		 break;
	     default:
		 nb = ncv;
		 break;
	 }
	 pind = _DK_MALLOC( int, sizeof( int ) * nb, -1);
	 if ( kls != NULL && pind != NULL )
	 {
	    kls->indexes = pind;
	    for ( i = 0; i < nb; i++ )
	    {
	       DK_in_animLValue( file, &ind );
	       kls->indexes[i] = (int) ind;
	    }
	 }

	 DK_in_animToken( file );	/* read the "}" */
      }
      else if ( strcmp( DK_CLUSTER_STATE_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animLValue( file, &state );
	 if ( kls != NULL )
	    kls->state = (int) state;
      }
      else if ( strcmp( "{", DK_animToken ) == 0 )
      {
      }
      else if ( strcmp( "}", DK_animToken ) == 0 )
      {
	 return;
      }
      else
	 printf(" token [%s] seems invalid here\n", DK_animToken );

      DK_in_animToken( file );
   }
   
}
