/*******************************************************************************
********************************************************************************

   $Source: /drive3/SI2/creative/RCS/DKit/src/tinyActor.c,v $
   $Revision: 1.12.14.1 $ $Date: 1995/06/09 21:59:52 $
   Checkin by: $Author: poirierm $

   This file contains a subset of the SOFTIMAGE's Actor functions.

   Written by: Colin Hui

   (c) Copyright 1991, 1992 SOFTIMAGE Inc.

********************************************************************************
*******************************************************************************/

#include "tinyByteswap.h"
#include "tinySoftType.h"
#include "tinySceneToken.h"
#include "tinyUtils.h"
#include "tinyModel.h"

static char *s20 = "                     ";

/* static routines */
static void DK_envelopVertWriteBinaryChunk (  FILE *, DK_EnvVert *);
static void DK_envelopVertReadBinaryChunk (  FILE *, DK_EnvListVert *, DK_FileProt *);
static DK_EnvVert *DK_oEnvelopeVertReadBinaryChunk (  FILE *);

/*******************************************************************************

   $$L DK_invKinAllocate

   This function allocates an inverse kinematic variable structure.

   Returned Value: DK_InvKin structure

*******************************************************************************/

DK_InvKin *DK_invKinAllocate( void )
{
   DK_InvKin *ik;

   ik = (DK_InvKin *) calloc( 1, sizeof( DK_InvKin ));
   ik->type = 0;
   ik->nbjts = 0;
   _DK_VectorInit( ik->prot, 0.0, 0.0, 0.0 );

   return( ik );
}

/*******************************************************************************

   $$L DK_envelopDataAllocate

   This function allocates an envelopment data structure for a joint.

   Returned Value: DK_EnvData structure

*******************************************************************************/

DK_EnvData *DK_envelopDataAllocate( int level )
{
   DK_EnvData *data;

   data = _DK_MALLOC( DK_EnvData, sizeof( DK_EnvData ), -1 );

   data->level = level;
   data->envlist = NULL;
   _DK_MatrixIdInit( data->transmat );
   _DK_VectorInit( data->initvrot, 0.0, 0.0, 0.0 );
   _DK_VectorInit( data->oldvrot, 0.0, 0.0, 0.0 );
   _DK_VectorInit( data->rootscl, 1.0, 1.0, 1.0 );
   _DK_VectorInit( data->rootrot, 0.0, 0.0, 0.0 );
   _DK_VectorInit( data->roottrs, 0.0, 0.0, 0.0 );
   data->jointarea = 0.2;
   data->indefcrv.type = DK_FINGER;
   data->indefcrv.data = NULL;
   data->indefcrv.beg = 0.0;
   data->indefcrv.end = 1.0;
   data->indefcrv.ampl = 0.0;
   data->indefcrv.scale = 1.0;
   data->outdefcrv.type = DK_FINGER;
   data->outdefcrv.data = NULL;
   data->outdefcrv.beg = 0.0;
   data->outdefcrv.end = 1.0;
   data->outdefcrv.ampl = 0.0;
   data->outdefcrv.scale = 1.0;

   return( data );
}

/*******************************************************************************

   $$L DK_envelopListVertAllocate

   This function allocates an envelope list structure.

   Returned Value: DK_EnvListVert structure

*******************************************************************************/

DK_EnvListVert *DK_envelopListVertAllocate( struct DK_model *env )
{
   DK_EnvListVert *list;

   list = _DK_MALLOC( DK_EnvListVert, sizeof( DK_EnvListVert ), -1 );

   list->env = env;
   list->segvert = NULL;
   list->prev = NULL;
   list->next = NULL;

   return( list );
}

/*******************************************************************************

   $$L DK_envelopVertAllocate

   This function allocates an envelope vertex variable structure.

   Returned Value: DK_EnvVert structure

*******************************************************************************/

DK_EnvVert *DK_envelopVertAllocate( void )
{
   DK_EnvVert *vert;

   vert = _DK_MALLOC( DK_EnvVert, sizeof( DK_EnvVert ), -1 );

   vert->index = 0;
   /* Obsolete */
   _DK_VectorInit( vert->coord, 0.0, 0.0, 0.0 );
   _DK_MatrixIdInit( vert->trmat );
   vert->ratio		= 0.0;
   vert->thickness	= 0.0;
   vert->weight		= 100.0;
   vert->next = NULL;

   return( vert );
}

/*******************************************************************************

   $$L DK_invKinDispose

   This function disposes an DK_InvKin structure.

   Returned Value: None

*******************************************************************************/

void DK_invKinDispose
    (
	DK_InvKin **ik
    )
{
    if (ik != NULL)
	_DK_FREE(*ik, -1);
}

/*******************************************************************************

   $$L DK_envelopDataDispose

   This function disposes a DK_EnvData structure.

   Returned Value: None

*******************************************************************************/

void DK_envelopDataDispose
    (
	DK_EnvData **ei
    )
{
    if (ei != NULL && *ei != NULL)
    {
        DK_EnvListVert *envlist = (*ei)->envlist;

        while (envlist)
        {
           DK_EnvListVert *save = envlist;
           DK_envelopVertDispose( &envlist->segvert );
           envlist = envlist->next;
           free( save );
        }

        _DK_FREE((*ei)->indefcrv.data, -1);
        _DK_FREE((*ei)->outdefcrv.data, -1);
        _DK_FREE(*ei, -1);
    }
}

/*******************************************************************************

   $$L DK_envelopVertDispose

   This function disposes an DK_EnvVert structure.

   Returned Value: None

*******************************************************************************/

void DK_envelopVertDispose
    (
	DK_EnvVert **ev
    )
{
    if (ev != NULL)
    {
        DK_EnvVert *e = *ev;

        while ( e )
        {
           DK_EnvVert *save = e;
           e = e->next;
           free( save );
        }

        *ev = NULL;
    }
}

/*******************************************************************************

   $$L DK_modelAlgoTypeWriteAscii

   - Writes information concerning the algo type.

   Returned Value: None.

*******************************************************************************/

static void DK_modelAlgoTypeWriteAscii
   (
       FILE	*file,
      DK_Model	*mdl,
      char	*space
   )
{
   switch ( mdl->type )
   {
      case DK_MDL_JNT:
	 _DK_OUT_TOKEN(DK_A_CHAIN_TOKEN );
	 fprintf( file, "   " );
	 _DK_OUT_TOKEN(DK_A_CHNJNT_TOKEN );
	 break;
      case DK_MDL_NILL:
	 switch ( mdl->algotype )
	 {
	    case DK_ASTD :
	    case DK_ADYNMDL :
	       break;
	    case DK_ADYNA :
	    case DK_AIKIN :
	    case DK_ADYNLEAF :
	    case DK_AIKLEAF :
	       _DK_OUT_TOKEN(DK_A_CHAIN_TOKEN );
	       break;
	    default :
	       _DK_OUT_TOKEN( DK_A_DYNCTL_TOKEN );
	       break;
	 }
	 fprintf( file, "   " );
	 switch ( mdl->algotype )
	 {
	    case DK_ADYNA :
	    case DK_AIKIN :
	       _DK_OUT_TOKEN(DK_A_CHNROOT_TOKEN );
	       break;
	    case DK_ADYNLEAF :
	    case DK_AIKLEAF :
	       _DK_OUT_TOKEN(DK_A_CHNLEAF_TOKEN );
	       break;
	    case DK_ADYNGRAV :
	       _DK_OUT_TOKEN( DK_A_DYN_GRAVITY_TOKEN );
	       break;
	    case DK_ADYNDGRAV :
	       _DK_OUT_TOKEN( DK_A_DYN_DEFGRAV_TOKEN );
	       break;
	    case DK_ADYNFRCE :
	       _DK_OUT_TOKEN( DK_A_DYN_FORCE_TOKEN );
	       break;
	    case DK_ADYNWIND :
	       _DK_OUT_TOKEN( DK_A_DYN_WIND_TOKEN );
	       break;
	    case DK_ADYNFAN :
	       _DK_OUT_TOKEN( DK_A_DYN_FAN_TOKEN );
	       break;
	    case DK_ADYNNAIL :
	       _DK_OUT_TOKEN( DK_A_DYN_NAIL_TOKEN );
	       break;
	 }
	 break;
   }
}

/*******************************************************************************

   $$L DK_modelAlgoTypeReadAscii

   - Reads information concerning the algo type.

   Returned Value: None.

*******************************************************************************/

static void DK_modelAlgoTypeReadAscii
   (
       FILE	*file,
      DK_Model	*mdl
   )
{

   if ( strcmp(DK_A_CHAIN_TOKEN, DK_animToken ) == 0 )
   {
      mdl->type = DK_MDL_NILL;
      DK_in_animToken( file );
      if ( strcmp(DK_A_CHNJNT_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->algotype = DK_AIKIN;
	 mdl->type = DK_MDL_JNT;
      }
      else if ( strcmp(DK_A_CHNROOT_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->algotype = DK_AIKIN;
      }
      else if ( strcmp(DK_A_CHNLEAF_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->algotype = DK_AIKLEAF;
      }
      else
      {
      }
      return;
   }

   if ( strcmp( DK_A_DYNCTL_TOKEN, DK_animToken ) == 0 )
   {
      mdl->algotype = DK_ASTD;
      DK_in_animToken( file );
      if ( strcmp( DK_A_DYN_GRAVITY_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->algotype = DK_ADYNGRAV;
      }
      else if ( strcmp( DK_A_DYN_DEFGRAV_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->algotype = DK_ADYNDGRAV;
      }
      else if ( strcmp( DK_A_DYN_FORCE_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->algotype = DK_ADYNFRCE;
      }
      else if ( strcmp( DK_A_DYN_WIND_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->algotype = DK_ADYNWIND;
      }
      else if ( strcmp( DK_A_DYN_FAN_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->algotype = DK_ADYNFAN;
      }
      else if ( strcmp( DK_A_DYN_NAIL_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->algotype = DK_ADYNNAIL;
      }
   }
}

/*******************************************************************************

   $$L DK_invKinWriteBinaryChunk

   - Write ik data to file.

   Returned Value: None.

*******************************************************************************/

void DK_invKinWriteBinaryChunk
   (
       FILE *file,	/* destination file (I) */
      DK_InvKin *ikvar	/* ik data to write (I) */
   )
{
   if ( ikvar != NULL )
   {
      DK_byteswap_fwrite( &ikvar->type, sizeof(int), 1, file );
      DK_byteswap_fwrite( &ikvar->nbjts, sizeof(int), 1, file );
      DK_Vector_fwrite( &ikvar->prot, sizeof(DK_Vector), 1, file );
   }
}


/*******************************************************************************

   $$L DK_invKinReadBinaryChunk

   - Read ik data from file.

   Returned Value: pointer to ik data.

*******************************************************************************/

DK_InvKin * DK_invKinReadBinaryChunk
   (
       FILE *file	/* source file (I) */
   )
{
   DK_InvKin *ikvar;
   int nb;

   if ( (ikvar = DK_invKinAllocate()) != NULL )
   {
      DK_byteswap_fread( &ikvar->type, sizeof(int), 1, file );
      DK_byteswap_fread( &ikvar->nbjts, sizeof(int), 1, file );
      DK_Vector_fread( &ikvar->prot, sizeof(DK_Vector), 1, file );
   }

   return( ikvar );
}

/*******************************************************************************

   $$L invKinWriteAscii

   - Write ik data to file.

   Returned Value: None.

*******************************************************************************/

static void invKinWriteAscii
   (
       FILE *file,	/* destination file (I) */
      DK_InvKin *ikvar,	/* ik data to write (I) */
      char	*space
   )
{
   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN(DK_A_IKTYPE_TOKEN );
   fprintf(file,"         ");
   _DK_OUT_SVALUE( ikvar->type );

   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN(DK_A_IKNBJNTS_TOKEN );
   fprintf(file,"       ");
   _DK_OUT_SVALUE( ikvar->nbjts );

   fprintf(file,"\n%s  ",space);
   _DK_OUT_TOKEN(DK_A_IKPROT_TOKEN );
   fprintf(file,"  ");
   _DK_OUT_FVALUE( ikvar->prot.x );
   _DK_OUT_FVALUE( ikvar->prot.y );
   _DK_OUT_FVALUE( ikvar->prot.z );

   _DK_OUT_NEWLINE();
}

/*******************************************************************************

   $$L invKinReadAscii

   - Read ik data from ascii file.

   Returned Value: None.

*******************************************************************************/

static DK_InvKin *invKinReadAscii
   (
       FILE 	*file,
      char	*lastToken
   )
{
   DK_InvKin	*p;
   short	 ii;

   p = DK_invKinAllocate();
   if ( p == NULL &&  DK_option.verbose == TRUE)
   {
      fprintf( DK_option.msgFd,"invKinReadAscii : Allocate failed ! \n");
      return( NULL );
   }

   DK_in_animToken(file);
   while( TRUE )
   {
      if ( strcmp(DK_A_IKTYPE_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &ii );
	 p->type = (int) ii;
      }
      else if ( strcmp(DK_A_IKNBJNTS_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &ii );
	 p->nbjts = (int) ii;
      }
      else if ( strcmp(DK_A_IKPROT_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->prot.x) );
	 DK_in_animFValue( file, &(p->prot.y) );
	 DK_in_animFValue( file, &(p->prot.z) );
      }
      else
      {
	 strcpy( lastToken, DK_animToken );
	 return( p );
      }

      DK_in_animToken( file );
   }
}

/*******************************************************************************

   $$L DK_envelopVertWriteBinaryChunk

   - Write data of envelope's vertices assigned to a skeleton's joint.

   Returned Value: None.

*******************************************************************************/

static void DK_envelopVertWriteBinaryChunk
   (
       FILE *file,
      DK_EnvVert *ptr
   )
{
   DK_byteswap_fwrite( &ptr->index, sizeof(long), 1, file );	/* 1 */
   DK_byteswap_fwrite( &ptr->ratio, sizeof(float), 1, file );	/* 3 */
   DK_byteswap_fwrite( &ptr->thickness, sizeof(float), 1, file );	/* 4 */
   if ( DK_VERSION >= DK_v2_6 )
      DK_byteswap_fwrite( &ptr->weight, sizeof(float), 1, file );	/* 6 */
}

/*******************************************************************************

   $$L DK_envelopVertReadBinaryChunk

   - Read data of envelope's vertices assigned to a skeleton's joint.

   Returned Value: None.

*******************************************************************************/

static void DK_envelopVertReadBinaryChunk
   (
       FILE 		*file,
      DK_EnvListVert   	*envl,
      DK_FileProt          *prot
   )
{
   DK_EnvVert *newvert, *tmp;
   short    tok, ind = 0;

   newvert = DK_envelopVertAllocate();

   if ( newvert == NULL )
      return;

   if ( envl->segvert == NULL )
      envl->segvert = newvert;
   else
   {
      tmp = envl->segvert;
      envl->segvert = newvert;
      envl->segvert->next = tmp;
   }

   if( prot->envelopeVertex != NULL )
      tok = prot->envelopeVertex[ ind ++ ];
   else
      DK_byteswap_fread( &tok, sizeof(short), 1, file );

   while( tok != 0 ) /* End of envelop vertice data definition */
   {
      switch( tok )
      {
	 case 1:
	    DK_byteswap_fread( &newvert->index, sizeof(long), 1, file );
	    break;
	 case 2:
	    /* Obsolete */
	    DK_Vector_fread( &newvert->coord, sizeof(DK_Vector), 1, file );
	    break;
	 case 3:
	    DK_byteswap_fread( &newvert->ratio, sizeof(float), 1, file );
	    break;
	 case 4:
	    DK_byteswap_fread( &newvert->thickness, sizeof(float), 1, file );
	    break;
	 case 5:
	    /* Obsolete */
	    DK_Matrix_fread( newvert->trmat, sizeof(DK_Matrix), 1, file );
	    newvert->coord.x = 0.0;
	    newvert->coord.y = 0.0;
	    newvert->coord.z = 0.0;
	    break;
	 case 6:
	    DK_byteswap_fread( &newvert->weight, sizeof(float), 1, file );
	    break;

         default:
            DK_fileSkipField( file, tok );
      }
      if( prot->envelopeVertex != NULL )
         tok = prot->envelopeVertex[ ind ++ ];
      else
         DK_byteswap_fread( &tok, sizeof(short), 1, file );
   }
}

/*******************************************************************************

   $$L DK_envelopWriteBinaryChunk

   - Write envelopment data of a skeleton's joint.

   Returned Value: None.

*******************************************************************************/

void DK_envelopWriteBinaryChunk
   (
       FILE *file,
      DK_EnvData *pdata
   )
{
   short i, nb;
   long j, nb2;
   DK_EnvListVert *ptr;
   DK_EnvVert *ptr2;

   if ( pdata != NULL )
   {
      DK_byteswap_fwrite( &pdata->level, sizeof(int), 1, file );		  /* 1 */
      DK_Vector_fwrite( &pdata->initvrot, sizeof(DK_Vector), 1, file );	  /* 2 */
      DK_Vector_fwrite( &pdata->rootscl, sizeof(DK_Vector), 1, file );	  /* 22 */
      DK_Vector_fwrite( &pdata->rootrot, sizeof(DK_Vector), 1, file );	  /* 23 */
      DK_Vector_fwrite( &pdata->roottrs, sizeof(DK_Vector), 1, file );	  /* 24 */
      DK_byteswap_fwrite( &pdata->jointarea, sizeof(float), 1, file );	  /* 5 */
      DK_byteswap_fwrite( &pdata->indefcrv.type, sizeof(DK_DefType), 1, file );  /* 4 */
      DK_byteswap_fwrite( &pdata->indefcrv.beg, sizeof(float), 1, file );	  /* 7 */
      DK_byteswap_fwrite( &pdata->indefcrv.end, sizeof(float), 1, file );	  /* 8 */
      DK_byteswap_fwrite( &pdata->indefcrv.ampl, sizeof(float), 1, file );	  /* 9 */
      DK_byteswap_fwrite( &pdata->indefcrv.scale, sizeof(float), 1, file );	  /* 11 */
      DK_byteswap_fwrite( &pdata->outdefcrv.type, sizeof(DK_DefType), 1, file ); /* 12 */
      DK_byteswap_fwrite( &pdata->outdefcrv.beg, sizeof(float), 1, file );	  /* 13 */
      DK_byteswap_fwrite( &pdata->outdefcrv.end, sizeof(float), 1, file );	  /* 14 */
      DK_byteswap_fwrite( &pdata->outdefcrv.ampl, sizeof(float), 1, file );	  /* 15 */
      DK_byteswap_fwrite( &pdata->outdefcrv.scale, sizeof(float), 1, file );  /* 16 */

      nb = 0;
      if ( pdata->indefcrv.data == NULL )			  /* 17 */
	 DK_byteswap_fwrite( &nb, sizeof(short), 1, file );
      else
      {
	 nb = 2* DK_SAMP;
	 DK_byteswap_fwrite( &nb, sizeof(short), 1, file );
	 for ( i = 0; i < nb; i++ )
	    DK_byteswap_fwrite( &pdata->indefcrv.data[ i ], sizeof(float), 1, file );
      }

      nb = 0;
      if ( pdata->outdefcrv.data == NULL )			  /* 18 */
	 DK_byteswap_fwrite( &nb, sizeof(short), 1, file );
      else
      {
	 nb = 2* DK_SAMP;
	 DK_byteswap_fwrite( &nb, sizeof(short), 1, file );
	 for ( i = 0; i < nb; i++ )
	    DK_byteswap_fwrite( &pdata->outdefcrv.data[ i ], sizeof(float), 1, file );
      }

      /* 20 : envelop list */
      for ( ptr = pdata->envlist, nb = 0; ptr != NULL; ptr = ptr->next, nb++ );
      DK_byteswap_fwrite( &nb, sizeof(short), 1, file );

      /* 21 : vertices for each envelop */
      for ( ptr = pdata->envlist, i = 0; i < nb; ptr = ptr->next, i++ )
      {
	 for ( ptr2 = ptr->segvert, nb2 = 0; ptr2 != NULL;
	       ptr2 = ptr2->next, nb2++ );
	 DK_byteswap_fwrite( &nb2, sizeof(long), 1, file );
	 for ( ptr2 = ptr->segvert, j = 0; j < nb2; ptr2 = ptr2->next, j++ )
	    DK_envelopVertWriteBinaryChunk( file, ptr2 );
      }
   }
}

/*******************************************************************************

   $$L DK_envelopReadBinaryChunk

   - Read envelopment data of a skeleton's joint.

   Returned Value: DK_Pointer to data, or NULL if failed.

*******************************************************************************/

DK_EnvData * DK_envelopReadBinaryChunk
   (
       FILE 		*file,
      DK_Model		*mdl,
      DK_FileProt          *prot
   )
{
   short tok, ind = 0;
   short i, nb, nbe;
   long j, nb2;
   int ii, inb;
   float *ptr, dummyf;
   DK_Vector dummyv;
   DK_EnvData *pdata;
   DK_EnvListVert *envl, *last;

   pdata = DK_envelopDataAllocate( 0 );
   if ( pdata == NULL )
      return( NULL );

   if ( mdl->parent != NULL && _DK_MODEL_IS_CHNROOT( mdl->parent ) )
   {
      _DK_VectorInit( pdata->rootscl, mdl->parent->scaling.x,
		   mdl->parent->scaling.y, mdl->parent->scaling.z );
      _DK_VectorInit( pdata->rootrot, mdl->parent->rotation.x,
		   mdl->parent->rotation.y, mdl->parent->rotation.z );
      _DK_VectorInit( pdata->roottrs, mdl->parent->translation.x,
		   mdl->parent->translation.y, mdl->parent->translation.z );
   }

   if ( _DK_MODEL_IS_CHNLEAF( mdl ) )
   {
      _DK_VectorInit( pdata->initvrot,
		   mdl->rotation.x, mdl->rotation.y, mdl->rotation.z );
   }

   if( prot->envelopeData != NULL )
      tok = prot->envelopeData[ ind ++ ];
   else
      DK_byteswap_fread( &tok, sizeof(short), 1, file );

   while( tok != 0 ) /* End of envelopment data definition */
   {
      switch( tok )
      {
	 case 1:
	    DK_byteswap_fread( &pdata->level, sizeof(int), 1, file );
	    break;
	 case 2:
	    DK_Vector_fread( &pdata->initvrot, sizeof(DK_Vector), 1, file );
	    break;
	 case 3:
	    DK_Vector_fread( &dummyv, sizeof(DK_Vector), 1, file );
	    break;
	 case 4:
	    DK_byteswap_fread( &pdata->indefcrv.type, sizeof(DK_DefType), 1, file );
	    break;
	 case 5:
	    DK_byteswap_fread( &pdata->jointarea, sizeof(float), 1, file );
	    break;
	 case 6:
	    DK_byteswap_fread( &dummyf, sizeof(float), 1, file );
	    break;
	 case 7:
	    DK_byteswap_fread( &pdata->indefcrv.beg, sizeof(float), 1, file );
	    break;
	 case 8:
	    DK_byteswap_fread( &pdata->indefcrv.end, sizeof(float), 1, file );
	    break;
	 case 9:
	    DK_byteswap_fread( &pdata->indefcrv.ampl, sizeof(float), 1, file );
	    break;
	 case 11:
	    DK_byteswap_fread( &pdata->indefcrv.scale, sizeof(float), 1, file );
	    break;
	 case 12:
	    DK_byteswap_fread( &pdata->outdefcrv.type, sizeof(DK_DefType), 1, file );
	    break;
	 case 13:
	    DK_byteswap_fread( &pdata->outdefcrv.beg, sizeof(float), 1, file );
	    break;
	 case 14:
	    DK_byteswap_fread( &pdata->outdefcrv.end, sizeof(float), 1, file );
	    break;
	 case 15:
	    DK_byteswap_fread( &pdata->outdefcrv.ampl, sizeof(float), 1, file );
	    break;
	 case 16:
	    DK_byteswap_fread( &pdata->outdefcrv.scale, sizeof(float), 1, file );
	    break;
	 case 17:
	    DK_byteswap_fread( &nb, sizeof(short), 1, file );
	    if ( nb > 0 )
	    {
	       ptr = _DK_MALLOC( float, nb * sizeof(float), -1 );
	       for ( i = 0; i < nb; i++ )
		  DK_byteswap_fread( &ptr[ i ], sizeof(float), 1, file );
	       pdata->indefcrv.data = ptr;
	    }
	    break;
	 case 18:
	    DK_byteswap_fread( &nb, sizeof(short), 1, file );
	    if ( nb > 0 )
	    {
	       ptr = _DK_MALLOC( float, nb * sizeof(float), -1 );
	       for ( i = 0; i < nb; i++ )
		  DK_byteswap_fread( &ptr[ i ], sizeof(float), 1, file );
	       pdata->outdefcrv.data = ptr;
	    }
	    break;
	 case 10:   /* note: this token is not written anymore */
	 case 19:   /* note: this token is not written anymore */
	    pdata->envlist = DK_envelopListVertAllocate( NULL );
	    if ( tok == 10 )
	       DK_byteswap_fread( &inb, sizeof(int), 1, file );
	    else
	    {
	       DK_byteswap_fread( &nb, sizeof(short), 1, file );
	       inb = (int) nb;
	    }
	    for ( ii = 0; ii < inb; ii++ )
	       DK_envelopVertReadBinaryChunk( file, pdata->envlist, prot );
	    break;
	 case 20:
	    DK_byteswap_fread( &nbe, sizeof(short), 1, file );
	    break;
	 case 21:
	    for ( i = 0; i < nbe; i++ )
	    {
	       envl = DK_envelopListVertAllocate( NULL );
	       DK_byteswap_fread( &nb2, sizeof(long), 1, file );
	       for ( j = 0; j < nb2; j++ )
		  DK_envelopVertReadBinaryChunk( file, envl, prot );

	       if ( pdata->envlist == NULL )
		  pdata->envlist = envl;
	       else
	       {
		  last->next = envl;
		  envl->prev = last;
	       }
	       last = envl;
	    }
	    break;
         case 22:
            DK_Vector_fread( &pdata->rootscl, sizeof(DK_Vector), 1, file );
            break;
         case 23:
            DK_Vector_fread( &pdata->rootrot, sizeof(DK_Vector), 1, file );
            break;
         case 24:
            DK_Vector_fread( &pdata->roottrs, sizeof(DK_Vector), 1, file );
            break;

         default:
            DK_fileSkipField( file, tok );

      }

      if( prot->envelopeData != NULL )
         tok = prot->envelopeData[ ind ++ ];
      else
         DK_byteswap_fread( &tok, sizeof(short), 1, file );
   }

   return ( pdata );
}

/*******************************************************************************

   $$L DK_envelopVertWriteAscii

   - Write data of envelope's vertices assigned to a skeleton's joint.

   Returned Value: None.

*******************************************************************************/

static void DK_envelopVertWriteAscii
   (
       FILE *file,
      DK_EnvVert *ptr,
      char *space
   )
{
   fprintf(file,"\n%s    {",space);
   fprintf(file,"\n%s     ",space);
   _DK_OUT_TOKEN( DK_W_ENVVERT_INDEX_TOKEN );
   _DK_OUT_SVALUE( ptr->index );

   /* Obsolete : BEGIN */
   fprintf(file,"\n%s     ",space);
   _DK_OUT_TOKEN( DK_W_ENVVERT_TRMAT_TOKEN );
   fprintf(file,"%*.*f  %*.*f  %*.*f  %*.*f",  DK_FM,  DK_FD, ptr->trmat[0][0],
					       DK_FM,  DK_FD, ptr->trmat[0][1],
					       DK_FM,  DK_FD, ptr->trmat[0][2],
					       DK_FM,  DK_FD, ptr->trmat[0][3] );
   fprintf(file,"\n%s%s     ",space, s20);
   fprintf(file,"%*.*f  %*.*f  %*.*f  %*.*f",  DK_FM,  DK_FD, ptr->trmat[1][0],
					       DK_FM,  DK_FD, ptr->trmat[1][1],
					       DK_FM,  DK_FD, ptr->trmat[1][2],
					       DK_FM,  DK_FD, ptr->trmat[1][3] );
   fprintf(file,"\n%s%s     ",space, s20);
   fprintf(file,"%*.*f  %*.*f  %*.*f  %*.*f",  DK_FM,  DK_FD, ptr->trmat[2][0],
					       DK_FM,  DK_FD, ptr->trmat[2][1],
					       DK_FM,  DK_FD, ptr->trmat[2][2],
					       DK_FM,  DK_FD, ptr->trmat[2][3] );
   fprintf(file,"\n%s%s     ",space, s20);
   fprintf(file,"%*.*f  %*.*f  %*.*f  %*.*f",  DK_FM,  DK_FD, ptr->trmat[3][0],
					       DK_FM,  DK_FD, ptr->trmat[3][1],
					       DK_FM,  DK_FD, ptr->trmat[3][2],
					       DK_FM,  DK_FD, ptr->trmat[3][3] );
   /* Obsolete : END */

   fprintf(file,"\n%s     ",space);
   _DK_OUT_TOKEN( DK_W_ENVVERT_RATIO_TOKEN );
   _DK_OUT_FVALUE( ptr->ratio );

   fprintf(file,"\n%s     ",space);
   _DK_OUT_TOKEN( DK_W_ENVVERT_THICK_TOKEN );
   _DK_OUT_FVALUE( ptr->thickness );

   if ( DK_VERSION >= DK_v2_6 )
   {
      fprintf(file,"\n%s     ",space);
      _DK_OUT_TOKEN( DK_W_ENVVERT_WEIGHT_TOKEN );
      _DK_OUT_FVALUE( ptr->weight );
   }

   fprintf(file,"\n%s    }",space);
}

/*******************************************************************************

   $$L envelopWriteAscii

   - Write envelopment data to file.

   Returned Value: None.

*******************************************************************************/

static void envelopWriteAscii
   (
       FILE *file,
      DK_EnvData *pdata,
      char *space
   )
{
   short i, nb;
   long j, nb2;
   DK_EnvListVert *ptr;
   DK_EnvVert *ptr2;

   if ( pdata != NULL )
   {
      DK_incIndent(1);

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_LEVEL_TOKEN );
      _DK_OUT_SVALUE( pdata->level );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_INITVROT_TOKEN );
      _DK_OUT_FVALUE( pdata->initvrot.x );
      _DK_OUT_FVALUE( pdata->initvrot.y );
      _DK_OUT_FVALUE( pdata->initvrot.z );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_ROOTSCL_TOKEN );
      _DK_OUT_FVALUE( pdata->rootscl.x );
      _DK_OUT_FVALUE( pdata->rootscl.y );
      _DK_OUT_FVALUE( pdata->rootscl.z );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_ROOTROT_TOKEN );
      _DK_OUT_FVALUE( pdata->rootrot.x );
      _DK_OUT_FVALUE( pdata->rootrot.y );
      _DK_OUT_FVALUE( pdata->rootrot.z );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_ROOTTRS_TOKEN );
      _DK_OUT_FVALUE( pdata->roottrs.x );
      _DK_OUT_FVALUE( pdata->roottrs.y );
      _DK_OUT_FVALUE( pdata->roottrs.z );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_JNTAREA_TOKEN );
      _DK_OUT_FVALUE( pdata->jointarea );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_INCRVTYP_TOKEN );
      _DK_OUT_FVALUE( pdata->indefcrv.type );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_INCRVBEG_TOKEN );
      _DK_OUT_FVALUE( pdata->indefcrv.beg );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_INCRVEND_TOKEN );
      _DK_OUT_FVALUE( pdata->indefcrv.end );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_INCRVAMP_TOKEN );
      _DK_OUT_FVALUE( pdata->indefcrv.ampl );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_INCRVSCL_TOKEN );
      _DK_OUT_FVALUE( pdata->indefcrv.scale );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_OUTCRVTYP_TOKEN );
      _DK_OUT_FVALUE( pdata->outdefcrv.type );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_OUTCRVBEG_TOKEN );
      _DK_OUT_FVALUE( pdata->outdefcrv.beg );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_OUTCRVEND_TOKEN );
      _DK_OUT_FVALUE( pdata->outdefcrv.end );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_OUTCRVAMP_TOKEN );
      _DK_OUT_FVALUE( pdata->outdefcrv.ampl );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_OUTCRVSCL_TOKEN );
      _DK_OUT_FVALUE( pdata->outdefcrv.scale );

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_INCRVNBS_TOKEN );
      if ( pdata->indefcrv.data == NULL )
	 _DK_OUT_SVALUE( 0 );
      else
      {
	 _DK_OUT_SVALUE( 2* DK_SAMP );
	 fprintf(file,"\n%s  ",space);
	 _DK_OUT_TOKEN( DK_W_ENVELOP_INCRVDAT_TOKEN );
	 for ( i = 0; i < 2* DK_SAMP; i++ )
	 {
	    _DK_OUT_FVALUE( pdata->indefcrv.data[ i ] );
	    fprintf(file,"   ");
	    _DK_OUT_FVALUE( pdata->indefcrv.data[ ++i ] );
	    fprintf(file,"\n%s%s  ",space, s20);
	 }
      }

      fprintf(file,"\n%s  ",space);
      _DK_OUT_TOKEN( DK_W_ENVELOP_OUTCRVNBS_TOKEN );
      if ( pdata->outdefcrv.data == NULL )
	 _DK_OUT_SVALUE( 0 );
      else
      {
	 _DK_OUT_SVALUE( 2* DK_SAMP );
	 fprintf(file,"\n%s  ",space);
	 _DK_OUT_TOKEN( DK_W_ENVELOP_OUTCRVDAT_TOKEN );
	 for ( i = 0; i < 2* DK_SAMP; i++ )
	 {
	    _DK_OUT_FVALUE( pdata->outdefcrv.data[ i ] );
	    fprintf(file,"   ");
	    _DK_OUT_FVALUE( pdata->outdefcrv.data[ ++i ] );
	    fprintf(file,"\n%s%s  ",space, s20);
	 }
      }

      fprintf(file,"\n%s  ",space);
      for ( ptr = pdata->envlist, nb = 0; ptr != NULL; ptr = ptr->next, nb++ );
      _DK_OUT_TOKEN( DK_W_ENVELOP_NBENV_TOKEN );
      _DK_OUT_SVALUE( nb );

      for ( ptr = pdata->envlist, i = 0; i < nb; ptr = ptr->next, i++ )
      {
	 fprintf(file,"\n%s  ",space);
	 for ( ptr2 = ptr->segvert, nb2 = 0; ptr2 != NULL;
	       ptr2 = ptr2->next, nb2++ );
	 _DK_OUT_TOKEN( DK_W_ENVELOP_NBVERT_TOKEN );
	 _DK_OUT_SVALUE( nb2 );
	 for ( ptr2 = ptr->segvert, j = 0; j < nb2; ptr2 = ptr2->next, j++ )
	    DK_envelopVertWriteAscii( file, ptr2, space );
      }

      _DK_OUT_NEWLINE();
   }
}

/*******************************************************************************

   $$L DK_envelopVertReadAscii

   - Read envelope vertices data from ascii file.

   Returned Value: None.

*******************************************************************************/

static void DK_envelopVertReadAscii
   (
       FILE *file,
      DK_EnvListVert *lst
   )
{
   short nb, i;
   float *pm;
   long l;
   DK_EnvVert *newv, *tmp;

   DK_in_animToken( file );
   if ( strcmp( DK_animToken, "{" ) == 0 )
   {
      newv = DK_envelopVertAllocate();
      if ( newv == NULL &&  DK_option.verbose == TRUE)
      {
	 fprintf( DK_option.msgFd,"DK_envelopVertReadAscii : Allocate failed ! \n");
	 return;
      }

      if ( lst->segvert == NULL )
	 lst->segvert = newv;
      else
      {
	 tmp = lst->segvert;
	 lst->segvert = newv;
	 lst->segvert->next = tmp;
      }

      DK_in_animToken(file);
      while( strcmp( DK_animToken, "}" ) != 0 )
      {
	 if ( strcmp( DK_R_ENVVERT_INDEX_TOKEN, DK_animToken ) == 0 )
	 {
	    DK_in_animLValue( file, &(l) );
	    newv->index = (int) l;
	 }
	 else if ( strcmp( DK_R_ENVVERT_TRMAT_TOKEN, DK_animToken ) == 0 )
	 {
	    /* Obsolete */
	    pm = &( newv->trmat[0][0] );
	    for ( i = 0; i < 16; i++, pm++ )
	       DK_in_animFValue( file, pm );
	 }
	 else if ( strcmp( DK_R_ENVVERT_RATIO_TOKEN, DK_animToken ) == 0 )
	    DK_in_animFValue( file, &(newv->ratio) );
	 else if ( strcmp( DK_R_ENVVERT_THICK_TOKEN, DK_animToken ) == 0 )
	    DK_in_animFValue( file, &(newv->thickness) );
	 else if ( strcmp( DK_R_ENVVERT_WEIGHT_TOKEN, DK_animToken ) == 0 )
	    DK_in_animFValue( file, &(newv->weight) );

	 DK_in_animToken( file );
      }
   }
}

/*******************************************************************************

   $$L envelopReadAscii

   - Read envelopment data from ascii file.

   Returned Value: pointer to envelopment data or NULL if failed.

*******************************************************************************/

static DK_EnvData *envelopReadAscii
   (
       FILE 	*file,
      char	*lastToken
   )
{
   DK_EnvData *p;
   DK_EnvListVert *lst, *last;
   short nb, i;
   long nb2, j, l;
   float *dat;

   p = DK_envelopDataAllocate( 0 );
   if ( p == NULL &&  DK_option.verbose == TRUE)
   {
      fprintf( DK_option.msgFd,"envelopReadAscii : Allocate failed ! \n");
      return( NULL );
   }

   DK_in_animToken(file);
   while( TRUE )
   {
      if ( strcmp( DK_R_ENVELOP_LEVEL_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animLValue( file, &(l) );
	 p->level = (int) l;
      }
      else if ( strcmp( DK_R_ENVELOP_INITVROT_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->initvrot.x) );
	 DK_in_animFValue( file, &(p->initvrot.y) );
	 DK_in_animFValue( file, &(p->initvrot.z) );
      }
      else if ( strcmp( DK_R_ENVELOP_ROOTSCL_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->rootscl.x) );
	 DK_in_animFValue( file, &(p->rootscl.y) );
	 DK_in_animFValue( file, &(p->rootscl.z) );
      }
      else if ( strcmp( DK_R_ENVELOP_ROOTROT_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->rootrot.x) );
	 DK_in_animFValue( file, &(p->rootrot.y) );
	 DK_in_animFValue( file, &(p->rootrot.z) );
      }
      else if ( strcmp( DK_R_ENVELOP_ROOTTRS_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animFValue( file, &(p->roottrs.x) );
	 DK_in_animFValue( file, &(p->roottrs.y) );
	 DK_in_animFValue( file, &(p->roottrs.z) );
      }
      else if ( strcmp( DK_R_ENVELOP_JNTAREA_TOKEN, DK_animToken ) == 0 )
	 DK_in_animFValue( file, &(p->jointarea) );
      else if ( strcmp( DK_R_ENVELOP_INCRVTYP_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &(i) );
	 p->indefcrv.type = (DK_DefType) i;
      }
      else if ( strcmp( DK_R_ENVELOP_INCRVBEG_TOKEN, DK_animToken ) == 0 )
	 DK_in_animFValue( file, &(p->indefcrv.beg) );
      else if ( strcmp( DK_R_ENVELOP_INCRVEND_TOKEN, DK_animToken ) == 0 )
	 DK_in_animFValue( file, &(p->indefcrv.end) );
      else if ( strcmp( DK_R_ENVELOP_INCRVAMP_TOKEN, DK_animToken ) == 0 )
	 DK_in_animFValue( file, &(p->indefcrv.ampl) );
      else if ( strcmp( DK_R_ENVELOP_INCRVSCL_TOKEN, DK_animToken ) == 0 )
	 DK_in_animFValue( file, &(p->indefcrv.scale) );
      else if ( strcmp( DK_R_ENVELOP_OUTCRVTYP_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &(i) );
	 p->outdefcrv.type = (DK_DefType) i;
      }
      else if ( strcmp( DK_R_ENVELOP_OUTCRVBEG_TOKEN, DK_animToken ) == 0 )
	 DK_in_animFValue( file, &(p->outdefcrv.beg) );
      else if ( strcmp( DK_R_ENVELOP_OUTCRVEND_TOKEN, DK_animToken ) == 0 )
	 DK_in_animFValue( file, &(p->outdefcrv.end) );
      else if ( strcmp( DK_R_ENVELOP_OUTCRVAMP_TOKEN, DK_animToken ) == 0 )
	 DK_in_animFValue( file, &(p->outdefcrv.ampl) );
      else if ( strcmp( DK_R_ENVELOP_OUTCRVSCL_TOKEN, DK_animToken ) == 0 )
	 DK_in_animFValue( file, &(p->outdefcrv.scale) );
      else if ( strcmp( DK_R_ENVELOP_INCRVNBS_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &(nb) );
	 if ( nb > 0 )
	 {
	    DK_in_animToken( file );
	    if ( strcmp( DK_R_ENVELOP_INCRVDAT_TOKEN, DK_animToken ) == 0 )
	    {
	       dat = _DK_MALLOC( float, nb * sizeof( float ), -1 );
	       for ( i = 0; i < nb; i++ )
		  DK_in_animFValue( file, &(dat[ i ]) );
	       p->indefcrv.data = dat;
	    }
	 }
      }
      else if ( strcmp( DK_R_ENVELOP_OUTCRVNBS_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &(nb) );
	 if ( nb > 0 )
	 {
	    DK_in_animToken( file );
	    if ( strcmp( DK_R_ENVELOP_OUTCRVDAT_TOKEN, DK_animToken ) == 0 )
	    {
	       dat = _DK_MALLOC( float, nb * sizeof( float ), -1 );
	       for ( i = 0; i < nb; i++ )
		  DK_in_animFValue( file, &(dat[ i ]) );
	       p->outdefcrv.data = dat;
	    }
	 }
      }
      else if ( strcmp( DK_R_ENVELOP_NBENV_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &(nb) );
	 for ( i = 0; i < nb; i++ )
	 {
	    lst = DK_envelopListVertAllocate( NULL );
	    DK_in_animToken( file );
	    if ( strcmp( DK_R_ENVELOP_NBVERT_TOKEN, DK_animToken ) == 0 )
	    {
	       DK_in_animLValue( file, &(nb2) );
	       for ( j = 0; j < nb2; j++ )
		  DK_envelopVertReadAscii( file, lst );

	       if ( p->envlist == NULL )
		  p->envlist = lst;
	       else
	       {
		  last->next = lst;
		  lst->prev = last;
	       }
	       last = lst;
	    }
	 }
      }
      else
      {
	 strcpy( lastToken, DK_animToken );
	 return( p );
      }

      DK_in_animToken( file );
   }
}

/*******************************************************************************

   $$L DK_oEnvelopeVertReadBinaryChunk

   - Read data of envelope's vertices assigned to a skeleton's joint.

   Returned Value: DK_Pointer to data, or NULL if failed.

*******************************************************************************/

static DK_EnvVert * DK_oEnvelopeVertReadBinaryChunk
   (
       FILE *file
   )
{
   DK_EnvVert *newvert, *ptr, *vert;
   int i, nb;

   DK_byteswap_fread( &nb, sizeof(int), 1, file );

   vert = NULL;
   ptr = NULL;
   for ( i = 0; i < nb; i++ )
   {
      if ( ( newvert = DK_envelopVertAllocate() ) != NULL )
      {
	 DK_byteswap_fread( &newvert->index, sizeof(int), 1, file );
	 DK_Vector_fread( &newvert->coord, sizeof(DK_Vector), 1, file );
	 DK_byteswap_fread( &newvert->ratio, sizeof(float), 1, file );
	 DK_byteswap_fread( &newvert->thickness, sizeof(float), 1, file );
      }
      else
	 return NULL;

      if ( ptr == NULL )
      {
	 ptr = newvert;
	 vert = ptr;
      }
      else
      {
	 ptr->next = newvert;
	 ptr = ptr->next;
      }
      ptr->next = NULL;
   }

   return ( vert );
}


/*******************************************************************************

   $$L DK_oEnvelopeReadBinaryChunk

   - Read envelopment data of a skeleton's joint.

   Returned Value: DK_Pointer to data, or NULL if failed.

*******************************************************************************/

DK_EnvData * DK_oEnvelopeReadBinaryChunk
   (
       FILE *file
   )
{
   DK_EnvData *pdata;
   float dummyfloat;
   DK_Vector dummyvect;

   if ( ( pdata = DK_envelopDataAllocate( 0 ) ) != NULL )
   {
      DK_byteswap_fread( &pdata->level, sizeof(int), 1, file );
      DK_Vector_fread( &pdata->initvrot, sizeof(DK_Vector), 1, file );
      DK_Vector_fread( &dummyvect, sizeof(DK_Vector), 1, file );
      DK_byteswap_fread( &pdata->indefcrv.type, sizeof(DK_DefType), 1, file );
      DK_byteswap_fread( &pdata->jointarea, sizeof(float), 1, file );
      DK_byteswap_fread( &dummyfloat, sizeof(float), 1, file );
      DK_byteswap_fread( &pdata->indefcrv.beg, sizeof(float), 1, file );
      DK_byteswap_fread( &pdata->indefcrv.end, sizeof(float), 1, file );
      DK_byteswap_fread( &pdata->indefcrv.ampl, sizeof(float), 1, file );
      if ( ( pdata->envlist = DK_envelopListVertAllocate( NULL ) ) != NULL )
	 pdata->envlist->segvert = DK_oEnvelopeVertReadBinaryChunk( file );
   }

   return ( pdata );
}

/*******************************************************************************

   $$P DK_modelActorDataWriteAscii

   - Write all tokens associated with actor (invkin,dynamic,envelop) data.

   Returned Value: None.

*******************************************************************************/

void DK_modelActorDataWriteAscii
   (
       FILE	*file,
      DK_Model	*mdl,
      char	*space
   )
{
   char sspace[80];

   fprintf( file, "\n%sactor_data", space );

   sprintf( sspace, "%s  ", space );

   if ( mdl->type == DK_MDL_NILL || mdl->type == DK_MDL_JNT )
   {
      fprintf( file, "\n%s", sspace );
      DK_modelAlgoTypeWriteAscii( file, mdl, sspace );
   }

   fprintf( file, "\n%s", sspace );
   _DK_OUT_TOKEN(DK_A_ENVTYPE_TOKEN );
   fprintf( file, "       %d\n", mdl->envtype );

   if ( mdl->dchnvar != NULL )
   {
      fprintf( file, "\n%s", sspace );
      _DK_OUT_TOKEN( DK_A_DYNROOT_TOKEN );
   }

   if ( mdl->dmatter != NULL )
   {
      fprintf( file, "\n%s", sspace );
      _DK_OUT_TOKEN( DK_A_DYNMATTER_TOKEN );
      DK_dynamatterWriteAscii( file, mdl->dmatter, sspace );
   }

   if ( mdl->djntinfo != NULL )
   {
      fprintf( file, "\n%s", sspace );
      _DK_OUT_TOKEN( DK_A_DYNJNTPROP_TOKEN );
      DK_dynajntpropWriteAscii( file, mdl->djntinfo, sspace );
   }

   if ( mdl->ikvar != NULL )
   {
      fprintf( file, "\n%s", sspace );
      _DK_OUT_TOKEN(DK_A_INVKINDATA_TOKEN );
      invKinWriteAscii( file, mdl->ikvar, sspace );
   }

   if ( mdl->envdata != NULL )
   {
      fprintf( file, "\n%s", sspace );
      _DK_OUT_TOKEN( DK_W_ENVELOPDATA_TOKEN );
      envelopWriteAscii( file, mdl->envdata, sspace );
   }

   _DK_OUT_NEWLINE();
}

/*******************************************************************************

   $$P actorDataReadAscii

   - Read all tokens associated with actor (invkin,dynamic,envelop) data.

   Returned Value: None.

*******************************************************************************/

void DK_modelActorDataReadAscii
   (
       FILE	*file,
      DK_Model	*mdl,
      char	*lastToken
   )
{
   int	readtok = TRUE;
   short typ;

   DK_in_animToken( file );
   while ( TRUE )
   {
      if ( strcmp(DK_A_CHAIN_TOKEN, DK_animToken ) == 0 )
      {
	 DK_modelAlgoTypeReadAscii( file, mdl );
      }
      else if ( strcmp(DK_A_ENVTYPE_TOKEN, DK_animToken ) == 0 )
      {
	 DK_in_animSValue( file, &(typ) );
	 mdl->envtype = (DK_EnvelopeType) typ;
      }
      else if ( strcmp( DK_A_DYNCTL_TOKEN, DK_animToken ) == 0 )
      {
	 DK_modelAlgoTypeReadAscii( file, mdl );
      }
      else if ( strcmp( DK_A_DYNROOT_TOKEN, DK_animToken ) == 0 )
      {
         mdl->dchnvar = DK_dynaChnVarAllocate();
      }
      else if ( strcmp( DK_A_DYNMATTER_TOKEN, DK_animToken ) == 0 )
      {
         mdl->dmatter = DK_dynamatterReadAscii( file, lastToken );
	 readtok = FALSE;
      }
      else if ( strcmp( DK_A_DYNJNTPROP_TOKEN, DK_animToken ) == 0 )
      {
         mdl->djntinfo = DK_dynajntpropReadAscii( file, lastToken );
	 readtok = FALSE;
      }
      else if ( strcmp(DK_A_INVKINDATA_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->ikvar = invKinReadAscii( file, lastToken );
	 readtok = FALSE;
      }
      else if ( strcmp( DK_R_ENVELOPDATA_TOKEN, DK_animToken ) == 0 )
      {
	 mdl->envdata = envelopReadAscii( file, lastToken );
	 readtok = FALSE;
      }
      else
      {
	 strcpy( lastToken, DK_animToken );
	 return;
      }

      if ( readtok )
	 DK_in_animToken( file );
      readtok = TRUE;
   }
}
