#include <math.h>
#include "tinyModel.h"
#include "DK_filter.h"

#ifdef _WIN32
#include <irixMath.h>
#endif

void printUsage();
void modelTaper( Model *mdl, float radius );

main( int argc, char *argv[] )
{
   Model	*mdl;
   String	 inname = NULL;
   String	 outname = NULL;
   FileFormat	 fmt = FIL_BINARY;
   float	 radius;

#ifdef _WIN32
   DK_streamInitialize();
#endif

   if( argc != 4 )
      printUsage();

   inname = argv[1];
   radius = atof( argv[2] );
   outname = argv[3];

printf( "Taper %s %f %s\n", inname, radius, outname );

   if( mdl = modelReadFile( inname ) )
   {
      modelTaper( mdl, radius );
      if( ! modelWriteFile( outname, mdl, fmt ) )
        printf( "ERROR: Cannot write output file: %s(.hrc)\n", outname );
   }
   else
     printf( "ERROR: Cannot read input file: %s(.hrc)\n", inname );
}

void printUsage()
{
   printf( "usage: taper <infile> <radius> <outfile>\n" );
   printf( "\n" );
   exit( -1 );
}

void modelTaper( Model *mdl, float radius )
{
   Mesh		*msh;
   Vertex	*vrt;
   Vector	*vct;
   Patch	*ptch;
   int		 i, nbp;
   float	 sizeY,  minY, maxY, rad, cylRad;
   float	 fct, scl;

   switch( mdl->type )
   {
      case MDL_MESH:
         msh = (Mesh *) mdl->definition;

        /* found bounding cylinder */
         minY = maxY = msh->vertices[0].position.y;
         cylRad = 0.0;

         for( i = 0, vrt = msh->vertices; i < msh->nbVertices; i++, vrt++ )
         {
            if( vrt->position.y > maxY )   maxY = vrt->position.y;
            if( vrt->position.y < minY )   minY = vrt->position.y;

            rad = sqrt( vrt->position.x * vrt->position.x +
            		vrt->position.z * vrt->position.z   );

            if( rad > cylRad )   cylRad = rad;
         }

         if( (sizeY = (maxY - minY)) < 0.001 )
            break;

         for( i = 0, vrt = msh->vertices; i < msh->nbVertices; i++, vrt++ )
         {
            fct = (vrt->position.y - minY) / sizeY * 2.0 * M_PI;
            scl = cos( fct ) * 0.5 * radius  + 1.0 - 0.5  * radius;
            vrt->position.x *= scl;
            vrt->position.z *= scl;
         }
         break;

      case MDL_PTCH:
         ptch = (Patch *) mdl->definition;

        /* found bounding cylinder */
         minY = maxY = ptch->points[0].y;
         cylRad = 0.0;

         nbp = ptch->uNbPoints * ptch->vNbPoints;
         for( i = 0, vct = ptch->points; i < nbp; i++, vct++ )
         {
            if( vct->y > maxY )   maxY = vct->y;
            if( vct->y < minY )   minY = vct->y;

            rad = sqrt( vct->x * vct->x + vct->z * vct->z );

            if( rad > cylRad )   cylRad = rad;
         }

         if( (sizeY = (maxY - minY)) < 0.001 )
            break;

         for( i = 0, vct = ptch->points; i < nbp; i++, vct++ )
         {
            fct = (vct->y - minY) / sizeY * 2.0 * M_PI;
            scl = cos( fct ) * 0.5 * radius  + 1.0 - 0.5  * radius;
            vct->x *= scl;
            vct->z *= scl;
         }
         break;
   }

   if( mdl->sibling )
      modelTaper( mdl->sibling, radius );

   if( mdl->child )
      modelTaper( mdl->child, radius );
}
