
/*******************************************************************************
********************************************************************************

   scnload.c

   This file reads and writes a scene file into/from the DKit scene structure

   (c) Copyright 1993 Softimage Inc.

********************************************************************************
*******************************************************************************/
static char copyright[] = "(c) Copyright 1993 Softimage Inc.";

#include <getopt.h>
#include <string.h>
#include "tinySoftType.h"
#include "tinyFile.h"
#include "tinySceneToken.h"
#include "tinyScene.h"
#include "tinyScnDump.h"
#include "libdbdk.h"

static int versionMaxIdx = sizeof(DK_VERSION_NAME) / sizeof(float) - 1;

void printUsage (char *execName)
{
   int i; 
   printf("\n\n\n%s - Version %5.2f\n", execName, DK_VERSION_NAME[versionMaxIdx]);
   printf("(c) Copyright 1993, SOFTIMAGE Inc.\n");
   printf("----------------------------------\n\n");

   printf("Usage : %s <scene in> <scene out> [-r <resources>]\n", execName );
   printf("                 [-l]\n");
   printf("                 [-v <version>]\n");
   printf("                 [-d <input db>]\n");
   printf("                 [-O <output db>]\n");
   printf("\n");
   printf("<scene in>  : complete scene name to be read\n");
   printf("<scene out> : output scene name (without prefix and version numbers)\n");
   printf("-r          : path where the resources can be found\n");
   printf("              DEFAULT: /usr/softimage/rsrc\n");
   printf("-l          : use the latest version of the input scene\n" );
   printf("-v          : output scene versions can be\n");
   for (i = 0; i < versionMaxIdx; i++) {
      printf("                   %5.2f,\n", DK_VERSION_NAME[i]);
   }
   printf("                   %5.2f\n", DK_VERSION_NAME[versionMaxIdx]);
   printf("              current directory is used for 2.5 & older,\n");
   printf("              DEFAULT: version %5.2f\n", 
                         DK_VERSION_NAME[versionMaxIdx]);
   printf("-d          : input database\n"); 
   printf("              DEFAULT: 1st db in the DatabaseDir.rsrc file\n");
   printf("-O          : output database\n");
   printf("              DEFAULT: DKIT db if it exists; otherwise,\n");
   printf("                       1st db in the DatabaseDir.rsrc file\n");
}

main( int argc, char *argv[] )
{
   int		err_flag = 0,i,c, l_flag = 0;
   char		*rsrc_path = NULL;
   DK_String	inname = NULL;
   DK_String	outname = NULL;
   DK_String	indb, outdb;
   DK_Scene	*DK_scn;
   DK_Database	*pdb = NULL;
   extern	int	optind,opterr;

#ifdef _WIN32
   DK_streamInitialize();
#endif


   indb  = NULL;
   outdb = "DKIT";

   if ( argc < 3 )
      err_flag++;
   else
   {
      inname  = argv[1];
      outname = argv[2];
      optind = 3;
      opterr = 0;
      while ( (c = getopt(argc, argv, "h:r:v:ld:O:")) != GETOPTDONE )
      {
	 switch(c)
	 {
	    case 'r':
	       rsrc_path = strdup( optarg );
	       break;
	    case 'd':
	       indb = strdup( optarg );
	       break;
	    case 'O':
	       outdb = strdup( optarg );
	       break;
	    case 'v': {
               int i;
               float version = atof (optarg);
               for (i = 0; 
                    i <= versionMaxIdx && version != DK_VERSION_NAME[i]; 
                    i++);
	       if (i > versionMaxIdx) {
                  err_flag++;
               } else {
                  DK_VERSION = i;
               }
	       break;
            }
	    case 'l':
	       l_flag = TRUE;
	       break;
	    case '?':
	    case 'h':
	       err_flag++;
	       break;
	 }
      }
   }
   if ( err_flag || inname == NULL || outname == NULL )
   {
      printUsage(argv[0]);
      exit( -1 );
   }
   if ( rsrc_path == NULL )
      rsrc_path = "/usr/softimage/rsrc";

   DK_scn = DK_sceneAllocate();
   DK_scn->name = strdup( inname );

   /* To Ask for printed status use TRUE here */
   DK_setVerbose( FALSE );

   /*-----------------------------------*/
   /* Need this to init database stuff. */
   /*-----------------------------------*/
   DK_dbInit( rsrc_path, TRUE );	

   /*-----------------------------------------------------------*/
   /* Get a pointer towards the INPUT database. Replace the     */
   /* string by a database name taken from the DatabaseDir.rsrc */
   /* file. If the name is invalid a ptr to the default         */
   /* (first) database is returned.			        */
   /*-----------------------------------------------------------*/
   pdb = (DK_Database *) DK_dbGet( indb );
   if ( pdb == NULL )
      exit( -1 );

   /*-------------------------------------------*/
   /* Get the latest version of the scene that  */
   /* exist in the given database.		*/
   /*-------------------------------------------*/
   if ( l_flag )
   {
      inname = DK_dbGetLatestSceneVersion( inname, pdb );
      printf( "Latest version : uses [%s]\n", inname );
   }

   /*--------------------------------------------*/
   /* Read the scene from default database.      */
   /* Returns a ptr to the allocated DKit scene. */
   /*--------------------------------------------*/
   DK_scn = DK_dbSceneLoad( inname, pdb );	
   if ( DK_scn == NULL )
      exit( -1 );

   /*-----------------------------------*/
   /* Ascii print of the scene contents */
   /*-----------------------------------*/
   DK_scnDump( DK_scn );	

   DK_scn->name = strdup( outname );

   if ( DK_VERSION == DK_v2_5 ) /* old scn format output */
      DK_sceneWriteAsciiAnimation( DK_ASCII_FILE_HEADER, 
		DK_VERSION_NAME[DK_VERSION], DK_scn );
   else
   {
      /*---------------------------------------------------------*/
      /* Get a pointer towards the OUPUT database. Replace the   */
      /* dots by a database name taken from the DatabaseDir.rsrc */
      /* file. If the name is invalid a ptr to the default       */
      /* (first) database is returned.			         */
      /*---------------------------------------------------------*/
      pdb = (DK_Database *) DK_dbGet( outdb );
      if ( pdb == NULL )
	 exit( -1 );

      DK_dbSceneSave( DK_scn, pdb, FALSE ); /* saves scene in given database */
   }


   DK_sceneDispose( &DK_scn );
}
